/* dsa.c - DSA verification
 *	Copyright (C) 1999 Werner Koch (dd9jn).
 *
 * This file is part of SFSV.
 *
 * SFSV is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * SFSV is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include "config.h"
#include <stdio.h>
#include "types.h"
#include "bigint.h"
#include "dsa.h"
#include "sfsv.h"


typedef struct {
    BIGINT p;	    /* prime */
    BIGINT q;	    /* prime factor of p-1 (160 bits) */
    BIGINT g;	    /* generator */
    BIGINT y;	    /* g^x mod p */
} PUBLIC_KEY;

#include "keys.h" /* file generated by makekeys */



/****************
 * Returns true if the signature composed from R and S is valid.
 */
static int
verify( PUBLIC_KEY *pk, BIGINT *m, BIGINT *r, BIGINT *s )
{
    BIGINT zero;
    BIGINT tmp, w, u1, u2, v;
    BIGINT *base[2];
    BIGINT *exp[2];

    bi_init( &zero, 0 ),
    bi_init( &w,  bi_nlimbs(&pk->q) );
    bi_init( &u1, bi_nlimbs(&pk->q) );
    bi_init( &u2, bi_nlimbs(&pk->q) );
    bi_init( &v,  bi_nlimbs(&pk->p) );
    bi_init( &tmp,  bi_nlimbs(&pk->p) );

    /* check that R is less than Q */
    if( bi_cmp( r, &zero ) <= 0 || bi_cmp( r, &pk->q ) >= 0 )
	return 0;
    /* check that S is less than Q */
    if( bi_cmp( s, &zero ) <= 0 || bi_cmp( s, &pk->q ) >= 0 )
	return 0;

    /* w = s^(-1) mod q */
    bi_invm( &w, s, &pk->q );

    /* u1 = (m * w) mod q */
    bi_mulm( &u1, m, &w, &pk->q );

    /* u2 = r * w mod q  */
    bi_mulm( &u2, r, &w, &pk->q );

    /* v =  g^u1 * y^u2 mod p mod q */
    base[0] = &pk->g; exp[0] = &u1;
    base[1] = &pk->y; exp[1] = &u2;
    bi_mulexpm( &tmp, base, exp, 2, &pk->p );
    bi_mod( &v, &tmp, &pk->q );

    return !bi_cmp( &v, r );
}


int
verify_signature( const char *hash, SIG_packet *sig )
{
    PUBLIC_KEY pk;
    BIGINT m;
    byte tmpbuf[22];
    int i;

    /* find the public key */
    for(i=0; keytable[i].p ; i++ ) {
	if( keytable[i].keyid[0] == sig->keyid[0]
	    && keytable[i].keyid[1] == sig->keyid[1] )
	    break;
    }
    if( !keytable[i].p )
	return SFSV_NO_KEY;

    bi_init( &pk.p, 0 );
    bi_init( &pk.q, 0 );
    bi_init( &pk.g, 0 );
    bi_init( &pk.y, 0 );
    bi_scan( &pk.p, keytable[i].p );
    bi_scan( &pk.q, keytable[i].q );
    bi_scan( &pk.g, keytable[i].g );
    bi_scan( &pk.y, keytable[i].y );

    /* encode the hash */
    tmpbuf[0] = 0;
    tmpbuf[1] = 160;
    memcpy( tmpbuf+2, hash, 20 );
    bi_read( &m, tmpbuf, 22 );

    /* and verify */
    return verify( &pk, &m, &sig->r, &sig->s )? 0 : SFSV_BAD_SIG;
}

