/* [wkrc_2.c wk 1.6.91] WKRC: Proc Resource Plain
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * $Header: /usr/src/master/libs/wkswn/wkrc_2.c,v 1.3 1997/01/07 15:18:01 wk Exp $
 *
 * Resource: Plain binary data
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/string.h>

#include "wkrc.h"

/******* constants *********/
#define MAX_BLOCKSIZE  16384	/* max. Size of plain text in this impl. */

/****** types ******/

/***** globals *****/
static char    *memBlock;   /* Ptr to memory Block  */
static size_t  usedMem;     /* # of bytes used in memBlock */
static int     lastWasString; /* flag for string concatenation */
static int     sepEncFlag;   /* set if sep was encountered */

/******* local protos ******/
static int ProcLine( const char *line );
static const char *ProcEntry( const char *buf, int *retErr );
/******** functions *********/

/*
 * Dies Funktion verarbeitet die Zeilen des Resourcetypes:
 *     Plain ( raw binary data format )
 * Mode = 0 : Initialisieren
 *	  1 : Zeile
 *	  2 : Endbehandlung
 * line = bei mode 1 : Sourceline
 *	  sonst      : ResourceName
 * Blanks am Anfang und am Ende einer Zeile sind bereits entfernt.
 * Returns: ErrorCode
 */

int ProcRcPlain( const char *line, int mode )
{
    int err ;

    if( mode == 1 )
	err = *line ? ProcLine( line ) : 0 /* skip empty lines */;
    else if( !mode  ) { /* init */
	memBlock = xmalloc( MAX_BLOCKSIZE );
	usedMem = 0;
	lastWasString = 0;
	sepEncFlag = 1; /* fake an separator at start */
	/* (line contains resourcename) */
	printf("Processing Plain-Resource `%s'\n", line );
	err = 0;
    }
    else { /* exit */
	ushort uhlp;
	ulong lhlp;

	if( sepEncFlag && usedMem ) /* usedMem, caus we faked an sep. */
	    err = ERR_SYNTAX;
	else {
	    StartTempFile( line ); /* contains resourcename */
	    uhlp = RC_PLAIN;
	    WriteTempFile( &uhlp, sizeof uhlp );    /* Resource ID */
	    lhlp = usedMem;
	    WriteTempFile( &lhlp, sizeof lhlp );    /* size of plain data */
	    WriteTempFile( memBlock, usedMem );     /* plain data */
	    err = 0 ;
	}

	free( memBlock ) ;
    }

    return err;
}



/*
 * Eine Zeile verarbeiten, die Zeile kann Format elemente der folgenden
 * Form enthalten:
 *   n	= dezimaler Wert
 *  0n	= oktales Wert
 * 0xn	= heyadezimaler Wert
 * 'c'  = Zeichenwert ( C-Escapeseq. sind erlaubt )
 * "string" = C-String ( die 0 wird mitgespeichert ! )
 * z.Z. drfen die Werte nur im Bereich 0 .. 255 liegen
 * Erweiterungsmglichkeit: Wort, DoppelWort und Floatingpoint erlauben
 * Returns: ErrorCode
 * Die Werte werden im globalen Array memBlock gespeichert.
 * String concatenation wird untersttzt.
 * Die Zeile enthlt mindestens ein Zeichen , Blanks an den Enden
 * sind nicht vorhanden.
 */

static int ProcLine( const char *line )
{
    int err;

    err = 0;
    do {
	/* check for separator */
	if( *line == ',' ) {
	    line++;
	    if( sepEncFlag )
		err = ERR_SYNTAX;   /* two ',' */
	    else {
		sepEncFlag++;
		line += strspn( line, "\t " );
	    }
	}

	/* syntax check between elements */
	if( *line == '\"' ) { /* next entry is a string */
	    if( lastWasString && sepEncFlag )
		lastWasString = 0; /* can't concatenate any more */
	    else if( !lastWasString && !sepEncFlag )
		err = ERR_SYNTAX ;
	}
	else if( !sepEncFlag )
	    err = ERR_SYNTAX;

	if( !err && *line ) {
	    /* proc element */
	    line = ProcEntry( line, &err );
	    sepEncFlag = 0;
	    if( !err )
		line += strspn( line, "\t " ); /* skip white spaces */
	}
    } while( !err && *line );

    return err ;
}



/*
 * Ein einzelnes Element bearbeiten:
 * buf zeigt auf den Anfang eines Elements, wobei white spaces entfernt sind
 * Ein Komma am Anfang ist bereits entfernt.
 * Returns: Pointer to next character in buf
 */

static const char *ProcEntry( const char *buf, int *retErr )
{
    int err;
    size_t stringLen, unused;
    int value;
    const char *newpos=NULL;
    char *pp;
    long hl;

    err = 0;
    if( *buf == '\"' ) { /* String entry */
	if( lastWasString )
	    usedMem--;	/* append String: remove last stringterminator */
	/* get the String */
	unused = MAX_BLOCKSIZE - usedMem ;
	if( unused < strlen(buf)+1 )
	    err = ERR_R2BIG; /* may be too big */
	else {
	    newpos = ProcStrg( memBlock+usedMem, buf, unused, &stringLen );
	    if( !newpos )
		err = ERR_INVSTR;
	    else {
		usedMem += stringLen;
		memBlock[usedMem++] = '\0' ; /* append String terminator */
	    }
	}
	lastWasString = 1;
    }
    else if( usedMem+1 >= MAX_BLOCKSIZE-1 ) /* we need room for 1 byte */
	err = ERR_R2BIG;    /* Resource will be get too large */
    else {  /* not a String */
	if( *buf == '\'' ) { /* a Character */
	    if( !(newpos = ProcChar( &value, buf )) )
		err = ERR_INVCHR;
	}
	else { /* take value */
	    hl = strtol( buf, &pp, 0 );
	    newpos = pp;
	    if( hl < 0L || hl > 0x00ff )
		err = ERR_RANGE;
	    else
		value = (int)hl;
	}

	memBlock[usedMem++] = (byte)(value & 0xff);
	lastWasString = 0;
    }

    *retErr = err;
    return newpos;
}

/**** end of file ****/
