/* [wkrc.c wk 1.6.91] Resource Compiler
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * $Header: /usr/src/master/libs/wkswn/wkrc.c,v 1.3 1997/01/07 14:47:13 wk Exp $
 *
 * Doku: siehe RC.DOC
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/string.h>
#include <wk/file.h>

#include "wkrc.h"

/******* constants *********/
/****** types ******/
/***** globals *****/
static char *tempRsName;
static char *tempIxName;	/* name of temp. Index File */
static FILE *tempRsStream;
static FILE *tempIxStream;
static int noOfResources;	/* # of resources yet processed */

static struct {
	int quiet;
    } opt ;

/******* local protos ******/
static void CleanUp(void);
static void CreateTempFile(void);
static void BuildLibrary( FILE *libSt );

/***** def'd functions *****/
#define is_octdigit( c ) ( (c) >= '0' && (c) <= '7' )
#define oct_value( c )	((byte)((c)-'0'))
#define hex_value( c )	(isdigit(c)?(byte)((c)-'0'):\
			(byte)((c)-(isupper(c)?'A':'a')+10))
/******** functions *********/

const char *CopyRight( int level )
{
    const char *p;
    switch( level ) {
      case 10:
      case 0:	p = "WKRC - Resource Compiler Ver. 1.01; "
		    "Copyright (c) 1991 by Werner Koch" ; break;
      case 1:
      case 11:	p = "usage: WKRC [options] Sourcefiles (-h for help)"; break;
      case 2:
      case 12:	p =
    "Syntax: WKRC [options] {file}\n\n"
    "Compiles Resourcefiles to output file \"r.out\"\n"
    "Options summary:\n"
    " -q = quiet mode\n"
    " -o<name>[<ext>] = Set output file to \"<name>.exr\"\n"
    " -h = help\n";
	break;
      case 13:	p = "WKRC"; break;
      case 14:	p = "1.01"; break;
      default: p = "?";
    }

    if( !level ) { puts( p ); putchar('\n'); }
    else if( level == 1 ) { fputs( p, stderr ); putc( '\n', stderr );exit(3);}
    else if( level == 2 ) {  /* help */ puts( p ); exit(0); }

    return p;
}

void main( int argc, char **argv )
{
    FILE *stream;
    int helper;
    long lineCount;
    int err;
    char *libFileName, *s;

    Initialize_Main();
  #ifndef UNIX
    if( ArgExpand( &argc, &argv, 4|1 ) )
	Error(4,GetStr(12));
  #endif

    libFileName = "r.out";
    for( s=""; --argc && **++argv == '-' && *s != '-'; )
	for( s = *argv + 1 ; *s && *s != '-' ; s++ )
	    switch( *s ) {
	      case 'q': opt.quiet++ ; break ;
	      case 'o':
		libFileName = xmalloc( strlen( s+1 )+1+4 ); /* for Extension */
		strcpy( libFileName, s+1 );
		if( !FileExtCheck( libFileName ) )
		    FileExtAppend( libFileName, "exr" );
		while( *s ) s++; s--;
		break;
	      case 'h':
	      case '?' : CopyRight(0) ; CopyRight(2); break;
	      default  : CopyRight(1);
	    }
    if( argc < 1 )
	CopyRight(1);
    if( !opt.quiet )
	CopyRight(0);

    atexit( CleanUp ) ;
    CreateTempFile();

    err = 0;
    for( ; argc ; argc--, argv++ ) {
	if( !(stream = fopen( *argv, "r") ) ) {
	    fprintf(stderr,"Error opening file `%s'\n", *argv);
	    exit(2);
	}
	else
	    printf("Processing file: `%s'\n", *argv);
	lineCount = 1L;
	helper = 0;
	do {
	    err = ProcNextResource( stream, &lineCount, *argv, &helper );
	} while( !(err || feof(stream) || ferror(stream) ) );

	fclose(stream);
	DeleteAllMacros();
    }


    if( err )
	printf( "Compilation errors (last: %s)\n", GetErrorString(err));
    else {
	/* Library neu erzeugen und Header schreiben */
	stream = xfopen( libFileName , "w+b" ) ;
	fputs( "RC\x01 Copyright (c) 1991 by DD9JN ", stream);
	putc( '\0', stream) ;
	putc( '\0', stream) ; /* No CompressInfo */

	BuildLibrary( stream ) ;
	fclose( stream );
	printf("Successful compiled to '%s'\n", libFileName);
    }

    exit( err ? 2 : 0 );
} /* main() */




/********************
 *  General Cleanup function
 */

static void CleanUp()
{
    static int inst = 0 ;

    if( inst )
	return ;    /* call this function only once */
    inst++;
    if( tempIxStream )
	fclose( tempIxStream );
    if( tempRsStream )
	fclose( tempRsStream );
    if( tempIxName )
	remove( tempIxName ), free( tempIxName );
    if( tempRsName )
	remove( tempRsName ), free( tempRsName ); ;
}


/************************
 * Den Temporren File zum Zwischenspeichern aller bereits
 * verarbeiteten Resourcen erzeugen und entsprechende
 * Statistiken initialisieren.
 * Bei	einem Fehler bricht die Funktion sofort ab.
 */

static void CreateTempFile()
{
    if( !(tempIxName = CreateTmpFile("rci")) )
	Error(1004,GetStr(8)) ;
    if( !(tempRsName = CreateTmpFile("rcd")) )
	Error(1004,GetStr(8)) ;
    tempIxStream = xfopen( tempIxName, "w+b" ) ;
    tempRsStream = xfopen( tempRsName, "w+b" ) ;
    noOfResources = 0;
}


/************************
 * Die Funktion startet einen Temporren File, dabei wird
 * der Name der Resource angegeben.
 * Intern werd der Name in den IndexFile geschrieben mit dem Offset
 * des tempRsFile; dieser offset braucht dann spter nur noch
 * korrigiert werden.
 */

void StartTempFile( const char *name )
{
    long help;
    int err ;

    err = 0;
    help = ftell( tempRsStream );
    xassert( help >= 0L );
    if( fwrite( &help, sizeof help, 1 , tempIxStream ) != 1 )
	err++;
    else if( fwrite( name, strlen(name)+1, 1 , tempIxStream) != 1 )
	err++ ;
    if( err ) {
	fprintf(stderr,"Fatal error writing to temporary index file\n" );
	exit(4);
    }

    /* !!! hier noch prfen ob der Name nicht doppelt ist */

    noOfResources++;
}


/*************************
 * Daten an den Tempfile anhngen
 */

void WriteTempFile( const void *buffer, size_t size )
{
    if( fwrite( buffer, size, 1 , tempRsStream ) != 1 ) {
	fprintf(stderr,"Fatal error writing to temporary data file\n" );
	exit(4);
    }
}




/*********************************
 * Die Library aufbauen.
 * Der Header ist bereits geschrieben.
 */

static void BuildLibrary( FILE *libSt )
{
    long rOffset, lenOfIxFile, hl;
    int c ;

    /* calculate relocation offset */
    rOffset = ftell( libSt ); xassert( rOffset >= 0L );
    lenOfIxFile = ftell( tempIxStream ); xassert( lenOfIxFile >= 0L );
    /* we will add 2 longs to the index table */
    rOffset += lenOfIxFile + 2*sizeof(long);
    /* now we can use rOffset to relocate */
    /* the entries in the indextable */

    rewind( tempIxStream );
    /* Write Index Table */
    if( !opt.quiet )
	puts("Writing indextable");
    while( lenOfIxFile > 0L ) {
	if( fread( &hl, sizeof hl, 1 , tempIxStream) != 1 ) {
	    fprintf(stderr,"Error reading temporary index file\n");
	    exit(4);
	}
	lenOfIxFile -= sizeof hl ;
	hl += rOffset ;
	if( fwrite( &hl, sizeof hl, 1 , libSt ) != 1 ) {
	    fprintf(stderr,"Error writing to library\n");
	    exit(4);
	}

	while( lenOfIxFile--, c = getc(tempIxStream) )
	    putc( c, libSt ) ;
	putc( '\0', libSt ) ;
    }

    if( lenOfIxFile )
	Bug("%ld bytes remaining in tempIndexFile", lenOfIxFile );

    hl = -1L ;	/* end of indextable marker */
    if( fwrite( &hl, sizeof hl, 1 , libSt ) != 1 ) {
	fprintf(stderr,"Error writing to library\n");
	exit(4);
    }
    hl = 0L ;  /* reserved for future use */
    if( fwrite( &hl, sizeof hl, 1 , libSt ) != 1 ) {
	fprintf(stderr,"Error writing to library\n");
	exit(4);
    }


    /* now copy DataFile to Library File */
    if( !opt.quiet )
	puts("Writing datablocks");
    rewind( tempRsStream );
    while( (c = getc(tempRsStream)) != EOF ) {
	if( putc( c, libSt ) == EOF ) {
	    fprintf(stderr,"Error writing to library\n");
	    exit(4);
	}
    }

    /* Ready */
}



/**********************
 * Diese Funktion gibt eine  Fehlermeldung aus
 * wenn String ungleich NULL ist, so wird dies als erste Zeile gedruckt.
 */

void PrintError( const char *string, const char *file, long line,  int err )
{
    if( string )
	puts( string ) ;
    printf( "\"%s\" line %ld error: %s\n", file, line, GetErrorString(err)  );

}



/*********************
 * Einen ErrorCode auf einen ErrorText mappen
 */

const char *GetErrorString( int err )
{
    const char *p;

    switch(err) {
      case ERR_UNKOWN: p = "Unkown"; break;
      case ERR_DUPMAC: p = "Duplicated defined macro"; break;
      case ERR_INVMAC: p = "Invalid form of macro"; break;
      case ERR_UDFMAC: p = "Undefined macro"; break;
      case ERR_RANGE : p = "Value out of range"; break;
      case ERR_FILENF: p = "File not found"; break;
      case ERR_FATAL : p = "Fatal"; break;
      case ERR_SYNTAX: p = "Syntax error"; break;
      case ERR_INVKWD: p = "Invalid keyword"; break;
      case ERR_UKWKWD: p = "Unkown keyword"; break;
      case ERR_DUPNAM: p = "Duplicated name"; break;
      case ERR_DUPTYP: p = "Duplicated type"; break;
      case ERR_INVTYP: p = "Invalid type"; break;
      case ERR_PRMEOF: p = "Premature end-of-file"; break;
      case ERR_R2BIG : p = "Resource too big"; break;
      case ERR_R2MANY: p = "Too many resource entries"; break;
      case ERR_VALUE : p = "Invalid value"; break;
      case ERR_INVSTR: p = "Invalid string"; break;
      case ERR_INVCHR: p = "Invalid character"; break;
      case ERR_UBNEST: p = "Unbalanced nesting"; break;
      case ERR_ALDEF : p = "Already defined"; break;
      case ERR_INCMPL: p = "Incomplete data"; break;
      case ERR_2MBTNS: p = "Too many buttons"; break;
      default: p = "????";
    }

    return p;
}



/***************************************
 * ProcStrg	copy a string with escape characters
 *
 * const char *ProcStrg( char *d, const char *s, size_t max, size_t *copied );
 * char *d ;		  Destination Buffer
 * char *s ;		  Source String
 * size_t max ; 	  Lnge des Buffers
 * size_t copied;	  Anzahl der kopierten Zeichen
 *
 * Die Funktion kopiert einen String in einen Buffer:  Es wird auf den
 * Stringanfang ('\"') gewartet und die zeichen danach bis zum Stringende
 * ('\"' ) kopiert.  Die C Escapesequencen werden dabei
 * bercksichtigt.  Die
 * Stringbegrenzer werden nicht kopiert.  max gibt die maximal Lnge des
 * buffers an, wird diese erreicht, so wird der ein Fehler zurckgeliefert
 * Return Value
 *   NULL = Error: Buffer too short or string not terminated
 *   Pointer auf das nchste Zeichen nach dem Stringendezeichen
 */

const char *ProcStrg( char *d, const char *s, size_t n, size_t *copied )
{
    int escape, okay ;


    okay = 0;
    *copied = n ; /* save max. buffer length */

    /* Stringanfang suchen */
    while( *s )
	if( *s++ == '\"' )
	    break ;

    for( escape = 0 ; *s && n; s++ ) {
	if( escape ) {
	    if( is_octdigit( *s ) && is_octdigit(s[1]) && is_octdigit(s[2]) ) {
		*d++ = (char)(64*oct_value(*s) +
			       8*oct_value(s[1])+oct_value(s[2])) ;
		s += 2 ;
	    }
	    else
		switch( *s ) {
		  case 'x':
		    if( isxdigit(s[1]) && isxdigit(s[2]) ) {
			*d++ = (char)(16 * hex_value(s[1]) + hex_value(s[2]));
			s += 2 ;
		    }
		    else
			*d++ = 'x' ;
		    break ;
		  case 'a':   *d++ = '\a' ; break ;
		  case 'b':   *d++ = '\b' ; break ;
		  case 'f':   *d++ = '\f' ; break ;
		  case 'n':   *d++ = '\n' ; break ;
		  case 'r':   *d++ = '\r' ; break ;
		  case 't':   *d++ = '\t' ; break ;
		  case 'v':   *d++ = '\v' ; break ;
		  default:    *d++ = *s ;
		}
	    n-- ;
	    escape = 0 ;
	}
	else if( *s == '\\' )
	    escape = 1 ;
	else if( *s == '\"' ) {
	    okay++;
	    s++;
	    break ;
	}
	else {
	    *d++ = *s ;
	    n-- ;
	    escape = 0 ;
	}
    }

    *copied -= n ; /* actual copied charcters */

    return okay ? s : NULL ;
}




/***************************************
 * ProcChar	copy a c-formatted char characters
 *
 * const char *ProcChar( int *value, const char *s );
 * int	*value		  Returns value of char
 * char *s ;		  Source String
 *
 * Die Funktion ermittelt den Wert eines Zeichens, welches im C-Format
 * als String vorliegt. Es wird auf den Zeichenanfang '\''
 *  gewartet.
 * Return Value
 *   NULL = Error: Syntaxfehler
 *   Pointer auf das nchste Zeichen nach dem Zeichenendezeichen
 */

const char *ProcChar( int *value, const char *s )
{
    int escape, anyValue, okay;

    okay = anyValue = 0;
    /* Zeichenanfang suchen */
    while( *s )
	if( *s++ == '\'' )
	    break ;

    for( escape = 0 ; *s ; s++ ) {
	if( escape ) {
	    if( is_octdigit( *s ) && is_octdigit(s[1]) && is_octdigit(s[2]) ) {
		*value = (byte)(64*oct_value(*s) +
			       8*oct_value(s[1])+oct_value(s[2])) ;
		s += 2 ;
	    }
	    else
		switch( *s ) {
		  case 'x':
		    if( isxdigit(s[1]) && isxdigit(s[2]) ) {
			*value = (byte)(16 * hex_value(s[1]) + hex_value(s[2]));
			s += 2 ;
		    }
		    else
			*value = 'x' ;
		    break ;
		  case 'a':   *value = '\a' ; break ;
		  case 'b':   *value = '\b' ; break ;
		  case 'f':   *value = '\f' ; break ;
		  case 'n':   *value = '\n' ; break ;
		  case 'r':   *value = '\r' ; break ;
		  case 't':   *value = '\t' ; break ;
		  case 'v':   *value = '\v' ; break ;
		  default:    *value = *(byte*)s ; break;
		}
	    anyValue++;
	    escape = 0 ;
	}
	else if( *s == '\\' )
	    escape = 1 ;
	else if( *s == '\'' ) {
	    okay = anyValue == 1;
	    s++;
	    break ;
	}
	else {
	    *value = *(byte*)s ;
	    anyValue++;
	    escape = 0 ;
	}
    }

    return okay ? s : NULL ;
}


/**** end of file ****/
