/* [scrmod0.c wk 22.4.91] Screen Manager
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkCUA.
 *
 *  WkCUA is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkCUA is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Header: /usr/src/master/libs/wkswn/scrmod0.c,v 1.5 1997/01/07 15:18:00 wk Exp $
 *
 *  Main Module for Screen Manager
 */

#include <wk/tailor.h>

/*** to show PS at the second monitor uncomment the next line ***/
/* #define DEBUG_PS_AT_MONO 1 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <wk/lib.h>
#include <wk/mouse.h>
#include <wk/keys.h>
#if DEBUG_PS_AT_MONO && MSDOS
    #include <dos.h>
#endif

#include "scr_sy.h"

/****** constants *********/
#define MAX_HANDLES 20
/******** typedefs ********/

typedef struct {
	int useFlag;	    /* 0: unused*/
	int   px, py ;	    /* position of Window*/
	int   sx, sy ;	    /* size of Window*/
	int x1, x2, y1, y2 ;
	byte   attr;	    /* attribut of window*/
	ushort style;	    /* Flags from open*/
	ushort mode;	    /* Flags from open*/
	cell_t *data ;	    /* pointer to Data*/
	int    dSize;	    /* Size of data*/
	char  crsActiv;     /* Cursor is activ (not hidden)*/
	byte  crsForm;	    /* Cursors Form*/
	int crsX, crsY;     /* cursors Position*/
    } control_t ;


/******* globals **********/
int scrErrno = 0;   /* definition of scrErrno from scr.h */

static int initOkay ;
static int msCrsHidden ;
static control_t controlTbl[MAX_HANDLES] ; /*handle 0 reserved for internal use*/
static int lastWindow[MAX_HANDLES];
static struct {
	int x1, x2, y1, y2 ;	/* Limits of the PS*/
	int xSize, ySize;
	int ix1, ix2, iy1, iy2 ; /* invalidated Rectangle */
	int upd;		/* update Flag */
    } psLimit ;
static int rebuildPending ;
static int crsWindowHd; 	/* handle of Window with activ cursor */

static void (*DrvUpdate)(cell_t *,int,int,int,int,int);
static void (*DrvGetInfo)(scrPhyInf_t *);
static void (*DrvSetCrs)( int, int, int);
static void (*DrvDevCtrl)( int, scrPhyDevCtrl_t *);

/****** prototypes ********/
static const char *TheScrDriver(void);

static void ClearWindow( control_t *win );
static void SetInvRect( int x, int y );
static void ResetInvRect(void);

static void DummyDrvUpdate(cell_t*,int,int,int,int,int);
static void DummyDrvGetInfo(scrPhyInf_t *);
static void DummyDrvSetCrs( int, int, int);
static void DummyDrvDevCtrl( int, scrPhyDevCtrl_t *);

/***** def. Function ******/
#define EnterCritical()     /* Dummies*/
#define LeaveCritical()     /* Dummies*/
/******* Functions ********/

#if __MSC__
#pragma check_stack(off)
#endif
static const char *TheScrDriver()
{
    return DrvUpdate == DummyDrvUpdate || !initOkay ? NULL : "unkownDriver" ;
}
#if __MSC__
#pragma check_stack(on)
#endif



#ifdef DOCUMENTATION
@Summary ScrInitialize	    Initialisieren des Windowsystems
 #include <wk/scr.h>

 int ScrInitialize( 0,0,0,0,0,0,0 );
 --> Die Argumente sind fr Erweiterungen vorgesehen
     unsd sollten z.Z. alle 0 sein.
@Description
 Diese Funktion initialisiert das Windowsystem.
 Erst nach dieser Funktion knnen andere Funktionen
 aufgerufen werden. Ein Bildschirmtreiber wird aber noch
 nicht aktiviert.
@Return Value
 ErrorCode
@See Also
 ScrInitDriverStd ScrErrorText
#endif /*DOCUMENTATION*/

int ScrInitialize( int level, int sx, int sy, int a, int b, ushort c, ushort d)
{
    int i , err ;
    control_t *win ;

    err = 0 ;
    if( initOkay )
	{ err = ESCR_GENERAL_ERROR ; goto retlab ; }
    if( level || a || b || c ||d ) /* not yet defined*/
	{ err = ESCR_INVALID_VALUE ; goto retlab ; }
    if( !sx )
	sx = 80 ;
    if( !sy )
	sy = 25 ;

    EnterCritical() ;
    for(i=0; i < MAX_HANDLES ; i++ ) {
	controlTbl[i].useFlag = 0 ;
	lastWindow[i] = -1 ; /* invalidate all entries*/
    }
    initOkay = 1 ;
    rebuildPending = 1 ;   /* Show an empty PS */
    LeaveCritical() ;
    /** Now Open the PS (with handle 0) ***/
    psLimit.x1 = 0 ;
    psLimit.x2 = 79;
    psLimit.y1 = 0 ;
    psLimit.y2 = 24;
    psLimit.xSize = 80 ;
    psLimit.ySize = 25 ;
    ResetInvRect();
    SetInvRect( psLimit.x1, psLimit.y1 );
    SetInvRect( psLimit.x2, psLimit.y2 );
    /*** handle 0 is used for the PS ***/
    win = controlTbl ;
    win->useFlag = 1;
    win->px = psLimit.x1 ;
    win->py = psLimit.y1 ;
    win->sx = psLimit.xSize ;
    win->sy = psLimit.ySize ;
    win->x1 = psLimit.x1 ;
    win->x2 = psLimit.x2 ;
    win->y1 = psLimit.y1 ;
    win->y2 = psLimit.y2 ;
    win->style = 0;
    win->mode  = 0;
    win->dSize = psLimit.xSize * psLimit.ySize;
  #if DEBUG_PS_AT_MONO
    win->data = MK_FP( 0xb000, 0 );
  #else
    if( !(win->data  = malloc(win->dSize * sizeof *win->data )) )
	{ err = ESCR_OUT_OF_MEMORY ; goto retlab ; }
  #endif
    /** Fill PS with Blanks **/
    MouseInitialize();
    crsWindowHd = -1;
    msCrsHidden = 1;
    ClearWindow(win);
    ScrChangeRedirector(1, TheScrDriver );

  retlab:
    return scrErrno = err ;
}



#ifdef DOCUMENTATION
@Summary ScrOpen	Open a Screen Window
 #include <wk/scr.h>

 int ScrOpen( retHandle, level, px, py, sx, sy, attr, sFlags, mFlags );
 int *retHandle;	Return: Handle des Windows
 int level;		muss 0 sein
 int px,py,sx,sy;	Position und Gre des Windows
 byte attr;		Attribut des Windows
 ushort sFlags; 	Style Flags ( SCR_STYLE_... ) (Bitweise)
 uhort	mFlags; 	ModeFlags   ( SCR_MODE_... ) (Bitweise)
@Description
 Diese Funktion erzeugt ein Window der Gre sx * sy an Position px,py
 im PS. sx/sy sind die innere nutzbare Gre, d.h. ohne einen
 eventuellen Frame; ebenso ist die Position px/py die obere linke Ecke
 diese nutzbaren Area. Wird fr px oder py -1 angegeben, so wird das
 Window auf der entsprechenden Achse zentriert.
 Mit den Style Flags knnen Rahmen, Schatten und spter eventuell
 weitere Eigenschaften des Windows definiert werden.
 Das Attribut gibt die Farbe des Windows und damit insbesondere die
 eines evntl. Rahmens an.
 Der zurckgegeben Handle ist bei allen weiteren Operationen auf dieses
 Window zu verwenden.
 Wird als Mode SCR_MODE_HIDDEN mit angegeben, so muss zum Sichtbarmachen
 die Funktion ScrShow() aufgerufen werden.
@Return Value
 ErrorCode
 und falls dieser 0 ist, in retHandle den Handle des neuen Windows
@See Also
 ScrClose
#endif /*DOCUMENTATION*/

int ScrOpen( int *retHandle,
	     int level, int px, int py, int sx, int sy, byte attr,
	     ushort styleFlags, ushort modeFlags )
{
    int i, err ;
    control_t *win ;
    cell_t aCell;

    err = 0 ;
    win = NULL ;
    EnterCritical() ;
    for(i=1; i < MAX_HANDLES ; i++ )
	if( !controlTbl[i].useFlag ) {
	    win = controlTbl + i;
	    win->useFlag++ ;
	    win->data = NULL ;
	    *retHandle = i ;
	    break ;
	}
    LeaveCritical() ;
    if( !win )
	{ err = ESCR_OUT_OF_HANDLES ; goto retlab ; }

    if( px == -1 )
	px = (psLimit.xSize - sx)/2 ;
    if( py == -1 )
	py = (psLimit.ySize - sy)/2 ;

    if( px < psLimit.x1 || px > psLimit.x2 ||
	py < psLimit.y1 || py > psLimit.y2 ||
	sx < 1 || sy < 1 || attr > 127 ||
	px+sx > psLimit.xSize || py+sy > psLimit.ySize
      ) {
	err = ESCR_OUT_OF_RANGE;
	goto retlab;
    }
    win->px = px ;
    win->py = py ;
    win->sx = sx ;
    win->sy = sy ;
    win->x1 = px ;
    win->x2 = px + sx -1 ;
    win->y1 = py ;
    win->y2 = py + sy -1 ;
    win->attr = attr ;
    win->style = styleFlags ;
    win->mode  = modeFlags ;
    win->dSize = sy * sx ;
    win->crsActiv = 0;	   /* Cursor inactiv */
    win->crsForm = 1;	   /* Form A*/
    win->crsX = win->crsY = 0;
    if( win->style & (SCR_STYLE_FRAME|SCR_STYLE_DBLFRAME) ) {
	/** check for correct values if a frame is specified **/
	if( !win->x1 || win->x2 == psLimit.x2 ||
	    !win->y1 || win->y2 == psLimit.y2 )
	    { err = ESCR_OUT_OF_RANGE ; goto retlab ; }
	SetInvRect( win->x1-1, win->y1-1 );
	SetInvRect( win->x2+1, win->y2+1 );
    }
    else {
	SetInvRect( win->x1, win->y1 );
	SetInvRect( win->x2, win->y2 );
    }
    if( !(win->data  = malloc(win->dSize * sizeof *win->data )) )
	{ err = ESCR_OUT_OF_MEMORY ; goto retlab ; }
    /** Fill Window with Blanks **/
    aCell.c = (char)' ' ;
    aCell.a = attr ;
    for( i=0; i < win->dSize; i++ )
	win->data[i] = aCell ;

    EnterCritical() ;
    for(i=0; i < MAX_HANDLES ; i++ )
	if( lastWindow[i] == -1 ) { /* insert into list as last one */
	    lastWindow[i] = *retHandle ;
	    break;
	}
    LeaveCritical() ;
    xassert( i < MAX_HANDLES );

  retlab:
    if( win && err ) { /* Cleanup if error*/
	free( win->data ) ;
	EnterCritical() ;
	win->useFlag = 0 ;
	LeaveCritical() ;
    }
    return scrErrno = err ;
}



#ifdef DOCUMENTATION
@Summary ScrClose	Close an Window
 #include <wk/scr.h>

 int ScrClose( hd );
 int hd;	handle des Windows
@Description
 Dies ist die inverse Funktion zu ScrOpen().
 das Window wird vom PS entfernt und damit ist der Handle
 ungltig und kann vom System neu vergeben werden.
 Das Entfernen ist selbstvertndlich erst nach dem nchsten
 Aufruf von ScrUpdate() fr den User sichtbar.
@Return Value
 ErrorCode ( InvalidHandle )
@See Also
 ScrOpen
#endif /*DOCUMENTATION*/

int ScrClose(int handle)
{
    control_t *win ;
    int i ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    free( win->data ) ;
	    if( crsWindowHd == handle )
		crsWindowHd = -1;
	    EnterCritical() ;
	    for(i=0; i < MAX_HANDLES ; i++ )
		if( lastWindow[i] == handle ) { /* handle found*/
		    for(; i < MAX_HANDLES-1; i++ )
			lastWindow[i] = lastWindow[i+1] ;
		    lastWindow[MAX_HANDLES-1] = -1 ;
		    break;
		}
	    xassert( i < MAX_HANDLES );
	    win->useFlag = 0 ;
	    rebuildPending = 1 ;
	    LeaveCritical() ;
	    return scrErrno = 0 ;
	}
    return scrErrno = ESCR_INVALID_HANDLE ;
}



#ifdef DOCUMENTATION
@Summary ScrWriteCell	    Write a Datacell onto the Window
 #include <wk/scr.h>

 int ScrWriteCell( hd, x, y, c, a );
 int hd;	Handle des Windows
 int x,y;	Position der Zelle
 char c;	zu schreibendes Zeichen
 byte a;	Attribut des Zeichens
@Description
 Gibt ein Zeichen aus. Die Kombination von Zeichen und Attribut
 wird als Zelle ( Cell ) bezeichnet. Es wird nur an gltigen
 Positionen ausgegeben; scrolling wird nicht durchgefhrt.
 Die Koordinaten sind auf die obere linke Ecke bezogen.
@Return Value
 ErrorCode ( InvalidHandle, OutOfRange )
@See Also
 ScrReadCell
#endif /*DOCUMENTATION*/

int ScrWriteCell( int handle, int x, int y, char c, byte a )
{
    control_t *win ;
    int i ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    i = y * win->sx + x ;
	    if( i >= 0 && i < win->dSize ) {
		if( win->data[i].c != c || win->data[i].a != a ) {
		    win->data[i].c = c ;
		    win->data[i].a = a ;
		    SetInvRect( win->px+x, win->py+y );
		}
		return scrErrno = 0;
	    }
	    else
		return scrErrno = ESCR_OUT_OF_RANGE ;
	}
    return scrErrno = ESCR_INVALID_HANDLE ;
}



#ifdef DOCUMENTATION
@Summary ScrReadCell	    Read a Datacell from the Window
 #include <wk/scr.h>

 int ScrReadCell( hd, x, y, c, a );
 int hd;	Handle des Windows
 int x,y;	Position der Zelle
 char *c;	!= NULL: Return: gelesenes Zeichen
 byte *a;	!= NULL: Return: gelesenes Attribut
@Description
 Liest ein Zeichen ein. Es wird nur von gltigen
 Positionen gelesen.
@Return Value
 ErrorCode ( InvalidHandle, OutOfRange )
@See Also
 ScrReadCell
#endif /*DOCUMENTATION*/

int ScrReadCell( int handle, int x, int y, char *c, byte *a )
{
    control_t *win ;
    int i ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    i = y * win->sx + x ;
	    if( i >= 0 && i < win->dSize ) {
		if( c )
		    *c = win->data[i].c ;
		if( a )
		    *a = win->data[i].a ;
		return scrErrno = 0;
	    }
	    else
		return scrErrno = ESCR_OUT_OF_RANGE ;
	}
     return scrErrno = ESCR_INVALID_HANDLE ;
}



#ifdef DOCUMENTATION
@Summary ScrGetSize	    Get Size of Window
 #include <wk/scr.h>

 void ScrGetSize( hd, sx, sy );
 int hd;	Handle des Windows
 int *sx;	!= NULL: Returns: Anzahl des Spalten des Windows
 int *sy;	!= NULL: Returns: Anzahl der zeilen des Windows
@Description
 Ermittelt die Gre eines Windows.
 Wird als Handle 0 angegeben, so wird statt der Windowgre
 die Gre des Presentationspaces ermittelt.
#endif /*DOCUMENTATION*/

void ScrGetSize( int handle, int *sx, int *sy )
{
    if( sx )
	*sx = controlTbl[handle].sx;
    if( sy )
	*sy = controlTbl[handle].sy;
}



#ifdef DOCUMENTATION
@Summary ScrGetPos	    Get Position of Window
 #include <wk/scr.h>

 void ScrGetPos( hd, mode, px, py );
 int hd;	Handle des Windows
 int mode;	0 = Get Position des Windows im PS
		1 = Get CursorPosition innerhalb des Windows
 int *px;	!= NULL: Returns: X-Wert
 int *py;	!= NULL: Returns: Y-Wert
@Description
 Ermittelt die Position des Windows im PS.
 Handle 0 anzugeben ist unsinnig.
#endif /*DOCUMENTATION*/

void ScrGetPos( int handle, int mode, int *px, int *py )
{
    if( px ) {
	switch( mode ) {
	  case 0:   *px = controlTbl[handle].px; break;
	  case 1:   *px = controlTbl[handle].crsX; break;
	}
    }
    if( py ) {
	switch( mode ) {
	  case 0:   *py = controlTbl[handle].py; break;
	  case 1:   *py = controlTbl[handle].crsY; break;
	}
    }
}




#ifdef DOCUMENTATION
@Summary ScrHide	Hide an Window
 #include <wk/scr.h>

 int ScrHide( hd );
 int hd;	Handle des Windows
@Description
 Versteckt ein Window, alle Operationen auf das Window bleiben aber
 gltig und werden ausgefhrt ( inclusive ScrUpdate(), wobei dort
 nur die Reihenfolge gendert wird - aber nicht angezeigt ).
 Sichtbar erst nach dem nchst ScrUpdate()
@Return Value
 ErrorCode ( InvalidHandle )
@See Also
 ScrShow
#endif /*DOCUMENTATION*/

int ScrHide( int handle )
{
    control_t *win ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    if( !(win->mode & SCR_MODE_HIDDEN ) ) {
		win->mode |= SCR_MODE_HIDDEN;
		EnterCritical() ;
		rebuildPending = 1 ;
		LeaveCritical() ;
	    }
	    return scrErrno = 0;
	}
     return scrErrno = ESCR_INVALID_HANDLE ;
}


#ifdef DOCUMENTATION
@Summary ScrShow	Show an Window ( Unhide )
 #include <wk/scr.h>

 int ScrShow( hd );
 int hd;	Handle des Windows
@Description
 Zeigt ein vertecktes Window wieder an.
 Sichtbar erst nach dem nchst ScrUpdate()
@Return Value
 ErrorCode ( InvalidHandle )
@See Also
 ScrHide
#endif /*DOCUMENTATION*/

int ScrShow( int handle )
{
    control_t *win ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    if( win->mode & SCR_MODE_HIDDEN ) {
		win->mode &= ~SCR_MODE_HIDDEN;
		EnterCritical() ;
		rebuildPending = 1 ;
		LeaveCritical() ;
	    }
	    return scrErrno = 0;
	}
     return scrErrno = ESCR_INVALID_HANDLE ;
}

#ifdef DOCUMENTATION
@Summary ScrIsVisible
 #include <wk/scr.h>

 int ScrIsVisible( handle );
 int handle;	    Handle des Windows oder 0 fr PS
@Description
 Bestimmt ob das Window Visible oder hidden ist
@Return Value
 False = Window ist nicht sichtbar
 True  = Window ist sichtbar
@Notes
 Hier gibt es keine Fehlermeldung; stattdesssen wird False
 zurureckgegeben.
#endif /*DOCUMENTATION*/

int ScrIsVisible( int handle )
{
    control_t *win ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag )
	    if( !(win->mode & SCR_MODE_HIDDEN) )
		return 1;
    return 0;
}

#ifdef DOCUMENTATION
@Summary ScrHasFocus
 #include <wk/scr.h>

 int ScrHasFocus( handle );
 int handle;	    Handle des Windows
@Description
 Bestimmt ob das Window den Focus hat
@Return Value
 True  = Window hat den Focus
@Notes
 Hier gibt es keine Fehlermeldung; stattdesssen wird False
 zurureckgegeben.
#endif /*DOCUMENTATION*/

int ScrHasFocus( int handle )
{
    /* eigentlich muesste auch was mit crsWindowHd gemacht werden .. */
    /* die implementation ist da nicht sehr korrekt */
    return lastWindow[0] == handle;
}


#ifdef DOCUMENTATION
@Summary ScrUpdate	Update / Focus on Window
 #include <wk/scr.h>

 int ScrUpdate( handle );
 int handle;	    Handle des Windows oder 0 fr PS
@Description
 Diese Funktion dient der Anzeige eines Windows auf dem
 PS ( Presentation Space = Bildschirm ). Erst nach dem Aufruf
 dieser Funktion kann davon ausgeganen werden, dass der User
 die bis dato durchgefhrten Ausgaben sehen kann. Weiterhin ist
 es mit dieser Funktion mglich zu bestimmen, welches dass
 oberste Window ist ( die Windows knnen sich gegenseitig verdecken ).
 Wird die Funktion mit dem Handle eines Windows aufgerufen, so wird
 dieses Window zuoberst angezeigt. Soll die Stapelreihenfolge der
 Windows nicht gendert werden, so ist der dafr reservierte
 Handle 0 zu benutzen ( Update PS ) oder der entsprechende negative
 Wert des handles zu benutzen.
@Return Value
 Error Code
#endif /*DOCUMENTATION*/


int ScrUpdate( int handle )
{
    control_t *win, *ps ;
    int i , j , x, y , frame , err , ready , nxtWin=0 , rebuild=0 , noFocus;
    int doCopy, doShadows, doDriver , doFocus;
    cell_t aCell ;

    if( noFocus = handle <= 0 )
	handle = -handle;

    ready = err = 0 ;
    frame = 0 ; /* we must have a default value ! */
    doFocus = 0;
    if( handle >= 0 && handle < MAX_HANDLES ) {
	EnterCritical() ;
	rebuild = rebuildPending ;
	rebuildPending = 0 ;
	LeaveCritical() ;
	nxtWin = MAX_HANDLES - 1 ;
	if( !handle )
	    rebuild++ ; /* force rebuild when handle is 0 */
	else if( lastWindow[0] != handle && !noFocus ) {/*Focus, if ! on top*/
	    EnterCritical() ;
	    /* first: remove window with this handle from list */
	    for(i=0; i < MAX_HANDLES; i++ )
		if( lastWindow[i] == handle ) { /* handle found */
		    for(; i < MAX_HANDLES-1; i++ )
			lastWindow[i] = lastWindow[i+1] ;
		    lastWindow[MAX_HANDLES-1] = -1 ;
		    break;
		}
	    /* second: shift list and insert handle as first one */
	    for(i=MAX_HANDLES-1; i > 0 ; i-- )
		lastWindow[i] = lastWindow[i-1] ;
	    lastWindow[0] = handle ; /* this is the top one*/
	    LeaveCritical() ;
	    rebuild++;
	}
    }
    else
	err = ESCR_INVALID_HANDLE ;

    ps = controlTbl ; /* PS addressed by (internal) handle 0 */
    if( rebuild ) {
	ClearWindow( ps ) ; /* start from empty PS */
	ResetInvRect();
	SetInvRect( psLimit.x1, psLimit.y1 );
	SetInvRect( psLimit.x2, psLimit.y2 );
    }
    while( !err && !ready) {
	doCopy = doShadows = doDriver = 0 ;
	if( rebuild ) {
	    for( win = NULL ; nxtWin >= 0; nxtWin--, win=NULL )
		if( lastWindow[nxtWin] != -1 )
		    if( (win=controlTbl+lastWindow[nxtWin])->useFlag )
			break ;
	    if( win ) {
		nxtWin-- ;
		doCopy++ ;
	    }
	    else if( !handle )
		ready = doDriver = 1;
	    else if( (win=controlTbl+handle)->useFlag )  /* now my Handle*/
		ready = doDriver = doShadows = 1;
	    else
		err = ESCR_INVALID_HANDLE;
	}
	else if( (win=controlTbl+handle)->useFlag )
	    ready = doCopy = doShadows = doDriver = 1;
	else
	    err = ESCR_INVALID_HANDLE;

	if( win )
	    if( win->mode & SCR_MODE_HIDDEN )
		doCopy = doShadows = doDriver = 0;

	if( doCopy ) {
	    /**** part 1 : Build PS *****/
	    frame = win->style & (SCR_STYLE_FRAME|SCR_STYLE_DBLFRAME) ? 1:0;
	    if( frame ) {
		aCell.a = win->attr ;
		/** upper frameLine **/
		aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_UL:C_WF_DUL;
		i = (win->y1-1) * ps->sx + win->x1-1;
		ps->data[i] = aCell ;
		aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_HL:C_WF_DHL;
		for( x=0; x < win->sx ; x++ )	{
		    i = (win->y1-1) * ps->sx + win->x1+x;
		    ps->data[i] = aCell ;
		}
		aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_UR : C_WF_DUR;
		i = (win->y1-1) * ps->sx + win->x2+1;
		ps->data[i] = aCell ;
	    }
	    for( y=0; y < win->sy ; y++ ) {
		if( frame ) {
		    /* Left frame character */
		    aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_VL:C_WF_DVL;
		    i = (win->y1+y) * ps->sx + win->x1-1;
		    ps->data[i] = aCell ;
		}
		/* Window contents */
		for( x=0; x < win->sx ; x++ ) {
		    i = (win->y1+y) * ps->sx + win->x1+x;
		    j = y * win->sx + x;
		    ps->data[i] = win->data[j] ;
		}
		if( frame ) {
		    /* Right frame character */
		    aCell.c = win->style & SCR_STYLE_FRAME? C_WF_VL:C_WF_DVL;
		    i = (win->y1+y) * ps->sx + win->x2+1;
		    ps->data[i] = aCell ;
		}
	    }
	    if( frame )  {
		/* lower frameLine */
		aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_LL:C_WF_DLL ;
		i = (win->y2+1) * ps->sx + win->x1-1;
		ps->data[i] = aCell ;
		aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_HL:C_WF_DHL ;
		for( x=0; x < win->sx ; x++ )	{
		    i = (win->y2+1) * ps->sx + win->x1+x;
		    ps->data[i] = aCell ;
		}
		aCell.c = win->style & SCR_STYLE_FRAME ? C_WF_LR:C_WF_DLR;
		i = (win->y2+1) * ps->sx + win->x2+1;
		ps->data[i] = aCell ;
	    }
	}

	if( doShadows ) {
	    /*** part 2: handle shadows ***/
	    /** handle shadow here: **/
	    /** - remove all the old Shadows **/
	    /** - draw this Shadow **/
	    /*
	     * ... to be done here ...
	     */
	}

	if( doDriver ) {
	    /*** part 4: tell Driver to update the PS ***/
	  #if 0
	    if( rebuild )
		DrvUpdate( ps->data, ps->sx, ps->x1, ps->y1, ps->x2, ps->y2);
	    else
		DrvUpdate( ps->data, ps->sx, win->x1-frame, win->y1-frame,
					     win->x2+frame, win->y2+frame);
	  #endif
	    if( psLimit.upd ) {
		DrvUpdate( ps->data, ps->sx, psLimit.ix1, psLimit.iy1,
					     psLimit.ix2, psLimit.iy2);
		ResetInvRect();
	    }
	    if( handle && !noFocus ) {
		if( crsWindowHd == -1 )
		    DrvSetCrs( 1, 0, 0 ) ; /* no cursor: switch off */
		else if( crsWindowHd == handle ) {
		    DrvSetCrs( 0, win->px+win->crsX, win->py+win->crsY ) ;
		    DrvSetCrs( win->crsForm == 1 ? 101 : 102, 0,0 );
		    DrvSetCrs( win->crsActiv ? 2 : 1, 0, 0 );
		}
	    }
	}
    }

    return scrErrno = err ;
}



#ifdef DOCUMENTATION
@Summary ScrCursor	Cursoroperationen auf Window
 #include <wk/scr.h>

 int ScrCursor( hd, mode, x, y );
 int handle;	    Handle des Windows
 int mode;	    Art der Operation ( SCR_CRS_... )
 int x,y;	    Paramter je nach Art der Operation
@Description
 Fhrt verschieden Operation auf dem Cursor des
 angegebenen Windows aus:
 SCR_CRS_SET	: Setzt Cursor auf Position x,y
 SCR_CRS_SETREL : ndert Cursorposition um x,y
 SCR_CRS_SHOW	: Macht den Cursor sichtbar
 SCR_CRS_HIDE	: Macht den Cursor unsichtbar
 SCR_CRS_FORM_A : Setzt die Cursorform A
 SCR_CRS_FORM_B : Setzt die Cursorform B
 SCR_MCRS_SHOW	: Mouse Cursor sichtbar   (wirkt sofort, nicht kummulativ
					   auch mit handle 0 = PS)
 SCR_MCRS_HIDE	: Mouse Cursor unsichtbar (wirkt sofort, nicht kummulativ
					   auch mit handle 0 = PS)
@Return Value
 ErrorCode  ( InvalidHandle, InvalidValue )
@See Also
 ScrSetAttr
#endif /*DOCUMENTATION*/


int ScrCursor( int handle, int mode, int x, int y )
{
    control_t *win ;

    if( handle > 0 && handle < MAX_HANDLES ||
	( !handle && ( mode == SCR_MCRS_SHOW || mode == SCR_MCRS_HIDE ) ) )
	if( (win=controlTbl+handle)->useFlag ) {
	    switch( mode ) {
	      case SCR_CRS_SET:
		win->crsX = 0 ;
		win->crsY = 0 ; /* fall thru*/
	      case SCR_CRS_SETREL:
		win->crsX += x ;
		win->crsY += y ;
		if( win->crsX < 0 )
		    win->crsX = 0 ;
		if( win->crsX >= win->sx )
		    win->crsX = win->sx-1 ;
		if( win->crsY < 0 )
		    win->crsY = 0 ;
		if( win->crsY >= win->sy )
		    win->crsY = win->sy-1 ;
		break ;

	      case SCR_CRS_SHOW   :
		win->crsActiv = 1 ;
		crsWindowHd = handle;
		break ;

	      case SCR_CRS_HIDE   :
		win->crsActiv = 0 ;
		if( crsWindowHd == handle )
		    crsWindowHd = -1;
		break ;

	      case SCR_CRS_FORM_A :
		win->crsForm = 1 ;
		break ;
	      case SCR_CRS_FORM_B :
		win->crsForm = 2 ;
		break ;
	      case SCR_MCRS_SHOW :
		if( msCrsHidden ) {
		    MouseShow();
		    msCrsHidden = 0;
		}
		break;
	      case SCR_MCRS_HIDE :
		if( !msCrsHidden ) {
		    MouseHide();
		    msCrsHidden++;
		}
		break;

	      case SCR_CRS_UPDATE:
		if( crsWindowHd == -1 )
		    DrvSetCrs( 1, 0, 0 ) ; /* no cursor: switch off */
		else if( crsWindowHd == handle ) {
		    DrvSetCrs( 0, win->px+win->crsX, win->py+win->crsY ) ;
		    DrvSetCrs( win->crsForm == 1 ? 101 : 102, 0,0 );
		    DrvSetCrs( win->crsActiv ? 2 : 1, 0, 0 );
		}
		break;

	      default:
		return scrErrno = ESCR_INVALID_VALUE ;
	    }
	    return scrErrno = 0 ;
	}
     return scrErrno = ESCR_INVALID_HANDLE ;
}


#ifdef DOCUMENTATION
@Summary ScrSetAttr	Setzen des WindowAttributs
 #include <wk/scr.h>

 int ScrSetAttr( hd, a );
 int hd;	Handle des Windows
 byte a;	neues Attribut
@Description
 Vernder das Attribut des Windows fr alle folgenden Operationen.
 Dies gilt nicht fr Rahmenoperationen und ScrWriteCell().
@Return Value
 ErrorCode ( InvalidHandle )
#endif /*DOCUMENTATION*/

int ScrSetAttr( int handle, byte attr )
{
    control_t *win ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    win->attr = attr ;
	    return scrErrno = 0 ;
	}
    return scrErrno = ESCR_INVALID_HANDLE ;
}


#ifdef DOCUMENTATION
@Summary ScrPrintF	    printf() fr Window
 #include <wk/scr.h>

 int ScrPrintF( hd, s, ... );
 int hd;	Handle des Windows
 const char *s; Format string
@Description
 Ein String wird auf dem angegebenen Window ausgegeben.
 Die Funktion verhlt sich wie printf(), allerdings mit der
 einschrnkung, dass der interne Buffer auf ^b2000 Bytes^n
 begrenzt ist und das Steuerzeichen '\f' den Bildschirm lscht.
 Alle anderen Zeichen verhalten sich wie gewohnt.
@Return Value
 ErrorCode ( InvalidHandle, OutOfMemory )
#endif /*DOCUMENTATION*/


int ScrPrintF( int handle, const char *s, ... )
{
    control_t *win ;
    char *buf, *p ;
    va_list argPtr ;
    int i ;

    if( handle > 0 && handle < MAX_HANDLES )
	if( (win=controlTbl+handle)->useFlag ) {
	    if( buf = malloc( 2000 ) ) {
		va_start( argPtr, s );
		vsprintf(buf, s, argPtr ) ;
		va_end( argPtr );
		p = buf ;
		i = win->crsY * win->sx + win->crsX ;
		SetInvRect( win->px+win->crsX, win->py+win->crsY );
		for( p=buf; *p ; p++, i++ ) {
		    if( i >= win->dSize )
			i = 0 ;
		    win->data[i].c = *p ;
		    win->data[i].a = win->attr ;
		}
		win->crsX = i % win->sx ;
		win->crsY = i / win->sx ;
		SetInvRect( win->px+win->crsX, win->py+win->crsY );
		free(buf) ;
		return scrErrno = 0 ;
	    }
	    else
		return scrErrno = ESCR_OUT_OF_MEMORY ;
	}
    return scrErrno = ESCR_INVALID_HANDLE ;
}


#ifdef DOCUMENTATION
@Summary ScrMsStat
 #include <wk/scr.h>

 int ScrMsStat( int *hd, int *px, int *py );
@Description
 Ermittelt den Mouse Status innerhalb des Windows. Ist die Mouse Position
 nicht innerhalb des Windows, so wird die MousePosition auf den PS bezogen
 zurckgegeben und deswegen in hd 0 fr PS zurckgegeben.
 Soll die Position nur auf den PS bezogen ermittelt werden, so ist fr hd
 NULL anzugeben. px und py  knnen falls das Window einen Rahmen hat auch auf
 diesen zeigen; d.h. z.B. eine Position -1/-1 zeigt an, da der MouseCursor
 sich in der oberen linken Ecke des Rahmens befindet.
@Return Value
 Bitweise: siehe mouse.h
@See Also
 ScrCursor
#endif /*DOCUMENTATION*/

int ScrMsStat( int *whd, int *px, int *py )
{
    int btn, i, fnd, mx, my;
    control_t *win=NULL; /* controlled by fnd */


    btn = MouseGet( &mx, &my );
    if( whd ) {
	/* look thru all windows from top to down */
	for(fnd=i=0; i < MAX_HANDLES && !fnd ; i++ )
	    if( lastWindow[i] != -1 ) {
		win = controlTbl + lastWindow[i];
		if( win->style & (SCR_STYLE_FRAME|SCR_STYLE_DBLFRAME) ) {
		    if( mx >= (win->x1-1) && mx <= (win->x2+1) &&
			my >= (win->y1-1) && my <= (win->y2+1)	 )
			fnd++;
		}
		else {
		    if( mx >= win->x1 && mx <= win->x2 &&
			my >= win->y1 && my <= win->y2	 )
			fnd++;
		}
	    }
	if( fnd ) {
	    --i;
	    *whd = lastWindow[i];
	    *px = (int)mx - win->px;
	    *py = (int)my - win->py;
	}
	else {
	    *whd = 0;	/* not in a window, tell PS-Coordinates */
	    *px = mx;
	    *py = my;
	}
    }
    else {
	*px = mx;   /* user wants PS-Coordinates */
	*py = my;
    }

    return btn ;
}



#ifdef DOCUMENTATION
@Summary ScrDrvCtrl
 #include <wk/scr.h>

 int ScrDrvCtrl( int dummy, int opcode, int p1, int p2 );
@Description
@Return Value
#endif /*DOCUMENTATION*/


int ScrControl( int dummy, int opcode, int p1, int p2 )
{
    int i;

    switch( opcode ) {
      case SCR_CTRL_REPAINT:
	DrvDevCtrl(SCRDRV_SYNCMIRROR, NULL);
	rebuildPending = 1;
	break;

      case SCR_CTRL_ORGSCR:
	DrvDevCtrl(SCRDRV_RESTORGSCR, NULL);
	break;

      case SCR_CTRL_SAVEORGSCR:
	DrvDevCtrl(SCRDRV_SAVEORGSCR, NULL);
	break;

      case SCR_CTRL_CLOSEALL:
	for(i=1; i < MAX_HANDLES ; i++ )
	    ScrClose(i);
	break;

      case SCR_CTRL_DEINIT:
	if( initOkay ) {
	    ScrChangeRedirector(0, NULL );
	    free( controlTbl[0].data );
	    initOkay = 0;
	}
	break;
    }
    return 0;
}



#ifdef DOCUMENTATION
@Summary ScrBeep
 #include <wk/scr.h>

 void ScrBeep(void);
@Description
 Sound a beep
#endif /*DOCUMENTATION*/

void ScrBeep()
{
    DrvDevCtrl(SCRDRV_SOUNDBEEP, NULL);
}



#ifdef DOCUMENTATION
@Summary ScrErrorText	    String mit einem Text zurckgeben
 #include <wk/scr.h>

 const char *ScrErrorText( nr );
 int nr;	ErrorCode
@Description
 Mapped einen ErrorCode auf einen String um.
@Return Value
 Pointer to String
#endif /*DOCUMENTATION*/

const char *ScrErrorText(int n)
{
    const char *p ;
    static char buffer[30] ;

    if( n == -1 )
	n = scrErrno ;
    switch( n ) {
      case 0:			   p = "No Error"; break;
      case ESCR_GENERAL_ERROR	 : p = "General error" ; break;
      case ESCR_INVALID_HANDLE	 : p = "Invalid handle" ; break;
      case ESCR_INVALID_VALUE	 : p = "Invalid value" ; break;
      case ESCR_OUT_OF_RANGE	 : p = "Out of range" ; break;
      case ESCR_OUT_OF_HANDLES	 : p = "Out of handles" ; break;
      case ESCR_OUT_OF_MEMORY	 : p = "Out of memory" ; break;
      default: sprintf(buffer, "ScrError %d", n ) ; p = buffer ; break ;
    }
    return p ;
}




/* [internal]
 * Funktion zur Kommunikation mit einem ScreenDriver
 * hier: neuen Driver anmelden
 */

int ScrRegisterDriver( int *retDrvHandle,
		       void (*D1)(cell_t*,int,int,int,int,int), /* Update*/
		       void (*D2)(scrPhyInf_t *),   /* Get Phy Info*/
		       void (*D3)( int, int, int),  /* Set Cursor*/
		       void (*D4)( int, scrPhyDevCtrl_t *) /* Device Control*/
		     )
{
    EnterCritical();
    DrvUpdate  = D1 ;
    DrvGetInfo = D2 ;
    DrvSetCrs  = D3 ;
    DrvDevCtrl = D4 ;
    *retDrvHandle = 1107 ; /* some value*/
    LeaveCritical();
    return 0 ;
}


/* [internal]
 * Funktion zur Kommunikation mit einem ScreenDriver+
 * hier: Driver abmelden
 */

int ScrDeRegisterDriver( int driverHandle )
{
    if( driverHandle == 1107 ) {
	EnterCritical();
	DrvUpdate  = DummyDrvUpdate  ;
	DrvGetInfo = DummyDrvGetInfo ;
	DrvSetCrs  = DummyDrvSetCrs  ;
	DrvDevCtrl = DummyDrvDevCtrl ;
	LeaveCritical();
	return 0 ;
    }
    else
	return ESCR_INVALID_HANDLE ;

}




/* [internal to Screen Drivers]
 * Read the Attribute Table:
 * level: nur 1 definiert
 */

int ScrGetAttrTbl( int level, const char *key, byte *table, ushort tableSize )
{
    char  buf[3] ;
    ushort n ;
    byte  x;
    static char *attrTbl[] = {	"07",  /*  0: Grau auf Schwarz */
				"0F",  /*  1: Wei auf Schwarz */
				"17",  /*  2: Grau auf Blau */
				"1F",  /*  3: Wei auf Blau */
				"1E",  /*  4: Gelb auf Blau */
				"37",  /*  5: Grau auf Trkis */
				"3F",  /*  6: Wei auf Trkis */
				"3E",  /*  7: Gelb auf Trkis */
				"47",  /*  8: Grau auf Rot */
				"4F",  /*  9: Wei auf Rot */
				"4E",  /* 10: Gelb auf Rot */
				"70",  /* 11: Schwarz auf Grau */
				"F0",  /* 12: Schwarz auf Wei */
				"10",  /* 13: Schwarz auf Blau */
				"30",  /* 14: Schwarz auf Trkis */
				"40",  /* 15: Schwarz auf Rot */
				"0C",  /* 16: Rot auf Schwarz */
				"2F",  /* 17: Wei auf Grn */
				"20",  /* 18: Schwarz auf Grn */
				"0A"   /* 19: Grn auf Schwarz */
					   } ;


    if( level != 1 )
	return ESCR_INVALID_VALUE ;

    for(n=0; n < tableSize; n++ ) {
	strncpy( buf, attrTbl[n], DIM(buf)-1 );
	x = ((isdigit(buf[0]) ? (buf[0]-'0'):(buf[0]-'A'+10)) << 4) & 0xf0;
	x |= (isdigit(buf[1]) ? (buf[1]-'0'):(buf[1]-'A'+10)) & 0x0f;
	table[n] = x ;
    }

    return n < 16 ? ESCR_INVALID_ATABLE : 0 ;
}



/******************************
 *** Local Functions **********
 ******************************/

static void ClearWindow( control_t *win )
{
    cell_t aCell ;
    int i;

  #ifdef UNIX
    aCell.c = (char)' ' ;
  #else
    aCell.c = (char)'' ;
  #endif
    aCell.a = 0 ; /* Standard*/
    for( i=0; i < win->dSize; i++ )
	win->data[i] = aCell ;
}



static void SetInvRect( int x, int y )
{
    if( x < psLimit.ix1 )
	psLimit.ix1 = x ;
    if( x > psLimit.ix2 )
	psLimit.ix2 = x ;
    if( y < psLimit.iy1 )
	psLimit.iy1 = y ;
    if( y > psLimit.iy2 )
	psLimit.iy2 = y ;
    psLimit.upd=1;
}

static void ResetInvRect()
{
    psLimit.ix1 = psLimit.x2+1;
    psLimit.ix2 = psLimit.x1-1;
    psLimit.iy1 = psLimit.y2+1;
    psLimit.iy2 = psLimit.y1-1;
    psLimit.upd = 0;
}

/******************************
 *** Dummy Functions **********
 ******************************/
#if __MSC__
#pragma check_stack(off)
#endif

static void DummyDrvUpdate(cell_t*k,int l,int a,int b,int c,int d)
{

}

static void DummyDrvGetInfo(scrPhyInf_t *a)
{
}

static void DummyDrvSetCrs( int a, int b, int c)
{
}

static void DummyDrvDevCtrl( int a, scrPhyDevCtrl_t *b)
{
}


/*********************** bottom of file ***********************************/
