/* [scrdrv1.c wk t 22.4.91] Screen Driver using Curses
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkCUA.
 *
 *  WkCUA is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkCUA is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#if 0
  #define MINICURSES 1
#endif
#ifdef __linux__
  #include <ncurses.h>
#else
  #include <curses.h>
#endif
#define INCL_CTRL_KEYCODES 1
#define INCL_ALT_KEYCODES  1
#include <wk/lib.h>
#include <wk/event.h>

#include "scr_sy.h"

/****** constants **********/

#define AMASK 15
static int attrTbl[16] = {
/* nr and default color --- Dialog default usage */
/*-----------------------------------------------*/
/*  0: Grau auf Schwarz 			    */ A_DIM,
/*  1: Wei auf Schwarz --- Selected		    */ A_BLINK,
/*  2: Grau auf Blau	--- Standard, Button	    */ 0,
/*  3: Wei auf Blau	--- Window, Field, ButtonOn */ A_BOLD,
/*  4: Gelb auf Blau	--- ButtonAccChar	    */ A_UNDERLINE,
/*  5: Grau auf Trkis				    */ 0,
/*  6: Wei auf Trkis	--- FieldFocus, ButtonFocus */ A_REVERSE,
/*  7: Gelb auf Trkis	--- AccChar		    */ A_UNDERLINE,
/*  8: Grau auf Rot				    */ 0,
/*  9: Wei auf Rot				    */ 0,
/* 10: Gelb auf Rot				    */ 0,
/* 11: Schwarz auf Grau 			    */ 0,
/* 12: Schwarz auf Wei 			    */ 0,
/* 13: Schwarz auf Blau 			    */ 0,
/* 14: Schwarz auf Trkis			    */ 0,
/* 15: Schwarz auf Rot				    */ 0
};

/******* types *************/

/******** globals **********/

static int driverHandle;	/* Handle from Register Driver*/
static struct {
	int sx, sy, sxMul2;
	int x2, y2 ;
	int crsX, crsY ;
	int crsForm ;
	int crsActiv ;
	int size ;		/*total size*/
    } screen ;			/* Screen Parameters*/
static cell_t *mirror ;


/********** prototypes *********/
static void Cleanup( void *dummy );

static void DrvUpdate(cell_t*,int,int,int,int,int);
static void DrvGetInfo(scrPhyInf_t *);
static void DrvSetCrs( int, int, int);
static void DrvDevCtrl( int, scrPhyDevCtrl_t *);

static int KbdFnc(void);

/*********** defined functions ********/
/*********** functions ****************/

#ifdef DOCUMENTATION
@Summary ScrInitDriverCurses
 #include <wk/scr.h>

 int ScrInitDriverCurses( level, mode );
 int level;	    0 = Init, 1= deinit
 int mode ;	    reserviert; muss 0 sein
@Description
 Initialisiert den Curses Screen Driver.
 Vorher muss ScrInitialize() aufgerufen worden sein.
@Return Value
 0 on success
@See Also
 ScrInitialize
@Example
 ..
 if( err = ScrInitialize(0,0,0,0,0,0,0) )
    error(4,0,"Can't init Screen: %s", ScrErrorText(err));
 if( err = ScrInitDriverCurses(0,0) )
    error(4,0,"Can't init Curses: %s", ScrErrorText(err));
 ...
#endif /*DOCUMENTATION*/


int ScrInitDriverCurses(int level, int mode )
{
    static int isInitialized = 0;
    static int cleanupOkay = 0;
    int err, cols, rows ;

    if( !cleanupOkay ) {
	AddCleanUp( Cleanup, NULL );
	cleanupOkay = 1;
    }
    err = 0 ;
    if( mode ) /* nothing yet defined*/
	BUG();

    if( level ) { /* deinit */
	if( isInitialized ) {
	    isInitialized = 0;
	    FREE( mirror );
	    ScrDeRegisterDriver( driverHandle );
	    move(LINES-1,0);
	  #ifndef MINICURSES
	    clrtoeol();
	  #endif
	    refresh();
	    endwin();
	}
	return 0;
    }

    /** init curses */
    initscr();
    isInitialized = 1;
  /*raw();   */
    cbreak();
    nonl();
    noecho();
    idlok(stdscr, TRUE);
  #ifndef MINICURSES
    keypad(stdscr, TRUE);
  #endif


    /**** Setup Screen Parameters ****/
    rows = LINES;
    cols = COLS;

    screen.sx = cols ;
    screen.sxMul2 = cols*2 ;
    screen.sy = rows ;
    screen.x2 = cols -1 ;
    screen.y2 = rows -1 ;
    screen.size = screen.sx * screen.sy ;
    if( !(mirror = calloc( screen.size, sizeof *mirror )) )
	{ err = ESCR_OUT_OF_MEMORY; goto retlab ; }
    /** invalidate mirror ***/
    memset( mirror, 255, screen.size * sizeof *mirror );


    /**** Register this Driver ****/
    if( err = ScrRegisterDriver( &driverHandle, DrvUpdate,
				 DrvGetInfo, DrvSetCrs, DrvDevCtrl ) )
	goto retlab ;

    /**** Init Mouse Functions ****/
    DrvSetCrs( 101, 0, 0 ); /* Set Cursor Form to 1*/
    DrvSetCrs( 1, 0, 0 ) ;  /* hide Cursor*/

    EventRegisterKbd( KbdFnc, 0); /* this driver is not able to poll the kbd */

  retlab:
    return scrErrno = err ;
}


static void Cleanup( void *dummy )
{
    ScrInitDriverCurses(1,0);
}



/*
 * copy the Presentation Space to the Physical Screen
 * Abbildung zur Zeit: 1:1 ; rechts und unten wird abgeschnitten
 */

static void DrvUpdate( cell_t *ps, int psXSize,
		       int psX1, int psY1, int psX2, int psY2 )
{
    register int x ;
    int y , py, px , sy, sx ;
    cell_t  aCell ;

    /*** z.Z. nur rechts und unten clippen ***/
    px = psX1 ;
    py = psY1 ;
    sx = screen.sx ;
    sy = screen.sy ;
    if( px + psX2 < sx )
	sx = px + psX2 + 1 ;
    if( py + psY2 < sy )
	sy = py + psY2 + 1 ;

    for( y=py ; y < sy ; y++ ) {
	for( x=px ; x < sx ; x++ ) {
	    aCell = ps[y*psXSize+x] ;
	    if( memcmp( &aCell, &mirror[y*screen.sx+x], sizeof aCell) ) {
		mirror[y*screen.sx+x] = aCell ;
		attrset( attrTbl[aCell.a & AMASK ] );
	      #ifdef MINICURSES
		move( y, x );
		addch( aCell.c );
	      #elif OS2
		mvaddrawch( y, x, aCell.c );
	      #else
		mvaddch( y, x, aCell.c );
	      #endif
	    }
	}
    }
    if( screen.crsActiv )
	move( screen.crsY, screen.crsX );
    else
	move( 0, 0 );
    refresh();
}




static void DrvGetInfo( scrPhyInf_t *info )
{
    memset( info, 0, sizeof *info );
    info->sx = screen.sx ;
    info->sy = screen.sy ;
    info->col = 0 ;
}



/*
 * Den Cursor Setzen bzw in der Form ndern.
 * X und Y beziehen sich auf den Presentation Space.
 * level 0 - Cursor Position setzen
 * level 1 - Cursor entfernen	(x=y=0)
 * level 2 - Cursor aktivieren	(x=y=0)
 * level 10- Mouse Cursor setzen ( nicht impl. )
 * level 11- Mouse Cursor entfernen (x=y=0) ( nicht impl. )
 * level 12- Mouse Cursor aktivieren (x=y=0) ( nicht impl. )
 * level 101 - Cursor auf Form 1 setzen (x=y=0)
 * level 102 - Cursor auf form 2 setzen (x=y=0)
 */

static void DrvSetCrs( int level, int x, int y )
{
    switch(level) {
      case 0:	/* Set Position*/
	if( screen.crsY != y || screen.crsX != x ) {
	   screen.crsY=y;
	   screen.crsX=x ;
	   if( screen.crsActiv )
	       move( screen.crsY, screen.crsX );
	   else
	       move( 0, 0 );
	}
	refresh();
	break ;

      case 1:	/* Cursor aus*/
	screen.crsActiv = 0 ;
      #if 0
	cursoff();
      #endif
	move( 0, 0 );
	refresh();
	break;

      case 2:	/* Cursor ein*/
	screen.crsActiv = 1 ;
      #if 0
	curson();
      #endif
	move( screen.crsY, screen.crsX );
	refresh();
	break;

      case 10: /* Mouse setzen*/
	break ;

      case 11:	/* MouseCursor aus*/
	break;

      case 12:	/* MouseCursor ein*/
	break;

      case 101: /* CursorForm auf 1 setzen*/
	screen.crsForm = 1 ;
	break ;

      case 102: /* CursorForm auf 2 setzen*/
	screen.crsForm = 2 ;
	break ;
    }
}


/*
 * Device Control Function Dispatcher
 */

static void DrvDevCtrl( int opCode, scrPhyDevCtrl_t *ctrl )
{
    static int sema = 0 ;

    sema-- ;
    switch( opCode ) {
      case SCRDRV_SYNCMIRROR :
	memset( mirror, 255, screen.size * sizeof *mirror );
	break ;

      case SCRDRV_RESTORGSCR :
	break;

      case SCRDRV_SAVEORGSCR :
	break;

      case SCRDRV_SOUNDBEEP :
	beep();
	break;

      case SCRDRV_SYNCAREA :
      default:
	if( !sema )
	    Bug("invalid Opcode %d passed to DrvDevCtrl", opCode ) ;
    }
    sema++ ;
}



static int KbdFnc()
{
    int c, org, wait, meta;

    meta = 0;
    do {
	c = wait = 0;
	if( (c = org = getch()) == EOF )
	    c = 0;
	else if( meta ) {
	    switch(c) {
	      case KEY_BACKSPACE : c = K_ARUBOUT   ; break;
	      case KEY_F(1)	 : c = K_AF1	   ; break;
	      case KEY_F(2)	 : c = K_AF2	   ; break;
	      case KEY_F(3)	 : c = K_AF3	   ; break;
	      case KEY_F(4)	 : c = K_AF4	   ; break;
	      case KEY_F(5)	 : c = K_AF5	   ; break;
	      case KEY_F(6)	 : c = K_AF6	   ; break;
	      case KEY_F(7)	 : c = K_AF7	   ; break;
	      case KEY_F(8)	 : c = K_AF8	   ; break;
	      case KEY_F(9)	 : c = K_AF9	   ; break;
	      case KEY_F(10)	 : c = K_AF10	   ; break;
	      case KEY_F(11)	 : c = K_AF11	   ; break;
	      case KEY_F(12)	 : c = K_AF12	   ; break;
	      case 'A' & ~0x40   : c = K_CHOME     ; break;
	      case 'E' & ~0x40   : c = K_CEND      ; break;
	      case 'H' & ~0x40   : c = K_CLEFT     ; break;
	      case 'J' & ~0x40   : c = K_CRIGHT    ; break;
	      case 'K' & ~0x40   : c = K_CUP       ; break;
	      case 'L' & ~0x40   : c = K_CDOWN     ; break;
	      case '[' & ~0x40   : c = K_ESCAPE    ; break;
	      case 'a': case 'A' : c = K_AA        ; break;
	      case 'b': case 'B' : c = K_AB        ; break;
	      case 'c': case 'C' : c = K_AC        ; break;
	      case 'd': case 'D' : c = K_AD        ; break;
	      case 'e': case 'E' : c = K_AE        ; break;
	      case 'f': case 'F' : c = K_AF        ; break;
	      case 'g': case 'G' : c = K_AG        ; break;
	      case 'h': case 'H' : c = K_AH        ; break;
	      case 'i': case 'I' : c = K_AI        ; break;
	      case 'j': case 'J' : c = K_AJ        ; break;
	      case 'k': case 'K' : c = K_AK        ; break;
	      case 'l': case 'L' : c = K_AL        ; break;
	      case 'm': case 'M' : c = K_AM        ; break;
	      case 'n': case 'N' : c = K_AN        ; break;
	      case 'o': case 'O' : c = K_AO        ; break;
	      case 'p': case 'P' : c = K_AP        ; break;
	      case 'q': case 'Q' : c = K_AQ        ; break;
	      case 'r': case 'R' : c = K_AR        ; break;
	      case 's': case 'S' : c = K_AS        ; break;
	      case 't': case 'T' : c = K_AT        ; break;
	      case 'u': case 'U' : c = K_AU        ; break;
	      case 'v': case 'V' : c = K_AV        ; break;
	      case 'w': case 'W' : c = K_AW        ; break;
	      case 'x': case 'X' : c = K_AX        ; break;
	      case 'y': case 'Y' : c = K_AY        ; break;
	      case 'z': case 'Z' : c = K_AZ        ; break;
	      case '0':            c = K_A0        ; break;
	      case '1':            c = K_A1        ; break;
	      case '2':            c = K_A2        ; break;
	      case '3':            c = K_A3        ; break;
	      case '4':            c = K_A4        ; break;
	      case '5':            c = K_A5        ; break;
	      case '6':            c = K_A6        ; break;
	      case '7':            c = K_A7        ; break;
	      case '8':            c = K_A8        ; break;
	      case '9':            c = K_A9        ; break;
	      default: c = 0; break;
	    }
	}
	else {
	    switch(c) {
	      case KEY_BREAK	 : c = K_BREAK	   ; break;
	      case KEY_DOWN	 : c = K_DOWN	   ; break;
	      case KEY_UP	 : c = K_UP	   ; break;
	      case KEY_LEFT	 : c = K_LEFT	   ; break;
	      case KEY_RIGHT	 : c = K_RIGHT	   ; break;
	      case KEY_HOME	 : c = K_HOME	   ; break;
	      case KEY_BACKSPACE : c = K_RUBOUT    ; break;
	      case KEY_F(1)	 : c = K_F1	   ; break;
	      case KEY_F(2)	 : c = K_F2	   ; break;
	      case KEY_F(3)	 : c = K_F3	   ; break;
	      case KEY_F(4)	 : c = K_F4	   ; break;
	      case KEY_F(5)	 : c = K_F5	   ; break;
	      case KEY_F(6)	 : c = K_F6	   ; break;
	      case KEY_F(7)	 : c = K_F7	   ; break;
	      case KEY_F(8)	 : c = K_F8	   ; break;
	      case KEY_F(9)	 : c = K_F9	   ; break;
	      case KEY_F(10)	 : c = K_F10	   ; break;
	      case KEY_F(11)	 : c = K_F11	   ; break;
	      case KEY_F(12)	 : c = K_F12	   ; break;
	      case KEY_DL	 : c = K_ADEL	   ; break;
	      case KEY_IL	 : c = K_AINS	   ; break;
	      case KEY_DC	 : c = K_DEL	   ; break;
	      case KEY_IC	 : c = K_INS	   ; break;
	      case KEY_EIC	 : c = K_INS	   ; break;
	      case KEY_CLEAR	 : c = K_CHOME	   ; break;
	      case KEY_EOS	 : c = K_CDEL	   ; break;
	      case KEY_EOL	 : c = K_CEND	   ; break;
	      case KEY_SF	 : c = K_CUP	   ; break;
	      case KEY_SR	 : c = K_CDOWN	   ; break;
	      case KEY_NPAGE	 : c = K_PGDN	   ; break;
	      case KEY_PPAGE	 : c = K_PGUP	   ; break;
	      case KEY_STAB	 : c = K_BACKTAB   ; break;
	      case KEY_CTAB	 : c = 0	   ; break;
	      case KEY_CATAB	 : c = 0	   ; break;
	      case KEY_ENTER	 : c = K_RETURN    ; break;
	      case KEY_SRESET	 : c = 0	   ; break;
	      case KEY_RESET	 : c = 0	   ; break;
	      case KEY_PRINT	 : c = 0	   ; break;
	      case KEY_LL	 : c = 0	   ; break;
	    #if 0
	      case 'A' & ~0x40   : c = K_HOME      ; break;
	      case 'H' & ~0x40   : c = K_LEFT      ; break;
	      case 'J' & ~0x40   : c = K_RIGHT     ; break;
	      case 'K' & ~0x40   : c = K_UP        ; break;
	      case 'L' & ~0x40   : c = K_DOWN      ; break;
	      case 'F' & ~0x40   : c = K_TAB       ; break;
	      case 'B' & ~0x40   : c = K_BACKTAB   ; break;
	      case 'E' & ~0x40   : c = K_END       ; break;
	    #endif
  /* escape*/ case '[' & ~0x40   : meta++; wait++; break;
	      default: break;
	    }
	}
    } while( wait );

 /* fprintf(stderr,"Char=%02x -> %02x \n", org, c ); */
    return c;
}


/**** end of file ***/
