/* [scrdrv0.c wk 22.4.91] Standard Screen Driver
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkCUA.
 *
 *  WkCUA is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkCUA is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Header: /usr/src/master/libs/wkswn/scrdrv0.c,v 1.3 1997/01/07 14:47:08 wk Exp $
 *
 *  Standard Driver for Screen Manager
 *  MSDOS:direct to video memory
 *  OS/2: via VioWrtCellStr() or Curses
 *  UNIX: via Curses
 *
 * Die Zortech MouseSupport-Implementation erlaubt den Aufruf der Mouse-
 * funktionen in jedem Fall, da die Mouse Funktion in jedem fall selber
 * prfen, ob das Mouse Package initialisiert ist.
 * Wir bentigen hier nur: ..Show und ..HideCursor
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#if __ZTC__
#include <sound.h>
#endif
#include <wk/lib.h>
#if MSDOS
#elif OS2
#elif DOS386
#elif DOS16RM
  #include <zpmapi.h>
#else
    #error Operating system must be MSDOS or OS2
#endif

#include <wk/mouse.h>
#include <wk/bios.h>
#if defined(OS2)
    #define INCL_SUB 1
    #include <os2.h>
#else
    #include <dos.h>
#endif

#include "scr_sy.h"

/****** constants **********/
#define MAX_ATTR 20

/******* types *************/
typedef struct {
	unsigned fo : 4 ;   /* foreground */
	unsigned ba : 4 ;   /* background */
	unsigned iv : 1 ;   /* (internal invisible flag) */
    } attr_t ;

/******** globals **********/
#ifdef OS2
    static ushort *outLine = NULL ;
#else
    static volatile ushort _far *screenBase ;
#endif

static int driverHandle;	    /* Handle from Register Driver*/
static int isColor ;		  /* flag for color-mode */
static int mstransX;
static struct {
	int sx, sy, sxMul2;
	int x2, y2 ;
	int crsX, crsY ;
	int crsForm ;
	int crsActiv ;
	int size ;		  /*total size*/
    } screen ;			  /* Screen Parameters*/
static cell_t *mirror ;
static attr_t attrTbl[MAX_ATTR] ;


static struct {
	int videoMode;
	ushort cursorForm;
	int crsX, crsY;
	ushort orgScrSize;	    /* # of elem in orgScr */
      #if OS2
	ushort orgScrRows;
	ushort orgScrCols;
      #endif
	ushort *orgScr; 	    /* area to save org Screen */
    } save ;			    /* save original screen Environment*/

/********** prototypes *********/
static void DrvUpdate(cell_t*,int,int,int,int,int);
static void DrvGetInfo(scrPhyInf_t *);
static void DrvSetCrs( int, int, int);
static void DrvDevCtrl( int, scrPhyDevCtrl_t *);

static void SaveScreen(void);
static void RestScreen(void);

/*********** defined functions ********/
/*********** functions ****************/

#ifdef DOCUMENTATION
@Summary ScrInitDriverStd
 #include <wk/scr.h>

 int ScrInitDriverStd( level, mode );
 int level;	    0 = Init, 1= deinit
 int mode ;	    reserviert; muss 0 sein
@Description
 Initialisiert den Standard Screen Driver.
 Vorher muss ScrInitialize() aufgerufen worden sein.
@Return Value
 0 on success
@See Also
 ScrInitialize
@Example
 ..
 if( err = ScrInitialize(0,0,0,0,0,0,0) )
    error(4,0,"Can't init Screen: %s", ScrErrorText(err));
 if( err = ScrInitDriverStd(0,0) )
    error(4,0,"Can't init ScreenDriver: %s", ScrErrorText(err));
 ...
#endif /*DOCUMENTATION*/


int ScrInitDriverStd(int level, int mode )
{
    int err, cols, rows, i ;
    byte *aTable ;

    aTable= NULL ;
    err = 0 ;
    save.videoMode = -1 ;
    if( mode ) /* nothing yet defined*/
	BUG();

    if( level ) { /* deinit */
	FREE( mirror );
      #ifdef OS2
	FREE(outLine);
      #endif
	FREE( save.orgScr );
	ScrDeRegisterDriver( driverHandle );
	return 0;
    }

    isColor = 1 ; /* default*/
    switch( save.videoMode = WKBiosGetVideoMode( &cols, &rows ) ) {
      case 0x07: isColor = 0; save.videoMode = -1;break;
      case 0x02: case 0x03: case 0x23: case 0x4D: case 0x17:
      case 0x4F: case 0x57: case 0x19: case 0x1A: case 0x24:
      case 0x45: case 0x32: case 0x2A: case 0x4E: case 0x40:
      case 0x18: case 0x33: case 0x22: case 0x44: case 0x43:
      case 0x26: case 0x5A: case 0x34:	save.videoMode = -1; break;
      default:
	rows = 25;
	WKBiosSetVideoMode( 3 ) ; /* change Vidomode to Standard*/
	break ;
    }

    /**** Setup Screen Parameters ****/
    mstransX = cols == 40 ? 16 : 8 ;
    screen.sx = cols ;
    screen.sxMul2 = cols*2 ;
    screen.sy = rows ;
    screen.x2 = cols -1 ;
    screen.y2 = rows -1 ;
    screen.size = screen.sx * screen.sy ;
    if( !(mirror = calloc( screen.size, sizeof *mirror )) )
	{ err = ESCR_OUT_OF_MEMORY; goto retlab ; }
    /** invalidate mirror ***/
    memset( mirror, 255, screen.size * sizeof *mirror );

  #ifdef OS2
    if( !(outLine = malloc( screen.sxMul2+2 )) )
	{ err = ESCR_OUT_OF_MEMORY; goto retlab ; }
  #else
    screenBase = HW_MakePtrToMemory( isColor ? 0xb8000 : 0xb0000,
							  screen.size*2 );
  #endif

    save.orgScrSize = screen.size;
  #if OS2
    save.orgScrRows = screen.sy;
    save.orgScrCols = screen.sx;
  #endif
    if( !(save.orgScr = malloc( save.orgScrSize * sizeof *save.orgScr)) )
	{ err = ESCR_OUT_OF_MEMORY; goto retlab ; }
    SaveScreen();

    /**** Read the Attribute Table *****/
    if( !(aTable = malloc( MAX_ATTR +1 )) )
	{ err = ESCR_OUT_OF_MEMORY; goto retlab ; }

    if( err=ScrGetAttrTbl(1, isColor? "SCR$EGA":"SCR$MDA",aTable,MAX_ATTR) )
	goto retlab ;
    for( i = 0 ; i < MAX_ATTR ; i++ ) {
	attrTbl[i].fo = aTable[i] & 0x0f ;
	attrTbl[i].ba = aTable[i] >> 4 ;
	attrTbl[i].iv = /*  i == 8  ? 1 : */ 0;
    }

    /**** Register this Driver ****/
    if( err = ScrRegisterDriver( &driverHandle, DrvUpdate,
				 DrvGetInfo, DrvSetCrs, DrvDevCtrl ) )
	goto retlab ;

    /**** Init Mouse Functions ****/
    DrvSetCrs( 101, 0, 0 ); /* Set Cursor Form to 1*/
    DrvSetCrs( 1, 0, 0 ) ;  /* hide Cursor*/


  retlab:
    free( aTable ) ;
    if( err && save.videoMode != -1 )  /* Reset Videomode on error*/
	WKBiosSetVideoMode( save.videoMode ) ;

    return scrErrno = err ;
}





static void SaveScreen()
{
    ushort *mPtr;
  #if OS2
    ushort x,y;
  #else
    volatile ushort _far * sPtr;
    ushort cnt;
  #endif

  #if __ZTC__ && DEBUG && !OS2 /* Tell ZDB to switch to output Screen */
    int dmy1, dmy2;	/* by calling video Bios */
    WKBiosGetVideoMode( &dmy1, &dmy2 );
  #endif
    MouseHide();
    mPtr = save.orgScr;
  #if OS2
    for( y=0 ; y < save.orgScrRows ; y++ ) {
	x = save.orgScrCols * sizeof(unsigned);
	VioReadCellStr( (PCH)mPtr, &x, y, 0, 0 ) ;
	mPtr += save.orgScrCols;
    }
  #else
    sPtr = screenBase;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(mPtr++) = *(sPtr++);
  #endif
    MouseShow();
    save.cursorForm = WKBiosGetCursorForm() ;
    WKBiosGetCursorPos( 0, &save.crsY, &save.crsX );
}


static void RestScreen()
{
    ushort *mPtr;
  #if OS2
    unsigned y;
  #else
    volatile ushort _far * sPtr;
    ushort cnt;
  #endif

  #if __ZTC__ && DEBUG && !OS2	/* Tell ZDB to switch to output Screen */
    int dmy1, dmy2;	/* by calling video Bios */
    WKBiosGetVideoMode( &dmy1, &dmy2 );
  #endif
    MouseHide();
    mPtr = save.orgScr;
  #if OS2
    for( y=0 ; y < save.orgScrRows ; y++ ) {
	VioWrtCellStr( (PCH)mPtr,
		       save.orgScrCols * sizeof(unsigned), y, 0, 0 ) ;
	mPtr += save.orgScrCols;
    }
  #else
    sPtr = screenBase;
    cnt  = save.orgScrSize;
    while( cnt-- )
	*(sPtr++) = *(mPtr++);
  #endif
    MouseShow();
    WKBiosSetCursorPos( 0, save.crsY, save.crsX ) ;
    WKBiosSetCursorForm( save.cursorForm ) ;
}



/*
 * copy the Presentation Space to the Physical Screen
 * Abbildung zur Zeit: 1:1 ; rechts und unten wird abgeschnitten
 */

static void DrvUpdate( cell_t *ps, int psXSize,
		       int psX1, int psY1, int psX2, int psY2 )
{
    register int x ;
    int y , py, px , sy, sx ;
    union {
	struct {
	   unsigned c  : 8 ;
	   unsigned fo : 4 ;
	   unsigned ba : 4 ;
	} a ;
	ushort o ;
    } ibm ;
    int anyupd=0 , hideMs;
    cell_t  aCell ;
    attr_t  attr ;

    /*** z.Z. nur rechts und Unten clippen ***/
    px = psX1 ;
    py = psY1 ;
    sx = screen.sx ;
    sy = screen.sy ;
    if( px + psX2 < sx )
	sx = px + psX2 + 1 ;
    if( py + psY2 < sy )
	sy = py + psY2 + 1 ;


    for( y=py ; y < sy ; y++ ) {
      #ifdef OS2
	if( !anyupd ) {
	    MouseHideCond( psX1, psY1, psX2, psY2);
	    anyupd++ ;
	}
	for( x=px ; x < sx ; x++ ) {
	    aCell = ps[y*psXSize+x] ;
	    attr = attrTbl[aCell.a] ;
	    ibm.a.fo = attr.fo ;
	    ibm.a.ba = attr.ba ;
	    ibm.a.c =  attr.iv ? ' ' : aCell.c ;
	    outLine[x] = ibm.o ;
	}
	VioWrtCellStr( (PCH)outLine+px*sizeof(*outLine),
			(sx-px)*2, y,px, 0 ) ;
      #else /* DOS*/
	if( !anyupd ) {
	  #if __ZTC__ && DEBUG	/* Tell ZDB to switch to output Screen */
	    int dmy1, dmy2;	/* by calling video Bios */
	    WKBiosGetVideoMode( &dmy1, &dmy2 );
	  #endif
	    MouseHideCond( psX1, psY1, psX2, psY2);
	    anyupd++ ;
	}
	for( x=px ; x < sx ; x++ ) {
	    aCell = ps[y*psXSize+x] ;
	    if( memcmp( &aCell, &mirror[y*screen.sx+x], sizeof aCell) ) {
		mirror[y*screen.sx+x] = aCell ;
		attr = attrTbl[aCell.a] ;
		ibm.a.fo = attr.fo ;
		ibm.a.ba = attr.ba ;
		ibm.a.c =  attr.iv ? ' ' : aCell.c ;
		screenBase[y*screen.sx+x] = ibm.o ;
	    }
	}
      #endif
    }
    if( anyupd )
	MouseShow();
}



static void DrvGetInfo( scrPhyInf_t *info )
{
    memset( info, 0, sizeof *info );
    info->sx = screen.sx ;
    info->sy = screen.sy ;
    info->col = isColor ;
}



/*
 * Den Cursor Setzen bzw in der Form ndern.
 * X und Y beziehen sich auf den Presentation Space.
 * level 0 - Cursor Position setzen
 * level 1 - Cursor entfernen	(x=y=0)
 * level 2 - Cursor aktivieren	(x=y=0)
 * level 10- Mouse Cursor setzen ( nicht impl. )
 * level 11- Mouse Cursor entfernen (x=y=0) ( nicht impl. )
 * level 12- Mouse Cursor aktivieren (x=y=0) ( nicht impl. )
 * level 101 - Cursor auf Form 1 setzen (x=y=0)
 * level 102 - Cursor auf form 2 setzen (x=y=0)
 */

static void DrvSetCrs( int level, int x, int y )
{
    switch(level) {
      case 0:	/* Set Position*/
	/* hier fehlt noch das umrechnen*/
	screen.crsY=y;
	screen.crsX=x ;
	if( screen.crsActiv )
	    WKBiosSetCursorPos( 0, screen.crsY, screen.crsX ) ;
	break ;

      case 1:	/* Cursor aus*/
	screen.crsActiv = 0 ;
	WKBiosSetCursorPos( 0, screen.sy, 0) ;	/* Hide Cursor*/
	break;

      case 2:	/* Cursor ein*/
	screen.crsActiv = 1 ;
	WKBiosSetCursorPos( 0, screen.crsY, screen.crsX ) ;
	break;

      case 10: /* Mouse setzen*/
	break ;

      case 11:	/* MouseCursor aus*/
	break;

      case 12:	/* MouseCursor ein*/
	break;

      case 101: /* CursorForm auf 1 setzen*/
	#if OS2
	WKBiosSetCursorForm( 0x0e0f ) ;
	#else
	WKBiosSetCursorForm( isColor? 0x0707:0x0b0c ) ;
	#endif
	screen.crsForm = 1 ;
	break ;

      case 102: /* CursorForm auf 2 setzen*/
	#if OS2
	WKBiosSetCursorForm( 0x070f ) ;
	#else
	WKBiosSetCursorForm( isColor? 0x0307:0x060c ) ;
	#endif
	screen.crsForm = 2 ;
	break ;
    }
}


/*
 * Device Control Function Dispatcher
 */

static void DrvDevCtrl( int opCode, scrPhyDevCtrl_t *ctrl )
{
    static int sema = 0 ;

    sema-- ;
    switch( opCode ) {
      case SCRDRV_SYNCMIRROR :
	memset( mirror, 255, screen.size * sizeof *mirror );
	break ;

      case SCRDRV_RESTORGSCR :
	RestScreen();
	break;

      case SCRDRV_SAVEORGSCR :
	SaveScreen();
	break;

      case SCRDRV_SOUNDBEEP :
      #if __ZTC__
	sound_beep(300);
      #else
	fputc('\a', stderr); fflush(stderr);
      #endif
	break;

      case SCRDRV_SYNCAREA :
      default:
	if( !sema )
	    Bug("invalid Opcode %d passed to DrvDevCtrl", opCode ) ;
    }
    sema++ ;
}




/**** end of file ***/
