/* [rcmgr1.c wk 2.6.91] ResourceManager
 *	Copyright (c) 1988-93 by Werner Koch (dd9jn)
 *  This file is part of WkCUA.
 *
 *  WkCUA is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkCUA is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Header: /usr/src/master/libs/wkswn/rcmgr1.c,v 1.3 1997/01/07 14:47:06 wk Exp $
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/string.h>
#include <wk/file.h>

#include <wk/rcmgr.h>

/******* constants *********/
#define MAX_HANDLES 5
/****** types ******/

typedef struct {
	ushort value;
	ushort offset;
    } sBlkIndex_t;

typedef struct {
	int rcType;	/* 0 = unused slot */
	union {
	    struct {
		ushort	noEntries;
		sBlkIndex_t *index;
		char	*data;
	    } sBlk ;
	    struct {
		ushort entries;  /* number of Strings */
		ushort bufSize;
		char *buffer;	 /* Buffer to hold last retrieved String */
		long fileOff;	 /* ftell() from 1. Offset */
	    } sMsg;
	    struct {
		long size;    /* # of bytes in Resource */
		long nread;   /* # of bytes read */
		long fileOff;  /* fileoffset of data */
	    } plain;
	    struct {
		ushort entries; /* # of entries */
		char *nameTbl;	/* pointer to NameTbl */
		long fileOff;  /* fileoffset of data */
	    } pdmenu;
	    struct {
		ushort items;	/* # of items */
		char *textTbl;	/* pointer to allocated NameTbl */
		long offStart;	/* fileOff of itemTbl */
		long offRover;
		ushort itemsRead;
	    } dlgwin;
	} u ;
    } handleTbl_t;


/***** globals *****/
int rcmInitialized = 0 ; /* true if RCMGR is initialized */

static handleTbl_t handleTbl[MAX_HANDLES];
static FILE *rcStream;

/******* local protos ******/
static void FindResource( const char *name );
static ushort ReadWord(void);
static ulong  ReadLongWord(void);

/******** functions *********/


#ifdef DOCUMENTATION
@Summary RcmInitialize	     Resource-Manager initialization
 #include "wk/rcmgr.h"

 void RcmInitialize( const char *p );
 const char *p;     Name des ExeProgramms mit LoadPath
@Description
 Diese Funktion initialisiert den ResourceManager.
 Dazu wird ihr der Name des ExeProgramms bergeben
 incl. Pfadangabe ( dies ist ab MSDOS 3.0 argv[0] ).
 Intern wird dann daraus der Name der Resourcelibrary
 ermittelt, indem die Extension durch ".EXR" ersetzt wird.
 Die Resourcelibrary wird dann geffnet oder das Programm mit einem
 entsprechenden Fehlerhinweis abgebrochen.
#endif /*DOCUMENTATION*/

void RcmInitialize( const char *exeName )
{
    char *rcFileName;
    int i;

    if( rcmInitialized )
	return ;    /* no double initialization */

    i = strlen( exeName ) ;
    rcFileName = xmalloc( i+4+1 );
    strcpy(rcFileName, exeName);
  #ifdef MSDOSFILESYSTEM
    if( i < 5 ) /* minimum is "?.exe" */
	BUG();
    else if( strcmpl( rcFileName+i-4, ".exe" ) )
	strcat(rcFileName, ".exr");
    else
	rcFileName[i-1] = 'r' ;
  #else
    strcat(rcFileName, ".exr");
  #endif

    if( !(rcStream = fsopen( rcFileName , "b" )) ) {
	Fatal("Can't open resource `%s'\n", rcFileName );
	exit(4);
    }
    for(i=0; i < MAX_HANDLES; i++ )
	handleTbl[i].rcType = 0;    /* set all to unused */
    free( rcFileName );
    rcmInitialized = 1 ;

}




#ifdef DOCUMENTATION
@Summary RcmLoadResource   Eine Resource laden
 #include "wk/rcmgr.h"

 int RcmLoadResource( const char *name, int type )
 const char *name;	Resourcename
 int type		Resourcetype
@Description
 Diese Funktion ldt eine Resource mit namen name und dem angegebenen typ.
 Der Name allein ist der Schssel, der typ dient nur der typprfung.
 Es werden die Verwaltungsinfos der Resource geladen; der Zugriff auf die
 Resource erfolgt durch andere Funktionen, denen dann der von dieser Funktion
 gelieferte Handle bergeben wird.
 Wird die Resource nicht mehr bentigt, so kann Sie durch RcmFreeResource
 wieder freigegeben werden. Wird die Resource nicht gefunden, oder
 ist ein falscher Typ angegeben, so bricht die Funktion das Programm
 mit "internal error" ab.
@Return Value
 Handle
@See Also
 RcmFreeResource
#endif /*DOCUMENTATION*/

int RcmLoadResource( const char *name, int type )
{
    ushort ilen, dlen;
    int handle;
    handleTbl_t *tbl;

    FindResource( name );
    if( ReadWord() != type )
	Bug("ResourceType conflict `%s'", name );

    /* get a new handle */
    for(handle=0; handle < MAX_HANDLES; handle++ )
	if( !handleTbl[handle].rcType )
	    break;
    if( !(handle < MAX_HANDLES) )
	Bug("out of resource handles");
    tbl = handleTbl + handle;
    tbl->rcType = type ;

    if( type == RC_STRBLK ) {
	ilen = ReadWord() ;
	dlen = ReadWord() ;
	tbl->u.sBlk.data  = xmalloc( dlen );/* may be transferred, so alloc first */
	tbl->u.sBlk.index = xmalloc( ilen );
	tbl->u.sBlk.noEntries = ilen / sizeof( *tbl->u.sBlk.index );
	if( fread( tbl->u.sBlk.index, ilen, 1 , rcStream ) != 1 )
	    Bug("Error reading Resource `%s' (index)", name );
	if( fread( tbl->u.sBlk.data , dlen, 1 , rcStream ) != 1 )
	    Bug("Error reading Resource `%s' (data)", name );
    }
    else if( type == RC_STRMSG ) {
	tbl->u.sMsg.entries = ReadWord() ;
	tbl->u.sMsg.bufSize = ReadWord() + 1;
	tbl->u.sMsg.buffer = xmalloc( tbl->u.sMsg.bufSize );
	tbl->u.sMsg.fileOff = ftell( rcStream );
    }
    else if( type == RC_PLAIN ) {
	tbl->u.plain.size = ReadLongWord() ;
	tbl->u.plain.nread = 0L;
	tbl->u.plain.fileOff = ftell( rcStream );
    }
    else if( type == RC_PDMENU ) {
	tbl->u.pdmenu.entries = ReadWord();  /* # of entries */
	dlen = ReadWord(); /* size of nameTbl */
	tbl->u.pdmenu.nameTbl = xmalloc( dlen );
	if( fread( tbl->u.pdmenu.nameTbl, dlen, 1 , rcStream ) != 1 )
	    Bug("Error reading Resource `%s' (NameTbl)", name );
	tbl->u.pdmenu.fileOff = ftell( rcStream );
    }
    else if( type == RC_DLGWIN ) {
	tbl->u.dlgwin.items = ReadWord();  /* # of entries */
	dlen = ReadWord(); /* size of nameTbl */
	tbl->u.dlgwin.textTbl = xmalloc( dlen );
	if( fread( tbl->u.dlgwin.textTbl, dlen, 1 , rcStream ) != 1 )
	    Bug("Error reading Resource `%s' (TextTbl)", name );
	tbl->u.dlgwin.offStart = ftell( rcStream );
	tbl->u.dlgwin.offRover = -1L;
    }
    else
	Bug("Resource '%s' has unkown type %d", name, type );

    return handle;
}


#ifdef DOCUMENTATION
@Summary RcmFreeResource
 #include <wk/rcmgr.h>

 void RcmFreeResource( int handle );
@Description
 Gibt die Resourcen fr den allokierten StringBlock
 wieder frei. Danach ist dieser handle ungltig.
#endif /*DOCUMENTATION*/

void RcmFreeResource( int handle )
{
    handleTbl_t *tbl;

    if( handle >= 0 && handle < MAX_HANDLES ) {
	switch( (tbl = handleTbl + handle)->rcType ) {
	  case RC_STRBLK:
	    free( tbl->u.sBlk.index ) ;
	    free( tbl->u.sBlk.data  ) ;
	    break;
	  case RC_STRMSG:
	    free( tbl->u.sMsg.buffer );
	    break;
	  case RC_PLAIN:
	    break;
	  case RC_PDMENU:
	    free( tbl->u.pdmenu.nameTbl );
	    break;
	  case RC_DLGWIN:
	    free( tbl->u.dlgwin.textTbl );
	    break;

	  default: Bug("Can't free unloaded resource; type %d", tbl->rcType);
	}
	tbl->rcType = 0;  /* mark slot unused */
    }
    else
	BUG();
}


#ifdef DOCUMENTATION
@Summary RcmGetStrBlk
 #include <wk/rcmgr.h>

 const char *RcmGetStrBlk( int hd, ushort no);
 hd	     handle
 ushort      Nummer des Strings
@Description
 Retrieves an entry from a string block.
@Return Value
 Ptr auf den String
#endif /*DOCUMENTATION*/

const char *RcmGetStrBlk( int handle, ushort number)
{
    handleTbl_t *tbl;
    sBlkIndex_t *index;
    ushort	 n, low, high, mid;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_STRBLK );

    index = tbl->u.sBlk.index;
    if( n = tbl->u.sBlk.noEntries ) {
	low = 0;
	high = n - 1;
	while( low <= high ) {
	    mid = (low + high) / 2;
	    if( index[mid].value > number ) {
		if( !mid )
		    break;
		high = mid - 1;
	    }
	    else if( index[mid].value < number ) {
		if( mid == n )
		    break;
		low = mid + 1;
	    }
	    else
		return (const char *)(tbl->u.sBlk.data + index[mid].offset) ;
	}
    }

    Bug("StrBlk-Resource # %u not found", number );
    return 0; /* NEVER REACHED */
}


#ifdef DOCUMENTATION
@Summary RcmTransferStrBlk
 #include <wk/rcmgr.h>

 char *RcmTransferStrBlk( int hd );
 hd	     handle
@Description
 Diese Funktion gibt einen Zeiger auf den Datenbereich zurck, indem
 die Strings allokiert wurden. Das AnwenderProgramm kann diesen Daten-
 bereich danach selbt freigeben ( via free() ).
 Allle weiteren Aufrufe mit diesem handle drfen nicht mehr durch-
 gefhrt, da der interne Datenbereichszeiger nicht mehr existiert;
 einzige noch zulsseige und auch sinnvolle Aufruf ist:
 RcmFreeResource() um das handel wieder freizugeben.
 Mit dieser Funktion ist es mglich ein Pointer-Array mit Pointern
 zu Strings zu fllen ( via RcmGetStrBlk() ) danach diese Funktion
 und RcmFreeResource() aufrufen, um die Resource freizugen, die Strings
 aber weiterhin im Speicher zu halten.
 der zurckgegeben Pointer kann ignoriert werden, oder abgespeichert
 werden, um den Speicher fr die Strings irgendwann mal freizugeben.
@Return Value
 Ptr auf den Datenbereichanfang
#endif /*DOCUMENTATION*/


char *RcmTransferStrBlk( int handle )
{
    handleTbl_t *tbl;
    char *p;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_STRBLK );
    p = tbl->u.sBlk.data ;
    tbl->u.sBlk.data = NULL; /* damit free() in FreeResource nichts macht*/
    return p ;
}




#ifdef DOCUMENTATION
@Summary RcmGetStrMsg	  Get String Message
 #include <wk/rcmgr.h>

 const char *RcmGetStrMsg( int hd, ushort no);
 hd	     handle
 ushort      Nummer der Message
@Description
 Ermittelt eine StringMsg mit der angegebenen Nummer.
 wird diese Nummer nicht gefunden, so wird NULL zurckgegeben.
 Diese Funktion benutzt einen Buffer, dessen Inhalt bei dem nchsten
 Aufruf (mit gleichem Handle) berschrieben wird.
@Return Value
 Ptr auf den String
#endif /*DOCUMENTATION*/


const char *RcmGetStrMsg( int handle, ushort number)
{
    handleTbl_t *tbl;
    ushort	 n;
    int c;
    char *p;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_STRMSG );

    if( fseek( rcStream, tbl->u.sMsg.fileOff, SEEK_SET) )
	goto errLabel ;
    if( number > tbl->u.sMsg.entries )
	goto errLabel;

    /* set filepointer to start of string with number */
    while( number > 0 )
	if( (c=getc(rcStream)) == EOF )
	    goto errLabel;
	else if( !c )
	    number-- ;

    /* copy string from file to buffer */
    n = tbl->u.sMsg.bufSize;
    p = tbl->u.sMsg.buffer;
    while( n-- ) {  /* count, may be file was damaged */
	if( (c=getc(rcStream)) == EOF )
	    goto errLabel;
	*(p++) = (c & 0xff);
	if( !c )
	    return *(p=tbl->u.sMsg.buffer) ? (const char *)p : NULL;
    }

  errLabel:
    clearerr(rcStream);
    return NULL;      /* Message invalid  or not found */
}



#ifdef DOCUMENTATION
@Summary RcmReadPlain	Read data from plain resource
 #include <wk/rcmgr.h>

 size_t RcmReadPlain( int hd, void *buffer, size_t n );
 hd	     handle
 buffer      to hold data
 n	     # of bytes to read
@Description
 Die Funktion liest plain data aus der angegebenen Resource.
 Es werden n bytes gelesen. Es handelt sich um seq. lesen.
@Return Value
 Anzahl der wirklich gelesenen Bytes.
@See Also
 RcmSeekPlain
#endif /*DOCUMENTATION*/

size_t RcmReadPlain( int handle, void *buffer, size_t nbytes )
{
    handleTbl_t *tbl;
    size_t n;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_PLAIN );

    n = 0;
    if( tbl->u.plain.nread + nbytes > tbl->u.plain.size )
	nbytes = tbl->u.plain.size - tbl->u.plain.nread;
    if( nbytes )
	if( !fseek(rcStream, tbl->u.plain.fileOff+tbl->u.plain.nread,
							    SEEK_SET))
	    n = fread( buffer, 1, nbytes, rcStream );

    clearerr(rcStream);
    return n;
}


#ifdef DOCUMENTATION
@Summary RcmSeekPlain	Set datapointer for next RcmReadPlain
 #include <wk/rcmgr.h>

 long RcmSeekPlain( int hd, long offset, int origin );
 hd	     handle
 offset      offset
 origin      SEEK_SET, SEEK_CUR, SEEK_END
@Description
 Wie fseek() aber nur fr RsmReadPlain()
@Return Value
 neue aktuelle Position innerhalb der Resource
 oder -1L bei einem Fehler
@See Also
 RcmReadPlain
#endif /*DOCUMENTATION*/

long RcmSeekPlain( int handle, long offset, int origin )
{
    handleTbl_t *tbl;
    long size, nread;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_PLAIN );

    size = tbl->u.plain.size;
    nread = tbl->u.plain.nread ;
    if( origin == SEEK_SET )
	nread = offset;
    else if( origin == SEEK_CUR )
	nread += offset;
    else /* SEEK_END */
	nread = size - offset;

    if( nread < 0L || nread > size )
	return -1L;
    else
	return tbl->u.plain.nread = nread;

}



#ifdef DOCUMENTATION
@Summary RcmQryPDMenu  Get Info about Pull down Menu
 #include <wk/rcmgr.h>

 const char *RcmQryPDMenu( int hd, ushort *noOfEntries );
 hd	     handle
 noOfEntries to hold data
@Description
 Die Funktion ermittelt die Infos zu einem Pull Down Menu:
@Return Value
 Pointer to the NameTbl (allocate inside the RCMGR)
@See Also
 RcmGetPDMenu
#endif /*DOCUMENTATION*/

const char *RcmQryPDMenu( int handle, ushort *entries )
{
    handleTbl_t *tbl;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_PDMENU );

    *entries = tbl->u.pdmenu.entries;
    return tbl->u.pdmenu.nameTbl;
}


#ifdef DOCUMENTATION
@Summary RcmGetPDMenu  Get Data of Pull down Menu
 #include <wk/rcmgr.h>

 void RcmGetPDMenu( int hd, ushort no, ushort *buffer )
 hd	     handle
 no	     zu lesende Nummer
 Buffer      Return: must be large enough for 3 shorts
@Description
 Die Funktion liest die Infos zu einem Menuitem in den Buffer ein
 zurck. Bei einer ungltigen Nummer enthlt der Buffer nur Nullen.
 Dadurch kann in einer einfachen Schleife gelesen werden.
@See Also
 RcmQryPDMenu
#endif /*DOCUMENTATION*/

void RcmGetPDMenu( int handle, ushort number, ushort *buffer )
{
    handleTbl_t *tbl;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_PDMENU );

    memset( buffer, 0, 3 * sizeof *buffer );
    if( number < tbl->u.pdmenu.entries )
	if( !fseek(rcStream,
		   tbl->u.pdmenu.fileOff+ 3*sizeof *buffer * number,
		   SEEK_SET) ) {
	    /*fread( buffer, 3*sizeof *buffer, 1, rcStream );*/
	    buffer[0] = ReadWord();
	    buffer[1] = ReadWord();
	    buffer[2] = ReadWord();
	}
    clearerr(rcStream);
}


#ifdef DOCUMENTATION
@Summary RcmTransferDlgWin
 #include <wk/rcmgr.h>

 char *RcmTransferDlgWin( int hd );
 hd	     handle
@Description
 Diese Funktion gibt einen Zeiger auf die TextTbl zurck, indem
 die Strings allokiert wurden. Das AnwenderProgramm kann diesen Daten-
 bereich danach selbt freigeben ( via free() ).
@Return Value
 Ptr auf den Datenbereichanfang
#endif /*DOCUMENTATION*/

char *RcmTransferDlgWin( int handle )
{
    handleTbl_t *tbl;
    char *p;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_DLGWIN );
    p = tbl->u.dlgwin.textTbl;
    tbl->u.dlgwin.textTbl = NULL; /* damit free() in FreeResource nichts macht*/
    return p ;
}



#ifdef DOCUMENTATION
@Summary RcmReadDlgWin	Read Data of DialogWindow
 #include <wk/rcmgr.h>

 size_t RcmReadDlgWin( int hd, int mode, void *buffer, size_t bufSize )
 hd	     handle
 mode	     Lesemodus
 buffer      Will return an item record
 bufSize     Size of Buffer
@Description
 Einlesen der Items in den Buffer; falls die angegebene Bufferlnge nicht
 ausreicht bricht das Programm ab.
 der Buffer enthlt jeweils genau ein Item, wobei das Lngenbyte im
 Itemrecord nicht dabei ist; d.h. das erste Byte ist der type des Items.
 mode  0 = rewind to first Item and Return # of Items
 mode  1 = Read next Item
@Return Value
 Anzahl der gltigen Bytes im Buffer; wobei 0 EOF bedeutet.
 bzw bei Mode 0: Anzahl der Items in der Resource.
@See Also
 RcmTransferDlgWin
#endif /*DOCUMENTATION*/

size_t RcmReadDlgWin( int handle, int mode, void *buffer, size_t bufSize )
{
    handleTbl_t *tbl;
    size_t nbytes;
    byte reclen;

    tbl = handleTbl + handle;
    xassert( tbl->rcType == RC_DLGWIN );

    if( !mode ) {   /* Reset */
	tbl->u.dlgwin.offRover = tbl->u.dlgwin.offStart;
	tbl->u.dlgwin.itemsRead = 0;
	return tbl->u.dlgwin.items;
    }
    else if( tbl->u.dlgwin.offRover == -1L )
	return 0;   /* EOF was reached */
    else if( tbl->u.dlgwin.itemsRead >= tbl->u.dlgwin.items )
	return 0;   /* EOF was reached */
    else {
	nbytes = 0;
	if( fseek(rcStream, tbl->u.dlgwin.offRover, SEEK_SET) )
	    tbl->u.dlgwin.offRover = -1L; /* must be EOF */
	else if( fread( &reclen, 1, 1, rcStream ) == 1 ) {
	    nbytes = reclen+1 ;
	    if( bufSize < nbytes )
		Bug("RCMGR: DlgWin item buffer");
	    if( fread( buffer, nbytes, 1, rcStream ) != 1 ) {
		tbl->u.dlgwin.offRover = -1L; /* must be EOF */
		nbytes = 0;
	    }
	    else {
		tbl->u.dlgwin.offRover += 1+nbytes;
		tbl->u.dlgwin.itemsRead++;
	    }
	}
	clearerr(rcStream);
	return nbytes;
    }
}


/******* local helper functions *******/

/*
 * Diese Funktion sucht in der library
 * nach der Resource mit dem angegebenen Namen.
 * Die Funktion positioniert auf den angegebenen offset.
 * Die Funktion bricht bei einem Fehler ab
 */

static void FindResource( const char *name )
{
    int c;
    long off;

    rewind( rcStream );
    c = EOF ; /* default, for error detection */
    if( getc(rcStream) == 'R' )
	if( getc(rcStream) == 'C' )
	    if( getc(rcStream) == '\x01' )
		while( (c=getc(rcStream)) != EOF )
		    if( !c )
			break;
    if( c == EOF )
	Bug("Resource '%s': Invalid ID", name );
    if( (c=getc(rcStream)) != '\0' )
	Bug("Resource '%s' unkown C-Mode %d", name, c );
    off = ScanOffsetTbl( name, rcStream ) ;
    if( off == -1L ) /* entry not found */
	Bug("Resource '%s' not found", name );

    if( fseek( rcStream, off, SEEK_SET) )
	BUG();
}


/*
 * Die Funktion liest ein wort vom Stream
 * und bricht ab bei einem Fehler
 */

static ushort ReadWord()
{
    int c1, c2;

    c1 = getc(rcStream) ;
    c2 = getc(rcStream) ;
    if( c1 == EOF || c2 == EOF )
	Bug("Error reading resource");

  #if WKLIB_BIG_ENDIAN
    return (( c1 & 0xff ) << 8 ) | (c2 & 0xff) ;
  #else
    return (( c2 & 0xff ) << 8 ) | (c1 & 0xff) ;
  #endif
}

/*
 * Die Funktion liest ein longwort vom Stream
 * und bricht ab bei einem Fehler
 */

static ulong ReadLongWord()
{
    ushort c1, c2;

    c1 = ReadWord() ; /* low word */
    c2 = ReadWord() ; /* high word */
  #if WKLIB_BIG_ENDIAN
    return (( c1 & 0xffff ) << 16 ) | (c2 & 0xffff) ;
  #else
    return (( c2 & 0xffff ) << 16 ) | (c1 & 0xffff) ;
  #endif
}



#ifdef TEST
/*** TEST TEST TEST TEST TEST TEST TEST TEST TEST TEST ****/


main( int argc, char ** argv )
{
    int handle, handle2 ;
    long i;
    const char *p;

    puts( "Init" );
    RcmInitialize( *argv );
    argc-- , argv++ ;

    puts( "Load" );
    handle = RcmLoadResource( *argv, RC_STRMSG );
    argc--; argv++;


    for( ; argc ; argc--, argv++ ) {
	p = RcmGetStrMsg( handle, atoi( *argv ) );
	printf("Msg is: `%s'\n", p );
    }

    RcmFreeResource(handle);

    puts( "Ready" );
}


#endif /* TEST */
/**** end of file ****/
