/* [hlpwin.c wk 12.6.91] Help Window
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * History:
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/lib.h>
#include <wk/cmpr.h>
#include <wk/scr.h>
#include <wk/event.h>
#include <wk/file.h>
#include <wk/cua.h>

/**** constants ***/
#define ATTR_STD  3	/* For frame */
#define ATTR_INV  1	/* For arrows */
#define ATTR_BASE 2	/* Base for Helptext */
#define ATTR_CRS  1	/* selected Index in helptext */

    /* Bitmask values internal for ShowEntry() */
#define CRS_ANY   1	/* Indices sind vorhanden			 */
#define CRS_SEL   2	/* Cursor ist selektiert			 */
#define CRS_LEFT  4	/* links vom selektierten ist noch ein Index	 */
#define CRS_RIGHT 8	/* rechts vom selektierten ist noch ein Index	 */
#define CRS_UP	  16	/* oberhalb vom selektierten ist noch ein Index  */
#define CRS_DOWN  32	/* unterhalb vom selektierten ist noch ein Index */


#define MAX_HANDLES 3
#define MAX_DEEP   5	/* 5 Stufen merken */

/**** types ******/
typedef struct {
	int    n;	/* nummer des Indices in line; 0 = keiner ausgewhlt*/
	ushort line;	/* Zeilennummer des indices */
	ushort keyNr;	/* keyNr des ausgewhlten indices */
    } crs_t;

typedef struct {
	int   useFlag;	/* true if in use */
	FILE  *st;	/* helpStream */
	void  *rdHd;	/* Handle for Compress...() */
	long  keyTblOff;/* Offset of keyTbl */
	ushort keyNr;	/* actual keyNr */
	int px,py,sx,sy;/* Position and Size of Window */
	char **text;	/* buffer for helptext */
	crs_t crs;	/* Cursor informations */
	crs_t cLeft, cUp, cRight, cDown; /* zum merken */
	int scrHd;	/* handle of screen ( -1 = no Screen activ ) */
	ushort line;	/* actual linenumber */
	ushort nlines;	/* total # of lines */
	int doUpdate;
	unsigned flag;
	crs_t stk[MAX_DEEP]; /* to remember the last keyNr's */
	int   stkInd;
    } data_t;		/* Struct to hold Informations about an helpinstance */

/**** globals ****/
static data_t handleTbl[MAX_HANDLES];
static char *defaultDataBase;
static const char indexEntryKey[] = "$INDEX$";

/*** local Protos ***/
static int ReadHelp( data_t *d, const char *keyName, ushort keyNr );
static int MakeWindow( data_t * );
static int ReadEntry( FILE *st, void *rdHd , char ***, unsigned *,ushort*);
static int ShowEntry( char**,ushort,int,int,int,
		     crs_t*,crs_t*,crs_t*,crs_t*,crs_t*,int,int,int);
/**** def'd Functions ****/
#define EnterCritical()     /* Dummies*/
#define LeaveCritical()     /* Dummies*/
/**** Global Functions ****/

/*
 * ffnet das Hilfesystem und gibt in handle einen Handle
 * darauf zurck.
 * wird als dbname NULL angegeben, so wird der DB-Name, der beim allerersten
 * Aufruf dieser Funktion verwendet wurde benutzt.
 * wird fr py, px,sy oder sx -1 angegeben, so werden standard werte benutzt.
 * Returns: ErrorCode
 */

int HelpOpen( int *handle, const char *dbName, int px, int py, int sx, int sy )
{
    int c, i, err;
    data_t *d=NULL; /* controlled by err */

    err = DLGE_NOHDS;
    EnterCritical();
    for( i=0; i < MAX_HANDLES; i++ )
	if( !handleTbl[i].useFlag ) {
	    d = handleTbl + i;
	    d->useFlag++;
	    err = 0;
	    break;
	}
    LeaveCritical();
    if( err )
	goto retLabel;
    *handle = i;

    d->st = NULL;
    d->rdHd = NULL;
    d->text = NULL;
    d->scrHd = -1;
    if( sx < 5 )
	sx = 20;
    if( sy < 3 )
	sy = 5;
    d->px = px;
    d->py = py;
    d->sx = sx;
    d->sy = sy;

    if( !defaultDataBase )
	defaultDataBase = xstrdup( dbName );

    if( !(d->st = fsopen( dbName? dbName:defaultDataBase , "b" )) ) {
	err = DLGE_NODBS;
	goto retLabel;
    }

    c = EOF ; /* default, for error detection */
    if( getc(d->st) == 'H' )
	if( getc(d->st) == 'L' )
	    if( getc(d->st) == '\x01' )
		while( (c=getc(d->st)) != EOF )
		    if( !c )
			break;
    if( c == EOF ) {
	err = DLGE_VER;
	goto retLabel;
    }
    if( !(d->rdHd = CompressOpen(d->st)) ) {
	err = DLGE_READ;
	goto retLabel;
    }
    d->keyTblOff = ftell( d->st );
    d->stkInd = 0;

  retLabel:
    if( err && err != DLGE_NOHDS ) {
	if( d->rdHd )
	    CompressClose(d->rdHd);
	if( d->st )
	    fclose(d->st);
	EnterCritical();
	d->useFlag = 0;
	LeaveCritical();
    }

    return err;
}


int HelpClose( int handle )
{
    data_t *d;

    if( handle >= 0 && handle < MAX_HANDLES )
	if( (d =handleTbl+handle)->useFlag ) {
	    CompressClose( d->rdHd );
	    fclose( d->st );
	    if( d->scrHd != -1 )
		ScrClose( d->scrHd );
	    if( d->text )
		free( *d->text );
	    free( d->text );
	    EnterCritical();
	    d->useFlag = 0;
	    LeaveCritical();
	    return 0;
	}
    return DLGE_INVHD;
}


/*
 * Get Help for the given Key
 * use NULL to get INDEX Page
 */

int HelpGet( int handle, const char *key )
{
    if( handle >= 0 && handle < MAX_HANDLES )
	if( handleTbl[handle].useFlag )
	    return ReadHelp( handleTbl + handle,
			     key ? key: indexEntryKey , 0 );
    return DLGE_INVHD;
}



/*
 * Die Editaktionen am Window durchfhren, dabei wird jeweils ein keycode
 * ausgegeben.
 * der erste Aufruf bewirkt das Anzeigen des Windows
 * K_CMD_EDITSTART  Editieranfang; bewirkt nur ein Anzeigen und kehrt zurck
 * K_CMD_EDITEND    Editierende; Window entfernen
				( aber nicht die gelesenen Daten)
 * Returns: ErrorCode
 */

int HelpEdit( int handle )
{
    int i, err, upd, kusd, ctrl, sel, keyCode=0, ex;
    data_t *d=NULL;
    unsigned eventWord=0, btns;
    int glidPos=0, glidLen, x,y;

    err = DLGE_INVHD;
    if( handle >= 0 && handle < MAX_HANDLES )
	if( (d=handleTbl+handle)->useFlag )
	    err = 0;
    if( err )
	return err;

    upd =  d->doUpdate;
    for(ex=0; !ex && !err ;) {
	ctrl = kusd = 0;
	if( upd )
	    keyCode = K_CMD_USER_0;
	else {
	    eventWord = EventWait();
	    if( eventWord & EVT_CMD )
		keyCode = EventGetCmd();
	    else if( eventWord & EVT_KBD )
		keyCode = EventGetKbd();
	    else if( eventWord & EVT_MS ) {
		btns = ScrMsStat( &i, &x, &y );
		if( i == d->scrHd ) {
		    if( x == d->sx-1 ) { /* in scroll bar */
			if( eventWord & EVT_MSLB )
			    keyCode = K_CMD_USER_1; /* left mouse btn clicked */
			else if( eventWord & EVT_MSMV )
			    keyCode = K_CMD_USER_2; /* dragged */
		    }
		    else if( x && y && y < d->sy && eventWord & EVT_MSLB ) {
			EventClear( EVT_MSMV );
			keyCode = K_CMD_USER_3;
		    }
		    else {
			EventClear( EVT_MSMV );
			keyCode = K_CMD_DUMMY_0;
		    }
		}
		else {
		    EventClear( EVT_MSMV );
		    keyCode = K_CMD_DUMMY_0;
		}
	    }
	    else
		keyCode = K_CMD_DUMMY_0;
	}

	switch(keyCode) {
	  case K_CMD_USER_0 :
	    break;

	  case K_CMD_USER_1:	/* left Button is down */
	    if( y == 1 ) { /* Up-Arrow */
		if( d->line ) {
		    d->line-- ;
		    upd = 1;
		}
	    }
	    else if( y == d->sy-2 )  { /* Down Arrow */
		if( d->text[d->line+1] ) {
		    d->line++ ;
		    upd = 1;
		}
	    }
	    else if( y > 1 && y < d->sy-2 ) {
		/* in scrollbar, but not inside glider */
		if( !(y >= glidPos && y < glidPos+glidLen) ) {
		    i = ((y-2)) * d->nlines/(d->sy-4);
		    if( i >= 0 && i < d->nlines ){
			d->line = i ;
			upd = 1;
		    }
		}
	    }
	    EventClear( EVT_MSLB );
	    break;

	  case K_CMD_USER_2:	/* mouse moved with button down */
	    if( glidLen ) {
		if( y >= glidPos && y < glidPos+glidLen ) { /* inside glider */
		  #if 0
		    int dmy1, dmy2;
		    unsigned dir;
		  #endif

		    i = ((y-2)) * d->nlines/(d->sy-4);
		    if( i >= 0 && i < d->nlines ){
			d->line = i ;
			upd = 1;
		    }
		  #if 0
		    i = d->sy > 4 ? (d->nlines / (d->sy-4)) : 1;
		    if( (dir=EventGetMs(&dmy1,&dmy2)) & EVT_MS_MV_U ) {
			if( d->line > i )
			    d->line -= i ;
			else
			    d->line = 0;
			upd = 1;
		    }
		    else if( dir & EVT_MS_MV_D ) {
			if( d->line+i < d->nlines )
			    d->line += i;
			else
			    d->line = d->nlines-1;
			upd = 1;
		    }
		  #endif
		}
	    }
	    EventClear( EVT_MSMV );
	    break;

	  case K_CMD_USER_3:	/* select index */
	    upd=1;
	    d->crs.n = 0;   /* delete last selected index */
	    EventClear( EVT_MSLB );
	    break;

	  case K_TAB: /* to next indice or first index */
	    if( !d->crs.n ) {  /* no selection yet */
		if( d->cUp.n ) { /* there is at least one index */
		    d->crs = d->cUp;  /* set to first index */
		    upd = 1;
		}
	    }
	    else {  /* one is selected: switch to the next one */
		if( d->cRight.n )
		    d->crs = d->cRight;
		else if( d->cDown.n )
		    d->crs = d->cDown ;
		else	/* no more indices, remove seection */
		    d->crs.n = 0;
		upd = 1;
	    }
	    kusd++;
	    break;

	  case K_BACKTAB: /* to prev indice or last index */
	    if( !d->crs.n ) {  /* no selection yet */
		if( d->cDown.n ) {  /* ther is at least one index */
		    d->crs = d->cDown;	/* set to last index */
		    upd = 1;
		}
	    }
	    else {  /* one is selected: switch to the prev one */
		if( d->cLeft.n )
		    d->crs = d->cLeft;
		else if( d->cUp.n )
		    d->crs = d->cUp ;
		else	/* no more indices, remove selection */
		    d->crs.n = 0;
		upd = 1;
	    }
	    kusd++;
	    break;

	  case K_RIGHT:
	    if( d->crs.n ) {  /* only if selected */
		if( d->cRight.n ) {
		    d->crs = d->cRight;
		    upd = 1;
		}
	    }
	    kusd++;
	    break;

	  case K_LEFT:
	    if( d->crs.n ) {  /* only if selected */
		if( d->cLeft.n ) {
		    d->crs = d->cLeft;
		    upd = 1;
		}
	    }
	    kusd++;
	    break;

	  case '-':
	  case K_CLEFT:
	    if( d->crs.n ) {  /* only if selected */
		if( d->cUp.n ) {
		    d->crs = d->cUp;
		    upd = 1;
		}
		else if( d->line ) {
		    d->line-- ;
		    ctrl = 2; /* get the first index in the first line */
		    upd = 1;
		}
	    }
	    kusd++;
	    break;

	  case '+':
	  case K_CRIGHT:
	    if( d->crs.n ) {  /* only if selected */
		if( d->cDown.n ) {
		    d->crs = d->cDown;
		    upd = 1;
		}
		else if( d->text[d->line+1] ) {
		    d->line++ ;
		    ctrl = 1; /* get the first index in the last line */
		    upd = 1;
		}
	    }
	    kusd++;
	    break;

	  case K_UP :
	    if( d->line ) {
		d->line-- ;
		upd = 1;
	    }
	    kusd++;
	    break;

	  case K_DOWN:
	    if( d->text[d->line+1] ) {
		d->line++ ;
		upd = 1;
	    }
	    kusd++;
	    break;

	  case K_PGUP:
	    if( d->line >= d->sy-1 )
		d->line -= d->sy-1;
	    else
		d->line = 0;
	    upd = kusd = 1;
	    break;

	  case K_PGDN:
	    for( i=1; i < d->sy && d->text[d->line]; i++, d->line++ )
		;
	    if( !d->text[d->line] && d->line )
		d->line--;
	    upd = kusd = 1;
	    break;

	  case '4' : d->px--;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '6' : d->px++;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '8' : d->py--;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '2' : d->py++;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '7' : d->sx--;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '3' : d->sx++;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '9' : d->sy--;   err = MakeWindow( d ); upd = kusd = 1; break;
	  case '1' : d->sy++;   err = MakeWindow( d ); upd = kusd = 1; break;

	  case K_RETURN:
	    if( (d->crs.n && d->crs.keyNr != 0xffff) || !d->crs.n ) {
		if( (i=d->stkInd) >= MAX_DEEP ) {  /* shift stack */
		    for( i=1; i < MAX_DEEP; i++ )
			d->stk[i-1] = d->stk[i];
		    i--;
		    d->stkInd--;
		}
		d->stk[i].keyNr = d->keyNr;
		if( d->crs.n ) {
		    d->stk[i].n = d->crs.n;
		    d->stk[i].line = d->crs.line;
		    i = 0;
		}
		else { /* no selection, so Show index */
		    d->stk[i].n = 0;
		    d->stk[i].line = d->line;
		    i = 1;
		}
		d->stkInd++;
		err = ReadHelp( d, i? indexEntryKey:NULL, i ? 0:d->crs.keyNr);
		upd = 1;
	    }
	    kusd++;
	    break;

	  case K_RUBOUT:
	    if( d->stkInd ) {
		i = --d->stkInd;
		d->keyNr     = d->stk[i].keyNr;
		err = ReadHelp( d, NULL, d->keyNr );
		d->crs.n     = d->stk[i].n;
		d->crs.line  = d->stk[i].line;
		d->crs.keyNr = 0xffff;
		d->line      = d->crs.line;
		upd = 1;
	    }
	    kusd++;
	    break;

	  case K_CMD_EDITSTART: upd = kusd = 1; ex++; break;
	  case K_CMD_EDITEND:
	    if( d->scrHd != -1 ) {
		ScrClose(d->scrHd);
		d->scrHd = -1;
		ScrUpdate(0);
	    }
	    ex++;
	    kusd = 1;
	    break;

	  default:
	    ex++;
	    break; /* nothing */
	}


	if( upd && !err ) {
	    d->doUpdate = 0;
	    if( d->scrHd == -1 )
		if( err = MakeWindow( d ) )
		    goto retLabel;
	    upd = 0;
	    if( keyCode != K_CMD_USER_3 )
		x = y = 0;

	    do {
		upd++;
		sel = ShowEntry( d->text, d->line, d->scrHd, d->sx, d->sy, &d->crs,
		      &d->cUp, &d->cRight, &d->cDown, &d->cLeft,
		      d->flag &1 ? 1:0, x, y);
		if( ctrl == 1 && d->cDown.n )
		    d->crs = d->cDown;
		else if( ctrl == 2 && d->cUp.n )
		    d->crs = d->cUp;
		else if( d->flag & 1 && !sel && upd < 2) {
		    d->crs.n = 1;
		    d->crs.line = d->line;
		}
		else
		    upd = 0;
	    } while( upd );

	    glidPos = ScrScrollBar( d->scrHd, d->sx-1, 1, 0,
				    d->nlines, d->line,
				    d->sy-2, d->sy-2, ATTR_STD, ATTR_INV,
				    &glidLen);
	    ScrUpdate(d->scrHd);
	    upd = 0;
	}

	if( kusd )
	    EventClear( eventWord & EVT_CMD ? EVT_CMD : EVT_KBD );

    }

  retLabel:
    return err;
}




/**** Local Functions *****/

/*
 *  Die Hilfe einlesen.
 *  Dazu Key verwenden es sein denn key == NULL, dann via keyNr lesen.
 *  Wird der key gefunden, so werden die alten Infos ( vom letzten Aufruf
 *  dieser Funktion aus dem Speciher entfernt ).
 *  Returns: ErrorCode
 */

static int ReadHelp( data_t *d, const char *keyName, ushort keyNr )
{
    int err;
    long off;
    ushort knr;

    err = 0;
    clearerr( d->st );
    if( fseek( d->st, d->keyTblOff, SEEK_SET) )
	err = DLGE_READ;
    else {
	off = keyName ? ScanOffsetTbl1( keyName, d->st, &knr ) :
			ScanOffsetTbl2( knr = keyNr, d->st);
	if( off == -1L )   /* key not found */
	    err = DLGE_KEYNF;
	else if( fseek( d->st, off, SEEK_SET) )
	    err = DLGE_READ;
	else {
	    if( d->text ) /* remove the last Infos */
		free( *d->text );
	    FREE( d->text );
	    if( !(err = ReadEntry( d->st, d->rdHd, &d->text,
				   &d->flag, &d->nlines )) ) {
		/* weitere Inits eventuell hier */
		d->line = 0;
		d->crs.line = 0;
		/* select first line , first entry if index */
		d->crs.n = d->flag & 1 ? 1 : 0;
		d->cUp.n = d->cRight.n = d->cDown.n = d->cLeft.n = 0;
		d->doUpdate = 1;
		d->keyNr = knr;
	    }
	}
    }
    return err;
}




/*
 * Ein Window aufbauen in der gre sx*sy
 * Wenn d->scrHd != -1 ist, so wird dieses Window zuerst geschlossen
 * Returns: errorCode
 */

static int MakeWindow( data_t *d )
{
    int x,y, hd, err, sx, sy;

    err = 0;
    hd = d->scrHd;
    if( hd != -1 )
	if( ScrClose( hd ) )
	    Error(2002,"Error Closing Window ");
    d->scrHd = -1;
    sx = d->sx;
    sy = d->sy;

    if( ScrOpen( &hd, 0,d->px, d->py, sx, sy, ATTR_STD, 0, 0 ) ) {
	/* error, try with default values */
	d->px = d->py = -1 ;
	sx = d->sx = 30; sy = d->sy = 5;
	err = ScrOpen( &hd, 0,d->px, d->py, sx, sy, ATTR_STD, 0, 0 );
    }

    if( !err ) {
	d->scrHd = hd;
	ScrGetPos( hd, 0, &d->px, &d->py );
	/* Frame malen */
	ScrWriteCell( hd, 0, 0,    '\xda', ATTR_STD );
	ScrWriteCell( hd, 0, sy-1, '\xc0', ATTR_STD );
	for(x=1; x < sx-1 ; x++ ) {
	    ScrWriteCell( hd, x, 0,    '\xc4', ATTR_STD );
	    ScrWriteCell( hd, x, sy-1, '\xc4', ATTR_STD );
	}
	ScrWriteCell( hd, x, 0,    '\xbf', ATTR_STD );
	ScrWriteCell( hd, x, sy-1, '\xd9', ATTR_STD );
	for( y = 1; y < sy-1; y++ ) {
	    ScrWriteCell( hd, 0,    y , '\xb3', ATTR_STD );
	    ScrWriteCell( hd, sx-1, y , '\xb3', ATTR_STD );
	}
    }
    return err;
}


/*
 *  Einlesen eines Helpentries in die interne Datenstruktur.
 *  welche hier allokiert wird.
 *
 *  Steuerzeichen im Text:
 *  '\x02'     = Start of Text
 *  '\x03'     = End of Text
 *  '\xfa'     = geschtztes Leerzeichen
 *  '\n'       = linefeed
 *  '\x10' c   = das Zeichen c transparent ausgeben (state = 1)
 *  '\x11' n   = Index Escape; n = keyNr (ushort)   (state = 2)
 *  '\x12' f   = Formatierzeichen f auswerten       (state = 3)
 *  '\x13' a   = Attribut a setzen a=(N=0,R=1,B=2,U=3) (state = 4)
 *  state = 5 : use last character and set state to 0
 *  state = 6 : Anzeigen des folgenden Wortes als einen Reference Key hinweis
 *  state = 8 : 1. Byte von IndexNr gelesen
 *  state = 15: Ende
 *
 *  Der FilePointer ist positioniert auf den Anfang des Entries
 *  Die ersten Bytes sind nicht comprimiert und  werden deshalb
 *  normal gelesen, danach kommt der rdHd zur Anwendung.
 *  Der Begin jeder Zeile enthlt ein Control_Byte, welches Bitweise
 *  codiert ist:
 *  Bit 3 - 0 = State at beginning of line
 *  Bit 5 - 4 = Attribute ( 0..3)
 *  Bit 7 - 6 = FormatFlag ( 0..3)
 *  --> Hinweis: Der Tabulator wird zur Zeit nicht gespeichert
 *	Struktur hierfr muss noch angelegt werden.
 * Returns: ErrorCode
 */

static int ReadEntry( FILE *st, void *rdHd, char ***textPar,
		      unsigned *flag, ushort *ret_nlines )
{
    ushort nlines, line;
    int  ncolumns;
    long nbytes; /* total number of bytes */
    char **text;
    int supChar;
    int state, attr, frmt, c=0, x, err;

    text = NULL ;
    err =0;
    /* read the header */
    if( fread( &nlines, sizeof(ushort), 1, st ) != 1 ) {
	err = DLGE_READ;
	goto retLabel;
    }
    *ret_nlines = nlines;
    if( (ncolumns = getc( st )) == EOF ) {
	err = DLGE_READ;
	goto retLabel;
    }
    *flag = getc( st ) ;
    if( fread( &nbytes, sizeof(long), 1, st ) != 1 ) {
	err = DLGE_READ;
	goto retLabel;
    }
    /* allocate storage */
    if( nbytes+nlines > 60000 ) {
	err = DLGE_2LRG;
	goto retLabel;
    }
    if( !(text = calloc( nlines+1, sizeof *text )) ) {
	err = DLGE_NOMEM;
	goto retLabel;
    }
    if( !(*text = xmalloc( nbytes+nlines+1 )) ) {
	err = DLGE_NOMEM;
	goto retLabel;
    }

    if( CompressSeek( rdHd, ftell(st)) == -1L ) {
	err = DLGE_READ;
	goto retLabel;
    }
    if( CompressRead( rdHd ) != '\x02' ) {
	err = DLGE_INV; /* No STX */
	goto retLabel;
    }

    attr = 0 ;
    frmt = 0 ;
    line = 0;
    x=0;
    for(state=0;state != 15 ;) {
	if( !x )
	    text[line][x++] = ( (frmt & 0x03) << 6 ) |
			      ( (attr & 0x03) << 4 ) | state & 0xff ;
	if( state == 5 )
	    state = 0 ;
	else {
	    c = CompressRead(rdHd);
	    if( c == EOF || !nbytes) {
		err = DLGE_INV; /* unexpected EOF or invalid length */
		goto retLabel;
	    }
	    nbytes--;
	}
	supChar = 0;
	switch( state ) {
	  case 0:
	    switch( c ) {
	      case '\x03':  state = 15; break; /* end of text */
	      case '\x10':  state =  1; break; /* write next character */
	      case '\x11':  state =  2; break; /* index escape */
	      case '\x12':  state =  3; break; /* Formatierkennzeichen folgt */
	      case '\x13':  state =  4; break; /* Attribut folgt */
	      case 0 : err = DLGE_INV; goto retLabel; /* "\\0 in textfile" */
	    }
	    break ;

	  case 1:   /* das Zeichen transparent ausgeben */
	    state = 0 ;
	    break ;
	  case 2:   /* Index Key  byte 1*/
	    state = 8;
	    break;

	  case 8:   /* Index Key  byte 2 */
	    state = 6 ;
	    break ;

	  case 6:   /* Wort nach index key */
	    if( isspace(c) || c == '\x03' ) {
		state = 5 ;
		supChar++ ;
	    }
	    break ;

	  case 3:   /* Formatierkennzeichen */
	    frmt = c & 0x03 ;
	    state = 0 ;
	    break ;

	  case 4:   /* Attribut setzen */
	    attr = c & 0x03;
	    state = 0 ;
	    break ;

	  default: Bug("Invalid state=%d", state ) ;
	}

	if( !supChar ) { /* store character */
	    if( !state && c == '\n' ) {
		text[line][x++] = '\0';
		text[line+1] = text[line] + x ;
		line++;
		if( !(line < nlines) ) {
		     err = DLGE_INV; /* invalid # of lines */
		     goto retLabel;
		}
		x = 0;
	    }
	    else
		text[line][x++] = c;
	}
    }
    text[line][x] = '\0';

  retLabel:
    if( err ) {
	if( text )
	    free( *text );
	FREE( text );
    }
    *textPar = text;
    return err;

} /* end ReadEntry() */




/*
 * Eine Entry der in text gespeichert anzeigen
 * hd ist der Handle des Screens
 * crs zeigt die gewnchte Selektierung an
 * in idxUp, idxRight, idxDown, idxLeft werden die Informationen
 * von den Nachbar indices zurckgegeben bzw, wenn keiner selektiert
 * wurde ( crs->n = 0 ) wird falls vorhanden in idxUp der erste
 * und falls vorhanden in idxDown der letze zurckgegeben
 * ber die vorhandenen Indices zurck, wobei allerdings die keyNr nicht
 * gesetzt wird.
 * Bedeutung von ctrl: 0 = standard; 1 = Index
 * Wenn X != 0, dann geben x und y einen Mouseclick position an.
 * Returns: false if no Selction
 *	    true if any selected
 */

static int ShowEntry( char **text, ushort line,
		      int hd, int sx, int sy,
		      crs_t *crs,
		      crs_t *idxUp,
		      crs_t *idxRight,
		      crs_t *idxDown,
		      crs_t *idxLeft ,
		      int ctrl, int msX, int msY )
{
    int supChar, state, attr, frmt, c=0, x, y, i, saFlag;
    const char *t;
    byte a;
    char ch;
    int sa,sb,sc; /* spaces am anfang, text, spaces am Ende einer Zeile */
    unsigned idxFlag;
    int idxNr;	    /* nummer des Indices in der Zeile ( 1.. ) */
    int idxCount;   /* total number of Indexs in window */
    int thisIsCrs;  /* This is the Cursor Flag */
    ushort keyNr;   /* store keynr here */

    sx -= 2; sy -= 2; /* leave space for the frame */
    text += line;

    idxFlag = 0;
    idxUp->n = idxRight->n = idxDown->n = idxLeft->n = 0; /* invalidate all */
    idxCount = idxNr = 0;
    thisIsCrs = 0;
    sa = saFlag = 0;
    x = y = 0;
    t = text[y];
    if( !t )
	return 0 ; /* end of list, do nothing */
    state = *t & 0x0f;
    attr  = (*t >> 4) & 0x03;
    frmt  = (*t >> 6) & 0x03;
    t++;
    while(state != 15 && y < sy ) {
	if( state == 5 )
	    state = 0 ;
	else
	    c = *t++;
	supChar = 0;
	switch( state ) {
	  case 0:
	    switch( c ) {
	      case '\x03':  state = 15;            break; /* end of text */
	      case '\x10':  state =  1; supChar++; break; /* write next character */
	      case '\x11':  state =  2; supChar++; break; /* index escape */
	      case '\x12':  state =  3; supChar++; break; /* Formatierkennzeichen folgt */
	      case '\x13':  state =  4; supChar++; break; /* Attribut folgt */
	      case '\xfa':  c = ' '; break; /* protected blank */
	    }
	    break ;

	  case 1:   /* das Zeichen transparent ausgeben */
	    state = 0 ;
	    break ;

	  case 2:   /* Index Key  byte 1*/
	    state = 8;
	    memcpy( &keyNr, t-1, sizeof keyNr ); /* get the 2 bytes of keyNr */
	    supChar++;
	    break;

	  case 8:   /* Index Key  byte 2 */
	    state = 6 ;
	    c = '\xaf' ; /* Doppelpfeil rechts */
	    if( ctrl )
		supChar++;
	    idxNr++;	 /* counter for indices in line */
	    idxCount++;  /* counter for indices in window  */
	    if( !crs->n && !msX ) {
		if( idxCount == 1 ) { /* Get infos about first index */
		    idxUp->n = idxNr;
		    idxUp->line = line+y;
		    idxFlag |= CRS_UP | CRS_DOWN;  /* first will be last */
		}
		idxDown->n = idxNr;
		idxDown->line = line+y;
	    }
	    else {
		if( idxFlag & CRS_SEL ) {
		    if( idxNr > crs->n && crs->line == line+y ) {
			if( !(idxFlag & CRS_RIGHT) ) {
			    idxRight->n = idxNr;
			    idxRight->line = line+y;
			    idxFlag |= CRS_RIGHT;
			}
		    }
		    else if( line+y > crs->line ) {
			if( !(idxFlag & CRS_DOWN) ) {
			    idxDown->n = idxNr;
			    idxDown->line = line+y;
			    idxFlag |= CRS_DOWN;
			}
		    }
		}
	    }
	    break ;

	  case 6:   /* Wort nach index key */
	    if( isspace(c) || !c || c == '\x03' ) {
		state = 5 ;
		supChar++ ;
		thisIsCrs = 0;
	    }
	    else {
		if( !(idxFlag & CRS_SEL) && (crs->n||msX) ) { /* no crs yet selected */
		    if( (crs->line == (line+y) && crs->n == idxNr) ||
			(msX == x && msY == y	)	       ) {
			idxFlag |= CRS_SEL;   /* Cursor selected */
			crs->keyNr = keyNr;
			thisIsCrs = 1;
			if( idxNr > 1 )
			    idxFlag |= CRS_LEFT; /* make infos valid */
			if( idxCount > idxNr )
			    idxFlag |= CRS_UP;	 /* make infos valid */
		    }
		    else {  /* remember values for left and up */
			idxLeft->n    = idxNr;
			idxLeft->line = line+y;
			if( line+y < crs->line ) {
			    idxUp->n	= idxNr;
			    idxUp->line = line+y;
			}
		    }
		}
		if( c == '\xfa' )
		    c = ' ';
	    }
	    break ;

	  case 3:   /* Formatierkennzeichen */
	    frmt = c & 0x03 ;
	    state = 0 ;
	    supChar++;
	    break ;

	  case 4:   /* Attribut setzen */
	    attr = c & 0x03;
	    state = 0 ;
	    supChar++;
	    break ;

	  default: Bug("Invalid state=%d", state ) ;
	}

	if( !supChar ) { /* show character */
	    if( (!state && !c) || state==15 ) {
		sc = sx - x;   /* es gilt immer: x <= sx */
		sb = sx - sc - sa;
		/* in every case we have to fill up the line with blanks */
		for( ; x < sx; x++ )
		    ScrWriteCell( hd, x+1, y+1, ' ', ATTR_STD );

		/* Format ^.n und ^.l sind z.Z identisch und dummies */
		if( frmt > 1 ) { /* justify right or center */
		    i = frmt == 2 ? sc : (sc-sa)/2;
		    if( i > 0 ) {  /* shift right */
			 for( x=sb-1 ; x >= 0; x-- ) {
			     ScrReadCell(hd, sa+x+1, y+1, &ch, &a );
			     ScrWriteCell(hd, sa+x+i+1, y+1, ch, a );
			 }
			 for( x=0; x < i; x++ )
			     ScrWriteCell(hd, sa+x+1, y+1, ' ', ATTR_STD );
		    }
		    else if( i < 0 ) {	/* shift left */
			 for( x=0 ; x < sb; x++ ) {
			     ScrReadCell(hd, sa+x+1, y+1, &ch, &a );
			     ScrWriteCell(hd, sa+x+i+1, y+1, ch, a );
			 }
			 for( x=0; x < -i; x++ )
			     ScrWriteCell(hd, sa+sb-x-1+1, y+1, ' ', ATTR_STD);
		    }
		}
		idxNr = x = 0;
		sa = saFlag = 0;
		t = text[++y] ;
		if( !t || state == 15 )
		    state = 15;
		else {
		    xassert( (byte)*t == (byte)( ((frmt & 0x03) << 6) |
				       ((attr & 0x03) << 4) | state & 0x0f) );
		    t++;
		}
	    }
	    else if( x < sx ) {
		if( !saFlag )
		    if( c == ' ' )
			sa++;
		    else
			saFlag++;
		ScrWriteCell( hd, x++ + 1, y+1, c,
			      thisIsCrs? ATTR_CRS : ATTR_BASE+attr );
	    }
	}
    }

    /* Hier noch bis zum Bildende auffllen */
    for( ; y < sy; y++ ) {
	for( x=0; x < sx; x++ )
	    ScrWriteCell( hd, x+1, y+1, ' ', 3 );
    }

    if( idxCount ) {
	idxFlag |= CRS_ANY;   /* any Index Found */
	if( crs->n ) {
	    if( !(idxFlag & CRS_LEFT) )
		idxLeft->n = 0; /* invalid it here too */
	    if( !(idxFlag & CRS_UP) )
		idxUp->n = 0; /* invalid it here too */
	}
    }

    return idxFlag & CRS_SEL;

} /* end ShowEntry() */



#if TEST

#include <wk/timer.h>

static void ShowInfo(int);
static void ShowTick(int);

void main( int argc, char **argv )
{
    int err, hd, c , hd1, hd2, ex, hdInfo;
    unsigned eventWord;

    ErrorStream(stdout);
    ErrorRgText( 2, ScrErrorText );
    if( ScrInitialize(0,0,0,0,0,0,0) )
	Error(2004,"Can't init Screen");
    if( ScrInitDriverStd(0,0) )
	Error(2004,"Can't init ScreenDriver");
    ScrOpen( &hdInfo, 0, 0, 24, 40, 1, 9, 0, 0 );
    EventInit();

    if( err = HelpOpen( &hd1, "h.out", 0, 0, 40, 24 ) )
	Error(2, "HelpOpen: %d", err );
    if( err = HelpGet( hd1, NULL ) )
	Error(2, "HelpGet: %d", err );
    EventPutCmd( K_CMD_EDITSTART );
    HelpEdit(hd1);
  #if 0
    if( err = HelpOpen( &hd2, NULL, 40, 0, 40, 24 ) )
	Error(2, "HelpOpen: %d", err );
    if( err = HelpGet( hd2, NULL ) )
	Error(2, "HelpGet: %d", err );
    EventPutCmd( K_CMD_EDITSTART );
    HelpEdit(hd2);
  #endif

    EventRgIdle( ShowInfo, hdInfo, 0 );

    hd = hd1;
    for(ex=0; !ex; ) {
	if( err = HelpEdit( hd ) )
	    Fatal("HelpEdit returns %d", err );
	eventWord = EventWait();
	if( eventWord & EVT_KBD )
	    c = EventGetKbd();
	else if( eventWord & EVT_CMD )
	    c = EventGetCmd();
	else
	    EventError();

	switch( c ) {
	  case 0: break;
	#if 0
	  case K_F1:
	    hd = hd1;
	    EventClear(EVT_KBD);
	    EventPutCmd( K_CMD_EDITSTART );
	    HelpEdit(hd);
	    break;

	  case K_F2:
	    hd = hd2;
	    EventClear(EVT_KBD);
	    EventPutCmd( K_CMD_EDITSTART );
	    HelpEdit(hd);
	    break;
	#endif

	  case K_F3 :
	     ShowTick(1); /* reset */
	     EventRgIdle( ShowTick, 0, 100 );
	     EventClear(EVT_KBD);
	     break;

	  case K_F4 :
	     EventDgIdle( ShowTick );
	     ShowTick(1); /* reset */
	     EventClear(EVT_KBD);
	     break;

	  case K_F5 :
	     if( err = HelpGet( hd, "Test_NochMehrHilfe" ) )
		 Error(2, "HelpGet: %d", err );
	     EventClear(EVT_KBD);
	     break;

	  case K_F7 :
	     if( err = HelpGet( hd, "WKRC2" ) )
		 Error(2, "HelpGet: %d", err );
	     EventClear(EVT_KBD);
	     break;

	  case K_F9 :
	    if( err = HelpGet( hd, "WKRC" ) )
		 Error(2, "HelpGet: %d", err );
	    EventClear(EVT_KBD);
	    break;

	  case K_ESCAPE:
	    ex++;
	    EventClear(EVT_KBD);
	    break;


	  default:
	    if( EventPoll() & ~EVT_KBD )
		EventError();
	    else
		EventClear(EVT_KBD);
	    break;
	}
    }


    if( err = HelpClose( hd1 ) )
	Error(2, "HelpClose: %d", err );
  #if 0
    if( err = HelpClose( hd2 ) )
	Error(2, "HelpClose: %d", err );
  #endif
    ScrClose( hdInfo );
    ScrUpdate(0);

    exit(0);
}


static void ShowInfo( int hd )
{
    static int hdTimer = -1;
    static int tIs;
    int tstat;

    if( hdTimer == -1 )
	if( (hdTimer = TimerOpen(0))==-1 )
	    Bug("out of Timers");
	else
	    tIs = 0;
    tstat = TimerQuery(hdTimer);
    if( !tstat ) {
	TimerStart( hdTimer, 2000 );
	tIs = !tIs;
    }

    ScrCursor( hd, SCR_CRS_SET, 0, 0 );
    ScrPrintF( hd, EventInfo() );
    ScrPrintF( hd, tIs? " Run" : " Stp" );
    ScrUpdate( -hd );
}




static void ShowTick( int mode ) /*0 = running, 1 = Reset */
{
    static long lastTime;
    static long evtCnt=-1L;
    static int hd, flag;
    long cnt;

    if( mode ) {
	evtCnt =-1L;
	if( flag ) {
	    ScrClose(hd);
	    flag = 0;
	}
    }
    else if( (cnt=EventGetCount(~0)) != evtCnt ){
	if( flag ) {  /* close last window */
	    ScrClose(hd);
	    flag = 0;
	}
	evtCnt = cnt;
	lastTime = -1L;
    }
    else {
	if( lastTime == -1L )
	    lastTime = TimeOfDay();
	cnt = (TimeOfDay() - lastTime)/1000L;
	if( cnt >= 1L ) {
	    if( !flag ) {
		ScrOpen( &hd, 0, -1, -1, 25, 1, 4, SCR_STYLE_DBLFRAME, 0);
		flag++;
	    }
	    ScrCursor( hd, SCR_CRS_SET, 1, 0 );
	    ScrPrintF( hd, "Waiting ... %3ld Seconds", cnt );
	    ScrUpdate( -hd );
	}
    }
}

#endif
/***** bottom of File *****/
