/* [hc.c wk 6.6.91] Help Compiler
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * History:
 * 19.06.93 wk	Version 1.01 (IBM C Set/2)
 */

#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/io.h>
#include <wk/lib.h>
#include <wk/file.h>
#include <wk/string.h>

/****** constants *********/
#if DOS386
#define MAX_KEYINFOS  50000
#define KEYTABLE_SIZE 1000000

#else
#define MAX_KEYINFOS  1000	/* max . numner of Info keys */
#define KEYTABLE_SIZE 20000	/* Size of offset/Key-Table */
				/* ca. 1000 keys w/ avg. Keylen of 15 */
#endif

#define MAX_LINELEN  256	/* max. Sourcefile LineLength */
#define MAX_KEYLEN   80 	/* max. length of a Keys */

#define MAX_TABS 10

/******** typedefs ********/
typedef struct {
	byte  val ;
	long  cnt ;
    } count_t ; 	    /* needed for sorting in analyse */

typedef struct {
	ushort keyNr;	    /* number of key */
	char   *text;	    /* string with key short Info */
    } keyInfo_t;	    /* type to hold key short informations */

/******* globals **********/
static struct {
	int tty;
	int verbose ;
	int cmprMode;
	int quiet;
	int index;
    } opt ;

static char *crfKeyFileName ;
static char *textFileName ;

static byte *keyTable ;
static int   keyTableLen ;

static keyInfo_t *keyInfoTbl;
static size_t keyInfoUsed;

static count_t analyseMap[256] ;
static char    cmprTable[14] ;

/****** prototypes ********/
static void CleanUp(void);
static void ProcFixups( FILE *textSt, FILE *crfSt );
static void ProcAllEntries( FILE *, FILE * );
static void WriteEntry( FILE *, FILE *,long,ushort *, byte *);
static void ProcSrcFile( const char *, FILE *, FILE *);
static void ProcSrcLine( const char *, int *, FILE *, FILE *);
static int AppendKeyTable( const char *, FILE *, ushort *);
static void ChangeKeyTable( ushort , long);
static int ScanKeyTable( const char *, ushort *);
static int ScanKeyTable4Nr( ushort keyNr );
static long WriteKeyTable( long, FILE * );
static void WriteKeyInfo( ushort keyNr, const char *text );
static void AddIndexEntry( FILE *textSt );
static int CompareInfo( const void*, const void*);

static void ResetCmprAna( void );
static int CmprAna( int );
static void CmprWriteInfo( FILE * ) ;
static void CmprWrite( int, FILE * ) ;
static int CompareMapEntry( const void *, const void *);

static void PutNibble( int c, FILE *st );

/***** def. Function ******/
/******* Functions ********/

const char *CopyRight( int level )
{
    const char *p;
    switch( level ) {
      case 10:
      case 0:	p = "HC - Help Compiler Ver. 1.01; "
		    "Copyright (c) 1991 by Werner Koch" ; break;
      case 1:
      case 11:	p = "usage: HC [options] Sourcefiles (-h for help)"; break;
      case 2:
      case 12:	p =
    "Syntax: HC [options] {file}\n\n"
    "Compiles Helpfiles to output file \"h.out\"\n"
    "Options summary:\n"
    " -o<name>[<ext>] = Set output file to \"<name>.exh\"\n"
    " -q    = quiet mode\n"
    " -v    = verbose informations\n"
    " -c<n> = force compressmode n (def. is 1)\n"
    " -ni   = do not create an indexentry\n"
    " -h = help\n";
	break;
      case 13:	p = "HC"; break;
      case 14:	p = "1.01"; break;
      default: p = "?";
    }

    if( !level ) { puts( p ); putchar('\n'); }
    else if( level == 1 ) { fputs( p, stderr ); putc( '\n', stderr );exit(3);}
    else if( level == 2 ) {  /* help */ puts( p ); exit(0); }

    return p;
}



void main( int argc, char **argv )
{
    char   *s ;
    FILE *textSt, *crfKeySt , *outSt ;
    char *outFileName ;
    long keyTableOffset;

    Initialize_Main();
  #ifndef UNIX
    if( ArgExpand( &argc, &argv, 4|1 ) )
	Error(4,GetStr(12));
  #endif

    outFileName = "h.out";
    opt.cmprMode = 1;
    opt.index  = 1;
    for( s=""; --argc && **++argv == '-' && *s != '-'; )
	for( s = *argv + 1 ; *s && *s != '-' ; s++ )
	    switch( *s ) {
	      case 'v': opt.verbose++ ; break ;
	      case 'c': opt.cmprMode = atoi(++s) ;
		while( *s && isdigit(*s) )
		    s++ ;
		s-- ;
		break ;
	      case 'q': opt.quiet++ ; break ;
	      case 'o':
		outFileName = xmalloc( strlen( s+1 )+1+4 ); /* for Extension */
		strcpy( outFileName, s+1 );
		if( !FileExtCheck( outFileName ) )
		    FileExtAppend( outFileName, "exh" );
		while( *s ) s++; s--;
		break;
	      case 'n':
		if( s[1] != 'i' )
		    Error(3,GetStr(15),s );
		else
		    opt.index = 0;
		s++;
		break;

	      case 'h':
	      case '?' : CopyRight(0) ; CopyRight(2); break;
	      default  : Error(3,GetStr(15),s );
	    }
    opt.tty = isatty(fileno(stdin)) && isatty(fileno(stdout)) ;
    if( !argc )
	CopyRight(1);
    if( !opt.quiet )
	CopyRight(0);

    if( !opt.cmprMode || opt.cmprMode == 1 )
	;
    else
	Error(3, "Invalid compress mode %d selected\n"
		 "\tvalid compressmodes are :\n"
		 "\t 0 = no compress\n"
		 "\t 1 = nibble compress", opt.cmprMode);

    atexit( CleanUp ) ;

    keyTable = xmalloc( KEYTABLE_SIZE+sizeof(long) ) ;
    keyTableLen = 0 ;	/* noch nichts eingetragen*/
    memset( keyTable, 255, KEYTABLE_SIZE+sizeof(long) ) ; /* fill with -1L */

    if( opt.index ) {
	keyInfoTbl = xcalloc( MAX_KEYINFOS+1, sizeof *keyInfoTbl );
	keyInfoUsed = 0;
    }

    if( !(textFileName = CreateTmpFile("hct")) )
	Error(1004,GetStr(8)) ;
    if( !(crfKeyFileName = CreateTmpFile("hcc")) )
	Error(1004,GetStr(8)) ;
    textSt   = xfopen( textFileName,   "w+b" ) ;
    crfKeySt = xfopen( crfKeyFileName, "w+b" ) ;


    /* Source Files verarbeiten */
    if( !opt.quiet )
	puts( "[scanning files]" ) ;
    for( ; argc ; argc--, argv++ )
	ProcSrcFile( *argv, textSt, crfKeySt ) ;

    if( opt.index ) {
	if( !opt.quiet )
	    puts( "[creating index]" ) ;
	qsort( keyInfoTbl, keyInfoUsed, sizeof(*keyInfoTbl), CompareInfo ) ;
	AddIndexEntry( textSt );
    }

    if( !opt.quiet )
	puts( "[processing fixups]" ) ;
    ProcFixups( textSt, crfKeySt ) ;


    /* Binary Outputfile erzeugen */
    outSt = xfopen( outFileName , "w+b" ) ;
    fputs( "HL\x01 Copyright (c) 1991 by DD9JN ", outSt);
    putc( '\0', outSt) ;
    CmprWriteInfo( outSt ) ;

    keyTableOffset = WriteKeyTable( -1L, outSt ) ;

    /* textEntry aus textSt verarbeiten */
    if( !opt.quiet )
	puts( "[writing entries]" ) ;
    ProcAllEntries( textSt, outSt ) ;

    /* rewrite the updated keyTable */
    WriteKeyTable( keyTableOffset, outSt ) ;

    if( !opt.quiet )
	printf("Successful compiled to '%s'\n", outFileName);
    exit(0) ;
} /* end main() */



/*
 * atexit() Funktion:
 * die temp. Files lschen
 */

static void CleanUp()
{
    size_t n;

  #ifndef UNIX
    fcloseall() ;
  #endif
    if( crfKeyFileName )
	remove( crfKeyFileName ) ;
    if( textFileName )
	remove( textFileName ) ;
    for(n=0; n < keyInfoUsed; n++ )
	free( keyInfoTbl[n].text );
    free( keyInfoTbl );
    free( keyTable ) ;
}


/*
 *  Loop ber Textfile und crfKeyFile
 *	Lesen des ersten Keys im crfKeyFile
 *	Testen ob dieser key dazu auch in der keyTable ist
 *	Wenn nicht:
 *	    Warnung ausgeben
 *	sonst:
 *	Im textfile Lesen bis der im crfkeyfile angegebene Offset erreicht ist
 *	und dann im textFile an dieser Stelle die Nummer des Keys aus der
 *	Tabelle eintragen.
 *  EndLoop
 */

static void ProcFixups( FILE *txtSt, FILE *crfSt )
{
    char keyBuf[MAX_KEYLEN+1] ;
    int c , i, txtFileState;
    ushort keyNr, dummyNr;
    long txtOff;    /* actual offset of txtSt */
    long off;	    /* offset as found in crfSt */
    long dataLen;   /* length of dataBlock in txtSt */

    rewind( txtSt ) ;
    txtOff = 0L;
    rewind( crfSt ) ;
    ResetCmprAna();

    txtFileState = 0;	/* 0 = in keyNr list */
			/* 1 = Read length of dataBlock */
			/* 2 = read data */
    while( fread( &off, sizeof off, 1, crfSt ) == 1 ) {
	for( i=0; (c = getc(crfSt)) && i < DIM(keyBuf)-1 ; i++ )
	    if( c != EOF )
		keyBuf[i] = (char)c;
	    else
		Bug("Premature EOF on temporary File (crfKeyList)") ;
	keyBuf[i] = '\0' ;
	if( ScanKeyTable( keyBuf, &keyNr ) == -1 )
	    fprintf(stderr, "Warning: Unresolved key: `%s'\n", keyBuf ) ;
	else {
	    /* read in textFile until offset reached */
	    /* and do Compress analysis */
	    while( txtOff < off ) {
		if( !txtFileState ) {
		    if( fread( &dummyNr, sizeof dummyNr, 1, txtSt ) != 1 )
			Bug("Premature EOF on temporary File (text)") ;
		    txtOff += sizeof dummyNr;
		    if( dummyNr == 0xffff )
			txtFileState = 1;
		}
		else if( txtFileState == 1 ) {
		    getc( txtSt ); /* read Flag */
		    txtOff++;
		    if( fread( &dataLen, sizeof dataLen, 1, txtSt ) != 1 )
			Bug("Premature EOF on temporary File (text)") ;
		    txtOff += sizeof dataLen;
		    txtFileState = 2 ;
		}
		else {
		    if( (c=getc(txtSt)) == EOF )
			Bug("Premature EOF on temp.File behind STX (text)") ;
		    txtOff++;
		    if( --dataLen < 0L )
			Bug("Inconsistent textFile ( length invalid )");
		    if( !dataLen ) {
			txtFileState = 0;
			if( c != '\x03' )
			    Bug("Last char isn't ETX but %#x", c );
		    }
		    CmprAna(c);
		}
	    }
	    if( txtFileState != 2 )
		Bug("Invalid temporary File (text)");
	    /* Now Fix up */
	    fseek( txtSt, 0L, SEEK_CUR ); /* switch to Writemode */
	    if( fwrite( &keyNr, sizeof keyNr, 1, txtSt ) != 1 )
		Bug("Error writing fixup (text)") ;
	    fseek( txtSt, 0L, SEEK_CUR ); /* switch back to Readmode */
	    CmprAna( keyNr & 0xff );
	    CmprAna( (keyNr >> 8) & 0xff );
	    txtOff += 2;
	    if( (dataLen -= 2) < 0L )
		Bug("Inconsistent textFile ( length invalid 1)");
	}
    } /* end endless */

    /* read to end of textfile, to complete Compressanalysis */
    while( txtFileState != 3 ) {
	if( !txtFileState ) {
	    if( fread( &dummyNr, sizeof dummyNr, 1, txtSt ) != 1 )
		txtFileState = 3 ; /* EOF */
	    else if( dummyNr == 0xffff )
		txtFileState = 1;
	}
	else if( txtFileState == 1 ) {
	    getc( txtSt ); /* read Flag */
	    if( fread( &dataLen, sizeof dataLen, 1, txtSt ) != 1 )
		Bug("Premature EOF on temporary File (text)") ;
	    txtFileState = 2 ;
	}
	else {
	    if( !dataLen )
		txtFileState = 0;
	    else {
		if( (c=getc(txtSt)) == EOF )
		    Bug("Premature EOF on temp.File behind STX (text)") ;
		CmprAna(c);
		if( --dataLen < 0L )
		    Bug("Inconsistent textFile ( length invalid 3 )");
	    }
	}
    }

}



/*
 *  Loop ber alle Entries im textFile
 *	Key(s) lesen
 *	Keys in keyTable suchen (sind dort vorhanden)
 *	aktuellen Fileoffset in keyTable schreiben
 *	textFile lesen und komprimiert schreiben.
 *  EndLoop
 */

static void ProcAllEntries( FILE *textSt, FILE *outSt )
{
    int  i, first;
    int flag;
    long textLen , saveOff;
    ushort lines;
    byte columns;
    ushort keyNr;

    rewind( textSt ) ;

    first = 1;
    for(;;) {
	if( fread( &keyNr, sizeof keyNr, 1, textSt ) != 1 ) {
	    if( first )
		break; /* for loop; its a true EOF */
	    else
		Bug("Premature EOF on temporary File (text)") ;
	}
	if( keyNr != 0xffff ) {
	    first = 0;
	    ChangeKeyTable( keyNr, ftell(outSt) ) ;
	}
	else {	/* write data*/
	    flag = getc( textSt ); /* read Flag */
	    saveOff = ftell(outSt) ;
	    /* space for header ( lines, cols, flag, len ) */
	    for( i=0; i < 2+1+1+4; i++ )
		putc( '\0' , outSt ) ;
	    fread( &textLen, sizeof textLen, 1, textSt ) ;
	    WriteEntry(textSt, outSt, textLen, &lines, &columns ) ;
	    fseek( outSt, saveOff, SEEK_SET ) ;
	    if( fwrite( &lines, sizeof lines, 1, outSt ) != 1 )
		BUG();
	    putc( columns , outSt ) ;
	    putc( flag ? '\x01' : '\0' , outSt ) ;      /* flag */
	    if( fwrite( &textLen, sizeof textLen, 1, outSt ) != 1 )
		BUG();

	    fseek( outSt, 0L, SEEK_END ) ;
	    first = 1;
	}
    } /* end endless */

}



/*
 * Eine Textentry schreiben,
 * dabei die Zeilen- und Spaltenanzahl bestimmen
 * und in ret_??? zurckgeben.
 * Die Funktion besteht im wesentlichen aus einer abgemagerten
 * Version der Funktion ShowPage aud Help...()
 */

static void WriteEntry( FILE *textSt, FILE *outSt, long textLen,
			ushort *ret_lin, byte *ret_col )
{
    int x , c=0 , state , supChar;
    ushort col;
    ushort y;

    col = 1 ;
    y = 0;
    state = x = 0;

    CmprWrite( 0, NULL ) ;   /* reset Compressor */
    for(state=0;state != 15 ;) {
	if( state == 5 )
	    state = 0 ;
	else {
	    c = getc(textSt);		/* read ... */
	    if( --textLen < 0L )
		Bug("Invalid length of tempFile (text); state=%d", state);
	    CmprWrite( c, outSt ) ;	/* and write compressed to output */
	}
	if( c == EOF )
	    Bug("Premature EOF on tempFile (text); state=%d", state) ;
	supChar = 0;
	switch( state ) {
	  case 0:
	    switch( c ) {
	      case '\x03':  state = 15;            break; /* end of text */
	      case '\x10':  state =  1; supChar++; break; /* write next character */
	      case '\x11':  state =  2; supChar++; break; /* index escape */
	      case '\x12':  state =  3; supChar++; break; /* Formatierkennzeichen folgt */
	      case '\x13':  state =  4; supChar++; break; /* Attribut folgt */
	    }
	    break ;

	  case 1:   /* das Zeichen transparent ausgeben */
	    state = 0 ;
	    break ;

	  case 2:   /* Index Key  byte 1*/
	    state = 8;
	    supChar++;
	    break;

	  case 8:   /* Index Key  byte 2 */
	    state = 6 ;
	    break ;

	  case 6:   /* Wort nach index key */
	    if( isspace(c) || c == '\x03' ) {
		state = 5 ;
		supChar++ ;
	    }
	    break ;

	  case 3:   /* Formatierkennzeichen */
	    state = 0 ;
	    supChar++;
	    break ;

	  case 4:   /* Attribut setzen */
	    state = 0 ;
	    supChar++;
	    break ;

	  default: Bug("Invalid state=%d", state ) ;
	}

	if( !supChar ) {
	    if( (!state && c == '\n') || state == 15 ) {
		if( x > col )
		    col = x ;
		y++;
		x=0;
	    }
	    else
		x++;
	}

    } /* end of state machine */

    CmprWrite( EOF, outSt ) ;	/* flush Compressor */
    if( textLen )
	Bug("Premature end of text-entry; remaining length=%ld", textLen );

    *ret_lin = y ;
    if( col > 254 )
	Bug("Too many columns in entry");
    *ret_col = (byte)(col+1) ;
}





/*
 * Eine Sourcefile verarbeiten
 */

static void ProcSrcFile( const char *fileName, FILE *textSt, FILE *crfKeySt )
{
    FILE *st ;
    char *buf ;
    int tabPosHlp ;
    int status , doEnd , textStatus, firstLine=0;
    int err , i ;
    long textEntryOff=0, hl ;
    int lines, columns, pages ;
    ushort keyNr;

    err = 0 ;
    buf = xmalloc( 1+MAX_LINELEN+1 ) ;	/* one more to insert LF */
    if( !(st=fsopen( fileName , "t" )) )
	Error(1000,"Skipping file '%s'", fileName) ;
    else {
	if( opt.verbose )
	    printf( "Scanning '%s'\n", fileName  ) ;

	/* loop over all lines expand tabs while reading line */
	status = 0 ;	/* waiting for an entry*/
	doEnd = 0 ;
	tabPosHlp = 0 ;
	while( FTabGets( buf, MAX_LINELEN, st, "", 0, "\n", 1, &tabPosHlp) ) {
	    if( *buf == '@' ) { /* key encountered*/
		if( doEnd )  { /*Endbehandlung*/
		    ProcSrcLine( NULL, &textStatus, textSt, crfKeySt);
		    hl = ftell(textSt) - textEntryOff - sizeof(long);
		    fseek( textSt, textEntryOff, SEEK_SET ) ;
		    fwrite( &hl, sizeof hl, 1, textSt ) ;
		    fseek( textSt, 0L, SEEK_END ) ;
		    doEnd = 0 ;
		    status = 0 ;
		}

		if( !buf[1] || isspace(buf[1]) ) /* No Key */
		    status = 0 ;
		else if( !strlen(strtok( buf+1, " ,\n\t" )) )
		    fprintf(stderr,"Invalid Key `%s' - skipped\n", buf+1 );
		else if( AppendKeyTable(buf+1, textSt, &keyNr ) )
		    fprintf(stderr, "Duplicate key `%s' - skipped\n",buf+1);
		else {
		    if( opt.verbose )
			printf("         `%s'\n", buf+1 ) ;
		    status = 1 ;    /* wait for next key*/
		    if( opt.index )
			WriteKeyInfo( keyNr, strtok( NULL, "\n" )  );
		}
	    }
	    else {  /* not in a keyline*/
		if( status == 1 ) {
		    keyNr = 0xffff ; /* Marker for end of keylist */
		    if( fwrite( &keyNr, sizeof keyNr, 1, textSt ) != 1 )
			Error(1002,"Error writing tempFile") ;
		    putc( '\0', textSt ); /* write Flag */
		    textEntryOff = ftell( textSt ) ; /* for later fixup */
		    hl = 0L ;
		    fwrite( &hl, sizeof hl, 1, textSt ) ;
		    putc( '\x02' /* STX */, textSt ) ;
		    status = 2 ;  /* kein Key sondern Textentry*/
		    firstLine = 1 ;
		    textStatus = 0 ; /* for Statemachine of ProcSrcLine()*/
		    lines = columns = pages = 1 ;
		    doEnd = 1 ;
		}
		if( status == 2 ) {
		    /* remove lf from lastline */
		    if( i = strlen(buf) )
			buf[--i] = '\0' ; /* remove LF */
		    if( firstLine )
			firstLine = 0 ;
		    else {
			memmove( buf+1, buf, i ) ;
			buf[0] = '\n' ;
		    }
		    ProcSrcLine( buf, &textStatus, textSt, crfKeySt);
		}
	    }
	} /* end loop over all lines */

	if( doEnd )  { /* letzte Endbehandlung*/
	    ProcSrcLine( NULL, &textStatus, textSt, crfKeySt);
	    hl = ftell(textSt) - textEntryOff - sizeof(long);
	    fseek( textSt, textEntryOff, SEEK_SET ) ;
	    fwrite( &hl, sizeof hl, 1, textSt ) ;
	    fseek( textSt, 0L, SEEK_END ) ;
	}

	fclose(st) ;
    }
    free( buf ) ;
}


/*
 * Eine Zeile auf dem Textstrem ausgeben, dabei die bersetzung
 * durchfhren, status gibst die Adresse eine Var. an, in der der
 * Status der Statemachine gespeichert wird ( beim ersten Aufruf
 * sollte diere auf 0 stehen.
 * Ist Buffer == NULL, so wird die Ausgabe ordnugsgemss beendet,
 * d.h. die Statemachine wird auf Init-State gebracht und ETX ausgegeben.
 * States:
 *  0 = Init
 *  1 = Escape Charcter ^ found
 *  2 = Escape Sequence ^. found
 *  3 = Crossreference Sequence ^# found
 *  4 = AttributKennzeichen und c ausgeben esc auf 0 setzen
 *  5 = FormatierKennzeichen und c ausgeben esc auf 0 setzen
 *  6 = keine Ausgabe, esc auf 0
 *  7 = Sequence ^x erkannt
 *  8 = Sequence ^xn erkannt
 *  9 = Transparentkennzeichen und c ausgeben und esc auf 0 setzen
 * 10 = CrfKey-Flag, crfKey und 0x00 ausgeben esc auf 0 setzen
 * 11 = FormatierKennzeichen und c ausgeben esc erhhen
 * 12 = Tabulatorliste warten auf Zeichen
 * 13 = Tabulatorliste Zeichen ausgeben
 * 14 = Tabulatorliste Zeichen und Endekennung ausgeben esc auf 0
 */

static void ProcSrcLine( const char *line, int *status,
			 FILE *textSt, FILE *crfKeySt )
{
    int c , state , chval=0 ;
    char crfKey[MAX_KEYLEN+1] ;
    int  crfKeyLen=0 ;
    long off;

    if( !line ) {
	/* falls nicht im init state: auf initstate bringen */
	/* ein paar Blanks sollten dazu ausreichen */
	if( *status )
	    ProcSrcLine( "    ", status, textSt, crfKeySt);
	if( *status )
	    Bug("Error getting down to init state" ) ;
	putc( '\x03' /*ETX*/ , textSt ) ;
	line = "" ; /* leere zeile erzwingt abbruch*/
    }

    state = *status ;
    while( c = *(line++) ) {
	switch( state ) {
	  case 0 :
	    if( c == '^' )
		state = 1 ;
	    else if( c == '\x02' || c == '\x03' )
		state = 6 ; /* STX und ETX unterdrcken*/
	    break ;

	  case 1 :
	    switch( c ) {
	      case '^' : state = 0 ; break ;
	      case '@' : state = 0 ; break ;
	      case '.' : state = 2 ; break ;
	      case '#' : state = 3 ; crfKeyLen = 0 ; break ;
	      case 'x' :
	      case 'X' : state = 7 ; break ;
	      case 'N' :
	      case 'n' : state = 4 ; c = 0 ; break ;
	      case 'R' :
	      case 'r' : state = 4 ; c = 1 ; break ;
	      case 'B' :
	      case 'b' : state = 4 ; c = 2 ; break ;
	      case 'U' :
	      case 'u' : state = 4 ; c = 3 ; break ;
	    #if 0  /* kein Tabulator z.Z. */
	      case 'T' :
	      case 't' : state = 0 ; c = '\x14' ; break ;
	    #endif
	      default  : state = 6 ;  break ; /* unterdrcken*/
	    }
	    break ;

	  case 2 :
	    switch( c ) {
	      case 'n':
	      case 'N': state = 5 ; c = 0 ; break ;
	      case 'l':
	      case 'L': state = 5 ; c = 1 ; break ;
	      case 'r':
	      case 'R': state = 5 ; c = 2 ; break ;
	      case 'c':
	      case 'C': state = 5 ; c = 3 ; break ;
	    #if 0 /* kein Tabulator z.Z. */
	      case 't':
	      case 'T': state = 11 ; c = 4; chval = 0; break ;
	    #endif
	      default : state = 6 ;  break ; /* unterdrcken*/
	    }
	    break ;

	  case 3 :  /* CrfKey Sequence erkannt*/
	    if( isspace(c) )
		state = 6 ;   /* crfKey verwerfen*/
	    else if( c == '#' ) { /* Ende des Keys*/
		crfKey[crfKeyLen++] = '\0' ;
		off = ftell( textSt ) + 1; /* 1 is for Index - Marker */
		fwrite( &off, sizeof off, 1, crfKeySt );
		fwrite( crfKey, crfKeyLen, 1 , crfKeySt ) ;
		state = 10 ;
	    }
	    else if( crfKeyLen < DIM(crfKey)-1 )
		crfKey[crfKeyLen++] = (char)c ;
	    break ;

	  case 7 :
	    if( isxdigit( c ) ) {
		chval = (isdigit(c)? c - '0': toupper(c)-'A'+10) * 16 ;
		state = 8 ;
	    }
	    else
		state = 6 ; /* unterdrcken*/
	    break ;

	  case 8 :
	    if( isxdigit( c ) ) {
		c = chval + (isdigit(c)? c - '0': toupper(c)-'A'+10) ;
		state = 9 ;
	    }
	    else
		state = 6 ; /* unterdrcken*/
	    break ;

	  case 12 : /* Tabulator liste aufbauen */
	  case 13 :
	    state = 12 ;
	    if( isdigit(c) )
		chval = chval*10 + c - '0' ;
	    else if( c == ',' || c == '\n' || c == ';' ) {
		if( chval < 1 && chval > 255 )
		    Error(0,"Tabposition must be in range 1..255;"
			      " it's: %d - ignored", chval );
		else {
		    state = c == ',' ? 13:14 ;
		    c = chval-1 ;
		}
		chval = 0 ;
	    }
	    break ;

	  default :
	    Bug( "Invalid state %d", state ) ;
	}

	switch( state ) {
	  case 0 : /* Normal ausgeben*/
	  case 13:
	    putc(  c , textSt ) ;
	    break ;

	  case 9 : /* TransparentKz und c ausgeben*/
	    putc( '\x10' , textSt ) ;
	    putc( c , textSt ) ;
	    state = 0 ;
	    break ;

	  case 14:  /* Tabulator ende ausgeben */
	    putc( c  , textSt ) ;
	    putc( '\xff'  , textSt ) ;
	    state = 0 ;
	    break ;

	  case 4 : /* AttributKz und c ausgeben*/
	     putc( '\x13'  , textSt ) ;
	     putc(  c  , textSt ) ;
	     state = 0 ;
	    break ;

	  case 5 : /* FormatKz und c ausgeben*/
	  case 11:
	     putc( '\x12'  , textSt ) ;
	     putc(  c  , textSt ) ;
	     state = state == 5 ? 0 : 12 ;
	    break ;

	  case 6: /* keine Ausgabe und state reset*/
	    state = 0 ;
	    break ;


	  case 10 : /* kz fr index + Dummy Bytes fr Crf Key Number ausgeben*/
	    putc( '\x11'  , textSt ) ;
	    putc( '\xff'  , textSt ) ;
	    putc( '\xff' , textSt ) ;
	    state = 0 ;
	    break ;
	}

    }
    *status = state ;

} /* end ProcSrcLine() */


/*
 * Den angegeben Key in die Keytabelle Eintragen:
 * Wenn der Key ereits dort vorhanden ist:
 *	Nicht eintragen
 * Wenn der Key noch nicht vorhanden ist:
 *	Key in die Keytabelle eintragen
 *	KeyNr auf dem TextSt ausgeben
 *
 * Returns: 0 = Key ist eingetragen worden
 *	   !0 = duplicate Key
 */

static int AppendKeyTable( const char *key, FILE *textSt, ushort *keyNr )
{
    int len ;
    long hl ;

    if( ScanKeyTable( key, keyNr ) == -1 ) {
	/* Okay: Key is not in table */
	len = strlen(key)+1 ;
	if( sizeof(long)+len < KEYTABLE_SIZE-sizeof(long) ) {
	    if( fwrite( keyNr, 2, 1, textSt ) != 1 )
		Error(1002,"Error writing tempFile") ;
	    hl = 0L ;
	    memcpy( keyTable+keyTableLen, &hl, sizeof hl ) ;
	    keyTableLen += sizeof hl ;
	    memcpy( keyTable+keyTableLen, key, len ) ;
	    keyTableLen += len ;
	}
	else
	    Error(2,"KeyTable overflow (tablesize="STRING(KEYTABLE_SIZE)")");
	return 0 ;
    }
    else
	return -1 ; /* duplicate key */
}



/*
 * Den angegeben offset zum Key in der Keytabelle eintragen
 * ( der Key ist dort vorhanden )
 */

static void ChangeKeyTable( ushort keyNr, long offset )
{
    int pos ;

    pos = ScanKeyTable4Nr( keyNr );
    if( pos < 0 || pos > KEYTABLE_SIZE - sizeof(long) )
	Bug("KeyTable and tempFile inconsistent") ;
    memcpy( keyTable+pos, &offset, sizeof offset ) ;
}



/*
 * Suchen in der Tabelle nach dem Key  (wie ScanOffsetTbl() aber im Speicher)
 * in retNumber wird die Nummer des Eintrags zurckgegeben.
 * Returns: Offset des Fileoffsets des Keys in der Tabelle
 *	    oder -1, wenn nicht vorhanden
 */

static int ScanKeyTable( const char *key, ushort *retNumber )
{
    long off ;
    const char *p ;
    byte *tblp ;
    int c , ret ;

    *retNumber = 0;
    off = -1L ;
    tblp = keyTable ;
    for(;;) {
	ret = tblp - keyTable ;
	memcpy( &off, tblp, sizeof(long)) ;
	tblp += sizeof(long) ;
	if( off == -1L ) /* ende der tabelle */
	    break ;
	for( p=key ; (c = *tblp++) == *p  ; p++ )
	    if( !*p )
		break ;
	if( !*p && !c )
	    break ;
	if( c )
	    while( *tblp++ )
		;
	++*retNumber;
    }
    return off == -1L ? -1 : ret ;
}


/*
 * Suchen in der Tabelle nach dem Eintrag mit der Nummer keyNr.
 * Returns: Offset des Fileoffsets des Keys in der Tabelle
 *	    oder -1, wenn nicht vorhanden (sollte nicht vorkommen)
 */

static int ScanKeyTable4Nr( ushort keyNr )
{
    long off ;
    byte *tblp ;
    ushort number;

    tblp = keyTable ;
    for(number=0; number != keyNr ; ++number) {
	memcpy( &off, tblp, sizeof(long)) ;
	tblp += sizeof(long) ;
	if( off == -1L ) /* end of table - not found */
	    return -1;
	while( *tblp++)
	    ;
    }
    return tblp - keyTable ;
}


/*
 * Die Keytable auf den angegebenen Stream schreiben
 * Dabei vorher auf den angegebenen Offset zurckspulen
 * ist der angegebene Offset -1L, so nicht spulen, sondern
 * den aktuellen Offset benutzen.
 * Returns: off = fileoffset des Tabellenstarts im File
 *
 */

static long WriteKeyTable( long offset, FILE *st )
{
    long hl ;

    if( offset == -1L )
	offset = ftell(st);
    else if( fseek( st, offset, SEEK_SET ))
	Error(1002,GetStr(7), "[OutFile]") ;

    if( fwrite( keyTable, keyTableLen, 1 , st ) != 1 )
	Error(2,GetStr(3), "[OutFile (1)]" );
    hl = -1L ;	/* end of table flag*/
    if( fwrite( &hl, sizeof hl, 1 , st ) != 1 )
	Error(2,GetStr(3), "[OutFile (2)]" );
    hl = 0L ;	/* reserved Table flag*/
    if( fwrite( &hl, sizeof hl, 1 , st ) != 1 )
	Error(2,GetStr(3), "[OutFile (3)]" );

    return offset ;
}



/*
 * KeyEintrag in die keyinfoTbl schreiben
 * Parms: Keyzeile ( ohne Key )
 *	  keyNr
 */

static void WriteKeyInfo( ushort keyNr, const char *text )
{
    size_t len, n;
    char *p;

    if( keyInfoUsed >= MAX_KEYINFOS )
	Error(2,"Too many keys for index" );

    if( text ) {
	for( ; *text && isspace(*text); text++ )
	    ;
	for( len = strlen(text); len ; len-- )
	    if( !isspace( text[len-1] ) )
		break;
    }
    else
	len = 0;

    if( !len ) {
	text = "[Unkown]";
	len  = 8;
    }
    if( len ) {
	p = xcalloc( len+1, 1 );
	for( n=0; n < len; n++ )
	    p[n] = text[n] == ' ' ? '\xfa' : text[n] ;
	keyInfoTbl[keyInfoUsed].text = p;
    }
    keyInfoTbl[keyInfoUsed].keyNr = keyNr;
    keyInfoUsed++;
}


/*
 * Aus der KeyInfoTable einen Entry aufbauen und diesen
 * auf den textSt schreiben ( wird als letzter Eintrag geschrieben )
 * Dabei wird direkt das Format des textSt erzeugt.
 */

static void AddIndexEntry( FILE *textSt )
{
    static char idxKey[] = "$INDEX$";
    ushort keyNr;
    long entryOff, hl;
    size_t n;
    keyInfo_t *ki;

    if( AppendKeyTable(idxKey, textSt, &keyNr ) )
	Error(2,"Indexentry key `%s' already used\n",idxKey);
    keyNr = 0xffff ; /* Marker for end of keylist */
    if( fwrite( &keyNr, sizeof keyNr, 1, textSt ) != 1 )
	Error(1002,"Error writing tempFile") ;
    putc( '\x01', textSt ); /* write Flag: is an index */
    entryOff = ftell( textSt ) ; /* for later fixup of length */
    hl = 0L ;
    fwrite( &hl, sizeof hl, 1, textSt ) ;
    putc( '\x02' /* STX */, textSt ) ;

    for(n = 0, ki=keyInfoTbl; n < keyInfoUsed; n++, ki++ ) {
	if( n )  /* we don't want the LF in the last line, so do it here*/
	    putc( '\n', textSt );
	putc( ' ', textSt );
	putc( '\x11', textSt );
	fwrite( &ki->keyNr, sizeof(ushort), 1, textSt );
	fputs( ki->text, textSt );
    }
    putc( '\x03' /* ETX */, textSt ) ;
    hl = ftell(textSt) - entryOff - sizeof(long);
    fseek( textSt, entryOff, SEEK_SET ) ;
    fwrite( &hl, sizeof hl, 1, textSt ) ;
    fseek( textSt, 0L, SEEK_END ) ;
}



/*
 * Vergleichsfunktion fr keyInfo_t (used by qsort())
 */

static int CompareInfo( const void*a, const void*b )
{
    return strcmpl( ((keyInfo_t*)a)->text, ((keyInfo_t*)b)->text ) ;
}



/*
 * Den Analyse Buffer fr die Kompression zurcksetzen
 */

static void ResetCmprAna()
{
    int i ;

    for(i=0; i < 256 ; i++ )
	analyseMap[i].val = (byte)i ;
}


/*
 * Den Kompressionsbuffer weiterschreiben mit dem Zeichen c
 * Returns: das Zeichen c
 */

static int CmprAna( int c )
{
    analyseMap[ c & 0xff ].cnt++ ;
    return c ;
}



/*
 *
 * Die Kompressinfos bestimmen und schreiben
 */

static void CmprWriteInfo( FILE *st )
{
    int i ;

    qsort( analyseMap, 256, sizeof(*analyseMap), CompareMapEntry ) ;
    for( i=0; i < 14 ; i++ )
	cmprTable[i] = analyseMap[i].val ;
    if( !opt.cmprMode )
	putc( 0, st ) ;
    else if( opt.cmprMode == 1 ) {
	putc( '\x01', st );
	for(i=0; i < 14 ; i++ )
	    putc( cmprTable[i], st );
    }
    else
	BUG();
}


/*
 * Ein Byte komprimiert ausgeben
 * wird fr c EOF angegeben , so wird ein reset durchgefhrt
 */

#define MIN_RPT     2		/* ab MIN_RPT Zeichen lohnt es sich*/
#define MIN_IRPT    4
#define MAX_RPT     (15+MIN_RPT)    /* max. Anzahl von Wiederholungen*/
#define MAX_IRPT    (15+MIN_IRPT)

static void CmprWrite( int c, FILE *st )
{
    int i ;
    static int rptcnt;	       /*Anzahl zu wiederholenender Zeichen*/
    static int rptflag;        /* Zeichen ist normal*/
    static int rptchr;	       /* Zeichen*/
    static int rptind;	       /* falls !rptflag: Index in tabelle*/
    static int rptmax;	       /* max. mgliche Anzahl von wiederholungen*/

    if( !opt.cmprMode ) {
	if( c != EOF && st )
	    putc( c, st ) ;
	/* reset und Flush nicht notwendig */
    }
    else { /* nibble Compress */
	if( !st )  { /* reset */
	    PutNibble( -1, NULL ) ; /* reset Buffer */
	    rptcnt = 0 ;
	    rptchr = -1 ;
	}
	else {	/* compress one character */
	    if( rptchr == c && rptcnt < rptmax && c != EOF )
		rptcnt++ ;
	    else {
		if( rptcnt ) {	/* anything to output*/
		    if( rptflag ) {
			if( rptcnt < MIN_RPT )
			    for(i=0; i < rptcnt ; i++ ) {
				PutNibble( 0x0f, st ) ;
				PutNibble( (rptchr & 0xf0)>>4, st ) ;
				PutNibble( rptchr & 0x0f, st ) ;
			    }
			else {
			    PutNibble( 0x0e, st ) ;
			    PutNibble( rptcnt-MIN_RPT, st ) ;
			    PutNibble( 0x0f, st ) ;
			    PutNibble( (rptchr & 0xf0)>>4, st ) ;
			    PutNibble( rptchr & 0x0f, st ) ;
			}
		    }
		    else {
			if( rptcnt < MIN_IRPT )
			    for(i=0; i < rptcnt ; i++ )
				PutNibble( rptind, st ) ;
			else {
			    PutNibble( 0x0e, st ) ;
			    PutNibble( rptcnt-MIN_IRPT, st ) ;
			    PutNibble( rptind, st ) ;
			}
		    }
		}

		if( c != EOF ) {
		    for(i=0; i < 14; i++ )
			if( c == cmprTable[i] )
			    break ;
		    if( i < 14 ) {
			rptind = i ;
			rptflag = 0 ;
			rptmax	= MAX_IRPT ;
		    }
		    else {
			rptflag = 1 ;
			rptmax	= MAX_RPT ;
		    }
		    rptchr = c ;
		    rptcnt = 1 ;    /* 1 Zeichen haben wir schon*/
		}
		else {
		    PutNibble( 0x0e, st ) ;  /* issue EOF-Flag ( E 0 E )*/
		    PutNibble( 0x00, st ) ;
		    PutNibble( 0x0e, st ) ;
		    PutNibble( -1, st ) ;    /* flush last nibble*/
		}
	    }
	} /* end compress one character */
    } /* end nibble compress */
}



/*
 * Ausgabe eines Nibbles
 * Sonderfunktion wenn c == 1 :
 *	falls ein Stream angegeben ist, so wird geflushed
 *	ansonsten interner reset
 */

static void PutNibble( int c, FILE *st )
{
    static int nib, cnt ;

    if( c == -1 ) { /* flush*/
	if( st && cnt )
	    putc( nib << 4, st ) ;
	cnt = 0 ;
    }
    else if( !cnt ) {
	nib = c ;
	cnt++ ;
    }
    else {
	c = nib << 4 | c ;
	putc( c, st ) ;
	cnt = 0 ;
    }
}


/*
 * Vergleichsfunktion fr count_t (used by qsort())
 */

static int CompareMapEntry( const void *a, const void *b )
{
    if( ((count_t*)a)->cnt < ((count_t*)b)->cnt )
	return 1 ;
    else if( ((count_t*)a)->cnt > ((count_t*)b)->cnt )
	return -1 ;
    else
	return 0 ;
}


/*************************************************************
 ******** weiter Infos ***************************************
 *************************************************************
 *  Algorithmus:
 * -------------
 *  Tabelle:   keyTable <-- reset
 *  temporre Files: crfkeyFile und textFile erzeugen
 *  Loop
 *	nchsten Sourcefile einlesen:
 *	Auf ersten Key warten
 *	Loop
 *	    Key(s) noch nicht in keyTable
 *		Key bzw direkt folgenden Keys in keyTable mit Offset 0
 *		    speichern
 *		Key(s) als Header auf textFile wegschreiben
 *		KeyNr ermitteln und diese mit der Keykurzinfo
 *		in InfoTabelle schreiben.
 *		STX schreiben
 *		File umwandeln und auf textFile wegschreiben
 *		Wenn darin Indexkey gefunden:
 *			auf crfkeyFile zusammen mit key und Offset
 *			im TextFile schreiben.
 *			und 2 byte ( 0xffff ) in den Textfile als Platz
 *			halter schreiben.
 *		ETX schreiben
 *	    auf nchsten Key oder EOF warten
 *	EndLoop
 *  EndLoop
 *
 *  Statistik fr TextFile ein
 *  Loop ber Textfile
 *	Lesen des ersten Keys im crfKeyFile
 *	Testen ob dieser key dazu auch in der keyTable ist
 *	Wenn nicht:
 *	    Warnung ausgeben
 *	sonst:
 *	Im textfile Lesen bis der im crfkeyfile angegebene Offset erreicht ist
 *	und dann im textFile an dieser Stelle die Nummer des Keys aus der
 *	Tabelle eintragen.
 *  EndLoop
 *  Statistik fr TextFile aus
 *
 *  Library neu erzeugen
 *  Library Header schreiben
 *  keyTable schreiben um Platz im File zu reservieren
 *  Kompressionsmethode aus Statistik bestimmen
 *  und CompressWrite initialisieren
 *  KompressionsInfoblock schreiben
 *  Loop ber alle Entries im textFile
 *	Key(s) lesen
 *	Keys in keyTable suchen (sind dort vorhanden)
 *	aktuellen Fileoffset in keyTable schreiben
 *	textFile lesen und komprimiert schreiben.
 *  EndLoop
 *  keyTable erneut im File abspeichern.
 *
 *  cleanUp
 *  eoj
 *
 *  Aufbau eines HelpFiles:
 * -------------------------
 *  'HL'    Kennnung fr Helpfile
 *  0x01    interne Versionsinformation
 *  asciiz  Copyright
 *  n bytes Kompressinfoblock (n hngt von Kompressionsverfahren ab)
 *	    (wenn das erste byte den Wert 0 hat, so folgen keine weiteren
 *	     Bytes)
 *  {
 *    long     Offset des Entries im File
 *    string   Helpkey ( C-String bel. Lnge incl. 0x00 )
 *  }
 *  long = -1L	Ende kennung
 *  long = 0L	reserviert
 *  --- ab hier hintereinander die Entries
 *    ushort: Number of lines
 *    byte  : Max. Number of columns
 *    byte  : Flag: Bit 0 set: this is an index entry
 *    long  : # of bytes im folgenden Block
 *	 ab hier komprimiert:
 *	4. Byte: STX
 *	n Bytes text
 *	4+n+1. byte ETX
 *
 *  Aufbau des temp. File textFile
 * --------------------------------
 *  {
 *	ushort*n    Nummer der Keys aus der Tabelle
 *	ushort=0xffff  Ende Kennung
 *	byte	    Flag
 *	long	    Lnge des folgenden Datenblocks
 *	n byte	    Data inclusive STX, ETX
 *  }
 *
 * Aufbau des temp. File crfKeyFile
 * ---------------------------------
 *  {
 *	long	    offset dieses Keys im textFile (d.h der
 *		    beiden Platzhalter bytes )
 *	String	    mit dem KEy (C-String)
 *  }
 */

/** bottom of file **/
