/* [event.c wk 17.6.91] Event Handler
 *	Copyright (c) 1991 by Werner Koch (dd9jn)
 * History:
 * 20.06.93 wk	Changes for OS/2, curses etc
 * 14.07.93 wk	KbdFlag eingebaut (wg. nicht pollfaehigen kbds)
 */


#include <wk/tailor.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#define INCL_ALT_KEYCODES 1
#include <wk/lib.h>
#include <wk/keys.h>
#include <wk/mouse.h>
#include <wk/event.h>
#if MSDOS || DOS386
#include <bios.h>
#include <wk/bios.h>
#elif defined(OS2)
    #define INCL_DOSPROCESS 1
    #define INCL_KBD 1
    #if OS20
       #define INCL_DOSDEVIOCTL 1
    #endif
    #include <os2.h>
#endif


#if  MOUSE_LEFT != EVT_MS_BT_L || MOUSE_RIGHT != EVT_MS_BT_R ||      \
     MOUSE_LEFTDBL != EVT_MS_BT_LD || MOUSE_RIGHTDBL != EVT_MS_BT_RD
  #error value mismatch between MOUSE_xxxx and EVT_MS_yyyy
#endif

/**** constants ***/
#define MAX_IDLEFNC 5

/**** types ******/
typedef struct {
	void (*fnc)(int);   /* pointer to function or NULL if unused */
	long   tslc;	    /* time slice to call function */
	int    fncParm;
	long lsttm;
    } idlefnc_t;

/**** globals ****/
static unsigned eventWord;  /* holds the actual EventWord */
static unsigned lastKey;
static int lastCommand;
static idlefnc_t idlefnc[MAX_IDLEFNC];
static int msOkay;	   /* Okay: we got a mouse */
static int mouseX, mouseY, lstmsX, lstmsY;
static unsigned mouseBtns;
static int msCheckMove;

static long eventCntKbd;
static long eventCntCmd;
static long eventCntMs;

static int DefaultKbdFnc(void);
#if OS2 /* polls */
  #define DEFAULT_KBDFNCFLAG 1
#elif defined(UNIX) /* does not poll */
  #define DEFAULT_KBDFNCFLAG 0
#else  /* polls */
  #define DEFAULT_KBDFNCFLAG 1
#endif


static int (*KbdFnc)(void) = DefaultKbdFnc;
static unsigned kbdFncFlag = DEFAULT_KBDFNCFLAG;
static int kbdSkipCount;


/*** local Protos ***/
/**** def'd Functions ****/
/**** Global Functions ****/

#ifdef DOCUMENTATION
@Summary EventInit
 #include <wk/event.h>

 void EventInit(void);
@Description
 Sollte Anfangs einmal aufgerufen werden um das EventSystem zu initialisieren.
 Das Screensystm ist vorher zu initialisieren, da die Daten des Videomodes
 bentigt werden.
#endif /*DOCUMENTATION*/

void EventInit()
{
    static int isInitialized;

    if( !isInitialized ) {
	msOkay = MouseInitialize();
	isInitialized++;
    }
}


long EventGetCount( unsigned mask )
{
    long ret;

    ret = 0;
    if( mask & EVT_KBD ) {
	ret += eventCntKbd;
	eventCntKbd = 0;
    }
    if( mask & EVT_MS ) {
	ret += eventCntMs;
	eventCntMs = 0;
    }
    if( mask & EVT_CMD ) {
	ret += eventCntCmd;
	eventCntCmd = 0;
    }
    return ret;
}


void EventRegisterKbd( int (*f)(void), unsigned flag )
{
    KbdFnc = f ? f : DefaultKbdFnc;
    kbdFncFlag = f ? flag : DEFAULT_KBDFNCFLAG;
}


#ifdef DOCUMENTATION
@Summary EventPoll
 #include <wk/event.h>

 unsigned EventPoll(void);
@Description
 Dies Funktion sollte von Zeit zu Zeit aufgerufen werden, um in
 bestimmten Environments ( z.B. MSDOS ) dem System die Moeglichkeit zu geben,
 neue Events zu registrieren.
@Return Value
 Es wird das EventStatus Word zurckgegeben.
 Dies ist Bitweise codiert und Kann mit den Constanten
 EVT_... abgefragt Ierden. ist diese Word 0, so liegen keine Events vor.
    EVT_SIG  - Ein Signal ist registriert
    EVT_MS   - Ein Mouse Event ist registriert
    EVT_KBD  - Eine Taste wurde bettigt.
    EVT_CMD  - Ein Command wurde registriert.
@See Also
 EventClear  EventWait
@Notes
@Example
#endif /*DOCUMENTATION*/

unsigned EventPoll()
{
    int c, x, y;
    unsigned btns;


  #if MSDOS || DOS386 || DOS16RM
    /* special for Interrupts etc .. */
    if( event_Int1BPending ) {
	eventWord |= EVT_SIG;
	event_Int1BPending = 0;
      #if MSDOS || DOS386
	bioskey(0);		/* remove Dummy (0x0000) entry */
      #endif
	raise( SIGINT );
    }
  #endif

    /* now look at the keyboard */
    if( kbdFncFlag&1) { /* kbd is able to poll */
	if( c = KbdFnc() ) {
	    if( !lastKey ) {
	      #if MSDOS || DOS386 || DOS16RM
		if( c == 0x03 ) {  /* Control-C ? */
		    eventWord |= EVT_SIG;
		    raise(SIGINT);
		}
		else {
	      #endif
		    eventCntKbd++;
		    eventWord |= EVT_KBD;
		    lastKey = c;
	      #if MSDOS || DOS386 || DOS16RM
		}
	      #endif
	    }
	}
	else
	    Sleep(0);	/* be nice */
    }
    else { /* kbdfnc() waits for a keystroke */
	if( lastKey )
	    ; /* skip */
	else if( kbdSkipCount < 4 ) /* jeweils ein paar mal leer durchlaufen */
	    kbdSkipCount++;
	else if( c = KbdFnc() ) {
	    kbdSkipCount = 0;
	    eventCntKbd++;
	    eventWord |= EVT_KBD;
	    lastKey = c;
	}
    }

    /* and lets have a look at the mouse */
    if( msOkay ) {
	btns = MouseGet( &x, &y );
	if( btns ^ mouseBtns ) { /* button state changed */
	    if( btns & (MOUSE_LEFT | MOUSE_LEFTDBL) )
		eventWord |= EVT_MSLB;
	    if( btns & (MOUSE_RIGHT | MOUSE_RIGHTDBL) )
		eventWord |= EVT_MSRB;
	    mouseBtns = btns;
	    msCheckMove = btns & MOUSE_LEFT;
	    eventWord &= ~EVT_MSMV;  /* reset at any btn change */
	    lstmsX=mouseX = x; lstmsY=mouseY = y;
	    eventCntMs++;
	}
	if( msCheckMove )
	    if( x != mouseX || y != mouseY ) {
		eventWord |= EVT_MSMV;
		mouseBtns &= ~(EVT_MS_MV_U|EVT_MS_MV_R|EVT_MS_MV_D|EVT_MS_MV_L);
		if( x < mouseX )
		    mouseBtns |= EVT_MS_MV_L;
		else if( x > mouseX )
		    mouseBtns |= EVT_MS_MV_R;
		if( y < mouseY )
		    mouseBtns |= EVT_MS_MV_U;
		else if( y > mouseY )
		    mouseBtns |= EVT_MS_MV_D;
		lstmsX= mouseX; lstmsY=mouseY;
		mouseX = x; mouseY = y;
	    }
    }

    return eventWord;
}


#ifdef DOCUMENTATION
@Summary EventPeek
 #include <wk/event.h>

 unsigned EventPeek(void);
@Description
 Diese Funktion ist hnlich EventPoll(), mit dem Unterschied, das sie
 nicht nach events sieht, sondern nur das Statuswort zurckgibt.
 Kann von einer EventIdle-Function aufgerufen werden !
@Return Value
 Es wird das EventStatus Word zurckgegeben.
@See Also
 EventPoll
#endif /*DOCUMENTATION*/

unsigned EventPeek()
{
    return eventWord;
}




#ifdef DOCUMENTATION
@Summary EventWait
 #include <wk/event.h>

 unsigned EventWait(void);
@Description
 Diese Funktion ist hnlich EventPoll(), mit dem Unterschied, das sie
 falls kein Event anliegt auf den nchsten Event wartet.
@Return Value
 Es wird das EventStatus Word zurckgegeben.
@See Also
 EventPoll
#endif /*DOCUMENTATION*/

unsigned EventWait()
{
    unsigned which;

    do
	EventIdle(); /* min. 1 idle durchgang */
    while( !( which = EventPoll() ) );
    return which;
}



#ifdef DOCUMENTATION
@Summary EventClear
 #include <wk/event.h>

 void EventClear( unsigned mask )
@Description
 Lscht einen oder mehrere Events.
 als Mask ist eine der Konstamnten EVT_... anzugeben.
 wird 0 angegeben, so hat die Funktion keine Wirkung,
 wird ~0 angegeben, so werden alle anstehenden Events gelscht.
#endif /*DOCUMENTATION*/

void EventClear( unsigned which )
{
    if( which & EVT_KBD )
	lastKey = 0;
    if( which & EVT_CMD )
	lastCommand = 0;
    eventWord &= ~which;
}



#ifdef DOCUMENTATION
@Summary EventGetKbd
 #include <wk/event.h>

 int EventGetKbd(void);
@Description
 Gibt den letzten Tastencode zurck, ist keiner vorhanden, so
 wird 0 zurckgegeben. Der Tastencode bleibt solange vorhanden,
 bis er explizit durch EventClear( EVT_KBD ) zurckgesetzt wird.
@Return Value
 KeyCode (defined in wk/keys.h)
@See Also
 EventClear
#endif /*DOCUMENTATION*/

int EventGetKbd()
{
    return lastKey ;
}


#ifdef DOCUMENTATION
@Summary EventPutKbd
 #include <wk/event.h>

 void EventPutKbd( int keyCode );
@Description
 Setzt einen Keycode in die Eventqueue ab.
#endif /*DOCUMENTATION*/

void EventPutKbd( int keyCode )
{
    if( lastKey )
	Fatal("EVT: Keycode pending (Code=%d)", lastKey);

    eventWord |= EVT_KBD ;
    eventCntKbd++;
    lastKey = keyCode ;
}





#ifdef DOCUMENTATION
@Summary EventGetCmd
 #include <wk/event.h>

 int EventGetCmd(void);
@Description
 Gibt den letzten Commandcode zurck, ist keiner vorhanden, so
 wird 0 zurckgegeben. Der Code bleibt solange vorhanden,
 bis er explizit durch EventClear( EVT_CMD ) zurckgesetzt wird.
@Return Value
 CommandCode
#endif /*DOCUMENTATION*/


int EventGetCmd()
{
    return lastCommand;
}



#ifdef DOCUMENTATION
@Summary EventPutCmd
 #include <wk/event.h>

 void EventPutCmd( int cmdCode );
@Description
 Setzt einen Commandcode in die Eventqueue ab. Dieser CommandCode
 kann die Werte K_CMD_xxxxx oder alle Werte ab K_USER_BASE
 haben (defined in wk/keys.h)
#endif /*DOCUMENTATION*/


void EventPutCmd( int cmdCode )
{
    if( lastCommand )
	Fatal("EVT: Command pending (Code=%d)", lastCommand);
    if( !((cmdCode >= K_CMD_MENUSEL && cmdCode <= K_CMD_USER_4) ||
	cmdCode >= K_USER_BASE ) )
	Bug("EVT: cmdCode %d is invalid", cmdCode );

    eventWord |= EVT_CMD ;
    eventCntCmd++;
    lastCommand = cmdCode ;
}


#ifdef DOCUMENTATION
@Summary EventIdle
 #include <wk/event.h>

 void EventIdle(void);
@Description
 Kann aufgerufen werden um die Idlefuncktion durchzufhren.
 (EventWait() ruft diese z.b. laufend auf.
@See Also
 EventRgIdle
#endif /*DOCUMENTATION*/

void EventIdle()
{
    int i;
    long t;
    static long lastIdleTime ;

    t = TimeOfDay();
    for( i=0; i < MAX_IDLEFNC; i++ )
	if( idlefnc[i].fnc )
	    if( t >= idlefnc[i].lsttm + idlefnc[i].tslc ) {
		idlefnc[i].fnc(idlefnc[i].fncParm);
		idlefnc[i].lsttm = t;
	    }
    lastIdleTime = t;
}


#ifdef DOCUMENTATION
@Summary EventRgIdle
 #include <wk/event.h>

 int EventRgIdle( fnc, fncParm, interval );
 void (*fnc)(int);	Address of function
 int fncParm;		Parmeter fr die Funktin,
			wird bei jedem Aufruf bergeben
 long interval; 	Timeinterval for Function ( ms )
@Description
 Registriert eine Idle Funktion
@Return Value
 0 = okay
@See Also
 EventDgIdleFnc
#endif /*DOCUMENTATION*/

int EventRgIdle( void (*fnc)(int), int fncParm, long tslc )
{
    int i;

    for( i=0; i < MAX_IDLEFNC; i++ )
	if( !idlefnc[i].fnc ) {
	    idlefnc[i].fnc = fnc;
	    idlefnc[i].tslc= tslc;
	    idlefnc[i].fncParm= fncParm;
	    idlefnc[i].lsttm = 0L;
	    return 0;
	}
    return -1;
}



#ifdef DOCUMENTATION
@Summary EventDgIdle
 #include <wk/event.h>

 int EventDgIdle( fnc );
 void (*fnc)(void);	Addrerss of function
@Description
 Entfernt eine Funkction wieder aus der Idle Loop, die funktion
 wird anhand ihrer Adresse erkannt.
@Return Value
 0 = okay
@See Also
 EventRgIdleFnc
#endif /*DOCUMENTATION*/

int EventDgIdle( void (*fnc)(int) )
{
    int i;

    for( i=0; i < MAX_IDLEFNC; i++ )
	if( idlefnc[i].fnc == fnc ) {
	    idlefnc[i].fnc = NULLF;
	    return 0;
	}
    return -1;
}



#ifdef DOCUMENTATION
@Summary EventGetMs
 #include <wk/event.h>

 unsignedEventGetMs( int *px, int *py);
@Description
 Gibt die letzte ermittelte Position der Mouse zurck,
 den letzten Status der Buttons und ein Flag welches anzeigt, in welche
 Richtung die Mouse bewegt wurde.
@Return Value
 Bitweise: EVT_MS_...
#endif /*DOCUMENTATION*/

unsigned EventGetMs( int *px, int *py )
{
    unsigned ret;

    *px = lstmsX;
    *py = lstmsY;
    ret = mouseBtns;
    return ret;
}



#ifdef DOCUMENTATION
@Summary EventError
 #include <wk/event.h>

 void EventError(void);
@Description
 Kann ein event nicht mehr bearbeitet werden, so ist diese
 Funktion aufzurufen; sie versucht dann entsprechen Manahmnen durchzufhren.
#endif /*DOCUMENTATION*/

void EventError()
{
    Fatal("EVT: unused Mouseevents: %s %s %s",
	   eventWord & EVT_MSMV ? "move":"",
	   eventWord & EVT_MSLB ? "left":"",
	   eventWord & EVT_MSRB ? "right":"" );
    EventClear( EVT_MS );
    if( eventWord )
	Bug("EventError: Status is %X; lastkey is %d", eventWord, lastKey);
}



const char *EventInfo()
{
    static char buf[50];

    sprintf( buf, "%c%c%2s%1s %2d/%2d",
		  eventWord & EVT_MSLB ? 'L' : ' ' ,
		  eventWord & EVT_MSRB ? 'R' : ' ' ,
		  eventWord & EVT_MSMV ? "MV":"",
		  msCheckMove ? "c" : "",
		  mouseX, mouseY
					   );
    return buf;
}



static int DefaultKbdFnc()
{
    int c;
  #if OS2
    ushort  chr, scan;
    KBDKEYINFO kbci;
  #endif

    c = 0;
  #if OS2 /* CHANGE NOTE: Rembember to setup the DefaultKbdFlag */
    if( !KbdCharIn( &kbci, IO_NOWAIT, 0 ) ) {
	/* no error (error could be: no focus etc.) */
	if( kbci.fbStatus & ~(
			      #if OS20
				KBDTRF_SHIFT_KEY_IN | KBDTRF_INTERIM_CHAR_IN |
			      #else
				       SHIFT_KEY_IN |	     INTERIM_CHAR_IN |
			      #endif
					     CONVERSION_REQUEST   ) ) {
	    /* a useful character was entered */
	    scan = kbci.chScan;
	    chr  = kbci.chChar;
	    c = chr ? (chr & 0xff) : (256 + (scan&0xff));
	}
    }
  #elif defined(UNIX)
    if( (c = getchar()) == EOF )
	c = 0;
  #else
    if( bioskey(1) ) {
	c = bioskey(0);
	c = (c & 0xff)? (c & 0xff) : (((c >> 8) & 0xff) + 256);
    }
  #endif

    return c;
}



/***** bottom of File *****/
