/* [file15.c wk 29.3.89] OpenTmpFile()
 *	Copyright (c) 1988-96 by Werner Koch (dd9jn)
 *  This file is part of WkLib.
 *
 *  WkLib is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WkLib is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <wk/tailor.h>
RCSID("$Id: file15.c,v 1.3 1996/12/17 17:15:59 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <wk/lib.h>
#include <wk/file.h>


struct tempfile_list_struct {
    struct tempfile_list_struct *next;
    char *name;
    FILE *fp;
    int fp_valid; /* need this flag, because fp is a pointer */
};

static struct tempfile_list_struct *tempfiles;


static void
Cleanup(void)
{
    struct tempfile_list_struct *r, *r2;

    for( r=tempfiles; r; r = r2 ) {
	r2 = r->next;
	if( r->fp_valid ) {
	    r->fp_valid = 0;
	    fclose(r->fp);
	}
	if( r->name )
	    remove(r->name);
	free(r->name);
	free(r);
    }
    tempfiles = NULL;
}


#ifdef DOCUMENTATION
@Summary FOpenTmpFile3
@Description
 create and open a tmpfile, this function sets the name on a list
 of tmpfile, so they will be removed at programm-end.
@Return Value
  A FILE Pointer or NULL on error.
@See Also
 FCloseTmpFile
@Notes
@Example
#endif /*DOCUMENTATION*/

FILE *
FOpenTmpFile( char **ret_filename, const char *mode,
	     const char *prefix )
{
    return FOpenTmpFile3( ret_filename, mode, NULL, prefix, NULL );
}

FILE *
FOpenTmpFile2( char **ret_filename, const char *mode,
	     const char *direct, const char *prefix)
{
    return FOpenTmpFile3( ret_filename, mode, direct, prefix, NULL );
}

FILE *
FOpenTmpFile3( char **ret_filename, const char *mode,
	     const char *direct, const char *prefix, const char *suf)
{
    FILE *fp;
    char *name;
    struct tempfile_list_struct *entry;

    if( !tempfiles )
	RegisterCleanup( Cleanup );
    if( !(name = CreateTmpFile3(direct, prefix,suf)) )
	return NULL;
    if( !(fp = fopen(name, mode)) ) {
	free(name);
	return NULL;
    }
    entry = xcalloc(1, sizeof *entry );
    entry->name = name;
    entry->fp = fp;
    entry->fp_valid = 1;
    entry->next = tempfiles;
    tempfiles = entry;
    if( ret_filename )
	*ret_filename = name;
    return fp;
}

int
FCloseTmpFile( FILE *fp )
{
    struct tempfile_list_struct *r, *r2;

    for( r2=NULL,r=tempfiles; r; r2=r, r = r->next )
	if( r->fp == fp ) {
	    if( r->fp_valid ) {
		if( fclose(fp) )
		    return EOF;
		r->fp_valid = 0;
	    }
	    if( r->name && remove( r->name ) )
		return EOF;
	    free(r->name);
	    if( r2 )
		r2->next = r->next;
	    else
		tempfiles = r->next;
	    free(r);
	    return 0;
	}
    errno = EINVAL;
    return EOF;
}

const char *
FGetTmpFileName( FILE *fp )
{
    struct tempfile_list_struct *r;

    for( r=tempfiles; r; r = r->next )
	if( r->fp == fp )
	    return r->name;
    return "";
}


/****************
 * close old fp and make a new one with the new mode
 * closes and removes the tmpfile on error.
 */
FILE *
FReopenTmpFile( FILE *fp, const char *mode )
{
    struct tempfile_list_struct *r, *r2;
    int e;

    for( r2=NULL,r=tempfiles; r; r2=r, r = r->next )
	if( r->fp == fp ) {
	    if( !r->fp_valid || !fclose(fp) ) {
		r->fp_valid = 0;
		if( fp = fopen(r->name, mode) ) {
		    r->fp = fp;
		    r->fp_valid = 1;
		    return fp;
		}
	    }
	    /* failed: close */
	    e = errno;
	    if( r->name )
		remove( r->name );
	    errno = e;
	    free(r->name);
	    if( r2 )
		r2->next = r->next;
	    else
		tempfiles = r->next;
	    free(r);
	    return NULL;
	}
    errno = EINVAL;
    return NULL;
}

/******* bottom of file *******/
