/* [console.c wk 13.5.92] Console Module
 *	Copyright (c) 1992,94 by Werner Koch (dd9jn)
 *
 * Das Teil hat folgende 4 Windows:
 *     Ein umschaltbares Windows ( via Alt 1-8 )
 *     Eine StatusZeile
 *     Ein globales Windows - fuer Fehlermeldungen etc (Alt-9)
 *     Eine KommandoZeile
 *
 * History:
 * 15.11.94 wk	moved to wklib
 * 08.03.95 wk	replaced ScrDrvInkey by Inkey() - thie will not work
 *		but there is no ScrDrvInkey(), I have to think about
 *		extending Inkey() a little bit or write a new ScrDrvInkey().
 */

#include <wk/tailor.h>
RCSID("$Id: console.c,v 1.3 1995/03/08 16:53:58 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <wk/lib.h>
#include <wk/wscrdrv.h>
#include <wk/string.h>
#include <wk/keys.h>
#include <wk/errorhd.h>
#include <wk/console.h>

/***** constants *******/
#define MAX_WINDOWS 9
#define MAX_COMMANDPROCS 5  /* may # of registered command procedures */
#define MAX_IDLEPROCS	10  /* may # of registered idle procedures */
#define MAX_INKEYPROCS	 3  /* may # of registered inkey procedures */


/**** global vars ******/
static struct {
	int sxPS,syPS;	    /* size of Presentation Space */
	int pyMain, syMain; /* position and size of main window within PS */
	int aMain;	    /* attribute for main window */
	int pyStatus;	    /* position of Status line (size = 1 ) */
	int aStatus,aStatusEn,aStatusNew;
	int pyGlob, syGlob; /* position and size of global window within PS */
	int aGlob;
	int pyCmd;	    /* position of Command line (size = 1 ) */
	int aCmd;
    } screen;		       /* screen paramters */
static int dabfRegisterHd;

static struct {
	char inUse;		/* window is in use */
	char statusText[10];	/* text for the statusline */
	char newData;		/* new data received (in nonactive window) */
	int dabfHd;		/* handle of associated databuffer */
	int isTerm;		/* handle associated with a terminal buffer */
	ushort offset;
    } window[MAX_WINDOWS];     /* 0 = globales window, 1 - 8 Select Window */
static int activeWindowId;

static int (*commandProcTbl[MAX_COMMANDPROCS])(void*,int,const char*);
static void *commandProcData[MAX_COMMANDPROCS];
static void (*idleProcTbl[MAX_IDLEPROCS])(void*);
static void *idleProcData[MAX_IDLEPROCS];
static int (*inkeyProcTbl[MAX_INKEYPROCS])(int,int);

static int blackWhiteScreen;
static int normalTermination ;
static FILE *oldErrorStream;
static int errorRedirected;
static int runningFlag;

/****** protos *********/
static void MyVprintf( const char *s, va_list arg);
static void NotifyHandler( int dabfHd, int reason, ... );
static int MakeAttr( dabfAttr_t a );
static void MakeScreen(int);
static void ShowStatusLine(void);
static void ActivateWindow(int n);
static int  ShowWindow(void);
static int  PaintWindow( int inUse, int isTerm, int dabfHd, ushort off,
			 int py, int sy, int attr );
static void ScrollWindow( int nlines );
static void ShowInfo(int);
static void FullScreenLoop(void);
static void DispatchCommands(int);
static void DefaultIdleProc(void*);
static void CleanUp( void *dummy );

/***** functions *******/



#ifdef DOCUMENTATION
@Summary Console...
 #include <wk/cons.h>

 int ConsoleCreate( retHd, flags );
 int *retHd;	    Return: Handle of console
 unsigned flags;    Create flags:
		      Bit 0 : Run asyncron
		      Bit 1 : Redirect ErrorStream to file console.log
		      Bit 2 : use b/w color palette
		      Bit 3 : Use PM
 int ConsoleClose( hd ); /* only for asynchron console */
 void ConsoleStart( hd);
 void ConsoleStop( hd );
@Description
 Creates an Console. If Bit 1 of flags is set, all error output will be
 appended to a file in the current directory, named console.log; if
 this file exists, function will check for a special ID in the first line
 of the file to avoid overwrite of another file- Process will be
 terminated in this case.
 ConsoleStart() Starts the loop, in syncron mode, commandProcs or idleProcs
 must have been registered prior to ConsoleStart()
@Return Value
 0 = Okay else Error
@Notes
 Currently handles are not supported, you will always receive 0,
 but using them, may be good for later extensions.
#endif /*DOCUMENTATION*/


int
ConsoleCreate( int *retHd, unsigned flags )
{
    FILE *st;
    static const char errorFile[] = "console.log";
    static const char errorFileId[] = "WK==ConsoleLog==WK" ;
    char buf[18+1];
    int err, i, dabfHd;

    err = 0;

    if( flags & 1 )
	Bug( "Asynchron mode for console not yet supported" );
    dabfRegisterHd = -1; /* for cleanup */
    idleProcTbl[0] = DefaultIdleProc ;
    normalTermination = 0;
    errorRedirected = 0;
    runningFlag = 0;
    blackWhiteScreen = flags & 4;
    for(i=0; i < MAX_WINDOWS; i++ )
	window[0].inUse = 0;
    MakeScreen(flags & 8 ? 100:0);
    dabfRegisterHd = DaBfRegisterNotify( NotifyHandler );

    if( flags & 2 ) {
	if( st = fopen( errorFile, "r" ) ) {
	    if( !fgets( buf, 18+1, st ) )
		err++;
	    else if( strcmp( buf, errorFileId ) )
		err++;
	    else { /*okay : reopen file for append */
		fclose(st);
		if( !(st = fopen( errorFile, "a" )) )
		    Error(4,"Can't reopen logfile '%s'", errorFile );
	    }
	    if( err ) {
		fclose(st);
		Error(4,"Can't use logfile '%s' - not created for logging",
								  errorFile );
	    }
	}
	else if( !(st = fopen(errorFile, "a")) ) /* create file */
	    Error(4,"Can't create logfile '%s'", errorFile );
	else {	/* and write file Id */
	    fputs( errorFileId, st );
	    putc( '\n', st );
	    fflush( st );
	}
	errorRedirected++;
	oldErrorStream = ErrorStream( st );
    }

    /* First created DabBf will be associated with the global Window */
    /* also there is no need to save the dabf handle */
    if( err = DaBfCreate( &dabfHd, 0, 0, 0) )
	Error(4,"Can't create DaBf for global window (err=%d)", err );
    ErrorRgMonitor( MyVprintf ); /* link DaBf to global window */
    *retHd = 0;
    return err;
}


int
ConsoleClose( int hd )
{
    normalTermination++;
    exit(0);
    /*NOTREACHED*/
    return 0;
}


void
ConsoleStart(int hd)
{
    runningFlag = 1;
    FullScreenLoop();
}

void
ConsoleStop(int hd)
{
    runningFlag = 0; /* will stop the loop */
}


int
ConsoleRgIdle( void (*f)(void*), void *data )
{
    int i;

    if( idleProcTbl[0] == DefaultIdleProc )
	idleProcTbl[0] = NULL ; /* remove predefined idleproc */
    for(i=0; i < MAX_IDLEPROCS; i++ )
	if( !idleProcTbl[i] ) {
	    idleProcTbl[i] = f;
	    idleProcData[i] = data;
	    return 100 + i; /* idle handles are with offset 100 */
	}
    return -1; /* too mayn Idle functions */
}

int
ConsoleRgCmd( int (*f)(void *,int,const char*), void *data )
{
    int i;

    for(i=0; i < MAX_COMMANDPROCS; i++ )
	if( !commandProcTbl[i] ) {
	    commandProcTbl[i] = f;
	    commandProcData[i] = data;
	    return 200 + i; /* cmdProc handles are with offset 200 */
	}
    return -1; /* too mayn Cmd functions */
}

int
ConsoleRgKeyFilter( int (*f)(int,int) )
{
    int i;

    for(i=0; i < MAX_INKEYPROCS; i++ )
	if( !inkeyProcTbl[i] ) {
	    inkeyProcTbl[i] = f;
	    return 300 + i; /* inkey handles are with offset 300 */
	}
    return -1; /* too many Inkey functions */
}

int
ConsoleDeRg( int hd )
{
    if( hd >= 100 && hd < 100 + MAX_IDLEPROCS ) {
	idleProcTbl[hd-100] = NULL;
	idleProcData[hd-100]= NULL;
    }
    else if( hd >= 200 && hd < 200 + MAX_COMMANDPROCS )
	commandProcTbl[hd-200] = NULL;
    else if( hd >= 300 && hd < 300 + MAX_INKEYPROCS )
	inkeyProcTbl[hd-300] = NULL;
    else
	return -1;
    return 0;
}



static void
MyVprintf( const char *s, va_list arg)
{
    /* output to the first allocated databuffer */
    DaBfVprintf( window[0].dabfHd, s, arg );
}


/****************
 * Handler called by DaBf.... routines
 * (This would be the key to a multi-threaded version of this
 *  console module)
 */

static void
NotifyHandler( int dabfHd, int reason, ... )
{
    int n;
    unsigned flags;
    int type;
    long size;
    va_list arg_ptr;
    ushort x, y;
  #if 0
    dabfAttr_t a;
    byte c;
    int att;
  #endif

    switch(reason) {
      case DABF_N_CREATE: /* new databuffer created, associate a window */
	for(n=0; n < MAX_WINDOWS; n++ )
	    if( !window[n].inUse ) {
		window[n].inUse++;
		window[n].dabfHd = dabfHd;
		window[n].newData = 1; /* a new buffer contains new data */
		window[n].offset  = 0;
		DaBfGetInfo( dabfHd, &type, &flags, &size );
		window[n].isTerm = type == DABF_T_ANSI;
		sprintf( window[n].statusText,
			 window[n].isTerm ?"T-%X":" %X", dabfHd);
		ShowStatusLine();     /* even if empty data */
		ShowWindow();
		break; /* for loop */
	    }
	break;
      case DABF_N_CLOSE:  /* databuffer close, release associated window */
	for(n=0; n < MAX_WINDOWS; n++ )
	    if( window[n].inUse && window[n].dabfHd == dabfHd )  {
		window[n].inUse = 0;
		ShowStatusLine();
		break; /* for loop */
	    }
	break;
      case DABF_N_DATA:   /* new Data in Window - inform window about it */
	for(n=0; n < MAX_WINDOWS; n++ )
	    if( window[n].inUse && window[n].dabfHd == dabfHd )  {
		window[n].newData = 1;
		ShowStatusLine();
		ShowWindow();
		break;
	    }
	break;
      case DABF_N_CRSPOS:
      case DABF_N_CHAR:
	n = activeWindowId;
	if( window[n].dabfHd == dabfHd && window[n].inUse ) {
	    va_start(arg_ptr, reason);
	    x = va_arg( arg_ptr, ushort );
	    y = va_arg( arg_ptr, ushort );
	  #if 0
	    a = va_arg( arg_ptr, dabfAttr_t );
	    c = va_arg( arg_ptr, byte );
	    if( x < screen.sxPS &&  y < screen.syMain ) {
		ScrDrvWriteCell(x,screen.pyMain+y,
				a.concealed? '*':c, MakeAttr(a) );
		ScrDrvShow();
	    }
	    if( x < screen.sxPS &&  y < screen.syMain )
		ScrDrvWriteCell(x,screen.pyMain+y, ' ', 0xf0 );
	  #endif
	    va_end(arg_ptr);
	}
	break;
    #if 0
      case DABF_N_CLEL:
	n = activeWindowId;
	if( window[n].dabfHd == dabfHd && window[n].inUse ) {
	    va_start(arg_ptr, reason);
	    x = va_arg( arg_ptr, ushort );
	    y = va_arg( arg_ptr, ushort );
	    a = va_arg( arg_ptr, dabfAttr_t );
	    c = va_arg( arg_ptr, byte );
	    if( a.concealed )
		c = '*';
	    att = MakeAttr(a);
	    if( y < screen.syMain )
		for(; x < screen.sxPS; x++ )
		    ScrDrvWriteCell(x,screen.pyMain+y, c, att );
	    ScrDrvShow();
	    va_end(arg_ptr);
	}
	break;
      case DABF_N_CLS:
	n = activeWindowId;
	if( window[n].dabfHd == dabfHd && window[n].inUse ) {
	   va_start(arg_ptr, reason);
	   va_arg( arg_ptr, ushort );
	   va_arg( arg_ptr, ushort );
	   a = va_arg( arg_ptr, dabfAttr_t );
	   c = va_arg( arg_ptr, byte );
	   if( a.concealed )
	       c = '*';
	   att = MakeAttr(a);
	   for(y=0; y < screen.syMain; y++ )
	       for(x=0; x < screen.sxPS; x++ )
		   ScrDrvWriteCell(x,screen.pyMain+y, c, att );
	   ScrDrvShow();
	   va_end(arg_ptr);
	}
	break;
      case DABF_N_SCROLL:
	break;
    #endif
      default: break; /* unkown reason - skip */
    }
}

static int
MakeAttr( dabfAttr_t a )
{
    static byte map[8] = { 0, 4, 2, 6, 1, 5, 3, 7 };
    int att;



    if( a.reverse ) {
	att  = map[a.back&7];
	att |= map[a.fore&7]<<4;
    }
    else {
	att  = map[a.fore&7];
	att |= map[a.back&7]<<4;
    }
    if( a.blink | a.rblink )
	att |= 128;
    if( a.bold )
	att |= 16;
    return att ;
}

/****************
 * Create the Screen
 * and fill global vars
 * Initial Screen layout ist:
 *	7 lines global window
 *	1 line status
 *     16 lines main windows
 *	1 line command line
 * Mode 0 = Init
 * Mode 1 = grow global window
 * Mode -1 = shrink global window
 * Mode 2 = Restore all windows
 */

static void
MakeScreen(int mode)
{
    int bw, x, y;

    if( !mode ) {
	bw = ScrDrvOpen();
	AddCleanUp( CleanUp, NULL );
	if( blackWhiteScreen )
	    bw = 1;   /* force black & white palette */

	ScrDrvQrySize( &screen.sxPS, &screen.syPS );
	if( screen.syPS < 25 || screen.sxPS < 80 )
	    Bug("Inv. Screensize %d*%d; must have 25*80",
					screen.sxPS, screen.syPS);

	screen.aMain  = bw ? 0x0f : 0x0b ;
	screen.aStatus	 = bw ? 0x70 : 0x30 ;
	screen.aStatusEn = bw ? 0x07 : 0x3f ;
	screen.aStatusNew= bw ? 0xf0 : 0xb4 ;
	screen.aGlob  = bw ? 0x0f : 0x0f ;
	screen.aCmd   = bw ? 0x70 : 0x2f ;
	screen.pyMain = 0;	screen.syMain = 16;
	for(x=0; x < MAX_WINDOWS; x++ )
	    window[x].inUse = 0;
	activeWindowId = 0;
    }
    else if( mode == 1 && screen.syMain > 1 )
	screen.syMain--;
    else if( mode == -1 && screen.syGlob > 1 )
	screen.syMain++;
    screen.pyStatus = screen.pyMain + screen.syMain;
    screen.pyGlob = screen.pyStatus + 1;
    screen.syGlob = screen.syPS - screen.syMain - 1 - 1;
    screen.pyCmd  = screen.pyGlob + screen.syGlob;


    if( !mode ) {
	for(y=0; y < screen.syMain; y++ )
	    ScrDrvWriteNChar(0,screen.pyMain+y,' ',screen.aMain, screen.sxPS);
	for(y=0; y < screen.syGlob; y++ )
	    ScrDrvWriteNChar(0,screen.pyGlob+y,' ',screen.aGlob, screen.sxPS);
	ScrDrvWriteNChar(0,screen.pyCmd, ' ', screen.aCmd, screen.sxPS);
	ShowStatusLine();
	ScrDrvShow();
    }
    else {
	window[0].newData = 1;		    /* force Display */
	window[activeWindowId].newData = 1; /* force Display */
	ShowWindow();
	ShowStatusLine();
    }
}


/****************
 * Die Statuszeile mit allen Texten anzeigen
 */

static void
ShowStatusLine()
{
    int n,x,i,a,c;

    x = 0;
    for(n=1; n < MAX_WINDOWS; n++ ) {
	a = n == activeWindowId ? screen.aStatusEn :
	      window[n].newData ? screen.aStatusNew: screen.aStatus ;
	ScrDrvWriteCell( x++, screen.pyStatus, '0'+ n, a);
	ScrDrvWriteCell( x++, screen.pyStatus, ':', a);
	i=0;
	if( window[n].inUse )
	    for(;i < 7 && (c=window[n].statusText[i]); i++ )
		ScrDrvWriteCell( x++, screen.pyStatus, c, a);
	else
	    for(;i < 7 ; i++ )
		ScrDrvWriteCell( x++, screen.pyStatus, '-', a);
	for(;i < 8 ; i++ )
	    ScrDrvWriteCell( x++, screen.pyStatus, ' ', a);
    }
}


static void
ActivateWindow(int n)
{
    xassert(n >= 0 && n <= 8);
    activeWindowId = n;
    window[activeWindowId].newData = 1; /* force Display */
    ShowWindow();
    ShowStatusLine();
}


/****************
 * Returns number of show lines shown for scrollble window
 */

static int
ShowWindow(void)
{
    int n = 0, nlines=0;

    if( window[0].newData )  {
	/* no scroll for global window */
	PaintWindow( window[0].inUse, window[0].isTerm, window[0].dabfHd, 0,
		     screen.pyGlob, screen.syGlob, screen.aGlob );
	if( activeWindowId != 0 )  /* but scrollable if duplicated */
	    window[0].newData = 0;
	n++;
    }
    if( window[activeWindowId].newData )  {
	nlines = PaintWindow( window[activeWindowId].inUse,
			      window[activeWindowId].isTerm,
		     window[activeWindowId].dabfHd,
		     window[activeWindowId].offset,
		     screen.pyMain, screen.syMain, screen.aMain );
	if( !window[activeWindowId].offset )
	    window[activeWindowId].newData = 0;
	n++;
    }
    if( n ) /* may be called by notify function, so show the screen */
	ScrDrvShow();
    return nlines;
}


static int
PaintWindow( int inUse, int isTerm, int dabfHd, ushort offset,
	     int py, int sy, int attr )
{
    char *array[50];
    dabfAttr_t *attrArray[50];
    const char *p;
    int y, nlines, x;
    size_t len;

    xassert( sy < 50 );

    nlines = y = 0;
    if( inUse ) {
	if( isTerm ) {
	    DaBfGetALines( dabfHd, (const char**)array,
				   (const dabfAttr_t**)attrArray,
							sy+1 , 0, offset );
	    for(; p = array[y]; y++,nlines++ ) {
		len = strlen(p);
		if( len > screen.sxPS )
		    len = screen.sxPS;
	      #if 0
		ScrDrvWriteStr( 0, py+y, p, len, attr );
		ScrDrvWriteNChar(len,py+y, ' ', attr, screen.sxPS - len);
	      #else
		for(x=0; x < len ; x++ )
		    ScrDrvWriteCell( x, py+y, p[x],
			  attrArray[y] ? MakeAttr(attrArray[y][x]): attr );
	      #endif
	    }
	    DaBfUnlockPtrs( (const void**)array );
	}
	else {
	    DaBfGetLines( dabfHd, (const char**)array,
				   sy +(offset||py == screen.pyGlob? 1:0),
								1, offset );
	    for(; p = array[y]; y++,nlines++ ) {
		/* add some linewrapping and tabexpand here */
		len = strlen(p);
		if( len > screen.sxPS )
		    len = screen.sxPS;
		ScrDrvWriteStr( 0, py+y, p, len, attr );
		ScrDrvWriteNChar(len,py+y, ' ', attr, screen.sxPS - len);
	    }
	    DaBfUnlockPtrs( (const void**)array );
	    if( y < sy && py != screen.pyGlob ) { /* no end for the global window*/
		ScrDrvWriteStr( 0, py+y, "[---End---]", 11, attr );
		ScrDrvWriteNChar(11,py+y, ' ', attr, screen.sxPS - 11);
		y++;
	    }
	}
    }
    else {
	ScrDrvWriteStr( 0, py+y, "[Not Open]", 10, attr );
	ScrDrvWriteNChar(10,py+y, ' ', attr, screen.sxPS - 10);
	y++;
    }
    for( ; y < sy; y++ )
	ScrDrvWriteNChar(0,py+y, ' ', attr, screen.sxPS );
    return nlines;
}


static void
ScrollWindow( int nlines )
{
    ushort oldOffset;
    int n;

    oldOffset = window[activeWindowId].offset;
    if( !nlines ) { /* goto end of buffer */
	window[activeWindowId].offset  = 0;
	window[activeWindowId].newData = 1; /* force update */
	ShowWindow();
    }
    else if( nlines > 0 ) {
	window[activeWindowId].offset += nlines ;
	window[activeWindowId].newData = 1; /* force update */
	n = ShowWindow();
	if( n < nlines ) {
	    window[activeWindowId].offset = oldOffset + (nlines - n);
	    window[activeWindowId].newData = 1; /* force update */
	    ShowWindow();
	}
    }
    else { /* nlines < 0 */
	if( window[activeWindowId].offset > nlines )
	    window[activeWindowId].offset -= nlines ;
	else
	    window[activeWindowId].offset = 0;
	window[activeWindowId].newData = 1; /* force update */
	ShowWindow();
    }

}




static void
ShowInfo(int mode)
{
    static const char *text[] = {
       "  ============== Console Key Help ==============",
       "",
       "   F3 .............. Quit console",
       "   Alt-F1..F8 ...... Switch to window 1 .. 8",
       "   Alt-F9 .......... Duplicate global window to main area",
       "   Alt-F10 ......... Repaint screen",
       "   Ctrl-PgUp ....... Grow global window",
       "   Ctrl-PgDn ....... Shrink global window",
       "   Ctrl-Backspace .. Clear commandline",
       "   Return .......... Execute command",
       "",
       "  =======<Press any key to quit this help>======",
      NULL  };
    static const char *text2[] = {
      "",
      "    You are about to quit the console !",
      "   =====================================",
      "",
      "    - Press 'Y' to commit.",
      "",
      "    - Press any other key to continue.",
      NULL  };

    int y, len;
    const char *p;

    for(y=0; (p = mode ? text2[y] : text[y]) && y < screen.syPS-1; y++ ) {
	len = strlen(p);
	if( len > screen.sxPS )
	    len = screen.sxPS;
	ScrDrvWriteStr( 0, y, p, len, screen.aMain );
	ScrDrvWriteNChar(len,y, ' ', screen.aMain, screen.sxPS - len);
    }
    for( ; y < screen.syPS-1; y++ )
	ScrDrvWriteNChar(0,y, ' ', screen.aMain, screen.sxPS );
}



/****************
 * Process keyboard events and update the screen
 */

static void
FullScreenLoop(void)
{
    int i,j,c, crs, helpActive, quitActive=0;

    helpActive = crs = 0;
    ScrDrvSetCrs( crs, screen.pyCmd );
    ScrDrvShowCrs( 1 );
    ScrDrvShow();
    while(runningFlag) {
	while( !(c = Inkey()) && runningFlag ) {
	    for(c=0; c < MAX_IDLEPROCS; c++ )
		if( idleProcTbl[c] )
		    idleProcTbl[c]( idleProcData[c] );
	}
	if( helpActive || quitActive ) {
	    if( quitActive && (c == 'y' || c == 'Y') )
		ConsoleStop(0); /* we only got this handle at the moment */
	    helpActive = 0;
	    quitActive = 0;
	    MakeScreen(2);
	    c = 0;
	}
	for(i=j=0; i < MAX_INKEYPROCS; i++ ) {
	    if( inkeyProcTbl[i] ) {
		j = inkeyProcTbl[i]( window[activeWindowId].dabfHd, c );
		if( j == 1 ) { /* key was used */
		    c = 0;     /* do not use the key for commands */
		    break;
		}
		/* j == 2 : key accepted, but pass to next inkey handler*/
		/* j == 0 : key not accepted */
	    }
	}
	switch(c) {
	  case 0: break;
	  case K_F1:
	    helpActive = 1;
	    ShowInfo(0);
	    break;
	  case K_F3:  /* ready */
	    ShowInfo(1);
	    quitActive = 1;
	    break;
	  case K_AF1: ActivateWindow(1); break;
	  case K_AF2: ActivateWindow(2); break;
	  case K_AF3: ActivateWindow(3); break;
	  case K_AF4: ActivateWindow(4); break;
	  case K_AF5: ActivateWindow(5); break;
	  case K_AF6: ActivateWindow(6); break;
	  case K_AF7: ActivateWindow(7); break;
	  case K_AF8: ActivateWindow(8); break;
	  case K_AF9: ActivateWindow(0); break;
	  case K_AF10: MakeScreen(2); ScrDrvRepaint(); break;
	  case K_CPGUP: /* increase size of global window */
	    MakeScreen(1);
	    break;
	  case K_CPGDN: /* decrease size of global window */
	    MakeScreen(-1);
	    break;

	  case K_UP: ScrollWindow( 1 ); break;
	  case K_DOWN: ScrollWindow( -1 ); break;
	  case K_CHOME: ScrollWindow( 1000 );  /* should be enough */ break;
	  case K_CEND: ScrollWindow( 0 ); break;

	  case K_CF8: Info( CopyRight(10) );  break;
	  case K_CF9: Bug( "Abort test" );  break;
	  case K_CF10: Fatal( "Fatal %s", "test" ); break;

	  case K_RETURN: /* execute commandline */
	    DispatchCommands(0);
	    ScrDrvSetCrs( crs = 0, screen.pyCmd );
	    break;
	  case K_CRETURN: /* execute commandline */
	    DispatchCommands(1);
	    break;
	  case K_CRUBOUT: /* delete Command line */
	    ScrDrvWriteNChar(0,screen.pyCmd, ' ', screen.aCmd, screen.sxPS);
	    /* fall thru */
	  case K_HOME:	 /* goto first pos of commandline */
	    ScrDrvSetCrs( crs=0, screen.pyCmd );
	    break;
	  case K_LEFT:	/* commandline: cursor left */
	    if(crs )
		ScrDrvSetCrs( --crs, screen.pyCmd );
	    break;
	  case K_RUBOUT:/* commandline: backspace */
	    if(crs )
		ScrDrvSetCrs( --crs, screen.pyCmd );
	    ScrDrvWriteCell(crs,screen.pyCmd, ' ', screen.aCmd);
	    break;
	  case K_RIGHT: /* commandline: cursor right */
	    if(crs+1 < screen.sxPS )
		ScrDrvSetCrs( ++crs, screen.pyCmd );
	    break;
	  default: /* print at the commandline */
	    if( c < 256 ) {
		ScrDrvWriteCell( crs, screen.pyCmd, c, screen.aCmd);
		if(crs+1 < screen.sxPS )
		    ScrDrvSetCrs( ++crs, screen.pyCmd );
	    }
	    break;
	}
	ScrDrvShow();
    }
}


static void
DispatchCommands(int mode)
{
    int i, c, a, dabfHd;
    char buffer[80+1];

    dabfHd = window[activeWindowId].dabfHd ;
    for(i=0; i < 80; i++ ) {
	ScrDrvReadCell( i, screen.pyCmd, &c, &a);
	buffer[i] = c ? (byte)c : ' ';
    }
    buffer[i] = '\0';
    StripWSpaces(buffer);
    for(c=i=0; i < MAX_COMMANDPROCS; i++ )
	if( commandProcTbl[i] ) {
	    c = commandProcTbl[i]( commandProcData[i], dabfHd, buffer);
	    if( c == 1 ) /* command was accepted */
		break;
	    /* c == 2 : command accepted, but pass to next command handler*/
	    /* c == 0 : command not accepted */
	}
    if( !c )  /* command not accepted by any commandProc */
	Info("Invalid console command");
    if( !mode )
	ScrDrvWriteNChar(0,screen.pyCmd, ' ', screen.aCmd, screen.sxPS);
}


static void
DefaultIdleProc(void*p)
{
    Sleep(50);	/* Be kind to OS/2 and give up some CPU time */
}



static void
CleanUp( void *dummy )
{
    static const char text[] =
	"***** Process terminated ! - Press Escape to confirm *****";
    int len;

    DaBfDeRegisterNotify( dabfRegisterHd );
    ErrorRgMonitor( NULL );
    dabfRegisterHd = -1;
    if( !normalTermination ) {
	len = strlen(text);
	ScrDrvWriteNChar(0,0, ' ', 0x4f, screen.sxPS );
	ScrDrvWriteStr(   0, 1, text, len, 0x4f );
	ScrDrvWriteNChar(len,1, ' ', 0x4f, screen.sxPS - len);
	ScrDrvWriteNChar(0,  2, ' ', 0x4f, screen.sxPS );
	ScrDrvShow();
	while( Inkey() != K_ESCAPE )
	    ;
    }
    ScrDrvRestScreen();
    ScrDrvShow();
    ScrDrvClose();
}



/**** bottom of file ****/
