/* [wam/view.c wk 22.01.93] Class View
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Dieses Modul implementiert die Class View.
 * Diese Class kontolliert den Ablauf einer Anwendung
 * und ist Mittler zwischen Model und Form
 *
 ******************************************************
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: view.c,v 1.11 1996/09/25 16:20:54 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>

/**************************************************
 *************	Constants  ************************
 **************************************************/

DCLSYM sym_finishInitialization;

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

DCLSHAREDPART(View)

BEGIN_DCLPRIVATEPART
    id	form;	/* associated form of the view */
    id	args;	/* args of this view (view will free them when closed) */
    id	actDict;
    id	children;   /* IdArray of open children */
    id	parent;     /* id of parent view or nil */
    int parentLock; /* parent is locked */
    int deferClose; /* defer a close */
    int closePend;  /* a close is pendig */
END_DCLPRIVATEPART

static id openViews;

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static void RemoveChild( id self, id child);
static void SendOpenInitialization( id self );
static symbol_t ExtractSubIdFromName( symbol_t name, int *subId );

static id Open( id self, id var, symbol_t name, id args);

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

static void RemoveChild( id self, id child)
{
    DCL_var();
    if( var->children )
	msg1(var->children, sym_remove, child );
}


static void SendOpenInitialization( id self )
{
    DCL_var();
    var->deferClose = 1;
    msg(self, sym_openInitialization );
    if( !var->closePend )
	msg(var->form, sym_finishInitialization );
}


static symbol_t ExtractSubIdFromName( symbol_t name, int *subId )
{
    char *s, *s2;

    if( name && (s=symName(name)) && (s2=strchr(s, '.')) ) {
	*s2++ = 0;		   /* den punkt entfernen */
	name = WamCreateSymbol(s); /* und den Name erzeugen */
	*subId = atoi(s2);
    }
    else
	*subId = 0;
    return name;
}

/**************************************************
 ******************  Methods  *********************
 **************************************************/

/****************
 * Create an new Object and open it
 * Returns: Object of Class View....
 */

DCLFOBJFNC( open )
{
    id aView;
    id var2;

    if( !(aView = msg( self, sym(newInstanceRequest)))) {
	/* es soll eine neue Instance benutzt werden */
	aView = msg2( self, sym(privateOpen), nil, 0 );
	SendOpenInitialization( aView );
	/* no parent here */
	  /*  msg2(var->parent, sym_notify, aView, sym_ChildOpened ); */
	SET_varx(var2,aView); /* tricky; works only inside same Class Module */
	var2->deferClose = 0;
	if( var2->closePend ) {
	    msg( aView, sym_close );
	    aView = nil;
	}
    }
    return aView;
}

DCLFOBJFNC( newInstanceRequest )
{
    return nil;
}



DCLFOBJFNC( privateOpen )
{
    DCL_arg(id, args);
    DCL_arg(int, subId);
    id obj, var;

    obj = msgSuper( sym_new );
    SET_var(obj);
    var->actDict  = newObj(SymbolDictionary);
    var->children = newObj(IdArray);
    msg(obj, sym_initialize );
    msg1(openViews, sym_add, obj );
    if( subId )
	msg1(var->form, sym(setSubId), subId);
    msg(var->form, sym_open );
    var->args = args;
    return obj;
}


/****************
 * Factory method to
 * get an Array of all the currently open Views
 * This is NOT a copy!
 */

DCLFOBJFNC( openViews )
{
    return openViews;
}


/****************
 * lock all existing windows
 */

DCLFOBJFNC( lock )
{
    id aView;

    if( msg(openViews, sym_enumOpen ) )
	while( aView = msg(openViews, sym_enumGet) )
	    msg( aView, sym_lockWindow );
    return self;
}


/****************
 * unlock all windows
 */

DCLFOBJFNC( unlock )
{
    id aView;

    if( msg(openViews, sym_enumOpen ) )
	while( aView = msg(openViews, sym_enumGet) )
	    msg( aView, sym_unlockWindow );
    return self;
}



DCLOBJFNC( free )
{
    DCL_var();
    freeObj( var->form	  );
    freeObj( var->actDict );
    freeObj( var->args );
    freeObj( var->children );
    return msgSuper( sym_free );
}



/****************
 * This method should ininitialze a View
 * Es ist noch nicht alles initialisiert,
 * Normalerweise werden hier die Form und Item Attachments durchgefuehrt.
 */

DCLOBJFNC( initialize )
{
    return msg(self, sym_subclassResponsibility );
}



/****************
 * Setzt die Form der View, nur wenn noch nicht geschehen
 * Wird verwendet, wenn es eine eigene FormClass gibt.
 */

DCLOBJFNC( attachForm )
{
    DCL_arg(id,f);
    DCL_var();

    if( !var->form ) {
	var->form = newObj(f);
	msg1(var->form, sym_setView, self );
    }
    return self;
}



/****************
 * Setzt die Form der View, nur wenn noch nicht geschehen
 * Hat die gleiche Funktionalitt wie AttachForm, wird aber verwendet,
 * wenn automatisch zu erzeugende Form verwendet werden.
 * Returns: erzeugte bzw. bereits vorhandenes FormObjekt.
 */

DCLOBJFNC( createForm )
{
    DCL_var();

    if( !var->form ) {
	var->form = newObj(AutomaticForm);
	msg1(var->form, sym_setView, self );
    }
    return var->form;
}


/****************
 * Wird von Class AutoForm aufgerufen, um einer View
 * zur ermoeglichen, ein openInitialization fuer die
 * Form durchzufuehren.
 */

DCLOBJFNC( openInitializationForForm )
{
 /* DCL_arg(id, form ); */
    return self;
}



/****************
 * Oeffnet eine View von sich selbst aus und kann argumente uebergeben
 * An den Namen kann mit einem Punkt getrennt eine Zahl angehangen werden
 * diese wird dann als zusaetzlicher Key beim Speicher und wiederherstellen
 * der Bildschirmposition benutzt.
 */

DCLOBJFNC( open )
{
    DCL_arg( symbol_t, name );	/* name of View to be opened */
    DCL_arg( id, args );	/* args for this View (view will free them)*/
    id ret;
    DCL_var();

    ret = Open( self, var, name, args );
    return ret ? self : nil;
}

/****************
 * Wie open aber der Parent wird gelocked.
 */

DCLOBJFNC( openAndLock )
{
    DCL_arg( symbol_t, name );	/* name of View to be opened */
    DCL_arg( id, args );	/* args for this View (view will free them)*/
    id ret;
    DCL_var();

    if( ret = Open( self, var, name, args ) )
	msg(ret, sym_lockParent);
    return ret;
}


static id
Open( id self, id var, symbol_t name, id args)
{
    id o, aView, var2;
    int subId;

    name = ExtractSubIdFromName( name, &subId );

    aView = nil;
    if( (o=WamGetFactoryByName(name)) && WamIsKindOf( o, factory ) ) {
	if( !(aView = msg( o, sym(newInstanceRequest))) ) {
	    /* es soll eine neue Instance benutzt werden */
	    aView = msg2( o, sym(privateOpen), args, subId );
	    msg1( aView, sym(setParent), self );
	    msg1( var->children, sym_add, aView );
	    SendOpenInitialization( aView );
	    SET_varx(var2,aView); /* tricky; works only inside same Class Module */
	    var2->deferClose = 0;
	    if( var2->closePend ) {
		msg( aView, sym_close );
		aView = nil;
	    }
	    else
		msg2(self, sym_notify, aView, sym_ChildOpened );
	}
    }
    else
	msg2(self, sym_errorMsg,
		    "Cannot open unknown object '%s'\n",symName(name));
    return aView;
}

/****************
 * Wie #open, aber falls eine View dieser Klasse bereits als Child
 * offen ist, so wird diese nicht geoeffnet, sondern ihr ein #reopen gesendet.
 */

DCLOBJFNC( openSingle )
{
    DCL_arg( symbol_t, name );	/* name of View to be opened */
    DCL_arg( id, args );	/* args for this View (view will free them)*/
    id o, aView;
    int subId;
    symbol_t orgname;
    DCL_var();

    name = ExtractSubIdFromName( orgname=name, &subId );

    if( (o=WamGetFactoryByName( name )) && WamIsKindOf( o, factory ) ) {
	/* is such a view already open ? */
	aView = nil;
	if( msg(var->children, sym_enumOpen ) )
	    while( aView = msg(var->children, sym_enumGet) )
		if( WamIsMemberOf( aView, o ) )
		    break;
	msg(var->children, sym_enumClose );
	if( !aView ) /* no: */
	    return msg2(self, sym_open, orgname, args );
	/* yes: send a reopen request */
	return msg1( aView, sym_reopen, args );
    }
    else
	msg2(self, sym_errorMsg,
	     "Cannot open unknown object '%s'\n",symName(name));
    return nil;
}


/****************
 * Dies ist wie openSingle, wird aber nur durchgefhrt, falls
 * die angegeben view bereits offen ist.
 */
DCLOBJFNC( refreshOpen )
{
    DCL_arg( symbol_t, name );	/* name of View to be opened */
    DCL_arg( id, args );	/* args for this View (view will free them)*/
    id o, aView;
    int subId;
    DCL_var();

    name = ExtractSubIdFromName( name, &subId );

    if( (o=WamGetFactoryByName( name )) && WamIsKindOf( o, factory ) ) {
	/* is such a view already open ? */
	aView = nil;
	if( msg(var->children, sym_enumOpen ) )
	    while( aView = msg(var->children, sym_enumGet) )
		if( WamIsMemberOf( aView, o ) )
		    break;
	msg(var->children, sym_enumClose );
	if( !aView ) {/* no: */
	    freeObj(args);
	    return nil;
	}
	/* yes: send a reopen request */
	return msg1( aView, sym_reopen, args );
    }
    else
	msg2(self, sym_errorMsg,
	     "Cannot refreshOpen unknown object '%s'\n",symName(name));
    return nil;
}


/****************
 * Dies ist die default Methode fuer #reopen : Sie setzt den Focus
 * auf die view und sendet den (neuen) args ein free.
 * Soll eine andere Reaktion durchgefuehrt werden, so ist
 * diese methode zu subclassen, wobei dann aber nicht an super
 * gesendet werden sollte, falls zwischenzeitlich setArgs benutzt
 * wurde!
 */

DCLOBJFNC( reopen )
{
    DCL_arg( id, args );
    DCL_var();

    freeObj(args);
    msg( var->form, sym(restoreIfNotMaximized) );
    msg( self, sym_setFocus );
    return self;
}


DCLOBJFNC( setParent )
{
    DCL_arg(id, par);
    DCL_var();
    var->parent = par;
    return self;
}

DCLOBJFNC( parent )
{
    DCL_var();
    return var->parent;
}



/****************
 * Hier ist alles initialsiert, aber die View wahrscheinlich noch
 * nicht angezeigt,
 */

DCLOBJFNC( openInitialization )
{
    /* may be subclassed */
    return self;
}


/****************
 * Wird irgendwann nach openInitailization aufgerufen, sobald
 * das GUI das Window angezeigt hat.
 */

DCLOBJFNC( postOpenInitialization )
{
    /* may be subclassed */
    return self;
}


DCLOBJFNC( closeRequested )
{
    return msg(self, sym_close );
}


DCLOBJFNC( close )
{
    id tmp, par;
    DCL_var();

    if( var->deferClose ) {
	var->closePend=1;
	return self;
    }
    if( par = var->parent ) {
	if( var->parentLock ) {
	    msg( var->parent, sym_unlockWindow );
	    var->parentLock = 0;
	}
	msg2(var->parent, sym_notify, self, sym_ChildClosed );
    }
    /* first cancel all children */
    if( msg(var->children, sym_enumOpen ) )
	while( tmp = msg(var->children, sym_enumGet) )
		msg( tmp, sym_cancel );
    msg(var->children, sym_clear);
    if( var->parent ) {
	RemoveChild(var->parent, self);
	var->parent = nil;
    }
    /* then do the rest */
    msg(var->form, sym_close );
    msg1(openViews, sym_remove, self );
    if( par ) {
	msg(par, sym_restore );
	msg(par, sym_setFocus );
    }
    freeObj(self);
    return nil;
}


DCLOBJFNC( cancelChildren )
{
    id tmp, children;
    DCL_var();

    children = msg(var->children, sym_copy);
    if( msg(children, sym_enumOpen ) )
	while( tmp = msg(children, sym_enumGet) )
	    msg( tmp, sym_cancelChild );
    freeObj(children);
    return self;
}


/****************
 * Zum subclassen, um bestimmte Childs fuer ein reopen vorzubereits
 * diese koeenten dann immer zusaamen mit dem Parent angezeigt werden und
 * wechseln dann immer die Daten.
 */

DCLOBJFNC( cancelChild )
{
    return msg(self, sym_cancel );
}


DCLOBJFNC( cancel )
{
    return msg(self, sym_close );
}



DCLOBJFNC( event )
{
    DCL_arg(symbol_t,itemName);
    symbol_t actSel;
    DCL_var();

    if( actSel = s_msg1(var->actDict, sym_at, itemName ) )
	msg1(self, actSel, itemName );
    else {
      #if 0
	Info("Warning: No action performed for item '%s'\n",
							 symName(itemName));
      #endif
    }

    return self;
}


DCLOBJFNC( attachActionTo ) /* attach action event to method */
{
    DCL_arg(symbol_t, itemName);
    DCL_arg(symbol_t, actSel);	/* action selector */
    DCL_var();

    msg2(var->actDict, sym_atPut, itemName, actSel );
    return self;
}


DCLOBJFNC( doesNotRecognize )
{
    DCL_arg(symbol_t,m);
    DCL_var();

    if( !WamIsMemberOf( self, factory ) )
	/* special message for all subclasses if it is a attached method */
	if( msg1(var->actDict, sym(keyFromValue), m ) ) {
	    /* there is an attached method */
	    if( m != sym_open ) {/* try to use the Selector as ViewName */
	      #if 0
		if( o = WamGetFactoryByName( m ) ) /* only with views */
		    if( WamIsKindOf( o, factory ) ) {
			msg( o, sym_open );
			return self;
		    }
	       #endif
		return msg2( self, sym_open, m, nil );
	    }
	    msg2(self, sym_errorMsg,
		 "Attached Action method '%s' not found\n",symName(m));
	    return nil;
	}

    return msgSuper1( sym_doesNotRecognize, m );
}


/****************
 * Sendet ein Objekt an die ElternView, soll diese dort benutzt
 * werden, so ist daran zu denken, das es sich nicht um eine Kopie handelt.
 * Existiert die ElternView nicht, so geschieht nichts.
 * Returns: returnvalue from parent or nil if no parent
 */

DCLOBJFNC( sendToParent )
{
    DCL_arg( id, args );
    DCL_var();

    if( var->parent )
	return msg2( var->parent, sym(argsFromChild), self, args );
    return nil;
}

DCLOBJFNC( argsFromChild )
{
  /*DCL_arg(id, childView );
    DCL_arg(id, args );*/
    return nil;
}




DCLOBJFNC( form )   /*return the form */
{
    DCL_var();

    return var->form;
}

DCLOBJFNC( item )   /* get form item */
{
    DCL_arg(symbol_t, name);	/* name of item */
    DCL_var();

    return msg1(var->form, sym_item, name );
}


DCLOBJFNC( getValue )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();

    return msg1(var->form, sym_getValue, item );
}


DCLOBJFNC( putValue )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_arg(const void *, p);	 /* pointer to value */
    DCL_var();
    return msg2(var->form, sym_putValue, item, p );
}

DCLOBJFNC_s( getSymbol )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();

    return s_msg1(var->form, sym_getSymbol, item );
}


DCLOBJFNC( putSymbol )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_arg(symbol_t, sym);
    DCL_var();
    return msg2(var->form, sym_putSymbol, item, sym );
}


DCLOBJFNC( putLabel )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_arg(id, val);
    DCL_var();
    return msg2(var->form, sym_putLabel, item, val );
}


DCLOBJFNC( get )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();

    return msg1(var->form, sym_get, item );
}

DCLOBJFNC( getCheckedItem )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();

    return msg1(var->form, sym_getCheckedItem, item );
}


DCLOBJFNC( put )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_arg(id, val);		/* new value for item */
    DCL_var();
    return msg2(var->form, sym_put, item, val );
}



/****************
 * Diese Methode ist eine Vereinfachung eines oft
 * vorkommenden Kopiervorgangs.
 * Same as put.  put the args are a table object,
 * the name of a table field and the name of an item (this may be nil
 * to take the same name as the tablefield).
 */
DCLOBJFNC( putFromTable )
{
    DCL_arg(id, tbl);		/* table object */
    DCL_arg(symbol_t, fname);	/* name of table field */
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();
    id val, rc;

    if( !item )
	item = fname;
    val = msg1(tbl, sym_get, fname );
    rc = msg2(var->form, sym_put, item, val );
    freeObj(val);
    return rc;
}


/****************
 * Diese Methode ist eine Vereinfachung eines oft
 * vorkommenden Kopiervorgangs.
 * Same as get, get the value from the item in the view and put it
 * into the table in the field with the given name.
 * Returns: 0 := Der Wert in der Table hat sich *nicht* gendert,
 *	    1 := Der Wert in der Table hat sich gendert.
 */
DCLOBJFNC_i( getToTable )
{
    DCL_arg(id, tbl);		/* table object */
    DCL_arg(symbol_t, fname);	/* name of table field */
    DCL_arg(symbol_t, item);	/* name of item */
    id val;
    int flag;

    if( !item )
	item = fname;
    val = msg1(self, sym_get, item );
    flag = i_msg2(tbl, sym_put, fname, val );
    freeObj(val);
    return flag? 1:0;
}


DCLOBJFNC( clearValue )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();
    return msg1(var->form, sym_clearValue, item );
}

DCLOBJFNC( modified )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();
    return msg1(var->form, sym_modified, item );
}

DCLOBJFNC( resetModify )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_var();
    return msg1(var->form, sym_resetModify, item );
}

DCLOBJFNC( putAF )
{
    DCL_arg(symbol_t, item);	/* name of item */
    DCL_arg(id, val);		/* new value for item */
    DCL_var();
    msg2(var->form, sym_put, item, val );
    freeObj(val);
    return self;
}


/****************
 * Standard Methode um eine SQL - Fehler anzuzeigen, eventuell
 * kann hier eine spezielle Box aufgemacht werden, die eine genauere
 * Fehlermeldung anzeigt. Diese Funktion ist unmmittelbar nach eine SQL-Command
 * aufzurufen, da dort immer nur der letzte Fehler gespeichert wird.
 */

DCLOBJFNC( sqlFailure )
{
    id o;
    DCL_var();

  #if 0
    rc = i_msg(Database, sel(getLastError));
    sprintf(buffer, "SQL Fehler: rc = %d", rc );
  #endif
    o = msg(Database, sel(getLastErrorString) );
    msg2( var->form, sym_messageBox, WAM_ERROR, getString(o) );
    freeObj(o);
    return self;
}



DCLOBJFNC( infoMsg )
{
    DCL_arg(const char*, s);
    char buffer[500];
    DCL_var();

    vsprintf(buffer, s, arg_ptr );
    msg2( var->form, sym_messageBox, WAM_INFO, buffer );
    return self;
}

DCLOBJFNC( warningMsg )
{
    DCL_arg(const char*, s);
    char buffer[300];
    DCL_var();

    vsprintf(buffer, s, arg_ptr );
    msg2( var->form, sym_messageBox, WAM_WARNING, buffer );
    return self;
}

DCLOBJFNC( errorMsg )
{
    DCL_arg(const char*, s);
    char buffer[500];
    DCL_var();

    vsprintf(buffer, s, arg_ptr );
    msg2( var->form, sym_messageBox, WAM_ERROR, buffer );
    return self;
}

DCLOBJFNC( fatalMsg )
{
    DCL_arg(const char*, s);
    char buffer[300];
    DCL_var();

    vsprintf(buffer, s, arg_ptr );
    msg2( var->form, sym_messageBox, WAM_FATAL, buffer );
    return self;
}



DCLOBJFNC_i( verifyBox )
{
    DCL_arg(const char*, s);
    char buffer[300];
    DCL_var();

    vsprintf(buffer, s, arg_ptr );
    return i_msg2( var->form, sym_messageBox, WAM_VERIFY, buffer );
}

DCLOBJFNC_i( queryBox )
{
    DCL_arg(const char*, s);
    char buffer[300];
    DCL_var();

    vsprintf(buffer, s, arg_ptr );
    return i_msg2( var->form, sym_messageBox, WAM_QUERY, buffer );
}


/****************
 * get an Array of all the currently open Views
 * This is NOT a copy!
 */

DCLOBJFNC( openViews )
{
    return openViews;
}


/****************
 * Den Titel der View setzen
 */

DCLOBJFNC( setTitle )
{
    DCL_arg(const char *, p);	/* title String */
    DCL_var();
    if( !p )
	p = "";
    return msg1( var->form, sym_setTitle, p );
}


/****************
 * Den Focus bestimmen
 * Returns: 0 oder Name des Items mit dem Focus
 */

DCLOBJFNC_s( focus )
{
    id fitem;
    DCL_var();

    fitem = msg( var->form, sym_focus );
    if( fitem )
	return s_msg( fitem, sym_name );
    else
	return 0;
}


/****************
 * Den Focus auf diese View (MainForm) setzen
 */

DCLOBJFNC( setFocus )
{
    DCL_var();
    return msg( var->form, sym_setFocus );
}


/****************
 * Den Focus auf ein Item der View setzen
 */

DCLOBJFNC( setFocusTo )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_setFocusTo, name );
}



/****************
 * Werden von Form an View gesendet, und ist zum
 * subClassen vorhanden
 * Diese Methode wird nicht rekursiv aufgerufen, es sind nur
 * Notifications, der Focus muss neu gesetzt werden, wenn
 * das Item ihn nicht verlieren soll.
 */

DCLOBJFNC( gotFocus )
{
  /*DCL_arg(symbol_t, itemName);  or 0 bei Form */
    return self;
}

DCLOBJFNC( lostFocus )
{
    /*DCL_arg(symbol_t, itemName);  or 0 bei Form */
    return self;
}

/****************
 * Diese Methode informiert ueber eine Aenderung des Focus
 */

DCLOBJFNC( focusChange )
{
  /*DCL_arg(symbol_t, lostName);  or 0 if no old focus */
  /*DCL_arg(symbol_t, gotName );  or 0 if no new focus */
    return self;
}



DCLOBJFNC( enable )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_enable, name );
}

DCLOBJFNC( disable )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_disable, name );
}

DCLOBJFNC( setEnablement )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_arg(int,state); 	/* new state */
    DCL_var();

    if( state )
	return msg1( var->form, sym_enable, name );
    else
	return msg1( var->form, sym_disable, name );
}


DCLOBJFNC( readWrite )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_readWrite, name );
}

DCLOBJFNC( readOnly )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_readOnly, name );
}


DCLOBJFNC( insertMode )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_insertMode, name );
}

DCLOBJFNC( overtypeMode )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return msg1( var->form, sym_overtypeMode, name );
}

/****************
 * Wenn ein Feld edittable (readwrite) ist, so kann
 * der Benutzer eingaben vornehmen, ansonsten ist es
 * ein geschtzes Feld, auf welches auch kein Focus gesetzt
 * wird.
 */
DCLOBJFNC( setEditable )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_arg(int,state); 	/* new state */
    DCL_var();

    if( state )
	return msg1( var->form, sym_readWrite, name );
    else
	return msg1( var->form, sym_readOnly, name );
}


DCLOBJFNC( hideWindow )
{
    DCL_var();
    return msg( var->form, sym_hideWindow );
}

DCLOBJFNC( showWindow )
{
    DCL_var();
    return msg( var->form, sym_showWindow );
}


DCLOBJFNC( lockWindow )
{
    DCL_var();
    return msg( var->form, sym_lockWindow );
}

DCLOBJFNC( unlockWindow )
{
    DCL_var();
    return msg( var->form, sym_unlockWindow );
}


/****************
 * Dies ist eine Art automatic lock, welche nur funktioniert, wenn
 * ein parent vorhanden ist, und den lock wieder freigibt, sobald
 * die view geschlossen wird.
 */

DCLOBJFNC( lockParent )
{
    DCL_var();
    if( var->parent && !var->parentLock ) {
	msg( var->parent, sym_lockWindow );
	var->parentLock = 1;
    }
    return self;
}


DCLOBJFNC( restore )
{
    DCL_var();
    return msg( var->form, sym_restore );
}

DCLOBJFNC( minimize )
{
    DCL_var();
    return msg( var->form, sym_minimize );
}

DCLOBJFNC( maximize )
{
    DCL_var();
    return msg( var->form, sym_maximize );
}

DCLOBJFNC( setSelection )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_arg(int, sel ); 	/* selection index */
    DCL_var();
    return msg2( var->form, sym_setSelection, name, sel );
}

DCLOBJFNC_i( getSelection )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();
    return i_msg1( var->form, sym_getSelection, name );
}

DCLOBJFNC( groupItems )
{
    DCL_var();
    return WamVSendMsg( var->form, sym_groupItems, arg_ptr );
}


/****************
 * Return my Args
 */

DCLOBJFNC( args )
{
    DCL_var();
    return var->args;
}

/****************
 * Return one Args (assuming it its a dictionary)
 */

DCLOBJFNC( argAt )
{
    DCL_arg(symbol_t, name);	/* item name */
    DCL_var();

    if( WamIsKindOf( var->args, Dictionary ) )
	return msg1(var->args, sym_at, name);
    else
	return nil;
}



/****************
 * Die Argumente einer View aendern, wird gewoehnlich nur von
 * #reopen benutzt um die dort vorhandenen neuen Args auch der view
 * mitzuteilen. Eine Automaik ist nicht sinnvoll, da die args normaler
 * konstant sind und die View die Moeglichkeit haben muss, eine Aenderung
 * mitzubekommen.
 */

DCLOBJFNC( setArgs )
{
    DCL_arg(id, args);
    DCL_var();
    freeObj( var->args );
    var->args = args;
    return self;
}


void WamSUC_View()
{
    id self = View;
    CREATECLASS("View");
    WamSubclassClass( sym_Object, self );
    WamSetAtticLimit( self, 20 );

    DCLFMTHD( open );
    DCLFMTHD( newInstanceRequest );
    DCLFMTHD( privateOpen );
    DCLFMTHD( openViews );
    DCLFMTHD( lock );
    DCLFMTHD( unlock );

    DCLMTHD( free );
    DCLMTHD( initialize );
    DCLMTHD( attachForm );
    DCLMTHD( createForm );
    DCLMTHD( openInitializationForForm );
    DCLMTHD( open );
    DCLMTHD( openAndLock );
    DCLMTHD( openSingle );
    DCLMTHD( refreshOpen );
    DCLMTHD( reopen );
    DCLMTHD( setParent );
    DCLMTHD( parent );
    DCLMTHD( openInitialization );
    DCLMTHD( postOpenInitialization );
    DCLMTHD( closeRequested );
    DCLMTHD( close );
    DCLMTHD( cancel );
    DCLMTHD( cancelChildren );
    DCLMTHD( cancelChild    );
    DCLMTHD( event );
    DCLMTHD( attachActionTo );
    DCLMTHD( doesNotRecognize );
    DCLMTHD( sendToParent );
    DCLMTHD( argsFromChild );
    DCLMTHD( form );
    DCLMTHD( item );
    DCLMTHD( getValue );
    DCLMTHD( putValue );
    DCLMTHD( getSymbol);
    DCLMTHD( putSymbol);
    DCLMTHD( putLabel );
    DCLMTHD( get );
    DCLMTHD( getCheckedItem );
    DCLMTHD( put );
    DCLMTHD( putFromTable );
    DCLMTHD( getToTable );
    DCLMTHD( clearValue );
    DCLMTHD( modified );
    DCLMTHD( resetModify );
    DCLMTHD( putAF );
    DCLMTHD( sqlFailure );
    DCLMTHD( infoMsg );
    DCLMTHD( warningMsg );
    DCLMTHD( fatalMsg );
    DCLMTHD( errorMsg );
    DCLMTHD( verifyBox );
    DCLMTHD( queryBox );
    DCLMTHD( focus );
    DCLMTHD( setFocus);
    DCLMTHD( setFocusTo);
    DCLMTHD( gotFocus );
    DCLMTHD( lostFocus );
    DCLMTHD( focusChange );
    DCLMTHD( openViews );
    DCLMTHD( setTitle);
    DCLMTHD( enable  );
    DCLMTHD( disable );
    DCLMTHD( setEnablement );
    DCLMTHD( setEditable );
    DCLMTHD( readWrite );
    DCLMTHD( readOnly  );
    DCLMTHD( insertMode);
    DCLMTHD( overtypeMode);
    DCLMTHD( hideWindow );
    DCLMTHD( showWindow );
    DCLMTHD( lockWindow );
    DCLMTHD( unlockWindow );
    DCLMTHD( lockParent );
    DCLMTHD( restore );
    DCLMTHD( minimize );
    DCLMTHD( maximize );
    DCLMTHD( setSelection );
    DCLMTHD( getSelection );
    DCLMTHD( groupItems );
    DCLMTHD( args    );
    DCLMTHD( argAt   );
    DCLMTHD( setArgs );

    openViews = newObj(IdArray);
}


/**** end of file ****/
