/* [wam/table.c wk 31.01.93] Class Table
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Dieses Modul implementiert die Class Table.
 * Dabei handelt sich um eine Sammlung von Daten unterschiedlicher
 * Typen die in Spalten und Zeilen organisiert sind.
 * Es wird ein aktueller Record gehalten (initialisiert auf
 * den ersten Record) auf dessen Felder dann einfach ueber den
 * Feldnamen zugegriffen werden kann.
 ******************************************************
 * BUGS: Hier wird heftig ge-casted um memcpys zu vermeiden ...
 */

#include <wk/tailor.h>
RCSID("$Id: table.c,v 1.13 1996/01/25 20:18:09 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <wk/string.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define ROWTBL_FIRSTCHUNK    2	/* beim ersten mal nich soviel allokieren */
#define ROWTBL_CHUNK	    20	/* wahrscheinlich brauchen wir dann doch mehr*/
#define DYNA_THRESHOLD	    8	/* strings with a max. len greater of this */
				/* will be allocated dynamically */
DCLSYM sym_killDependency;

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/
typedef struct sparseHead_s {
    ushort len;  /* len of this sparse block */
    struct sparseHead_s *next;
    /* data follows here (no need to access it)*/
} sparseHead_t;

typedef struct {
    symbol_t name;  /* name of this column */
    byte     type;  /* datatype of column WAM_DT_...  or 0 if column invalid */
    byte     len;   /* column len or 0 if ptr */
    ushort   off;   /* offset of this column */
    ushort   flen;  /* formatierte laenge */
} columnDes_t;

typedef struct {
    int (*Fnc)(id,void*,int);
    void *parm;
    sparseHead_t *sparse;
} scrollDes_t;


DCLSHAREDPART(Table)

BEGIN_DCLPRIVATEPART
    ushort    refCount;     /* we have only shallow copies, so count them */
    ushort	nCols;	    /* number of columns in Table */
    columnDes_t *colDes;    /* allocated Array of column descrptors */
    ushort	rowSize;    /* total Number of bytes in a row */
			    /* including NULL-indicators */
			    /* min is sizeof(sparseHead), so we can link them*/
			    /* together in an sparselist */
    char	**data;     /* Array of ptrs to the rows */
    ushort	dataSize;   /* number of allocated rows in data */
    ushort	nRows;	    /* current number of rows */
    ushort	nullOff;    /* offset in each row of nullindicators */
    ushort	n;	    /* helper var */
    char       *p;	    /* helper var */
    ushort	curRow;     /* current row */
    id		deps;	    /* IdArray of dependent object or nil */
    int 	inBlock;
    int 	anyChanges;
    int 	pendingRows;
    scrollDes_t *scroll;    /* allocted if this is a scrollable table */
    int 	freePending;
END_DCLPRIVATEPART


/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static columnDes_t *GetColDesByName( id var, symbol_t name, size_t *retX );
static void BroadcastChange(id self, id var);
static int FormatDate( char *buf, long dt );
static int FormatTime( char *buf, long tm );
static int ScrollTable(id self, id var );
static void EraseTableData(id self, id var );
static void ReleaseScrollInfo(id var);
static void FixupFlen(id self, id var);


/**************************************************
 *************	Local Functions  ******************
 **************************************************/

static columnDes_t *
GetColDesByName( id var, symbol_t name, size_t *retX )
{
    size_t x;
    symbol_t upperName;
    columnDes_t *colDes;

    for(x=0,colDes=var->colDes; x < var->nCols; x++, colDes++ )
	if( colDes->name == name ) {
	    *retX = x;
	    return colDes;
	}
    /* not found: try again with capitalized name */
    upperName = WamCapitalizeSymbol(name);
    for(x=0,colDes=var->colDes; x < var->nCols; x++, colDes++ )
	if( colDes->name == upperName ) {
	    /* fix column name; so later access will hit with the first try */
	    /* Hint to improve performance: */
	    /* there is also a possibility to create a translation Table */
	    /* and change the column name, when the column is described */
	    colDes->name = name;
	    *retX = x;
	    return colDes;
	}
    return NULL; /* really not found */
}


/****************
 * An alle Dependents eine message senden
 */

static void
BroadcastChange(id self, id var)
{
    id o;

    if( var->inBlock )
	var->anyChanges = 1;
    else {
	if( var->deps ) {
	    if( msg(var->deps, sym_enumOpen ) )
		while( o = msg(var->deps, sym_enumGet) )
		    msg1( o, sym_changed, self );
	}
    }
}




static int
FormatDate( char *buf, long dt )
{
    int d,m,y;

    if( !JD2Date( dt, &d, &m, &y ) )
	return sprintf(buf,"??.??.????");
    return sprintf(buf,"%2d.%02d.%04d",d,m,y);
}

static int
FormatTime( char *buf, long tm )
{
    int h,m,s,mil;

    mil = tm % 1000;
    tm /= 1000;
    h = tm / 3600;
    m = (tm % 3600) / 60;
    s = tm % 60;
    return sprintf(buf,"%2d:%02d:%02d.%03d", h,m,s,mil );
}



/****************
 * Table scrollen
 * Returns: 0 = Scroll erfolgreich; d.h. weitere Rows wurden gefunden.
 */

static int
ScrollTable(id self, id var )
{
    EraseTableData(self,var);
    /* Read next rows */
    if( var->scroll->Fnc(self, var->scroll->parm, 0 ) ) { /* fehler */
	/* release DB-Resources if they still exist */
	ReleaseScrollInfo(var);
	return -1;
    }
    return 0;
}


/****************
 * Den Inhalt der Table loeschen (aber nicht die Beschreibung)
 */

static void
EraseTableData(id self, id var )
{
    char *row, *p;
    size_t x,y;
    columnDes_t *colDes;

    /* free the data rows */
    for(y=0; y < var->nRows; y++ ) {
	row = var->data[y];
	/* free dynamic string storage */
	for(x=0; x < var->nCols; x++ ) {
	    colDes = var->colDes + x;
	    if( colDes->type == WAM_DT_STRING ) {
		if( !colDes->len ) {
		    if( !(*(row + var->nullOff + x / 8) & (1<<(x % 8)) ) ) {
			p = *(char**)(row+colDes->off);
			free(p);
		    }
		}
	    }
	}

	if( var->scroll ) { /* link the rows to a sparselist */
	    sparseHead_t *s;

	    s = (void*)row;
	    s->len = var->rowSize;
	    s->next = var->scroll->sparse;
	    var->scroll->sparse = s;
	}
	else
	    free( row );
    }
    var->nRows = 0;
}



static void
ReleaseScrollInfo(id var)
{
    sparseHead_t *rover, *next;

    if( var->scroll ) {
	if( var->scroll->Fnc )
	    var->scroll->Fnc(nil, var->scroll->parm, 0 );
	for(rover=var->scroll->sparse; rover; rover = next) {
	    next = rover->next;
	    free(rover);
	}
	FREE(var->scroll);
    }
}



/****************
 * Die Formatierte Laenge der einzelen Spalten der Table neu bestimmen
 */

static void
FixupFlen(id self, id var)
{
    size_t y, x;
    char *row;
    columnDes_t *colDes;
    char buf[40];
    size_t n;
    long hl;

    /* set min. length */
    for(x=0; x < var->nCols; x++ ) {
	colDes = var->colDes + x;
	switch( colDes->type ) {
	  case WAM_DT_STRING:
	  case WAM_DT_INTEGER:
	  case WAM_DT_FLOAT:
	    colDes->flen = 1;
	    break;

	  default: break; /* don't fix them */
	}
    }

    /* determine max. length */
    for(y=0; y < var->nRows; y++ ) {
	row = var->data[y];
	for(x=0; x < var->nCols; x++ ) {
	    colDes = var->colDes + x;
	    if( !colDes->type )
		continue; /* skip invalid/removed columns */
	    if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) ) {
		/* NULL-Value */
		n = 0;
	    }
	    else {
		switch( colDes->type ) {
		  case WAM_DT_STRING:
		    if( !colDes->len )
			n = strlen( *(char**)(row+colDes->off) );
		    else
			n = strlen( row+colDes->off );
		    break;

		  case WAM_DT_INTEGER:
		    hl = *(long*)(row+colDes->off);
		    n =  hl < 0 ? 2 : 1;
		    while( hl = hl / 10 )
			n++;
		    break;

		  case WAM_DT_FLOAT:
		    n = sprintf(buf,"%.2f", *(double*)(row+colDes->off) );
		    break;

		  default: n = colDes->flen; break;
		}
	    }
	    if( n > colDes->flen )
		colDes->flen = n;
	}
    }
}

/**************************************************
 *************	Global Functions  *****************
 **************************************************/

/****************
 * Besonderheiten:
 * Type -1 = Begin description and create Table object:
 *	      - self should be 0.
 *	      - len tells the number of columns to describe.
 *	-2 = End description.
 *     -102= End description for scrollable Table.  len contains
 *	     max. number of columns in the scrollable table.
 * Returns: self.
 */

id
WamDescribeTableColumns( id self, symbol_t name, int type, size_t len )
{
    columnDes_t *colDes;

    xassert( len <= 0xffff );
    if( type == -1 ) {	/* begin description */
	id var, obj;

	xassert( !self );
	obj = msg( factory, sym_new );
	SET_var(obj);
	self = var;
	xassert( !var->nCols );  /* darf nur einmalig aufgerufen werden */
	var->refCount = 1;
	var->colDes = xcalloc( len, sizeof *var->colDes );
	var->n	   = len; /* for assertion check */
	var->pendingRows=0;
	var->scroll = NULL; /* we don't know yet */
    }
    else if( type == -2 || type == -102 ) { /* finish description */
	DCL_var();

	var->dataSize = ROWTBL_FIRSTCHUNK;
	if( type == -102 ) {
	    var->scroll = xcalloc( 1, sizeof *var->scroll );
	    if( len > var->dataSize )
		var->dataSize = len;
	}
	var->data = xcalloc( var->dataSize, sizeof *var->data );

	var->n = 0;
	var->nullOff = var->rowSize;
	var->rowSize += (var->nCols + 7) / 8;

	/* correct the size, so that we can use the memory to link */
	/* unused blocks together */
	if(var->rowSize < sizeof(sparseHead_t) )
	    var->rowSize += sizeof(sparseHead_t) - var->rowSize;

	/* *** now we are ready to fill the table with data *** */
    }
    else {  /* describe one column */
	DCL_var();
	xassert( var->nCols < var->n );
	colDes = var->colDes + var->nCols;
	colDes->name = name;
	colDes->type = type;
	switch( type ) {
	  case WAM_DT_STRING:
	    colDes->len = len > DYNA_THRESHOLD ? 0 : len + 1;
	    colDes->flen = len;
	    break;
	  case WAM_DT_SYMBOL:
	    colDes->len = sizeof(symbol_t);
	    colDes->flen = 5;
	    break;
	  case WAM_DT_INTEGER:
	    colDes->len = sizeof(long);
	    colDes->flen = 5;
	    break;
	  case WAM_DT_FLOAT:
	    colDes->len = sizeof(double);
	    colDes->flen = 8;
	    break;
	  case WAM_DT_DATE:
	    colDes->len = sizeof(long);
	    colDes->flen = 10;	   /* dd.mm.yyyy */
	    break;
	  case WAM_DT_TIME:
	    colDes->len = sizeof(long);
	    colDes->flen = 12;	  /* hh:mm:ss.xxx */
	    break;
	  case WAM_DT_TSTAMP:
	    colDes->len = sizeof(timeStamp_t);
	    colDes->flen = 23;	  /* dd.mm.yyyy-hh:mm:ss.xxx */
	    break;
	  default: Bug("Invalid type %d in WamDescribeTableColumns()", type );
	}
	xassert( var->dataSize+colDes->len+sizeof(void*) < 0xffff );
	colDes->off  = var->rowSize;
	var->rowSize += colDes->len ? colDes->len : sizeof(void*);
	var->nCols++;
    }
    return self;
}



/****************
 * Destroy a Table; even a partially created one
 */

void WamDestroyTable( id self )
{
    DCL_var();

    if( !self )
	return;
    free(var->p);
    msg(self, sym_free);
}



/****************
 * Besonderheiten:
 * mode -1 = Begin row create:
 *	-2 = End row create - make it permanent
 *	-3 = Notification: Table complete
 *	-4 = set rest of columns to NULL
 *     -13 = Notification: Table complete but limited in some way
 *    -103 = Scrollable Table is complete
 *    -113 = Scrollable Table break
 *	 0 = Create next column
 * Returns: self
 */

void WamCreateTableRow( id self,
		      int mode,
		      void *data,
		      size_t dataLen) /* for strings */
{
    columnDes_t *colDes;
    char *p;
    DCL_var();

    if( mode == -1 ) {	/* begin create row */
	xassert( var->nRows < 0xffff );
	if( var->scroll && var->scroll->sparse ) {
	    xassert( var->scroll->sparse->len == var->rowSize );
	    var->p = (void*)var->scroll->sparse;
	    var->scroll->sparse = var->scroll->sparse->next;
	    memset(var->p, 0, var->rowSize);
	}
	else
	    var->p = xcalloc( var->rowSize, 1 );
	var->n = 0;
    }
    else if( mode == -2 ) { /* finish create row -- make permanent */
	if( !(var->nRows < var->dataSize) ) {
	    /* we need more memory */
	    var->data = xrealloc( var->data, (var->dataSize + ROWTBL_CHUNK) *
							  sizeof *var->data  );
	    var->dataSize += ROWTBL_CHUNK;
	}
	var->data[var->nRows] = var->p;
	if( !var->nRows )
	    var->curRow = 0;
	var->nRows++;
	p = NULL;
    }
    else if( mode == -3 ) { /* notification: table complete */
	BroadcastChange(self, var);
	var->pendingRows=0;
    }
    else if( mode == -13 ) { /* notification: table incomplete */
	BroadcastChange(self, var);
	var->pendingRows=1;
    }
    else if( mode == -103 ) { /* notification: scrolltable complete */
	BroadcastChange(self, var);
	var->pendingRows=0;
	ReleaseScrollInfo(var);
    }
    else if( mode == -113 ) { /* notification: scolltable break */
	BroadcastChange(self, var);
	var->pendingRows=1; /* but we can retrieve them */
    }
    else if( mode == -4 ) { /* set rest of all columns to NULL */
	for( ;var->n < var->nCols; var->n++ ) {
	    p = var->p + var->nullOff + var->n / 8;
	    *p |= (1<<(var->n % 8));
	}
    }
    else {  /* create next column */
	xassert( var->n < var->nCols );
	if( data ) {
	    colDes = var->colDes + var->n;
	    switch( colDes->type ) {
	      case WAM_DT_STRING:
		if( !colDes->len ) { /* use dynamic storage for this column */
		    p = xmalloc( dataLen+1 );
		    mem2str(p, data, dataLen+1);
		    *(char**)(var->p + colDes->off) = p;
		}
		else {
		    xassert( colDes->len > dataLen );
		    mem2str( var->p + colDes->off, data, dataLen+1);
		}
		break;

	      case WAM_DT_SYMBOL:
		*(symbol_t*)(var->p + colDes->off) = *(symbol_t*)data;
		break;

	      case WAM_DT_INTEGER:
		*(long*)(var->p + colDes->off) = *(long*)data;
		break;

	      case WAM_DT_FLOAT:
		*(double*)(var->p + colDes->off) = *(double*)data;
		break;

	      case WAM_DT_DATE:
		*(long*)(var->p + colDes->off) = *(long*)data;
		break;

	      case WAM_DT_TIME:
		*(long*)(var->p + colDes->off) = *(long*)data;
		break;

	      case WAM_DT_TSTAMP:
		*(timeStamp_t*)(var->p + colDes->off) = *(timeStamp_t*)data;
		break;

	      default: BUG();
	   }
	}
	else { /* NULL Value */
	    p = var->p + var->nullOff + var->n / 8;
	    *p |= (1<<(var->n % 8));
	}
	var->n++;
    }
}


/****************
 * Diese Funktion setzt eine Funktion aus dem DatenbankModul, welche zu
 * benutzen ist, wenn weitere Rows benoetigt werden.
 * Sie wird vom dB-Modul aufgerufen, wenn dieses mit einem negativen LoadLimit
 * aufgerufen wird. (Weitere dox sind dort zu finden)
 */

void WamSetTableScrollFnc( id self, int(*ScrollFnc)(id,void*,int), void*parm1)
{
    DCL_var();

    /* kann auch aufgerufen werden bereits keine Rows mehr da
     * sind, wodurch dann scroll bereist nicht mehr gueltig ist
     */
    if( var->scroll ) {
       var->scroll->Fnc = ScrollFnc;
       var->scroll->parm = parm1;
    }

}


/**************************************************
 ******************  Methods  *********************
 **************************************************/

/****************
 * Erzeugt eine neue Table:
 * Als Argument wird ein Array erwatet mit der im hdr unter
 * tableDefinition_t beschriebenen Struktur
 */

DCLFOBJFNC( create )
{
    DCL_arg(tableDefinition_t *, tblDef );
    size_t n;
    id tbl;

    for(n=0; tblDef[n].name ; n++ )
	;
    tbl = WamDescribeTableColumns( nil, 0, -1, n );
    for(n=0; tblDef[n].name ; n++ )
	WamDescribeTableColumns( tbl, WamCreateSymbol( tblDef[n].name ),
				 tblDef[n].type, tblDef[n].len );
    WamDescribeTableColumns( tbl, 0, -2, 0 );
    return tbl;
}


DCLOBJFNC( free )
{
    DCL_var();

    if( --var->refCount )
	return nil;

  #if 0 /* preserved */
    if( var->deps ) {
	if( msg(var->deps, sym_enumOpen ) )
	    while( o = msg(var->deps, sym_enumGet) )
		msg1( o, sym_killDependency, self );
	freeObj( var->deps );
    }
  #else
    if( var->deps ) {
	if( i_msg(var->deps, sym_size ) ) {
	    var->freePending = 1;
	    return nil;
	}
	freeObj( var->deps );
    }
  #endif

    ReleaseScrollInfo(var);
    EraseTableData(self,var);
    free( var->data );
    free( var->colDes );

    return msgSuper( sym_free );
}



/****************
 * Eine Row hinzufuegen. Alle felder werden auf NULL gesetzt.
 * Der aktuelle RowPointer wird auf diese neue Row gesetzt.
 */

DCLOBJFNC( addRow )
{
    DCL_var();

    WamCreateTableRow( self, -1, NULL, 0 );
    WamCreateTableRow( self, -4, NULL, 0 );
    WamCreateTableRow( self, -2, NULL, 0 );
    var->curRow = var->nRows -1;
    return self;
}



/****************
 * Eine Column aus der Table entfernen
 */

DCLOBJFNC( removeColumn )
{
    DCL_arg(symbol_t, name);
    columnDes_t *colDes;
    char *row,*p;
    size_t x,y;
    DCL_var();

    colDes = GetColDesByName( var, name, &x );
    if( !colDes ) {
	msg2(self, sym_debug, "Table field '%s' not found", symName( name ) );
	return nil;
    }

    /* Hier wird nur der dynamisch llokierte Stringteil freigegeben */
    /* da der type nachher nicht mehr bestimmt werden kann */
    colDes = var->colDes + x;
    if( colDes->type == WAM_DT_STRING ) {
	if( !colDes->len ) {
	    for(y=0; y < var->nRows; y++ ) {
		row = var->data[y];
		if( !(*(row + var->nullOff + x / 8) & (1<<(x % 8)) ) ) {
		    p = *(char**)(row+colDes->off);
		    free(p);
		}
	    }
	}
    }
    colDes->type = 0;

    return self;
}




/****************
 * Die aktuelle Position zurueckgeben
 */

DCLOBJFNC_I( position )
{
    DCL_var();
    return var->curRow;
}

/****************
 * Die aktuelle Position setzen
 */

DCLOBJFNC( seekTo )
{
    DCL_arg(size_t, n);
    DCL_var();

    if( n < var->nRows )
	var->curRow = n;
    else
	msg2(self, sel(bounds), n, var->nRows-1 );
    return self;
}


/****************
 * Auf die letzte Zeile setzen
 */

DCLOBJFNC( seekToEnd )
{
    DCL_var();

    if( var->nRows )
	var->curRow = var->nRows-1;
    return self;
}


/****************
 * Die aktuelle Position eins weitersetzen
 * Returns: self = okay; nil = keine weitere row mehr da.
 */

DCLOBJFNC( next )
{
    DCL_var();

    if( var->curRow+1 < var->nRows ) {
	var->curRow++;
	return self;
    }
    else if( var->scroll ) {
	if( !ScrollTable(self,var) )
	    return self;
    }
    return nil;
}

/****************
 * Die aktuelle Position eins zuruecksetzen
 * Returns: self = okay; nil = erste row war bereits erreicht.
 */

DCLOBJFNC( previous )
{
    DCL_var();

    if( var->curRow ) {
	var->curRow--;
	return self;
    }
    return nil;
}



DCLOBJFNC_I( size )
{
    DCL_var();
    return var->nRows;
}

DCLOBJFNC_i( isEmptyOrNil )
{
    DCL_var();
    return !var->nRows;
}


DCLOBJFNC( show )
{
    msg2(self, sym(printOnFile), stdout, 1 );
    return self;
}



/****************
 * Ausgabe auf file.
 * Mode ist bitcodiert:
 *  Bit 0: Header mit ausgeben
 *  Bit 1: Comma delimited format
 */

DCLOBJFNC( printOnFile )
{
    DCL_arg(FILE*,fp);
    DCL_arg(int, mode);
    size_t y, x;
    timeStamp_t ts;
    char *row;
    columnDes_t *colDes;
    char buf[40];
    int i, any;
    ushort flen;
    DCL_var();

  #if 0
    printf("Table with %u columns and %u rows:\n", var->nCols, var->nRows);
  #endif
    FixupFlen(self,var);
    if( mode & 1) {
	any = 0;
	for(x=0; x < var->nCols; x++ ) {
	    colDes = var->colDes + x;
	    flen = mode & 2 ? 0 : colDes->flen;
	    if( !colDes->type )
		continue; /* skip invalid/removed columns */
	    if( any )
	       putc(mode & 2? ',':' ', fp);
	    else
	       any++;
	    WamQuerySymbolName( colDes->name, buf, DIM(buf) );
	  #if 0
	    fprintf(fp,"%s:%u/%u/%u/%u) ", buf,
		   colDes->type, colDes->flen, colDes->len, colDes->off );
	  #else
	    fprintf(fp,"%-*s", flen, buf);
	  #endif
	}
	putc('\n',fp);
    }
    for(y=0; y < var->nRows; y++ ) {
	row = var->data[y];
	any = 0;
	for(x=0; x < var->nCols; x++ ) {
	    colDes = var->colDes + x;
	    flen = mode & 2 ? 0 : colDes->flen;
	    if( !colDes->type )
		continue; /* skip invalid/removed columns */
	    if( any )
	       putc(mode & 2? ',':' ', fp);
	    else
	       any++;
	    if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) ) {
		/* NULL-Value */
		if( !(mode & 2) )
		    for(i=0; i < flen; i++ )
			putc(' ',fp);
	    }
	    else {
		switch( colDes->type ) {
		  case WAM_DT_STRING:
		    if( !colDes->len )
			fprintf(fp,"%-*s", flen,
				       *(char**)(row+colDes->off) );
		    else
			fprintf(fp,"%-*s", flen, row+colDes->off );
		    break;

		  case WAM_DT_SYMBOL:
		    fprintf(fp,"#%s", symName( *(symbol_t*)(row+colDes->off)));
		    break;

		  case WAM_DT_INTEGER:
		    fprintf(fp,"%*ld", flen, *(long*)(row+colDes->off) );
		    break;

		  case WAM_DT_FLOAT:
		    fprintf(fp,"%*.2f", flen, *(double*)(row+colDes->off) );
		    break;

		  case WAM_DT_DATE:
		    FormatDate( buf, *(long*)(row+colDes->off));
		    fputs(buf,fp);
		    break;

		  case WAM_DT_TIME:
		    FormatTime( buf, *(long*)(row+colDes->off));
		    fputs(buf,fp);
		    break;

		  case WAM_DT_TSTAMP:
		    ts = *(timeStamp_t*)(row+colDes->off);
		    FormatDate( buf, ts.d);
		    fprintf(fp,"%s-", buf);
		    FormatTime( buf, ts.t);
		    fputs(buf,fp);
		    break;

		  default: BUG();
		}
	    }
	}
	putc('\n',fp);
    }
    return self;
}



DCLOBJFNC( at )
{
    DCL_arg(symbol_t, name);
    Info("Dictionary: Please use #get instead of #at");
    return msg1(self, sym_get, name );
}

/****************
 * Das erzeugte Objekt muss wieder freigegeben werden !
 */
DCLOBJFNC( get )
{
    DCL_arg(symbol_t, name);
    columnDes_t *colDes;
    char *row;
    void *p;
    size_t x;
    DCL_var();

    colDes = GetColDesByName( var, name, &x );
    if( !colDes ) {
	msg2(self, sym_debug, "Table field '%s' not found", symName( name ) );
	return nil;
    }
    if( !var->nRows )
	return nil;	   /* table is empty */
    xassert( var->curRow < var->nRows );
    row = var->data[var->curRow];
    if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) )
	return nil;

    p = row+colDes->off;
    switch( colDes->type ) {
      case 0: /* removed column */
	return nil;

      case WAM_DT_STRING:
	if( !colDes->len )
	    return newString( *(char**)p );
	return newString( (char*)p   ); /* cast only for brevity */

      case WAM_DT_SYMBOL:
	return newSymbol( *(symbol_t*)p );

      case WAM_DT_INTEGER:
	return newInteger( *(long*)p );

      case WAM_DT_FLOAT:
	return newFloat( *(double*)p );

      case WAM_DT_DATE:
	return msg1(Date, sym_create, *(long*)p );

      case WAM_DT_TIME:
	return msg1(Time, sym_create, *(long*)p );

      case WAM_DT_TSTAMP:
	return msg1(Timestamp, sym_create, *(timeStamp_t*)p );

      default: BUG();
    }
    /*NOTREACHED*/
    return self;
}


/****************
 * Testet ob das Feld NAME eine NULL-Wert hat oder nicht.
 */
DCLOBJFNC_i( isNull )
{
    DCL_arg(symbol_t, name);
    columnDes_t *colDes;
    char *row;
    size_t x;
    DCL_var();

    colDes = GetColDesByName( var, name, &x );
    if( !colDes ) {
	msg2(self, sym_debug, "Table field '%s' not found", symName( name ) );
	return 1;
    }
    if( !var->nRows )
	return 1;    /* table is empty */
    xassert( var->curRow < var->nRows );
    row = var->data[var->curRow];
    if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) )
	return 1;
    return 0;
}


DCLOBJFNC_i( existsColumn )
{
    DCL_arg(symbol_t, name);
    size_t x;
    columnDes_t *colDes;
    DCL_var();

    colDes = GetColDesByName( var, name, &x );
    return  colDes && colDes->type ? 1:0;
}



/****************
 * Erzeugt ein Array mit Kopien der Werte eine Spalte
 * (d.h. mit deepFree freigeben )
 * Besonderheit: Sind in der Column nil vorhanden, so
 * wird nicht nil eingetragen sondern das Object "Null";
 * Damit ist es einfacher eine Afrage Schleife aufzubauen.
 */

DCLOBJFNC( fieldValues )
{
    DCL_arg(symbol_t, name);
    columnDes_t *colDes;
    char *row;
    void *p;
    size_t x, y;
    id arr, o;
    DCL_var();

    colDes = GetColDesByName( var, name, &x );
    if( !colDes ) {
	msg2(self, sym_debug, "Table field '%s' not found", symName( name ) );
	return nil;
    }
    if( !colDes->type ) {
	msg2(self, sym_debug, "Table field '%s' was removed", symName( name ) );
	return nil;
    }
    arr = newObj(IdArray);

    /* implementation kann noch verbessert werden */
    /* (wg. invarianzen) */
    for(y=0; y < var->nRows; y++ ) {
	row = var->data[y];
	if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) )
	    o = Null;
	else {
	    p = row+colDes->off;
	    switch( colDes->type ) {
	      case WAM_DT_STRING:
		if( !colDes->len )
		    o = newString( *(char**)p );
		else
		    o = newString( (char*)p   ); /* cast only for brevity */
		break;

	      case WAM_DT_SYMBOL:
		o = newSymbol( *(symbol_t*)p );
		break;

	      case WAM_DT_INTEGER:
		o = newInteger( *(long*)p );
		break;

	      case WAM_DT_FLOAT:
		o = newFloat( *(double*)p );
		break;

	      case WAM_DT_DATE:
		o = msg1(Date, sym_create, *(long*)p );
		break;

	      case WAM_DT_TIME:
		o = msg1(Time, sym_create, *(long*)p );
		break;

	      case WAM_DT_TSTAMP:
		o = msg1(Timestamp, sym_create, *(timeStamp_t*)p );
		break;

	      default: BUG();
	    }
	}
	msg1(arr, sym_add, o );
    }
    return arr;
}



DCLOBJFNC_i( atPut )
{
    DCL_arg(symbol_t, name);
    DCL_arg(id, val );
    Info("Dictionary: Please use #put instead of #atPut");
    return i_msg2(self, sym_put, name, val );
}

/****************
 * Returns: False wenn die Werte gleich waren
 *	    True  wenn es ein neuer Wert ist
 */

DCLOBJFNC_i(put)
{
    DCL_arg(symbol_t, name);
    DCL_arg(id, newVal );
    columnDes_t *colDes;
    char *row;
    char *p, *s;
    timeStamp_t ts;
    int chg;
    long longVal;
    symbol_t symbolVal;
    double doubleVal;
    size_t x;
    DCL_var();

    colDes = GetColDesByName( var, name, &x );
    if( !colDes ) {
	msg2(self, sym_debug, "Table field '%s' not found", symName( name ) );
	return 0;
    }
    if( !colDes->type ) {
	msg2(self, sym_debug, "Table field '%s' was removed", symName( name ) );
	return 0;
    }
    if( !var->nRows ) {
	msg1(self, sym_debug, "Table is empty" );
	return 0;	 /* table is empty */
    }
    xassert( var->curRow < var->nRows );
    row = var->data[var->curRow];
    p = row + var->nullOff + x / 8;
    if( !newVal ) { /* set field to NULL-Value */
	if( *p & (1<<(x % 8)) )
	    return 0; /* already NULL-value */
	*p |= (1<<(x % 8));

	/* ... should also free dynamic allocated String .....*/

	BroadcastChange(self, var);
	return 1;
    }

    chg = 0;
    if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) ) {
	/* was NULL-value */
	chg++;
	/* reset NULL-Value indicator */
	*p &= ~(1<<(x % 8));
    }
    switch( colDes->type ) {
      case WAM_DT_STRING:
	if( WamIsKindOf( newVal, String ) )
	    s = getString( newVal );
	else {
	    msg1(self, sym_debug, "Invalid object for String field" );
	    return 0;
	}
	if( !colDes->len ) { /* use dynamic storage for this column */
	    if( !chg ) { /* old value was not NULL */
		if( !(p = *(char**)(row + colDes->off)) )
		    chg++;  /* kann p ueberhaupt NULL sein .. ? */
		else if( chg = strcmp(p, s) )
		    free( p );
	    }
	    if( chg ) {
		p = xstrdup( s );
		*(char**)(row + colDes->off) = p;
	    }
	}
	else {
	    chg = strcmp( row + colDes->off, s );
	    mem2str( row + colDes->off, s, colDes->len);
	}
	break;

      case WAM_DT_SYMBOL:
	if( WamIsKindOf( newVal, Symbol ) )
	    symbolVal = getSymbol(newVal);
	else {
	    msg1(self, sym_debug, "Invalid object for Symbol field" );
	    return 0;
	}
	if( *(symbol_t*)(row + colDes->off) != symbolVal ) {
	    *(symbol_t*)(row + colDes->off)  = symbolVal;
	    chg=1;
	}
	break;

      case WAM_DT_INTEGER:
	if( !WamIsKindOf( newVal, Integer ) ) {
	    if( !WamIsKindOf( newVal, Float ) ) {
		msg1(self, sym_debug, "Invalid object for Integer field" );
		return 0;
	    }
	    longVal = (long)getFloat(newVal);
	}
	else
	    longVal = getInteger(newVal);
	if( *(long*)(row + colDes->off) != longVal ) {
	    *(long*)(row + colDes->off) = longVal;
	    chg=1;
	}
	break;

      case WAM_DT_FLOAT:
	if( !WamIsKindOf( newVal, Float ) ) {
	    if( !WamIsKindOf( newVal, Integer ) ) {
		msg1(self, sym_debug, "Invalid object for Float field" );
		return 0;
	    }
	    doubleVal = getInteger(newVal);
	}
	else
	    doubleVal = getFloat(newVal);
	if( *(double*)(row + colDes->off) != doubleVal ) {
	    *(double*)(row + colDes->off) = doubleVal;
	    chg=1;
	}
	break;

      case WAM_DT_DATE:
	if( !WamIsKindOf( newVal, Date ) ) {
	    msg1(self, sym_debug, "Invalid object for Date field" );
	    return 0;
	}
	longVal = getInteger(newVal);
	if( *(long*)(row + colDes->off) != longVal ) {
	    *(long*)(row + colDes->off) = longVal;
	    chg=1;
	}
	break;

      case WAM_DT_TIME:
	if( !WamIsKindOf( newVal, Time ) ) {
	    msg1(self, sym_debug, "Invalid object for Time field" );
	    return 0;
	}
	longVal = getInteger(newVal);
	if( *(long*)(row + colDes->off) != longVal ) {
	    *(long*)(row + colDes->off) = longVal;
	    chg=1;
	}
	break;

      case WAM_DT_TSTAMP:
	if( !WamIsKindOf( newVal, Timestamp ) ) {
	    msg1(self, sym_debug, "Invalid object for Timestamp field" );
	    return 0;
	}
	ts = *(timeStamp_t*)v_msg(newVal, sym_getValue);
	if(   (*(timeStamp_t*)(row + colDes->off)).d != ts.d
	    ||(*(timeStamp_t*)(row + colDes->off)).t != ts.t ) {
	    *(timeStamp_t*)(row + colDes->off) = ts;
	    chg=1;
	}
	break;

      default: BUG();
    }
    if( chg )
	BroadcastChange(self, var);
    return chg;
}


/****************
 * Erzeugt ein neues Dictionary, mit Kopien der Werte
 */

DCLOBJFNC( asDictionary )
{
    id dict, o;
    size_t x;
    char *row,*p;
    columnDes_t *colDes;
    DCL_var();

    dict = newObj( Dictionary );
    if( !var->nRows )
	return dict; /* table is empty */

    row = var->data[var->curRow];
    for(x=0; x < var->nCols; x++ ) {
	colDes = var->colDes + x;
	if( !colDes->type ) /* field was removed */
	    continue;

	p = row+colDes->off;
	if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) ) { /*NULL-Value*/
	    o = nil;
	}
	else switch( colDes->type ) {
	  case WAM_DT_STRING:
	    if( !colDes->len )
		o = newString( *(char**)p );
	    else
		o = newString( (char*)p   ); /* cast only for brevity */
	    break;

	  case WAM_DT_INTEGER:
		o = newInteger( *(long*)p );
	    break;

	  case WAM_DT_SYMBOL:
		o = newSymbol( *(symbol_t*)p );
	    break;

	  case WAM_DT_FLOAT:
		o = newFloat( *(double*)p );
	    break;

	  case WAM_DT_DATE:
		o = msg1(Date, sym_create, *(long*)p );
	    break;

	  case WAM_DT_TIME:
		o = msg1(Time, sym_create, *(long*)p );
	    break;

	  case WAM_DT_TSTAMP:
		o = msg1(Timestamp, sym_create, *(timeStamp_t*)p );
	    break;


	  default: BUG();
	}
	msg2( dict, sym_atPut, colDes->name, o );
    }

    return dict;
}




DCLOBJFNC( addDependent )
{
    DCL_arg(id, dep);
    DCL_var();

    if( !var->deps )
	var->deps = newObj( Array );
    msg1( var->deps, sym(addSingle), dep );
    return self;
}

DCLOBJFNC( removeDependent )
{
    DCL_arg(id, dep);
    DCL_var();

    if( var->deps ) {
	msg1( var->deps, sym_remove, dep );
	if( var->freePending ) /* try again to free */
	    msg(self, sym_free);
    }
    return self;
}



/****************
 * Die Table formatieren und as StringArray zurueckgeben
 * dieses muss der Aufrufer dann freigeben
 */

DCLOBJFNC( formatTable )
{
    id arr;
    size_t y, x, n, len;
    char *row;
    columnDes_t *colDes;
    char *line;
    DCL_var();

    arr = newObj( StringArray );
    if( !var->nRows )
	return arr; /* table is empty */

    FixupFlen(self,var);

    /* calculate lineLen */
    colDes = var->colDes;
    n = 0;
    for(x=0; x < var->nCols; x++ ) {
	colDes = var->colDes + x;
	switch( colDes->type ) {
	  case 0: break; /* removed column */
	  case WAM_DT_STRING: n += colDes->flen + 10; break;
	  case WAM_DT_SYMBOL: n += colDes->flen + 10; break;
	  case WAM_DT_INTEGER:n += colDes->flen + 10; break;
	  case WAM_DT_FLOAT:  n += colDes->flen + 10; break;
	  case WAM_DT_DATE:   n += colDes->flen + 10; break;
	  case WAM_DT_TIME:   n += colDes->flen + 10; break;
	  case WAM_DT_TSTAMP: n += colDes->flen + 10; break;
	  default: BUG();
	}
    }
    line = xmalloc(n+1);

    /* format data */
    for(y=0; y < var->nRows; y++ ) {
	row = var->data[y];
	n = 0;
	for(x=0; x < var->nCols; x++ ) {
	    colDes = var->colDes + x;
	    if( !colDes->type )
		continue;   /* removed column */

	    len = colDes->flen;
	    if( *(row + var->nullOff + x / 8) & (1<<(x % 8)) ) { /*NULL-Value*/
		memset(line+n, ' ', len );
		n += len;
	    }
	    else {
		switch( colDes->type ) {
		  case WAM_DT_STRING:
		    if( !colDes->len )
			n += sprintf(line+n, "%-*.*s", (int)len, (int)len,
				       *(char**)(row+colDes->off) );
		    else
			n += sprintf(line+n,"%-*.*s", (int)len, (int)len,
						  row+colDes->off );
		    break;

		  case WAM_DT_SYMBOL:
		    n+=sprintf(line+n,"%*lu",(int)len,(ulong)*(symbol_t*)
						    (row+colDes->off));
		    break;

		  case WAM_DT_INTEGER:
		    n+=sprintf(line+n,"%*ld",(int)len,*(long*)(row+colDes->off));
		    break;

		  case WAM_DT_FLOAT:
		    n+=sprintf(line+n,"%*.2f",(int)len, *(double*)(row+colDes->off));
		    break;

		  case WAM_DT_DATE:
		    n += FormatDate( line+n, *(long*)(row+colDes->off) );
		    break;

		  case WAM_DT_TIME:
		    n += FormatTime( line+n, *(long*)(row+colDes->off) );
		    break;

		  case WAM_DT_TSTAMP:
		    n += FormatDate( line+n,
			    (*(timeStamp_t*)(var->p + colDes->off)).d  );
		    line[n++] = '-';
		    n += FormatTime( line+n,
			    (*(timeStamp_t*)(var->p + colDes->off)).t  );
		    break;

		  default: BUG();
		}
	    }
	    line[n++] = 0x91;/* column separator (see MapIso2Ibm()) */
	}
	line[n?(n-1):n] = 0;  /* remove last column separator */
	msg1(arr, sym_add, line);
    }
    free( line );
    return arr;
}




/****************
 * Erzeugt ein Objekt der Class TableRow. Als Argument wird die
 * Rownumber erwartet. Gibt nil zurueck, falls Index out of Range
 */

DCLOBJFNC( asTableRowAt )
{
    DCL_arg(size_t, n);
    id obj;
    DCL_var();

    obj = nil;
    if( n < var->nRows )
	obj = msg2( TableRow, sym_create, self, n );
    else
	msg2(self, sel(bounds), n, var->nRows-1 );
    return obj;
}


/****************
 * die beiden folgenden Methode , dienen dazu, mehrere Operationen
 * auf eine Table ausfuehren zukoenne, ohne das Performance-Verluste
 * durch laufende Broadcasts ueber die Aenderungen auftreten;
 * Sind Aenderungen durchgefuehrt worden, so wird das Broadcast am Ende
 * durch #endBlock durchgefuehrt. Die Blcke knnen verschachtelt werden.
 */

DCLOBJFNC( beginBlock )
{
    DCL_var();
    if( !var->inBlock )
	var->anyChanges = 0;
    var->inBlock++;
    return self;
}

DCLOBJFNC( endBlock )
{
    DCL_var();
    var->inBlock--;
    if( !var->inBlock && var->anyChanges ) {
	BroadcastChange(self, var);
	var->anyChanges = 0;
    }
    return self;
}


DCLOBJFNC( pendingRows )
{
    DCL_var();
    return var->pendingRows? True : False;
}


DCLOBJFNC( scrollable )
{
    DCL_var();
    return var->scroll? True : False;
}


DCLOBJFNC( copy )
{
    DCL_var();

    var->refCount++;
    return self;
}


void WamSUC_Table()
{
    id self = Table;
    CREATECLASS("Table");
    WamSubclassClass( sym_Object, self );
    WamSetAtticLimit( self, 100 );

    DCLFMTHD( create );

    DCLMTHD( free );
    DCLMTHD( removeColumn );
    DCLMTHD( addRow );
    DCLMTHD( position );
    DCLMTHD( seekTo );
    DCLMTHD( seekToEnd );
    DCLMTHD( next );
    DCLMTHD( previous );
    DCLMTHD( size );
    DCLMTHD( isEmptyOrNil );
    DCLMTHD( show );
    DCLMTHD( printOnFile );
    DCLMTHD( at ); /* obsolete */
    DCLMTHD( get );
    DCLMTHD( isNull );
    DCLMTHD( existsColumn );
    DCLMTHD( fieldValues );
    DCLMTHD( atPut ); /* obsolete */
    DCLMTHD( put );
    DCLMTHD( asDictionary );
    DCLMTHD( addDependent );
    DCLMTHD( removeDependent );
    DCLMTHD( formatTable );
    DCLMTHD( asTableRowAt );
    DCLMTHD( beginBlock );
    DCLMTHD( endBlock );
    DCLMTHD( pendingRows );
    DCLMTHD( scrollable );
    DCLMTHD( copy );
}


/**** end of file ****/
