/* [wam/symbols.c wk 22.01.93] Symbols (Atoms)
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Symbols sind Dartsellungen von Strings, die im System
 * nur in voraussehbarer Menge vorkommen und auf die
 * schnelle Operationen angewendet werden sollen.
 * Jeder String wird nur einmal gespeichert und als
 * ein integraler Typ (Pointer to String converted to
 * Integer) behandelt. Es ist deswegen ein einfacher
 * Vergleich mit == moeglich und das Symbol kann als
 * case Label benutzt werden.
 *
 * Z.Z. wird nur fuer OS/2 implementiert und hierfuer
 * die OS/2 Atom-APIs benutzt. Eine Standard Implemetation
 * ist ohne weiters moeglich, wird aber erst implementiert,
 * wenn dies noetig sein sollte; zur Performance-Steigerung
 * sollte diese Implementation dann auch einen Hash-Algorithmus
 * benutzen (wie OS/2 Atoms)
 ******************************************************
 * History:
 */


#error OLD Source
#include <wk/tailor.h>
RCSID("$Id: symbols.c,v 1.4 1993/11/20 19:34:28 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#if defined(OS20)
    #define INCL_WINATOM 1
    #include <os2.h>
#else
    #error Current implementation is only for OS/2 v2.0
#endif


#include <wk/wam.h>


/**************************************************
 *************	Constants  ************************
 **************************************************/

#define MAX_SYMBOLLEN 128


/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

HATOMTBL  atomTblHd;	/* atom-table handle or NULL if not initialized */


/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static void TerminationHandler( void *);

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

/****************
 * Dieser wird bei Process Ende ( via CleanUpQueue )
 * aufgerufen und gibt die SymbolTable wieder frei
 */

static void TerminationHandler( void *dummy )
{
    WinDestroyAtomTable( atomTblHd );
}


/**************************************************
 *************	Global Functions  *****************
 **************************************************/

#ifdef DOCUMENTATION
@Summary WamInitializeSymbols
 #include <wk/wam.h>

 void WamInitializeSymbols(void);
@Description
 Initialisiert das Symbol SubSystem.
 Die Funktion is einmalig am Anfang einer Applikation aufzurufen;
 erst danach darf mit Symbols gearbeitet werden. Da diese Funktion zum
 einen nicht unbedingt reentrant sein muss und zum anderen Symbols
 ein grundlegender Bestandteil von WAM sind, sollte ihr Aufruf auf
 jeden Fall vor dem Aufruf von WamEnterProcess() erfolgen.
 Anmerkung: WamInitialize() fuehrt alle notwendigen Initialisierungen
 aus, und sollte deswegen benutzt werden.
@See Also
 WamInitialize
 WamCreateSymbol
 WamQuerySymbolName
@Notes
 Diese Funktion installiert ihren eigenen Ternination-Handler
#endif /*DOCUMENTATION*/


void WamInitializeSymbols(void)
{
    if( atomTblHd )
	return; /* already initialized */
    atomTblHd = WinCreateAtomTable( 0, 0 );
    if( !atomTblHd )
        Error(4,"Error initializing symbol system");
    AddCleanUp( TerminationHandler, NULL );
}



#ifdef DOCUMENTATION
@Summary WamCreateSymbol
 #include <wk/wam.h>

 symbol_t WamCreateSymbol( const char *name );

@Description
 Gibt fuer den angegebenen Namen das Symbol zurueck, ist der Name noch
 nicht benutzt worden, so wird er neu angelegt.
 Die Laenge eines Names ist auf 127 Zeichen zu begrenzen, das erste
 Zeichen soll ein Buchstabe (A-Z,a-z) oder ein Unterstrich sein,
 alle weiteren Zeichen koennen beliebig sein.
 Ein Symbol darf nicht fuer Bewegungsdaten erzeugt werden, da die
 Anzahl der Symbole durch die Symboltabelle begrentzt wird.
 Sollte das Anlegen eines Symbols nicht moeglich sein, so wird
 die Anwendung mit Error(4,.. ) abgebrochen.
@Return Value
 symbol zu dem angegebenen Namen.
@See Also
 WamQuerySymbolName
@Notes
 Aus Performance Gruenden wird auf eine Gueltigkeitspruefung des Names
 verzichtet.
#endif /*DOCUMENTATION*/


symbol_t WamCreateSymbol( const char *name )
{
    ATOM atom;

    atom = WinAddAtom( atomTblHd, (PSZ)name );
    if( !atom )
        Error(4,"Error creating symbol named \"%s\"", name );
    return (symbol_t)(ulong)atom;
}



#ifdef DOCUMENTATION
@Summary WamQuerySymbolName
 #include <wk/wam.h>

 char *WamQuerySymbolName( symbol_t sym, char *buf, size_t buflen );

@Description
 Gibt den Namen des angegebenen Symbols zurueck. Der Aufrufer
 muss einen Buffer mit einer ausreichenden Laenge angeben.
 Da Symbolnamen eine max. Laenge von 127 Zeichen haben, ist ein
 Buffer von 128 Zeichen zur Verfuegung zu stellen und diese Laenge
 (oder mehr) in buflen anzugeben.
 Falls kein Buffer angegeben wird, so wird ein statischer Buffer benutzt!
@Return Value
 buf
@See Also
 WamCreateSymbol
@Notes
 Wird ein ungueltiges Symbol angegeben, so wird die Anwendung
 mit Bug() abgebrochen.
#endif /*DOCUMENTATION*/

char *WamQuerySymbolName( symbol_t sym, char *buf, size_t buflen )
{
    static char helpBuf[MAX_SYMBOLLEN];

    if( !buf ) {
	buf = helpBuf;
	buflen = MAX_SYMBOLLEN;
    }

    if( !WinQueryAtomName( atomTblHd, (ATOM)sym, (PSZ)buf, buflen ) )
        Bug("error retrieving name for symbol %#lx", sym );
    return buf;
}


symbol_t WamCapitalizeSymbol( symbol_t name )
{
    char buf[MAX_SYMBOLLEN];

    WamQuerySymbolName( name, buf, MAX_SYMBOLLEN );
    strupr(buf);
    return WamCreateSymbol(buf);
}


/**************************************************
 *************	Test & Debug Suite ****************
 **************************************************/


/**** end of file ****/
