/* [wam/symbol.c wk 06.04.93] Class Symbol
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 *
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: symbol.c,v 1.7 1996/01/10 19:02:09 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <wk/string.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>
#include "wammain.h"

#ifdef __GNUC__
  #define INLINE inline
#else
  #define INLINE
#endif

/**************************************************
 *************	Constants  ************************
 **************************************************/

#define INITIAL_BUCKETS  1013

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

typedef struct entry_s {
    id	obj;		    /* link to the object */
    struct entry_s *next;   /* next entry (for collisions) */
    char name[1];	    /* dynamicly extended */
} entry_t;


static entry_t **buckets;
static size_t  noOfBuckets;


static int initializationPhase;

DCLSHAREDPART(Symbol)

BEGIN_DCLPRIVATEPART
    entry_t *val;   /* points direct to the correct entry */
END_DCLPRIVATEPART


/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static INLINE ulong Hash( const char *s );
static void TerminationHandler( void *);
static id CreateSymbol( const char * s );
static void FixupRawObjects(void);

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

static INLINE ulong Hash( const char *s )
{
    ulong hashVal = 0, carry;

    if( s )
	for( ; *s ; s++ ) {
	    hashVal = (hashVal << 4) + *s;
	    if( carry = (hashVal & 0xf0000000) ) {
		hashVal ^= (carry >> 24);
		hashVal ^= carry;
	    }
	}

    return hashVal % noOfBuckets;
}



static void TerminationHandler( void *dummy )
{
    int n;
    entry_t *entry, *tmp;

    for(n=0; n < noOfBuckets; n++ )
	for( entry = buckets[n] ; entry ; entry = tmp ) {
	    free( entry->obj ); /* can't send sym_free here */
	    tmp = entry->next;
	    free( entry );
	}
    free(buckets);
    noOfBuckets = 0;
}


static id
CreateSymbol( const char * s )
{
    id	var;
    entry_t *entry;
    ulong hash;

    hash = Hash(s);
    entry = buckets[hash];
    for( ; entry ; entry = entry->next )
	if( !strcmp( entry->name, s ) )
	    return entry->obj;	/* found */


    /* no instance with that name; create a new one */
    entry = xmalloc(sizeof *entry + strlen(s));
    strcpy( entry->name, s);
    if( initializationPhase )
	entry->obj  = xcalloc( 1, sizeof(struct s_objPRIVATE) );
    else
	entry->obj  = WamSendMsgSuper(factory,factory,sym_new);
    var = entry->obj; /* assuming the privateOffset is zero */
    var->val = entry;
    entry->next = buckets[hash];
    buckets[hash] = entry;
    return entry->obj;
}


static void
FixupRawObjects()
{
    int n;
    entry_t *entry;

    for(n=0; n < noOfBuckets; n++ )
	for( entry = buckets[n] ; entry ; entry = entry->next )
	    WamSetupRawObject( factory, entry->obj );
    initializationPhase = 0;
}


/**************************************************
 ******************  Methods  *********************
 **************************************************/

DCLFOBJFNC( new )
{
    return msg1(self, sym_create, NULL);
}

DCLFOBJFNC( create )
{
    DCL_arg(const char*, s);
    return CreateSymbol(s);
}



DCLOBJFNC( free )
{
    /* nothing to free, symbols are reusable */
    return self;
}


DCLOBJFNC( copy )
{
    return self;
}



DCLOBJFNC_s( getValue )  /* gibt den Wert zurueck */
{
    return self;
}


DCLOBJFNC_I( size )
{
    DCL_var();
    return strlen(var->val->name);
}


DCLOBJFNC_i( isEqual )
{
    DCL_arg(id, other );
    return self == other;
}


DCLOBJFNC( show )
{
    DCL_var();
    const char *p;
    p = var->val->name;
    fputs(p, stdout);
    return self;
}



/****************
 * Symbole werden einfach als String zurueckgegeben -
 * ohne Anfuehrungszeichen; sinnvoll fuer Konstanten wie Tablenames
 * oder z.B. USER
 */

DCLOBJFNC_p( allocAsDBString )
{
    DCL_var();
    return xstrdup(var->val->name);
}

DCLOBJFNC( isTrue )
{
    return True;
}

DCLOBJFNC( asString )
{
    char *p;
    id obj;
    DCL_var();

    p = xmalloc( strlen(var->val->name)+2 );
    *p = '#';
    strcpy(p+1, var->val->name);
    obj = newString(p);
    free(p);
    return obj;
}

DCLOBJFNC( asNumber )
{
    DCL_var();

    return newFloat( atof(var->val->name) );
}


void WamSUC_Symbol()
{
    id self = Symbol;
    CREATECLASS("Symbol");
    WamSubclassClass( sym_Object, self );

    DCLFMTHD( new );
    DCLFMTHD( create );

    DCLMTHD( free );
    DCLMTHD( copy );
    DCLMTHD( getValue );
    DCLMTHD( size );
    DCLMTHD( isEqual );
    DCLMTHD( show );
    DCLMTHD( allocAsDBString );
    DCLMTHD( isTrue	     );
    DCLMTHD( asString	     );
    DCLMTHD( asNumber	     );

    FixupRawObjects();
}


/**************************************************
 *************	Global Functions  *****************
 **************************************************/

#ifdef DOCUMENTATION
@Summary WamInitializeSymbols
 #include <wk/wam.h>

 void WamInitializeSymbols(void);
@Description
 Initialisiert das Symbol SubSystem.
 Die Funktion is einmalig am Anfang einer Applikation aufzurufen;
 erst danach darf mit Symbols gearbeitet werden. Da diese Funktion zum
 einen nicht unbedingt reentrant sein muss und zum anderen Symbols
 ein grundlegender Bestandteil von WAM sind, sollte ihr Aufruf auf
 jeden Fall vor dem Aufruf von WamEnterProcess() erfolgen.
 Anmerkung: WamInitialize() fuehrt alle notwendigen Initialisierungen
 aus, und sollte deswegen benutzt werden.
@See Also
 WamInitialize
 WamCreateSymbol
 WamQuerySymbolName
@Notes
 Diese Funktion installiert ihren eigenen Ternination-Handler
#endif /*DOCUMENTATION*/


void WamInitializeSymbols(void)
{
    if( buckets )
	return; /* already initialized */
    initializationPhase = 1;
    noOfBuckets = INITIAL_BUCKETS;
    buckets = xcalloc( noOfBuckets, sizeof *buckets );
    AddCleanUp( TerminationHandler, NULL );
}



#ifdef DOCUMENTATION
@Summary WamCreateSymbol
 #include <wk/wam.h>

 symbol_t WamCreateSymbol( const char *name );
 symbol_t WamCreateSymbolWithLen( const char *name, site_t len );

@Description
 Gibt fuer den angegebenen Namen das Symbol zurueck, ist der Name noch
 nicht benutzt worden, so wird er neu angelegt.
 Die Laenge eines Names ist auf 127 Zeichen zu begrenzen, das erste
 Zeichen soll ein Buchstabe (A-Z,a-z) oder ein Unterstrich sein,
 alle weiteren Zeichen koennen beliebig sein.
 Ein Symbol darf nicht fuer Bewegungsdaten erzeugt werden, da die
 Anzahl der Symbole durch die Symboltabelle begrentzt wird.
 Sollte das Anlegen eines Symbols nicht moeglich sein, so wird
 die Anwendung mit Error(4,.. ) abgebrochen.
@Return Value
 symbol zu dem angegebenen Namen.
@See Also
 WamQuerySymbolName
@Notes
 Aus Performance Gruenden wird auf eine Gueltigkeitspruefung des Names
 verzichtet.
#endif /*DOCUMENTATION*/


symbol_t
WamCreateSymbol( const char *name )
{
    return CreateSymbol(name);
}


symbol_t
WamCreateSymbolWithLen( const char *name, size_t len )
{
    char buf[50], *p;
    symbol_t s;

    if( len < DIM(buf)-1 ) {
	mem2str(buf, name, len+1 );
	s = CreateSymbol(buf);
    }
    else {
	p = mem2str(NULL, name, len+1 );
	s = CreateSymbol(p);
	free(p);
    }
    return s;
}


/****************
 * Testen on ein Symbol mit dem angegebenen Namen esistiert, wenn
 * ja, dann dieses zurueckgeben, ansonsten nil.
 */

symbol_t
WamFindSymbol( const char *s )
{
    entry_t *entry;
    ulong hash;

    hash = Hash(s);
    entry = buckets[hash];
    for( ; entry ; entry = entry->next )
	if( !strcmp( entry->name, s ) )
	    return entry->obj;	/* found */
    return nil;
}




#ifdef DOCUMENTATION
@Summary WamQuerySymbolName
 #include <wk/wam.h>

 char *WamQuerySymbolName( symbol_t sym, char *buf, size_t buflen );

@Description
 Gibt den Namen des angegebenen Symbols zurueck. Der Aufrufer
 muss einen Buffer mit einer ausreichenden Laenge angeben.
 Da Symbolnamen eine max. Laenge von 127 Zeichen haben, ist ein
 Buffer von 128 Zeichen zur Verfuegung zu stellen und diese Laenge
 (oder mehr) in buflen anzugeben.
 Falls kein Buffer angegeben wird, so wird ein statischer Buffer benutzt!
@Return Value
 buf
@See Also
 WamCreateSymbol
@Notes
 Wird ein ungueltiges Symbol angegeben, so wird die Anwendung
 mit Bug() abgebrochen.
#endif /*DOCUMENTATION*/

char *WamQuerySymbolName( symbol_t sym, char *buf, size_t buflen )
{
    static char helpBuf[128];

    if( !buf ) {
	buf = helpBuf;
	buflen = DIM(helpBuf);
    }

    mem2str(buf, sym->val->name, buflen );
    return buf;
}


symbol_t WamCapitalizeSymbol( symbol_t name )
{
    char buf[128];

    mem2str(buf, name->val->name, DIM(buf));
    strupr(buf);
    return CreateSymbol(buf);
}



/**** end of file ****/
