/* [wam/string.c wk 25.02.93] Class String
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 *
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: string.c,v 1.9 1996/09/18 13:16:32 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <wk/string.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>

/**************************************************
 *************	Constants  ************************
 **************************************************/

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

DCLSHAREDPART(String)

BEGIN_DCLPRIVATEPART
    char *val;	   /* allocated text of string */
END_DCLPRIVATEPART


/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static id CreateDBString( id self, id var, int c );

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

static id CreateDBString( id self, id var, int c )
{
    char *p, *s, *string;
    unsigned len;
    id obj;

    /* benoetigte Stringlaenge bestimmen */
    for(len=0,p=var->val; *p ; p++, len++ )
	if( *p == c )
	    len++;

    s = string = xmalloc( len + 2 + 1 );
    /* kopieren */
    for(*s++ = c, p=var->val; *p ; p++ ) {
	if( *p == c )
	    *s++ = c;
	*s++ = *p;
    }
    *s++ = c;
    *s = 0;
    obj = WamSendMsgSuper(factory,factory,sym_new);
				  /* !-- send it from the class object */
    SET_var(obj);
    var->val = string;
    return obj;
}

/**************************************************
 ******************  Methods  *********************
 **************************************************/

DCLFOBJFNC( new )
{
    return msg1(self, sym_create, "");
}

DCLFOBJFNC( create )
{
    DCL_arg(const char *, text);
    id obj;
    id	var;

    obj = msgSuper( sym_new );
    SET_var(obj);
    var->val = xstrdup(text);
    return obj;
}

DCLFOBJFNC( createWithLen )
{
    DCL_arg(const char *, text);
    DCL_arg(unsigned, len);
    id obj;
    id	var;

    obj = msgSuper( sym_new );
    SET_var(obj);
    var->val = mem2str(NULL,text,len+1);
    return obj;
}

DCLFOBJFNC( createBlankString )
{
    DCL_arg(unsigned, len);
    id obj;
    id	var;

    obj = msgSuper( sym_new );
    SET_var(obj);
    var->val = xmalloc( len + 1 );
    memset( var->val, ' ', len );
    var->val[len] = 0;
    return obj;
}


/****************
 * Dies ist eine besondere Funktion:
 * Sie erzeugt einen String von einem SQL String.
 * d.h. das erste und letzte Zeichen muss ein Apostroph sein und
 * inner Apostrophe mueesen verdoppelt sein.
 * Bei einem Syntax Fehler wird nil zurueckgegeben.
 * Blanks am Anfang werden uebersprungen. Zeichen nach dem Ende-Apostroph
 * werden nicht beruecksichtigt.
 */

DCLFOBJFNC( createFromSQLString )
{
    DCL_arg(const char *, text);
    const char *s;
    char *d;
    unsigned n;
    id obj;
    id	var;
    int esc;

    while( isspace( *text ) )
	text++;
    s = text;
    n = 0;
    if( *s == '\'') {
	for(esc=0,s++; *s; s++ )
	    if( esc ) {
		if( *s == '\'' )
		    esc=0, n++;
		else
		    break;
	    }
	    else if( *s == '\'' )
		esc++;
	    else
		n++;
	if( esc ) { /* dieser String ist gueltig und hat die laenge n */
	    obj = msgSuper( sym_new );
	    SET_var(obj);
	    d = var->val = xmalloc(n+1);
	    s = text;
	    for(esc=0,s++; *s; s++ )
		if( esc ) {
		    if( *s == '\'' )
			esc=0, *d++ = *s;
		    else
			break;
		}
		else if( *s == '\'' )
		    esc++;
		else
		    *d++ = *s;
	    *d = 0;
	    return obj;
	}
    }
    return nil;
}


DCLOBJFNC( free )
{
    DCL_var();

    free( var->val );
    return msgSuper( sym_free );
}


DCLOBJFNC( copy )
{
    DCL_var();

    return msg1(factory, sym_create, var->val );
}

DCLOBJFNC( asString )
{
    DCL_var();

    return msg1(factory, sym_create, var->val );
}


DCLOBJFNC( addString )
{
    DCL_arg(const char*, str);
    DCL_var();
    char *p;

    if( *str ) {
	p = xmalloc( strlen(var->val) + strlen(str) + 1 );
	stpcpy( stpcpy( p, var->val ), str );
	free( var->val );
	var->val = p;
    }
    return self;
}

DCLOBJFNC( addNString )
{
    DCL_arg(const char*, str);
    DCL_arg(unsigned, n);
    DCL_var();
    char *p, *p2;

    if( n ) {
	p = xmalloc( strlen(var->val) + n + 1 );
	p2 = stpcpy( p, var->val );
	memcpy(p2, str, n);
	p2[n] = 0;
	free( var->val );
	var->val = p;
    }
    return self;
}


/****************
 * Das Zeichen C N mal an den String anfgen.
 */
DCLOBJFNC( appendCharacters )
{
    DCL_arg(int, c);
    DCL_arg(unsigned, n);
    DCL_var();
    char *p, *p2;

    if( n ) {
	p = xmalloc( strlen(var->val) + n + 1 );
	p2 = stpcpy( p, var->val );
	memset(p2, c , n );
	p2[n] = 0;
	free( var->val );
	var->val = p;
    }
    return self;
}


DCLOBJFNC_p( getValue )  /* gibt den Wert zurueck */
{
    DCL_var();
    return var->val;
}

DCLOBJFNC( putValue )
{
    DCL_arg(const char *, text);
    DCL_var();

    free( var->val );
    var->val = xstrdup(text);
    return self;
}


DCLOBJFNC_I( size )
{
    DCL_var();
    return strlen(var->val);
}


DCLOBJFNC_i( isEqual )
{
    DCL_arg(id, other );    /* sollte auch string sein */
    id othvar;
    DCL_var();

    if( !WamIsMemberOf( other, factory ) )
	return 0;   /* from other class - cannot be equal */
    SET_varx(othvar,other); /* tricky; works only inside same Class Module */
    return !StrCmpNTS(var->val, othvar->val);
}


DCLOBJFNC_i( isEmptyOrNil )
{
    DCL_var();
    return !StrLenNTS(var->val);
}

DCLOBJFNC_i( isGT )
{
    DCL_arg(id, other );    /* sollte auch string sein */
    id othvar;
    DCL_var();

    if( !WamIsMemberOf( other, factory ) )
	return 0;   /* from other class - cannot compare */
    SET_varx(othvar,other); /* tricky; works only inside same Class Module */
    return StrCmpNTS(var->val, othvar->val) > 0 ;
}



DCLOBJFNC_i( isGE )
{
    DCL_arg(id, other );    /* sollte auch string sein */
    id othvar;
    DCL_var();

    if( !WamIsMemberOf( other, factory ) )
	return 0;   /* from other class - cannot compare */
    SET_varx(othvar,other); /* tricky; works only inside same Class Module */
    return StrCmpNTS(var->val, othvar->val) >= 0 ;
}


DCLOBJFNC_i( isLT )
{
    DCL_arg(id, other );    /* sollte auch string sein */
    id othvar;
    DCL_var();

    if( !WamIsMemberOf( other, factory ) )
	return 0;   /* from other class - cannot compare */
    SET_varx(othvar,other); /* tricky; works only inside same Class Module */
    return StrCmpNTS(var->val, othvar->val) < 0 ;
}

DCLOBJFNC_i( isLE )
{
    DCL_arg(id, other );    /* sollte auch string sein */
    id othvar;
    DCL_var();

    if( !WamIsMemberOf( other, factory ) )
	return 0;   /* from other class - cannot compare */
    SET_varx(othvar,other); /* tricky; works only inside same Class Module */
    return StrCmpNTS(var->val, othvar->val) <= 0 ;
}




DCLOBJFNC( show )
{
    DCL_var();
    fputs(var->val, stdout);
    return self;
}


/****************
 * Ich denke, diese Methode ist nicht ganz korrekt ?
 * (already quoted???)
 */

DCLOBJFNC_p( allocAsDBString )
{
    char *p, *s, *string;
    unsigned len;
    DCL_var();

    /* feststellen ob anfuehrungszeichen im String sind und laenge bestimmen */
    for(len=0,p=var->val; *p ; p++, len++ )
	if( *p == '\'' ) {
	    if( p[1] == '\'' ) { /* already quoted */
		p++;
		len++;
	    }
	    else
		len++; /* add room for an quote character */
	}

    s = string = xmalloc( len + 2 + 1 );
    /* kopieren */
    for(*s++ = '\'', p=var->val; *p ; p++ ) {
	if( *p == '\'' ) {
	    if( p[1] == '\'' )  /* already quoted */
		*s++ = *p++;
	    else
		*s++ = '\'';  /* quote character */
	}
	*s++ = *p;
    }
    *s++ = '\'';
    *s = 0;
    return string;
}


DCLOBJFNC( asDBString )
{
    DCL_var();
    return CreateDBString( self, var, '\'');
}


/****************
 * Wie as DBString, aber in Anfuehrungszeichen geklammert.
 */

DCLOBJFNC( asCSVString )
{
    DCL_var();
    return CreateDBString( self, var, '\"');
}



DCLOBJFNC( isTrue )
{
    DCL_var();
    return atof(var->val) ? True : False;
}



DCLOBJFNC( asNumber )
{
    DCL_var();

    return newFloat( atof(var->val) );
}


/****************
 * Dies Methode kann benutzt werden, um einen string zu krzen.
 * Returns: self (String die mit getValue geholt wurden sind
 *		  nicht mehr gltig)
 */
DCLOBJFNC( setSizeTo )
{
    DCL_arg(unsigned, newlen);
    DCL_var();

    if( strlen(var->val) > newlen )
	var->val[newlen] = 0; /* should we rerallocate the string ? */
    return self;
}


/****************
 * Dies Methode kann benutzt werden, um rechts mit Blanks aufzufllen.
 * Returns: self (String die mit getValue geholt wurden sind
 *		  nicht mehr gltig)
 */
DCLOBJFNC( padRightTo )
{
    DCL_arg(unsigned, newlen);
    DCL_var();
    char *p;
    unsigned len;

    if( (len=strlen(var->val)) >= newlen )
	return self; /* nichts zum auffllen da */

    p = xmalloc( newlen+1 );
    strcpy(p, var->val);
    memset(p+len,' ', newlen - len);
    p[newlen] = 0;

    free( var->val );
    var->val = p;
    return self;
}

/****************
 * Dies Methode kann benutzt werden, um links mit Blanks aufzufllen.
 * Returns: self (String die mit getValue geholt wurden sind
 *		  nicht mehr gltig)
 */
DCLOBJFNC( padLeftTo )
{
    DCL_arg(unsigned, newlen);
    DCL_var();
    char *p;
    unsigned len;

    if( (len=strlen(var->val)) >= newlen )
	return self; /* nichts zum auffllen da */

    p = xmalloc( newlen+1 );
    memset(p,' ', newlen - len);
    strcpy(p+newlen-len, var->val);

    free( var->val );
    var->val = p;
    return self;
}

/****************
 * Dies Methode kann benutzt werden, um links und rechts mit Blanks aufzufllen.
 * Returns: self (String die mit getValue geholt wurden sind
 *		  nicht mehr gltig)
 */
DCLOBJFNC( padLeftAndRightTo )
{
    DCL_arg(unsigned, newlen);
    DCL_var();
    char *p;
    unsigned len, left;

    if( (len=strlen(var->val)) >= newlen )
	return self; /* nichts zum auffllen da */

    p = xmalloc( newlen+1 );
    left = (newlen - len)/2;
    memset(p,' ', left);
    strcpy(p+left, var->val);
    memset(p+left+len,' ', newlen - len - left);
    p[newlen] = 0;

    free( var->val );
    var->val = p;
    return self;
}



void
WamSUC_String()
{
    id self = String;
    CREATECLASS("String");
    WamSubclassClass( sym_Object, self );
    WamSetAtticLimit( self, 500 );

    DCLFMTHD( new );
    DCLFMTHD( create );
    DCLFMTHD( createWithLen );
    DCLFMTHD( createBlankString );
    DCLFMTHD( createFromSQLString );

    DCLMTHD( free );
    DCLMTHD( copy );
    DCLMTHD( addString );
    DCLMTHD( addNString );
    DCLMTHD( appendCharacters );
    DCLMTHD( getValue );
    DCLMTHD( putValue );
    DCLMTHD( size );
    DCLMTHD( isEqual );
    DCLMTHD( isEmptyOrNil );
    DCLMTHD( isGT );
    DCLMTHD( isGE );
    DCLMTHD( isLT );
    DCLMTHD( isLE );
    DCLMTHD( show );
    DCLMTHD( allocAsDBString );
    DCLMTHD( isTrue );
    DCLMTHD( asString );
    DCLMTHD( asDBString );
    DCLMTHD( asCSVString );
    DCLMTHD( asNumber );
    DCLMTHD( setSizeTo );
    DCLMTHD( padRightTo );
    DCLMTHD( padLeftTo );
    DCLMTHD( padLeftAndRightTo );

}


/**** end of file ****/
