/* [wam/process.c wk 12.12.92]
 *	Copyright (c) 1992 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * ----------- System Dependend Source ----------------
 *-----------------------------------------------------
 * Hier werden Funktionen implementiert, die der
 * Processsteuerung dienen; d.h. System abhngige Sachen
 ******************************************************
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: process.c,v 1.6 1996/01/25 20:17:59 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>
#if defined(OS20)
    #define INCL_NOCOMMON 1
    #define INCL_DOSPROCESS 1
    #define INCL_DOSSEMAPHORES 1
    #define INCL_DOSERRORS 1
    #include <os2.h>
    #ifdef __WATCOMC__
      #include <process.h>
    #endif
#endif


#include <wk/wam.h>

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define MAX_THREADS  4	/* Anzahl der moeglichen Threads     */
			/* (laufende und beendete)	     */


/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/
typedef enum {
	tUNUSED=0,   /* Unused */
	tCREAT,      /* Thread is in state of creation */
	tRUN,	     /* Thread is running */
	tFROZ,	     /* Thread is frozen */
	tWAIT	     /* Thread terminated, but not released */
       } threadState_t;

struct s_thread {
	threadState_t state;	/* current thread state      */
	int tid;		/* tid for this thread	     */
      #if OS20
	TID osTID;		/* tid internal to the OS    */
      #endif
	volatile int cc;	/* termination code	     */
	int  (*fnc)(long);	/* used to pass the function */
	long parm;		/* and its parameter	     */
    };

typedef struct s_thread thread_t;

static thread_t threadTbl[MAX_THREADS];

#ifdef OS20
static HMTX serializeSemaphore; /* for internal use */
#endif

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static void TerminationHandler( void *dummy );
#ifdef __WATCOMC__
static void  RunThread( thread_t *);
#else
static void _wamapi RunThread( thread_t *);
#endif

static thread_t *AllocThreadSlot(void);
static thread_t *GetThreadSlot( int tid );

static void RequestSerialization(void);
static void ReleaseSerialization(void);

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

/****************
 * Dieser wird bei Process Ende ( via CleanUpQueue )
 * aufgerufen und kann alles moegliche freigeben, etc
 */

static void TerminationHandler( void *dummy )
{
    /* nothing yet to do */
}


/****************
 * Diese Funktion ist die hauptfunktion fuer alle
 * hier erzeugten Threads, Sie ruft die eigentliche
 * Funktion des Threads auf und sorgt nach deren Beendigung
 * fuer ein geordnetes Ende. Funktion kehrt nicht zurueck.
 */

#ifdef __WATCOMC__
static void
#else
static void _wamapi
#endif
RunThread( thread_t *thread /* ptr to slot */ )
{
    int cc, i,j, tid;

    tid  = thread->tid;

    thread->state = tRUN;
    /* call the function */
    cc = thread->fnc(thread->parm);
    /* termination processing */

    RequestSerialization();
    /* more than one thread running ? */
    for(i=j=0; i < MAX_THREADS; i++ )
	if( threadTbl[i].state == tRUN )
	    j++;
    if( thread->cc ) /* die quick, no WaitThread will be used */
	thread->state = tUNUSED;    /* release thread infos */
    else {
	thread->cc = cc;
	thread->state = tWAIT;
    }
    ReleaseSerialization();

    if( j > 1 ) {/* more than one running, so exit only the thread */
      #ifdef OS20
	if( thread->osTID == 1 )  { /* main thread, should not do an exit */
				    /* while others are running */
	   DosSuspendThread(1);
	   DosExit(EXIT_PROCESS, 0);
	}
       #if  __WATCOMC__ || EMX
	_endthread();
       #else
	DosExit(EXIT_THREAD, 0);
       #endif
      #endif
	BUG();	/* NOTREACHED */
    }
    WamLeaveProcess(0,cc); /* terminate all */
    /* may be the leave Process was already called by another thread: */
    for(;;)	/* wait until OS Kills the process and this thread */
      #ifdef OS20
	DosSleep(100);
      #else
	Sleep(100);
      #endif
}



/****************
 * Diese Funktion belegt einen ThreadSlot in der InfoTable,
 * setzt den State auf tCREAT und weist eine tid zu
 * Returns: NULL = Keine Slots mehr frei
 *	    ptr  = Pointer auf den Slot
 * ACHTUNG: Der erste Aufruf reserviert die TID 1, welche dann
 * nicht wieder vergeben wird!
 */

static thread_t *AllocThreadSlot()
{
    static int	lastTID=0;     /* round robin counter for tid */
			       /* so we will won't get easyly the same tid */
    int i,flag;
    thread_t *thread;

  #ifdef OS20
    DosEnterCritSec();
  #endif
    thread = NULL;
    for(i=0; i < MAX_THREADS; i++ )
	if( threadTbl[i].state == tUNUSED ) {
	    thread = threadTbl + i;
	    thread->state = tCREAT;
	    if( !lastTID )  /* first time execution of this function */
		lastTID = 1;
	    else {    /* allocate a new tid */
		do {  /* I use very much precaution to assure unique tids */
		    lastTID++;
		    if( lastTID < 2 ) /* wrap around may have occured */
			lastTID = 2;
		    for(i=flag=0; i < MAX_THREADS; i++ )
			if( threadTbl[i].tid == lastTID ) {
			    flag++; /* recently used or currently in use */
			    break;
			}
		} while( flag );
	    }
	    thread->tid = lastTID;
	    thread->cc = 0;
	    break;
	}
  #ifdef OS20
    DosExitCritSec();
  #endif
    return thread;
}


/****************
 * Ptr auf thread bestimmen, sollte immer benutzt werden, um es
 * dem System freizustellen, die ThreadTable eventuell mal zu packen
 * (dann muesste allerdings aber auch ein lock/unlock her)
 * Returns: NULL   = invalid TID
 *	    Ptr to thread of given tid
 */

static thread_t *GetThreadSlot( int tid )
{
    int i;

    for(i=0; i < MAX_THREADS; i++ )
	if( threadTbl[i].state != tUNUSED )
	    if( threadTbl[i].tid == tid )
		return threadTbl + i;
    return NULL;
}


static void RequestSerialization(void)
{
    int rc;
  #ifdef OS20
    if( rc = DosRequestMutexSem( serializeSemaphore , 1000 ) )
  #else
    rc = -1;
  #endif
	Error(4,"Error requesting internal serialization: rc=%d", rc );
}

static void ReleaseSerialization(void)
{
    int rc;
  #ifdef OS20
    if( rc = DosReleaseMutexSem( serializeSemaphore ) )
  #else
    rc = -1;
  #endif
	Error(4,"Error releasing internal serialization: rc=%d", rc );
}

/**************************************************
 *************	Global Functions  *****************
 **************************************************/

#ifdef DOCUMENTATION
@Summary WamEnterProcess
 #include <wk/wam.h>

 void WamEnterProcess( fnc, parm );
 int (*fnc)(long);  Function to execute as MainThread
 long parm;	    Paramater passsed to function
@Description
 Diese Funktion Startet das WAM, sie wird niemals zurueckkehren.
 Vor Aufruf dieser Funktion duerfen keine Threads erzeugt werden
 oder ExitHAndler etc. installiert werden. Empfelenswert ist lediglich
 die Auswertung von Kommandozeilenparamtern.
 Die Funktion die hierdurch aufgerufen wird, wird als originaler
 Thread (MainExecutionThread) weitergefuehrt und bekommt die feste
 TID 1 zugeordnet (diese wird nicht wiederverwendet).
 Die Implementation dieser Funktion ist notwendig, um den bereits vom
 OS erzeugten MainExecutionThread iin den Griff zu bekommen; d.h. ihn
 in die interne Threadverwaltung mit aufzunehmen. Dieser kann dann wie
 jeder andere Thread behandelt werden.
 Diese Funktion wird nicht zurueckkehren.
@See Also
 WamLeaveProcess
#endif /*DOCUMENTATION*/

void WamEnterProcess( int (*fnc)(long), long parm)
{
    static int initialized = 0;
    thread_t *thread;
    int rc;

    if( initialized )
	Bug("WamEnterProcess called twice");
    else {
	if( !(thread = AllocThreadSlot()) )
	    Bug("Wam: No initial thread slot");
      #ifdef OS20
	if( rc = DosCreateMutexSem( NULL, &serializeSemaphore, 0, FALSE ) )
      #else
	rc = -1;
      #endif
	    Error(4,"Error creating internal serializeSemaphore: rc=%d", rc );
	initialized = 1;
	AddCleanUp( TerminationHandler, NULL );
	/* now startup the (already running) thread */
	thread->fnc = fnc;
	thread->parm = parm;
      #ifdef OS20
	thread->osTID= 1;
      #endif
	RunThread( thread );
	BUG();	/* NOTREACHED */
    }
}


#ifdef DOCUMENTATION
@Summary WamLeaveProcess
 #include <wk/wam.h>

 void WamLeaveProcess( flag, cc );
 int flag;	Reserved; should be 0
 int cc;	ConditionCode; returned to OS (errorlevel)
@Description
 Diese Funktion wird aufgerufen um den Process zu beenden.
 Sie wird nicht mehr zurueckkehren.
 Der Aufruf von exit() fuehrt intern auch zum Aufruf dieser
 Funktion und kann deswegen in anderen Library Funktionen
 vorkommen.
 Sie wird auch aufgerufen , wenn der letzte Thread beendet wird.
 Letztendlich ist ihr Aufruf deswegen nicht notwendig!
@See Also
 WamEnterProcess
#endif /*DOCUMENTATION*/

void WamLeaveProcess( int r, int cc )
{
    static volatile int sentinel=0;
    if( !sentinel ) {
      #ifdef OS20
	DosEnterCritSec();
      #endif
	if( !sentinel ) {  /* Wir brauchen hier nichts zu machen, da das */
	    sentinel = 1;  /* Cleanup ueber den lokalen TerminationHandler */
	  #ifdef OS20
	    DosExitCritSec();
	  #endif
	    exit(cc);	   /* durchgefuehrt wird */
	}
      #ifdef OS20
	DosExitCritSec();
      #endif
    }
}




#ifdef DOCUMENTATION
@Summary WamCreateThread
 #include <wk/wam.h>

 WamCreateThread( tid, fnc, parm , stack);
 int *tid;	    Return: Thread identifier
 int (*fnc)(long);  Function to execute by thread
 long parm;	    Paramater passsed to function
 size_t stack;	    Size of thread s stack (in bytes)
@Description
 Diese Funktion erzeugt einen neuen Thread, der dann die Funktion
 fnc mit dem Paramter parm aufruft. Wird fuer stack 0 angegeben, so wird
 ein default Wert von 8K angenommen.
 Die Resourcen fuer den Thread werden hier intern verwaltet und nach
 Beendigung freigegeben.
 Die Thread Funktion kann einen Return Wert zurueckgeben, der dann nach
 Beendigung der Funktion abgefragt werden kann. Will ein Thread diese
 Information nicht zurueckgeben, so kann er diese Information vor
 seiner Beendigung bereits lschen.
 Die Anzahl der laufenden und beendeten Threads ist begrentzt. Die beendeten
 Threads werden geloescht wenn WamWaitThread diesen ermittelt hat, oder
 fuer den Thread WamSetReleaseThreadFlag() aufgerufen wurde.
@Return Value
 0 = Okay oder Errorcode E_WAM_...
@See Also
 WamSetReleaseThreadFlag
 WamWaitThread
#endif /*DOCUMENTATION*/

int WamCreateThread( int *retTid, int (*fnc)(long), long parm, size_t stk )
{
    thread_t *thread;

    if( !(thread = AllocThreadSlot()) )
	return E_WAM_RESOURCE;
    if( !stk )
	stk = 8192;
    thread->fnc = fnc;
    thread->parm = parm;

#ifdef OS20
  #if __WATCOMC__ || EMX
    thread->osTID = _beginthread( RunThread, NULL, stk, thread );
    if( thread->osTID == -1 ) { /* creation error */
	thread->state = tUNUSED;
	return E_WAM_THREAD;
    }
  #else
    if( DosCreateThread( &thread->osTID,
			 (PFNTHREAD)RunThread,
			 (ULONG)thread,
			 0, /* Immediate Execution, default stack initial. */
			 stk )	    ) {
	/* creation error */
	thread->state = tUNUSED;
	return E_WAM_THREAD;
    }
    xassert( thread->osTID != 1 );
  #endif
#endif
    *retTid = thread->tid;
    return 0;
}



#ifdef DOCUMENTATION
@Summary WamSetReleaseThreadFlag
 #include <wk/wam.h>

 int WamSetReleaseThreadFlag( tid );
 int tid;	Thread ID or 0 for current thread
@Description
 Nach >Bendigung eines Threads werden einige Informationen ueber
 den Thread, und damit auch Resourcen, solange behalten, bis
 WamWaitThread diese Informationen abgeliefert hat.
 Um nicht unnoetigerweise Resourcen zu blockieren (z.B. bei haeufig
 erzeugten Threads, die keine RueckgabeWerte liefern muessen) kann der
 Thread selbst, oder jeder andere Thread, zu jedem Zeitpunkt hiermit
 ein Flag setzen, welches diese Resourcen unmittelbar nach dem Ende
 des Thread freigibt; dieses hat dann auch zur Folge das mit WamWaitThread
 nicht auf diesen Thread gewartet werden kann.
@Return Value
 0 = Okay or E_WAM_INVTID
@See Also
 WamWaitThread
#endif /*DOCUMENTATION*/

int WamSetReleaseThreadFlag( int tid )
{
    thread_t *t;

    if( t = GetThreadSlot( tid ? tid : WamGetMyTID() ) ) {
	t->cc = -1 ;
	return 0;
    }
    return E_WAM_INVTID;
}


#ifdef DOCUMENTATION
@Summary WamWaitThread
 #include <wk/wam.h>

 int WamWaitThread( tid, rcc );
 int *tid;	    Ptr to a threadID or 0 for any thread
		    Returns: tid of terminated thread
 int *rcc;	    Return: Completion Code of thread
@Description
 Wartet auf die Beendigung des angegebenen bzw. eines beliebigen
 Threads und gibt dessen Return Code zurueck.
@Return Value
 0 = Okay: a thread has terminated
 E_WAM_INVTID  = No Thread with this tid is running
 E_WAM_NOTRUN  = No other threads running
@See Also
 WamCheckThread
 WamCreateThread
 WamReleaseThreadFlag
#endif /*DOCUMENTATION*/

int WamWaitThread( int *ptid, int *rcc )
{
    int err;

    while( (err = WamCheckThread(ptid,rcc)) == E_WAM_NOTTERM )
      #ifdef OS20
	DosSleep(0);	/* give up timeslice */
      #else
	Sleep(0);
      #endif
    return err;
}



#ifdef DOCUMENTATION
@Summary WamCheckThread
 #include <wk/wam.h>

 int WamCheckThread( tid, rcc );
 int *tid;	    Ptr to a threadID or 0 for any thread
		    Returns: tid of terminated thread
 int *rcc;	    Return: Completion Code of thread
@Description
 Testet ob der angegebenen bzw. ein beliebiger
 Thread beendet ist und gibt dessen Return Code zurueck.
@Return Value
 0 = Okay: a thread has terminated
 E_WAM_INVTID  = No Thread with this tid is running
 E_WAM_NOTRUN  = No other threads running
 E_WAM_NOTTERM = No threads have terminated (but still some are running)
@See Also
 WamWaitThread
 WamCreateThread
 WamReleaseThreadFlag
#endif /*DOCUMENTATION*/

int WamCheckThread( int *ptid, int *rcc )
{
    int tid,i;
    thread_t *t;

    tid = *ptid;
    if( tid )
	t = GetThreadSlot( tid );
    else {
	for(t=NULL,i=0; i < MAX_THREADS; i++ ) {
	    if( threadTbl[i].state == tWAIT ) {
		t = threadTbl + i;
		break;
	    }
	}
    }
    if( !t )
	return tid ? E_WAM_INVTID : E_WAM_NOTRUN;
    if( t->state != tWAIT )
	return E_WAM_NOTTERM;
    *ptid = t->tid;
    *rcc  = t->cc;
    t->state = tUNUSED;
    return 0;
}



#ifdef DOCUMENTATION
@Summary WamSuspendThread
 #include <wk/wam.h>

 int WamSuspendThread( tid );
 int tid;	ThreadID or 0 for all threads but the current
@Description
 Friert die Ausfuehrung des angegebenen Threads ein; dies kann auch der eigene
 Thread sein. Falls 0 als tid angegeben werden, so werden alle anderen Threads
 angehalten und nur der eigene weiterausgefuehrt (dies entspricht dann
 EnterCriticalSection). Falls der eigene angegeben wird, so kehrt die
 Funktion natuerlich erst zurueck, wenn dieser wieder von einem anderen
 Process aufgetaut wurde.
@Return Value
 0 = Okay: thread(s) frozen
 E_WAM_INVTID  = Invalid ThreadID
@See Also
 WamResumeThread
#endif /*DOCUMENTATION*/

int WamSuspendThread( int tid )
{
    BUG(); /*NOTREACHED*/
    return -1;
}


#ifdef DOCUMENTATION
@Summary WamResumeThread
 #include <wk/wam.h>

 int WamResumeThread( tid );
 int tid;	ThreadID or 0 for all threads but the current
@Description
 Taut die Ausfuehrung des angegebenen Threads auf; dies kann naturgemaess
 nicht der eigene Thread sein. Falls 0 als tid angegeben werden,
 so werden alle anderen Threads aufgetaut. (dies entspricht dann
 LeaveCriticalSection).
@Return Value
 0 = Okay: thread(s) thawed
 E_WAM_INVTID  = Invalid ThreadID
@See Also
 WamSuspendThread
#endif /*DOCUMENTATION*/

int WamResumeThread( int tid )
{
    BUG(); /*NOTREACHED*/
    return -1;
}



#ifdef DOCUMENTATION
@Summary WamGetMyTID
 #include <wk/wam.h>

 int WamGetMyTID(void);
@Description
 Ermittelt die TID des aktuellen Threads.
@Return Value
 TID des aktuelle Prozesses
#endif /*DOCUMENTATION*/


int WamGetMyTID()
{
  #ifdef OS20
    TIB   *ptib;
    PIB   *ppib;
    TID   osTID;
    int i;

    DosGetInfoBlocks( &ptib, &ppib );
    osTID = ptib->tib_ptib2->tib2_ultid;

    for(i=0; i < MAX_THREADS; i++ )
	if( threadTbl[i].state != tRUN ||
	    threadTbl[i].state != tFROZ   ) /* may be i'm suspended meanwhile*/
	    if( threadTbl[i].osTID == osTID )
		return threadTbl[i].tid;
    Bug("WamGetMyTID: Orphaned (osTID=%#lx)", osTID );
    return -1;
  #else
    return -1;
  #endif
}


#ifdef DOCUMENTATION
@Summary WamSemaphore
 #include <wk/wam.h>

 int WamSemaphore( wamSemaphore_t *mark, int opCode, ulong timeout );
@Description
 Dies ist eine Mehrzweckfunktion, die nur ueber die Macros:
	DCL_SERIALIZE(a);	/* Hilfsvariable a definieren */
	BEGIN_SERIALIZE(a);	/* Beginn des serialisierten Zugriffs */
	BEGIN_SERIALIZE2(a,t);	/* dito, mit timeout Angabe */
	END_SERIALIZE(a);	/* Ende des serialisierten Zugriffs */

	RESET_SEMAPHORE(a);
	POST_SEMAPHORE(a);
	WAIT_SEMAPHORE(a);
	WAIT_SEMAPHORE2(a,t);
 benutzt werden soll.
 "mark" ist das Semaphore.
 "opCode" kann sein:
    1 = Request Mutex Semaphore
    2 = Release Mutex Semaphore
    3 = Reset Event Semaphore
    4 = Post Event Semaphore
    5 = Wait Event Semaphore
 "timeout" kann folgende Werte haben (bei opCode 1 und 5):
    0 = es wird nicht gewartet
    n = es wird n Millisekunden gewartet
    -1L = (0xffffffff) es wird beliebig lange gewartet.
@Return Value
 0 = Okay
 E_WAM_TIMEOUT = Timeout aufgetreten
@Notes
 Es kann nur der Timeout-Fehler auftrtetn, da andernfalls die Anwendung
 mit einem schweren Fehler abgebrochen wird.
#endif /*DOCUMENTATION*/


int WamSemaphore( wamSemaphore_t *sem, int opCode, ulong timeout )
{
    int rc;
    ulong dummy;

  /*printf("Enter: WamSemaphore(%p, %d, %lu)\n", sem, opCode, timeout );*/
    switch( opCode ) {
      case 1: /* request mutex semaphore */
	/* first check wether we have to create the semaphore */
	if( !*sem ) { /* once again, but protected by a semaphore */
	    RequestSerialization();
	    if( !*sem ) {
	      #ifdef OS20
		if( rc = DosCreateMutexSem( NULL, sem, 0, FALSE ) )
		    Error(4,"Error creating semaphore: rc=%d", rc );
	      #endif
		if( !*sem )  /* need this for automatic semaphore creation */
		    Bug("Semaphore with handle 0 has been created");
	    }
	    ReleaseSerialization();
	}
	/*  ... and request it */
      #ifdef OS20
	if( rc = DosRequestMutexSem( *sem , timeout ) )
	    if( rc == ERROR_TIMEOUT && timeout != -1L )
		{ rc = E_WAM_TIMEOUT; goto leave; }
	    else
		Error(4,"Error requesting semaphore: rc=%d", rc );
      #endif
	break;

      case 2: /* release mutex semaphore */
      #ifdef OS20
	if( rc = DosReleaseMutexSem( *sem ) )
	    Error(4,"Error releasing semaphore: rc=%d", rc );
      #endif
	break;

      case 3: /* reset event semaphore */
	/* first check wether we have to create the semaphore */
	if( !*sem ) { /* once again, but protected by a semaphore */
	    RequestSerialization();
	    if( !*sem ) {
	       #ifdef OS20
		if( rc = DosCreateEventSem( NULL, sem, 0, TRUE ) )
		    Error(4,"Error creating event semaphore: rc=%d", rc );
	       #endif
		if( !*sem )  /* need this for automatic semaphore creation */
		    Bug("EventSemaphore with handle 0 has been created");
	    }
	    ReleaseSerialization();
	}
	/*  ... and reset it */
      #ifdef OS20
	if( rc = DosResetEventSem( *sem , &dummy ) )
	    if( rc == 300 ) {
		puts("Double reset of semaphore (rc=300)");
		rc = 0;
	    }
	    else
		Error(4,"Error resetting semaphore: rc=%d", rc );
      #endif
	break;

      case 4: /* post event semaphore */
	if( !*sem )
	    Bug("Attempt to post an not created semaphore");

      #ifdef OS20
	if( rc = DosPostEventSem( *sem ) )
	    if( rc == 299 ) /* 299 = already posted */
		rc = 0;
	    else
		Error(4,"Error posting semaphore: rc=%d", rc );
      #endif
	break;

      case 5: /* wait event semaphore */
	/* first check wether we have to create the semaphore */
	if( !*sem ) { /* once again, but protected by a semaphore */
	    RequestSerialization();
	    if( !*sem ) {
	      #ifdef OS20
		if( rc = DosCreateEventSem( NULL, sem, 0, TRUE ) )
		    Error(4,"Error creating event semaphore: rc=%d", rc );
	      #endif
		if( !*sem )  /* need this for automatic semaphore creation */
		    Bug("EventSemaphore with handle 0 has been created");
	    }
	    ReleaseSerialization();
	}
	/*  ... and wait */
      #ifdef OS20
	if( rc = DosWaitEventSem( *sem , timeout ) )
	    if( rc == ERROR_TIMEOUT && timeout != -1L )
		{ rc = E_WAM_TIMEOUT; goto leave; }
	    else
		Error(4,"Error waiting for semaphore: rc=%d", rc );
      #endif
	break;


      default: BUG(); /*NOTREACHED*/
    }
  leave:
    /*printf("Leave: WamSemaphore: rc=%d\n", rc );*/
    return rc;
}



/**************************************************
 *************	Test & Debug Suite ****************
 **************************************************/





/**** end of file ****/
