/* [wam/pmmain.c wk 31.1.93] PM Main Module
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Dies implemetiert das GUI fuer den OS/2 2.0
 * Presentation Manager.
 * Ihr arbeiten 2 Thread die untereinander syncronsiert werden
 *  - GUI-Thread (intern)
 *  - Execution-Thread (Kommunikation mit andren Modulen)
 *
 * leider muessen wir immer noch den execution Thread anhalten
 * bis eine Rueckmeldung vom GUI-Thread kommt, da ansonsten
 * die Messages in einer Falschen reihenfolge ankommen und es einen
 * deadlock gibt.
 ******************************************************
 */

#include <wk/tailor.h>
RCSID("$Id: pmmain.c,v 1.18 1996/09/25 16:20:54 wk Exp $")
#if !defined(OS20)
    #error This implementation is only for OS/2 v2.0
#endif
#define OS20_PM 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <wk/lib.h>
#include <wk/string.h>
#include <wk/environ.h>
#include <wk/errorhd.h>

#define INCL_WIN 1
#define INCL_GPI 1
#include <os2.h>


#include <wk/wam.h>
#define WAM_PM_MODULE -1
#include "version.h"
#include "wammain.h"
#include "wamgui.h"


#define DEBUG_MSG_MAIN	    0
#define DEBUG_MSG_DLG	    0
#define DEBUG_MSG_ENTRY     0
#define DEBUG_MSG_BUTTON    0
#define DEBUG_MSG_LSTBOX    0
#define DEBUG_MSG_SCROLLBAR 0
#define DEBUG_STORE_EVENT   0

#define MAKE_DLL 1

HMODULE GetModuleHandle(void);

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define DEFAULT_LISTBOXFONT "10.System Monospaced"
#define MAX_FORMS 50
#define MAX_ADDMENUES 5  /* max. Anzahl hinzuzufuegender menueitems mit */
			 /* einer Operation (willkuerlich gewaehlter Wert)*/

#define WM_USER_CLOSE (WM_USER +  1 )	 /* close dialog */

#define WM_USER_OPEN  (WM_USER +  2 )	 /* open dialog */
    /* mp1 := Ptr (formInfo_t*) to slot in FormTbl */

#define WM_USER_GETITEMTEXT   (WM_USER + 3 )
    /* mp1 := itemText_t*  */

#define WM_USER_PUTITEMTEXT   (WM_USER + 4 )
    /* mp1 := itemText_t*  */

#define WM_USER_MESSAGEBOX    (WM_USER + 5 )
    /* mp1 := msgBox_t*  */

#define WM_USER_DELETEALL     (WM_USER + 6 )
    /* mp1 := itemText_t* (only dlgItem used)*/

#define WM_USER_INSERTITEM    (WM_USER + 7 )
    /* mp1 := itemText_t* */

#define WM_USER_ENABLEITEM    (WM_USER + 8 )
    /* mp1 := itemBool_t* */

#define WM_USER_SETFOCUS      (WM_USER + 9 )
    /* mp1 := 0  oder dlgItem id */
    /* mp2 := 0 */

#define WM_USER_SETTITLE      (WM_USER + 10)
    /* mp1 := title string */

#define WM_USER_GETSELECTION  (WM_USER + 11)
    /* mp1 := selection_t* */

#define WM_USER_PUTSELECTION  (WM_USER + 12)
    /* mp1 := selection_t* */

#define WM_USER_GETPUTCHECKSTATE (WM_USER + 13)
    /* mp1 := selection_t* */

#define WM_USER_SPINBUTTON	 (WM_USER + 14)
    /* mp1 := spinbtn_t* */

#define WM_USER_VISIBILITY	 (WM_USER + 15)
    /* mp1 := long mode */

#define WM_USER_ADDMENU 	 (WM_USER + 16)
    /* mp1 := menuDesc_t */

#define WM_USER_QUERYFOCUS	 (WM_USER + 17)
    /* mp1 := ptr to Variable to hold the item/formId */

#define WM_USER_SETPOINTER	 (WM_USER + 18)

#define WM_USER_SETITEMSTYLE	 (WM_USER + 19 )

#define WM_USER_SETGRIDVALUE	 (WM_USER + 20 )

    /* codes for event reasons */
#define REASON_TERMINATED 0
#define REASON_POSTOPEN   1
#define REASON_QUIT	  2
#define REASON_CMD	  3
#define REASON_GOTFOCUS   5
#define REASON_LOSTFOCUS  6
#define REASON_TICK	  7
#define REASON_CHANGE	  8


#define IDM_ATTENTION 42
#define IDM_ATTENTION1 51
#define IDM_ATTENTION2 52
#define IDM_ATTENTION3 53
#define IDM_ATTENTION4 54
#define IDM_ATTENTION5 55
#define IDM_ATTENTION6 56
#define IDM_ATTENTION7 57
#define IDM_ATTENTION8 58
#define IDM_ATTENTION9 59


#define ID_TIMER  42
#if ID_TIMER >= TID_USERMAX
  #error Invalid timer id
#endif

static char *sysMenuText[] = {
	NULL,
	"~Unterbrechen"
    };
static MENUITEM sysMenuInfo[] = {
	{ MIT_END, MIS_SEPARATOR,0, 0	      },
	{ MIT_END, MIS_TEXT,	 0, IDM_ATTENTION  },
    };

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

/* resolve extern declarations for this module */
/* so that we do not need the  symbol collect pass */
id sym_postOpenInitialization;
id sym_closeRequested;
id sym_lostFocus;
id sym_terminated;
id sym_command;
id sym_gotFocus;
id sym_timerTick;
id sym_itemModified;


static wamSemaphore_t startupSemaphore;
static wamSemaphore_t eventOccuredSem;
DCL_SERIALIZE(eventQueueAccessSem);
#if 0
    /* access to formTbl should also be serialized */
#endif
static wamSemaphore_t waitForResultSem;

static HAB hab;
static HWND mainWindowHandle;
static HMODULE moduleHandle;

typedef struct s_itemInfo itemInfo_t;
struct s_itemInfo {
	itemInfo_t *nxt;
	id  item;
	int dlgItem;
	int itemType;
    };

typedef struct {
	id  form;	/* Form Object or 0 if unnused Slot */
	int dlg;	/* Resource ID of Dialog */
	HWND hwnd;	/* window handle (valid after open) */
	itemInfo_t *items; /* linked list of items */
	int  formValid;  /* nur wenn true, darf etwas an form gesendet werden*/
	volatile int  released;  /* scheduled for releasing */
	int  subId;
    } formInfo_t;

static formInfo_t formTbl[MAX_FORMS];

static HPOINTER   pointerTbl[4];
static volatile int currentPointer; /* current index in pointerTbl */

static int applyPositionsFlag = 1; /* default is to apply positions */
static char *applyPositionExt;	 /* malloced or NULL */
static int infoInBox = 0;
static int debugHeap;
static char *resourceDLL;	 /* malloced or NULL */
static int enterIsTab;
static int placeAtHome;
static char *listBoxFont;	 /* malloced or NULL */

/****************
 * My EventQueue (one item only)
 */

static struct {
	id form;
	id formItem;
	int reason;
    } lastEvent;


typedef struct {
	int    dlgItem;
	char   *buffer;
	size_t buflen;
    } itemText_t;

typedef struct {
	int    dlgItem;
	int    itemType;
	int    *flag;
    } itemBool_t;

typedef struct {
	const char *text;
	const char *title;
	char *allocedText;
	ulong flags;
	ulong rc;
	int   nowait;
    } msgBox_t;

typedef struct {
	int dlgItem;
	int itemType;
	int selection; /* -1 = none/disable all, 0.. 32767 index */
    } selection_t;     /* also used for getput checkstate */

typedef struct {
	int dlgItem;
	int mode;   /* 0 = set range, 1 = get value (returned in lower) */
	int lower;
	int upper;
    } spinBtn_t;

typedef struct {
	int dlgItem;
	int x, y;	/* 1 based */
	int datatype;
	const void *data;
	size_t datalen;
    } Tgridvalue;

typedef struct {
    int parent;
    int pos;
    char **textArr;  /* text Array, NULL indicates EndOfArray */
    int  *idArr;
} menuDesc_t;


/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/

static void  Err( const char *str );
static void  ErrIfError( const char *str );
static void  Fat( const char *str );
static void MyVprintf( const char *s, va_list arg);
static void Cleanup( void *dummy );
static void LoadResource(void);

static int ThreadMain(long);
static void SaveApplyDialogSizes( HWND hwnd, int mode );
static void AddMenue( HWND hwnd, menuDesc_t *mdes );
static void TranslateMenueStrings( HWND hwnd, int indent );
MRESULT EXPENTRY MainWndProc( HWND, ULONG, MPARAM, MPARAM );
MRESULT EXPENTRY StdDialogProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2);
static void StoreControlEvent(HWND hwnd,int item,int notiCode,long ctrlData);
static void StoreEvent( int reason, HWND hwnd, int item );
static void DoStoreEvent( int reason, id form, id formItem );
static itemInfo_t *FormFromHwnd( HWND hwnd, int dlgItem, id *formId );
static itemInfo_t *FormFromCtrlHwnd( HWND hwnd, HWND ctrlHwnd, id *formId );
static void GCFormTable(int);

static formInfo_t *GetFormSlot( id form );
static formInfo_t *GetFormAndItem( id form, id fitem, itemInfo_t **retitm );



#define StrMessage(a)	WamWindowStrMessage(a)

static void SubClass(HWND);


/**************************************************
 *************	Local Functions  ******************
 **************************************************/

static void
Err( const char *str )
{
    printf("Error: %#lx - %s\n", WinGetLastError(hab), str );
}

static void
ErrIfError( const char *str )
{
    ulong x = WinGetLastError(hab);

    if( x )
	printf("Error: %#lx - %s\n", x, str );
}

static void Fat( const char *str )
{
    Err(str);
    exit(4);
}

static void MyVprintf( const char *s, va_list arg)
{
    DCL_SERIALIZE(sentinel);
    static char buf[300];

    if( !BEGIN_SERIALIZE2( sentinel, 0 ) ) {
	vsprintf(buf, s, arg );
	if( !AreWSpaces( buf ) )
	    WamShowMessageBox( 0, WAM_INFO, buf );
	END_SERIALIZE( sentinel );
    }
}


static void Cleanup( void *dummy )
{
    GCFormTable(1);
}

static void
LoadResource()
{
    int rc;
    void *p;

    if( !resourceDLL ) {
	moduleHandle = 0;
	return;
    }

    if( !(p = WKLoadModule(resourceDLL,"#SYS-MODULE-HANDLE#",&rc)) )
	Error(2,"Error loading resource DLL '%s': rc=%d ", resourceDLL, rc);
    moduleHandle = (HMODULE)(ulong)p;
}


static int ThreadMain( long dummy )
{
    static ULONG frameFlags = FCF_TASKLIST;
    static char clientClassName[] = "WAMMainWindow";
    HMQ hmq;
    HWND hwndFrame, hwndClient;
    QMSG qmsg;
    int  snoopActive;


    if( !(hab = WinInitialize(0)) )
	Fat("retrieving anchor block");
  #if 0
    {  ulong list[20];
       int rc,i;
       rc = WinQueryCpList(hab, 20, list );
       if( !rc )
	    Fat("WinQueryCpList");
       for(i=0; i < rc ; i++ )
	   printf("CodePage: %lu\n", list[i] );
    }
  #endif
    if( !(hmq = WinCreateMsgQueue(hab,0)) )
	Fat("creating message queue" );

    if( !WinRegisterClass( hab, (PSZ)clientClassName, MainWndProc,
				CS_SIZEREDRAW, sizeof(PVOID) ) )
	Fat("Error registering class" );

    if( !(hwndFrame = WinCreateStdWindow(
		    HWND_DESKTOP,   /* parent window handle */
		    0,		    /* style of frame */
		    &frameFlags,    /* ptr to control data */
	       (PSZ)clientClassName,/* Client window class name */
		    NULL,	    /* titlebar text */
		    0L, 	    /* style of client window */
		    moduleHandle,   /* module handle for resources */
		    0,		    /* ID of resource */
		    &hwndClient ))) /* returns: client window handle */
	Fat("creating main window" );
    if( !hwndClient )
	Fat("creating client window" );

    mainWindowHandle = hwndClient;

    WinSendMsg( hwndFrame, WM_SETICON,
		   (MPARAM)WinLoadPointer(HWND_DESKTOP, 0, 15000 ),
			   NULL );

    pointerTbl[0] = WinQuerySysPointer( HWND_DESKTOP, SPTR_ARROW, FALSE);
    pointerTbl[1] = WinQuerySysPointer( HWND_DESKTOP, SPTR_WAIT, FALSE);
    pointerTbl[2] = WinLoadPointer(HWND_DESKTOP, moduleHandle, 15001 );
    pointerTbl[3] = WinLoadPointer(HWND_DESKTOP, moduleHandle, 15002 );
    currentPointer = 0;


    snoopActive = ErrorSnoop(0);
    if( !snoopActive && infoInBox )
	ErrorRgMonitor( MyVprintf ); /* for Error(), Bug() etc. */
    /* tell other threads about working message queue */
    POST_SEMAPHORE( startupSemaphore );

    /* no events yet */
    RESET_SEMAPHORE( eventOccuredSem );

    WamPMInitializeHelp( hab, hwndFrame );
    while( WinGetMsg( hab, &qmsg, 0, 0, 0 ) )
	WinDispatchMsg( hab, &qmsg );


    if( !snoopActive && infoInBox ) /* fixme: was ist wenn infoInBox */
	ErrorRgMonitor( NULL );     /* zwischenzeitlich umgestellt wurde? */

    StoreEvent( REASON_TERMINATED, 0, 0);

    WamPMTerminateHelp();
    WinDestroyWindow( hwndFrame );
    WinDestroyMsgQueue( hmq );
    WinTerminate( hab );
    return 0;
}



/****************
 * mode 0 = Save, mode 1 = apply
 * Application is: "WkApps:<String from Call to CopyRight(16)><user>"
 *	   Key is: "Dlg<ResourceId desDialogs>.0Pos"
 *			(.0 = reserved for more instances)
 *	  Data is: binary (structure SWP)
 * user is optional and may be:
 *		"(" <username> ")"
 */

static void SaveApplyDialogSizes( HWND hwnd, int mode )
{
  /*SWP   swp;
    ulong cx;*/
    formInfo_t *fi;
    char appbuf[60];
    char keybuf[20];


    if( mode && !applyPositionsFlag )
	return;

    fi = WinQueryWindowPtr(hwnd, 0);
    if( !fi ) {
	puts("fi missing");
	return;
    }
    if( !fi->dlg ) {
	puts("fi->dlg missing");
	return;
    }
    strcpy( appbuf, "WkApps:" );
    mem2str( appbuf+7, wamApplicationName, DIM(appbuf)-2-7 );
    if( applyPositionExt ) {
	size_t n;
	strcat(appbuf,"(");
	n = strlen(appbuf);
	mem2str(appbuf+n, applyPositionExt, DIM(appbuf)-1-n);
	strcat(appbuf,")");
    }

    sprintf( keybuf, "Dlg%d.%dPos", fi->dlg, fi->subId );

  #if 0
    if( !mode ) { /* save current size */
	if( !WinQueryWindowPos( hwnd, &swp ) )
	    Err("getting size of dlg window");
	else if( !PrfWriteProfileData(HINI_PROFILE, appbuf, keybuf,
				 &swp, sizeof swp) )
	    Err("Error writing to profile");
    }
    else { /* apply size from ini file if any stored */
	cx = sizeof swp;
	if(!PrfQueryProfileData(HINI_PROFILE, appbuf, keybuf, &swp, &cx) )
	    Err("Error reading from profile");
	else if( cx != sizeof swp )
	    puts("Invalid data in profile");
	else {
	    if( !WinSetWindowPos(hwnd, 0,
				 swp.x,   /* x pos */
				 swp.y,   /* y pos */
				 swp.cx,  /* x size */
				 swp.cy,  /* y size */
				 SWP_MOVE|SWP_SIZE ) )
		Err("resizing dialog window");
	}
    }
  #else /* there is an API! */
    if( !mode ) { /* save current size */
	/*Info("Saving windowpos as   '%s' '%s'", appbuf, keybuf );*/
	if( !WinStoreWindowPos(appbuf, keybuf, hwnd) )
	    Err("WinRestoreWindowPos");
    }
    else {
	/*Info("restor windowpos from '%s' '%s'", appbuf, keybuf );*/
	if( !WinRestoreWindowPos(appbuf, keybuf, hwnd) )
	    Err("WinRestoreWindowPos");
    }
  #endif
}


/****************
 * Helper function to add an menue
 */

static void
AddMenue( HWND hwnd, menuDesc_t *mdes )
{
    MENUITEM mi[MAX_ADDMENUES];
    MENUITEM miHelp;
    int i,n, idx;
    char *p;
    HWND hwndMenu, subMenu;

    idx = mdes->pos;
    for(i=0; i < MAX_ADDMENUES && (p=mdes->textArr[i]); i++ ) {
	mi[i].iPosition   = idx == -1 ? MIT_END : idx++;
	mi[i].afStyle	  = *p ? MIS_TEXT : MIS_SEPARATOR;
	mi[i].afAttribute = 0;
	mi[i].id	  = mdes->idArr[i];
	mi[i].hwndSubMenu = 0;
	mi[i].hItem	  = 0;
    }
    n = i;


    hwndMenu = WinWindowFromID(hwnd,FID_MENU);
    if( !mdes->parent )
	subMenu = hwndMenu;
    else {
	WinSendMsg(hwndMenu, MM_QUERYITEM,
		    MPFROM2SHORT(mdes->parent, TRUE),
		    MPFROMP( &miHelp )	      );
	subMenu = miHelp.hwndSubMenu;
	if( !subMenu ) { /* create submenu */
	  #if 0
	    subMenu = WinCreateWindow(HWND_DESKTOP,
			WC_MENU, NULL,	0,  0, 0, 0,0,	0,
			0,  0,	NULL,  NULL);
	    if( !subMenu )
		Err("creating menuwindow");
	    miHelp.afStyle    |= MIS_SUBMENU;
	    miHelp.hwndSubMenu = subMenu;
	    WinSendMsg(hwndMenu, MM_SETITEM,
			    MPFROM2SHORT(0, TRUE),
			    MPFROMP( &miHelp )	);
	    WinSendMsg(hwndMenu, MM_QUERYITEM,
			MPFROM2SHORT(mdes->parent, TRUE),
			MPFROMP( &miHelp )	  );
	    subMenu = miHelp.hwndSubMenu;
	  #endif
	}
    }
    for(i=0; i < n; i++ )
	WinSendMsg(subMenu, MM_INSERTITEM,
		    MPFROMP( mi + i ),
		    MPFROMP( mdes->textArr[i] )  );
    WinSendMsg( hwnd, WM_UPDATEFRAME, MPFROMLONG(FCF_MENU), NULL );
}


/****************
 * Die Strings in den Menues aus latin-1 in local CP wandeln.
 * Die maximale Verschachtelungstiefe ist hier willkrlich
 * auf 20 festgelegt.
 */
static void
TranslateMenueStrings( HWND hwnd, int indent )
{
    char buffer[50];
    byte *p;
    ushort itemid, pos, itemcount;
    int i;
    MENUITEM mi;
    long al;

    itemcount=SHORT1FROMMR(WinSendMsg(hwnd, MM_QUERYITEMCOUNT, MPVOID, MPVOID));
    if( itemcount > 100 ) /* remember: This API was defined my M$ */
	itemcount = 100;  /* so lets be a little bit paranoid */
    for(pos=0; pos < itemcount ; pos++ ) {
	al = LONGFROMMR(WinSendMsg( hwnd, MM_ITEMIDFROMPOSITION,
				   MPFROMSHORT(pos), MPVOID ));
	if( al == MIT_ERROR )
	    continue; /* no id (Separator) */
	itemid = (ushort)al;
	i = SHORT1FROMMR(WinSendMsg( hwnd, MM_QUERYITEMTEXT,
			     MPFROM2SHORT(itemid, DIM(buffer)),
					      MPFROMP(buffer) ));
	buffer[i] = 0;
      #if 0
	printf("%*smenu hwnd=%lu pos=%hu id=%hu text='%s'\n", indent*3,"",
		(ulong)hwnd, pos, itemid, buffer);
      #endif
	for(p=buffer; *p ; p++ )
	    if( *p & 0x80 )
		break;
	if( *p ) {
	    for(p=buffer; *p; p++ )
		if( *p & 0x80 )
		    *p = MapIso2Ibm( *p, 1 );
	    if( !WinSendMsg(hwnd, MM_SETITEMTEXT,
			      MPFROMSHORT(itemid), MPFROMP( buffer ) ))
		Err("MM_SETITEMTEXT");
	}
	if( !WinSendMsg(hwnd, MM_QUERYITEM,
		    MPFROM2SHORT(itemid, FALSE), MPFROMP( &mi ) ))
	    Err("MM_QUERYITEM");
	else if( mi.hwndSubMenu && indent < 20 )
	    TranslateMenueStrings(mi.hwndSubMenu, indent + 1 );
    }
}



/****************
 * MainWindow (should never get visible)
 */

MRESULT EXPENTRY MainWndProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
 /* HPS   hps;
    RECTL rcl, rclInvalid;*/
    HWND h;
    formInfo_t *fi;
    msgBox_t *msgBox;


  #if DEBUG_MSG_MAIN
    printf("MainWnd %08lx mp1=%08lx mp2=%08lx %s\n",
				 hwnd, mp1, mp2, StrMessage(msg) );
  #endif
    switch( msg ) {
      case WM_USER_OPEN:
	fi = mp1;
	h = WinLoadDlg(HWND_DESKTOP,hwnd, StdDialogProc,
					  moduleHandle, fi->dlg, fi );
	if( !h ) {
	    fi->hwnd = 0;
	    Fat("Error Loading Dialog");
	}
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_CLOSE:
	/* terminate everything */
	WinSendMsg( hwnd, WM_CLOSE, 0, 0 );
	POST_SEMAPHORE( waitForResultSem );
	break;


      case WM_USER_MESSAGEBOX:
	msgBox = mp1;
	msgBox->rc = WinMessageBox(HWND_DESKTOP, hwnd,
				   (PSZ)msgBox->text,
				   (PSZ)msgBox->title,
				   0, msgBox->flags	 );
	if( msgBox->allocedText ) {
	    FREE(msgBox->allocedText);
	    msgBox->text = "";
	}
	if( !msgBox->nowait )
	    POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_SETPOINTER:
	WinSetPointer(HWND_DESKTOP, pointerTbl[currentPointer] );
	break;

      default: return WinDefWindowProc( hwnd, msg, mp1, mp2 );
    }
    return 0;
}

HMODULE
GetModuleHandle()
{
    return moduleHandle;
}


MRESULT EXPENTRY StdDialogProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    int ret = 0;
    int i;
    HWND mn, h;
    RECTL rfr, rmn;
    ulong cx,cy;
    formInfo_t *fi;
    itemText_t *itemText;
    msgBox_t *msgBox;
    itemBool_t *itemBool;
    selection_t *sel;
    itemInfo_t *itm;
    SWP swp;

  #if DEBUG_MSG_DLG
    printf("DIALOG %08lx mp1=%08lx mp2=%08lx %s\n",
				 hwnd, mp1, mp2, StrMessage(msg) );
  #endif
    switch( msg ) {
      case WM_INITDLG:
	fi = mp2;
	if( !WinSetWindowPtr(hwnd, 0, fi ) ) /* store fi for quick access */
	    Fat("set window ptr");
	fi->hwnd = hwnd;
	WinSendMsg( hwnd, WM_SETICON,
		       (MPARAM)WinLoadPointer(HWND_DESKTOP,moduleHandle,15000),
			       NULL );
	SubClass( hwnd );

	if( !(mn = WinLoadMenu(hwnd, moduleHandle, fi->dlg)) )
	    Err("Loading Menu");
	else {
	    TranslateMenueStrings( mn, 0 );
	    WinSendMsg( hwnd, WM_UPDATEFRAME, MPFROMLONG(FCF_MENU), NULL );
	    if( !WinQueryWindowRect( hwnd, &rfr) )
		Err("getting window size of dlg");
	    if( !WinQueryWindowRect( mn, &rmn) )
		Err("getting window size of menu");
	  #if 0
	    printf("dialg: L=%ld B=%ld R=%ld T=%ld\n",
			   rfr.xLeft, rfr.yBottom, rfr.xRight, rfr.yTop );
	    printf("menue: L=%ld B=%ld R=%ld T=%ld\n",
			   rmn.xLeft, rmn.yBottom, rmn.xRight, rmn.yTop );
	  #endif
	    cx = rfr.xRight - rfr.xLeft;
	    cy = rfr.yTop - rfr.yBottom + rmn.yTop - rmn.yBottom;
	    if( !WinSetWindowPos(hwnd, 0, 0, 0, cx, cy, SWP_SIZE ) )
		Err("sizing dialog window");
	}
	SaveApplyDialogSizes( hwnd, 1 /* apply */ );

	{  /* systemmenue erweitern */
	    MENUITEM miSysMenu;
	    HWND     hwndSysMenu, hwndSysSubMenu;
	    short    idSysMenu;

	    hwndSysMenu = WinWindowFromID( hwnd,FID_SYSMENU);
	    idSysMenu = SHORT1FROMMR( WinSendMsg( hwndSysMenu,
				       MM_ITEMIDFROMPOSITION, NULL, NULL));
	    WinSendMsg( hwndSysMenu, MM_QUERYITEM,
			MPFROM2SHORT( idSysMenu, FALSE ),
			MPFROMP( &miSysMenu ) );
	    hwndSysSubMenu = miSysMenu.hwndSubMenu;
	    for(i=0; i < DIM( sysMenuInfo ); i++ )
		WinSendMsg( hwndSysSubMenu, MM_INSERTITEM,
			    MPFROMP( sysMenuInfo + i ),
			    MPFROMP( sysMenuText[i]  )	);
	}
	break;

      case WM_COMMAND:
	switch( COMMANDMSG(&msg)->cmd ) {
	  case IDM_HELPHELPFORHELP:
	  case IDM_HELPEXTENDED:
	  case IDM_HELPINDEX:
	  case IDM_HELPTUTORIAL:
	  case IDM_HELPABOUT:
	    WamPMDisplayHelp(hwnd, COMMANDMSG(&msg)->cmd);
	    break;
	  case IDM_ATTENTION:
	    if( WinMessageBox( HWND_DESKTOP, hwnd,
		(PSZ)"Mchten Sie ein Unterbrechnungssignal "
		     "an diese Anwendung senden?\n\n"
		 "(Die Anwendung selbst wird hierdurch nicht abgebrochen)",
		(PSZ)wamApplicationName, 0,
		MB_OKCANCEL | MB_ICONQUESTION) == MBID_OK )
		WamUserAttention(0);
	    break;

	  case IDM_ATTENTION1:
	  case IDM_ATTENTION2:
	  case IDM_ATTENTION3:
	  case IDM_ATTENTION4:
	  case IDM_ATTENTION5:
	  case IDM_ATTENTION6:
	  case IDM_ATTENTION7:
	  case IDM_ATTENTION8:
	  case IDM_ATTENTION9:
	    WamUserAttention(COMMANDMSG(&msg)->cmd - IDM_ATTENTION1 + 11 );
	    break;

	  default:
	    StoreEvent( REASON_CMD, hwnd, COMMANDMSG(&msg)->cmd );
	}
	break;

      case WM_HELP:
	WamPMDisplayHelp(hwnd, SHORT1FROMMP(mp1) );
	/*WamPMDisplayHelp(hwnd, 0 );*/
	break;

      case HM_HELPSUBITEM_NOT_FOUND:
	break; /* return False = display extended help */


      case WM_CONTROL:
	StoreControlEvent( hwnd,  SHORT1FROMMP(mp1), /* controlid */
				  SHORT2FROMMP(mp1), /* notification code */
				  LONGFROMMP(mp2)    /* controlspecific */
						      );
	break;

      case WM_SYSCOMMAND:
	if( COMMANDMSG(&msg)->cmd == SC_CLOSE )
	    StoreEvent( REASON_QUIT, hwnd, 0 );
	else
	    ret = -1;
	break;

      case WM_TIMER:
	if( SHORT1FROMMP(mp1) == ID_TIMER )
	    StoreEvent( REASON_TICK, hwnd, 0 );
	else
	    ret = -1;
	break;

      case WM_MOUSEMOVE:
	WinSetPointer(HWND_DESKTOP, pointerTbl[currentPointer] );
	ret = 1;
	break;

      case WM_FOCUSCHANGE:
	if( WinIsWindowEnabled(hwnd) )
	    ret = -1; /* normal weiter */
	else if( !SHORT1FROMMP(mp2) ) /* loosing focus */
	    ret = -1; /* normal weiter */
	else /* getting focus, but disabled */
	    puts("NoFocus");
	break;

      case WM_ADJUSTWINDOWPOS:
	if( WinIsWindowEnabled(hwnd) )
	    ret = -1; /* normal weiter */
	else {/* getting focus, but disabled */
	    PSWP pswp;

	    pswp = PVOIDFROMMP(mp1);
	    /* put behind all windows */
	    pswp->hwndInsertBehind = HWND_BOTTOM;
	    return 0;
	}
	break;

      case WM_USER_CLOSE:
	SaveApplyDialogSizes( hwnd, 0 /* save */ );
	WinSendMsg( hwnd, WM_CLOSE, 0, 0 );
	fi = mp1;
	fi->released=1;
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_GETITEMTEXT:
	itemText = mp1;
	if( itemText->buffer ) {
	    i=WinQueryDlgItemText( hwnd, itemText->dlgItem,
					 itemText->buflen,
					 itemText->buffer );
	    /* errorabfrage anscheinend hier nicht moeglich */
	    itemText->buffer[i] = 0;
	}
	else { /* bad kludge: set the max length of an entry field */
	    if( !WinSendDlgItemMsg( hwnd, itemText->dlgItem,
					EM_SETTEXTLIMIT,
			   MPFROMSHORT((short)itemText->buflen), MPVOID ) )
		ErrIfError("EM_SETTEXTLIMIT");

	}
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_PUTITEMTEXT:
	itemText = mp1;
	if( !WinSetDlgItemText( hwnd, itemText->dlgItem,
					 itemText->buffer ))
	    Err("SetDlgItemText");
	if( placeAtHome )
	    WinSendDlgItemMsg( hwnd, itemText->dlgItem, EM_SETSEL,
					 MPFROM2SHORT(0,0), MPVOID );
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_SETGRIDVALUE: {
	Tgridvalue *g = mp1;

	if( !WinSendDlgItemMsg( hwnd, g->dlgItem,
			   VM_SETITEM,
			   MPFROM2SHORT(g->y, g->x),
			   MPFROMP( g->data )
			 ) )
	    Err("VM_SETITEM");
	POST_SEMAPHORE( waitForResultSem );
      } break;

      case WM_USER_DELETEALL:
	itemText = mp1;
	if( !WinSendDlgItemMsg( hwnd, itemText->dlgItem,
				      LM_DELETEALL, 0, 0 ))
	    ErrIfError("SendDlgItemMsg:LM_DELETEALL");
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_INSERTITEM:
	itemText = mp1;
	WinSendDlgItemMsg( hwnd, itemText->dlgItem, LM_INSERTITEM,
				MPFROMLONG(LIT_END), itemText->buffer );
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_MESSAGEBOX:
	msgBox = mp1;
	msgBox->rc = WinMessageBox(HWND_DESKTOP, hwnd,
				   (PSZ)msgBox->text,
				   (PSZ)msgBox->title,
				   0, msgBox->flags	 );
	if( msgBox->allocedText ) {
	    FREE(msgBox->allocedText);
	    msgBox->text = "";
	}
	if( !msgBox->nowait )
	    POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_ENABLEITEM:
	itemBool = mp1;
	if( itemBool->itemType == ITEMTYPE_MENU ) {
	    if( !WinEnableMenuItem(WinWindowFromID(hwnd,FID_MENU),
				   itemBool->dlgItem, *itemBool->flag%2 ) )
		Err("enabling menuitem");
	}
	else {	/* es handelt sich um ein anderes Item */
	    if( *itemBool->flag < 2 ) { /* disable/enable */
		if( !WinEnableControl(hwnd, itemBool->dlgItem,*itemBool->flag))
		    Err("enabling control");
		WinSendDlgItemMsg( hwnd, itemBool->dlgItem,
					WM_USER_SETITEMSTYLE,
			   MPFROMLONG(	 2 ),  /* margin */
			   MPFROM2SHORT((*itemBool->flag), 0) );
	    }
	    else if( *itemBool->flag < 4 ) {/* 2 := readonly / 3 := readwrite */
		WinSendDlgItemMsg( hwnd, itemBool->dlgItem,
					EM_SETREADONLY,
			   MPFROM2SHORT((*itemBool->flag==2), 0), 0 );
		WinSendDlgItemMsg( hwnd, itemBool->dlgItem,
					WM_USER_SETITEMSTYLE,
			   MPFROMLONG( 1 | 2 ),  /* tabstop & margin */
			   MPFROM2SHORT((*itemBool->flag==3), 0) );
	    }
	    else if( *itemBool->flag < 6 ) { /* 4 := insert / 5 := replace */
		WinSendDlgItemMsg( hwnd, itemBool->dlgItem,
					EM_SETINSERTMODE,
			   MPFROM2SHORT((*itemBool->flag==4), 0), 0 );
	    }
	}
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_SETFOCUS:
	i = LONGFROMMP(mp1);
	if( !i ) {
	    if( WinQueryFocus(HWND_DESKTOP) != hwnd )
		if( !WinSetFocus(HWND_DESKTOP, hwnd ) )
		    Err("setting focus to me");
	}
	else {	/* set focus to item */
	    if( !WinSetFocus(HWND_DESKTOP, WinWindowFromID(hwnd,i) ) )
		Err("setting focus to DialogItem");
	}
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_QUERYFOCUS: {
	id *idp;

	idp = mp1;
	h = WinQueryFocus(HWND_DESKTOP);
	if( h ) {
	    itm = FormFromCtrlHwnd( hwnd, h, NULL );
	    *idp = itm? itm->item : nil;
	}
	POST_SEMAPHORE( waitForResultSem );
      } break;


      case WM_USER_SETTITLE:
	{  char *p;
	   p = PVOIDFROMMP(mp1);
	if( !WinSetWindowText(hwnd, p ) )
	    Err("setting title");
	POST_SEMAPHORE( waitForResultSem );
	}
	break;

      case WM_USER_GETSELECTION:
	sel = mp1;
	sel->selection = LONGFROMMR(WinSendDlgItemMsg( hwnd, sel->dlgItem,
					LM_QUERYSELECTION,
					MPFROMSHORT(LIT_FIRST), 0 ));
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_PUTSELECTION:
	sel = mp1;
	if( sel->itemType == ITEMTYPE_VALUESET ) {
	    /*hacked, so we can set x/y coordinates */
	    ushort x, y;
	    x = (ulong)sel->selection & 0xffff;
	    y = ((ulong)sel->selection >>16) & 0xffff;
	    WinSendDlgItemMsg( hwnd, sel->dlgItem,
			       VM_SELECTITEM,
			       MPFROM2SHORT(y, x), 0 );
	}
	else {
	    if( !WinSendDlgItemMsg( hwnd, sel->dlgItem, LM_SELECTITEM,
				MPFROMSHORT(sel->selection), MPFROMLONG(TRUE) ))
		Err("select item");
	}
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_GETPUTCHECKSTATE:
	sel = mp1;
	if( sel->itemType == ITEMTYPE_MENU ) {
	    if( sel->selection == -1 )
		sel->selection = WinIsMenuItemChecked(
				    WinWindowFromID(hwnd,FID_MENU),
							sel->dlgItem);
	    else
		(void)WinCheckMenuItem(WinWindowFromID(hwnd,FID_MENU),
					     sel->dlgItem, sel->selection );
	}
	else {	/* es handelt sich um ein anderes Item */
	    i = WinQueryButtonCheckstate(hwnd, sel->dlgItem);
	    if( sel->selection != -1 )
		(void)WinCheckButton(hwnd,  sel->dlgItem, sel->selection );
	    sel->selection = i;
	}
	POST_SEMAPHORE( waitForResultSem );
	break;

      case WM_USER_SPINBUTTON: {
	spinBtn_t *a = mp1;

	if( !a->mode ) {
	    if( !WinSendDlgItemMsg( hwnd, a->dlgItem, SPBM_SETLIMITS,
					    MPFROMLONG(a->upper),
					    MPFROMLONG(a->lower) ) )
		Err("SPBM_SETLIMITS");
	}
	else if( a->mode == 1 ) {
	    if( !WinSendDlgItemMsg( hwnd, a->dlgItem, SPBM_QUERYVALUE,
				    MPFROMP(&a->lower), 0 ) )
		Err("SPBM_QUERYVALUE");
	}
	else {
	    if( !WinSendDlgItemMsg( hwnd, a->dlgItem, SPBM_SETCURRENTVALUE,
				    MPFROMLONG(a->lower), 0 ) )
		Err("SPBM_SETCURRENTVALUE");
	}
	POST_SEMAPHORE( waitForResultSem );
      } break;

      case WM_USER_VISIBILITY:
	switch( LONGFROMMP(mp1) ) {
	  case -111:
	    if( !WinSetWindowPos(hwnd, 0, 0, 0, 0, 0, SWP_RESTORE ) )
		Err("restoring dialog window");
	    else if( !WinShowWindow( hwnd, TRUE ) )
		Err("showing window");
	    else
		StoreEvent( REASON_POSTOPEN, hwnd, 0 );
	    break;

	  case -2: /* hide */
	    WinShowWindow( hwnd, FALSE );
	    break;
	  case -1: /* show */
	    WinShowWindow( hwnd, TRUE );
	    break;
	  case	1: /* restore */
	    WinSetWindowPos(hwnd, 0, 0, 0, 0, 0, SWP_RESTORE );
	    break;
	  case	2: /* minimize */
	    WinSetWindowPos(hwnd, 0, 0, 0, 0, 0, SWP_MINIMIZE);
	    break;
	  case	3: /* maximize */
	    WinSetWindowPos(hwnd, 0, 0, 0, 0, 0, SWP_MAXIMIZE);
	    break;
	  case 5: /* lock */
	    WinEnableWindow( hwnd, FALSE );
	    break;
	  case 6: /* unlock */
	    WinEnableWindow( hwnd, TRUE );
	    break;

	  case	11: /* restore if not maximized */
	    if( WinQueryWindowPos( hwnd, &swp ) )
		if( !(swp.fl & SWP_MAXIMIZE) )
		    WinSetWindowPos(hwnd, 0, 0, 0, 0, 0, SWP_RESTORE );
	    break;

	  default: BUG();
	}
	break;

      case WM_USER_ADDMENU:
	AddMenue( hwnd, mp1 );
	POST_SEMAPHORE( waitForResultSem );
	break;

      default: ret = -1; break;
    }
    return ret < 0 ? WinDefDlgProc( hwnd, msg, mp1, mp2 ) : MRFROMLONG(ret);
}




static void
StoreControlEvent( HWND hwnd,  int dlgItem,
		   int notiCode, long ctrlData )
{
    id form;
    itemInfo_t *itm;

    itm = FormFromHwnd( hwnd, dlgItem, &form );
    if( !itm )
	return; /* not registered */

  #if DEBUG_STORE_EVENT
    printf("ControlEvent: item=%d, type=%d, noti=%d, data=%08lx for itm %p\n",
			  dlgItem, itm->itemType, notiCode, ctrlData, itm );
  #endif
    if( itm->itemType == ITEMTYPE_ENTRY ) {
	/* 526,527 are set/kill-focus for spinbuttons
	 * this is not documented, but I need them
	 * why the hell all the different values?
	 */
	if( notiCode == EN_SETFOCUS || notiCode == 526 ) {
	    DoStoreEvent( REASON_GOTFOCUS, form, itm->item );
	    if( placeAtHome )
		WinSendDlgItemMsg( hwnd, dlgItem, EM_SETSEL,
				    MPFROM2SHORT(0,0), MPVOID );
	}
	else if( notiCode == EN_KILLFOCUS || notiCode == 527  )
	    DoStoreEvent( REASON_LOSTFOCUS, form, itm->item );
      #if 0 /* bad performance */
	else if( notiCode == EN_CHANGE	)
	    DoStoreEvent( REASON_CHANGE, form, itm->item );
      #endif
    }
    else if( itm->itemType == ITEMTYPE_LIST ) {
	if( notiCode == LN_ENTER )
	    DoStoreEvent( REASON_CMD, form, itm->item );
    }
    else if( !itm->itemType ) {
	if( notiCode == BN_CLICKED  ||	/* check boxes */
	    notiCode == BN_DBLCLICKED ) /* check boxes */
	    DoStoreEvent( REASON_CMD, form, itm->item );
    }
}



/****************
 * Liefert zu einem WindowHandle das entsprechende FormObject
 * zurck und falls ein DialogItem angegeben ist, eine Ptr
 * auf dessen InfoBlock.
 */

static itemInfo_t *FormFromHwnd( HWND hwnd, int dlgItem, id *formId )
{
    int i;
    id form;
    itemInfo_t *itm;

    itm = NULL;
    /* find the form */
    for(form=0, i=0; i < MAX_FORMS; i++ )
	if( formTbl[i].hwnd == hwnd ) {
	    form = formTbl[i].form;
	    itm  = formTbl[i].items;
	    break;
	}
    if( !form )
	Bug("Orphaned window (not linked to a form)");

    /* find the item */
    if( dlgItem ) {
	for( ; itm ; itm = itm->nxt )
	    if( itm->dlgItem == dlgItem )
		break;
    }

    *formId = form;
    return itm;
}


static itemInfo_t *FormFromCtrlHwnd( HWND hwnd, HWND ctrlHwnd, id *formId )
{
    int i;
    id form;
    itemInfo_t *itm;

    itm = NULL;
    /* find the form */
    for(form=0, i=0; i < MAX_FORMS; i++ )
	if( formTbl[i].hwnd == hwnd ) {
	    form = formTbl[i].form;
	    itm  = formTbl[i].items;
	    break;
	}
    if( form && itm && ctrlHwnd ) { /* look out for the item */
	for( ; itm ; itm = itm->nxt )
	    if( WinWindowFromID(hwnd,itm->dlgItem) == ctrlHwnd )
		break;
    }

    if( formId )
	*formId = form;
    return form ? itm : NULL;
}



/****************
 * Einen event speichern.
 */

static void
StoreEvent( int reason, HWND hwnd, int item )
{
    id form, formItem;
    itemInfo_t *itm;

    formItem = 0;
    if( reason == REASON_TERMINATED )
	form = Form; /* use the factory */
    else {
	itm = FormFromHwnd( hwnd, item, &form );
	if( item ) {
	    if( !itm )
		puts("Warning: Item not registered; sending to form");
	    else
		formItem = itm->item;
	}

    }
    DoStoreEvent( reason, form, formItem );
}


static void
DoStoreEvent( int reason, id form, id formItem )
{
    if( BEGIN_SERIALIZE2( eventQueueAccessSem, 60 ) ) {
	if( reason == REASON_TERMINATED )
	    Error(4,"Termination forced by GUI");
	puts("EVENT LOSSED!");
	return;     /* do not violate 1/10 second rule !*/
    }
	lastEvent.form = form;
	lastEvent.formItem = formItem;
	lastEvent.reason = reason;
    POST_SEMAPHORE( eventOccuredSem );
    END_SERIALIZE( eventQueueAccessSem );
}




/**************************************************
 *************	Global Functions  *****************
 **************************************************/


#ifdef DOCUMENTATION
@Summary WamStartupGui
 #include <wk/wam.h>

 void WamStartupGui(void);
@Description
 Initialisiert und startet das Gui-Subsystem.
 Die Funktion kehrt erst zurueck, sobald das GUI System bereit ist
 (dies laueft in einem neu erzeugten Thread)
#endif /*DOCUMENTATION*/

void _wamapi
WamStartupGui(void)
{
    int rc, tid;

    sym_postOpenInitialization = sym(postOpenInitialization);
    sym_closeRequested	       = sym(closeRequested	   );
    sym_lostFocus	       = sym(lostFocus		   );
    sym_terminated	       = sym(terminated 	   );
    sym_command 	       = sym(command		   );
    sym_gotFocus	       = sym(gotFocus		   );
    sym_timerTick	       = sym(timerTick		   );
    sym_itemModified	       = sym(itemModified	   );

    AddCleanUp( Cleanup, NULL );
    LoadResource();
    RESET_SEMAPHORE( startupSemaphore );
    if( rc =  WamCreateThread( &tid, ThreadMain, 0, 16384 ) )
	Error(4,"Cannot start thread: rc=%d", rc );
    else
	WAIT_SEMAPHORE( startupSemaphore );
    return;
}


/****************
 * Set an option for the GUI. This is a multi-purpose fuction to
 * set different options. Currently implemented:
 *  "applyPos" := VAL bestimmt, ob gespeicherte Position benutzt werden
 *		  soll, wenn ein Window wieder geffnet wird.
 *		  Kann auch vor der Initialisierung benutzt werden.
 *		  Default ist TRUE
 *  "applyPosExt:" := Kann benutzt werden, um eine Extension an den Key,
 *		      der das Window beschreibt, anzungen. VAL ist
 *		      hier nicht definiert und sollte 0 sein; der Key selbst
 *		      folgt direkt dem Doppelpunkt in OPTION.
 *  "infoInBox" := VAL bestimmt, ob die Funktionen Error(), Info() etc,
 *		   ihre Ausgabe auch in eine Message Box leiten sollen.
 *		   Dies funkltioniert nur, fall die Funktion ErrorSnoop()
 *		   false zurckliefert. Die Funktion ist vor der Initialisierung
 *		   des GUIs aufzurufen, zu einem spteren Zeitpunkt hat sie
 *		   keine Wirkung mehr. Default ist FALSE.
 *  "resourceDLL:" := Setzt den Namen der Resource DLL. Diese Option ist
 *		   vor der Initialisierung des GUIs zu setzne.
 *  "debugHeap"
 *  "enterIsTab"   : (bool) enter key in an  entry fields acts like
 *		     the TAB_KEY (note: DefPushButton can't not work)
 *  "placeAtHome"  : (bool) Wenn ein entryfeld den Focus erhlt, so
 *		     wird der cursor immer auf den Anfang des Felds gesetzt.
 *  "listBoxFont:" : font to be used in list-boxes
 *		     (defaults to "10.System Monospaced")
 */
void _wamapi
WamSetGUIOption( const char *option, int val )
{
    const char *p;
    if( !strcmp(option,"applyPos") )
	applyPositionsFlag = val;
    else if( (p = strstr(option, "applyPosExt:")) && option == p ) {
	FREE(applyPositionExt);
	if( p[12] )
	    applyPositionExt = xstrdup(p+12);
    }
    else if( !strcmp(option, "infoInBox") )
	infoInBox = val;
    else if( (p = strstr(option, "resourceDLL:")) && option == p ) {
	FREE(resourceDLL);
	if( p[12] )
	    resourceDLL = xstrdup(p+12);
    }
    else if( !strcmp(option, "debugHeap") )
	debugHeap = val;
    else if( !strcmp(option, "enterIsTab") )
	enterIsTab = val;
    else if( !strcmp(option, "placeAtHome") )
	placeAtHome = val;
    else if( (p = strstr(option, "listBoxFont:")) && option == p ) {
	FREE(listBoxFont);
	if( p[12] )
	    listBoxFont = xstrdup(p+12);
    }
}


/****************
 * Diese Funktion wartet auf einen Event und
 * und sendet hann eine entsprechende Message
 * mode 0 := normal
 *	-1 := peek only
 * Returns: 0 = Okay , -1 for quit, sonst fehlercode
 */

int _wamapi
WamWaitForEvent(int mode)
{
    id form, formItem;
    int reason;
    symbol_t sel;

  #ifdef MEM_DEBUG
    if( debugHeap )
	memdbg_check();
  #endif
    if( mode == -1 ) {	/* peek */
	static int level;
	level++;
	if( level > 1 )
	    Info("WamWaitForEvent(-1) overflow");
	level--;
	if( WAIT_SEMAPHORE2( eventOccuredSem , 0 ) )
	    return 0;
    }
    else {
	WAIT_SEMAPHORE( eventOccuredSem );
    }
    BEGIN_SERIALIZE( eventQueueAccessSem );
      RESET_SEMAPHORE( eventOccuredSem );
	form	 = lastEvent.form;
	formItem = lastEvent.formItem;
	reason	 = lastEvent.reason;
    END_SERIALIZE( eventQueueAccessSem );

    /* translate reason to selectors */
    switch( reason ) {
      case REASON_TERMINATED: sel = sym_terminated; break;
      case REASON_POSTOPEN:   sel = sym_postOpenInitialization; break;
      case REASON_QUIT:       sel = sym_closeRequested; break;
      case REASON_CMD:	      sel = sym_command; break;
      case REASON_GOTFOCUS:   sel = sym_gotFocus; break;
      case REASON_LOSTFOCUS:  sel = sym_lostFocus; break;
      case REASON_TICK:       sel = sym_timerTick; break;
      case REASON_CHANGE:     sel = sym_itemModified; break;
      default: BUG(); /*NOTREACHED*/
    }
    if( sel ) {
	/* Form suchen um sicherzustellen, das die noch gueltig ist*/
	if( GetFormSlot(form) ) {
	    if( sel == sym_itemModified )
		msg(formItem, sel);
	    else
		msg1(form, sel, formItem);
	}
	if( sel == sym_terminated )
	    return E_WAM_FRCDTERM;
    }
    return 0;
}


static void GCFormTable( int force )
{
    int i;
    formInfo_t *fi;
    itemInfo_t *itm,*ihlp;

    for(fi=formTbl, i=0; i < MAX_FORMS; i++,fi++ )
	if( fi->released || force ) {
	    for( itm = fi->items; itm ; itm = ihlp ) {
		ihlp = itm->nxt;
		free( itm );
	    }
	    fi->items = NULL;
	    fi->hwnd = 0;
	    fi->released = 0;
	    fi->form = nil; /* free this slot */
	}
}


/****************
 * Verbindet eine Form mit einem Dialog-Window
 */

void _wamapi
WamRegisterForm( id form,     /* Form Object */
		 int dlg )    /* Resource ID of dialog */
{
    int i;
    formInfo_t *fi;

    GCFormTable(0);

    /* and use that room */
    for( i=0; i < MAX_FORMS; i++ )
	if( !formTbl[i].form ) {
	    fi = formTbl + i;
	    fi->form = form;
	    fi->dlg  = dlg;
	    fi->hwnd = 0; /* noch nicht bekannt */
	    fi->items = 0; /* noch keine definiert */
	    fi->formValid = 1; /* formObject is valid */
	    fi->released=0;
	    fi->subId = 0;
	    return;
	}
    Bug("Out of free slots in formTbl (max=%d)", MAX_FORMS);
}



/****************
 * Verbindet ein FormItem mit einer form
 */

void _wamapi
WamRegisterFormItem( id form,	  /* form of this item */
		     id fItem,	  /* the formItem */
		     int dlgItem, /* resource Id of Formitem */
		     int iType	) /* ITEMTYPE_... or 0 */
{
    formInfo_t *fi;
    itemInfo_t *itm;

    if( !(fi=GetFormSlot(form)) )
	BUG();

    if( !fi->items )
	itm = fi->items = xcalloc( 1, sizeof *itm );
    else {
	for( itm = fi->items; itm->nxt ; itm = itm->nxt )
	    ;
	itm = itm->nxt = xcalloc( 1, sizeof *itm );
    }

    itm->item = fItem;
    itm->dlgItem = dlgItem;
    itm->itemType = iType;
  #if 0
    printf("REGISTER: item=%d, type=%d for itm %p\n", dlgItem, iType, itm );
  #endif
}


/****************
 * Oeffnet den Dialog.
 */

void _wamapi
WamOpenWindow( id form, int subId, void *reserved )
{
    formInfo_t *fi;
    HWND hwndParent;

    hwndParent = mainWindowHandle;

    if( !(fi=GetFormSlot(form)) )
	Bug("Attempt to open not registered form");
    xassert( fi->dlg );
    xassert( !fi->hwnd );
    fi->subId = subId;

    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( hwndParent, WM_USER_OPEN, MPFROMP( fi ), 0 ) )
	Err("posting WM_USER_OPEN");
    else
	WAIT_SEMAPHORE( waitForResultSem );

}


/****************
 * Nach Aufruf dieser Funktion darf nicht mehr auf diese Form
 * zugegriffen werden.
 */

void _wamapi
WamCloseWindow( id form )
{
    int i;
    formInfo_t *fi;

    if( form ) {
	/* locate the slot of the form */
	for(fi=NULL, i=0; i < MAX_FORMS; i++ )
	    if( formTbl[i].form && formTbl[i].form == form ) {
		fi = formTbl + i;
		break;
	    }
	if( !(fi=GetFormSlot(form)) ) {
	    Fatal("Attempt to close not registered form");
	    return;
	}
	xassert( fi->dlg );
	xassert( fi->hwnd );
	fi->formValid = 0; /* Form Object may now be invalid */
    }
    else {  /* close the (invisible) main window */
	fi->hwnd = mainWindowHandle;
    }

    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_CLOSE, MPFROMP(fi), 0 ) )
	Err("posting WM_USER_CLOSE");
    else
	WAIT_SEMAPHORE( waitForResultSem );

}


/****************
 * Den Text eines Items holen.
 * Hier habe ich einen miesen kludge eingebaut:
 * Wenn buffer NULL ist, so kann idese Funkltion benutzt werden,
 * um die max. Lnge eines Entryfileds zu setzen.
 */
void _wamapi
WamGetItemText( id form, id fitem, char *buffer, size_t buflen )
{
    formInfo_t *fi;
    itemInfo_t *itm;
    itemText_t mp1;
    byte *p;

    fi = GetFormAndItem(form,fitem, &itm);
    mp1.dlgItem = itm->dlgItem;
    mp1.buffer	= buffer;
    mp1.buflen	= buflen;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_GETITEMTEXT, MPFROMP( &mp1 ), 0 ) )
	Err("posting WM_USER_GETITEMTEXT");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    if( buffer )
	for(p=buffer; *p; p++ )
	    if( *p & 0x80 )
		*p = MapIbm2Iso( *p, 1 );
}


void _wamapi
WamPutItemText( id form, id fitem, char *buffer )
{
    int converted=0;
    formInfo_t *fi;
    itemInfo_t *itm;
    itemText_t mp1;
    byte *p;

    fi = GetFormAndItem(form,fitem, &itm);
    mp1.dlgItem = itm->dlgItem;
    mp1.buffer	= buffer;
    mp1.buflen	= 0; /* not needed */
    /* FUTURE:
     * das warten kann auch durch einen intelligenteren Mechanismus
     * ersetzt werden, der bracht nur dem FormItem zu Melden, das es angezeigt
     * wurde, und diese liefert zwischenzeitlich seinen eigenen Wert, ohne
     * hier nachzufragen (buffer darf sich aber zwischenzeitlich nicht
     * aendern, da hier keine Kopie gemacht wird).
     * Kann auch generalisiert werden, so dass Aenderungen von Items
     * nach einem Setzen einmalig an das FormItem gemeldet werden, welches
     * nur in diesem Fall bei einem GetValue nach dem aktuelle Stand
     * fragen muss.
     */
    if( buffer )
	for(p=buffer; *p; p++ )
	    if( *p & 0x80 ) {
		converted = 1;
		*p = MapIso2Ibm( *p, 1 );
	    }

    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_PUTITEMTEXT, MPFROMP( &mp1 ), 0 ) )
	Err("posting WM_USER_PUTITEMTEXT");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    if( converted ) {
	for(p=buffer; *p; p++ )
	    if( *p & 0x80 )
		*p = MapIbm2Iso( *p, 1 );
    }
}


/****************
 * Set a value into an item with an coordinate system
 * DATATYPE should be 0 for a string, DATALEN should then be 0.
 * returns: 0 in success
 */
int _wamapi
WamSetGridValue( id form, id fitem, int x, int y,
		 int datatype, const void *data, size_t datalen )
{
    formInfo_t *fi;
    itemInfo_t *itm;
    Tgridvalue mp1;

    if( datatype != 0 )
	return -1;
    fi = GetFormAndItem(form,fitem, &itm);
    mp1.dlgItem = itm->dlgItem;
    mp1.x	= x;
    mp1.y	= y;
    mp1.datatype= datatype;
    mp1.data	= data	  ;
    mp1.datalen = datalen ;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_SETGRIDVALUE, MPFROMP( &mp1 ), 0) )
	Err("posting WM_USER_SETGRIDVAlUE");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    return 0;
}



/****************
 * Enbale /diaable an item:
 * mode 0 := disable item
 *	1 := enable item
 *	2 := set Item to read-only
 *	3 := set item to read-write
 *	4 := set insert-mode for item
 *	5 := set replace-mode for item
 */

void _wamapi
WamEnableItem( id form, id fitem, int mode )
{
    formInfo_t *fi;
    itemInfo_t *itm;
    itemBool_t mp1;

    fi = GetFormAndItem(form,fitem, &itm);
    mp1.dlgItem = itm->dlgItem;
    mp1.itemType= itm->itemType;
    mp1.flag	= &mode;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_ENABLEITEM, MPFROMP( &mp1 ), 0) )
	Err("posting WM_USER_ENABLEITEM");
    else
	WAIT_SEMAPHORE( waitForResultSem );
}



/****************
 * fitem = nil : set focus to form
 *	   sonst setFocus to that item
 */

void _wamapi
WamSetWindowFocus( id form, id fitem )
{
    int dlgItem;
    formInfo_t *fi;
    itemInfo_t *itm;

    fi = GetFormSlot(form);
    xassert( fi );

    dlgItem = 0;
    if( fitem )
	for(itm = fi->items ;itm ; itm = itm->nxt )
	    if( itm->item == fitem ) {
		dlgItem = itm->dlgItem;
		break;
	    }
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_SETFOCUS, MPFROMLONG(dlgItem), 0 ) )
	Err("posting WM_USER_SETFOCUS");
    else
	WAIT_SEMAPHORE( waitForResultSem );
}



/****************
 * Bestimmt das Item, welches den Focus hat.
 * form = nil := Returns Form, welches den Focus hat
 *		 (noch nicht implementiert!)
 *  sonst     := Returns ItemId, welches den Focus hat
 *		 oder nil, wenn kein Item den Focus hat
 */

id _wamapi
WamQueryWindowFocus( id form )
{
    formInfo_t *fi;
    id retId;

    fi = GetFormSlot(form);
    xassert( fi );

    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_QUERYFOCUS, MPFROMP(&retId), 0 ))
	Err("posting WM_USER_QUERYFOCUS");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    return retId;
}


void _wamapi
WamSetWindowTitle( id form, const char *string )
{
    int converted=0;
    byte *p;
    formInfo_t *fi;
    const char *mp1;

    fi = GetFormSlot(form);
    if( !fi )
	return;
    /* FIXME: FIXME: It is a realy bad, bad hack, to cast
     * a const away and write into the string
     * (but I know, that this is not a constant string!)
     * We should use some preallocated buffers to do the conversion.
     */
    if( string )
	for(p=(byte*)string; *p; p++ )
	    if( *p & 0x80 ) {
		converted = 1;
		*p = MapIso2Ibm( *p, 1 );
	    }
    mp1 = string;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_SETTITLE, MPFROMP( mp1 ), 0 ) )
	Err("posting WM_USER_SETTITLE");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    if( converted ) {
	for(p=(byte*)string; *p; p++ )
	    if( *p & 0x80 )
		*p = MapIbm2Iso( *p, 1 );
    }
}

/****************
 *  In eine List box ein neues Item setzen oder lschen
 *  mode == -1 && !string := Lschen
 *  mode == -2		  := Insert at End
 */

void _wamapi
WamInsertListItem( id form, id fitem, int mode, char *string )
{
    int converted=0;
    byte *p;
    formInfo_t *fi;
    itemInfo_t *itm;
    itemText_t mp1;

    if( string )
	for(p=string; *p; p++ )
	    if( *p & 0x80 ) {
		converted = 1;
		*p = MapIso2Ibm( *p, 1 );
	    }
    fi = GetFormAndItem(form, fitem, &itm);
    mp1.dlgItem = itm->dlgItem;
    mp1.buffer	= string;
    mp1.buflen	= 0; /* not needed */
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, mode == -1 ? WM_USER_DELETEALL :
					    WM_USER_INSERTITEM,
					    MPFROMP( &mp1 ), 0 ) )
	Err("posting WM_USER_DELETEALL/INSERTITEM");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    if( converted ) {
	for(p=string; *p; p++ )
	    if( *p & 0x80 )
		*p = MapIbm2Iso( *p, 1 );
    }
}



int _wamapi
WamShowMessageBox( id form, int msgType, const char *text )
{
    byte *p;
    const char *s;
    formInfo_t *fi;
    msgBox_t msgBox;
    HWND hwnd;

    fi = GetFormSlot(form);
    hwnd = fi ? fi->hwnd : mainWindowHandle;

    msgBox.allocedText = NULL;
    for(s=text; *s; s++ ) {
	if( *s & 0x80 )
	    break;
    }
    if( *s ) {
	p = msgBox.allocedText	= xstrdup(text);
	msgBox.text  = msgBox.allocedText;
	for(; *p; p++ )
	    if( *p & 0x80 )
		*p = MapIso2Ibm( *p, 1 );
    }
    else
	msgBox.text  = text;

    msgBox.flags = MB_MOVEABLE;
    if( msgType & WAM_MSGBOX_NOWAIT ) {
	msgType &= ~WAM_MSGBOX_NOWAIT;
	msgBox.nowait = 1;
    }
    else
	msgBox.nowait = 0;


    switch( msgType ) {
      case WAM_INFO:	msgBox.title  = "Information";
			msgBox.flags |= MB_OK | MB_CUANOTIFICATION;
			break;
      case WAM_WARNING: msgBox.title  = "Warnung";
			msgBox.flags |= MB_OK | MB_CUAWARNING;
			break;
      case WAM_ERROR:	msgBox.title  = "Fehler";
			msgBox.flags |= MB_OK | MB_CUACRITICAL;
			break;
      case WAM_FATAL:	msgBox.title  = "Schwerer Fehler";
			msgBox.flags |= MB_OK | MB_CUACRITICAL;
			break;
      case WAM_ABORT:	msgBox.title  = "Schwerer Systemfehler";
			msgBox.flags |= MB_OK | MB_CUACRITICAL;
			break;
      case WAM_QUERY:	msgBox.title  = "Frage";
			msgBox.flags |= MB_YESNO | MB_ICONQUESTION;
			break;
      case WAM_VERIFY:	msgBox.title  = "Frage";
			msgBox.flags |= MB_OKCANCEL | MB_ICONQUESTION;
			break;
      case WAM_DEBUG:	msgBox.title  = "Debug Info";
			msgBox.flags |= MB_OK | MB_NOICON;
			break;
      default: BUG();
    }


    if( msgBox.nowait )
	;
    else
	RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( hwnd, WM_USER_MESSAGEBOX, MPFROMP( &msgBox ), 0 ) )
	Err("posting WM_USER_MESSAGEBOX");
    else if( !msgBox.nowait )
	WAIT_SEMAPHORE( waitForResultSem );
    else
	return WAM_OKAY;

    if( msgType == WAM_ABORT )
	exit(3);

    switch( msgBox.rc ) {
      case MBID_YES:
      case MBID_RETRY:
      case MBID_ENTER:
      case MBID_OK:	return WAM_OKAY;
      case MBID_NO:
      case MBID_IGNORE: return WAM_NOTOKAY;
      case MBID_HELP:
      case MBID_ABORT:
      case MBID_CANCEL: return WAM_CANCEL;
      default:
	Fatal("Fehler beim anzeigen einer MessageBox");
	return WAM_CANCEL;
    }
}



/****************
 * Bei start wird z.Z. nur -1 unterstuetzt
 * Returns: -1 = nichts selektiert
 *	    0.. = index des selektierten Items
 */

int _wamapi
WamGetListSelection( id form, id fitem, int start )
{
    formInfo_t *fi;
    itemInfo_t *itm;
    selection_t sel;

#if LIT_NONE != -1
    #error LIT_NONE has wrong value
#endif

    xassert(start == -1);

    fi = GetFormAndItem(form, fitem, &itm);
    sel.dlgItem = itm->dlgItem;
    sel.itemType= itm->itemType;
    sel.selection = -1;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_GETSELECTION, MPFROMP( &sel ), 0 ) )
	Err("posting WM_USER_GETSELECTION");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    if( sel.selection < -1 || sel.selection > 32767 ) {
	Fatal("invalid value %d return from QUERYSELECTION", sel.selection);
	sel.selection = -1;
    }
    return sel.selection;
}

/****************
 * set the selection.
 * a special mode is used to set selections for grids:
 * (assuming 32 bit integers):
 *  low-16 bits is x
 *  high-16 bits is y
 */

void _wamapi
WamPutListSelection( id form, id fitem, int index )
{
    formInfo_t *fi;
    itemInfo_t *itm;
    selection_t sel;

    fi = GetFormAndItem(form, fitem, &itm);
    sel.dlgItem = itm->dlgItem;
    sel.itemType= itm->itemType;
    sel.selection = index;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_PUTSELECTION, MPFROMP( &sel ), 0 ) )
	Err("posting WM_USER_PUTSELECTION");
    else
	WAIT_SEMAPHORE( waitForResultSem );
}



/****************
 * Setzt den CheckState oder fragt ihn ab.
 * state= -1 := Query Check state
 *	   0 := Reset Check Mark
 *	   1 := Set Check Mark
 */

int _wamapi
WamGetPutCheckState( id form, id fitem, int state )
{
    formInfo_t *fi;
    itemInfo_t *itm;
    selection_t sel;

    fi = GetFormAndItem(form, fitem, &itm);
    sel.dlgItem = itm->dlgItem;
    sel.itemType= itm->itemType;
    sel.selection = state;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_GETPUTCHECKSTATE, MPFROMP( &sel ), 0 ))
	Err("posting WM_USER_GETPUTCHECKSTATE");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    return sel.selection;
}



/****************
 * Kontrolliert (numerische) SpinButtons
 * state=  0 := Set range
 *	   1 := Query Value
 *	   2 := Set Value
 */

int _wamapi
WamSpinButtonControl( id form, id fitem, int mode, int lower, int upper)
{
    formInfo_t *fi;
    itemInfo_t *itm;
    spinBtn_t  a;

    fi = GetFormAndItem(form, fitem, &itm);
    a.dlgItem = itm->dlgItem;
    a.mode = mode;
    a.lower= lower;
    a.upper= upper;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_SPINBUTTON, MPFROMP( &a ), 0 ))
	Err("posting WM_USER_SPINBUTTON");
    else
	WAIT_SEMAPHORE( waitForResultSem );
    return a.lower;
}


/****************
 * Window  anzeigen / verstecken, restoren etc.
 *	mode -111= initialy Show Window (posts postopen)
 *	mode -2 = Hide Window
 *	mode -1 = Show Window
 *	mode  1 = Restore Window
 *	mode  2 = Minimize Window
 *	mode  3 = Maximize Window
 *	mode  5 = lock Window (disable)
 *	mode  6 = unlock Window (enable)
 *	mode 11 = restore if not maximized
 */

int _wamapi
WamWindowVisibility( id form, int mode )
{
    formInfo_t *fi;

    fi = GetFormSlot(form);
    xassert( fi );
    if( !WinPostMsg( fi->hwnd, WM_USER_VISIBILITY, MPFROMLONG( mode ), 0 ) )
	Err("posting WM_USER_VISIBILITY");
    return 0;
}


/****************
 * Ein Menue hinzufuegen
 *  parent := ItemId unter dem die neuen Menuepunkte eingefuegt
 *	      werden sollen. ( 0 = in Action Bar )
 *     pos := -1 := Am Ende einfuegen, ansonsten an diesem Index
 *		    einfuegen ( 0 based )
 *     Ein NULL Eintrag im textArr kennzeichnet das Ende des Array
 *	ein leerer String wird als SEPARATOR dargestellt.
 *     idArr muss genausoviele Eintraege wie textArr haben.
 */
void _wamapi
WamAddWindowMenu( id form, int parent, int pos,
		  char **textArr, int *idArr )
{
    formInfo_t *fi;
    menuDesc_t mdes;

    fi = GetFormSlot(form);
    xassert( fi );
    mdes.parent  = parent;
    mdes.pos	 = pos;
    mdes.textArr = textArr;
    mdes.idArr	 = idArr;
    RESET_SEMAPHORE( waitForResultSem );
    if( !WinPostMsg( fi->hwnd, WM_USER_ADDMENU, MPFROMP( &mdes ), 0 ) )
	Err("posting WM_USER_ADDMENU");
    else
	WAIT_SEMAPHORE( waitForResultSem );
}


void _wamapi
WamSetWindowTimer( id form, ulong milliseconds )
{
    formInfo_t *fi;

    fi = GetFormSlot(form);
    xassert( fi );
    if( (long)milliseconds == -1L ) {
	if( !WinStopTimer(hab, fi->hwnd, ID_TIMER ) )
	    Err("WinStopTimer failed");

    }
    else if( !WinStartTimer(hab, fi->hwnd, ID_TIMER, milliseconds ) )
	Err("WinStartTimer failed");
}



void _wamapi
WamSetWindowPointer( int nr )
{
    if( nr >= 0 && nr < DIM(pointerTbl) ) {
	currentPointer = nr;
	WinPostMsg( mainWindowHandle, WM_USER_SETPOINTER, 0, 0);
    }
}


static formInfo_t *GetFormSlot( id form )
{
    int i;

    for(i=0; i < MAX_FORMS; i++ )
	if( formTbl[i].form && formTbl[i].form == form && formTbl[i].formValid)
	    return formTbl + i;
    return NULL;
}



static formInfo_t *GetFormAndItem( id form, id fitem, itemInfo_t **retitm )
{
    int i;
    itemInfo_t *itm;
    formInfo_t *fi;

    for(fi=NULL,i=0; i < MAX_FORMS; i++ )
	if( formTbl[i].form && formTbl[i].form==form && formTbl[i].formValid){
	    fi = formTbl + i;
	    itm= fi->items;
	    break;
	}
    xassert( fi );
    xassert( itm );
    for( ; itm ; itm = itm->nxt )
	if( itm->item == fitem )
	    break;
    xassert( itm );
    xassert( itm->dlgItem );
    *retitm = itm;
    return fi;
}



/******************************
 -----------------------------*/

PFNWP OldCtrlEntryWndProc;
PFNWP OldCtrlButtonWndProc;
PFNWP OldCtrlLstBoxWndProc;
PFNWP OldCtrlScrollWndProc;





MRESULT EXPENTRY WndProcWamCtrlEntry(HWND hwnd, ULONG msg,
				     MPARAM mp1, MPARAM mp2)
{
    int ret = 0;
 /* HPS hps;
    RECTL rcl;*/
    ULONG val, mask;
    MRESULT mr;

  #if DEBUG_MSG_ENTRY
    printf("Entry  %08lx mp1=%08lx mp2=%08lx %s\n",
			      hwnd, (ulong)mp1, (ulong)mp2, StrMessage(msg) );
  #endif
    switch(msg) {
    #if 0
      case WM_PAINT:
	if( hps = WinBeginPaint(hwnd, NULL, NULL ) ) {
	    puts("painting...");
	    GpiErase(hps);
	    if( WinQueryWindowRect( hwnd, &rcl ) ) {
		if( !WinDrawText( hps, -1, "wk", &rcl,
				  CLR_NEUTRAL, CLR_BACKGROUND,
				  DT_CENTER
				) )
		    Err("WinDrawText");
	    }
	    else
		Err("WinQueryWindowRect");
	    if( !WinEndPaint(hps) )
		Err("WinEndPaint");
	}
	else
	    Err("WinBeginPaint");
	break;
     #endif


      case WM_SETFOCUS:
	val = WinQueryWindowULong( hwnd, QWL_STYLE );
	/** die logik kapier ich nicht mehr (wk 02.05.96) */
	if( SHORT1FROMMP(mp2) ) {
	    if( val & ES_RIGHT ) {
		val &= ~ES_RIGHT;
		if( !WinSetWindowULong( hwnd, QWL_STYLE, val ) )
		    Err("WinSetWindowULong");
		else {
		    WinSendMsg(hwnd, EM_SETSEL, MPFROM2SHORT(0, 0x3fff), 0 );
		    WinInvalidateRect( hwnd, NULL, FALSE);
		}
	    }
	}
	else {
	    if( WinQueryWindowULong( hwnd, QWL_USER ) & 1 ) {
		val |= ES_RIGHT;
		if( !WinSetWindowULong( hwnd, QWL_STYLE, val ) )
		    Err("WinSetWindowULong");
		else
		    WinInvalidateRect( hwnd, NULL, FALSE);
	    }

	}
	ret = -1;
	break;

      case WM_USER_SETITEMSTYLE:
	mask = 0;
	val = LONGFROMMP(mp1); /* what */
	if( val & 1 )
	    mask |= WS_TABSTOP;
	if( val & 2 )
	    mask |= ES_MARGIN;
	if( SHORT1FROMMP(mp2) ) { /* set tabstop and margin bits */
	    WinSetWindowBits( hwnd, QWL_STYLE, ~0, mask );
	}
	else { /* clear tabstop and margin bits */
	    WinSetWindowBits( hwnd, QWL_STYLE,	0, mask );
	}
	if( mask & ES_MARGIN )
	    WinInvalidateRect( hwnd, NULL, FALSE);
	ret = 0;
	break;

      case WM_CHAR:
	if( enterIsTab && ( SHORT2FROMMP( mp2 ) == VK_ENTER
			    || SHORT2FROMMP( mp2 ) == VK_NEWLINE ) ) {
	    mp2 = MPFROM2SHORT( SHORT1FROMMP(mp2), VK_TAB );
	}
	ret = -1;
	break;

      default: ret = -1; break;
    }

    mr = ret < 0 ? OldCtrlEntryWndProc( hwnd, msg, mp1, mp2 )
		   : MRFROMLONG(ret);
    return mr;
}



MRESULT EXPENTRY WndProcWamCtrlButton(HWND hwnd, ULONG msg,
				      MPARAM mp1, MPARAM mp2)
{
    int ret = 0;

  #if DEBUG_MSG_BUTTON
    printf("Button %08lx mp1=%08lx mp2=%08lx %s\n",
				 hwnd, mp1, mp2, StrMessage(msg) );
  #endif
    switch(msg) {

      default:	ret = -1; break;
    }
    return ret < 0 ? OldCtrlButtonWndProc( hwnd, msg, mp1, mp2 )
		   : MRFROMLONG(ret);
}


MRESULT EXPENTRY WndProcWamCtrlLstBox(HWND hwnd, ULONG msg,
				      MPARAM mp1, MPARAM mp2)
{
    int ret = 0;

  #if DEBUG_MSG_LSTBOX
    printf("LstBox %08lx mp1=%08lx mp2=%08lx %s\n",
				 hwnd, mp1, mp2, StrMessage(msg) );
  #endif
    switch(msg) {

  #if 0
      case WM_MEASUREITEM: {
	    FONTMETRICS Metrics;
	    hps = WinGetPS(hwnd);
	    GpiQueryFontMetrics(hps, (LONG)sizeof(FONTMETRICS), &Metrics );
	    WinReleasePS(hps);
	    lItemHeight = Metrics.lMaxBaselineExt;
	    wpResult = MRFROM2SHORT( LOUSHORT(Metrics.lMaxBaselineExt),
				     L_NOTES_TEXT * LOUSHORT(Metrics.lAveCharWidth));	      /* item WIDTH */
	}
	break;


      case WM_DRAWITEM: {
	 HOOK_DATA  DrawData;

	 POWNERITEM pOwn = (POWNERITEM)mp2;

	 USHORT     usFlags = DT_LEFT	  |	 /* Horizontally left justified */
			      DT_VCENTER  |	 /* Vertically	 centered */
			      DT_ERASERECT;	 /* auto ERASE rectangle, Thanks! */

	 if ( (pOwn->fsState	== pOwn->fsStateOld) || /* not a selection change */
	      (pOwn->fsStateOld && pOwn->fsState) )
	 {
	   /* Get text of item to DRAW */

	   WinSendMsg(pOwn->hwnd,
		      LM_QUERYITEMTEXT,
		      MPFROM2SHORT(pOwn->idItem, sizeof(DrawData)),
		      (MPARAM)&DrawData);

	   /* Determine if "half tone" drawing is required... */

	   if ( DrawData.useHalfTone == '1' )
	     usFlags |= DT_HALFTONE;

	   /* Draw using specified COLOR */

	    WinDrawText(pOwn->hps,	    /* target HPS */
		       -1,		   /* using PSZ string, let PM calculate */
		       DrawData.szMsg,	   /* item text */
		       &pOwn->rclItem,	   /* target list item rectangle */
   ExternalColorTranslation[pSpyData->LogicalColorTranslation[DrawData.logCLR-COLOR_BASE].iExtColor].lClrValue,
		       SYSCLR_ENTRYFIELD,  /* BG color for ListBox */
		       usFlags);
	 }
	 wpResult = MRFROMSHORT(TRUE);	  /* Keep PM from drawing it too */
	}
	break;

      case WM_SIZE:
	WinQueryWindowRect(hwnd,&rectl);
	WinSetWindowPos(hwndMsgList,
			HWND_TOP,
			0,0,
			(SHORT)rectl.xRight,
			(SHORT)rectl.yTop,
			SWP_SIZE | SWP_MOVE);
	sNbrOfItems = (SHORT)(rectl.yTop / max(1L, lItemHeight));
	break;
  #endif

      default: ret = -1; break;
    }
    return ret < 0 ? OldCtrlLstBoxWndProc( hwnd, msg, mp1, mp2 )
		   : MRFROMLONG(ret);
}


MRESULT EXPENTRY
WndProcWamCtrlScroll(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
    int ret = 0;

  #if DEBUG_MSG_SCROLLBAR
    printf("Scrollbar %08lx mp1=%08lx mp2=%08lx %s\n",
				 hwnd, mp1, mp2, StrMessage(msg) );
  #endif
    switch(msg) {


      default:	ret = -1; break;
    }
    return ret < 0 ? OldCtrlScrollWndProc( hwnd, msg, mp1, mp2 )
		   : MRFROMLONG(ret);
}

static void
SubClass( HWND hwnd )
{
    /*CLASSINFO classinfo;*/
    PFNWP pWindowProc;
    HENUM  henum;
    HWND   hwndNext;
    char buffer[50];
    ULONG val;


    if( !(henum = WinBeginEnumWindows(hwnd)) )
	Fat("WinBeginEnumWindows");

    while( hwndNext = WinGetNextWindow(henum) ) {
	val = WinQueryWindowULong( hwndNext, QWL_USER );
	if( val )
	    Fatal("non zero QWL_USER found =%08lx\n", val );
	if( !WinQueryClassName(hwndNext, DIM(buffer), buffer ) )
	    Err("WinQueryClassName");
	else if( *buffer == '#' && buffer[1] == '3' && !buffer[2] ) {
	    /* WC_BUTTON */
	    if( !(pWindowProc = WinSubclassWindow(hwndNext,
						    WndProcWamCtrlButton)) )
		Err("WinSubclassWindow");
	    if( OldCtrlButtonWndProc && OldCtrlButtonWndProc != pWindowProc )
		Bug("WinSubclass conflict: %s/%08lx-%08lx\n",
			buffer, OldCtrlButtonWndProc,  pWindowProc );
	    OldCtrlButtonWndProc = pWindowProc;
	}
	else if( *buffer == '#' && buffer[1] == '6' && !buffer[2] ) {
	    /* WC_ENTRYFIELD */
	    if( !(pWindowProc = WinSubclassWindow(hwndNext,
						    WndProcWamCtrlEntry)) )
		Err("WinSubclassWindow");
	    if( OldCtrlEntryWndProc && OldCtrlEntryWndProc != pWindowProc )
		Bug("WinSubclass conflict: %s/%08lx-%08lx\n",
			buffer, OldCtrlEntryWndProc,  pWindowProc );
	    OldCtrlEntryWndProc = pWindowProc;
	    val = WinQueryWindowULong( hwndNext, QWL_STYLE );
	    val = val & ES_RIGHT ? 1 : 0;
	    if( val )
		if( !WinSetWindowULong( hwndNext, QWL_USER, val ) )
		    Err("WinSetWindowULong");

	    /* eigentlich sollte in QWL_USER der ptr auf item */
	    /* gespeichert werden, fuer alle Items, ist z.Z. aber noch */
	    /* nicht vordringlich notwendig */

	}
	else if( *buffer == '#' && buffer[1] == '7' && !buffer[2] ) {
	    /* WC_LISTBOX */
	    FONTMETRICS fm;
	    HPS hps;


	    if( !(pWindowProc = WinSubclassWindow(hwndNext,
						    WndProcWamCtrlLstBox)) )
		Err("WinSubclassWindow");
	    if( OldCtrlLstBoxWndProc && OldCtrlLstBoxWndProc != pWindowProc )
		Bug("WinSubclass conflict: %s/%08lx-%08lx\n",
			buffer, OldCtrlLstBoxWndProc,  pWindowProc );
	    OldCtrlLstBoxWndProc = pWindowProc;

	    if( !listBoxFont )
		listBoxFont = xstrdup(DEFAULT_LISTBOXFONT);
	    WinSetPresParam( hwndNext, PP_FONTNAMESIZE,
			    (ULONG)strlen(listBoxFont)+1, listBoxFont);

	    hps = WinGetPS(hwndNext);
	    GpiQueryFontMetrics(hps, (long)sizeof(FONTMETRICS), &fm);
	    WinReleasePS(hps);

	    WinSendMsg( hwndNext, LM_SETITEMHEIGHT,
			(MPARAM)fm.lMaxBaselineExt, 0L);
	}
	else if( *buffer == '#' && buffer[1] == '8' && !buffer[2] ) {
	    /* WC_SCROLLBAR */
	    if( !(pWindowProc = WinSubclassWindow(hwndNext,
						    WndProcWamCtrlScroll)) )
		Err("WinSubclassWindow");
	    if( OldCtrlScrollWndProc && OldCtrlScrollWndProc != pWindowProc )
		Bug("WinSubclass conflict: %s/%08lx-%08lx\n",
			buffer, OldCtrlScrollWndProc,  pWindowProc );
	    OldCtrlScrollWndProc = pWindowProc;
	}

    }
    if( !WinEndEnumWindows(henum) )
	Err("WinEndEnumWindows");




  #if 0
    if( !WinQueryClassInfo(hab, WC_ENTRYFIELD, &classinfo) )
	Fat("queryClassInfo");

    printf("Old Class info:\n"
	   "style: %08lx\n"
	   " proc: %08lx\n"
	   " data: %08lx\n" ,
	    classinfo.flClassStyle,
	    classinfo.pfnWindowProc,
	    classinfo.cbWindowData  );
  #endif


}


/****************************************************
 ************ DLL Initialization Stuff **************
 ***************************************************/
#ifdef MAKE_DLL

const char * _wamapi
WamDLLInfoGUI( int level )
{
    const char *p;

    switch( level ) {
      case 10:
      case 0:	p = "WAM GUI-Library for OS/2 PM"
		    "; Copyright 1993,95 by Werner Koch (dd9jn)" ; break;
      case 1:
      case 11:	p = ""; break;
      case 16:
      case 13:	p = "WAMOS2PM"; break;
      case 14:	p = VERSION; break;
      case 15:	p = ""; break;
      case 23:	p = __DATE__ " " __TIME__; break;
      case 71:	p = WAMGUI_INTERFACE_VERSION; break;
      default: p = "?";
    }
    return p;
}

#endif /* MAKE_DLL */


/**** bottom of file ****/
