/* [wam/object.c wk 22.01.93] Root of all classes
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Dieses Modul implementiert die Class Object.
 * Object ist die Root aller Classes und hat deswegen einige
 * Besonderheiten.
 ******************************************************
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: object.c,v 1.11 1996/01/25 20:17:51 wernerk Exp $")
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>
#include "wamgui.h"

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define ATTIC_LIMIT 3

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/


id True  = NULL;
id False = NULL;
id Null  = NULL;

DCLSHAREDPART(Object)

BEGIN_DCLPRIVATEPART
END_DCLPRIVATEPART

static int trace_subclassing;
static int shutdownInProgress;

DCLSYM sym_notification;

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static isa_t	FindClassByName( symbol_t name );
static symbol_t FindClassByID( id );

static void CleanupAttics( void *dummy );

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

/****************
 * Eine Class anhand ihres Namens suchen
 * Returns: Id of Class or Nil if not found
 */

static isa_t FindClassByName( symbol_t name )
{
    isa_t rover;

    for( rover = Object->isa; rover ; rover = rover->nameLink )
	if( rover->name == name )
	    return rover;
    return Nil;
}


/****************
 * Eine Class anhand ihres Ptrs suchen
 * Returns: Name der Class oder 0 if not found
 */

static symbol_t FindClassByID( id obj )
{
    isa_t rover;

    for( rover = Object->isa; rover ; rover = rover->nameLink )
	if( rover == obj->isa ) {
	    xassert( rover->name );
	    return rover->name;
	};
    return 0;
}


/**************************************************
 *************	Global Functions  *****************
 **************************************************/


#ifdef DOCUMENTATION
@Summary WamSUCFinish
 #include <wk/wam.h>

 void WamSUCFinish(void);
@Description
 Diese Funktion ist als letzte SetupClass Funktion aufzurufen.
 Eventuell werden hier noch einige Operationen durchgefuehrt
 um die ClassHierachie aufzubauen.
@Return Value
@See Also
 WamSUC_Object
#endif /*DOCUMENTATION*/


void WamSUCFinish()
{
    /* Hier koennten z.B. auch HashTabellen aufgebaut werden */
    /* create predefined Objects */

    True  = newObj(Object);
    False = newObj(Object);
    Null  = newObj(Object);
    WamRehashMethodTables();
    AddCleanUp( CleanupAttics, NULL );
}




#ifdef DOCUMENTATION
@Summary WamSubclassClass
 #include <wk/wam.h>

 void WamSubclassClass( symbol_t parentName, id child );
@Description
 Diese Funktion dient der Initialisierung einer Class und wird deswegen
 nur aus der Funktion WamSUC_xxxx() aufgerufen. Alle notwendigen
 Daten der Class sollten bereits gesetzt sein. Die Class "parent"
 muss vor dem Aufruf bereits existieren.
  Nach dem Rueckkehr aus dieser Funktion ist die Class "parent" dann die
 SuperClass der Class "child"
#endif /*DOCUMENTATION*/


void
WamSubclassClass( symbol_t parentName, id child )
{
    isa_t parent;
    isa_t rover;

    if( !child->isa )
	Bug("Attempt to subclass class with isa not setup");
    if( !child->isa->name )
	Bug("Attempt to subclass unnamed class");
    if( trace_subclassing ) {
	Info("subclassing '%s' from '%s'", symName(child->isa->name),
				   parentName?symName(parentName):"?");
    }
    if( FindClassByID( child ) )
	Bug("Class (%s) already subclassed", symName(child->isa->name));
    if( child->isa->name == parentName )
	Bug("Class (%s) subclassed to itself", symName(parentName));

    if( !(parent = FindClassByName( parentName )) ) {
	if( parentName == sym_Array ) {
	    /* Bad Hack, but array is a better name than idArray but it is
	     * only a synonym, so .. */
	    WamSubclassClass( sym(IdArray) , child );
	    return;
	}

	Bug("SuperClass (%s) not found", parentName?symName(parentName):"?");
	/* statt Bug koennte auch gewartet werden bis die ParentClass */
	/* definiert ist ? - ist aber komplizierter, da die class annimmt, */
	/* sie sei bereits initialisiert, eventuell direkt unter objekt */
	/* einhngen und spter an die richtige stelle ... lieber nicht */
    }

    /* append class to list of all classes */
    for( rover = Object->isa; rover->nameLink ; rover = rover->nameLink )
	;
    rover->nameLink = child->isa;
    child->isa->nameLink = Nil;

    /* set super class */
    child->isa->super = parent;

    /* setup information about varsize
     * we do not really allocate space for the isa pointer of every class,
     * except for the RootClass (Object), the isa Pointer of the root
     * is shared by every class because this isa points to the Class and not
     * to Object - so we decrement the private Size and the offset.
     * This will not be used for class Object - but Object has no parent,
     * so there is no need for it.
     */
    child->isa->privateOff   = parent->privateSize - sizeof( isa_t );
    child->isa->privateSize += parent->privateSize - sizeof( isa_t );

    /* setup informations about the attic (reusable instances) */
    child->isa->attic.list = NULL;
    child->isa->attic.used = 0;
    child->isa->attic.limit = ATTIC_LIMIT;
}



id WamGetFactoryByName( symbol_t name )
{
    isa_t isa;
    char symbuf[128];

    xassert(name);
    isa = FindClassByName( name );
    if( !isa )
	return nil;

    if( !isa->selfLink )
	Bug("SelfLink of Factory Object '%s' is invalid",
	     WamQuerySymbolName( name, symbuf, DIM(symbuf) ));
    return isa->selfLink;
}


int WamIsMemberOf( id self, id aClass )
{
    if( self && aClass )
	return self->isa == aClass->isa;
    if( !self && !aClass )
	return 1; /* nil is member of Nil */
    return 0;
}

int WamIsKindOf( id self, id aClass )
{
    isa_t isa;

    if( self && aClass ) {
	for( isa = self->isa; isa; isa = isa->super )
	    if( isa == aClass->isa )
		return 1;
	return 0;
    }
    if( !self && !aClass )
	return 1; /* nil is kind of Nil */
    return 0;
}


void WamSetAtticLimit( id self, size_t newLimit )
{
    isa_t isa;

    if( !self )
	return ; /*not for nil*/
    isa = self->isa;
    if( isa->attic.used <= newLimit )
	isa->attic.limit = newLimit;
    else
	Error(0,"Warning: Can't decrease attic limit");
}


static void CleanupAttics( void *dummy )
{
    isa_t rover;
    id obj;

    for( rover = Object->isa; rover ; rover = rover->nameLink ) {
	while( obj = rover->attic.list ) {
	    rover->attic.list = obj->isa;
	    free(obj);
	}
	rover->attic.used = 0;
    }
}


/****************
 * This is a kind of hack, but the only way to solve the
 * bootstrapping problem: We need Symbols prior to the creation of
 * the Class Object but Symbols are in Class Symbol, which is
 * a Subclass of Object ..
 */

void WamSetupRawObject( id aClass, id obj )
{
    isa_t isa;
    isa = aClass->isa;
    isa->icount++;
    obj->isa = isa;
}

/**************************************************
 ******************  Methods  *********************
 **************************************************/


DCLFOBJFNC( new )
{
    id obj;
    isa_t isa;

    isa = self->isa;
    isa->icount++;
    if( isa->attic.used ) {
	obj = isa->attic.list;
	isa->attic.list = obj->isa;  /* is used as the link pointer */
	isa->attic.used--;
	memset(obj,0,isa->privateSize);
    }
    else {
      #ifdef MEM_DEBUG
	obj = xcalloc_debug(1, isa->privateSize, isa->eye, (int)isa->icount );
      #else
	obj = xcalloc(1, isa->privateSize );
      #endif
    }
    obj->isa = isa;
    return obj;
}



DCLOBJFNC( free )
{
    void *tmp;
    isa_t isa;

    if( self == True || self == False || self == Null  )
	return nil;  /* diese werden niemals freigegeben und auch keine */
		     /* kopien erstellt */

    isa = self->isa;
    isa->icount--;
    if( isa->attic.used < isa->attic.limit ) {
      #ifdef MEM_DEBUG /* set memory to an easy recognizeble value */
	memset(self,0xdd,isa->privateSize);
      #endif
	/* store this instance in the attic */
	tmp = isa->attic.list;
	isa->attic.list = self;
	self->isa = tmp;  /* abusing isa as link pointer */
	isa->attic.used++;
    }
    else
	free(self);
    return nil;
}



DCLOBJFNC( doesNotRecognize )	/* args: symbol */
{
    DCL_arg(symbol_t,m);
    char buf[200];

    WamQuerySymbolName( m, buf, DIM(buf)-15 );
    strcat(buf, " not recognized");
    msg1(self,sym_error, buf);
    return nil;
}


DCLOBJFNC( errorNoSuperClass )
{
    Error(0,"no super class for this object - returning nil");
    return nil;
}

DCLOBJFNC( subclassResponsibility )
{
    return msg1(self,sym_error, "Subclass should override this message");
}


DCLOBJFNC( error )
{
    DCL_arg(const char*, p);

    Error(2,"Error: %s\n", p);
    return self;
}



DCLOBJFNC( bounds )
{
    DCL_arg(size_t, n);
    DCL_arg(size_t, mx);

    msg4(self, sym_debug,
	 "Index %u is out of range [0...%u] in Instance of Class %s",
		n, mx, self->isa->eye );
    return self;
}


DCLOBJFNC( display )
{
    DCL_arg(const char*, s);
    WamShowMessageBox( 0, WAM_INFO | WAM_MSGBOX_NOWAIT, s );
    return self;
}


DCLOBJFNC( debug )
{
    DCL_arg(const char*, s);
    char buffer[800];

    vsprintf(buffer, s, arg_ptr );
    msg2( Form, sym_messageBox, WAM_DEBUG, buffer );
    return self;
}


DCLOBJFNC( terminated )
{
    Error(0,"Application terminated (propably by GUI)");
    exit(0); /*NOTREACHED*/
    return self;
}


DCLOBJFNC( shutdown )
{
    if( !shutdownInProgress ) {
	shutdownInProgress = 1;
     /* msg(Form, sym(terminate) ); */
	/* falls der andere Thread via user_close an mainwindow */
	/* beendet, so gibt es ein trap 0d, (thread problem) */
	/* mit exit gehts aber auch ;-) */
	printf("%s terminated.\n", wamApplicationName);
	exit(0); /*NOTREACHED*/
    }
    return self;
}


DCLOBJFNC_s( name )
{
    return self->isa->name;
}


DCLOBJFNC( class )
{
    return self->isa->selfLink;
}


DCLOBJFNC_i( isMemberOf ) /* for quick access: WamIsMemberOf() */
{
    DCL_arg(id,aClass);

    return self->isa == aClass->isa;
}

DCLOBJFNC_i( isKindOf )   /* for quick access: WamIsKindOf() */
{
    DCL_arg(id,aClass);
    isa_t isa;

    for( isa = self->isa; isa; isa = isa->super )
	if( isa == aClass->isa )
	    return 1;
    return 0;
}



DCLOBJFNC_I( size )
{
    return 0;
}


DCLOBJFNC( show )
{
    if( self == True )
       Info("Object 'True'");
    else if( self == False )
       Info("Object 'False'");
    else if( self == Null )
       Info("Object 'Null'");
    else
       Info("Instance of Class '%s'", symName(self->isa->name) );
    return self;
}

DCLOBJFNC( copy )
{
    if( self == True || self == False || self ==Null )
	return self;
    return msg(self->isa->selfLink,sym_new);
}



/****************
 * Ein String Objekt aus dem Objekt erzeugen
 */

DCLOBJFNC( asString )
{
    char *p;

    if( self == True )
	return newString("True");
    else if( self == False )
	return newString("False");
    else if( self == Null )
	return newString("Null");
    p = symName(self->isa->name);
    *p = tolower(*p);
    return newString(p);
}


DCLOBJFNC( asDBString )
{
    char *p;
    id tmp;

    p = p_msg(self, sym_allocAsDBString);
    tmp = newString(p);
    free(p);
    return tmp;
}

DCLOBJFNC( asCSVString )
{
    return msg(self, sym_asDBString);
}


DCLOBJFNC( asNumber )
{
    if( self == True )
	return newInteger(1);
    return newInteger(0);
}


/****************
 * Special Method, gibt ein Object als String fuer die datenbank
 * zurueck, dabei wird ein Pointer zurueckgegeben, dessen Speicher
 * bereich der aufrufer dann verwalten (e.g. freigeben) muss.
 */
DCLOBJFNC_p( allocAsDBString )
{
    char *s, *p;

    if( self == True )
	return xstrdup("TRUE");
    if( self == False)
	return xstrdup("FALSE");
    if( self == Null )
	return xstrdup("NULL");
    p = symName(self->isa->name);
    s = xmalloc( 19 + strlen(p) + 2 );
    strcpy(s, "'Instance of Class ");
    strcat(s, p );
    strcat(s, "'" );
    return s;
}


DCLOBJFNC( changed )
{
    return self;
}



DCLOBJFNC( notification )  /* only as stub */
{
  #if 0
    DCL_arg( id, from );    /* nil = unkown */
    DCL_arg( symbol_t, reason );
  #endif
    return self;
}


DCLOBJFNC( notify )
{
    DCL_arg(id, from);
    DCL_arg(symbol_t, reason);
    return msg2( self, sym_notification, from, reason );
}




/****************
 * Returns the number of Objects in the System
 * Returns: An object of class Integer
 */

DCLFOBJFNC( objects )
{
    isa_t rover;
    long count = 0;

    for( rover = Object->isa; rover ; rover = rover->nameLink )
	count += rover->icount + 1; /* 1: a class is an object too */
    return newInteger(count);
}


/****************
 * Returns the number of Objects in this class
 * Returns: An object of class Integer
 */

DCLOBJFNC( objects )
{
    return newInteger(self->isa->icount);
}



DCLOBJFNC(isTrue)
{
    return self == True? True : False;
}










#ifdef DOCUMENTATION
@Summary WamSUC_Object	SetUp Class Object
 #include <wk/wam.h>

 void WamSUC_Object(void);
@Description
 Um mit jedem Compiler arbeiten zu koennen (d.h. auch wenn keine
 Constructors moeglich sind), ist die Verbindung der Classes
 untereinander einzeln zu programmieren. Zu diesem Zweck hat jede
 Class eine Funktion mit Namen "WamSUC_xxxxx()", wobei xxxx der
 Name der Class ist, die (in der richtigen Reihenfolge am Anfang
 der Anwendung genau einmal aufzurufen ist. Erst nach dieser
 Initialisierung duerfen weitere Threads gestartet werden!
  Dies hier ist die wichtigste, da sie die Wurzel der
 Class-Hierachie definiert und ist deswegen immer als erste aufzurufen.
 Nach der letzen SetupClass Fubnktion ist noch die Funktion
 WamSUCFinish() aufzurufen.
@See Also
 WamSUCFinish
#endif /*DOCUMENTATION*/

void WamSUC_Object()
{
    id self = Object;

    if( getenv("TRACESUBCLASSING") )
	trace_subclassing=1;
    CREATECLASS( "Object" );
    self->isa->super  = NULL ; /* Gibt es nur hier, wird bei anderen Classes*/
			       /* durch WamSubclassClass() gesetzt */
    self->isa->nameLink = NULL; /* Noch keine weiteren Classes definiert */
    self->isa->privateOff = 0;	/* 0 for the root class */
    self->isa->attic.list = NULL;
    self->isa->attic.used = 0;
    self->isa->attic.limit = ATTIC_LIMIT;

    /* setup factory methods */
    DCLFMTHD( new );

    /* setup instance methods */
    DCLMTHD( free );
    DCLMTHD( doesNotRecognize );
    DCLMTHD( errorNoSuperClass );
    DCLMTHD( error );
    DCLMTHD( terminated );
    DCLMTHD( shutdown );
    DCLMTHD( name );
    DCLMTHD( allocAsDBString );
    DCLMTHD( class );
    DCLMTHD( isMemberOf );
    DCLMTHD( isKindOf );
    DCLMTHD( copy );
    DCLMTHD( size );
    DCLMTHD( show );
    DCLMTHD( asString );
    DCLMTHD( asDBString );
    DCLMTHD( asCSVString );
    DCLMTHD( asNumber );
    DCLMTHD( bounds );
    DCLMTHD( display );
    DCLMTHD( debug );
    DCLMTHD( subclassResponsibility );
    DCLMTHD( changed );
    DCLMTHD( notification );
    DCLMTHD( notify );
    DCLFMTHD( objects );
    DCLMTHD( objects );
    DCLMTHD( isTrue );

}

/**** end of file ****/
