/* [wam/messager.c wk 22.01.93] Dispatch messages
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Dieses Modul verteilt Messages und implementiert
 * die SelektorTable
 ******************************************************
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: messager.c,v 1.8 1996/01/25 20:17:49 wernerk Exp $")
#include <stdio.h>
#include <stdlib.h>

#include <wk/wam.h>
#include "wammain.h"

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define MAX_METHODS 1000
#define MAX_ASENDS  3

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

typedef struct {
	symbol_t sel;	/* selector or 0 if unused slot */
	objfnc_t fnc;	/* function implementing this method */
	isa_t	 isa;	/* class implementing this method */
	byte	fMthd;	/* isfactoryMethod */
    } methodTbl_t;

static methodTbl_t *methodTbl;
static int methodTblUsed;    /* current size of method table */
static int methodTblSize;    /* allocated size of method table */

static int traceFlag;
static int localTableFlag;
static int recursionLevel;


/****************
 * Structure of method tables linked to every class
 */

typedef struct {
	symbol_t sel;	/* selector or 0 for the postlast entry*/
	objfnc_t fnc;	/* function implementing this method */
    } localTbl_t;

typedef struct {
	symbol_t sel;
	id	 recv;
	id	 args;
    } queue_t;


    /* TODO: Protect Queue with semaphores */
static queue_t aSendQueue[MAX_ASENDS];
static volatile aSendSize; /* current size of above queue */

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static void InvalidReceiver( id receiver, symbol_t selector, int w );
static void InvalidIsa( isa_t isa, symbol_t selector, int w );

static id Send( id, symbol_t, va_list);
static id SendSuper( id, id, symbol_t, va_list);
static symbol_t CreateMethod( isa_t isa, const char *name,
			      objfnc_t fnc, int fMthd );
static void RehashMethods(int);
static int WaitEvent(void);
static void Cleanup( void *dummy );

/**************************************************
 *************	Local Functions  ******************
 **************************************************/
#define CHECKReceiver(a,s,w) do { \
    if( (unsigned)(a) & 0x80000000 ) \
	InvalidReceiver((a),(s),(w));	      \
    } while(0)

#define CHECKIsa(a,s,w) do { \
    if( !a || (unsigned)(a) & 0x80000000 ) \
	InvalidIsa((a),(s),(w));	 \
    } while(0)


static void
InvalidReceiver( id receiver, symbol_t selector, int where )
{
    if( traceFlag ) /* hier drfen wir symName nutzen (fnc never returns)*/
	printf( "MsgTrc:%*s %s %c> INVALID RECEIVER %p\n",
		((recursionLevel-1)&0x1f) * 2, "", symName(selector),
						   where?'=':'-',
						   receiver);
    if( where )
	Fatal("During sending to super:");
    if( (unsigned)receiver == 0xffffffff )
	Bug("Message %s to a freed receiver", symName(selector));
    if( (unsigned)receiver == 0xeeeeeeee )
	Bug("Message %s to an uninitialized receiver", symName(selector));
    if( (unsigned)receiver == 0xdddddddd )
	Bug("Message %s to a receiver in the attic", symName(selector));
    Bug("Message %s to an invalid receiver with address %p",
					    symName(selector), receiver);
}


static void
InvalidIsa( isa_t isa, symbol_t selector, int where )
{
    if( traceFlag ) /* hier drfen wir symName nutzen (fnc never returns)*/
	printf( "MsgTrc:%*s %s %c> INVALID ISA %p (%s)\n",
		((recursionLevel-1)&0x1f) * 2, "", symName(selector),
						   where?'=':'-',
						   isa, isa? isa->eye:"");
    if( where )
	Fatal("During sending to super:");
    if( !isa )
	Bug("Message %s to a nil-isa", symName(selector));
    if( (unsigned)isa == 0xffffffff )
	Bug("Message %s to a freed isa", symName(selector));
    if( (unsigned)isa == 0xeeeeeeee )
	Bug("Message %s to an uninitialized isa", symName(selector));
    if( (unsigned)isa == 0xdddddddd )
	Bug("Message %s to an isa in the attic", symName(selector));
    Bug("Message %s to an invalid isa with address %p",
					    symName(selector), isa);
}


static id
Send( id receiver, symbol_t selector, va_list arg_ptr )
{
    isa_t isa;
    int i, fMthd;
    char symbuf[ 40 ]; /* should be 128 */
    localTbl_t *lt;

    CHECKReceiver(receiver,selector,0);
    isa = receiver->isa;
    CHECKIsa(isa,selector,0);
    fMthd = isa->selfLink == receiver ? 1 : 0; /*we need the discrete values*/
    if( traceFlag )
	printf(  fMthd ? "MsgTrc:%*s %s -> %sClass : " :
			 "MsgTrc:%*s %s -> %s : ",
	     ((recursionLevel-1)&0x1f) * 2, "",
	     WamQuerySymbolName( selector, symbuf, DIM(symbuf) ), isa->eye);

    if( localTableFlag )
	if( lt = fMthd? isa->fMthTbl : isa->iMthTbl ) {
	    /* zugriff ueber private ?MthdTbl versuchen */
	    for( ; lt->sel ; lt++ )
		if( lt->sel == selector ) {
		    if( traceFlag )
			printf( "found in local method table\n" );
		    return lt->fnc( receiver, arg_ptr );
		}
	}

    /* ueber die privaten Method Tables konnte die Methode nicht */
    /* gefunden werden, also in der globalen nachsehen */
    do {
	for(i=0; i < methodTblUsed; i++ )
	    if( methodTbl[i].sel == selector && methodTbl[i].isa == isa &&
		methodTbl[i].fMthd == fMthd			     ) {
		if( traceFlag )
		    printf( "%s will perform it\n", isa->eye );
		return methodTbl[i].fnc( receiver, arg_ptr );
	    }

	isa = isa->super;
    } while( isa );

    if( fMthd ) /* last attempt is to find an instance method in Object */
	for(i=0; i < methodTblUsed; i++ )
	    if( methodTbl[i].sel == selector &&
		methodTbl[i].isa == Object->isa &&
		!methodTbl[i].fMthd			     ) {
		if( traceFlag )
		    printf( "instance method of Object will perform it\n" );
		return methodTbl[i].fnc( receiver, arg_ptr );
	    }

    if( traceFlag )
	puts("method not recognized");
    if( selector == sel(doesNotRecognize) )
	Bug( "method 'doesNotRecognize' not recognized");
    return WamSendMsg( receiver, sel(doesNotRecognize), selector );
}


static id
SendSuper( id myFact, id receiver, symbol_t selector, va_list arg_ptr )
{
    isa_t isa;
    int i, fMthd;
    char symbuf[ 40 ]; /* should be 128 */
    localTbl_t *lt;

    CHECKReceiver(myFact,selector,1);
    isa = myFact->isa;	  /* factory of originator of msg */
    CHECKIsa(isa,selector,1);

    fMthd = receiver->isa->selfLink == receiver ? 1 : 0;
    if( traceFlag )
	printf(  fMthd ? "MsgTrc:%*s %s => %sClass : " : "MsgTrc:%*s %s => %s : ",
	     ((recursionLevel-1)&0x1f) * 2, "",
	     WamQuerySymbolName( selector, symbuf, DIM(symbuf) ), isa->eye);
    isa = isa->super;
    if( !isa )
	return Send( receiver, sel(errorNoSuperClass), arg_ptr );

    if( localTableFlag )
	if( lt = fMthd? isa->fMthTbl : isa->iMthTbl ) {
	    /* zugriff ueber private ?MthdTbl versuchen */
	    for( ; lt->sel ; lt++ )
		if( lt->sel == selector ) {
		    if( traceFlag )
			printf( "found in local method table\n" );
		    return lt->fnc( receiver, arg_ptr );
		}
	}

    do {
	for(i=0; i < methodTblUsed; i++ )
	    if( methodTbl[i].sel == selector && methodTbl[i].isa == isa &&
		methodTbl[i].fMthd == fMthd			     ) {
		if( traceFlag )
		    printf( "%s will perform it\n", isa->eye );
		return methodTbl[i].fnc( receiver, arg_ptr );
	    }

	isa = isa->super;
    } while( isa );

    if( traceFlag )
	puts("method not recognized by super class");
    if( selector == sel(doesNotRecognize) )
	Bug( "method 'doesNotRecognize' not recognized by super class");
    return WamSendMsgSuper( myFact, receiver, sel(doesNotRecognize),selector);
}


static symbol_t
CreateMethod( isa_t isa, const char *name, objfnc_t fnc, int fMthd )
{
    int i;
    symbol_t selector;

    selector = WamCreateSymbol(name);
    xassert( isa );
    xassert( selector );
    xassert( fnc );
    /* Testen ob Method bereits vorhanden */
    for(i=0; i < methodTblUsed; i++ ) {
	xassert( methodTbl[i].sel );
	xassert( methodTbl[i].fnc );
	xassert( methodTbl[i].isa );
	if( methodTbl[i].sel == selector && methodTbl[i].isa == isa &&
	    methodTbl[i].fMthd == fMthd   )
	    Bug("Method already defined in this class");
    }
    if( methodTblUsed >= methodTblSize )
	Bug("Method table is full");
    methodTbl[methodTblUsed].sel = selector;
    methodTbl[methodTblUsed].fnc = fnc;
    methodTbl[methodTblUsed].isa = isa;
    methodTbl[methodTblUsed].fMthd = fMthd;
    methodTblUsed++;

    return selector;
}



/****************
 * (das eigentliche Hashen ist noch nicht implementiert)
 * Das Verfahren setzt vorraus, das die GlobaleTabelle weiterhin
 * beibehalten wird.
 */

static void
RehashMethods( int fMthd )
{
    isa_t rover, isa;
    localTbl_t *lt;
    int n,i, j;
    symbol_t sel;

    /* loop over all classes */
    for( rover = Object->isa; rover ; rover = rover->nameLink ) {
	/* eventuelle alte Tabelle freigeben */
	if( fMthd )
	   FREE(rover->fMthTbl);
	else
	   FREE(rover->iMthTbl);

	/* count number of recognized methods.
	 * Hier werden noch zu viele gezaehlt, soll im Moment aber egal
	 * sein. Ansonsten muessste noch eine Tabelle gefuehrt werden
	 * mit den bekannten Methodenamen  ... but who cares about
	 * memory
	 * Instance Methode von Objekt werden nicht in dies private Tabelle
	 * mit aufgenommen (kommt wahrscheinlich selten vor), bei
	 * factory Table.
	 */
	for(isa=rover, n=0; isa; isa = isa->super )
	    for(i=0; i < methodTblUsed; i++ )
		if( methodTbl[i].sel && methodTbl[i].isa == isa &&
		    methodTbl[i].fMthd == fMthd )
			n++;
	/* jetzt die Tabelle allokieren + 1 ende entry */
	lt = xcalloc( n+1, sizeof *lt );

	/* und alle Methoden eintragen */
	for(isa=rover, n=0; isa; isa = isa->super )
	    for(i=0; i < methodTblUsed; i++ )
		if( (sel = methodTbl[i].sel) && methodTbl[i].isa == isa &&
					       methodTbl[i].fMthd == fMthd ) {
		    for(j=0; j < n; j++ )
			if( lt[j].sel == sel )
			    break;
		    if(j==n) { /* okay, ist noch nicht drin (=not subclassed)*/
			lt[n].sel = sel;
			lt[n].fnc = methodTbl[i].fnc;
			n++;
		    }
		}

	/* und Tabelle setzen - fertig */
	if( fMthd )
	    rover->fMthTbl = lt;
	else
	    rover->iMthTbl = lt;
    }
}


static void
Cleanup( void *dummy )
{
    isa_t rover;

    /* loop over all classes */
    if( Object ) {
	for( rover = Object->isa; rover ; rover = rover->nameLink ) {
	    FREE(rover->fMthTbl);
	    FREE(rover->iMthTbl);
	}
	localTableFlag = 0;
    }
}

/**************************************************
 *************	Global Functions  *****************
 **************************************************/

void
WamInitializeMessager()
{
    if( methodTbl )
	return; /* bereits initialisiert */

    methodTbl = xcalloc( MAX_METHODS, sizeof *methodTbl );
    methodTblSize = MAX_METHODS;
    methodTblUsed = 0;
    traceFlag = 0;
    localTableFlag = 1;
    AddCleanUp( Cleanup, NULL );
  #if 0
    Info("Messager addr=%p", Send );
  #endif
}



symbol_t
WamCreateFMethod( isa_t isa, const char *name, objfnc_t fnc )
{
    return CreateMethod( isa, name, fnc, 1 );
}


symbol_t
WamCreateIMethod( isa_t isa, const char *name, objfnc_t fnc )
{
    return CreateMethod( isa, name, fnc, 0 );
}


/****************
 * private Method tables fuer alle Classes neu aufbauen
 */

void
WamRehashMethodTables(void)
{
    RehashMethods(1); /* factory */
    RehashMethods(0); /* instance */
}


#ifdef DOCUMENTATION
@Summary WamSendMsg...
 #include <wk/wam.h>

 id WamSendMsg( id receiver, symbol_t selector);
 id WamSendMsgSuper( id receiver, symbol_t selector);

 msg( receiver, selector );
 msgSuper( receiver, selector );
@Description
  Sendet eine Message an ein Objekt. self ist der Sender,
  receiver das Objekt, an dsa die Message gesendet wird und
  selector der Name der Message.
  Diese Methode soll nicht direkt benutzt werden, sondern das
  Macro msg(), welches mehr dem Smalltalk-Syntax entpricht
@Return Value
 id
#endif /*DOCUMENTATION*/

id
WamSendMsg( id receiver, symbol_t selector, ...)
{
    va_list arg_ptr;
    id ret;
    va_start( arg_ptr, selector ) ;

    recursionLevel++;
    if( receiver )
	ret = Send(  receiver, selector, arg_ptr );
    else
	ret = WamSendMsgToNil( selector, arg_ptr, traceFlag, recursionLevel );
    recursionLevel--;
    return ret;
}

id
WamVSendMsg( id receiver, symbol_t selector, va_list arg_ptr )
{
    id ret;

    recursionLevel++;
    if( receiver )
	ret = Send( receiver, selector, arg_ptr );
    else
	ret = WamSendMsgToNil( selector, arg_ptr, traceFlag, recursionLevel );
    recursionLevel--;
    return ret;
}


id
WamSendMsgSuper( id myFact, id receiver, symbol_t selector, ...)
{
    va_list arg_ptr;
    id ret;
    va_start( arg_ptr, selector ) ;
    xassert(receiver); /* an super von nil senden waere voellig unsinnig */
    recursionLevel++;
    ret = SendSuper( myFact, receiver, selector, arg_ptr );
    recursionLevel--;
    return ret;
}


/****************
 * Testen ob der Receiver den selector versteht.
 * returns: NULL := does not respond
 *	    or the address of the function.
 */
objfnc_t
WamRespondsTo( id receiver, symbol_t selector )
{
    isa_t isa;
    int i, fMthd;
    localTbl_t *lt;

    if( !receiver )
	return (objfnc_t)WamSendMsgToNil( selector, NULL, 0, 0 );

    CHECKReceiver(receiver,selector,0);
    isa = receiver->isa;
    CHECKIsa(isa,selector,0);
    fMthd = isa->selfLink == receiver ? 1 : 0; /*we need the discrete values*/
    if( localTableFlag )
	if( lt = fMthd? isa->fMthTbl : isa->iMthTbl ) {
	    for( ; lt->sel ; lt++ )
		if( lt->sel == selector )
		    return lt->fnc;
	}

    do {
	for(i=0; i < methodTblUsed; i++ )
	    if( methodTbl[i].sel == selector && methodTbl[i].isa == isa &&
		methodTbl[i].fMthd == fMthd			     ) {
		return methodTbl[i].fnc;
	    }

	isa = isa->super;
    } while( isa );

    if( fMthd ) /* last attempt is to find an instance method in Object */
	for(i=0; i < methodTblUsed; i++ )
	    if( methodTbl[i].sel == selector &&
		methodTbl[i].isa == Object->isa &&
		!methodTbl[i].fMthd			     ) {
		return methodTbl[i].fnc;
	    }
    return NULL;
}



#ifdef DOCUMENTATION
@Summary WamMessageLoop
 #include <wk/wam.h>

 void WamMessageLoop( int (*termfnc)(int, void *), void *parm )
@Description
 Dies ist die MessageLoop, sie laeft so lange bis sie entweder
 durch die GUI abgebrochen wird, oder die Abbruch durch die optionale
 funktion termfnc() veranlasst wird.
 termfnc soll True zurueckliefern, wenn die MessageLoop beendet werden
 soll. Als Argument erhaelt si zu Infozwecken den aktuellen Level
 der MessageLoop.
#endif /*DOCUMENTATION*/

void
WamMessageLoop( int (*termfnc)(int,void*), void *parm )
{
    static int level;
    int rc;

    level++;
    Info("Starting MessageLoop at level %d", level);
    while( !(rc = WaitEvent()) )
	if( rc = termfnc ? termfnc(level,parm) : 0 ) {
	    Info("MessageLoop at level %d terminated (rc=%d)", level, rc );
	    level--;
	    return;
	}
    Info("MessageLoop at level %d cancelled (rc=%d)", level, rc );
    level--;
    return;
}

static int
WaitEvent()
{
    int i;
    id recv, args;
    symbol_t sel;

    /* Process ASendQueue */
    while( aSendSize ) {
	sel = aSendQueue[0].sel;
	recv= aSendQueue[0].recv;
	args= aSendQueue[0].args;
	for(i=1; i < aSendSize ; i++ )
	    aSendQueue[i-1] = aSendQueue[i];
	aSendSize--;
	msg1(recv, sel, args);
    }
    return WamWaitForEvent(0);
}


/****************
 * Send a Message Asynchron, this can only be a Message with obne Arg.
 * Returns: 0 := okay
 *	    err := Error (Queue is full)
 */

int
WamASend( id receiver, symbol_t selector, id args)
{
    if( aSendSize >= MAX_ASENDS )
	return E_WAM_AQFULL;
    aSendQueue[aSendSize].sel  = selector;
    aSendQueue[aSendSize].recv = receiver;
    aSendQueue[aSendSize].args = args;
    aSendSize++;
    return 0;
}




void
WamSetMessagerOption( int option, int value )
{
    if( option == 1 ) {
	if( traceFlag != value )
	    puts( value? "\nMsgTrc: Trace started" :
			 "MsgTrc: Trace stopped\n" );
	traceFlag = value;
    }
    else if( option == 2 ) {
	if( localTableFlag != value )
	    puts( value? "\nMsgTrc: Using LOCAL method tables" :
			   "MsgTrc: Using GLOBAL method table\n" );
	localTableFlag = value;
    }
}

int
WamGetMessagerOption( int option )
{
    if( option == 1 )
	return traceFlag;
    else if( option == 2 )
	return localTableFlag;
    return -1;
}

/**************************************************
 *************	Test & Debug Suite ****************
 **************************************************/


/**** end of file ****/
