/* [wam/symbldct.c wk 27.02.93] Class Dictionary
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 * Standard Dictionary
 ******************************************************
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: dctnry.c,v 1.7 1997/02/12 13:09:23 wk Exp $")
#include <stdio.h>
#include <stdlib.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>
#include "wammain.h"

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define FIRSTCHUNK   10  /* beim ersten mal nich soviel allokieren */
#define CHUNK	     50  /* wahrscheinlich brauchen wir dann doch mehr*/

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

typedef struct {
	symbol_t key;
	id	 val;
    } entry_t;


DCLSHAREDPART(Dictionary)

BEGIN_DCLPRIVATEPART
    ushort    refCount; /* we have only shallow copies, so count them */
    ushort    autoFreeAll;
    entry_t *contents;
    size_t capacity; /* allocated size of contents */
    id	  autoFreeSet;	/* set of elements to be freed */
    size_t enSize;   /* used for enumeration */
    size_t enIdx;    /* used for enumeration */
END_DCLPRIVATEPART


/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

static id AtPut( id self, id var, symbol_t key, id val );


/**************************************************
 ******************  Methods  *********************
 **************************************************/

DCLFOBJFNC( new )
{
    id obj;
    id	var;

    obj = msgSuper( sym_new );
    SET_var(obj);
    var->refCount = 1;
    return obj;
}

DCLOBJFNC( free )
{
    symbol_t sym;
    size_t n;
    DCL_var();

    if( var->autoFreeAll )
	return msg(self, sym_deepFree );

    if( --var->refCount )
	return nil;

    if( var->autoFreeSet ) {
	if( msg(var->autoFreeSet, sym_enumOpen ) )
	    while( sym = s_msg(var->autoFreeSet, sym_enumGet) )
		for(n=0; n < var->capacity; n++ )
		    if( var->contents[n].key == sym )
			freeObj(var->contents[n].val);
    }
    free( var->contents );
    free( var->autoFreeSet );
    return msgSuper( sym_free );
}


/****************
 * Free the Array and send an free to every value
 * see also: setAutoFree
 */

DCLOBJFNC( deepFree )
{
    size_t n;
    DCL_var();

    if( --var->refCount )
	return nil;

    if( var->contents )
	for(n=0; n < var->capacity; n++ )
	    if( var->contents[n].key )
		freeObj(var->contents[n].val);
    free( var->contents );
    free( var->autoFreeSet );
    return msgSuper( sym_free );
}

/****************
 * Add VALUE under key.
 */

DCLOBJFNC( atPut )
{
    DCL_arg(symbol_t, key );
    DCL_arg(id, val );
    DCL_var();

    return AtPut( self, var, key, val );
}

/****************
 * Add a copy of VALUE under KEY.
 */

DCLOBJFNC( put )
{
    DCL_arg(symbol_t, key );
    DCL_arg(id, val );
    DCL_var();

    return AtPut( self, var, key, msg(val,sym_copy) );
}

/****************
 * Add a Key/Value pair and set value to autofree
 */

DCLOBJFNC( atPutAF )
{
    DCL_arg(symbol_t, key );
    DCL_arg(id, val );
    id ret;
    DCL_var();

    ret = AtPut( self, var, key, val );
    if( !var->autoFreeSet )
	var->autoFreeSet = newObj(SymbolSet);
    msg1(var->autoFreeSet, sym_addSymbol, key);
    return ret;
}


static id AtPut( id self, id var, symbol_t key, id val )
{
    size_t n;

    if( !var->contents )
	var->contents = xcalloc(var->capacity=FIRSTCHUNK,sizeof *var->contents);
    /* find an empty slot */
    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key == key ) { /* replace */
	    if( var->autoFreeSet )
		if( i_msg1( var->autoFreeSet, sym_contains, key ) )
		    freeObj(var->contents[n].val);
	    var->contents[n].val = val;
	    return self;
	}
	else
	    if( !var->contents[n].key )
		break;
    if( n == var->capacity ) { /* need some more space */
	var->contents = xrealloc(var->contents,
				 (var->capacity+CHUNK) * sizeof *var->contents);
	memset( var->contents + var->capacity,
			    0 , CHUNK * sizeof *var->contents );
	var->capacity += CHUNK;
    }
    /* store value */
    var->contents[n].key = key;
    var->contents[n].val = val;
    return self;
}



DCLOBJFNC( at )
{
    DCL_arg(symbol_t, key );
    size_t n;
    DCL_var();

    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key == key )
	    return var->contents[n].val;
    return nil;
}

/****************
 * get: same as #at, but return a copy
 */
DCLOBJFNC( get )
{
    DCL_arg(symbol_t, key );
    size_t n;
    DCL_var();

    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key == key )
	    return msg(var->contents[n].val, sym_copy);
    return nil;
}


DCLOBJFNC( removeAt )
{
    DCL_arg(symbol_t, key );
    size_t n;
    DCL_var();

    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key == key ) {
	    var->contents[n].key = 0;
	    if( var->autoFreeSet )
		if( i_msg1( var->autoFreeSet, sym_contains, key ) )
		    freeObj(var->contents[n].val);
	    break;
	}
    return self;
}



DCLOBJFNC_s( keyFromValue )
{
    DCL_arg(id, val );
    size_t n;
    DCL_var();

    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key )
	    if( var->contents[n].val == val )
		return var->contents[n].key;
    return 0;
}



/****************
 * Returns: StringArray mit den Keys als Strings
 * See Also: valuesAsDBStringArray
 */

DCLOBJFNC( keysAsStringArray )
{
    size_t n;
    id arr;
    DCL_var();

    arr = newObj(StringArray);
    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key )
	    msg1(arr, sym_add, symName(var->contents[n].key) );
    return arr;
}


/****************
 * Returns: StringArray mit den Values als DBStrings
 *	    dabei haben diese die gleichen indices wie die
 *	    von der Methode keysAsStingArray zurckgelieferten
 * See Also: keysAsStringArray
 */

DCLOBJFNC( valuesAsDBStringArray )
{
    size_t n;
    id arr;
    DCL_var();

    arr = newObj(StringArray);
    for(n=0; n < var->capacity; n++ )
	if( var->contents[n].key )
	    msg1(arr, sym_addAsDBString, var->contents[n].val );
    return arr;
}


/****************
 * Returns: Einen Array mit den Ids, dies sit ein normales C-Array
 *	    welches vom Caller mit free() wieder freigegeben werden muss.
 *    Note: Damit auch Nil Werte im Array dargestllet werden koennen
 *	    enthaelt das erste Element kein Object, sondern die Anzahl
 *	    der im Array enthaltenen Elemente (cast it to ulong)
 */

DCLOBJFNC_a( valuesAsCArray )
{
    size_t i;
    ulong n;
    id *arr;
    DCL_var();

    n = 0;
    for(i=0; i < var->capacity; i++ )
	if( var->contents[i].key )
	    n++;
    arr = xmalloc( (n+1) * sizeof *arr );
    arr[0] = (id)n;
    n = 1;
    for(i=0; i < var->capacity; i++ )
	if( var->contents[i].key )
	    arr[n++] = var->contents[i].val;
    return arr;
}




DCLOBJFNC_I( size )
{
    size_t n, cnt;
    DCL_var();

    for(n=cnt=0; n < var->capacity; n++ )
	if( var->contents[n].key )
	    cnt++;
    return cnt;
}


/****************
 * An alle hiermit registrierten Entries  wird ein freeObj
 * gesendet, sobald das Dictionary selbst ein free erhaelt
 */

DCLOBJFNC( setAutoFree )
{
    DCL_arg( symbol_t, key );
    DCL_var();

    if( !var->autoFreeSet )
	var->autoFreeSet = newObj(SymbolSet);
    msg1(var->autoFreeSet, sym_addSymbol, key);

    return self;
}

/****************
 * Statt eines free wird immer ein deepFree gemacht.
 */

DCLOBJFNC( autoFreeAll )
{
    DCL_arg( symbol_t, key );
    DCL_var();

    var->autoFreeAll = 1;
    return self;
}


/****************
 * Seq. Zugriff: aber einmal gleichzeitig
 * (fuer Anderes ist die Sequence Methode zu benutzen)
 * Returns: self or nil; bei nil braucht close nicht mehr
 * benutzt zu werden, da es ein leeres Array anzeigt.
 */

DCLOBJFNC( enumOpen )
{
    DCL_var();

    if( var->enSize ) {
	msg1(self,sym_error,"nested enumBegin in Dictionary");
	return nil;
    }

    var->enSize  = var->capacity +1;
    var->enIdx	 = 0;
    return self;
}


/****************
 * Returns Keys
 */

DCLOBJFNC( enumGet )
{
    DCL_var();

    if( !var->enSize ) {
	msg1(self,sym_error,"enumGet without enumBegin in Dictionary");
	return NULL;
    }

    for( ; var->enIdx < var->enSize-1; var->enIdx++ )
	if( var->contents[var->enIdx].key )
	    return var->contents[var->enIdx++].key;

    var->enSize = 0;
    return 0;
}


DCLOBJFNC( enumClose )
{
    DCL_var();
    var->enSize = 0;
    return nil;
}


DCLOBJFNC( copy )
{
    DCL_var();

    var->refCount++;
    return self;
}


void WamSUC_Dictionary()
{
    id self = Dictionary;
    CREATECLASS("Dictionary");
    WamSubclassClass( sym_Object, self );
    WamSetAtticLimit( self, 100 );

    DCLFMTHD( new );

    DCLMTHD( free );
    DCLMTHD( deepFree );
    DCLMTHD( atPut );
    DCLMTHD( put );
    DCLMTHD( atPutAF );
    DCLMTHD( at );
    DCLMTHD( get );
    DCLMTHD( removeAt );
    DCLMTHD( keyFromValue );
    DCLMTHD( keysAsStringArray );
    DCLMTHD( valuesAsDBStringArray );
    DCLMTHD( valuesAsCArray );
    DCLMTHD( size );
    DCLMTHD( setAutoFree );
    DCLMTHD( autoFreeAll );
    DCLMTHD( enumOpen );
    DCLMTHD( enumGet  );
    DCLMTHD( enumClose);
    DCLMTHD( copy );
}


/**** end of file ****/
