/* [wam/dbodbc.c wk 08.12.95] Access primitives for ODBC
 *	Copyright (c) 1995 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 *  ODBC tested with Watcom.
 ******************************************************
 * BUGS: I use int instead of RETCODE, why do they define
 *	 everthing in their specs - it would be better if they go and fix
 *	 the macro handling of their compiler - see the numerous comments
 *	 about macros (e.g. ATOI_...) in several of my sources.
 */

#include <wk/tailor.h>
RCSID("$Id: dbodbc.c,v 1.5 1997/03/25 14:46:22 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <wk/string.h>

#include <wk/wam.h>
#include "version.h"
#include "wammain.h"
#include "wamdb.h"
#ifdef EMX
  #define _System
#endif
#include <os2odbc.h>

#define MAKE_DLL 1

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define DEFAULT_LOADLIMIT 500

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/
typedef struct s_coldes {
    struct s_coldes *next; /* so we can hold them in a linked list */
	short	edt;	/* external data type */
	long	edl;	/* external data length */
	short	pdt;	/* program data type */
	int	idt;	/* internal datatype */
	size_t	idl;	/* internal datalength */
	long	fdl;	/* fetched data length */
	char	d[256]; /* data buffer (fixme: should be malloced)*/
} t_coldes;

typedef struct s_crsblk {
    struct s_crsblk *next;
    HSTMT    stmt;
    t_coldes *firstcol;
    short    nsi;
    int      loadLimit;
} t_crsblk;


static char *logonString; /* NULL or alloced logon infos */
			  /* fixme: we should burn the password
			   * this is also used as a flag to indicate that
			   * we are connected to the database */
static int debug = 0;
static int isInitialized = 0;
static HENV environment;
static HDBC connection; /* we only support one connection (handle) */
	  /* so there is no need for a table or something like this*/

static char lastSQLState[SQLSTATE_SIZE+1];
static char lastSQLErrorMsg[ 250 ];
static volatile int abort_fetch;

static t_coldes *coldes_pantry; /* linked list of unused coldes items */
static t_crsblk *crsblk_pantry; /* linked list of unused cursor blocks */
static t_crsblk *crsblk_active; /* linked list of active cursor blocks */

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static void Dbg( int sql_rc, const char *s, ... );
static void Cleanup(void);
static int Initialize(void);
static int MapRC( int sql_rc, HSTMT stmt );
static int MapRCPrint( int sql_rc, HSTMT stmt, const char *fncname );

static void ReleaseCursor( t_crsblk *cb );
static void ReleaseAllCursors(void);

static int ScrollFnc( id tbl, void *arg, int reserved );
static int FetchLoop( id tbl, HSTMT stmt, t_coldes *firstcol,
		      short nsi, int loadLimit );

/**************************************************
 *************	Functions  ************************
 **************************************************/

#define ATOI_1( p )   ( isdigit(*(p)) ? *(p) - '0' : 0 )
#define ATOI_2( p )   (( ATOI_1(p) * 10) + ATOI_1( (p)+1 ))
#define ATOI_3( p )   (( ATOI_2(p) * 10) + ATOI_1( (p)+2 ))
#define ATOI_4( p )   (( ATOI_2(p) * 100) + ATOI_2( (p)+2 ))

static void
Dbg( int sql_rc, const char *s, ... )
{
    va_list arg_ptr;

    if( !debug )
	return;
    fputs("ODBC dbg: ", stdout);
    va_start( arg_ptr, s ) ;
    vfprintf(stdout,s,arg_ptr) ;
    va_end(arg_ptr);
    if( sql_rc )
	fprintf(stdout, ": rc=%d%s", sql_rc,
	       sql_rc == SQL_SUCCESS	       ? " (success)" :
	       sql_rc == SQL_SUCCESS_WITH_INFO ? " (success with info)" :
	       sql_rc == SQL_STILL_EXECUTING   ? " (still executing)" :
	       sql_rc == SQL_NEED_DATA	       ? " (need data)" :
	       sql_rc == SQL_NO_DATA_FOUND     ? " (no data found)" :
	       sql_rc == SQL_ERROR	       ? " (error)" :
	       sql_rc == SQL_INVALID_HANDLE    ? " (invalid handle)" : "" );
    else
	fputs(": okay", stdout);
    putc('\n', stdout);
}


static void
Cleanup()
{
    if( !isInitialized )
	return;
    if( connection ) {
	WamSQLDisconnect(0);
	connection = NULL;
    }
    if( environment ) {
	SQLFreeEnv( &environment );
	environment = NULL;
    }
    isInitialized = 0;
}


static int
Initialize()
{
    int rc;

    if( isInitialized )
	return 0;
    if( getenv("TRACEWAMDBDRIVER" ) )
	debug=1;
    RegisterCleanup( Cleanup );
    rc = SQLAllocEnv( &environment );
    Dbg(rc,"SQLAllocEnv");
    if( rc != SQL_SUCCESS ) { /* fixme: handle SQL_SUCCESS_WITH_INFO */
	return MapRCPrint(rc,NULL,"AllocEnv" );
    }
    isInitialized++;
    return 0;
}


/****************
 * FIXME: we should store all errors records for later retrieval.
 */
static int
MapRC( int sql_rc, HSTMT stmt )
{
    int rc;

    if( sql_rc == SQL_ERROR ) {
	rc = SQLError( environment, connection, stmt, lastSQLState, NULL,
		       lastSQLErrorMsg, DIM(lastSQLErrorMsg), NULL );
	if( rc != SQL_SUCCESS )
	    Error(0,"ODBC error: "
		    "SQLError failed for sql_rc=%d: sql_rc=%d", sql_rc, rc );
	else if( !strcmp(lastSQLState,"23000") )
	    return WAMDB_E_DUP_KEY;
    }

    switch( sql_rc ) {
      case SQL_SUCCESS		:
      case SQL_SUCCESS_WITH_INFO: rc = 0; break;
      case SQL_STILL_EXECUTING	: rc = WAMDB_E_EXECUTING; break;
      case SQL_NEED_DATA	: rc = WAMDB_E_NEED_DATA; break;
      case SQL_NO_DATA_FOUND	: rc = WAMDB_E_NO_DATA; break;
      case SQL_ERROR		: rc = WAMDB_E_DB_ERROR; break;
      case SQL_INVALID_HANDLE	: rc = WAMDB_E_DB_HANDLE; break;
      default:			  rc = WAMDB_E_UNKNOWN_RC; break;
    }
    return rc;
}

int
MapRCPrint( int sql_rc, HSTMT stmt, const char *fncname )
{
    int rc = MapRC(sql_rc, stmt);

    Error(0,"ODBC error: %s failed: sql_rc=%d%s", fncname, sql_rc,
	       sql_rc == SQL_SUCCESS	       ? " (success)" :
	       sql_rc == SQL_SUCCESS_WITH_INFO ? " (success with info)" :
	       sql_rc == SQL_STILL_EXECUTING   ? " (still executing)" :
	       sql_rc == SQL_NEED_DATA	       ? " (need data)" :
	       sql_rc == SQL_NO_DATA_FOUND     ? " (no data found)" :
	       sql_rc == SQL_ERROR	       ? " (error)" :
	       sql_rc == SQL_INVALID_HANDLE    ? " (invalid handle)" : "" );
    return rc;
}


/****************
 * Connect to the Database.
 * we map connection one to one to our handles.
 * Currently there is only one connection supported.
 * Returns: 0 = Okay or Errorcode from DB
 */

int _wamapi
WamSQLConnect( int *hd,     /* Return: handle (in SQL jargon: cursor) */
	       const char *dbName,  /* Name of database to connect */
	       const char *user,    /* name of user */
	       const char *pass )   /* and finaly the password */
{
    char *p, *p1, *p2, *p3;
    int rc, rc2;
    int tries=0;

    if( rc = Initialize() )
	return rc;
    if( connection ) { /* already connected */
	if( rc = WamSQLDisconnect(0) )
	    return rc;
    }

    rc = SQLAllocConnect( environment, &connection );
    Dbg(rc,"SQLAllocConnect");
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	return MapRCPrint(rc,NULL, "AllocConnect" );

    p = xmalloc( strlen(dbName) + 1 + strlen(user) + 1 + strlen(pass) + 1 );
    stpcpy(stpcpy(stpcpy(stpcpy(stpcpy( p, dbName ),"/"),user),"/"), pass );

  once_more:
    p1 = xstrdup(dbName); p2 = xstrdup(user); p3 = xstrdup(pass);
    rc = SQLConnect( connection, p1, SQL_NTS, p2, SQL_NTS, p3, SQL_NTS );
    Dbg(rc,"SQLConnect(%s,%s,%s)", p1, p2, p3 );
    free(p1); free(p2); free(p3);
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO ) {
	rc = MapRCPrint(rc,NULL, "Connect");
	if( tries < 5 && rc == WAMDB_E_DB_ERROR
	    && !strcmp(lastSQLState, "08001")
	    && strstr(lastSQLErrorMsg, "required to start engine") ) {
	    if( !tries ) {
		/* try to start the Watcom engine */
		Info("No database engine - I will try to start it");
		/* use START because a detached watcom engine seems to
		 * burn cpu-cycles */
		p1 = malloc( 60 + strlen(dbName) );
		strcpy( stpcpy(p1,"start/c/win/min dbstart <con >con "),dbName);
		system(p1);
		free(p1);
	    }
	    else
		Info("Database not yet up - trying once more (%d)", tries);
	    Sleep(1000); /* wait a second */
	    tries++;
	    goto once_more;
	}
	rc2 = SQLFreeConnect( connection );
	Dbg(rc2,"SQLFreeConnect");
	if( rc2 != SQL_SUCCESS )
	    MapRCPrint(rc2,NULL, "FreeConnect" );
	else
	    connection = NULL;
	free(p);
	return rc;
    }
    rc = MapRC( rc, NULL );
    logonString = p;
    *hd = 0;
    if( !rc ) {
	rc = SQLSetConnectOption(connection, SQL_AUTOCOMMIT, 0);
	Dbg(rc,"SQLSetConnectOption(AutoCommit,off)");
	if( rc != SQL_SUCCESS ) {
	    rc = SQLFreeConnect( connection );
	    Dbg(rc,"SQLFreeConnect");
	    if( rc != SQL_SUCCESS )
		return MapRCPrint(rc,NULL, "FreeConnect" );
	    connection = 0;
	    return rc;
	}
	rc = 0;
    }
    return rc;
}


/****************
 * Disconnect from a database. This closes all cursors and releases all
 * recources for this database.
 * Returns: 0 for okay or errocode.
 */

int _wamapi
WamSQLDisconnect( int hd )
{
    int rc;

    if( hd || !connection )
	return WAMDB_E_INV_HANDLE;
    if( logonString ) {
	ReleaseAllCursors();
	rc = SQLDisconnect( connection );
	Dbg(rc,"SQLDisconnect");
	if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	    return MapRCPrint(rc,NULL,"Disconnect");
	FREE(logonString);
    }
    rc = SQLFreeConnect( connection );
    Dbg(rc,"SQLFreeConnect");
    if( rc != SQL_SUCCESS )
	return MapRCPrint(rc,NULL, "FreeConnect" );
    connection = 0;
    return 0;
}


/****************
 * Einen Cursor Block wieder freigeben
 */

static void
ReleaseCursor( t_crsblk *cb )
{
    t_crsblk *a, *a2;
    t_coldes *d, *d2;

    /* remove from active list */
    for(a2=NULL, a = crsblk_active; a; a2=a, a = a->next )
	if( a == cb )
	    break;
    if( !a )
	Bug("ODBC: releasing an inactive crsblk");
    if( a2 )
	a2->next = a->next;
    else
	crsblk_active = a->next;

    /* release this cursor block */
    if( cb->stmt ) {
	int rc;
	rc = SQLFreeStmt( cb->stmt, SQL_DROP );
	Dbg(rc,"SQLFreeStmt");
    }
    for(d=cb->firstcol; d; d = d2 ) {
	d2 = d->next;
	d->next = coldes_pantry;
	coldes_pantry = d;
    }
    cb->next = crsblk_pantry;
    crsblk_pantry = cb;
}


static void
ReleaseAllCursors()
{
    while( crsblk_active )
	ReleaseCursor( crsblk_active );
}



int _wamapi
WamSQLCommit( int hd )
{
    int rc;

    if( hd || !connection )
	return WAMDB_E_INV_HANDLE;
    else if( !logonString )
	return WAMDB_E_NO_CONNECT;
    rc = SQLTransact( environment, connection, SQL_COMMIT );
    Dbg(rc,"SQLTransact(commit)");
    if( rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO )
	rc = MapRC(rc,NULL);
    else
	rc = MapRCPrint(rc,NULL, "Transact(Commit)" );
    return rc;
}


int _wamapi
WamSQLRollback( int hd )
{
    int rc;

    xassert(!hd);
    if( hd || !connection )
	return WAMDB_E_INV_HANDLE;
    else if( !logonString )
	return WAMDB_E_NO_CONNECT;
    rc = SQLTransact( environment, connection, SQL_ROLLBACK );
    Dbg(rc,"SQLTransact(rollback)");
    if( rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO )
	rc = MapRC(rc,NULL);
    else
	rc = MapRCPrint(rc,NULL, "Transact(Rollback)" );
    return rc;
}



/****************
 * Eine signal an einew "Operation" senden.
 * z.Z: 1 := Aktuelle Datenbank Operation unterbrechen.
 */
int _wamapi
WamSQLAttention( int hd, int nr )
{
    abort_fetch = 1;
    return 0;
}



/****************
 * Fuehrt ein beliebiges SQL Command aus
 * Bind variables werden z.Z. noch nicht unterstuetzt, knnen
 * aber spter untersttzt werden ber weiter Argumente.
 */

int _wamapi
WamSQLExecute( int hd, const char *str, ... )
{
    int rc=0;
    HSTMT stmt=0;
    char *sql = NULL;

    if( hd || !connection )
	return WAMDB_E_INV_HANDLE;
    else if( !logonString )
	return WAMDB_E_NO_CONNECT;

    abort_fetch = 0;
    rc = SQLAllocStmt( connection, &stmt );
    Dbg(rc,"SQLAllocStmt");
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	{ rc = MapRCPrint(rc,stmt,"AllocStmt"); goto leave; }

    if( !(sql = strdup(str)) )
	{ rc = WAMDB_E_NO_CORE; goto leave; }
    rc = SQLExecDirect( stmt, sql, SQL_NTS );
    Dbg(rc,"SQLExecDirect(%s)", sql);
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	{ rc = MapRCPrint(rc,stmt,"ExecDirect"); goto leave; }
    rc = 0;

  leave:
    if( stmt ) {
	int rc2 = SQLFreeStmt( stmt, SQL_DROP);
	Dbg(rc2,"SQLFreeStmt");
    }
    free( sql );
    return rc;
}


/****************
 * Fuehrt ein Select Statement aus, dies ist besonders, da es Daten
 * zurueckliefert.  ....
 * Returns: DB Error Code
 */

int _wamapi
WamSQLSelect( int hd,
	      const char *sqlStr, /* SQL String */
	      int loadLimit,
	      id *retTbl	  /* created Table */
	    )
{
    short    nsi;   /* # of select items */
    t_coldes *firstcol = NULL, *cp, *cp2;
    t_crsblk *crsblk = NULL;
    HSTMT stmt=0;
    char *sql = NULL;
    id	tbl = nil;	  /* created Table Object */
    int rc=0, i;
    symbol_t sym;

    if( hd || !connection )
	return WAMDB_E_INV_HANDLE;
    else if( !logonString )
	return WAMDB_E_NO_CONNECT;
    if( !(sql = strdup(sqlStr)) )
	return WAMDB_E_NO_CORE;

    if( !loadLimit )
	loadLimit = DEFAULT_LOADLIMIT;


    abort_fetch = 0;
    rc = SQLAllocStmt( connection, &stmt );
    Dbg(rc,"SQLAllocStmt");
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	{ rc = MapRCPrint(rc,stmt,"AllocStmt"); goto leave; }
    rc = SQLPrepare( stmt, sql, SQL_NTS );
    Dbg(rc,"SQLPrepare(%s)", sql );
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	{ rc = MapRCPrint(rc,stmt,"Prepare"); goto leave; }
    rc = SQLNumResultCols( stmt, &nsi );
    Dbg(rc,"SQLNumResultCols nsi=%d", (int)nsi);
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	{ rc = MapRCPrint(rc,stmt,"NumResultCols"); goto leave; }
    if( nsi < 0 ) {
	Error(0,"ODBC fatal: negative number of columns: nsi=%d", (long)nsi );
	rc = WAMDB_E_DB_BUG; goto leave;
    }

    if( !(tbl = WamDescribeTableColumns( nil, 0, -1, nsi )) ) {
	Error(0,"ODBC fatal: failed to create the table" );
	rc = WAMDB_E_WAM_BUG; goto leave;
    }

    cp = NULL;
    for(i=0; i < nsi; i++ ) {
	char name[128], *p;
	short namelen;
	long al;

	/* allocate an description item for the column */
	cp2 = cp ;
	if( coldes_pantry ) {
	    cp = coldes_pantry;
	    coldes_pantry = cp->next;
	}
	else if( !(cp = malloc( nsi * sizeof *cp )) )
	    { rc = WAMDB_E_NO_CORE; goto leave; }
	cp->next = NULL;
	if( cp2 )
	    cp2->next = cp;
	else
	    firstcol = cp;

	/* get and process column infos */
	rc = SQLColAttributes( stmt, i+1, SQL_COLUMN_NAME,
			       name, DIM(name), &namelen, NULL );
	Dbg(rc,"SQLColAttributes(%d) name=%s", i+1, name);
	if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	    { rc = MapRCPrint(rc,stmt,"DescribeCol(name)"); goto leave; }
	if( namelen >= DIM(name) )
	    Info("ODBC warning: a name was truncated to %d bytes", DIM(name)-1);
	rc = SQLColAttributes( stmt, i+1, SQL_COLUMN_TYPE, 0, 0, 0, &al );
	Dbg(rc,"SQLColAttributes(%d) type=%ld", i+1, al);
	if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	    { rc = MapRCPrint(rc,stmt,"DescribeCol(type)"); goto leave; }
	cp->edt = al;
	rc = SQLColAttributes( stmt, i+1, SQL_COLUMN_LENGTH, 0, 0, 0, &al );
	Dbg(rc,"SQLColAttributes(%d) length=%ld", i+1, al);
	if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	    { rc = MapRCPrint(rc,stmt,"DescribeCol(length)"); goto leave; }
	cp->edl = al;

	/* map SQL types to WAM types */
	cp->idl = 0;
	switch( cp->edt ) {
	  case SQL_CHAR:
	  case SQL_VARCHAR:
	  case SQL_LONGVARCHAR:
	    cp->pdt = SQL_C_CHAR;
	    cp->idt = WAM_DT_STRING;
	    cp->idl = cp->edl;
	    break;

	  case SQL_TINYINT:
	    cp->pdt = SQL_C_TINYINT;
	    cp->idt = WAM_DT_INTEGER;
	    break;
	  case SQL_SMALLINT:
	    cp->pdt = SQL_C_SHORT;
	    cp->idt = WAM_DT_INTEGER;
	    break;
	  case SQL_INTEGER:
	    cp->pdt = SQL_C_LONG;
	    cp->idt = WAM_DT_INTEGER;
	    break;

	  case SQL_NUMERIC:
	  case SQL_DECIMAL:
	  case SQL_FLOAT:
	  case SQL_REAL:
	  case SQL_DOUBLE:
	    cp->pdt = SQL_C_DOUBLE;
	    cp->idt = WAM_DT_FLOAT;
	    break;

	  case SQL_DATE:
	    cp->pdt = SQL_C_DATE;
	    cp->idt = WAM_DT_DATE;
	    break;
	  case SQL_TIME:
	    cp->pdt = SQL_C_TIME;
	    cp->idt = WAM_DT_TIME;
	    break;
	  case SQL_TIMESTAMP:
	    cp->pdt = SQL_C_TIMESTAMP;
	    cp->idt = WAM_DT_TSTAMP;
	    break;

	  case SQL_BIT:
	  case SQL_BIGINT:
	  case SQL_LONGVARBINARY:
	  case SQL_VARBINARY:
	  case SQL_BINARY:
	  default:
	    Info("ODBC warning: unsupported datatype %u", (unsigned)cp->edt );
	    rc = WAMDB_E_DT_NOT_SUP;
	    goto leave;
	}

	/* Der Name kann noch mit CorrelationName der Table versehen sein
	 * Dieser sollte unbedingt entfernt werden, auch wenn dies doppelte
	 * Namen ergeben koennte */
	sym = WamCreateSymbol( (p = strchr( name, '.' ))? (p+1) : name );

	WamDescribeTableColumns( tbl, sym, cp->idt, cp->idl );
	if( debug )
	    Info("ODBC: bind %d: e/p/idt=%d/%d/%d  e/idl=%ld/%lu n='%s'",
		i+1, cp->edt, cp->pdt, cp->idt,
		(long)cp->edl, (ulong)cp->idl, name );
	rc = SQLBindCol( stmt, i+1, cp->pdt, cp->d, DIM(cp->d)-1, &cp->fdl );
	Dbg(rc,"SQLBindCol(%d)", i+1);
	if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	    { rc = MapRCPrint(rc,stmt,"BindCol"); goto leave; }
    }

    if( loadLimit < 0 ) { /* scrolling was requested */
	if( crsblk_pantry ) {
	    crsblk = crsblk_pantry;
	    crsblk_pantry = crsblk->next;
	}
	else if( !(crsblk = malloc( sizeof *crsblk  )) )
	    { rc = WAMDB_E_NO_CORE; goto leave; }
	crsblk->next	 = crsblk_active;
	crsblk_active	 = crsblk;
	crsblk->stmt	 = stmt;
	crsblk->firstcol = firstcol;
	crsblk->nsi	 = nsi;
	crsblk->loadLimit= loadLimit;
	WamDescribeTableColumns( tbl, 0, -102, 0 ); /* finish */
	WamSetTableScrollFnc( tbl, ScrollFnc, crsblk );
    }
    else
	WamDescribeTableColumns( tbl, 0, -2, 0 ); /* finish */

    rc = SQLExecute( stmt );
    Dbg(rc,"SQLExecute");
    if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO )
	{ rc = MapRCPrint(rc,stmt,"Execute"); goto leave; }
    if( rc = FetchLoop(tbl, stmt, firstcol, nsi, loadLimit ) )
	goto leave;

    rc = 0;

  leave:
    if( !crsblk ) { /* we dont need the description any more */
	for(cp=firstcol; cp; cp = cp2 ) {
	    cp2 = cp->next;
	    cp->next = coldes_pantry;
	    coldes_pantry = cp;
	}
	if( stmt ) {
	    int rc2 = SQLFreeStmt( stmt, SQL_DROP );
	    Dbg(rc2,"SQLFreeStmt");
	}
    }
    free( sql );
    if( rc ) {
	if( crsblk )
	    ReleaseCursor(crsblk);
	WamDestroyTable(tbl);
    }
    else
	*retTbl = tbl;
    return rc;
}




/****************
 * Diese Funktion wird von Class Table benutzt um
 * weitere Rows zu holen. Nachdem keine Rows mehr benoetigt werden,
 * soll diese funktion mit TBL=nil aufgerufen werden um
 * die Resourcen freizugeben.
 * (reserved kann einmal benutzt werden um rueckwaerts scrollen
 *  zu implementieren)
 */

static int
ScrollFnc( id tbl, void *arg, int reserved )
{
    int rc;
    t_crsblk *c = arg;

    if( tbl ) {
	rc = FetchLoop(tbl, c->stmt, c->firstcol, c->nsi, c->loadLimit );
	if( rc )
	    ReleaseCursor(c);
    }
    else {
	ReleaseCursor(c);
	rc=0;
    }
    return rc;
}


/****************
 * Actually get the data.  This function retrieves up to LOADLIMIT
 * rows and appends them to the table TBL.
 * This is also used to scroll thru the result-set.
 */
static int
FetchLoop( id tbl, HSTMT stmt, t_coldes *firstcol, short nsi, int loadLimit )
{
    int rc=0, i;
    timeStamp_t ts;
    t_coldes *cp;
    int scroll=0, eof=0;

    if( loadLimit < 0 ) {
	loadLimit = -loadLimit;
	scroll=1;
    }

    while( loadLimit ) {
	if( abort_fetch ) {
	    rc = WAMDB_E_CANCELED;
	    abort_fetch = 0;
	    goto leave;
	}
	loadLimit--;
	rc = SQLFetch( stmt );
	Dbg(rc,"SQLFetch");
	if( rc != SQL_SUCCESS && rc != SQL_SUCCESS_WITH_INFO ) {
	    if( rc == SQL_NO_DATA_FOUND ) {
		rc = 0;
		eof++;
	    }
	    else
		rc = MapRCPrint(rc,stmt,"Fetch");
	    goto leave;
	}
	rc = 0;
	WamCreateTableRow( tbl, -1, NULL, 0 ); /* Begin a new row */

	/* loop over all columns */
	for(i=0,cp=firstcol; cp && i < nsi; i++, cp = cp->next ) {
	    if( cp->fdl == SQL_NULL_DATA )
		WamCreateTableRow( tbl, 0, NULL, 0 );
	    else if( cp->fdl < 0 ) {
		Error(0,"ODBC fatal: fetch code %ul for %d. column",
							     cp->fdl, i+1 );
		rc = WAMDB_E_DB_BUG; goto leave;
		goto leave;
	    }
	    else {
		long along;
		double adouble;

		switch( cp->idt ) { /* some types need conversion */
		  case WAM_DT_STRING:
		    WamCreateTableRow( tbl, 0, cp->d, cp->fdl );
		    break;

		  case WAM_DT_INTEGER: /* fixme: bus errors may occur ?*/
		    if( cp->pdt == SQL_C_SHORT )
			along = *(short*)cp->d;
		    else if( cp->pdt == SQL_C_TINYINT )
			along = *(signed char *)cp->d;
		    else if( cp->pdt == SQL_C_LONG )
			along = *(long*)cp->d;
		    else
			BUG();
		    WamCreateTableRow( tbl, 0, &along, sizeof along );
		    break;

		  case WAM_DT_FLOAT:
		    adouble = *(double *)cp->d;
		    WamCreateTableRow( tbl, 0, &adouble, sizeof adouble );
		    break;

		  case WAM_DT_DATE: {
			DATE_STRUCT *s = (DATE_STRUCT*)cp->d;
			/* fixme: should we use CheckDate()? */
			along = Date2JD(s->day, s->month, s->year );
			WamCreateTableRow( tbl, 0, &along, sizeof along );
		    }
		    break;
		  case WAM_DT_TIME: {
			TIME_STRUCT *s = (TIME_STRUCT*)cp->d;
			along = (3600L*s->hour + 60*s->minute
					       + s->second ) * 1000;
			WamCreateTableRow( tbl, 0, &along, sizeof along );
		    }
		    break;
		  case WAM_DT_TSTAMP: {
			TIMESTAMP_STRUCT *s = (TIMESTAMP_STRUCT*)cp->d;
			ts.d = Date2JD(s->day, s->month, s->year );
			ts.t = (3600L*s->hour + 60*s->minute
				+ s->second ) * 1000 + (s->fraction/1000000u);
			WamCreateTableRow( tbl, 0, &ts, sizeof ts );
		    }
		    break;

		  default: BUG();
		}
	    }
	}
	xassert( !cp && i == nsi );

	/* make new row permanent */
	WamCreateTableRow( tbl, -2, NULL, 0 );
    }

  leave:
    /* notify table about completness */
    if( rc )
	; /* but not if terminated by error */
    else if( scroll )
	WamCreateTableRow( tbl, eof? -103:-113, NULL, 0 );
    else
	WamCreateTableRow( tbl, eof? -3:-13, NULL, 0 );
    return rc;
}



/****************
 * Holt eine Beschreibung zu dem angegebenen Fehlercode
 * mode should be 0
 * Returns: Allocated Buffer with Text (caller must free)
 *	    or NULL if there is no Description
 */

char * _wamapi
WamSQLErrorDescription( int ec, int mode )
{
    char *p;

    p = malloc(strlen(lastSQLState)+strlen(lastSQLErrorMsg)+30);
    if( p ) {
	const char *s = lastSQLErrorMsg;
	/* Es kann sein, dass die Meldung mit einigen Tags [..][...]
	 * beginnt, diese habe keine weitere Bedeutung und werden entfernt
	 */
	while( *s == '[' ) {
	    for(s++; *s ; )
		if( *s++ == ']' )
		    break;
	}
	if( !*s ) /* nanu! */
	    s = lastSQLErrorMsg;
	sprintf(p, "SQL Error code %s.\n%s", lastSQLState, s);
    }
    return p;
}




/****************************************************
 ************ DLL Initialization stuff **************
 ***************************************************/
#ifdef MAKE_DLL

const char * _wamapi
WamDLLInfoDB( int level )
{
    const char *p;
  #if __IBMC__
    int x;

    x = _dllentry;
  #endif


    switch( level ) {
      case 10:
      case 0:	p = "WAM ODBC-Interface"
		    "; Copyright 1995,96 by Werner Koch (dd9jn)" ; break;
      case 1:
      case 11:	p = ""; break;
      case 16:
      case 13:	p = "WAMODBC"; break;
      case 14:	p = VERSION; break;
      case 15:	p = ""; break;
      case 23:	p = __DATE__ " " __TIME__; break;
      case 71:	p = WAMDB_INTERFACE_VERSION; break;
      case 72:	p = "2"; break; /* DB Class */
      default: p = "?";
    }
    return p;
}

#endif /* MAKE_DLL */

/**** end of file ****/
