/* [wam/dbgupta.c wk 08.02.93] Access primitives for GUPTA
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************
 *
 ******************************************************
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: dbgupta.c,v 1.10 1996/09/18 13:16:31 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/string.h>

#include <wk/wam.h>
#include "version.h"
#include "wammain.h"
#include "wamdb.h"

#include "gupta.h" /* Gupta Header modified by me for OS2 2.0 and CSet/2*/

#error returncodes are outdated!
#error I must write a mapping function to continue support for gupta
#error This is due to changes for ODBS in database.c (see dbodbc.c)

#define MAKE_DLL 1

/**************************************************
 *************	Constants  ************************
 **************************************************/
#define DEFAULT_LOADLIMIT 500

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/
typedef struct {
	SQLTFSC pfc;	/* fetch status code */
	SQLTCDL cvl;	/* currentValue length */
	int	idt;	/* internal datatype */
	char	d[256]; /* data buffer */
} coldes_t;

typedef struct crsNode_s {
    int  used;	    /* this cursor is in use */
    SQLTCUR crs;    /* der Cursor */

    coldes_t *coldes;
    SQLTNSI nsi;
    int loadLimit;

    struct crsNode_s *next;
} crsNode_t;


static char *logonString; /* NULL or alloced logon infos */
static crsNode_t *cursorList;

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static crsNode_t *GetCursor(void);
static void ReleaseCursor( crsNode_t *c );
static void DisconnectAllCursors(void);

static int ScrollFnc( id tbl, void *arg, int reserved );
static int FetchLoop( id tbl, SQLTCUR crs,
		      coldes_t *coldes, SQLTNSI nsi, int loadLimit );

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

#define ATOI_1( p )   ( isdigit(*(p)) ? *(p) - '0' : 0 )
#define ATOI_2( p )   (( ATOI_1(p) * 10) + ATOI_1( (p)+1 ))
#define ATOI_3( p )   (( ATOI_2(p) * 10) + ATOI_1( (p)+2 ))
#define ATOI_4( p )   (( ATOI_2(p) * 100) + ATOI_2( (p)+2 ))



/**************************************************
 *************	Global Functions  *****************
 **************************************************/

/****************
 * Connect to the Database
 * Returns: 0 = Okay or Errorcode from DB
 */

int _wamapi
    WamSQLConnect( int *hd,	/* Return: handle (in SQL jargon: cursor) */
		   const char *dbName,	/* Name of database to connect */
		   const char *user,	/* name of user */
		   const char *pass )	/* and finaly the password */
{
    char *p;
    int rc;
    SQLTCUR  crs = 0;
    SQLTDPV  dpv;

    FREE(logonString);

    p = xmalloc( strlen(dbName) + 1 + strlen(user) + 1 + strlen(pass) + 1 );
    strcpy( p, dbName ); strcat( p, "/" );
    strcat( p, user );	 strcat( p, "/" );
    strcat( p, pass );

    rc = sqlcnc( &crs, p, 0 );
    if( !rc ) {
	/* should only be done at the first logon: */
	/* set parameter: return NULL-Value Indicator FETRNUL */
	dpv = 1;
	rc = sqlset(0, SQLPNIE, (SQLTDAP)&dpv, 0 );
	if( rc )
	    sqldis( crs );
    }

    if( rc )
	free(p);
    else
	logonString = p;

    *hd = crs;
    return rc;
}



int _wamapi
WamSQLDisconnect( int hd )
{
    SQLTCUR  crs = hd;
    int rc;
    DisconnectAllCursors();
    rc = sqldis( crs );
    if( !rc )
	FREE(logonString);
    return rc;
}



/****************
 * Einen Zusatz Cursor besorgen.
 * Diese werden nur intern benutzt, wobei der Handle
 * aber an andere Module durchgereicht werden kann.
 */

static crsNode_t *GetCursor()
{
    crsNode_t *c;
    int rc;

    if( !logonString ) {
	Error(0,"error: attempt to get a DB-Cursor while noone is logged in");
	return NULL;
    }

    for( c = cursorList; c ; c = c->next )
	if( !c->used ) { /* reuse this connection */
	    c->used++;
	    return c;
	}

    /* all connections ar in use: create a new one */
    Info("Establishing a new Database connection");
    c = xcalloc(1, sizeof *c );
    rc = sqlcnc( &c->crs, logonString, 0 );
    if( rc ) {
	Error(0,"Error creating a new DB-Cursor: rc=%d", rc);
	free(c);
	return NULL;
    }
    c->used++;
    c->next = cursorList;
    cursorList = c;
    return c;
}

/****************
 * Und so einen Cursor wieder freigeben
 */

static void ReleaseCursor( crsNode_t *node )
{
    crsNode_t *c;

    for( c = cursorList; c ; c = c->next )
	if( c == node ) {
	    if( !c->used )
		Error(0,"error: attempt to release an unused cursor");
	    else {
		FREE(c->coldes);
		c->used=0;
	    }
	    return;
	}
    Error(0,"error: releasing unknown cursor");
}


static void DisconnectAllCursors()
{
    crsNode_t *c, *next;
    int rc;

    for( c = cursorList; c ; c = next )
	if( c->used ) {
	    if( rc = sqldis( c->crs ) )
		Error(0,"error: DisconnectAllCursors: rc =%d", rc );
	    FREE(c->coldes);
	    c->used = 0;
	    next = c->next;
	    free( c );
	}
    cursorList = NULL;
}



int _wamapi
WamSQLCommit( int hd )
{
    SQLTCUR  crs = hd;	/* wird aber sowieso fuer alle cursor ausgefuehrt */
    return sqlcmt( crs );
}

int _wamapi
WamSQLRollback( int hd )
{
    SQLTCUR  crs = hd;	/* wird aber sowieso fuer alle cursor ausgefuehrt */
    return sqlrbk( crs );
}

int _wamapi
WamSQLAttention( int hd, int nr )
{
    SQLTCUR  crs = hd;
    return sqlcan( crs );
}



/****************
 * Fuehrt ein beliebiges SQL Command aus
 * Bind variables werden z.Z. noch nicht unterstuetzt, koenne
 * aber spaeter unterstuetzt werden ueber weiter Argumente
 */

int _wamapi
WamSQLExecute( int hd, const char *str, ... )
{
    SQLTCUR crs;
    SQLTDAP sql;
    int rc, i;

    sql = xstrdup(str);
    crs = hd;
    if( rc = sqlcom( crs, sql, 0 ) )
	goto errLabel;

    if( rc = sqlexe( crs ) )
	goto errLabel;

  errLabel:
    free( sql );
    return rc;
}


/****************
 * Fuehrt ein Select Statement aus, dies ist besonders, da es Daten
 * zurueckliefert.  ....
 * Returns: DB Error Code
 */

int _wamapi
WamSQLSelect( int hd,
		  const char *sqlStr, /* SQL String */
		  int loadLimit,
		  id *retTbl	      /* created Table */
		)
{
    SQLTCUR crs;
    SQLTNSI  nsi;   /* # of select items */
    SQLTDDT  edt;   /* external datatype */
    SQLTDDL  edl;   /* external data length */
    SQLTPDT pdt;    /* program datatype */
    char     chp[20]; /* column heading buffer */
    SQLTCHL  chl; /* column heading length */
    size_t   idl; /* internal data length */
    coldes_t *coldes, *cp;    /* allocated array[nsi] and a rover */
    SQLTDAP  sql;
    crsNode_t *crsNode;

    id	tbl;	    /* created Table Object */
    int rc, i;
    symbol_t sym;
    char *p;

    crsNode = NULL;
    sql = xstrdup(sqlStr);
    coldes = NULL;
    tbl = nil;
    if( !loadLimit )
	loadLimit = DEFAULT_LOADLIMIT;

    if( loadLimit < 0 ) { /* we need a new Database cursor */
	if( !(crsNode = GetCursor()) ) {
	    rc = 6002; /* this is the DB errorcode for: Too many DB2 cursors */
	    goto errLabel;
	}
	crs = crsNode->crs;
    }
    else
	crs = hd;

    if( rc = sqlcom( crs, sql, 0 ) )
	goto errLabel;
    if( rc = sqlnsi( crs, &nsi ) )
	goto errLabel;

    coldes = xcalloc( nsi, sizeof *coldes );

    tbl = WamDescribeTableColumns( nil, 0, -1, nsi );

    for(i=0,cp=coldes; i < nsi; i++, cp++ ) {
	if( rc = sqldsc( crs, i+1, &edt, &edl, chp,&chl,SQLNPTR,SQLNPTR ) )
	    goto errLabel;
	chp[chl] = 0;
	idl = 0;
	switch( edt ) {
	  case SQLECHR:  /* CHAR	     */
	  case SQLEVAR:  /* VARCHAR	     */
	    cp->idt = WAM_DT_STRING;
	    idl = edl;
	    pdt = SQLPBUF;
	    break;
	  case SQLEINT:  /* INTEGER	     */
	  case SQLESMA:  /* SMALLINT	     */
	    cp->idt = WAM_DT_INTEGER;
	    pdt = SQLPSLO;
	    break;
	  case SQLEDAT:  /* DATE	     */
	    cp->idt = WAM_DT_DATE;
	    pdt = SQLPBUF;
	    break;
	  case SQLETIM:  /* TIME	     */
	    cp->idt = WAM_DT_TIME;
	    pdt = SQLPBUF;
	    break;
	  case SQLETMS:  /* TIMESTAMP	     */
	    cp->idt = WAM_DT_TSTAMP;
	    pdt = SQLPBUF;
	    break;

	  case SQLEFLO:  /* FLOAT	     */
	  case SQLEDOU:  /* DOUBLE	     */
	    cp->idt = WAM_DT_FLOAT;
	    pdt = SQLPDOU;
	    break;

	  case SQLELON:  /* LONGVAR	     */
	  case SQLEDEC:  /* DECIMAL	     */
	  case SQLEMON:  /* MONEY	     */
	  case SQLEGPH:  /* GRAPHIC	     */
	  case SQLEVGP:  /* VAR GRAPHIC      */
	  case SQLELGP:  /* LONG VAR GRAPHIC */
	  case SQLEBIN:  /* BINARY	     */
	  case SQLEVBI:  /* VAR BINARY	     */
	  case SQLELBI:  /* LONG BINARY      */
	  case SQLEBOO:  /* BOOLEAN	     */
	  default:
	    Info("SQLBase returned unsupported datatype %u", edt );
	    rc = 5146; /* TLK IDT Invalid data type */
	    goto errLabel;
	}

	/* Der Name kann noch mit CorrelationName der Table versehen sein */
	/* Dieser sollte unbedingt entfernt werden, auch wenn dies doppelte */
	/* Namen ergeben koennte */
	if( p = strchr( chp, '.' ) )
	    sym = WamCreateSymbol( p+1 );
	else
	    sym = WamCreateSymbol( chp );

	WamDescribeTableColumns( tbl, sym, cp->idt, idl );
	if( rc = sqlssb( crs, i+1, pdt, cp->d, 255, 0, &cp->cvl, &cp->pfc ) ) {
	    Info("SQLSSB rc=%d: chp='%s' edt=%u pdt=%u\n", rc, chp, edt,pdt);
	    goto errLabel;
	}
    }

    WamDescribeTableColumns( tbl, 0, crsNode ? -102 : -2, 0 );

    if( rc = sqlexe( crs ) )
	goto errLabel;


    if( crsNode ) {  /* tell table how to continue */
	crsNode->coldes   = coldes;
	crsNode->nsi	  = nsi;
	crsNode->loadLimit= loadLimit;
	WamSetTableScrollFnc( tbl, ScrollFnc, crsNode );
    }

    if( rc = FetchLoop(tbl, crs, coldes, nsi, loadLimit ) )
	goto errLabel;


  errLabel:
    if( !crsNode )
	free( coldes );
    free( sql );
    if( rc ) {
	if( crsNode )
	    ReleaseCursor(crsNode);
	WamDestroyTable(tbl);
    }
    else
	*retTbl = tbl;
    return rc;
}




/****************
 * Diese Funktion wird von Class Table benutzt um
 * weitere Rows zu holen. Nachdem keine Rows mehr benoetigt werden,
 * soll diese funktion mit tbl=nil aufgerufen werden um
 * die Resourcen freizugeben.
 * (reserved kann einmal benutzt werden um rueckwaerts scrollen
 *  zu implementieren)
 */

static int ScrollFnc( id tbl, void *arg, int reserved )
{
    int rc;
    crsNode_t *c = arg;

    if( tbl ) {
	rc = FetchLoop(tbl, c->crs, c->coldes, c->nsi, c->loadLimit );
	if( rc )
	    ReleaseCursor(c);
    }
    else {
	ReleaseCursor(c);
	rc=0;
    }
    return rc;
}


static int FetchLoop( id tbl, SQLTCUR crs,
		      coldes_t *coldes, SQLTNSI nsi, int loadLimit )
{
    int rc=0, i;
    timeStamp_t tstmp;
    coldes_t *cp;
    int scroll, eof;

    if( loadLimit < 0 ) {
	loadLimit = -loadLimit;
	scroll=1;
    }
    else
	scroll=0;

    eof = 0;
    while(loadLimit) {
	loadLimit--;
	if( rc = sqlfet( crs ) )
	    if( rc == 1 ) {/* last row has been fetched */
		rc = 0;
		eof++;
		break;
	    }
	    else /* 2 = update, 3 = delete performed since last fetch */
		goto errLabel;
	/* Begin a new row */
	WamCreateTableRow( tbl, -1, NULL, 0 );

	/* loop over all columns */
	for(i=0,cp=coldes; i < nsi; i++, cp++ ) {
	    if( !cp->pfc ) {  /* success */
		int a, b , c;

		switch( cp->idt ) { /* some types need conversion */
		  case WAM_DT_DATE:
		    a = ATOI_4( cp->d );
		    b = ATOI_2( cp->d+5 );
		    c = ATOI_2( cp->d+8 );
		    tstmp.d = Date2JD( c, b, a );
		    WamCreateTableRow( tbl, 0, &tstmp.d , 0 );
		    break;

		  case WAM_DT_TIME:
		    tstmp.t  = ATOI_2( cp->d ) * 3600;
		    tstmp.t += ATOI_2( cp->d+3 ) * 60;
		    tstmp.t += ATOI_2( cp->d+6 );
		    tstmp.t *= 1000;
		    /* die letzten drei Stellen (6 st. Millisekunden) */
		    /* muessen leider entfallen -- evtl. hier noch runden */
		    /* oder andere Aktionen um mit Insert/Update zu */
		    /* synchronisieren */
		    tstmp.t += ATOI_3( cp->d+9 );
		    WamCreateTableRow( tbl, 0, &tstmp.t , 0 );
		    break;

		  case WAM_DT_TSTAMP:
		    a = ATOI_4( cp->d );
		    b = ATOI_2( cp->d+5 );
		    c = ATOI_2( cp->d+8 );
		    tstmp.d = Date2JD( c, b, a );
		    tstmp.t  = ATOI_2( cp->d+11) * 3600;
		    tstmp.t += ATOI_2( cp->d+14) * 60;
		    tstmp.t += ATOI_2( cp->d+17);
		    tstmp.t *= 1000;
		    tstmp.t += ATOI_3( cp->d+20);
		    WamCreateTableRow( tbl, 0, &tstmp , 0 );
		    break;

		  default:
		    WamCreateTableRow( tbl, 0, cp->d, cp->cvl );
		}
	    }
	    else if( cp->pfc == FETRNUL )   /* NULL Value */
		WamCreateTableRow( tbl, 0, NULL, 0 );
	    else {
		Fatal("SQL Fetch returned %u for %d. column", cp->pfc, i );
		rc = -1;
		goto errLabel;
	    }
	}

	/* make new row permanent */
	WamCreateTableRow( tbl, -2, NULL, 0 );
    }

    /* notify table about completness */
    if( scroll )
	WamCreateTableRow( tbl, eof? -103:-113, NULL, 0 );
    else
	WamCreateTableRow( tbl, eof? -3:-13, NULL, 0 );

  errLabel:
    return rc;
}







/****************
 * Holt den eine Beschreibung zu dem angegebenen Fehlercode
 * mode should be 0
 * Returns: Allocated Buffer with Text (caller must free)
 *	 or NULL when there is no Description
 */

char * _wamapi
WamSQLErrorDescription( int ec, int mode )
{
    SQLTRCD rcd;
    char *p;

    rcd = ec;
    p = xmalloc(1000);
    if( sqletx( rcd, SQLXMSG + SQLXREA + SQLXREM, p, 1000, NULL ) )
	sprintf(p, "SQL Error code %d\n\nNo Description available", ec);
    return p;
}




/**************************************************
 *************	Test & Debug Suite ****************
 **************************************************/
#if 0
int main( int argc, char**argv)
{
    SQLTCUR  crs = 0;
    SQLTRCD  rc  = 0;
    SQLTNSI  nsi; /* # of select items */
    SQLTDDT  ddt; /* database type */
    SQLTDDL  ddl; /* database len */
    char     chp[50]; /* column heading buffer */
    SQLTCHL  chl; /* column heading length */
    SQLTPDT  pdt; /* program type */
    SQLTPDL  pdl; /* program length */
    char   line[300], *p;
    int i, length;

    if( rc = sqlcnc( &crs, argv[1], 0 ) )
	Error(2,"Error connecting rc=%u", rc );

    puts("connected");

    if( rc = sqlcom( crs, "SELECT * from TPartner", 0 ) )
	Error(2,"Error compiling rc=%u", rc );

    if( rc = sqlnsi( crs, &nsi ) )
	Error(2,"Error getting # of cols rc=%u", rc );
    printf("number of columns: %u\n", nsi );

    p = line;
    for(i=0; i < nsi; i++ ) {
	if( rc = sqldsc( crs, i+1, &ddt, &pdl,
			 chp,&chl,SQLNPTR,SQLNPTR ) )
	    Error(2,"Error on describe rc=%u", rc );
	chp[chl] = 0;
	printf("dbtype: %2u  len: %2u '%s' %u", ddt, ddl, chp, chl );

	pdt = SQLPBUF;
	if( rc = sqlssb( crs, i+1, pdt, p, pdl, 0, SQLNPTR, SQLNPTR ) )
	    Error(2,"Error on SetSelectBuffer rc=%u", rc );
	printf(" pdl=%u\n", pdl );

	p += pdl + 1;
    }

    length = p - line;
    printf("total length=%d\n", length );

    if( rc = sqlexe( crs ) )
	Error(2,"Error executing rc=%u", rc );

    for(;;) {
	memset( line, ' ', length ); line[length] = 0;
	if( rc = sqlfet( crs ) )
	    if( rc == 1 )
		break;
	    else
		Error(2,"Error fetching rc=%u", rc );
	puts( line );
    }



    if( rc = sqldis( crs ) )
	Error(2,"Error disconnecting rc=%hu", rc );

    return 0;
}
#endif


/****************************************************
 ************ DLL Initialization Stuff **************
 ***************************************************/
#ifdef MAKE_DLL

const char * _wamapi
WamDLLInfoDB( int level )
{
    const char *p;
  #if __IBMC__
    int x;

    x = _dllentry;
  #endif


    switch( level ) {
      case 10:
      case 0:	p = "WAM DB-Interface for GUPTA"
		    "; Copyright 1993 by Werner Koch (dd9jn)" ; break;
      case 1:
      case 11:	p = ""; break;
      case 16:
      case 13:	p = "WAMDBGupta"; break;
      case 14:	p = VERSION; break;
      case 15:	p = ""; break;
      case 23:	p = __DATE__ " " __TIME__; break;
      case 71:	p = WAMDB_INTERFACE_VERSION; break;
      case 72:	p = "1"; break;
      default: p = "?";
    }
    return p;
}


#endif /* MAKE_DLL */

/**** end of file ****/
