/* [wam/database.c wk 9.02.93] Class Database
 *	Copyright (c) 1993 by Werner Koch (dd9jn)
 ******************************************************
 * Generic Database interface for SQL supporting dbs
 *
 * Besondere Flags, fuer SQL Strings.
 * Diese sollte am Anfang des String mit einem '/' beginnen
 * dann den Flagnamen enthalten und eventuell von einen'=' und einem
 * Wert gefolgt.
 * Sinnvolle Flags sind:
 * /LL=n ::= LoadLimit: Es werden max. n Datensaetze be einem Select
 *	     in die Tabelle gestellt, es erfolgt keine fehlermeldung
 *	     Ist n==0 oder nicht angegeben, so erfolgt
 *	     keine Fehlermeldung, wenn das SystemLoadLimit erreicht ist.
 *	     Dies wird nur bei SELECTS beartet.
 * /Prot=key ::= Nur wenn der SecurityKey gltig ist, so wird das SQL
 *	     durchgefhrt. Dieser Key muss vorher bei der Database
 *	     registriert werden. Diese wird dann normalerweise eine
 *	     vorher registrierte Methode aufrufen.
 *	     Dieses Flag kann mehrfach vorhanden sein und alle Aktionen
 *	     in dieser Rehenfolge geprft, bis eines das Ergebniss falsch
 *	     zurckgibt.
 *
 *  This file is part of WAM.
 *
 *  WAM is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  WAM is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * History:
 */

#include <wk/tailor.h>
RCSID("$Id: database.c,v 1.17 1997/03/25 14:46:21 wk Exp $")
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <wk/string.h>
#include <wk/environ.h>

#define CLASS_IMPLEMENTATION 1
#include <wk/wam.h>
#include "wammain.h"
#include "wamdb.h"
#include "wamgui.h"  /* for WAM Set WindowPtr etc */


/**************************************************
 *************	Constants  ************************
 **************************************************/

/**************************************************
 *************	Local Vars & Types ****************
 **************************************************/

static const char *(* _wamapi SQLInfo)( int level ) = NULL;
static int (*_wamapi SQLConnect)( int *, const char *,
					 const char *, const char * );
static int (*_wamapi SQLDisconnect)( int );
static int (*_wamapi SQLExecute)( int hd, const char *str);
static int (*_wamapi SQLSelect)( int hd, const char *sqlStr, int ll,id *retTbl);
static int (*_wamapi SQLCommit)( int hd );
static int (*_wamapi SQLRollback)( int hd );
static char *(*_wamapi SQLErrorDescription)( int ec, int mode );
static int (*_wamapi SQLAttention)( int hd, int nr );


DCLSHAREDPART(Database)

BEGIN_DCLPRIVATEPART
    /* no instance Vars, we will use the factory directly */
END_DCLPRIVATEPART


static char *dbName;	/* malloced Name of database, NULL if not set */
static char *dbUser;	/* malloced Name of logged on user or NULL */
static int type_of_database;
static int cursor;  /* the database Cursor - we will only use one connection */
		    /* we have to change this later */
static int insideTransaction;
static int lastRC; /* last errorcode */

static int silent_exec;

typedef struct {
    symbol_t keyword; /* oder 0 be unbenutzenm slot */
    symbol_t mthd;    /* method selector */
    id	     recv;    /* receiver of message */
} protection_t;

static protection_t protectionTbl[10];	 /* feste Table */
DCL_SERIALIZE(protectionTblSema);

/**************************************************
 *************	Local Prototypes  *****************
 **************************************************/
static const char *ProcessFlags( const char* string, int *ll );
static int TestProtection( const char *key );
static int IsDQLStatement( const char *string );
static char *CreateSQLString( const char *sql, id values );
static char *CreateInsertString( symbol_t tblName, id valueDict );
static char *CreateUpdateString( symbol_t tblName, id valueDict,
				 const char *cont, id where );
static int LoadModule( const char *moduleName );

/**************************************************
 *************	Local Functions  ******************
 **************************************************/

/****************
 * Die Flags auswerten und lastRC eventuell setzen.
 * Returns: NULL bei eine Fehler ansonsten
 *	    Pointer auf String hinter den Flags.
 */

static const char *
ProcessFlags( const char *string, int *loadLimit )
{
    char buf[50];
    int i;
    char *p;

    silent_exec = 0;
    if( loadLimit )
	*loadLimit = 0; /* default value */
    if( !string )
	return ""; /* return an empty string in this case */
    /* skip white spaces */
    for(;;) {
	while( isspace(*string) )
	    string++;
	if( *string != '/' )
	    return string; /* no (more) flags */
	/* get name and value */
	for(i=0,string++; i < DIM(buf)-1 && *string &&
			   *string !='/' &&  !isspace(*string); string ++)
	    buf[i++] = *string;
	buf[i] = 0;
	if( i == DIM(buf)-1 ) {
	    Error(0,"Warning: SQL-Flag truncated to: '%s'", buf);
	    while( *string != '/' && !isspace(*string) ) /* skip the rest */
		string++;
	}
	if( p = strchr(buf, '=') ) { /* got a value */
	    *p = 0;
	    p++;
	}

	if( !stricmp(buf, "Prot") ) { /* handle Protection */
	    if( !*p )
		Error(0,"Warning: No key specified for protection");
	    else if( TestProtection( p ) ) {
		lastRC = WAMDB_E_PROT_FAILED;
		if( !silent_exec )
		    Info("Securitycheck %s failed (SQL=\"%s\")", p, string);
		return NULL;   /* protection failed */
	    }
	}
	else if( !stricmp(buf, "LL") ) {  /* handle load limit */
	    if( !loadLimit )
		Info("Ignoring LoadLimit for this command");
	    else if( p )
		*loadLimit = atoi(p);
	}
	else if( !stricmp(buf, "silent") ) {
	    silent_exec = 1;
	}
	else
	    Error(0,"Warning: Skipping unknown SQL-Flag '%s'", buf);
    }
    /*NOTREACHED*/
}


/****************
 * Alle registrierten Methode fuer den gegebenen Protectionkey
 * aufrufen und beim ersten nicht erfuellten funktion beenden
 * Returns: 0 ::= alles okay
 *	    n ::= (z.Z. noch beliebiger Wert) Fehler
 */

static int
TestProtection( const char *key )
{
    symbol_t keyword;
    volatile symbol_t k;     /* do not optimize them away */
    volatile symbol_t mthd;
    volatile id       recv;
    protection_t *prot;
    int i, rc;

    keyword = WamCreateSymbol(key);

    for(prot=protectionTbl, i=0; i < DIM(protectionTbl); i++, prot++ )
	if( prot->keyword == keyword ) {
	    k	 = prot->keyword;
	    mthd = prot->mthd;
	    recv = prot->recv;
	    /* still valid entry (may be deregistered meanwhile) */
	    if( k && mthd && recv ) {
		rc = i_msg( recv, mthd );
		if( !rc )
		    return -1; /* security check failed */
	    }
	}
    return 0; /* security check passed */
}



/****************
 * Testet ob es sich bei dem String um ein SQL-DQL Kommando
 * handelt (SELECT)
 * Kann erst benutzt werden, wenn die Flags entfernt sind.
 */

static int
IsDQLStatement( const char *string )
{
    const char *p;

    if( !(p = stristr(string, "SELECT" )) )
	return 0;
    /* testen ob nicht etwa create View oder in einem String vorkommt */
    for( ; *string != *p; string++ )
	if( !isspace(*string) )
	    return 0;  /* doch keins */

    return 1;  /* is an DQL-Statement */
}



/****************
 * Aus einem String ein vollstaendiges SQL-Kommando aufbauen
 * Die Stellen wo Einsetzungen erwartet werden sind durch
 * ":<nonspace> " zu kennzeichnen, erst das Blank oder das Stringende
 * kennzeichnet das Ende dieser Markierung (ebenso: '(',')' und ',').
 * Die Markierung: ":#" , bzw. alle die so anfangen sind dummy markierungen
 * die nicht eingesetzt werden, sondern nur der argpointer weter setzen
 * Markierungen die mit ":@" beginnen werden als normaler String eingesetzt und
 * nicht in einen SQL String umgewandelt.
 *
 * Die Objekte im IDArray muessen die methode asDBString verstehen
 * Falls keine Einsetzungen notwendig sin, kann dort auch nil stehen
 *
 * Returns: Nil on Error or an allocated Sring (caller must free)
 */

static char *
CreateSQLString( const char *sql, id values )
{
    id data;
    int nIns; /* Anzahl der einzufuegenden daten */
    int nVal; /* anzahl der values */
    char *string, *s;
    const char *p, *r;
    int instring =0;

    nVal = values ? i_msg( values, sym_size ) : 0;
    for(p=sql,nIns=0; *p ; p++ )
	if( instring ) {
	    if( *p == '\'' ) {
		if( p[1] == '\'' )
		    p++; /* skip escaped string delimiter */
		else
		    instring = 0;
	    }
	}
	else if( *p == '\'' )
	    instring++;
	else if( *p == ':' )
	    nIns++;
    if( instring ) {
	lastRC = WAMDB_E_UNCLOSED_STR;
	return NULL;
    }
    if( !nIns && !nVal ) /* keine Einsetzungen */
	return xstrdup(sql);
    if( nIns != nVal ) {
	lastRC = WAMDB_E_VAR_MISMATCH;
	Error(0,"Error: SQL requests %d inserts; %d values are supplied",
							       nIns, nVal);
	return NULL;
    }

    data = msg( values, sym(asDBStringArray) );
    string = xmalloc( strlen(sql) + I_msg( data, sym(cumulativeLength) ) );

    if( !msg(data, sym_enumOpen ) )
	BUG();	/* size is ja > 0 ! */
    for(p=sql,s=string; *p ; p++ )
	if( instring ) {
	    if( *p == '\'' ) {
		if( p[1] == '\'' )
		    *s++ = *p++; /* skip escaped string delimiter */
		else
		    instring = 0;
	    }
	    *s++ = *p;
	}
	else if( *p == '\'' ) {
	    instring++;
	    *s++ = *p;
	}
	else if( *p != ':' )
	    *s++ = *p;
	else {
	    r = p_msg(data, sym_enumGet);
	    xassert(r);
	    if( p[1] != '#' ) { /* '#xxx' is a dummy arg */
		if( p[1] == '@' && *r == '\'' ) {
		    r++;
		    while( *r ) {
			if( *r == '\'' && !r[1] )
			    break;  /* end of string */
			*s++ = *r++;
			if( *r == '\'' && r[1] == '\'' )
			    r++;  /* remove escaped string delimiter */
		    }
		}
		else {
		    while( *r )
			*s++ = *r++;
		}
	    }
	    /* jetzt noch p ueber den doppelpunkt hinweg */
	    *s++ = ' '; /* schonmal ein blank hintendran */
	    while( *++p )
		if( isspace( *p ) || *p=='(' || *p==')' || *p == ',' )
		    break;
	    p--;
	}
    *s = 0;
    xassert( !msg(data, sym_enumGet) ); /* duerfen keine mehr ueber sein */
    freeObj(data);
    return string;
}


/****************
 * Ein INSERT Command aufbauen: Argumente sind:
 * 1. Der Name der Table
 * 2. Ein Dictionary mit dem Namen und dem Wert.
 */

static char *
CreateInsertString( symbol_t tblName, id valueDict )
{
    id keys, vals;
    char *string, *s;
    const char *r;
    size_t len;

    keys = msg( valueDict, sel(keysAsStringArray) );
    vals = msg( valueDict, sel(valuesAsDBStringArray) );
    len  = I_msg( keys, sym(cumulativeLength) ) +
	   I_msg( vals, sym(cumulativeLength) ) ;
    r = symName(tblName);
    len += 12 + strlen(r) + 2 + 7 + 2;
    string = xmalloc( len+5 );

    s = stpcpy( string, "INSERT INTO ");
    s = stpcpy( s, r   );
    s = stpcpy( s, "(" );
    if( msg(keys, sym_enumOpen ) )
	while( r = p_msg(keys, sym_enumGet) ) {
	    s = stpcpy( s, r );
	    s = stpcpy( s, "," );
	}
    s[-1] = ')';  /* replace last comma */
    freeObj(keys);
    s = stpcpy( s, " VALUES(");
    if( msg(vals, sym_enumOpen ) )
	while( r = p_msg(vals, sym_enumGet) ) {
	    s = stpcpy( s, r );
	    s = stpcpy( s, "," );
	}
    s[-1] = ')';  /* replace last comma */
    freeObj(vals);

    return string;
}



/****************
 * Ein UPDATE Command aufbauen: Argumente sind:
 * 1. Der Name der Table
 * 2. Ein Dictionary mit dem Namen und dem Wert.
 * 3. Der WHERE String mit den ueblichen Einsetzungen, wie
 *    unter CreateSQLString beschrieben (inklusive WHERE)
 * 4. Die Argumente zu dem WHERE-String
 */

static char *CreateUpdateString( symbol_t tblName, id valueDict ,
				 const char *whereStr, id whereArgs )
{
    id keys, vals;
    char *string, *s;
    const char *r;
    char *wherePart;
    size_t len;

    /*Erstmal den WhereTeil zusammenbauen */
    if( !(wherePart = CreateSQLString( whereStr, whereArgs )) )
	return NULL;

    keys = msg( valueDict, sel(keysAsStringArray) );
    vals = msg( valueDict, sel(valuesAsDBStringArray) );
    len  = I_msg( keys, sym(cumulativeLength) ) +
	   I_msg( vals, sym(cumulativeLength) ) ;
    r = symName(tblName);
    len += 2 * I_msg( keys, sym_size ); /*" xx=yy"*/
    len += 7 + strlen(r) + 5 + 2 + 7 + 2 + 1 + strlen(wherePart);
    string = xmalloc( len+5 );

    s = stpcpy( string, "UPDATE ");
    s = stpcpy( s, r   );
    s = stpcpy( s, " SET ");
    if( msg(keys, sym_enumOpen ) )
	if( msg(vals, sym_enumOpen ) ) /* kann eigentlich nicht false sein */
	    while( r = p_msg(keys, sym_enumGet) ) {
		s = stpcpy( s, r );
		s = stpcpy( s, "=" );
		if( r = p_msg(vals, sym_enumGet) ) {
		    s = stpcpy( s, r );
		    s = stpcpy( s, "," );
		}
	    }
    s[-1] = ' '; /* replace last comma */
    freeObj(keys);
    freeObj(vals);
    stpcpy( s, wherePart );
    free( wherePart );
    return string;
}


/**************************************************
 ******************  Methods  *********************
 **************************************************/

DCLFOBJFNC( new )
{
    Bug("There is no new for class Database");
    return nil;  /* NOT REACHED */
}


DCLFOBJFNC( load )
{
    DCL_arg(id,moduleName);

    if( LoadModule( getString(moduleName) ) )
	return nil;
    else
	return self;
}

DCLFOBJFNC( isLoaded )
{
    return SQLInfo ? True : False;
}

/****************
 * Set the Name of the database
 */

DCLFOBJFNC( access )
{
    DCL_arg(const char*,name);

    FREE(dbName);
    if( name )
	dbName = xstrdup(name);
    return self;
}


/****************
 * Logons to the Database, name must have been set
 * Returns: 0  = Okay
 *	    rc = returncode from SQL
 */

DCLFOBJFNC_i( logon )
{
    DCL_arg(const char*,user);
    DCL_arg(const char*,pass);
    int rc;


    if( !user )
	user = "";
    if( !pass )
	pass = "";
    FREE(dbUser); /* delete last user */
    WamSetWindowPointer( POINTER_WAIT );
    rc = WamSQLConnect( &cursor, dbName, user, pass );
    WamSetWindowPointer( 0 );
    if( !rc ) {
	dbUser = xstrdup(user);
	strupr(dbUser);
	insideTransaction = 0;
    }
    else {
	lastRC = rc;
    }
    return rc;
}

/****************
 * Logoffs from the Database ( if any connection )
 * will print a message directly if logoff fails
 */

DCLFOBJFNC( logoff )
{
    int rc;

    insideTransaction = 0;
    if( dbUser ) {
	FREE(dbUser);
	WamSetWindowPointer( POINTER_WAIT );
	rc = WamSQLDisconnect( cursor );
	WamSetWindowPointer( 0 );
	if( rc ) {
	    lastRC = rc;
	    Fatal( "SQL Disconnect failed: rc=%d", rc);
	}
    }
    return self;
}


/****************
 * Ein Stringobjekt mit dem Namen des angemeldeten Users zurueckgeben
 * falls kein User angemeldet ist, wird nil zuriueckgegeben
 * Retunrs: neues Objekt mit dem Usernamen oder nil
 */

DCLFOBJFNC( user )
{
    return dbUser ? newString(dbUser) : nil;
}



/****************
 * Allgemeine Methode um ein SQL-Command abzusetzen.
 * Returns: Object of Table bei einem SELECT Command
 *	    self (Database Factory Object) bei anderen Commands
 *	    nil bei einem Fehler
 * Der String kann die Flags enthalten (siehe oben)
 * in den Values koennen naturgemaess keine Nils (NULL) vorkommen.
 */

DCLFOBJFNC( sql )
{
    DCL_arg(const char*,s);
    DCL_arg(id,values); /* IdArray of Values */
    int rc, loadLimit;
    char *str;
    id tbl;

    if( !(s = ProcessFlags( s, &loadLimit )) )
	return nil;
    if( !(str = CreateSQLString( s, values )) )
	return nil;
    if( !silent_exec )
	Info("SQL=\"%s\"", str );
    if( IsDQLStatement( str ) ) {
	WamSetWindowPointer( POINTER_READ );
	rc = WamSQLSelect( cursor, str, loadLimit, &tbl );
	WamSetWindowPointer( 0 );
	free(str);
	if( rc ) {
	    lastRC = rc;
	    Error(0, "SQLSelect failed: rc=%d", rc);
	    return nil;
	}
	return tbl;
    }

    /* DML, DDL or DCL statement or invalid statement */

    WamSetWindowPointer( POINTER_WRITE );
    rc = WamSQLExecute( cursor, str );
    WamSetWindowPointer( 0 );
    free(str);
    if( rc ) {
	lastRC = rc;
	Error(0, "SQLExecute failed: rc=%d", rc);
	return nil;
    }
    return self;
}




/****************
 * Allgemeine Methode um SQL INSERT abzusetzen.
 * Wird dieses ausserhalb einer Transaktion durchgefuehrt, so
 * wird auch ein COMMIT durchgefuehrt.
 * In den values zu den keys hier koennen auch Nils vorkommen, die
 * dann in NULL umgesetzt werden (Dies ist bei einem INSERT natuerlich
 * unsinning)
 */

DCLFOBJFNC( insert )
{
    DCL_arg(symbol_t, tblName);  /* Table which will receive the values */
    DCL_arg(id,values); 	 /* Dictionary mit Values */
    DCL_arg(const char *, s);	 /* String mit den flags */
    int rc;
    char *str;

    if( !(s = ProcessFlags( s, NULL )) )
	return nil;
    if( !(str = CreateInsertString( tblName, values )) )
	return nil;
    if( !silent_exec )
	Info("SQL=\"%s\"", str );
    WamSetWindowPointer( POINTER_WRITE );
    rc = WamSQLExecute( cursor, str );
    free(str);
    if( rc ) {
	lastRC = rc;
	Error(0, "SQLExecute failed: rc=%d", rc);
	WamSetWindowPointer( 0 );
	return nil;
    }
    if( !insideTransaction ) {/* do an automatic commit */
	Info("performing an automatic commit");
	if( rc = WamSQLCommit( cursor ) ) {
	    WamSetWindowPointer( 0 );
	    /* do not set lastRC */
	    Fatal( "SQLCommit failed: rc=%d", rc);
	    return nil;
	}
    }
    WamSetWindowPointer( 0 );
    return self;
}




/****************
 * Allgemeine Methode um SQL UPDATE abzusetzen.
 * Wird dieses ausserhalb einer Transaktion durchgefuehrt, so
 * wir auch ein COMMIT durchgefuehrt.
 * In den values zu den keys hier koennen auch Nils vorkommen, die
 * dann in NULL umgesetzt werden.
 */

DCLFOBJFNC( update )
{
    DCL_arg(symbol_t, tblName);  /* Table which will receive the values */
    DCL_arg(id,values); 	 /* Dictionary mit Values */
    DCL_arg(const char*, whereStr); /* where string */
    DCL_arg(id, whereArgs);	    /* args for the where string */
    int rc;
    char *str;

    if( !(whereStr = ProcessFlags( whereStr, NULL )) )
	return nil;
    if( !(str = CreateUpdateString( tblName, values, whereStr, whereArgs )) )
	return nil;
    if( !silent_exec )
	Info("SQL=\"%s\"", str );
    WamSetWindowPointer( POINTER_WRITE );
    rc = WamSQLExecute( cursor, str );
    free(str);
    if( rc ) {
	lastRC = rc;
	WamSetWindowPointer( 0 );
	Error(0, "SQLExecute failed: rc=%d", rc);
	return nil;
    }
    if( !insideTransaction ) { /* do an automatic commit */
	Info("performing an automatic commit");
	rc = WamSQLCommit( cursor );
	WamSetWindowPointer( 0 );
	if( rc ) {
	    /* do not set lastRC */
	    WamSetWindowPointer( 0 );
	    Fatal( "SQLCommit failed: rc=%d", rc);
	    return nil;
	}
    }
    WamSetWindowPointer( 0 );
    return self;
}



DCLFOBJFNC( beginTran )
{
    int rc;
    if( insideTransaction ) { /* rollback all uncommitted changes */
	Info("Rolling back work from unterminated Transaction");
	if( rc = WamSQLRollback( cursor ) ) {
	    Fatal( "SQLRollback failed: rc=%d", rc);
	}
    }
    Info("BEGIN Transaction");
    insideTransaction = 1;
    return self;
}



DCLFOBJFNC( endTran )
{
    int rc;

    if( !insideTransaction ) {
	Info("Rolling back work for not startet Transaction");
	if( rc = WamSQLRollback( cursor ) ) {
	    Fatal( "SQLRollback failed: rc=%d", rc);
	}
    }
    else {
	if( rc = WamSQLCommit( cursor ) ) {
	    lastRC = rc;
	    Fatal( "SQLCommit failed: rc=%d", rc);
	    return nil;
	}
    }
    Info("END Transaction");
    insideTransaction = 0;
    return self;
}


DCLFOBJFNC( abortTran )
{
    int rc;

    if( rc = WamSQLRollback( cursor ) ) {
	Fatal( "SQLRollback failed: rc=%d", rc);
	return nil;
    }
    Info("ABORT Transaction");
    insideTransaction = 0;
    return self;
}



/****************
 * Erzeugt ein Semaphore innerhalb einer transaction mit dem
 * key type und einer beliebigen Id die zu diesem Key passen muss.
 * Wird dies dann auch von einem anderen Seesion aufgerufen, so wartet diese
 * Methode so lange, bis die entsprechende Transaction beendet ist.
 */

DCLFOBJFNC( lock )
{
  /*  DCL_arg(symbol_t,type);
    DCL_arg(id,key);*/

    if( !insideTransaction )
	Fatal( "Acquiring lock semaphore outside a transaction");
    else
	Info("Semaphore locks in Database not yet supported; ignored");
    return self;
}

DCLFOBJFNC( unlock )
{
  /*DCL_arg(symbol_t,type);
    DCL_arg(id,key);*/

    /* not yet supported; unlock will occur at the end of a transaction
     * automatically. This method can be used to release the semaphore
     * prior to the end of the transaction.
     */
    return self;
}


/****************
 * Ermittelt den letzten Fehlercode - Dabei ist zu beachten, das
 * ein durch ein Rollback verursachter Fehler nicht registriert
 * wird; dies hat den Vorteil, eine Transaktion abbrechen zu koenne
 * und dann den Fehler anzeigen zu koenne, der die Abbruchursache
 * darstellt.
 */

DCLFOBJFNC_i( getLastError )
{
    return lastRC;
}

/****************
 * Creates String Object with error description
 */

DCLFOBJFNC( getLastErrorString )
{
    char *buf, *p, *s;
    id o;

    switch( lastRC ) {
      case WAMDB_E_NO_CORE:    p = "out of core in Wam interface"; break;
      case WAMDB_E_INV_HANDLE: p = "invalid Wam DB handle"; break;
      case WAMDB_E_DB_BUG:     p = "Wam detected a bug in the DB driver";break;
      case WAMDB_E_WAM_BUG:    p = "Wam detected a bug in itself"; break;
      case WAMDB_E_NO_CONNECT: p = "no connection to database"; break;
      case WAMDB_E_DT_NOT_SUP: p = "Datentyp wird von Wam nicht untersttzt";
	break;
      case WAMDB_E_CANCELED:   p = "SQL Befehl wurde abgebrochen."; break;
      case WAMDB_E_NO_DATA:    p = "no (more) data found"; break;
      case WAMDB_E_MISSING_VAL:
	p = "Speichern ist nicht mglich, da zumindest ein "
	    "Pflichtfeld nicht gefllt ist.\n\n"
	    "Fllen Sie bitte alle relevanten Felder aus!";
	break;

      case WAMDB_E_INV_DBNAME:
	p = "Die angegebene Datenbank existiert nicht oder kann\n"
	    "nicht geffnet werden.\n\n"
	    "Geben Sie eine gltige Datenbank an oder benachrichtigen\n"
	    "Sie den Systemadministrator!";
	break;
      case WAMDB_E_INV_USER:
	p = "Die angegebene Benutzerkennung ist nicht "
	    "gltig.\n\n"
	    "Geben Sie eine korrekte Benutzerkennung ein oder "
	    "erfragen Sie eine bei Ihrem Datenbankadministrator!";
	break;
      case WAMDB_E_INV_PASS:
	p = "Das angegebene Pawort ist nicht gltig.\n\n"
	    "Bitte geben Sie das korrekte Pawort ein. "
	    "(Das Pawort wird durch Sternchen angezeigt!)";
	break;
      case WAMDB_E_DUP_KEY:
	p = "Ein Datensatz mit diesem Schlsselwert existiert bereits!";
	break;

      case WAMDB_E_PROT_FAILED:
	p = "Der Datenbankzugriff konnte nicht durchgefhrt werden, "
	    "da eine notwendige Berechtigung nicht vorhanden ist!";
	break;
      case WAMDB_E_VAR_MISMATCH:
	p = "Hostvariable mismatch in supplied SQL-String!";
	break;
      case WAMDB_E_UNCLOSED_STR:
	p = "Unclosed string constant in SQL-String!";
	break;
      case WAMDB_E_MOD_NOT_AVAIL:
	p = "Es ist kein Datenbankmodul verfgbar!";
	break;
      case WAMDB_E_MOD_NOT_FOUND:
	p = "Das Datenbankmodul wurde nicht gefunden!";
	break;
      case WAMDB_E_MOD_INVALID:
	p = "Das Datenbankmodul ist ungltig!";
	break;
      case WAMDB_E_MOD_INV_VER:
	p = "Diese Version des Datenbankmoduls kann nicht benutzt werden!";
	break;
      case WAMDB_E_MOD_DUP_LOAD:
	p = "Es ist bereits ein Datenbankmodul geladen!";
	break;
      case WAMDB_E_EXECUTING:  p = "DB request still executing"; break;
      case WAMDB_E_NEED_DATA:  p = "DB needs data for further processing";break;
      case WAMDB_E_DB_ERROR:   p = NULL /*"DB detected an error"*/; break;
      case WAMDB_E_DB_HANDLE:  p = "Invalid handle passed to DB"; break;
      case WAMDB_E_DB_INV_ARG: p = "Invalid parm apssed to DB"; break;
      case WAMDB_E_UNKNOWN_RC: p = "Unknown returncode from DB"; break;
      case WAMDB_E_NOT_MAPPED: p = "No mapping for DB error available";break;
      default: p = NULL;
    }

    if( p )
	o = newString(p);
    else {  /* get the error description from the database module */
	buf = WamSQLErrorDescription( lastRC, 0 );
	if( buf ) {
	    if( type_of_database == 1) { /*gupta: remove most of the linefeeds*/
		int spaceCount = -1;

		for(s=p=buf; *p; p++ ) {
		    if( *p == '\r' ) /* skip CRS */
			;
		    else if( *p == '\n' ) {
			*s++ = ' ';
			spaceCount=0;
		    }
		    else if( spaceCount > 1 && *p == ' ' ) {
			*s++ = '\n' ;
			spaceCount = -1;
		    }
		    else {
			if( spaceCount == -1 )
			    ;
			else if( *p == ' ' )
			    spaceCount++;
			else
			    spaceCount=-1;
			*s++ = *p == '\t' ? ' ' : *p;
		    }
		}
		*s = 0;
	    }
	    o = newString(buf);
	    free(buf);
	}
	else
	    o = newString("Can't get error description (out of memory?)");
    }
    return o;
}




/****************
 * Eine Methode regsitrieren, die vor dem Absetzen eines
 * SQLs aufgerufen wird um eine Pruefung durchzufuehren.
 * Diese Methode muss treu zurueckliefren, wenn das SQL durchgefuehrt
 * werden soll, andernfalls  wird das SQL gar nicht durchgefuert, sondern
 * ein besonderer Fehlercode gesetzt.
 * Achtung, da ein Receiver registriert wird, muss darauf geachtet werden,
 * das dieser wieder deregistriert wird, bevor der Receiver destroyed wird.
 * Alle Argumene stellen den key dar und sind beim deregistrieren
 * zu benutzen.
 */

DCLFOBJFNC( registerProtection )
{
    DCL_arg(symbol_t, keyword); /* keyword auf den reagiert werden soll */
    DCL_arg(symbol_t, mthd);	/* Methodselector, der gesendet werden soll */
    DCL_arg(id, recv);	       /* An dieses Object soll gesendet werden */
    protection_t *prot;
    int i;

    xassert( keyword );
    xassert( mthd );
    /* find an empty slot */
    BEGIN_SERIALIZE(protectionTblSema);
    for(prot=NULL, i=0; i < DIM(protectionTbl); i++ )
	if( !protectionTbl[i].keyword )  {
	    prot = protectionTbl + i;
	    prot->keyword = keyword;
	    break;
	}
    END_SERIALIZE(protectionTblSema);
    if( !prot )
	Error(8,"Too many protection register request in " __FILE__
					    " (Max=%d)", DIM(protectionTbl));
    prot->mthd = mthd;
    prot->recv = recv;
    return self;
}


/****************
 * Eine registrierte Methode wieder entfernen.
 * es sind wieder alle Argumen, wie bei register anzugeben, verglichen
 * wird hier auf equivalenz und es werden alle gefundenen entries entfernt.
 */

DCLFOBJFNC( deRegisterProtection )
{
    DCL_arg(symbol_t, keyword); /* keyword */
    DCL_arg(symbol_t, mthd);	/* Methodselector */
    DCL_arg(id, recv);		/* Das Objekt */
    protection_t *prot;
    int i;

    xassert( keyword );
    xassert( mthd );
    /* find all slots */
    BEGIN_SERIALIZE(protectionTblSema);
    for(prot=NULL, i=0; i < DIM(protectionTbl); i++ )
	if( protectionTbl[i].keyword == keyword &&
	    protectionTbl[i].mthd    == mthd &&
	    protectionTbl[i].recv    == recv )	{
	    protectionTbl[i].keyword = 0;
	    protectionTbl[i].mthd    = 0;
	    protectionTbl[i].recv    = nil;
	}
    END_SERIALIZE(protectionTblSema);
    return self;

}


DCLFOBJFNC( sqlFailure )
{
    id o;

    o = msg(self, sym(getLastErrorString) );
    WamShowMessageBox( nil, WAM_ERROR, getString(o) );
    freeObj(o);
    return self;
}

/****************
 * This function may be used in case of a dup-key produced by a
 * key which contains a timestamp. It is a central method to
 * wait for a period of time which should assure that the next
 * CURRENT TIMESTAMP of the RDBMS will be different. May also be used
 * to syncronize Timestamps in a networking environment.
 * different labes should be used for different timestamp semantics.
 */
DCLFOBJFNC( waitOnNextTimestamp )
{
   /* DCL_arg(symbol_t,alabel) */
    WamSetWindowPointer( POINTER_WAIT );
    Sleep(1200);
    WamSetWindowPointer( 0 );
    return self;
}

void WamSUC_Database()
{
    id self = Database;
    CREATECLASS("Database");
    WamSubclassClass( sym_Object, self );

    DCLFMTHD( new );
    DCLFMTHD( load );
    DCLFMTHD( isLoaded );
    DCLFMTHD( access );
    DCLFMTHD( logon );
    DCLFMTHD( logoff );
    DCLFMTHD( user );
    DCLFMTHD( sql    );
    DCLFMTHD( insert );
    DCLFMTHD( update );
    DCLFMTHD( beginTran );
    DCLFMTHD( endTran );
    DCLFMTHD( abortTran );
    DCLFMTHD( lock );
    DCLFMTHD( unlock );
    DCLFMTHD( getLastError );
    DCLFMTHD( getLastErrorString );
    DCLFMTHD( registerProtection );
    DCLFMTHD( deRegisterProtection );
    DCLFMTHD( sqlFailure );
    DCLFMTHD( waitOnNextTimestamp );

}

/*************************************************
 **************** DLL Load Stuff *****************
 ************************************************/


/* Wrapper functions */

const char * _wamapi
WamDLLInfoDB( int level )
{
    const char *p;

    if( SQLInfo )   /* DB Module is loaded */
	return SQLInfo(level);

    switch( level ) {
      case 10:
      case 0:	p = "WAM DB-Dummy"; break;
      case 1:
      case 11:	p = ""; break;
      case 16:
      case 13:	p = "WAMDB?"; break;
      default: p = "?";
    }
    return p;
}


int _wamapi
WamSQLConnect( int *a, const char *b, const char *c, const char *d)
{
    if( SQLInfo )
	return SQLConnect(a,b,c,d);
    return WAMDB_E_MOD_NOT_AVAIL;
}

int _wamapi
WamSQLDisconnect( int a)
{
    if( SQLInfo )
	return SQLDisconnect(a);
    return WAMDB_E_MOD_NOT_AVAIL;
}

int _wamapi
WamSQLExecute( int hd, const char *str, ... )
{
    if( SQLInfo )
	return SQLExecute(hd,str);
    return WAMDB_E_MOD_NOT_AVAIL;
}

int _wamapi
WamSQLSelect( int hd, const char *sqlStr, int loadLimit, id *retTbl )
{
    if( SQLInfo )
	return SQLSelect(hd,sqlStr,loadLimit,retTbl);
    return WAMDB_E_MOD_NOT_AVAIL;
}

int _wamapi
WamSQLCommit( int hd )
{
    if( SQLInfo )
	return SQLCommit(hd);
    return WAMDB_E_MOD_NOT_AVAIL;
}

int _wamapi
WamSQLRollback( int hd )
{
    if( SQLInfo )
	return SQLRollback(hd);
    return WAMDB_E_MOD_NOT_AVAIL;
}


int _wamapi
WamSQLAttention( int reserved, int nr )
{
    if( SQLInfo )
	return SQLAttention(cursor,nr);
    return 0;
}

char * _wamapi
WamSQLErrorDescription( int ec, int mode )
{
    if( SQLInfo )
	return SQLErrorDescription( ec, mode );
    return "DB-Module is not available";
}


/****************
 * Returns the type of the Database:
 *   1 := SQL conforming to GUPTA
 *   2 := SQL conforming to ODBC
 */
int
WamSQLType()
{
    return type_of_database;
}


static int LoadModule( const char *moduleName )
{
    struct {
	char *name;
	void **addr;
    } tbl[] = {
	{ "WamDLLInfoDB"           , (void*)&SQLInfo }, /* must be ths first */
	{ "WamSQLConnect"          , (void*)&SQLConnect     },
	{ "WamSQLDisconnect"       , (void*)&SQLDisconnect  },
	{ "WamSQLExecute"          , (void*)&SQLExecute     },
	{ "WamSQLSelect"           , (void*)&SQLSelect      },
	{ "WamSQLCommit"           , (void*)&SQLCommit      },
	{ "WamSQLRollback"         , (void*)&SQLRollback    },
	{ "WamSQLErrorDescription" , (void*)&SQLErrorDescription },
	{ "WamSQLAttention"        , (void*)&SQLAttention   },
	{ NULL }
    };
    const char *p, *p2;
    int i, rc;

    if( SQLInfo )
	return WAMDB_E_MOD_DUP_LOAD;

    if( !(*tbl[0].addr = WKLoadModule(moduleName,tbl[0].name,&rc)) ) {
	Error(0,"Error: LoadModule '%s': rc=%d ", moduleName, rc);
	return WAMDB_E_MOD_NOT_FOUND;
    }
    p = SQLInfo(71);
    p2 = WAMDB_INTERFACE_VERSION;
    if( !p || strcmp( p, p2 ) ) {
	Error(0,"Error: DB-Version is '%s' but we want '%s' ", p, p2);
	/* fixme: we have no WKUnloadModule(), would be nice here */
	*tbl[0].addr = NULL;
	return WAMDB_E_MOD_INV_VER;
    }
    type_of_database = atoi(SQLInfo(72));
    if( type_of_database != 1 && type_of_database != 2 ) {
	Error(0,"Error: DB-Type %d is not supported, using type 1",
						      type_of_database );
	type_of_database = 1; /* gupta syntax */
    }

    for(i=1; tbl[i].name; i++ )
	if( !(*tbl[i].addr = WKLoadModule(moduleName,tbl[i].name,&rc)) ) {
	    Error(0,"Error: LoadModule '%s'(%d): rc=%d ", moduleName,i,rc);
	    return WAMDB_E_MOD_NOT_FOUND;
	}

    return 0;
}



/**** end of file ****/
