//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2006 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
// This file contains Original Code and/or Modifications of Original Code as
// defined in and that are subject to the GNU Lesser General Public License.
// You may not use this file except in compliance with the License.
// BY USING THIS FILE YOU AGREE TO ALL TERMS AND CONDITIONS OF THE LICENSE.
// A copy of the License is provided with the Original Code and Modifications,
// and is also available at http://www.dfsee.com/txwin/lgpl.htm
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License,
// or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; (lgpl.htm) if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// Questions on TxWin licensing can be directed to: txwin@fsys.nl
//
// ==========================================================================
//
// TX standard dialogs
//
// Author: J. van Wijk
//
// JvW  11-06-2006 Initial version, hex editor dialog and control-class
// JvW  16-08-2017 TxLib 5.00; Allow very wide edit windows, more mouse suport
// JvW  17-07-2018 TxLib 5.13; Mouse-marking and clipboard support

#include <txlib.h>                              // public interface
#include <txwpriv.h>                            // txwa anchor interface

/*
 Minimum width is 76 for 16 byte/line, can use 32/64/128 or manual width too
 when wide enough, adds absolute position column. Layout for typical usage:

Ĵ Partition 03 type:07 D: HPFS   size: 1114.8 MiB [X]
[PgUp]   0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f [ItemUp]  [LineUp]
000000  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000010  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000020  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000030  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000040  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000050  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000060  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000070  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000080  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
000090  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
0000a0  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
0000b0  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
0000c0  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
0000d0  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
0000e0  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
0000f0  00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF [0123456789abcdef]
[PgDn] Mod Cursor:012345 = Abs:0x11223344556   relSN:00 [ItemDn]  [LineDn]
LSN:0x00000010 = PSN:0x001ea547 = Cyl:  125 H:1   S:11   item-description 
F1=Help F4=OK  Up/Dn=Row PgUp/PgDn=Page Alt-PgUp/PgDn=Item TAB=Hex/Ascii
*/

#define TXWD_WID_HEXED    0xff80                // window id hex editor
#define TXWD_HELPHEXED    (TXWH_SYSTEM_SECTION_BASE + 800)
#define TXWD_HELPHECTL    (TXWD_HELPHEXED + 10)

static char           *hexedhelp[] =
{
   "#000 Hex editor, generic dialog",
   "",
   " The Hex editor dialog is a single window with just one",
   " control-window, the Hex-editor control window."
   "",
   " The dialog can be used by applications like sector editors",
   " and binary-file editors if they do not need additional",
   " functionality like multiple windows ...",
   "",
   " For help on using the Hex-editor control itself, refer to the",
   " previous help-screen that has all the details about operating",
   " the hex editor with the keyboard and mouse.",
   "",
   TXHELPITEM(10,"Hex editor control window")
   " The Hex editor control is used to view and update unstructered",
   " binary information in hexadecimal and ASCII format, as used by",
   " applications like sector editors and binary-file editors.",
   "",
   " For generic info on the Hex-editor and possibly the way it is",
   " to be used within the current application, press <F1> again ...",
   "",
   "",
   " The control window has a relative position displayed at the left,",
   " a HEX-byte area, an ASCII area and an absolute position shown on",
   " the far right side (screen-size permitting :-)",
   "",
   " The number of columns used will be set to 8, 16, 32, 64 or 128 by",
   " default, depending on the available window-width. It can be changed",
   " to any value between 1 and 256 using the <Alt>+arrow keys.",
   "",
   " There can be a 'MARKED AREA', a range of bytes shown in a different",
   " color which can be FILLED, BYTE-REVERSED, COPIED/MOVED within the",
   " same item buffer, or copied-to/pasted-from the clipboard using one",
   " of several keys or mouse actions (see further below)",
   "",
   " The top and bottom lines include positioning information like the",
   " offset for the cursor from start of item, the absolute position",
   " in the complete edited object (Abs), and the relative sector within",
   " the item (relSN)",
   " ",
   " There are scroll-buttons for use with the mouse on those lines too.",
   "",
   " The informational line at the very bottom is application defined,",
   " and contains information about the data (Item) being edited, like",
   " sector numbering in some form (LSN/PSN/Block/Cluster/CHS info),",
   " and perhaps the type of the data.",
   "",
   " The 'Cursor:'   position is the RELATIVE byte position in the",
   "                 current item (sector/cluster) starting at 0.",
   "",
   " The 'Abs:'      position is the ABSOLUTE byte position in the",
   "                 whole edited object (disk, partition, file)",
   "                 also starting at 0.",
   "",
   " The 'relSN:'    is the relative sector in the current item.",
   "                 (default item size is 1 block, 8 sectors)",
   "",
   "",
   "                 Edit, HEX-editor, keyboard usage",
   "                 ================================",
   "",
   "   UP/DOWN/LEFT/RIGHT   Move the cursor within the area",
   "   arrow keys           moving to the PREVious and NEXT",
   "                        item when needed.",
   "",
   "   PgUp and PgDn        Move up or down by the size of the",
   "                        window (or data-item when smaller)",
   "                        moving to the PREVious and NEXT",
   "                        item when needed.",
   "",
   "   Alt + PgUp or PgDn   Move up or down by the size of the",
   "                        data-item (goto previous/next item)",
   "",
   "   Alt  + 1  up to      Set mark at current cursor position",
   "   Alt  + 0             with length 1 upto 10 bytes",
   "",
   "   Alt  + A             Mark whole item (sector/cluster/block)",
   "",
   "   Alt  + B             Block/Byte mark, set begin or end of",
   "                        of a byte-size oriented marked area",
   "                        in the CURRENT item buffer.",
   "",
#if defined (WIN32)
   "   Ctrl + C",
#endif
#ifndef UNIX
   "   Ctrl + INSERT",
#endif
   "   Alt  + C             Copy whole item, or the marked area, ",
   "                        to the clipboard, in one of 3 formats:",
   "",
   "              HEX-dump  When there is NO marked area, or the mark",
   "                        extends into the left/right offset columns,",
   "                        the (marked part of) the CURRENT buffer is",
   "                        formatted as a HEX-DUMP with two header",
   "                        lines that describe the buffer contents.",
   "                        This is followed by one or more lines with:",
   "                        offset, HEX-pairs, ascii-area and absolute",
   "                        position, in the same row/column layout as",
   "                        HEX-editor window itself.",
   "                        Very useful for documentation purposes.",
   "",
   "              ASCII     When there IS a marked area in any buffer,",
   "                        and the cursor is in the ascii area, the",
   "                        range of marked bytes is copied as ASCII",
   "                        characters, to provide 'readable' text.",
   "                        WARNING: Unprintable characters replaced",
   "                                 by a DOT, do NOT paste back!",
   "",
   "              HEX-pairs When there IS a marked area in any buffer,",
   "                        and the cursor is in the HEX-pairs area,",
   "                        the range of marked bytes is copied as",
   "                        space separated HEX-pairs like '41 ' and",
   "                        can safely be pasted back, overlayed",
   "                        in the same item, or another one, or be",
   "                        used in a different application altogether",
   "",
   "   Alt  + H             Toggle type of the mark between byte-range",
   "                        and Hex-Dump, that includes offset columns",
   "                        Note: The type of the mark is relevant for",
   "                        copying to the clipboard only",
   "",
   "",
   "   Alt  + O             Overlay mark-data at current position",
   "   Alt  + G             Get/copy mark-data to current position",
   "                        (restricted to CURRENT buffer only)",
   "",
   "   Alt  + E             Mark from cursor to end of object",
   "   Alt  + J             Mark from begin of line to cursor",
   "   Alt  + K             Mark from cursor to end of line",
   "",
   "   Alt  + L             Line mark, set begin or end of a line",
   "                        oriented marked area in the CURRENT",
   "                        item buffer.",
   "",
   "   Alt  + M             Copy mark-data to current position",
   "                        then move the mark there as well.",
   "",
   "   Alt  + R             Reverse byte-order in marked area,",
   "                        useful for endian correction etc.",
   "",
   "   Alt  + S             Mark from begin of object to cursor",
   "",
   "   Alt  + U             Unmark, remove any byte or line mark",
   "",
   "   Ctrl + V             Paste/overlay clipboard data in CURRENT",
   "   Alt  + V             buffer, contents in one of two formats:",
   "",
   "              ASCII     Any string NOT being the HEX-pairs below.",
   "                        WARNING: don't simply paste back data that",
   "                        was copied from the ASCII area, it will",
   "                        have all unprintable characters replaced",
   "                        by a DOT, losing information!",
   "",
   "              HEX-pairs String with ONLY pairs of HEX digits with",
   "                        optional '0x' prefixes and using spaces,",
   "                        tabs, newlines or commas as separator",
   "",
   "",
   "   Alt  + A             Toggle size between 16x32 and initial",
   "",
   "   Ctrl + A             Find Again, repeat the last find with",
   "                        same search arguments (if available),",
   "                        without presenting the dialog again",
   "",
   "   Ctrl + B or          Go Back to the item and position it",
   "   Alt  + Home          was started with (initial sector)",
   "",
   "   Ctrl + D             Delete all bytes AFTER current one (*)",
   "   Alt  + D             (see DELETE/INSERT, file-editing only)",
   "",
   "   Ctrl + E             Erase/Fill whole item or marked area",
   "   Alt  + F             with specified HEX value, default 00",
   "",
   "   Ctrl + F             Find data and go to the found item.",
   "   Ctrl + T             Collapse edit-window, then Ctrl-F",
   "                        (search is application specific)",
   "",
   "   Ctrl + G             Go to other item, to be specified by",
   "                        user, using sector number or other",
   "                        selection. (application specific)",
   "",
   "   Ctrl + W             Write back the curent item (sector).",
   "                        Normally this is done automatically",
   "                        on moving to other items or at exit.",
#if defined (UNIX)
   "",
#else
   "   Ctrl + Z"
#endif
   "   Alt  + Z             Undo changes to CURRENT item (sector).",
   "                        This reverts the item to the state it",
   "                        had when loaded (from disk) last time.",
   "                        To correct mistakes or to allow moving",
   "                        to another item after a write failure.",
   "",
   "   Ctrl + arrow key     Shift the cursor within the visible",
   "                        window without changing the actual",
   "                        cursor position in the data item.",
   "                        Can be used to align certain bytes",
   "                        with the start of the line/window.",
   "",
   "   Alt  + LEFT/RIGHT    Decrease or increase the number of columns",
   "                        used in the data area, and switch off auto",
   "                        sizing to 8/16/32/64/128 column width on",
   "                        resizing the window manually.",
   "                        Number of columns can be 1 .. 256",
   "",
   "   Alt  + UP/DOWN       Decrease or increase the number of rows",
   "                        used in the data area, and switch off",
   "                        auto size to desktop window height when",
   "                        resizing the window manually.",
   "                        Number of rows can be 1 .. 256",
   "",
   "   Alt  + F10           Maximize the edit window, most columns",
   "",
   "   Alt  + T or          Return to initial size, and auto-resize",
   "   Ctrl + F2            or set 16 columns with 32 rows (toggle)",
   "",
   "   Valid HEX   key      Change bytes values when in the HEX area",
   "",
   "   Valid ASCII key      Change bytes values when in the ASCII area",
   "",
   "   BACKSPACE            Change byte value to 0x00, move LEFT, HEX area",
   "   BACKSPACE            Change byte value to 0x20, move LEFT, ASCII area",
   "",
   "   DELETE               Delete byte at the cursor position (*)",
   "   INSERT               Insert byte at the cursor position (*)",
   "",
   "     *                  Insert or Delete may be possible at any",
   "                        position, at the end of the object only,",
   "                        or not all, as defined by application.",
   "                        (Usually possible for binary file edit)",
   "",
   "   F1                   Show help on the Hex-edit control",
   "",
   "   F2                   Activate alternate-format viewer, like a",
   "                        disassembly module application-specific",
   "",
   "   F3, F4, Enter        End the Hex-edit dialog and write",
   "   Enter, Esc           back changes in the current item.",
   "",
   "   F8  (or TAB)         Switch edit-cursor between the HEX",
   "                        and ASCII data area. The TAB key",
   "                        may not be available inside a more",
   "                        complex dialog where it switches",
   "                        between multiple control windows.",
   "",
   "   F9                   Switch position indicators between",
   "                        the default hexadecimal and decimal",
   "                        representation",
   "",
   "",
   " Whenever moving the cursor causes a change from one item to the",
   " next or previous one, any changes to the current item will be",
   " written back. The application may provide a confirmation popup",
   " in that case to allow user intervention.",
   "",
   "",
   "                 Edit, HEX-editor, mouse usage",
   "                 =============================",
   "",
   " Click:",
   " ------",
   ""
   " The MOUSE can be used to set the cursor position, or to MOVE",
   " the cursor in the data area using the shown 'scroll buttons'",
   " that move the cursor up/dn by either a full Item size (block)",
   " or by the number of lines in the display (page).",
   "",
   " Clicking the TOP or BOTTOM 'button' lines outside the buttons",
   " will move the cursor UP or DOWN, clicking on the left (position)",
   " column moves the cursor LEFT and clicking ON the right ASCII",
   " area ']' column will move the cursor to the RIGHT ...",
   "",
   " Clicking on the left ASCII area '[' column it will SCROLL the data",
   " up or down, simulating a scrollbar. When clicking near the top or",
   " bottom of those, movement will be one line, when clicking near the",
   " center of the display, it will move by a whole page.",
   "",
   " Clicking INSIDE the HEX or ASCII data area will activate that pane",
   " (HEX or ASCII) and position the cursor at the byte/character under",
   " the mouse cursor. This can be in the current item (bright text) or",
   " in a previous or next one. Whatever item the cursor ends up in,",
   " will be made the CURRENT item.",
   "",
   "",
   " Click+Drag:"
   " -----------",
   "",
   " When clicking inside the HEX or ASCII area, and dragging the mouse,",
   " a MARKED-AREA is created that defines a RANGE of subsequent bytes.",
   " It can be FILLED, BYTE-REVERSED, COPIED/MOVED with the buffer, or",
   " copied-to/pasted-from the clipboard using several keys (see above)",
   " or mouse actions (see below)",
   "",
   " When the drag starts in, or extends into the OFFSET column, the type",
   " of the mark will be 'Hex-Dump' instead of 'byte-range' and the visible",
   " mark color will now include the offset area for the marked lines.",
   " The type of the mark is relevant for copying to the clipboard only",
   "",
   "",
   " Alt-Click (or Ctrl-Click mouse button 2):",
   " -----------------------------------------",
   "",
   " Sets the cursor to the click-location, then copies the marked-area",
   " (or whole item, when no mark present) to the clipboard. The format",
   " will be HEX-pairs when in the HEX-area, and an ascii-string filtered",
   " on non-printables in the ASCII-area (same as Alt-C/Ctrl-C).",
   "",
   "",
   " Ctrl-Click:",
   " -----------",
   "",
   " Sets the cursor to the click-location, or to the START of the MARKED",
   " area if there is one, hen PASTES the contents of the clipboard to that",
   " position (overlaying the existing data, no insertion of new data!)",
   " The maximum number of bytes pasted is from the start position to the",
   " end of the item (sector) or the end of the marked area, if any.",
   " The FORMAT of the clipboard data needs to be HEX-pairs, seperated by",
   " whitespace or commas and optional '0x' prefixes, or an ASCII string.",
   "",
   "",
   " Double-Click:",
   " -------------",
   ""
   " INSIDE the HEX or ASCII area, this UNMARKs the marked area (like Alt-U).",
   " When in the left or right OFFSET areas, it will toggle the mark-type",
   " between byte-range and Hex-dump format, the latter indicated with the",
   " mark-color extending into the offset columns.",
   ""
   "",
   " Once help is displayed, another <F1> will call up the help info",
   " about the application level dialog or generic hex-editor dialog",
   "",
   NULL
};

// 08 versions of position are valid for #cols from 4/8 up to 16
#define TXHE_MIN_COLS        4                  // room for Cursor only
#define TXHE_MED_COLS        8                  // room for Cursor+relSN
#define TXHE_STD_COLS       16                  // room for everything
#define TXHE_CUR_P08        13
#define TXHE_CUR_POS        18                  // relative cursor position
#define TXHE_ABS_POS        31                  // absolute cursor position
#define TXHE_RSN_P08        29                  // relative sector in buffer
#define TXHE_RSN_POS        53
#define TXHE_AC_COLUMN(c)   (((c) * 3) + 8)     // ASCII start column
#define TXHE_TC_016_COLS    76                  // Total width is 4 * cols + 12
#define TXHE_TC_032_COLS    140
#define TXHE_TC_064_COLS    268
#define TXHE_TC_128_COLS    524
#define TXHE_ABS_WIDTH      14
#define TXHE_HIDE_CURSOR    0

static char txwhe_footer[]  = "F1=Help F2=%s F4=OK F8=Hex/Ascii F9=decimal PgUp/PgDn=Page "
#if defined (UNIX)
                              "Alt-Z"
#else
                              "Ctrl-Z"
#endif
                                    "=UNDO changes";

static TXTM  statusMsg;


// Execute callback, but graceful fail when not available
#define txwHexEditCallBack(act,he) ((callback) ? (callback)(act,he) : TX_CMD_UNKNOWN)


// Prepare CRC and difference buffer handling for current data buffer
static void txtHexEditPrepareBuf
(
   TXHEXEDIT         *he                        // IN    HEXEDIT data structure
);

// Show or hide real and 'alternate' cursor in current location, use modify
static void txwHexEditShowCursor
(
   TXWHANDLE           hwnd,                    // IN    current window
   BOOL                show,                    // IN    show in cursor color
   BOOL                mod,                     // IN    buffer is modified
   TXHEXEDIT          *he                       // IN    HEXEDIT data structure
);


// Save the marked-area of CURR buffer to memory as an ASCII / HEX-pairs range
static ULONG txwSaveToMemHE                     // RET   nr of bytes copied
(
   TXHEXEDIT          *he,                      // IN    HEXEDIT data structure
   char              **heText                   // OUT   allocated text buffer
);                                              //       (needs FreeMem)

// Save the whole CURR buffer to memory as an HEXDUMP, including offsets etc
static ULONG txwHdump2MemHE                     // RET   nr of bytes copied
(
   TXHEXEDIT          *he,                      // IN    HEXEDIT data structure
   char              **heText                   // OUT   allocated text buffer
);                                              //       (needs FreeMem)

// Paste string of ASCII charcs or HEX-pairs into buffer, at cursor or mark
static ULONG txwPasteDataHE                     // RET   nr of bytes copied
(
   TXHEXEDIT          *he,                      // INOUT HEXEDIT data structure
   char               *heText                   // IN    allocated text buffer
);

// Reload data to change the position to one of the NEXT buffers
static BOOL txwHexEditNextBuf                   // RET   target buffer is SET
(
   TXHEXEDIT          *he,                      // INOUT HEXEDIT data structure
   LONG                target                   // IN    target offset, from CURR
);                                              //       a known POSITIVE value
                                                //       larger than CURR size!

// Reload data to change the position to one of the PREV buffers
static BOOL txwHexEditPrevBuf                   // RET   target buffer is SET
(
   TXHEXEDIT          *he,                      // INOUT HEXEDIT data structure
   LONG                target                   // IN    target offset, from CURR
);                                              //       a known NEGATIVE value!


/*****************************************************************************/
// Initialize the Hex-editor dialog/control data structures
/*****************************************************************************/
BOOL txwInitHexeditDialogs
(
   void
)
{
   BOOL                rc = TRUE;               // function return

   ENTER();

   txwRegisterHelpText( TXWD_HELPHEXED, "txhexed", "Hex Editor help",  hexedhelp);

   BRETURN (rc);
}                                               // end 'txwInitHexeditDialogs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display standard Hex editor dialog, supporting 8 or 16 byte wide display
/*****************************************************************************/
ULONG txwHexEditor
(
   TXWHANDLE           parent,                  // IN    parent window
   TXWHANDLE           owner,                   // IN    owner  window
   TXHEXEDIT          *hedat,                   // IN    hex edit data
   char               *title,                   // IN    title for the dialog
   ULONG               helpid                   // IN    help on hexedit
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(("parent:%8.8x  owner:%8.8x  title:'%s'\n",  parent, owner, title));
   TRACES(("helpid:%8.8x = %u  flags:%8.8x\n", helpid, helpid, hedat->wFlags));
   TRHEAP(1);

   if (txwIsWindow( TXHWND_DESKTOP) && (hedat->curr != NULL))
   {
      TXRECT           position;                // reference size/position
      TXWHANDLE        eframe;                  // editor frame
      TXWHANDLE        ehexed;                  // editor hexedit control
      TXWINDOW         window;                  // setup window data
      ULONG            style;
      short            phsize;                  // parent window width
      short            pvsize;                  // parent window height
      int              lines;                   // nr of lines in window
      int              width;                   // nr of columns
      TXTM             footer;

      txwQueryWindowRect( parent, FALSE, &position); // get nett size (left/top = 0!)
      phsize = position.right;
      pvsize = position.bottom;

      TRACES(("IN  rows: %3hu   cols:%3hu   phsize:%hd  pvsize:%hd\n",
            hedat->rows, hedat->cols,       phsize,     pvsize));

      if (hedat->rows == 0)                     // match to full-screen
      {
         hedat->rows = pvsize - 5;
      }
      lines = hedat->rows + 5;

      if ((hedat->autosize) || (hedat->cols == 0))
      {
         hedat->autosize = TRUE;
         hedat->cols = (phsize >= TXHE_TC_128_COLS) ? 128 :
                       (phsize >= TXHE_TC_064_COLS) ?  64 :
                       (phsize >= TXHE_TC_032_COLS) ?  32 :
                       (phsize >= TXHE_TC_016_COLS) ?  16 : 8;
      }
      hedat->initialRows = hedat->rows;         // used to restore to original size
      hedat->initialCols = hedat->cols;         // when autosize is FALSE
      TRACES(("SET rows: %3hu  cols:%3hu\n", hedat->rows, hedat->cols));

      while (hedat->posCursor >= (hedat->rows * hedat->cols))
      {
         hedat->posCursor -= hedat->cols;
         hedat->posCurBuf += hedat->cols;
      }

      width = hedat->cols * 4 + 12;             // required width
      if (width + TXHE_ABS_WIDTH <= phsize)
      {
         width += TXHE_ABS_WIDTH;               // add room for Abs column
      }

      if (position.left  + width < phsize)      // center-left, 1/5
      {
         position.left   = ((phsize - width -1) / 5);
      }
      if (position.top   + lines < pvsize)
      {
         position.top   += ((min( pvsize, 60) - lines) / 4);
      }
      TRECTA( "pos/size", (&position));

      style = TXWS_DIALOG | TXWS_DISABLED | TXCS_CLOSE_BUTTON;
      if (hedat->wFlags & TXHE_MOVEABLE)
      {
         style |= TXWS_MOVEABLE;                // make frame move/sizeable
      }
      sprintf( footer, txwhe_footer, hedat->altDispText);
      txwSetupWindowData(
         position.top,                          // upper left corner
         position.left,
         lines,                                 // vertical size
         width,                                 // horizontal size
         style | TXWS_CAST_SHADOW,              // window frame style
         (helpid) ? helpid : TXWD_HELPHEXED,
         ' ', ' ', TXWSCHEME_COLORS,
         title, footer,
         &window);
      window.st.buf     = NULL;                 // NO artwork attached
      window.dlgFocusID = TXWD_WID_HEXED;       // Field to get Focus
      eframe = txwCreateWindow( parent, TXW_FRAME, 0, 0, &window, NULL);

      style = TXWS_HEXECTL;
      if (hedat->wFlags & TXHE_MOVEABLE)
      {
         style |= TXWS_HCHILD_SIZE;             // resize with parent
         style |= TXWS_VCHILD_SIZE;
      }
      txwSetupWindowData(
         0, 0,                                  // upper left corner
         lines -2,                              // vertical size
         width -2,                              // horizontal size
         style,                                 // window frame style
         TXWD_HELPHECTL,                        // help on hexedit control
         ' ', ' ', TXWSCHEME_COLORS, "",  "",
         &window);
      window.he = *hedat;
      ehexed = txwCreateWindow( eframe, TXW_HEXEDIT, eframe, 0, &window, txwHexEditWinProc);
      txwSetWindowUShort( ehexed, TXQWS_ID, TXWD_WID_HEXED);

      rc = txwDlgBox( parent, owner, txwDefDlgProc, eframe, NULL);
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN (rc);
}                                               // end 'txwHexEditor'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Prepare CRC and difference buffer handling for current data buffer
/*****************************************************************************/
static void txtHexEditPrepareBuf
(
   TXHEXEDIT          *he                       // IN    HEXEDIT data structure
)
{
   ENTER();

   if (he->curr && he->curr->data && he->curr->size)
   {
      he->currCrc = TxCrc32( he->curr->data, he->curr->size);
      if (he->diffSize < he->curr->size)
      {
         TxFreeMem( he->diff);                  // free if too small
      }
      if (he->diff == NULL)                     // allocate when needed
      {
         he->diffSize = max( 512, he->curr->size); // minimum alloc to avoid
         he->diff     = TxAlloc( 1, he->diffSize); // frequent re-alloc
      }
      else
      {
         memset( he->diff, 0, he->diffSize);
      }
      if (he->diff != NULL)                     // make fresh diff copy
      {
         TRACES(( "Copy CURR to DIFF\n"));
         memcpy( he->diff, he->curr->data, he->curr->size);
      }
   }
   VRETURN();
}                                               // end 'txtHexEditPrepareBuf'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Hex Editor, paint window for the HEXED class
/*****************************************************************************/
void txwPaintHexEdit
(
   TXWINBASE          *wnd                      // IN    current window
)
{
   TXWHANDLE           hwnd  = (TXWHANDLE) wnd;
   TXWINDOW           *win   = wnd->window;
   TXCELL             *hexLn = NULL;            // Line of char+color Cells, HEX   area
   TXCELL             *ascLn = NULL;            // Line of char+color Cells, ASCII area
   TXLN                line;                    // line  buffer
   TXLN                value;                   // to be displayed
   TXHEXEDIT          *dat = &win->he;
   USHORT              brow;
   USHORT              bcol;
   LONG                bIndex;
   short               sy = win->client.bottom - win->client.top   +1;
   short               sx = win->client.right  - win->client.left  +1;
   short               ln = 0;                  // line-nr while painting
   short               cl;                      // column  while painting
   short               ac;                      // ascii  column start col
   short               tc;                      // tail column, after ascii
   BYTE                posColor;
   TXHEBUF            *hexBuf;
   LONG                offset;
   BOOL                modified = FALSE;
   int                 pad;
   char               *hl;
   char                ascro[2] = {0};          // ascii bracket / scroll-indicator

   ENTER()

   if (dat->autosize)
   {
      dat->rows =  sy - 3;                      // adapt to resizing
      dat->cols = (sx >= (TXHE_TC_128_COLS - 2)) ? 128 : // correct for border (2)
                  (sx >= (TXHE_TC_064_COLS - 2)) ?  64 :
                  (sx >= (TXHE_TC_032_COLS - 2)) ?  32 :
                  (sx >= (TXHE_TC_016_COLS - 2)) ?  16 : 8;
      TRACES(("Recalculated cols from sx: %hd to be: %hd, rows: %hd\n", sx, dat->cols, dat->rows));
   }
   ac = TXHE_AC_COLUMN(dat->cols);
   tc = ac + dat->cols + 2;

   //- Draw top line with the 'Up' buttons and bytenumbers
   txwStringAt( ln,         0, "[",             TxwSC( cHexEdButBracket));
   txwStringAt( ln,         1, "PgUp",          TxwSC( cHexEdButtonText));
   txwStringAt( ln,         5, "]",             TxwSC( cHexEdButBracket));

   strcpy( line, "  ");
   for (bcol = 0, cl = dat->posCurBuf & 0x0f; bcol < dat->cols; bcol++, cl++)
   {
      sprintf( value, (dat->decimal) ? "%2d " : "%2x ", (cl % dat->cols));
      strcat( line, value);
   }
   txwStringAt( ln,         6, line,            TxwSC( cHexEdByteNumber));

   cl = ac;
   txwStringAt( ln,        cl++,  "[",          TxwSC( cHexEdButBracket));
   if (dat->cols >= 5)
   {
      txwStringAt( ln,     cl,    "ItemUp",     TxwSC( cHexEdButtonText));
      cl += 6;
      txwStringAt( ln,     cl++,  "]",          TxwSC( cHexEdButBracket));
   }
   else
   {
      txwStringAt( ln,     cl,    "Up",         TxwSC( cHexEdButtonText));
      cl += 2;
      txwStringAt( ln,     cl++,  "]",          TxwSC( cHexEdButBracket));
   }
   if (dat->cols >= 16)
   {
      txwStringAt( ln,     cl,    "  ",         TxwSC( cHexEdByteNumber));
      cl += 2;
      txwStringAt( ln,     cl++,  "[",          TxwSC( cHexEdButBracket));
      txwStringAt( ln,     cl,    "LineUp",     TxwSC( cHexEdButtonText));
      cl += 6;
      txwStringAt( ln,     cl++,  "]",          TxwSC( cHexEdButBracket));
   }
   if (tc > cl)
   {
      sprintf( line, "%*.*s", (tc - cl), (tc - cl), "");
      txwStringAt( ln,     cl,    line,         TxwSC( cHexEdByteNumber));
   }

   if (sx > tc)
   {
      if ((sx-tc) >= TXHE_ABS_WIDTH)            // add absolute byte positions
      {
         sprintf( line, " Abs byte Pos%*.*s", (sx-tc-13), (sx-tc-13), "");
         txwStringAt( ln,  tc,    line,         TxwSC( cHexEdByteNumber));
      }
      else
      {
         sprintf( line, "%*.*s", (sx-tc), (sx-tc), "");
         txwStringAt( ln,  tc,    line,         TxwSC( cHexEdByteNumber));
      }
   }
   if ((dat->curr) && (dat->curr->data))        // do we have any data ?
   {
      modified = (dat->currCrc != TxCrc32( dat->curr->data, dat->curr->size));
   }
   TRACES(("Offset:%16.16llx pre4->size:%4u desc:%s\n", dat->pre4->start, dat->pre4->size, dat->pre4->desc));
   TRACES(("Offset:%16.16llx pre3->size:%4u desc:%s\n", dat->pre3->start, dat->pre3->size, dat->pre3->desc));
   TRACES(("Offset:%16.16llx pre2->size:%4u desc:%s\n", dat->pre2->start, dat->pre2->size, dat->pre2->desc));
   TRACES(("Offset:%16.16llx prev->size:%4u desc:%s\n", dat->prev->start, dat->prev->size, dat->prev->desc));
   TRACES(("Offset:%16.16llx curr->size:%4u desc:%s\n", dat->curr->start, dat->curr->size, dat->curr->desc));
   TRACES(("Offset:%16.16llx next->size:%4u desc:%s\n", dat->next->start, dat->next->size, dat->next->desc));
   TRACES(("Offset:%16.16llx nex2->size:%4u desc:%s\n", dat->nex2->start, dat->nex2->size, dat->nex2->desc));
   TRACES(("Offset:%16.16llx nex3->size:%4u desc:%s\n", dat->nex3->start, dat->nex3->size, dat->nex3->desc));
   TRACES(("Offset:%16.16llx nex4->size:%4u desc:%s\n", dat->nex4->start, dat->nex4->size, dat->nex4->desc));

   if (((hexLn = (TxAlloc( dat->cols, 3 * sizeof( TXCELL)))) != NULL) && // 3 cells per HEX-pair
       ((ascLn = (TxAlloc( dat->cols, 1 * sizeof( TXCELL)))) != NULL)  ) // 1 cell  per ASCII char
   {
      for (ln = 1, brow = 0; dat->curr && (brow < dat->rows); brow++, ln++)
      {
         BOOL          lineHasMark = FALSE;     // to be calculated, this line has at least one marked byte

         bIndex = brow * dat->cols;
         if (dat->decimal)
         {
            sprintf( line, (dat->posCurBuf + bIndex < 0) ? "%5.5d  " : "%6.6d  ", dat->posCurBuf + bIndex);
         }
         else
         {
            sprintf( line, "%6.6X ", (dat->posCurBuf + bIndex) & 0xffffff);
         }
         offset = dat->posCurBuf + bIndex;
         if (offset < 0)                        // line is in PREV buffer
         {
            posColor = TxwSC( cHexEdRelPosPrev);
         }
         else if (offset < dat->curr->size)     // line is in CURR buffer
         {
            if ((dat->markSize) && (dat->markDump)) // is it a hexDump type mark? (include offset/position)
            {
               hexBuf = dat->curr;              // we ARE in current buffer

               for (bcol = 0; bcol < dat->cols; bcol++) // use existing logic from byte-painting (works :)
               {
                  offset = dat->posCurBuf + bIndex + bcol; // offset for THIS byte

                  if (((hexBuf->start + offset) >= (dat->markBase + dat->markStart)) &&
                      ((hexBuf->start + offset) <  (dat->markBase + dat->markStart + dat->markSize)))
                  {
                     lineHasMark = TRUE;        // 'inmark' = at least on byte within the mark
                  }
               }
            }
            if (lineHasMark)
            {
               posColor = TxwSC( c1MarkedActiveOrgStd);
            }
            else
            {
               posColor = TxwSC( cHexEdRelPosCurr);
            }
         }
         else                                   // line is in NEXT buffer
         {
            posColor = TxwSC( cHexEdRelPosNext);
         }
         txwStringAt( ln, 0, line, posColor);
         txwStringAt( ln, 7, " ",  TxwSC( c1NomarkActiveOrgStd));

         for (bcol = 0; bcol < dat->cols; bcol++)
         {
            BYTE       hexColor = c1NomarkDimmedOrgStd;   //- default color for the current HEX-pair
            BYTE       ascColor = c2NomarkDimmedOrgStd;   //- default color for the urrent ASCII char
            TXCELL    *cell = &( hexLn[ bcol * 3]);       //- 3 cells per BYTE in HEX area

            offset = dat->posCurBuf + bIndex + bcol;

            TRLEVX(700,("ln: %d, initial offset: %d curr->size: %u\n", ln, offset, dat->curr->size));

            if (offset < 0)                     // determine buffer for this line, and offset to the BYTE
            {
               if (!dat->prev  ||  !dat->prev->data  ||
                   (offset + (LONG) dat->prev->size) < 0)
               {
                  if (!dat->pre2  ||  !dat->pre2->data  ||
                      (offset + (LONG) dat->pre2->size + (LONG) dat->prev->size) < 0)
                  {
                     if (!dat->pre3  ||  !dat->pre3->data  ||
                         (offset + (LONG) dat->pre3->size + (LONG) dat->pre2->size
                                                          + (LONG) dat->prev->size) < 0)
                     {
                        if (!dat->pre4  ||  !dat->pre4->data  ||
                            (offset + (LONG) dat->pre4->size + (LONG) dat->pre3->size
                                                             + (LONG) dat->pre2->size
                                                             + (LONG) dat->prev->size) < 0)
                        {
                           hexBuf = NULL;       // byte is before PRE4 (draw --)
                        }
                        else
                        {
                           hexBuf  = dat->pre4;       // byte is in PRE4 buffer
                           offset += dat->pre4->size;
                           offset += dat->pre3->size;
                           offset += dat->pre2->size;
                           offset += dat->prev->size;
                        }
                     }
                     else
                     {
                        hexBuf  = dat->pre3;       // byte is in PRE3 buffer
                        offset += dat->pre3->size;
                        offset += dat->pre2->size;
                        offset += dat->prev->size;
                     }
                  }
                  else
                  {
                     hexBuf  = dat->pre2;       // byte is in PRE2 buffer
                     offset += dat->pre2->size;
                     offset += dat->prev->size;
                  }
               }
               else
               {
                  hexBuf  = dat->prev;          // byte is in PREV buffer
                  offset += dat->prev->size;
               }
            }
            else if (offset < dat->curr->size)
            {
               hexBuf   = dat->curr;            // byte is in CURR buffer, ACTIVE
               hexColor = c1NomarkActiveOrgStd;
               ascColor = c2NomarkActiveOrgStd; // update the default colors
            }
            else if (dat->next && dat->next->data &&
                     (offset < (dat->curr->size + dat->next->size)))
            {
               hexBuf  = dat->next;             // byte is in NEXT buffer
               offset -= dat->curr->size;
            }
            else
            {
               if (dat->nex2 && dat->nex2->data &&
                   (offset < (dat->curr->size + dat->next->size
                                              + dat->nex2->size)))
               {
                  hexBuf  = dat->nex2;          // byte is in NEX2 buffer
                  offset -= dat->next->size;
                  offset -= dat->curr->size;
               }
               else
               {
                  if (dat->nex3 && dat->nex3->data &&
                      (offset < (dat->curr->size + dat->next->size
                                                 + dat->nex2->size
                                                 + dat->nex3->size)))
                  {
                     hexBuf  = dat->nex3;       // byte is in NEX3 buffer
                     offset -= dat->nex2->size;
                     offset -= dat->next->size;
                     offset -= dat->curr->size;
                  }
                  else
                  {
                     if (dat->nex4 && dat->nex4->data &&
                         (offset < (dat->curr->size + dat->next->size
                                                    + dat->nex2->size
                                                    + dat->nex3->size
                                                    + dat->nex4->size)))
                     {
                        hexBuf  = dat->nex4;       // byte is in NEX4 buffer
                        offset -= dat->nex3->size;
                        offset -= dat->nex2->size;
                        offset -= dat->next->size;
                        offset -= dat->curr->size;
                     }
                     else
                     {
                        hexBuf  = NULL;         // byte is beyond NEX4 (draw --)
                     }
                  }
               }
            }

            //- Now determine the desired color for this byte, when not default (dimmed/active Nomark+Org+Std
            if (hexBuf && hexBuf->data && (modified || dat->markSize)) // may need non-default colors
            {
               BOOL          bmodif = FALSE;    // cursor pos is modified byte
               BOOL          inmark = FALSE;    // cursor pos is inside mark

               if (modified && dat->diff && (hexBuf == dat->curr))
               {
                  bmodif = (hexBuf->data[ offset] != dat->diff[ offset]);
               }
               if (dat->markSize)
               {
                  inmark = (((hexBuf->start + offset) >= (dat->markBase + dat->markStart)) &&
                            ((hexBuf->start + offset) <  (dat->markBase + dat->markStart + dat->markSize)));
               }
               TRLEVX(700,("bmodif: %s   inmark: %s\n", (bmodif) ? "YES" : "NO ", (inmark) ? "YES" : "NO "));

               if (bmodif)
               {
                  if (inmark)
                  {
                     hexColor = c1MarkedActiveModStd;
                     ascColor = c2MarkedActiveModStd;
                  }
                  else
                  {
                     hexColor = c1NomarkActiveModStd;
                     ascColor = c2NomarkActiveModStd;
                  }
               }
               else                             // unmodified byte
               {
                  if (inmark)
                  {
                     if (hexBuf == dat->curr)   // ACTIVE buffer
                     {
                        hexColor = c1MarkedActiveOrgStd;
                        ascColor = c2MarkedActiveOrgStd;
                     }
                     else
                     {
                        hexColor = c1MarkedDimmedOrgStd;
                        ascColor = c2MarkedDimmedOrgStd;
                     }
                  }
                  else
                  {
                     if (hexBuf == dat->curr)   // ACTIVE buffer
                     {
                        hexColor = c1NomarkActiveOrgStd;
                        ascColor = c2NomarkActiveOrgStd;
                     }
                     else
                     {
                        hexColor = c1NomarkDimmedOrgStd;
                        ascColor = c2NomarkDimmedOrgStd;
                     }
                  }
               }
            }

            TRLEVX(700,("ln: %d, hexBuf->data: %p  Buffer offset: %d\n", ln, (hexBuf) ? hexBuf->data : NULL, offset));

            if (hexBuf && hexBuf->data && (offset >= 0) && (offset < hexBuf->size))
            {
               BYTE    data = hexBuf->data[ offset];

               cell->ch  = (data >> 4);
               cell->ch += (cell->ch < 10) ? '0' : ('a' - 10); // convert upper nibble to ASCII
               cell->at  = TxwSC( hexColor);

               cell++;
               cell->ch  = (data & 0x0f);
               cell->ch += (cell->ch < 10) ? '0' : ('a' - 10); // convert lower nibble to ASCII
               cell->at  = TxwSC( hexColor);

               ascLn[ bcol].ch = TxPrintable( data);
            }
            else
            {
               cell->ch  = '-';
               cell->at  = TxwSC( hexColor);

               cell++;
               cell->ch  = '-';
               cell->at  = TxwSC( hexColor);

               ascLn[ bcol].ch = ' ';
            }
            cell++;
            cell->ch = ' ';
            cell->at = TxwSC( hexColor);

            ascLn[ bcol].at = TxwSC( ascColor);
         }
         txwDrawCellString( hwnd, ln, 8, hexLn, dat->cols * 3); // print HEX-area in all its colors

         if      (brow == 0)                    // add scroll-indicators
         {
            ascro[0] = '-';
         }
         else if (brow < (dat->rows -1))
         {
            ascro[0] = '[';
         }
         else
         {
            ascro[0] = '+';
         }
         txwStringAt( ln, ac, ascro,  TxwSC( cHexEdAscBracket));
         txwDrawCellString( hwnd, ln, ac + 1, ascLn, dat->cols); // print ASCII-area in all its colors
         txwStringAt( ln, tc -1, "]", TxwSC( cHexEdAscBracket));

         if (sx > tc)
         {
            if (lineHasMark)
            {
               posColor = TxwSC( c1MarkedActiveOrgStd);
            }
            else
            {
               posColor = TxwSC( cHexEdAbsBytePos);
            }
            if ((sx-tc) >= TXHE_ABS_WIDTH)      // add absolute byte positions
            {
               sprintf( line, (dat->decimal) ? " %13.13llu" : " 0x%11.11llX",
                               dat->posCurBuf + bIndex + dat->curr->start);
               txwStringAt( ln, tc, line, posColor);
               sprintf( line, "%*.*s", (sx-tc-TXHE_ABS_WIDTH),
                                       (sx-tc-TXHE_ABS_WIDTH), "");
               txwStringAt( ln, tc + TXHE_ABS_WIDTH, line, posColor);
            }
            else                                // just fill to end
            {
               sprintf( line, "%*.*s", (sx-tc), (sx-tc), "");
               txwStringAt( ln, tc, line, TxwSC( cHexEdAbsBytePos));
            }
         }
      }
      TxFreeMem( hexLn);                        // free the CELL line memory
      TxFreeMem( ascLn);
   }
   else
   {
      TRACES(("Error allocating HEX-CELL-line memory\n"));
   }

   ln = dat->rows + 1;                          // first line after hex data
   txwStringAt( ln, 0, "[",      TxwSC( cHexEdButBracket));
   txwStringAt( ln, 1, "PgDn",   TxwSC( cHexEdButtonText));
   txwStringAt( ln, 5, "]",      TxwSC( cHexEdButBracket));

   sprintf( line, "%s ", (modified) ? " Mod " : "     ");
   txwStringAt( ln,     6,    line,          TxwSC( cHexEdModifyText));

   if      (dat->cols < 4)
   {
      pad = (dat->cols - 1) * 3 +1;
      sprintf( line, "%*.*s", pad, pad , "");
      txwStringAt( ln, 11, line, TxwSC( cHexEdByteNumber));
   }
   else if (dat->cols < 8)
   {
      txwStringAt( ln, 11, "C:", TxwSC( cHexEdByteNumber));
      pad = (dat->cols - 4) * 3 + 1;
      sprintf( line, "%*.*s", pad, pad , "");
      txwStringAt( ln, 19, line, TxwSC( cHexEdByteNumber));
   }
   else
   {
      if (dat->cols < 16)
      {
         txwStringAt( ln, 11,    "C:",     TxwSC( cHexEdByteNumber));
         txwStringAt( ln, 20, "   relSN:", TxwSC( cHexEdByteNumber));
         pad = (dat->cols - 8) * 3 + 1;
         sprintf( line, "%*.*s", pad, pad , "");
         txwStringAt( ln, 31,    line,     TxwSC( cHexEdByteNumber));
      }
      else
      {
         txwStringAt( ln, 11,   "Cursor:", TxwSC( cHexEdByteNumber));
         txwStringAt( ln, 24, " = Abs:",   TxwSC( cHexEdByteNumber));
         txwStringAt( ln, 44, "   relSN:", TxwSC( cHexEdByteNumber));
         pad = (dat->cols - 16) * 3 + 2;
         sprintf( line, "%*.*s", pad, pad , "");
         txwStringAt( ln, 54,    line,     TxwSC( cHexEdByteNumber));
      }
   }

   cl = ac;
   txwStringAt( ln,        cl++, "[",      TxwSC( cHexEdButBracket));
   if (dat->cols >= 5)
   {
      txwStringAt( ln,     cl,   "ItemDn", TxwSC( cHexEdButtonText));
      cl += 6;
      txwStringAt( ln,     cl++, "]",      TxwSC( cHexEdButBracket));
   }
   else
   {
      txwStringAt( ln,     cl,   "Dn",     TxwSC( cHexEdButtonText));
      cl += 2;
      txwStringAt( ln,     cl++, "]",      TxwSC( cHexEdButBracket));
   }
   if (dat->cols >= 16)
   {
      txwStringAt( ln,     cl,   "  ",     TxwSC( cHexEdByteNumber));
      cl += 2;
      txwStringAt( ln,     cl++, "[",      TxwSC( cHexEdButBracket));
      txwStringAt( ln,     cl,   "LineDn", TxwSC( cHexEdButtonText));
      cl += 6;
      txwStringAt( ln,     cl++, "]",      TxwSC( cHexEdButBracket));
   }
   if (tc > cl)
   {
      sprintf( line, "%*.*s", (tc - cl), (tc - cl), "");
      txwStringAt( ln,     cl,    line,    TxwSC( cHexEdByteNumber));
   }
   if (sx > tc)
   {
      sprintf( line, "%*.*s", (sx-tc), (sx-tc), "");
      txwStringAt( ln,     tc,    line,    TxwSC( cHexEdByteNumber));
   }

   if (dat->curr)
   {
      sprintf( line, "%-*.*s", sx, sx, dat->curr->desc);
      txwStringAt( ln +1,      0, line,    TxwSC( cHexEdItemSnText));

      if ((hl = strstr( line, ":0")) != NULL)   // support at most two
      {                                         // highlighted numbers
         char            *sp;                   // starting with ':0'

         TxCopy( value, hl +1, TXMAXLN);
         if ((sp = strchr( value, ' ')) != NULL)
         {
            *sp = 0;                            // terminate ':0' at 1st space
         }
         txwStringAt( ln +1, hl - line +1, value,  TxwSC( cHexEdItemHlight));

         if ((hl = strstr( hl +1, ":0")) != NULL)
         {
            TxCopy( value, hl +1, TXMAXLN);
            if ((sp = strchr( value, ' ')) != NULL)
            {
               *sp = 0;                         // terminate ':0' at 1st space
            }
            txwStringAt( ln +1, hl - line +1, value, TxwSC( cHexEdItemHlight));
         }
      }
   }
   txwHexEditShowCursor( hwnd, TRUE, modified, dat);
   VRETURN();
}                                               // end 'txwPaintHexEdit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show or hide real and 'alternate' cursor in current location, use modify
/*****************************************************************************/
static void txwHexEditShowCursor
(
   TXWHANDLE           hwnd,                    // IN    current window
   BOOL                show,                    // IN    show in cursor color
   BOOL                mod,                     // IN    buffer is modified
   TXHEXEDIT          *he                       // IN    HEXEDIT data structure
)
{
   USHORT              bcol  = he->posCursor % he->cols;
   USHORT              row   = he->posCursor / he->cols +1;
   short               ac    = TXHE_AC_COLUMN( he->cols);
   BOOL                bmod  = FALSE;           // current byte is modified
   BOOL                mark  = FALSE;           // current byte not in MARK
   LONG                index = he->posCurBuf + he->posCursor;
   TXTS                value;
   BYTE                actHex;                  // color for active   Hex   PANE
   BYTE                dimHex;                  // color for inactive Hex   pane
   BYTE                actAsc;                  // color for active   Ascii PANE
   BYTE                dimAsc;                  // color for inactive Ascii pane

   if ((he->rows > 0) && ((index < he->curr->size) || (he->curr->size == 0)))
   {
      BYTE             cbyte = 0;

      if (he->curr->data)
      {
         cbyte = he->curr->data[ index];
      }

      if ((mod) && (he->diff))                  // buffer has modified bytes
      {
         bmod = (cbyte != he->diff[ index]);
      }
      if ((he->markSize) && (he->markBase == he->curr->start))
      {                                         // we have a marked area in CURR item
         mark = ((index >= he->markStart) && (index < (he->markStart + he->markSize)));
      }

      //- paint cursor-byte in ASCII area
      actAsc = (show) ? (mark) ? (bmod) ? c2MarkedActiveModCur : c2MarkedActiveOrgCur
                               : (bmod) ? c2NomarkActiveModCur : c2NomarkActiveOrgCur
                      : (mark) ? (bmod) ? c2MarkedActiveModStd : c2MarkedActiveOrgStd
                               : (bmod) ? c2NomarkActiveModStd : c2NomarkActiveOrgStd;
      dimAsc = (show) ? (mark) ? (bmod) ? c2MarkedDimmedModCur : c2MarkedDimmedOrgCur
                               : (bmod) ? c2NomarkDimmedModCur : c2NomarkDimmedOrgCur
                      : (mark) ? (bmod) ? c2MarkedActiveModStd : c2MarkedActiveOrgStd
                               : (bmod) ? c2NomarkActiveModStd : c2NomarkActiveOrgStd;

      sprintf( value, "%c", TxPrintable( cbyte));
      txwStringAt( row, ac + 1 + bcol, value, (he->ascii) ? TxwSC( actAsc) : TxwSC( dimAsc));

      //- paint cursor-byte in HEX area
      actHex = (show) ? (mark) ? (bmod) ? c1MarkedActiveModCur : c1MarkedActiveOrgCur
                               : (bmod) ? c1NomarkActiveModCur : c1NomarkActiveOrgCur
                      : (mark) ? (bmod) ? c1MarkedActiveModStd : c1MarkedActiveOrgStd
                               : (bmod) ? c1NomarkActiveModStd : c1NomarkActiveOrgStd;
      dimHex = (show) ? (mark) ? (bmod) ? c1MarkedDimmedModCur : c1MarkedDimmedOrgCur
                               : (bmod) ? c1NomarkDimmedModCur : c1NomarkDimmedOrgCur
                      : (mark) ? (bmod) ? c1MarkedActiveModStd : c1MarkedActiveOrgStd
                               : (bmod) ? c1NomarkActiveModStd : c1NomarkActiveOrgStd;

      sprintf( value, "%2.2hhx", cbyte);
      txwStringAt( row, 8 + bcol * 3, value,  (he->ascii) ? TxwSC( dimHex) : TxwSC( actHex));

      if (show)                                 // setting new position
      {
         ULONG         cursorpos = 0;           // default to 0, for EMPTY files
         ULONG         absolute  = 0;
         USHORT        rcol = (he->ascii) ? bcol + ac +1 : bcol * 3 + 8 + he->hexLsb;

         if (he->curr->size > 0)                // editable area, with cursor ?
         {
            cursorpos = (he->posCurBuf + he->posCursor) & 0xffffff;
            absolute  = (he->posCurBuf + he->posCursor) + he->curr->start;
            txwSetCursorPos( hwnd, row, rcol);  // set normal (real) cursor
         }

         if (he->cols >= TXHE_MIN_COLS)         // update status line numbers when needed
         {
            sprintf( value, (he->decimal) ? "%6.6u" : "%6.6X", cursorpos);
            txwStringAt( he->rows +1, (he->cols < TXHE_STD_COLS) ? TXHE_CUR_P08 : TXHE_CUR_POS,
                                                    value, TxwSC( cHexEdRelCursorP));

            if (he->cols >= TXHE_STD_COLS)      // absolute cursor position
            {
               sprintf( value, (he->decimal) ? "%13.13u " : "0x%11.11X ", absolute);
               txwStringAt( he->rows +1, TXHE_ABS_POS, value, TxwSC( cHexEdAbsByteCur));
            }

            if ((he->cols >= TXHE_MED_COLS) &&  // update Rsect value when enough room
                (he->curr->bps != 0))
            {
               sprintf( value, (he->decimal) ? "%2.2u" : "%2.2X", (cursorpos / he->curr->bps));
               txwStringAt( he->rows +1, (he->cols < TXHE_STD_COLS) ? TXHE_RSN_P08 : TXHE_RSN_POS,
                                                       value, TxwSC( cHexEdRelCursorP));
            }
         }
      }
   }
}                                               // end 'txwHexEditShowCursor'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Window procedure for the HEXED window class, handling mouse and keyboard
// Called as TOP-LEVEL winproc! Handles (most) class specific stuff only
/*****************************************************************************/
ULONG txwHexEditWinProc                         // RET   result
(
   TXWHANDLE           hwnd,                    // IN    current window
   ULONG               msg,                     // IN    message id
   TXWMPARAM           mp1,                     // IN    param 1
   TXWMPARAM           mp2                      // IN    param 2
)
{
   ULONG               rc  = NO_ERROR;          // WinProc return code
   ULONG               wr  = NO_ERROR;          // write-back return code
   TXWINBASE          *wnd = (TXWINBASE *) hwnd;
   TXWINDOW           *win;
   BOOL                upd = FALSE;             // window draw required
   BYTE                tmp;

   ENTER();
   TRCMSG( hwnd, msg, mp1, mp2);

   if ((wnd != NULL) && ((win = wnd->window) != NULL))
   {
      TXWHANDLE        parent   = (TXWHANDLE) wnd->parent;
      TXWHANDLE        owner    = (TXWHANDLE) wnd->owner;
      short            sy       = win->client.bottom - win->client.top  +1;
      TXHEXEDIT       *dat      = &win->he;
      TXHE_CALLBACK    callback = (TXHE_CALLBACK)  dat->actionCallback;
      LONG             offset   = dat->posCurBuf + dat->posCursor;
      LONG             pbytes   = + offset;     // max bytes to go backward (prev)
      LONG             nbytes   = - offset;     // max bytes to go forward  (next)
      LONG             wbytes   = dat->rows * dat->cols;
      BOOL             modified = FALSE;
      LONG             value;
      TXTS             str;
      short            delta_x;
      short            delta_y;
      ULONG            key = (ULONG) mp2;

      TRACES(( "sy:%hu  offset:%d  wbytes:%d  currsize:%u\n",
                sy,     offset,    wbytes, dat->curr->size));

      if ((dat->curr) && (dat->curr->data))
      {
         modified = (dat->currCrc != TxCrc32( dat->curr->data, dat->curr->size));

                                nbytes += dat->curr->size;
         if (dat->next != NULL) nbytes += dat->next->size;
         if (dat->nex2 != NULL) nbytes += dat->nex2->size;
         if (dat->nex3 != NULL) nbytes += dat->nex3->size;
         if (dat->nex4 != NULL) nbytes += dat->nex4->size;

         if (dat->prev != NULL) pbytes += dat->prev->size;
         if (dat->pre2 != NULL) pbytes += dat->pre2->size;
         if (dat->pre3 != NULL) pbytes += dat->pre3->size;
         if (dat->pre4 != NULL) pbytes += dat->pre4->size;
      }
      switch (msg)
      {
         case TXWM_CREATE:                      // hexed control just created
            wr = txwHexEditCallBack( TXHE_CB_INIT_POS, dat);
            txtHexEditPrepareBuf(dat);          // reset modify status & diff
            upd = TRUE;                         // even if no data at all :-)
            if (dat->wFlags & TXHE_MAXIMIZE)    // auto maximize HEX window
            {
               txwPostMsg( hwnd, TXWM_CHAR, mp1, (TXWMPARAM) TXa_F10); // maximize to parent
            }
            if (dat->altView != 0)
            {
               txwPostMsg( hwnd, TXWM_CHAR, mp1, (TXWMPARAM) TXa_F2); // use alternate view
            }
            break;

         case TXWM_CHAR:                        // first handle keys that are
            TRACES(( "key: %8.8x\n", key));     // safe to handle without data
            switch (key)
            {
               case TXk_F2:                     // switch to alt display fmt
               case TXa_F2:                     // switch to alt display fmt
               case TXc_A:                      // find again
               case TXc_F:                      // find data, cursor/buffer
               case TXc_T:
               case TXc_G:                      // goto specified location
                  if (modified)                 // pos to be set by callback!
                  {
                     wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                  }
                  if (wr == NO_ERROR)
                  {
                     switch (key)
                     {
                        case TXc_T:             // find, collapse edit window
                           txwPostMsg( owner,  TXWM_CHAR, mp1, (TXWMPARAM) TXk_F12);
                           txwHexEditCallBack( TXHE_CB_FINDDATA, dat);
                           txwPostMsg( owner,  TXWM_CHAR, mp1, (TXWMPARAM) TXk_F12);
                           break;

                        case TXc_A:             // find again
                        case TXc_F:             // find, keep window up
                           txwHexEditCallBack( (key == TXc_F) ?
                                               TXHE_CB_FINDDATA :
                                               TXHE_CB_FNDAGAIN, dat);
                           break;

                        case TXc_G:
                           txwHexEditCallBack( TXHE_CB_GOTOITEM, dat);
                           break;

                        default:
                           if (txwHexEditCallBack( TXHE_CB_ALT_DISP, dat) == TXDID_EXIT)
                           {
                              txwPostMsg( hwnd, TXWM_CLOSE, 0, 0); // close editor
                           }
                           break;
                     }
                     txtHexEditPrepareBuf( dat);
                     upd = TRUE;
                  }
                  break;

               case TXk_F8:                     // toggle hex/ascii area
               case TXk_TAB:                    // TAB may be unavailable when
                  dat->ascii   = !dat->ascii;   // multiple windows in dialog
                  upd = TRUE;
                  break;

               case TXk_F9:                     // toggle decimal/hex position
                  dat->decimal = !dat->decimal;
                  txwHexEditCallBack( TXHE_CB_UPD_DESC, dat);
                  upd = TRUE;
                  break;

               case TXc_HOME:
                  if (modified && dat->prev && dat->prev->data)
                  {
                     wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                  }
                  if (wr == NO_ERROR)
                  {
                     if (dat->prev && dat->prev->data)
                     {
                        txwHexEditCallBack( TXHE_CB_TO_START, dat);
                        txtHexEditPrepareBuf( dat);
                     }
                     dat->posCursor = 0;
                     dat->posCurBuf = 0;
                     upd = TRUE;
                  }
                  break;

               case TXc_END:
                  if (modified && dat->next && dat->next->data)
                  {
                     wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                  }
                  if (wr == NO_ERROR)
                  {
                     if (dat->next && dat->next->data)
                     {
                        txwHexEditCallBack( TXHE_CB_TO_FINAL, dat);
                        txtHexEditPrepareBuf( dat);
                     }
                     if (dat->curr->size <= wbytes)
                     {
                        dat->posCurBuf = 0;
                        dat->posCursor = dat->curr->size -1;
                     }
                     else
                     {
                        dat->posCurBuf = dat->curr->size - wbytes;
                        dat->posCursor = wbytes -1;
                     }
                     upd = TRUE;
                  }
                  break;

               case TXc_B:                      // back to begin ...
               case TXa_HOME:                   // INITIAL sector/pos
                  if (modified)
                  {
                     wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                  }
                  wr = txwHexEditCallBack( TXHE_CB_INIT_POS, dat);
                  if (wr == NO_ERROR)
                  {
                     txtHexEditPrepareBuf(dat); // reset modify status & diff
                     upd = TRUE;
                  }
                  break;

               case TXk_HOME:
                  dat->posCursor = 0;
                  dat->posCurBuf = 0;
                  upd = TRUE;
                  break;

               case TXk_END:
                  if (dat->curr->size <= wbytes)
                  {
                     dat->posCurBuf = 0;
                     dat->posCursor = dat->curr->size -1;
                  }
                  else
                  {
                     dat->posCurBuf = dat->curr->size - wbytes;
                     dat->posCursor = wbytes -1;
                  }
                  upd = TRUE;
                  break;

               case TXa_LEFT:
                  if (dat->cols > 1)
                  {
                     dat->cols--;
                     dat->autosize = FALSE;     // no auto change on window resize
                     if (dat->posCursor >= dat->rows * dat->cols)
                     {
                        dat->posCurBuf += dat->rows; // move view UP
                        dat->posCursor -= dat->rows; // to keep cursor visible
                     }
                     //- resize the parent (frame arround the actual HEXEDIT control) accordingly
                     delta_x = -4;
                     txwSetWindowPos( parent,  0,  0,  0,  delta_x, 0, TXSWP_SIZE | TXSWP_RELATIVE);
                     upd = TRUE;
                  }
                  break;

               case TXa_RIGHT:
                  if (dat->cols < 256)
                  {
                     dat->cols++;
                     dat->autosize = FALSE;     // no auto change on window resize

                     //- resize the parent (frame arround the actual HEXEDIT control) accordingly
                     delta_x = +4;
                     txwSetWindowPos( parent,  0,  0,  0,  delta_x, 0, TXSWP_SIZE | TXSWP_RELATIVE);
                     upd = TRUE;
                  }
                  break;

               case TXa_UP:
                  if (dat->rows > 1)
                  {
                     dat->rows--;
                     dat->autosize = FALSE;     // no auto change on window resize
                     if (dat->posCursor >= dat->rows * dat->cols)
                     {
                        dat->posCurBuf += dat->rows; // move view UP
                        dat->posCursor -= dat->rows; // to keep cursor visible
                     }
                     //- resize the parent (frame arround the actual HEXEDIT control) accordingly
                     delta_y = -1;
                     txwSetWindowPos( parent,  0,  0,  0,  0, delta_y, TXSWP_SIZE | TXSWP_RELATIVE);
                     upd = TRUE;
                  }
                  break;

               case TXa_DOWN:
                  if (dat->rows < 256)
                  {
                     dat->rows++;
                     dat->autosize = FALSE;     // no auto change on window resize

                     //- resize the parent (frame arround the actual HEXEDIT control) accordingly
                     delta_y = +1;
                     txwSetWindowPos( parent,  0,  0,  0,  0, delta_y, TXSWP_SIZE | TXSWP_RELATIVE);
                     upd = TRUE;
                  }
                  break;

               case TXc_F2:                     // set 16x32 size,
               case TXa_T:                      // or Toggle to initial
                  if (dat->autosize)
                  {
                     delta_x =  (16 - dat->cols) * 4;
                     delta_y =  (32 - dat->rows);
                     dat->rows = 32;            // set fixed size 16x32 (one 512 byte sector)
                     dat->cols = 16;
                     dat->autosize = FALSE;     // no change on window resize
                  }
                  else                          // back to initial, and auto resize
                  {
                     delta_x =  (dat->initialCols - dat->cols) * 4;
                     delta_y =  (dat->initialRows - dat->rows);
                     dat->rows = dat->initialRows;
                     dat->cols = dat->initialCols; // reset to intial (auto) size
                     dat->autosize = TRUE;
                  }
                  //- resize the parent (frame arround the actual HEXEDIT control) accordingly
                  txwSetWindowPos( parent,  0,  0,  0,  delta_x, delta_y, TXSWP_SIZE | TXSWP_RELATIVE);
                  upd = TRUE;
                  break;

               case TXc_UP:
                  if (dat->posCursor >= dat->cols) // keep cursor inside data area
                  {
                     dat->posCurBuf += dat->cols;
                     dat->posCursor -= dat->cols;
                     upd = TRUE;
                  }
                  break;

               case TXc_DOWN:
                  if (dat->posCursor < wbytes - dat->cols)
                  {
                     dat->posCurBuf -= dat->cols;
                     dat->posCursor += dat->cols;
                     upd = TRUE;
                  }
                  break;

               case TXc_LEFT:
                  if (dat->posCursor > 0)       // keep cursor inside data area
                  {
                     dat->posCurBuf += 1;
                     dat->posCursor -= 1;
                     upd = TRUE;
                  }
                  break;

               case TXc_RIGHT:
                  if (dat->posCursor < wbytes -1)
                  {
                     dat->posCurBuf -= 1;
                     dat->posCursor += 1;
                     upd = TRUE;
                  }
                  break;

               case TXk_UP:
               case TXk_PGUP:
               case TXa_PGUP:
                  switch (key)
                  {
                     case TXk_UP:   value = dat->cols;             break; // one line
                     case TXk_PGUP: value = min( wbytes, pbytes);  break; // one screen
                     default:       value = dat->curr->size;       break; // one item
                  }
                  TRACES(( "UP:%5d  posCurBuf:%5d  posCursor:%5d  wbytes:%5d  pbytes:%5d\n",
                           value, dat->posCurBuf, dat->posCursor, wbytes, pbytes));

                  if (offset >= value)          // in same buffer ?
                  {
                     if (dat->posCursor >= value)
                     {
                        txwHexEditShowCursor( hwnd, FALSE, modified, dat);
                        dat->posCursor -= value;
                        txwHexEditShowCursor( hwnd, TRUE,  modified, dat);
                     }
                     else                       // scroll buffer position
                     {
                        dat->posCurBuf -= value;
                        upd = TRUE;
                     }
                  }
                  else                          // move to PREV, PRE2 or PRE3
                  {
                     if (modified)
                     {
                        wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                     }
                     if (wr == NO_ERROR)
                     {
                        if (txwHexEditPrevBuf( dat, offset - value)) // BACKWARD, so negative
                        {
                           txtHexEditPrepareBuf( dat);

                           TRACES(( "Pr:%5d  posCurBuf:%5d  posCursor:%5d  offset:%5d\n",
                                    value, dat->posCurBuf, dat->posCursor, offset));

                           if (dat->posCursor >= value)
                           {
                              dat->posCursor -= value;
                           }
                           else                 // scroll buffer position
                           {
                              dat->posCurBuf -= value;
                           }

                           TRACES(( "US:%5d  posCurBuf:%5d  posCursor:%5d  offset:%5d\n",
                                    value, dat->posCurBuf, dat->posCursor, offset));

                           upd = TRUE;
                        }
                        else
                        {
                           txwPostMsg( hwnd, TXWM_CHAR, mp1, (TXWMPARAM) TXk_HOME);
                        }
                     }
                  }
                  break;

               case TXk_DOWN:
               case TXk_PGDN:
               case TXa_PGDN:
                  switch (key)
                  {
                     case TXk_DOWN: value = dat->cols;             break; // one line
                     case TXk_PGDN: value = min( wbytes, nbytes);  break; // one screen
                     default:       value = dat->curr->size;       break; // one item
                  }
                  TRACES(( "DN:%5d  posCurBuf:%5d  posCursor:%5d  wbytes:%5d  nbytes:%5d\n",
                           value, dat->posCurBuf, dat->posCursor, wbytes, nbytes));

                  if ((offset + value) < dat->curr->size) // in same buffer ?
                  {
                     if (dat->posCursor + value < wbytes)
                     {
                        txwHexEditShowCursor( hwnd, FALSE, modified, dat);
                        dat->posCursor += value;
                        txwHexEditShowCursor( hwnd, TRUE,  modified, dat);
                     }
                     else                       // scroll buffer position
                     {
                        dat->posCurBuf += value;
                        upd = TRUE;
                     }
                  }
                  else                          // try the next buffers
                  {
                     if (modified)
                     {
                        wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                     }
                     if (wr == NO_ERROR)
                     {
                        if (txwHexEditNextBuf( dat, offset + value)) // FORWARD, so POSITIVE
                        {
                           txtHexEditPrepareBuf( dat);

                           TRACES(( "Nx:%5d  posCurBuf:%5d  posCursor:%5d  offset:%5d\n",
                                    value, dat->posCurBuf, dat->posCursor, offset));

                           if (dat->posCursor < (wbytes - value))
                           {
                              dat->posCursor += value;
                           }
                           else                 // scroll buffer position
                           {
                              dat->posCurBuf += value;
                           }
                           TRACES(( "DS:%5d  posCurBuf:%5d  posCursor:%5d  offset:%5d\n",
                                    value, dat->posCurBuf, dat->posCursor, offset));

                           upd = TRUE;
                        }
                        else
                        {
                           txwPostMsg( hwnd, TXWM_CHAR, mp1, (TXWMPARAM) TXk_END);
                        }
                     }
                  }
                  break;

               case TXk_BACKSPACE:
                  if (dat->curr->data)
                  {
                     if (dat->ascii)
                     {
                        dat->curr->data[ offset] = ' ';
                     }
                     else
                     {
                        dat->curr->data[ offset] = 0;
                        dat->hexLsb = 0;        // move to leftmost nibble
                     }
                     upd = TRUE;
                  }
               case TXk_LEFT:
                  if (!dat->ascii && dat->hexLsb)
                  {
                     dat->hexLsb = 0;           // move to leftmost nibble
                     txwHexEditShowCursor( hwnd, TRUE,  modified, dat);
                  }
                  else                          // move to previous byte
                  {
                     dat->hexLsb = 1;           // move to rightmost nibble
                     if (offset > 0)            // not on first buffer byte
                     {
                        if (dat->posCursor >= 1)
                        {
                           txwHexEditShowCursor( hwnd, FALSE, modified, dat);
                           dat->posCursor -= 1;
                           txwHexEditShowCursor( hwnd, TRUE,  modified, dat);
                        }
                        else                    // scroll buffer position
                        {
                           dat->posCurBuf -= dat->cols;
                           dat->posCursor += dat->cols -1;
                           upd = TRUE;
                        }
                     }
                     else if (dat->prev && dat->prev->data) // move to PREV buffer
                     {
                        if (modified)
                        {
                           wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                        }
                        if (wr == NO_ERROR)
                        {
                           dat->posCurBuf += dat->prev->size;
                           txwHexEditCallBack( TXHE_CB_PREV_BUF, dat);
                           txtHexEditPrepareBuf( dat);

                           if (dat->posCursor > 0)
                           {
                              dat->posCursor -= 1;
                           }
                           else                 // scroll buffer position
                           {
                              dat->posCurBuf -= dat->cols;
                              dat->posCursor += dat->cols -1;
                           }
                           upd = TRUE;
                        }
                     }
                  }
                  break;


               default:                         // possible 'dangerous' key when no data
                  if (dat->curr->data)
                  {
                     switch (key)               // key value
                     {
                        case TXk_INSERT:        // when at last byte of a file
                           if (txwHexEditCallBack( TXHE_CB_INSERT_1, dat) == TX_PENDING)
                           {
                              txtHexEditPrepareBuf(dat); // reset modify status & diff
                              if (dat->diff)
                              {                 // reset first DIFF byte copied
                                 dat->diff[0] = 0; // from newly created buffer so
                                 dat->currCrc++; // status will be 'modified'
                                 modified = TRUE;
                              }
                           }
                           upd = TRUE;
                           break;

                        case TXk_DELETE:
                           if (txwHexEditCallBack( TXHE_CB_DELETE_1, dat) == TX_PENDING)
                           {
                              txtHexEditPrepareBuf(dat); // reset modify status & diff
                           }
                           upd = TRUE;
                           break;

                        case TXc_D:
                        case TXa_D:
                           if (txwHexEditCallBack( TXHE_CB_DELTOEND, dat) == TX_PENDING)
                           {
                              txtHexEditPrepareBuf(dat); // reset modify status & diff
                           }
                           upd = TRUE;
                           break;

                        case TXa_B:             // byte/block mark
                           if (dat->markSize == 0) // new mark
                           {
                              dat->markBase  = dat->curr->start;
                              dat->markStart = offset;
                              dat->markSize  = 1;
                           }
                           else                 // (re)size the mark
                           {
                              if      (offset > dat->markStart)
                              {
                                 dat->markSize  = offset - dat->markStart + 1;
                              }
                              else if (offset < dat->markStart)
                              {
                                 dat->markSize  = dat->markStart - offset + 1;
                                 dat->markStart = offset;
                              }
                           }
                           upd = TRUE;
                           break;

                        case TXa_E:             // mark to end of object
                           dat->markStart = offset;
                           dat->markSize  = dat->curr->size - offset;
                           upd = TRUE;
                           break;

                        case TXa_H:             // toggle markDump
                           if ((dat->markSize) && (dat->markBase == dat->curr->start))
                           {
                              dat->markDump = !dat->markDump; // toggle markDump if existing mark in CURR buffer
                           }
                           upd = TRUE;
                           break;

                        case TXa_J:             // mark from begin of line
                           dat->markBase  = dat->curr->start;
                           dat->markStart = offset - (dat->posCursor % dat->cols);
                           dat->markSize  = dat->posCursor % dat->cols +1;
                           upd = TRUE;
                           break;

                        case TXa_K:             // mark to end of line
                           dat->markBase  = dat->curr->start;
                           dat->markStart = offset;
                           dat->markSize  = dat->cols - (dat->posCursor % dat->cols);
                           upd = TRUE;
                           break;

                        case TXa_L:             // line mark
                           value = offset - (dat->posCursor % dat->cols);
                           if (dat->markSize == 0) // new mark
                           {
                              dat->markBase  = dat->curr->start;
                              dat->markStart = value;
                              dat->markSize  = dat->cols;
                           }
                           else                 // (re)size the mark
                           {
                              if      (value > dat->markStart)
                              {
                                 dat->markSize  = value - dat->markStart + dat->cols;
                              }
                              else if (value < dat->markStart)
                              {
                                 dat->markSize  = dat->markStart - value + dat->cols;
                                 dat->markStart = value;
                              }
                           }
                           upd = TRUE;
                           break;

                        case TXa_1: case TXa_2: case TXa_3: case TXa_4: case TXa_5:
                        case TXa_6: case TXa_7: case TXa_8: case TXa_9: case TXa_0:
                           dat->markBase  = dat->curr->start;
                           dat->markStart = offset;
                           switch (key)
                           {
                              case TXa_1: dat->markSize  =  1; break;
                              case TXa_2: dat->markSize  =  2; break;
                              case TXa_3: dat->markSize  =  3; break;
                              case TXa_4: dat->markSize  =  4; break;
                              case TXa_5: dat->markSize  =  5; break;
                              case TXa_6: dat->markSize  =  6; break;
                              case TXa_7: dat->markSize  =  7; break;
                              case TXa_8: dat->markSize  =  8; break;
                              case TXa_9: dat->markSize  =  9; break;
                              case TXa_0: dat->markSize  = 10; break;
                           }
                           if ((offset + dat->markSize) > dat->curr->size)
                           {
                              dat->markSize = dat->curr->size - offset;
                           }
                           upd = TRUE;
                           break;

                        case TXa_O:             // EOS2 compatible overlay :-)
                        case TXa_G:             // copy mark data to cursor
                        case TXa_M:             // copy mark data, move mark
                           if (dat->markSize)   // we have a mark, could be
                           {                    // in another object (sector)
                              value = min( dat->markSize, (dat->curr->size - offset));
                              memmove( dat->curr->data + offset,
                                       dat->curr->data + dat->markStart, value);
                              if (key == TXa_M)
                              {
                                 dat->markBase  = dat->curr->start;
                                 dat->markStart = offset;
                                 dat->markSize  = value;
                              }
                              upd = TRUE;
                           }
                           break;

                        case TXa_R:             // reverse bytes in mark area
                           if ((dat->markSize) && (dat->markBase == dat->curr->start))
                           {
                              BYTE  *lo = dat->curr->data + dat->markStart;
                              BYTE  *hi = lo + dat->markSize -1;

                              while (lo < hi)
                              {
                                 tmp   = *lo;
                                 *lo++ = *hi;
                                 *hi-- = tmp;
                              }
                              upd = TRUE;
                           }
                           break;

                        case TXa_S:             // mark from start of object
                           dat->markBase  = dat->curr->start;
                           dat->markStart = 0;
                           dat->markSize  = offset +1;
                           upd = TRUE;
                           break;

                        case TXa_U:             // unmark HEXED
                           dat->markStart = 0;
                           dat->markSize  = 0;
                           upd = TRUE;
                           break;

                        case TXa_A:             // mark ALL (whole item)
                           dat->markBase  = dat->curr->start;
                           dat->markStart = 0;
                           dat->markSize  = dat->curr->size;
                           upd = TRUE;
                           break;

                        case TXa_C:             // Copy to clipboard, HEX-pairs or pure ASCII
                        case TXc_C:
                        case TXc_INSERT:
                           {
                              char   *text = NULL;
                              ULONG   nr   = 0; // nr of bytes copied to clipboard

                              if ((dat->markSize != 0)   && // there is a marked area, and it is
                                  (dat->markDump == FALSE)) // NOT for markDump, copy byte-range
                              {
                                 nr = txwSaveToMemHE( dat, &text); // just the marked byte range
                              }
                              else              // create HEXDUMP format of (part of) current buffer
                              {
                                 nr = txwHdump2MemHE( dat, &text); // CURR item in HEXDUMP format
                              }
                              if (text != NULL)
                              {
                                 if (TxCopyClipBoardText( text) == NO_ERROR) // string to clipboard
                                 {
                                    if ((dat->markSize != 0)   && // there is a marked area, and it is
                                        (dat->markDump == FALSE)) // NOT for markDump, copy byte-range
                                    {
                                       sprintf( statusMsg, "Copied %u %s bytes to the clipboard as %s", nr,
                                                           (dat->markSize) ? "MARKED" : "ITEM",
                                                           (dat->ascii)    ? "ASCII" : "HEX-pairs");
                                    }
                                    else        // created HEXDUMP format of current buffer
                                    {
                                       sprintf( statusMsg, "Copied HEXDUMP for %u bytes to the clipboard in %lu characters",
                                                            nr, strlen( text));
                                    }
                                    txwSetSbviewStatus( statusMsg, cSchemeColor);
                                    TxSleep( 700);
                                 }
                                 TxFreeMem( text);
                              }
                           }
                           break;

                        case TXa_V:
                        case TXc_V:        // Paste from clipboard (overlay at cursor/marked-range)
                        case TXa_INSERT:
                           {                    // worst case, prefixed HEX pair + CR/LF as separator
                              ULONG   bufSize  = 6 * dat->curr->size + 1; // assume 6 chars per BYTE
                              char   *pasteBuf = TxAlloc( 1, bufSize);
                              ULONG   nr = 0;

                              if (pasteBuf)
                              {
                                 if (TxPasteClipBoardLines( bufSize, FALSE, pasteBuf) != 0) // multi line OK
                                 {
                                    if ((nr = txwPasteDataHE( dat, pasteBuf)) != 0)
                                    {
                                       sprintf( statusMsg, "Pasted %u bytes from clipboard at %s", nr,
                                                                 (dat->markSize) ? "MARK" : "CURSOR");
                                       txwSetSbviewStatus( statusMsg, cSchemeColor);
                                       TxSleep( 700);
                                    }
                                    upd = TRUE;
                                 }
                                 TxFreeMem( pasteBuf);
                              }
                           }
                           break;

                        case TXc_E:             // Erase/Fill current item
                        case TXa_F:
                           if (dat->curr->data)
                           {
                              BOOL marked = ((dat->markSize) && (dat->markBase == dat->curr->start));

                              strcpy( str, "00");
                              if (TxPrompt( 0, 5, str, "Specify hexadecimal value to erase "
                                           "the %s with", (marked) ? "marked area" : "entire item"))
                              {
                                 BYTE erase = 0;

                                 sscanf( str, "%hx", (USHORT *) &erase);
                                 memset( dat->curr->data + dat->markStart, erase,
                                        (marked) ?  dat->markSize : dat->curr->size);
                                 upd = TRUE;
                              }
                           }
                           break;

                        case TXc_W:             // unconditional write back
                           wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                           if (wr == NO_ERROR)
                           {
                              txtHexEditPrepareBuf(dat); // reset modify status & diff
                              upd = TRUE;
                           }
                           break;

                     #if !defined (UNIX)
                        case TXc_Z:             // c-Z is SIGSTP, suspend program! (resume with 'fg')
                     #endif
                        case TXa_Z:             // undo changes to CURR
                           if (dat->diff && dat->curr->data)
                           {
                              memcpy( dat->curr->data, dat->diff, dat->curr->size);
                              upd = TRUE;
                           }
                           break;

                        default:
                           if (txwPossibleAsciiKey(key))
                           {
                              if (dat->curr->data)
                              {
                                 if (dat->ascii)
                                 {
                                    dat->curr->data[ offset] = key & 0xff;
                                    upd = TRUE;
                                 }
                                 else if (isxdigit( key))
                                 {
                                    BYTE  hex = toupper( key) - '0';

                                    if (hex > 9)
                                    {
                                       hex -= 7;
                                    }
                                    if (dat->hexLsb)
                                    {
                                       dat->curr->data[ offset] &= 0xf0;
                                       dat->curr->data[ offset] |= hex;
                                    }
                                    else
                                    {
                                       dat->curr->data[ offset] &= 0x0f;
                                       dat->curr->data[ offset] |= (hex << 4);
                                    }
                                    upd = TRUE;
                                 }
                                 else if (key != TXk_RIGHT)
                                 {
                                    dat->hexLsb = 0; // stay on same byte for
                                 }              // non-hex keys on HEX
                              }
                           }
                           else if (key != TXk_RIGHT)
                           {
                              rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                              break;
                           }
                           if (!dat->ascii && !dat->hexLsb)
                           {
                              if (key != TXk_SPACE)
                              {
                                 dat->hexLsb = 1; // move to rightmost nibble
                                 txwHexEditShowCursor( hwnd, TRUE,  modified, dat);
                              }
                           }
                           else                 // move to next byte
                           {
                              dat->hexLsb = 0;  // move to leftmost nibble
                              if (offset < (dat->curr->size - 1)) // not on last buffer byte
                              {
                                 if (dat->posCursor < wbytes -1)
                                 {
                                    txwHexEditShowCursor( hwnd, FALSE, modified, dat);
                                    dat->posCursor += 1;
                                    txwHexEditShowCursor( hwnd, TRUE,  modified, dat);
                                 }
                                 else           // scroll buffer position
                                 {
                                    dat->posCurBuf += dat->cols;
                                    dat->posCursor -= dat->cols -1;
                                    upd = TRUE;
                                 }
                              }
                              else if (dat->next && dat->next->data) // move to NEXT buffer
                              {
                                 if (modified)
                                 {
                                    wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                                 }
                                 if (wr == NO_ERROR)
                                 {
                                    dat->posCurBuf -= dat->curr->size;
                                    txwHexEditCallBack( TXHE_CB_NEXT_BUF, dat);
                                    txtHexEditPrepareBuf( dat);

                                    if (dat->posCursor < wbytes -1)
                                    {
                                       dat->posCursor += 1;
                                    }
                                    else        // scroll buffer position
                                    {
                                       dat->posCurBuf += dat->cols;
                                       dat->posCursor -= dat->cols -1;
                                    }
                                    upd = TRUE;
                                 }
                              }
                           }
                           break;
                     }
                  }
                  else
                  {
                     rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
                  }
                  break;
            }
            break;

         case TXWM_BUTTONDOWN:
         case TXWM_BUTTONDBLCLK:
            if (txwMouseButtonDown( hwnd, msg, mp1, mp2) == TX_PENDING)
            {
               short  col = TXMOUSECOL()  - win->client.left;  //- relative to
               short  row = TXMOUSEROW()  - win->client.top;   //- the window!
               short  ac  = TXHE_AC_COLUMN( dat->cols);

               TRACES(( "BDOWN - row:%4hd col:%4hd  ac:%4hd  posCursor:%4d\n", row, col, ac, dat->posCursor));
               if ((row == 0) || (row > dat->rows)) // not data area, buttons
               {
                  if (col < 6)
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, mp1, (row) ? (TXWMPARAM) TXk_PGDN : (TXWMPARAM) TXk_PGUP);
                  }
                  else if ((col >= ac) && (col < (ac + 8)))
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, mp1, (row) ? (TXWMPARAM) TXa_PGDN : (TXWMPARAM) TXa_PGUP);
                  }
                  else                          // rest, line up/down
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, mp1, (row) ? (TXWMPARAM) TXk_DOWN : (TXWMPARAM) TXk_UP);
                  }
               }
               else if ((col < 8) || (col == ac) || (col > (ac + dat->cols)))
               {
                  if (col == ac)                // ON start ASCII area [ 'scrollbar'
                  {
                     ULONG   key = TXk_PGUP;

                     if      (row < 4)        key = TXc_DOWN;
                     else if (row + 6 > sy)   key = TXc_UP;
                     else if (row < (sy / 2)) key = TXk_PGDN;

                     TRACES(( "HEXEDIT 'scrollbar' click to key: %s\n", txwKeyDescription(key)));
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) key);
                  }
                  else if (msg == TXWM_BUTTONDBLCLK) // ignore single-click here (too easy)
                  {
                     if ((dat->markSize) && (dat->markBase == dat->curr->start))
                     {
                        dat->markDump = !dat->markDump; // toggle markDump if existing mark in CURR buffer
                        upd = TRUE;
                     }
                  }
               }
               else                             // in data area, place cursor, perform optional action
               {
                  txwHexEditShowCursor( hwnd, FALSE, modified, dat);
                  dat->posCursor = (row - 1) * dat->cols;
                  if (col > ac)
                  {
                     dat->posCursor += col - ac - 1;
                     dat->ascii = TRUE;
                  }
                  else                          // in HEX byte area
                  {
                     dat->posCursor += (col - 8) / 3;
                     dat->hexLsb = ((col - 8) % 3) ? 1 : 0;
                     dat->ascii = FALSE;
                  }
                  TRACES(( "Moved to posCursor:%4d   posCurBuf:%4d  for size: %u\n",
                                dat->posCursor, dat->posCurBuf, dat->curr->size));

                  if ((dat->posCurBuf + dat->posCursor) <  0)
                  {
                     TRACES(("Clicked in PREV buffer! Need to reposition/reload\n"));

                     if (modified)
                     {
                        wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                     }
                     if (wr == NO_ERROR)
                     {
                        if (txwHexEditPrevBuf( dat, dat->posCurBuf + dat->posCursor))
                        {
                           txtHexEditPrepareBuf( dat);
                           upd = TRUE;
                        }
                     }
                  }
                  else if ((dat->posCurBuf + dat->posCursor) >= dat->curr->size)
                  {
                     TRACES(("Clicked in NEXT buffer! Need to reposition/reload\n"));

                     if (modified)
                     {
                        wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
                     }
                     if (wr == NO_ERROR)
                     {
                        if (txwHexEditNextBuf( dat, dat->posCurBuf + dat->posCursor))
                        {
                           txtHexEditPrepareBuf( dat);
                           upd = TRUE;
                        }
                     }
                  }
                  txwHexEditShowCursor( hwnd, TRUE,  modified, dat);

                  //- perform optional click actions (AFTER setting the cursor-location to mouse position)
                  if      ((TXMOUSEKEYSCA() & TXm_KS_ALT) != 0) // Alt-click copies mark (or default area) to clipboard
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_C); // simulate keyboard c-copy action
                  }
                  else if ((TXMOUSEKEYSCA() & TXm_KS_CTRL) != 0) // Ctrl-click is PASTE from clipboard
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_V); // simulate keyboard c-paste action
                  }
                  else if (msg == TXWM_BUTTONDBLCLK) // unmark doubleclick HEXEDIT
                  {
                     txwPostMsg( hwnd, TXWM_CHAR, 0, (TXWMPARAM) TXa_U); // simulate keyboard unmark action
                  }
               }
            }
            break;

         case TXWM_SETMARKRECT:                 // set new mark area
            if (mp1 != NULL)                    // we have a rectangle
            {
               TXRECT  *mRect = (TXRECT *) mp1;
               short    col   = mRect->left   - win->client.left; //- START relative to upper left HEXED window
               short    row   = mRect->top    - win->client.top;
               short    lrCol = mRect->right  - win->client.left; //- END   relative to upper left HEXED window
               short    lrRow = mRect->bottom - win->client.top;
               short    ac    = TXHE_AC_COLUMN( dat->cols);       //- startcol ascii area
               LONG     lrEndposMarked;                           //- position in buffer, for end of mark
               LONG     newPosCursor = (row - 1) * dat->cols;     //- will be new cursorpos, when INSIDE CURR buffer

               TRACES(("row: %hd lrRow: %hd  col: %hd lrCol: %hd\n", row, lrRow, col, lrCol));

               TRACES(( "posCurBuf:%6d   posCursor:%6d   markBase:%16.16llx markStart:%6d   markSize:%6d  newPosCursor:%6d\n",
                    dat->posCurBuf, dat->posCursor, dat->markBase,     dat->markStart, dat->markSize,     newPosCursor));

               txwHexEditShowCursor( hwnd, FALSE, modified, dat); // hide the (shadow) cursors

               if ((col < 7) || (lrCol < 7) || (col > (ac + dat->cols)) || (lrCol > (ac + dat->cols)))
               {
                  dat->markDump = TRUE;         // any of the mark-rectangle corners outside HEX/ASCII area
               }
               else
               {
                  dat->markDump = FALSE;        // keep default mark, a byte-range, not hexDump
               }

               //- calculate and set the START of the marked area, as a position in the buffer
               if ((dat->posCurBuf + newPosCursor) < 0)     //- Cursor would be outside CURR item
               {
                  dat->posCursor = - dat->posCurBuf;        //- Update cursor position to START of item
                  dat->hexLsb    = 0;
               }
               else                                         //- Top-mark/Cursor will be inside CURR item
               {
                  if      (col < 7)                         //- limit column to left-side of HEX-area
                  {
                     col   = 8;
                     lrCol = ac -1;                         //- mark whole line, when touching offset area
                  }
                  else if (col == ac)                       //- limit column to right-side of ascii area
                  {
                     col = ac + 1;                          //- don't extend mark when on AC marker column
                  }
                  else if (col > (ac + dat->cols))          //- limit column to right-side of ascii area
                  {
                     col = ac + dat->cols;
                  }
                  if      (lrCol < 7)                       //- limit column to left-side of HEX-area
                  {
                     lrCol = 8;
                     col   = ac -1;                         //- mark whole line, when touching offset area
                  }
                  if      (lrCol > (ac + dat->cols))        //- limit column to right-side of ascii area
                  {
                     lrCol = ac + dat->cols;
                  }
                  else if (lrCol == ac)                     //- don't extend mark when on AC marker column
                  {
                     lrCol = ac - 1;
                  }
                  dat->posCursor = newPosCursor;            //- Update cursor position to mark-top
                  if (col > ac)                             //- START in ASCII area
                  {
                     dat->posCursor += col - ac - 1;        //- offset from start of row in ASCII area
                  }
                  else                          // START in HEX byte area
                  {
                     dat->posCursor += (col - 8) / 3;       //- offset from start of row in HEX area
                     dat->hexLsb = ((col - 8) % 3) ? 1 : 0; //- extra offset of on second nibble
                  }
               }
               dat->ascii = (col > ac) ? TRUE : FALSE;      //- set HEX versus ASCII depending on mouse column

               txwHexEditShowCursor( hwnd, TRUE,  modified, dat); // re-display the (shadow) cursors

               dat->markBase  = dat->curr->start;           //- Make sure Base is for CURR item
               dat->markStart = dat->posCurBuf + dat->posCursor;

               //- calculate the END of the marked area, as buffer position, then SIZE = START .. END
               lrEndposMarked = (lrRow - 1) * dat->cols;
               if (lrCol > ac)                  // END in ASCII area
               {
                  lrEndposMarked += lrCol - ac - 1;
               }
               else                             // END in HEX byte area
               {
                  lrEndposMarked += (lrCol - 8) / 3;
               }
               dat->markSize = lrEndposMarked - dat->posCursor + 1;

               if ((dat->markStart + dat->markSize) > dat->curr->size)
               {
                  dat->markSize = dat->curr->size - dat->markStart; // clip on item size!
               }
               TRACES(("lrEndposMarked: %d  dat->posCursor: %d  dat->markSize: %d\n",
                        lrEndposMarked,     dat->posCursor,     dat->markSize));

               //- upd = TRUE;  Invalidate/repaint will be done by sender of the message!
            }
            break;

         case TXWM_SIZE:                        // window has been resized
            dat->rows = sy - 3;                 // adapt to resizing
            if (dat->rows && (dat->posCursor >= dat->rows * dat->cols))
            {
               dat->posCurBuf += dat->cols;     // move view UP one line
               dat->posCursor -= dat->cols;     // to keep cursor visible
            }                                   // (paint to be done  by SIZE)
            break;

         case TXWM_DESTROY:                     // window will be destroyed
            if (dat->diff      &&               // determine REAL modify status
                dat->diffSize  &&               // determine REAL modify status
                dat->curr->data )               // for exact item size now.
            {                                   // currCrc wrong after delete!
               int     i;                       // (will display 'modified' :-)

               for (modified = FALSE, i = 0; i < dat->curr->size; i++)
               {
                  if (dat->diff[ i] != dat->curr->data[ i])
                  {
                     modified = TRUE;
                     break;
                  }
               }
            }
            if (modified)
            {
               wr = txwHexEditCallBack( TXHE_CB_WRITEBUF, dat);
            }
            TxFreeMem( win->he.diff);           // free difference buffer
            dat->diffSize = 0;
            break;

         default:
            rc = txwDefWindowProc( hwnd, msg, mp1, mp2);
            break;
      }
      if (upd)
      {
         rc = txwInvalidateWindow( hwnd, TRUE, TRUE);
      }
   }
   else
   {
      rc = TX_INVALID_HANDLE;
   }
   RETURN( rc);
}                                               // end 'txwHexEditWinProc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Save the whole CURR buffer to memory as an HEXDUMP, including offsets etc
/*****************************************************************************/
static ULONG txwHdump2MemHE                     // RET   nr of bytes copied
(
   TXHEXEDIT          *he,                      // IN    HEXEDIT data structure
   char              **heText                   // OUT   allocated text buffer
)                                               //       (needs FreeMem)
{
   ULONG               done = 0;                // bytes written to hexDump
   ULONG               bytes;                   // bytes todo
   ULONG               lines;                   // nr of hexdump lines
   ULONG               cpl;                     // output chars per LINE
   ULONG               column;                  // counter within the line
   ULONG               offset;                  // relative byte offset, start of line
   BYTE               *start;                   // first byte in a line
   BYTE               *first;                   // first byte location
   BYTE               *last;                    // last  byte location
   BYTE               *this;                    // current byte to copy
   char               *tbuf = NULL;             // text buffer to fill
   TXTM                word;                    // smallish buffer to be appended

   ENTER();

   if (he && he->curr && he->curr->data && he->curr->size) // there IS data
   {
      bytes = he->curr->size;
      first = he->curr->data;                   // ptr to first BYTE to dump
      last  = he->curr->data + bytes   - 1;     // ptr to last  BYTE to dump
      lines = ((bytes - 1) / he->cols) + 1;     // nr of lines in the hexdump
      cpl   = (he->cols * 4) + 30;              // 4 chars per BYTE + lead / tail
      TRACES(("Bytes todo: %u, cpl: %u  from: %p\n", bytes, cpl, first));

      if ((tbuf = TxAlloc( cpl, lines + 5)) != NULL)  //- hexdump including header/frame lines
      {
         //- print the header lines
         strcpy( tbuf, txwa->desktop->window->title); // application title, with version + info
         strcat( tbuf, "\n");
         strcat( tbuf, "offset| ");
         for (column = 0; column < he->cols; column++) // number the columns
         {
            sprintf( word, "%2x ", column);
            strcat(  tbuf, word);
         }

         //- print a framing seperator line ABOVE the actual dump
         sprintf( word, "[%-*.*s] Absolute offset\n", he->cols, he->cols, "     ascii");
         strcat(  tbuf, word);
         strcat( tbuf, "------+-");
         for (column = 0; column < he->cols; column++) // frame the columns
         {
            strcat(  tbuf, "---");
         }
         sprintf( word, "+%-*.*s+----------------\n", he->cols, he->cols, "----------------------------------------------------------------");
         strcat(  tbuf, word);

         //- now print each HEXDUMP line in turn, including leader/trailer parts
         for (offset = 0, start = first; start <= last; start += he->cols, offset += he->cols)
         {
            BOOL       lineHasMark = FALSE;        //- to be calculated, this line has at least one marked byte

            if ((he->markSize) && (he->markDump))  //- is it a hexDump type mark? (include offset/position)
            {
               for (column = 0; column < he->cols; column++) // use existing logic from byte-painting (works :)
               {
                  if (((he->curr->start + offset + column) >= (he->markBase + he->markStart)) &&
                      ((he->curr->start + offset + column) <  (he->markBase + he->markStart + he->markSize)))
                  {
                     lineHasMark = TRUE;        // 'inmark' = at least on byte within the mark
                  }
               }
            }
            else
            {
               lineHasMark = TRUE;              // consider all lines marked, when no mark at all
            }

            if (lineHasMark)                    // only do the 'marked' lines (to be refined, just bytes IN mark)
            {
               //- print offset for this line
               sprintf( word, "%06x| ", offset);
               strcat( tbuf,  word);            // append this word to output string

               //- print the HEX-pairs, filling out when not a full line
               for (this = start, column = 0; column < he->cols; column++, this++)
               {
                  BOOL inrange = (this <= last);

                  if (inrange && (he->markSize)) // test if byte is inside mark, if any
                  {
                     if (((he->curr->start + offset + column) <  (he->markBase + he->markStart)) ||
                         ((he->curr->start + offset + column) >= (he->markBase + he->markStart + he->markSize)))
                     {
                        inrange = FALSE;
                     }
                  }
                  if (inrange)                  // still within buffer, and mark (if any)
                  {
                     sprintf( word, "%2.2hhx ", *this);
                     strcat( tbuf,  word);      // append this word to output string
                     done++;                    // count the byte as done
                  }
                  else
                  {
                     strcat( tbuf, "-- ");      // fill out with dashes
                  }
               }

               //- print the ASCII representation, filling out when no a full line
               strcat( tbuf, "[");
               for (this = start, column = 0; column < he->cols; column++, this++)
               {
                  BOOL inrange = (this <= last);

                  if (inrange && (he->markSize)) // test if byte is inside mark, if any
                  {
                     if (((he->curr->start + offset + column) <  (he->markBase + he->markStart)) ||
                         ((he->curr->start + offset + column) >= (he->markBase + he->markStart + he->markSize)))
                     {
                        inrange = FALSE;
                     }
                  }
                  if (inrange)                  // still within buffer, and mark, if any
                  {
                     sprintf( word, "%c", TxPrintable(*this));
                     strcat( tbuf,  word);      // append this word to output string
                  }
                  else
                  {
                     strcat( tbuf,  " ");       // fill out with spaces
                  }
               }

               //- print the absolute address for line start, and end the line
               sprintf( word, "] 0x%13.13llx\n", offset + he->curr->start);
               strcat( tbuf,  word);            // append this word to output string
            }
         }
         //- print a framing seperator line BELOW the actual dump
         strcat( tbuf, "------+-");
         for (column = 0; column < he->cols; column++) // frame the columns
         {
            strcat(  tbuf, "---");
         }
         sprintf( word, "+%-*.*s+----------------\n", he->cols, he->cols, "----------------------------------------------------------------");
         strcat(  tbuf, word);
         strcat(  tbuf, he->curr->desc);
         strcat(  tbuf, "\n");
      }
   }
   *heText = tbuf;
   RETURN (done);
}                                               // end 'txwHdump2MemHE'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Save the marked-area of CURR buffer to memory as an ASCII / HEX-pairs range
/*****************************************************************************/
static ULONG txwSaveToMemHE                     // RET   nr of bytes copied
(
   TXHEXEDIT          *he,                      // IN    HEXEDIT data structure
   char              **heText                   // OUT   allocated text buffer
)                                               //       (needs FreeMem)
{
   ULONG               done = 0;                // nr of bytes done
   ULONG               todo;                    // bytes todo
   ULONG               cpb;                     // output chars per BYTE
   BYTE               *first;                   // first byte location
   BYTE               *this;                    // current byte to copy
   char               *tbuf = NULL;             // text buffer to fill
   TXTS                word;                    // output string for one byte

   ENTER();


   if (he && he->curr && he->curr->data && he->curr->size) // there IS data
   {
      if (he->markSize != 0)                    // there is a marked area
      {
         todo  = he->markSize;                  // just the mark
         first = he->curr->data + he->markStart;
      }
      else                                      // whole item (sector or cluster/block)
      {
         todo  = he->curr->size;
         first = he->curr->data;
      }
      cpb = (he->ascii) ? 1 : 3;                // ASCII or HEX-pair size
      TRACES(("todo: %u bytes, cpb: %u  from: %p\n", todo, cpb, first));

      if ((tbuf = TxAlloc( cpb, todo + 1)) != NULL) // #bytes info + terminating zero
      {
         for (this = first, done = 0; done < todo; done++, this++)
         {
            if (he->ascii)                      // one character
            {
               word[0] = TxPrintable( *this);   // replace unprintable by DOT
               word[1] = 0;
            }
            else                                // HEX-pair
            {
               sprintf( word, "%2.2hhx ", *this);
            }
            strcat( tbuf,  word);               // append this word to output string
         }
      }
   }
   *heText = tbuf;
   RETURN (done);
}                                               // end 'txwSaveToMemHE'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Paste string of ASCII charcs or HEX-pairs into buffer, at cursor or mark
/*****************************************************************************/
static ULONG txwPasteDataHE                     // RET   nr of bytes copied
(
   TXHEXEDIT          *he,                      // INOUT HEXEDIT data structure
   char               *heText                   // IN    allocated text buffer
)
{
   ULONG               done = 0;                // nr of bytes done
   LONG                todo;                    // (maximum) bytes todo, can be negative!
   BOOL                hexPairs = TRUE;         // input contain HEX pairs
   BYTE               *first;                   // first byte location
   BYTE               *this;                    // current byte to copy
   char               *s;                       // input string pointer
   int                 digits;                  // consecutive HEX digits
   ULONG               hexValue;                // convenient type for scanf
   LONG                offset;                  // start offset in (current) buffer

   ENTER();

   if (he && he->curr && he->curr->data && he->curr->size) // there IS data
   {
      TRACES(("markSize: %d  curr->size: %d  posCurBuf: %d  posCursor: %d\n",
           he->markSize, he->curr->size, he->posCurBuf, he->posCursor));

      if (  (he->markSize  != 0) &&             // there is a marked area
          (((he->posCurBuf + he->posCursor) >= he->markStart)                && // cursor is in
           ((he->posCurBuf + he->posCursor) < (he->markStart + he->markSize)))) // the mark area
      {
         offset = he->markStart;                // to START of marked area
         todo   = he->markSize;                 // limit to size of mark
      }
      else                                      // to cursor position
      {
         offset = he->posCurBuf + he->posCursor;
         todo   = he->curr->size - offset;
      }
      TRACES(("todo max:%4u bytes, to offset:%4u\n", todo, offset));

      if ((todo >= 0) && (offset >= 0) && ((offset + todo) <= he->curr->size)) // stay IN current buffer
      {
         first = he->curr->data + offset;       // points to first byte to be modified

         //- Check if input (only) contains valid hex pairs, otherwise consider ASCII
         for (digits = 0, s = heText; *s; s++)
         {
            if ((*s == 0x0a) || (*s == 0x0d) || (*s == '\t') || // whitespace, commas
                (*s == ' ' ) || (*s == ',' ) || isxdigit(*s)  ) // or hex-digits are OK
            {
               if (isxdigit(*s))
               {
                  if (++digits > 2)             // no pairs, but longer hex number
                  {
                     hexPairs = FALSE;
                     break;
                  }
                  else if ((*s == '0') && ((*(s+1) == 'x') || (*(s+1) == 'X'))) // hex prefix OK
                  {
                     s++;                       // extra skip '0'
                     digits = 0;                // and reset count
                  }
               }
               else
               {
                  digits = 0;                   // no hex digit, reset count
               }
            }
            else                                // invalid character for HEX-pairs
            {
               hexPairs = FALSE;
               break;
            }
         }
         TRACES(("Interpret as %s, destination: 0x%p\n", (hexPairs) ? "HEX-pairs" : "ASCII", first));

         for ( s = heText, done = 0,    this = first;
              *s &&       (done < todo);        // until end string, or the todo limit set
                           done++,      this++)
         {
            if (hexPairs)                       // parse one hex-pair
            {
               while ((*s == 0x0a) || (*s == 0x0d) || (*s == '\t') || // skip freespace, commas
                      (*s == ' ' ) || (*s == 'x' ) || (*s == ',' )  ) // and extra prefixes ...
               {
                  s++;
               }
               if ((*s == '0') && ((*(s+1) == 'x') || (*(s+1) == 'X'))) // hex prefix OK
               {
                  s += 2;                       // skip prefix
               }
               if (*s)                          // not at end now (trailing whitespace!)
               {
                  sscanf( s, "%x", &hexValue);  // hex value from string fragment
                  *this = (BYTE) hexValue;
                  s += 2;                       // skip the pair of hex digits
               }
               else
               {
                  done--;                       // fixup count (extra iteration done!)
               }
            }
            else
            {
               *this = *s++;                    // just copy the character
            }
         }
      }
   }
   RETURN (done);
}                                               // end 'txwPasteDataHE'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reload data to change the position to one of the NEXT buffers
/*****************************************************************************/
static BOOL txwHexEditNextBuf                   // RET   target buffer is SET
(
   TXHEXEDIT          *he,                      // INOUT HEXEDIT data structure
   LONG                target                   // IN    target offset, from CURR
)                                               //       a known POSITIVE value
{                                               //       larger than CURR size!
   BOOL                rc = FALSE;              // function return
   TXHE_CALLBACK       callback = (TXHE_CALLBACK) he->actionCallback;
   LONG                limit = he->curr->size;  // END of CURR buffer

   ENTER();
   TRACES(("ENTRY posCurBuf:%5d   posCursor:%5d\n", he->posCurBuf, he->posCursor));

   if (he->next && he->next->data)              // try NEXT
   {
      limit += he->next->size;
      TRACES(("Compare target :%5d  NEXT limit:%5d\n", target, limit));
      if (target <= limit)
      {
         he->posCurBuf -= he->curr->size;       // move to NEXT
         txwHexEditCallBack( TXHE_CB_NEXT_BUF, he);
         rc = TRUE;
      }
      else if (he->nex2 && he->nex2->data)      // try NEX2
      {
         limit += he->nex2->size;
         TRACES(("Compare target :%5d  NEX2 limit:%5d\n", target, limit));
         if (target <= limit)
         {
            he->posCurBuf -= he->curr->size;    // move to NEX2
            he->posCurBuf -= he->next->size;
            txwHexEditCallBack( TXHE_CB_NEX2_BUF, he);
            rc = TRUE;
         }
         else if (he->nex3 && he->nex3->data)   // try NEX3
         {
            limit += he->nex3->size;
            TRACES(("Compare target :%5d  NEX3 limit:%5d\n", target, limit));
            if (target <= limit)
            {
               he->posCurBuf -= he->curr->size; // move to NEX3
               he->posCurBuf -= he->next->size;
               he->posCurBuf -= he->nex2->size;
               txwHexEditCallBack( TXHE_CB_NEX3_BUF, he);
               rc = TRUE;
            }
            else if (he->nex4 && he->nex4->data) // try NEX4
            {
               limit += he->nex4->size;
               TRACES(("Compare target :%5d  NEX4 limit:%5d\n", target, limit));
               if (target <= limit)
               {
                  he->posCurBuf -= he->curr->size; // move to NEX4
                  he->posCurBuf -= he->next->size;
                  he->posCurBuf -= he->nex2->size;
                  he->posCurBuf -= he->nex3->size;
                  txwHexEditCallBack( TXHE_CB_NEX4_BUF, he);
                  rc = TRUE;
               }
            }
         }
      }
   }
   TRACES(("EXIT  posCurBuf:%5d   posCursor:%5d\n", he->posCurBuf, he->posCursor));
   BRETURN (rc);
}                                               // end 'txwHexEditNextBuf'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reload data to change the position to one of the PREV buffers
/*****************************************************************************/
static BOOL txwHexEditPrevBuf                   // RET   target buffer is SET
(
   TXHEXEDIT          *he,                      // INOUT HEXEDIT data structure
   LONG                target                   // IN    target offset, from CURR
)                                               //       a known NEGATIVE value!
{
   BOOL                rc = FALSE;              // function return
   TXHE_CALLBACK       callback = (TXHE_CALLBACK)  he->actionCallback;
   LONG                limit = 0;               // BEGIN of CURR buffer

   ENTER();
   TRACES(("ENTRY posCurBuf:%5d   posCursor:%5d\n", he->posCurBuf, he->posCursor));

   if (he->prev && he->prev->data)              // try PREV
   {
      limit -= he->prev->size;
      TRACES(("Compare target :%5d  PREV limit:%5d\n", target, limit));
      if (target >= limit)
      {
         he->posCurBuf += he->prev->size;       // move to PREV
         txwHexEditCallBack( TXHE_CB_PREV_BUF, he);
         rc = TRUE;
      }
      else if (he->pre2 && he->pre2->data)      // try PRE2
      {
         limit -= he->pre2->size;
         TRACES(("Compare target :%5d  PRE2 limit:%5d\n", target, limit));
         if (target >= limit)
         {
            he->posCurBuf += he->prev->size;    // move to PRE2
            he->posCurBuf += he->pre2->size;
            txwHexEditCallBack( TXHE_CB_PRE2_BUF, he);
            rc = TRUE;
         }
         else if (he->pre3 && he->pre3->data)   // try PRE3
         {
            limit -= he->pre3->size;
            TRACES(("Compare target :%5d  PRE3 limit:%5d\n", target, limit));
            if (target >= limit)
            {
               he->posCurBuf += he->prev->size; // move to PRE3
               he->posCurBuf += he->pre2->size;
               he->posCurBuf += he->pre3->size;
               txwHexEditCallBack( TXHE_CB_PRE3_BUF, he);
               rc = TRUE;
            }
            else if (he->pre4 && he->pre4->data) // try PRE4
            {
               limit -= he->pre4->size;
               TRACES(("Compare target :%5d  PRE4 limit:%5d\n", target, limit));
               if (target >= limit)
               {
                  he->posCurBuf += he->prev->size; // move to PRE4
                  he->posCurBuf += he->pre2->size;
                  he->posCurBuf += he->pre3->size;
                  he->posCurBuf += he->pre4->size;
                  txwHexEditCallBack( TXHE_CB_PRE4_BUF, he);
                  rc = TRUE;
               }
            }
         }
      }
   }
   TRACES(("EXIT  posCurBuf:%5d   posCursor:%5d\n", he->posCurBuf, he->posCursor));
   BRETURN (rc);
}                                               // end 'txwHexEditPrevBuf'
/*---------------------------------------------------------------------------*/

