//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2005 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
// This file contains Original Code and/or Modifications of Original Code as
// defined in and that are subject to the GNU Lesser General Public License.
// You may not use this file except in compliance with the License.
// BY USING THIS FILE YOU AGREE TO ALL TERMS AND CONDITIONS OF THE LICENSE.
// A copy of the License is provided with the Original Code and Modifications,
// and is also available at http://www.dfsee.com/txwin/lgpl.htm
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License,
// or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; (lgpl.htm) if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// Questions on TxWin licensing can be directed to: txwin@fsys.nl
//
// ==========================================================================
//
// Generic TX get library and OS version information
//
// Author: J. van Wijk
//
// JvW  24-07-2005 Initial version, split off from TXUTIL

#include <txlib.h>
#include <txvers.h>                             // TXLIB version definitions

#if defined (UNIX)
   #include <sys/utsname.h>
#endif

static  char           txVersion[ TXMAXTM];     // static version string


#if   defined (DOS32)
   static TXTM         exVersion;               // DOS extender version info
#elif defined (WIN32)

#define NTDLL_MODULENAME         "NTDLL.DLL"
#define NTDLL_RTLGETVERSION      "RtlGetVersion"

//- Native NT GetVersion function prototype (bypassing the WIN32 subsystem completely)
//- since the regular GetVersionEx call LIES beyond version 8, always reporting '6.2'
typedef LONG (STDAPICALLTYPE * NTVERS)(void * verData);


#define K32DLL_MODULENAME         "KERNEL32.DLL"
#define K32DLL_ISWOW64PROC        "IsWow64Process"

//- Dynamically loadable IsWow64Process, to test of (32bit) program is running on a 64-bit Windows
typedef BOOL (WINAPI * WOW64)(HANDLE process, BOOL *b64);

#elif defined (UNIX)

#if !defined (DARWIN)
// Get description for Linux/Unix distributions from /etc/xxx and other info
static int TxGetDistroDescription            // RET   length of description
(
   char               *distro                // OUT   Distro description
);
#endif

#else
   #define TXKVL       0x30000                  // length of kernel to search, 192 Kb
   #define TXKVS       "Internal revision "
   static TXTM         kernelrev = "";          // OS/2 kernel revision

// Get description for OS/2, eCS or ArcaOS distributions from various (file) info
static int TxGetOs2Description                  // RET   length of description
(
   char                bootdrive,               // IN    Boot drive
   char               *distro                   // OUT   Distro description
);

#endif



/*****************************************************************************/
// Get operatingsystem version major*minor*micro and optional descriptive text
/*****************************************************************************/
ULONG TxOsVersion                               // RET   number 000000..999999
(
   char               *descr                    // OUT   OS description or NULL
)
{
   ULONG               rc = 0;                  // function return
   TXTM                text = {0};
   #if defined(UNIX)
      size_t           at;
      struct utsname   un;
    #if defined (DARWIN)
      FILE            *fh;
      TXLN             line;
      int              count = 1;
    #endif
   #else
      double           mm = 1.0;                // major/minor double value
      #if defined (WIN32)
        TXTM          servicePack = {0};
        OSVERSIONINFOW osv;                     // RtlGetVersion is Unicode only!
      #endif
   #endif
   ULONG               major = 0;               // major version component
   ULONG               minor = 0;               // minor version component
   ULONG               micro = 0;               // micro version component

   ENTER();

   #if   defined (WIN32)
      {
         memset( &osv, 0, sizeof( OSVERSIONINFOW));
         osv.dwOSVersionInfoSize = sizeof( OSVERSIONINFO); // use ASCII version layout here

         if (GetVersionEx( (OSVERSIONINFO *) &osv)) // standard user-space version number
         {
            TRACES(("GetVersionEx, size: %u, Major:%u minor:%u build:%u\n",
                     osv.dwOSVersionInfoSize, osv.dwMajorVersion, osv.dwMinorVersion, osv.dwBuildNumber));

            TRHEXS( 111,  &osv,  sizeof(OSVERSIONINFO), "GetVersionEx osv, post call");

            major = (ULONG) osv.dwMajorVersion;
         }

         if (major > 5)                         // Win-NT up to XP is OK already but
         {                                      // GetVersionEx starts LYING after Win-8
            HMODULE    hNTdll;
            NTVERS     fnGetVersion;

            if ((hNTdll = LoadLibrary( NTDLL_MODULENAME)) != 0)
            {
               fnGetVersion = (NTVERS) GetProcAddress( hNTdll, NTDLL_RTLGETVERSION);

               if (fnGetVersion != NULL)
               {
                  memset( &osv, 0, sizeof( OSVERSIONINFOW));
                  osv.dwOSVersionInfoSize = sizeof( OSVERSIONINFOW); // RTL version is unicode!
                  (void) (fnGetVersion)(&osv);  // always returns SUCCESS!

                  TRHEXS( 111,  &osv,  sizeof(OSVERSIONINFOW) + 256, "RTL osv, post call");

                  TRACES(("RtlGetVersion size: %u, Major:%u minor:%u build:%u\n",
                           osv.dwOSVersionInfoSize, osv.dwMajorVersion, osv.dwMinorVersion, osv.dwBuildNumber));

                  major = (ULONG) osv.dwMajorVersion;
               }
               else
               {
                  TRACES(("RtlGetVersion dynamic call not found in NTDLL\n"));
               }
               FreeLibrary( hNTdll);
            }
            else
            {
               TRACES(("NTDLL not found for RtlGetVersion dynamic call\n"));
            }
         }

         if (major != 0)                        // translate RAW version to user-familiar info
         {
            TxUnicAsciiAppend( (USHORT *) osv.szCSDVersion, servicePack, 20);

            TRACES(("Report, Major:%u minor:%u build:%u PlatformId: %uServicepack: '%s'\n",
                     osv.dwMajorVersion, osv.dwMinorVersion, osv.dwBuildNumber, osv.dwPlatformId, servicePack));

            minor = (ULONG) osv.dwMinorVersion;

            mm = (double) (major);
            if      (minor <  10) mm += (double) ((double) minor) / 10.0;
            else if (minor < 100) mm += (double) ((double) minor) / 100.0;
            else                  mm += (double) ((double) minor) / 1000.0;

            strcpy( text, "Unknown-NT");
            switch (osv.dwPlatformId)
            {
               case VER_PLATFORM_WIN32s:           strcpy( text, "Win32s API"); break;
               case VER_PLATFORM_WIN32_NT:         strcpy( text, "Unknown-NT");
                  switch (major)
                  {
                     case 3:
                     case 4:
                        strcpy( text, "Windows-NT");
                        break;

                     case 5:
                        switch (minor)
                        {
                           case 00:    strcpy( text, "WinNT-2000"); break;
                           case 01:    strcpy( text, "Windows-XP"); break;
                           case 02:    strcpy( text, "WinNT-2003"); break;
                        }
                        break;

                     case 6:
                        switch (minor)
                        {
                           case 00:    strcpy( text, "Win-Vista "); break;
                           case 01:    strcpy( text, "Windows-7 "); break;
                           case 02:    strcpy( text, "Windows-8 "); break;
                           case 03:    strcpy( text, "Windows8.1"); break;
                        }
                        if (minor < 04)         //   04: Used for Windows-10 as well!
                        {
                           break;               // fall-through for 6.4 and later
                        }
                     case 10:
                     default:
                        strcpy(                text, "Windows-10");
                        switch (osv.dwBuildNumber) // translate to known release year/month
                        {
                           case 10240: strcpy( servicePack, "1507  =  2015 1st Official Release"); break;
                           case 10586: strcpy( servicePack, "1511  =  2015 November Update     "); break;
                           case 14393: strcpy( servicePack, "1607  =  2016 Anniversary Update  "); break;
                           case 15063: strcpy( servicePack, "1703  =  2017 Creators Update     "); break;
                           case 16299: strcpy( servicePack, "1709  =  2017 Fall Creators Update"); break;
                           case 17134: strcpy( servicePack, "1803  =  2018 Spring 2018 Update  "); break;
                           case 17672: strcpy( servicePack, "1809  =  2018 Fall 2018 Update    "); break;
                           default:    strcpy( servicePack, "Unknown  version, possible preview"); break;
                        }
                        break;
                  }
                  break;

               default:
                  strcpy( text, "Windows-9x");
                  major = 1;                    // force Win9x (=4.x) different from NT versions 3.x/4.x
                  break;
            }
            TRACES(("text: '%s'\n", text));
            if (descr != NULL)
            {
               if (strlen( servicePack) == 0)
               {
                  strcpy( servicePack, "No Service Pack");
               }
               if (major == 10)                 // use alternate format, with 'update' number
               {
                  sprintf( descr, "%s %s Build %lu", text, servicePack, osv.dwBuildNumber);
               }
               else
               {
                  sprintf( descr, "%s %4.2lf    build %lu: %s", text, mm, osv.dwBuildNumber, servicePack);
               }
            }
         }
      }
   #elif defined (DOS32)
      {
         union  REGS   regs;

         TxxClearReg( regs);
         regs.h.al = 0x00;                      // include OEM info
         TxxDosInt21( regs, TXDX_DOS_GETVERSION);

         major = (ULONG) regs.h.al;
         minor = (ULONG) regs.h.ah;

         mm = (double) (major);
         if      (minor <  10) mm += (double) ((double) minor) / 10.0;
         else if (minor < 100) mm += (double) ((double) minor) / 100.0;
         else                  mm += (double) ((double) minor) / 1000.0;

         if (descr)
         {
            BOOL  vm86 = (strstr( txDosExtDpmiInfo(), "VM86") != NULL);
            BOOL  mmgr = (strstr( txDosExtDpmiInfo(), ": NO") == NULL);

            if (major == 20)                    // major version, OS/2 dosbox
            {
               strcpy( text, "32bit OS/2");
            }
            else
            {
               switch (regs.h.bh)               // check OEM indicator
               {
                  case 0xff:
                     if ((mm == 5.0) && (vm86) && (!mmgr))
                     {
                        strcpy( text, "Windows-NT/2000/XP");
                     }
                     else if (mm >= 8.0)
                     {
                        strcpy( text, "Microsoft Windows-ME");
                     }
                     else if (mm >= 7.0)
                     {
                        strcpy( text, "Microsoft Windows-9x");
                     }
                     else
                     {
                        strcpy( text, "Microsoft MS-DOS");
                     }
                     break;

                  case 0x00: strcpy( text, "IBM PC-DOS");              break;
                  case 0x01: strcpy( text, "Compaq-DOS");              break;
                  case 0xee: strcpy( text, "DR-DOS");                  break;
                  case 0xef: strcpy( text, "Novell-DOS");              break;
                  case 0xfd: strcpy( text, "FreeDOS");                 break;
                  default:   strcpy( text, "DOS, unknown OEM");        break;
               }
            }
            sprintf( descr, "DOS       %5.2lf     %s %2.2hx: %s",
                     mm, (vm86) ? (mmgr) ? "MemMgr" :
                                           "DosBox" :
                                           "OemVer", regs.h.bh, text);
         }
      }
   #elif defined (UNIX)
      if (uname( &un) == -1)                    // get uname descriptions
      {                                         // only likely failure seems
         if (descr)                             // systemcall not implemented
         {
            strcpy( descr, "Unixlike, but no 'uname' info!");
         }
      }
      else
      {

         TRACES(( "sysname : '%s'\n", un.sysname));
         TRACES(( "release : '%s'\n", un.release));
         TRACES(( "version : '%s'\n", un.version));
         TRACES(( "machine : '%s'\n", un.machine));
         TRACES(( "nodename: '%s'\n", un.nodename));
         if (descr)
         {
            if ((at = strspn( un.release, ".0123456789")) != 0)
            {
               un.release[at] = 0;              // terminate after number
            }
            sprintf( descr, "%-6.6s %8s (%s) ", un.sysname, un.release, un.machine);

            //- get more descriptive OS descriptions when found
            #if defined (DARWIN)
               //- Try resolving real macOS version and add that to uname info (Darwin version)
               if ((fh = fopen( "/System/Library/CoreServices/SystemVersion.plist", "rb")) != NULL)
               {
                  while (!feof(fh) && !ferror(fh))
                  {
                     if (fgets( line, TXMAXLN, fh) != NULL)
                     {
                        TRACES(( "SystemVersion.plist line: '%s'\n", line));
                        count = sscanf(  line, "\t<string>%u.%u.%u</string>\n", &major, &minor, &micro);
                        if (count == 3)
                        {
                           break;
                        }
                     }
                  }
                  fclose( fh);
               }
               sprintf( text, "%macOS  %u.%u.%u = ", major, minor, micro);
               strcat(  descr, text);
               if (major != 10)                 // No (valid) macOS release found
               {
                  strcat( descr, "No OS-X or macOS");
               }
               else
               {
                  switch (minor)
                  {
                     case  0: strcat( descr, "Cheetah");             break;
                     case  1: strcat( descr, "Puma");                break;
                     case  2: strcat( descr, "Jaguar");              break;
                     case  3: strcat( descr, "Panther");             break;
                     case  4: strcat( descr, "Tiger");               break;
                     case  5: strcat( descr, "Leopard");             break;
                     case  6: strcat( descr, "Snow Leopard");        break;
                     case  7: strcat( descr, "Lion");                break;
                     case  8: strcat( descr, "Mountain Lion");       break;
                     case  9: strcat( descr, "Mavericks");           break;
                     case 10: strcat( descr, "Yosemite");            break;
                     case 11: strcat( descr, "El Capitan");          break;
                     case 12: strcat( descr, "Sierra");              break;
                     case 13: strcat( descr, "High Sierra");         break;
                     case 14: strcat( descr, "Mojave");              break;
                     default: strcat( descr, "New release?");        break;
                  }
               }
            #else                               // most likely a Linux distribution
               if (TxGetDistroDescription( text) != 0) // get description
               {
                  strcat( descr, text);
               }
               else
               {
                  strcat( descr, "unidentified distribution");
               }
            #endif
         }
         if (major == 0)                        // not resolved from macOS version
         {
            sscanf(  un.release, "%u.%u.%u", &major, &minor, &micro);
         }
      }
   #else
      {
         ULONG         sysinfo[QSV_VERSION_MINOR]; // major, minor, bootdrive etc

         if (DosQuerySysInfo( 1, QSV_VERSION_MINOR,  sysinfo,
                                 QSV_VERSION_MINOR * sizeof(ULONG)) == NO_ERROR)
         {
            major = sysinfo[QSV_VERSION_MAJOR -1];
            minor = sysinfo[QSV_VERSION_MINOR -1];

         }
         if (descr)
         {
            char       boot = (char) '@' + (char) sysinfo[QSV_BOOT_DRIVE-1];

            if (major == 20)                    // major version
            {
               switch (minor)                   // minor version
               {
                  case 00:  mm = 2.0;  break;
                  case 10:  mm = 2.1;  break;
                  case 11:  mm = 2.11; break;
                  case 30:  mm = 3.0;  break;
                  case 40:  mm = 4.0;  break;
                  default:  mm = 4.50; break;
               }
            }
            TxGetOs2Description( boot, text);
            sprintf( descr, "OS/2       %4.2lf  in %s", mm, text);
         }
      }
   #endif
   TRACES(("descr: '%s'\n", (descr) ? descr : ""));
   rc = (major *10000) + (minor *100) + micro;  // numeric version

   RETURN (rc);
}                                               // end 'TxOsVersion'
/*---------------------------------------------------------------------------*/

#if defined (UNIX)
 #if !defined (DARWIN)
/*****************************************************************************/
// Get description for Linux/Unix distributions from /etc/xxx and other info
/*****************************************************************************/
static int TxGetDistroDescription               // RET   length of description
(
   char               *distro                   // OUT   Distro description
)
{
   int                 rc = 0;
   FILE               *fh;
   TXLN                line;
   TXTT                pattern;
   char               *info = NULL;

   ENTER();

   *distro = 0;                                 // start empty

   if      ((fh = fopen( "/etc/os-release", "rb")) != NULL)
   {
      strcpy( pattern, "PRETTY_NAME=\"");
   }
   else if ((fh = fopen( "/etc/lsb-release", "rb")) != NULL)
   {
      strcpy( pattern, "DISTRIB_DESCRIPTION=\"");
   }

   if (fh != NULL)
   {
      while (!feof(fh) && !ferror(fh))
      {
         if (fgets( line, TXMAXLN, fh) != NULL)
         {
            TRACES(( "lsb-version line: '%s'\n", line));
            if ((info = strstr( line, pattern)) != NULL)
            {
               info += strlen( pattern);        // Distro-text starts at end of pattern
               TxCopy( distro, info, 34);       // limit to half of minimum screen width
               break;
            }
         }
      }
      fclose( fh);

      //- above works Ok for Suse Leap 42.1, Ubuntu 14.04 and Slacko Puppy
      //- to be refined, may add other specific distro files or mechanisms

      if ((info = strchr( distro, '"')) != NULL)
      {
         *info = 0;                             // terminate at ending quote
      }
      rc = strlen( distro);
   }

   TRACES(("%s\n", distro));
   RETURN (rc);
}                                               // end 'TxGetDistroDescription'
/*---------------------------------------------------------------------------*/
 #endif

#elif defined (DEV32)
/*****************************************************************************/
// Get description for OS/2, eCS or ArcaOS distributions from various (file) info
/*****************************************************************************/
static int TxGetOs2Description                  // RET   length of description
(
   char                bootdrive,               // IN    Boot drive
   char               *descr                    // OUT   Distro description
)
{
   int                 rc = 0;
   FILE               *fh;
   TXTM                fname;
   TXLN                line;
   TXTT                oname;
   TXTT                over;
   TXTT                olang;

   ENTER();

   strcpy( descr, "ArcaOS");
   sprintf( fname, "%c:\\sys\\install\\install.flg", bootdrive);
   fh = fopen( fname, "rb");
   if (fh == NULL)
   {
      strcpy( descr, "eComStation");
      sprintf( fname, "%c:\\ecs\\\\ecs_inst.flg", bootdrive);
      fh = fopen( fname, "rb");
   }
   if (fh != NULL)
   {
      if (fgets( line, TXMAXLN, fh) != NULL)
      {
         TRACES(( "install.flg line: '%s'\n", line));

         TxRepl( line, '\n', '\0');             // remove newline CR/LF
         strcat( line, " word");                // make sure there are more than 3 'words'

         sscanf( line, "%s %s %s *%s", oname, over, olang); // get first 3 words only

         sprintf( descr, "%s %s %s",  oname, over, olang);
      }
      fclose( fh);
   }
   else
   {
      strcpy( descr, "IBM OS/2 distribution");
   }

   TRACES(("%s\n", descr));
   RETURN (rc);
}                                               // end 'TxGetOs2Description'
/*---------------------------------------------------------------------------*/
#endif

/*****************************************************************************/
// Get OS additional info like terminal/user identification or kernel details
/*****************************************************************************/
ULONG TxOsAdditionalInfo                        // RET   NO_ERROR if info
(
   char               *leading,                 // IN    leading text (spaces)
   char               *descr                    // OUT   additional OS info
)
{
   ULONG               rc = TX_FAILED;          // function return

   ENTER();

   #if   defined (WIN32)
      {                                         // user/computername info
         TXLN          username;
         TXLN          computer;
         ULONG         namelen = TXMAXLN;

         if (GetUserName( username, &namelen))
         {
            username[40] = 0;                   // truncate name at 40 positions
            namelen = TXMAXLN;
            if (GetComputerName( computer, &namelen))
            {
               HMODULE hK32dll;
               WOW64   fnWinOnWin64;
               BOOL    on64bit = FALSE;

               if ((hK32dll = LoadLibrary( K32DLL_MODULENAME)) != 0)
               {
                  fnWinOnWin64 = (WOW64) GetProcAddress( hK32dll, K32DLL_ISWOW64PROC);

                  if (fnWinOnWin64 != NULL)
                  {
                     if ((fnWinOnWin64)(GetCurrentProcess(), &on64bit))
                     {
                        TRACES(("On 64 bit: %s\n", (on64bit) ? "yes" : "no"));
                     }
                     else
                     {
                        TRACES(("Error in " K32DLL_ISWOW64PROC "\n"));
                     }
                  }
                  else
                  {
                     TRACES(( K32DLL_ISWOW64PROC " dynamic call not found in " K32DLL_MODULENAME "\n"));
                  }
                  FreeLibrary( hK32dll);
               }
               else
               {
                  TRACES(( K32DLL_MODULENAME " not found for " K32DLL_ISWOW64PROC "dynamic call\n"));
               }

               sprintf( descr, "%sCurrent user : %s, on %s, running %s-bit Windows\n",
                        leading, username, computer, (on64bit) ? "64" : "32");
               rc = NO_ERROR;                   // signal info available
            }
         }
      }
   #elif defined (DOS32)                        // show Dos-extender DPMI info
      {
         sprintf( descr,  "%sDPMI version : %s\n", leading, txDosExtDpmiInfo());
      }
   #elif defined (UNIX)
      {
         char            *env;
         struct utsname   un;

         sprintf( descr, "%sCurrent user : ", leading);

         if ((env = getenv("SUDO_USER")) != NULL)
         {
            strcat( descr, "SUDO: ");
            strcat( descr, env);
         }
         else if ((env = getenv("LOGNAME")) != NULL)
         {
            strcat( descr, env);
         }
         else
         {
            strcat( descr, "someone");
         }

         if (uname( &un) != -1)                 // get uname descriptions
         {
            strcat( descr, " on ");
            strcat( descr, un.nodename);
         }

         env = getenv("TERM");                  // get terminal identification
         strcat( descr, " using TERM=");
         strcat( descr, (env != NULL) ? env : "unknown");
         strcat( descr, "\n");

         rc = NO_ERROR;                         // signal info available
      }
   #else                                        // OS/2, supply kernel detail & boot drive
      {
         TXTM          text;
         ULONG         sysinfo[QSV_VERSION_MINOR]; // major, minor, bootdrive etc

         if (DosQuerySysInfo( 1, QSV_VERSION_MINOR,  sysinfo,
                                 QSV_VERSION_MINOR * sizeof(ULONG)) == NO_ERROR)
         {
            if ((strlen(kernelrev) == 0) || (TxaOption('r')))
            {
               TXLN          kernel;
               FILE         *fp;
               char          boot = (char) '@' + (char) sysinfo[QSV_BOOT_DRIVE-1];

               sprintf( kernel, "%c:\\os2krnl", boot);
               if ((fp = fopen( kernel, "rb")) == NULL)
               {
                  strcat( kernel, "i");         // alternative (floppy) name
                  fp = fopen( kernel, "rb");
               }
               if (fp != NULL)
               {
                  BYTE *buf     = TxAlloc( 1, TXKVL);
                  char *kver;

                  if (buf != NULL)
                  {
                     if (fread( buf, 1, TXKVL, fp) != 0)
                     {
                        if ((kver = TxMemStr( buf, TXKVS, TXKVL)) != NULL)
                        {
                           memcpy( text, kver + strlen(TXKVS), 10);
                           text[10] = 0;
                           TRACES(( "kernel string: '%s'\n", text));

                           for (kver = text; *kver; kver++)
                           {
                              if (!isprint(*kver))
                              {
                                 *kver = 0;
                              }
                           }
                           sprintf( kernelrev, "Internal revision:  %s on drive %c:", text, boot);
                        }
                     }
                     TxFreeMem( buf);
                  }
                  fclose( fp);
               }
               if (strlen(kernelrev) == 0)
               {
                  ULONG   major = sysinfo[QSV_VERSION_MAJOR -1];
                  ULONG   minor = sysinfo[QSV_VERSION_MINOR -1];

                  sprintf( kernelrev, "Internal version :  %u.%u on drive %c:", major, minor, boot);
               }
            }
         }
         if (strlen( kernelrev) != 0)
         {
            sprintf( descr, "%sOS/2  Kernel : %s\n", leading, kernelrev);
            rc = NO_ERROR;                      // signal info available
         }
      }
   #endif

   TRACES(("descr: '%s'\n", descr));
   RETURN (rc);
}                                               // end 'TxOsAdditionalInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return 15-character TXLIB version string
/*****************************************************************************/
char  *txVersionString                          // RET   15-char version string
(
   void
)
{
   sprintf( txVersion, "%s %s", TXLIB_V, TXLIB_C);
   return(  txVersion);
}                                               // end 'txVersionString'
/*---------------------------------------------------------------------------*/

#if defined (DOS32)

/*****************************************************************************/
// Return DOSEXTENDER version string
/*****************************************************************************/
char  *txDosExtVersion                          // RET   version string
(
   void
)
{
   #if defined (CAUSEWAY)
      return ("Causeway   3.52 (c) 1992-2000: M.E. Devore");
   #else
      return ("DOS32A     7.20 (c) 1996-2002: DOS/32 Advanced Team");
   #endif
}                                               // end 'txDosExtVersion'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return DOSEXTENDER DPMI info string
/*****************************************************************************/
char  *txDosExtDpmiInfo                         // RET   DPMI info string
(
   void
)
{
   union  REGS         regs;
   struct SREGS        sreg;

   memset( &regs, 0, sizeof(regs));
   memset( &sreg, 0, sizeof(sreg));
   regs.w.ax = TXDX_DPMI_VERS;

   txDpmiCall( &regs, &sreg)

   sprintf( exVersion, "%1.1hu.%02.2hu switchmode %s     swap: %s",
            regs.h.ah, regs.h.al,
           (regs.w.bx & 0x02) ? "REAL" : "VM86",
           (regs.w.bx & 0x04) ? "NO"   : "YES");
   return( exVersion);
}                                               // end 'txDosExtDpmiInfo'
/*---------------------------------------------------------------------------*/

#endif
