//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2005 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
// This file contains Original Code and/or Modifications of Original Code as
// defined in and that are subject to the GNU Lesser General Public License.
// You may not use this file except in compliance with the License.
// BY USING THIS FILE YOU AGREE TO ALL TERMS AND CONDITIONS OF THE LICENSE.
// A copy of the License is provided with the Original Code and Modifications,
// and is also available at http://www.dfsee.com/txwin/lgpl.htm
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License,
// or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; (lgpl.htm) if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// Questions on TxWin licensing can be directed to: txwin@fsys.nl
//
// ==========================================================================
//
// Operating system specific functions
//
// Author: J. van Wijk
//
// JvW  24-07-2005 Initial version, split off from TXUTIL

#include <txlib.h>
#include <txwpriv.h>                            // private interface, txwa

#if   defined (DOS32)

/*****************************************************************************/
// Wait for specified nr of msec (approximation, +/- 32 msec)
/*****************************************************************************/
void TxBusyWait
(
   ULONG               msec                     // IN    nr of msec to wait
)
{
   TXTIMER             busy = TxTmrSetTimer( TMRMSEC( msec));

   ENTER();

   while (TxTmrTimerExpired( busy) == FALSE)
   {
      ;                                         // do nothing, just wait
   }
   VRETURN();
}                                               // end 'TxBusyWait'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set keyboard mapping using FreeDOS keyb and .kl files, optional codepage
/*****************************************************************************/
ULONG TxSetNlsKeyboard                          // RET   result
(
   char               *spec,                    // IN    keyb file basename
   char               *cp                       // IN    codepage string or ""
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                line;                    // full-path to kl file
   TXTM                keyb;                    // kl filename / keyb cmd
   FILE               *kl;                      // .kl file

   ENTER();

   sprintf( keyb, "key\\%s.kl", spec);
   if ((kl = TxFindAndOpenFile( keyb, "PATH", line)) == NULL)
   {
      sprintf( keyb, "%s.kl", spec);
      kl = TxFindAndOpenFile( keyb, "PATH", line);
   }
   if (kl != NULL)
   {
      fclose( kl);
      TxExternalCommand(   "keyb /u");          // unload
      sprintf( keyb, "keyb %2.2s,%s,%s", spec, cp, line);
      TxExternalCommand(    keyb);              // set new
      TxExternalCommand(   "keyb");             // show
   }
   else
   {
      rc = ERROR_FILE_NOT_FOUND;
   }
   RETURN (rc);
}                                               // end 'TxSetNlsKeyboard'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Allocate DPMI compatible memory
/*****************************************************************************/
void *txDpmiAlloc                               // RET   PM linear address
(
   size_t              nr,                      // IN    number of items
   size_t              size,                    // IN    size per item
   short              *selector                 // OUT   PM selector
)
{
   void               *rc = NULL;
   union  REGS         regs;
   struct SREGS        sreg;

   ENTER();
   TRARGS(("nr of items: %d  size: %d\n", nr, size));

   if ((nr != 0) && (size != 0))
   {
      memset( &regs, 0, sizeof(regs));
      memset( &sreg, 0, sizeof(sreg));

      regs.w.ax = TXDX_DPMI_ALLOC;
      regs.w.bx = (USHORT) (((ULONG) (nr * size) -1) / 16) +1; // 16 byte paragraphs

      TRACES(("Alloc DPMI memory, paragraphs: %4.4hx\n", regs.w.bx));

      txDpmiCall( &regs, &sreg)

      if (regs.x.cflag == 0)
      {
         *selector   =  regs.w.dx;
         rc = (char *) (regs.w.ax << 4);
      }
      TRACES(("seg: %4.4hx  sel: %4.4hx  cflag: %hu\n",
               regs.w.ax, regs.w.dx, regs.x.cflag));
   }
   RETURN (rc);
}                                               // end 'txDpmiAlloc'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Free DPMI compatible memory
/*****************************************************************************/
void txDpmiFree
(
   short               selector                 // IN    PM selector
)
{
   union  REGS         regs;
   struct SREGS        sreg;

   ENTER();

   if (selector != 0)
   {
      memset( &regs, 0, sizeof(regs));
      memset( &sreg, 0, sizeof(sreg));

      TRACES(("free  DPMI memory, selector: %4.4hx\n", selector));
      regs.w.ax = TXDX_DPMI_FREEM;
      regs.w.dx = selector;
      txDpmiCall( &regs, &sreg)
   }
   VRETURN();
}                                               // end 'txDpmiFree'
/*---------------------------------------------------------------------------*/

#elif defined (WIN32)

/*****************************************************************************/
// Format and return (static allocated) text corresponding to last Win32 error
/*****************************************************************************/
char *txNtLastError                             // RET   WIN error text
(
   void
)
{
   static TX1K         ntLastError;
   TX1K                ertext;
   DWORD               error = GetLastError();

   FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM |  // from LastError
                  FORMAT_MESSAGE_IGNORE_INSERTS,
                  NULL, error,
                  MAKELANGID( LANG_NEUTRAL, SUBLANG_DEFAULT),
                  (LPTSTR) ertext, TXMAX1K,     // resulting buffer
                   NULL);                       // no arguments

   if (ertext[ strlen(ertext) -1] == '\n')
   {
      ertext[ strlen(ertext) -1] = 0;           // remove end-of-line from error text
   }
   sprintf( ntLastError, "%d = %s", (int) error, ertext);
   return ( ntLastError);
}                                               // end 'txNtLastError'
/*---------------------------------------------------------------------------*/



#elif defined (UNIX)

/*****************************************************************************/
// Sleep for specified nr of msec
/*****************************************************************************/
void TxSleepMsec
(
   ULONG               msec                     // IN    nr of msec to wait
)
{
   struct timespec     requested;
   struct timespec     remaining;               // after EINTR from a signal

   requested.tv_sec  =  msec / 1000;
   requested.tv_nsec = (msec % 1000) * 1000000;

   nanosleep( &requested, &remaining);
}                                               // end 'TxSleepMsec'
/*---------------------------------------------------------------------------                    */

/*************************************************************************************************/
// Translate (Linux) errno value to corresponding Application + DOS/OS2/WIN like RC
/*************************************************************************************************/
ULONG TxRcFromErrno                             // RET    TXW return-code
(
   int                 err                      // IN    error number (errno)
)
{
   ULONG               rc;                      // function return

   ENTER();

   switch (errno)
   {
      case 0:                          rc = NO_ERROR;                      break;
      case ENOENT:                     rc = ERROR_FILE_NOT_FOUND;          break;
      case EACCES:                     rc = ERROR_ACCESS_DENIED;           break;
      case EEXIST:                     rc = ERROR_FILE_EXISTS;             break;
      case EMFILE:    case ENFILE:     rc = ERROR_TOO_MANY_OPEN_FILES;     break;
      case ENXIO:     case ENODEV:
      case ENOTDIR:   case EISDIR:     rc = ERROR_INVALID_DRIVE;           break;
      case EFBIG:                      rc = ERROR_WRITE_FAULT;             break;
      case ENOSPC:                     rc = ERROR_DISK_FULL;               break;
      case EINVAL:                     rc = ERROR_INVALID_PARAMETER;       break;
      case ENAMETOOLONG:               rc = ERROR_FILENAME_EXCED_RANGE;    break;
      case EBADF:                      rc = ERROR_INVALID_HANDLE;          break;
      case ENOSYS:                     rc = ERROR_INVALID_PARAMETER;       break;
      case ENOMEM:                     rc = TX_ALLOC_ERROR;                break;
      default:                         rc = ERROR_GEN_FAILURE;             break;
   }
   TRACES(( "Translated errno value: %d to RC: %lu\n", errno, rc));
   RETURN (rc);
}                                               // end 'TxRcFromErrno'
/*-----------------------------------------------------------------------------------------------*/

#else                                           // OS/2

// OS2-PM menu-ID for VIO window system-menu 'Mouse Actions' and DragText 'Mouse Marking'
#define SYSMENU_MOUSE_ACTIONS   0x00a3
#define DRAGTXT_MOUSE_MARKING   0x5447


   static BOOL          txPfnApiTested = FALSE;
   static TXF_OS2DLLAPI txPfnApi =
   {
      NULL, NULL, NULL,                         // Large File APIs (JvW)
      NULLHANDLE, NULLHANDLE, 0,                // PM handles and hwnd
      NULL, NULL, NULL, NULL, NULL, NULL,       // PM-WIN  initializing
      NULL, NULL, NULL, NULL,                   // Switchlist
      NULL, NULL, NULL, NULL, NULL,             // Windowing
      NULL, NULL, NULL, NULL, NULL, NULL        // Clipboard
   };


   #define TXF_DOSCALLS    ((PSZ)  "DOSCALLS")   // name of dll with DOS calls
   #define TXF_DOSOPEN_ORD ((ULONG)       981)
   #define TXF_DOSSEEK_ORD ((ULONG)       988)
   #define TXF_DOSSFSZ_ORD ((ULONG)       989)

   #define TXF_PMCTLS      ((PSZ)  "PMCTLS"  )   // name of dll with PM controls
   #define TXF_PMWIN       ((PSZ)  "PMWIN"   )   // name of dll with Window functions
   #define TXF_PMSHAPI     ((PSZ)  "PMSHAPI" )   // name of dll with PM-Shell functions

/*****************************************************************************/
// Test if OS2 large-file support (> 2GiB) is available; Optional return API's
// When possible, morph process-type to PM, so we can set the Window Title
/*****************************************************************************/
BOOL TxInitDllApiOS2                            // RET   large file API's OK
(
   TXF_OS2DLLAPI      **entrypoints             // OUT   LF-API entrypoints
)                                               //       or NULL
{
   BOOL                rc = FALSE;              // function return
   TXLN                dlmerror;                // one line of data
   HMODULE             hMod = 0;                // DLL module handle

   ENTER();

   if (txPfnApiTested == FALSE)
   {
      if ((DosLoadModule( dlmerror, TXMAXLN, TXF_DOSCALLS, &hMod)) == NO_ERROR)
      {
         DosQueryProcAddr( hMod, TXF_DOSOPEN_ORD, NULL, (PFN *) &(txPfnApi.DosOpenL));
         DosQueryProcAddr( hMod, TXF_DOSSEEK_ORD, NULL, (PFN *) &(txPfnApi.DosSeekL));
         DosQueryProcAddr( hMod, TXF_DOSSFSZ_ORD, NULL, (PFN *) &(txPfnApi.DosSfSzL));

         TRACES(( "DosOpenL : %p  DosSeekL : %p  DosSfSzL : %p\n", txPfnApi.DosOpenL, txPfnApi.DosSeekL, txPfnApi.DosSfSzL));
      }
      else
      {
         TRACES(("Error loading DLL: '%s'\n", TXF_DOSCALLS));
      }

      if ( DosQueryModuleHandle( "PMCTLS", &hMod ) == NO_ERROR )
      {
         if ((DosLoadModule( dlmerror, TXMAXLN, TXF_PMSHAPI,  &hMod)) == NO_ERROR)
         {
            void    * _Seg16 pWork16;                //- intermediate storage to preserve 16 bitness
            ULONG   *pulW16 = (PULONG) &pWork16;
            PFN      pfnWSTAI32;

            DosQueryProcAddr( hMod, ORD_WINSETTITLEANDICON, NULL, (PFN *) &pfnWSTAI32);
            pWork16           = (SHORT (_Far16 *)()) pfnWSTAI32;
            txPfnApi.pfnWSTAI = (TXF_WSTAI) *pulW16; // WATCOM Kludge to get proper 16-bit pointer

            DosQueryProcAddr( hMod, ORD_WIN32QUERYSWITCHLIST,     NULL, (PFN *) (&txPfnApi.pfnWQSL));
            DosQueryProcAddr( hMod, ORD_WIN32QUERYSWITCHENTRY,    NULL, (PFN *) (&txPfnApi.pfnWQSE));
            DosQueryProcAddr( hMod, ORD_WIN32QUERYSWITCHHANDLE,   NULL, (PFN *) (&txPfnApi.pfnWQSH));
            DosQueryProcAddr( hMod, ORD_WIN32CHANGESWITCHENTRY,   NULL, (PFN *) (&txPfnApi.pfnWCSE));

            TRACES(( "QswList  : %p  QswEntry : %p  ChangeSwE: %p\n", txPfnApi.pfnWQSL, txPfnApi.pfnWQSE, txPfnApi.pfnWCSE));
         }
         else
         {
            TRACES(("Error loading DLL: '%s'\n", TXF_PMSHAPI));
         }

         if ((DosLoadModule( dlmerror, TXMAXLN, TXF_PMWIN,    &hMod)) == NO_ERROR)
         {
            DosQueryProcAddr( hMod, ORD_WIN32INITIALIZE,          NULL, (PFN *) &(txPfnApi.pfnWINIT));
            DosQueryProcAddr( hMod, ORD_WIN32TERMINATE,           NULL, (PFN *) &(txPfnApi.pfnWTERM));
            DosQueryProcAddr( hMod, ORD_WIN32CREATEMSGQUEUE,      NULL, (PFN *) &(txPfnApi.pfnWCMQ ));
            DosQueryProcAddr( hMod, ORD_WIN32DESTROYMSGQUEUE,     NULL, (PFN *) &(txPfnApi.pfnWDMQ ));
            DosQueryProcAddr( hMod, ORD_WIN32CANCELSHUTDOWN,      NULL, (PFN *) &(txPfnApi.pfnWCS  ));

            DosQueryProcAddr( hMod, ORD_WIN32POSTMSG,             NULL, (PFN *) &(txPfnApi.pfnWPM  ));
            DosQueryProcAddr( hMod, ORD_WIN32SENDMSG,             NULL, (PFN *) &(txPfnApi.pfnWSM  ));
            DosQueryProcAddr( hMod, ORD_WIN32QUERYWINDOW,         NULL, (PFN *) &(txPfnApi.pfnWQW  ));
            DosQueryProcAddr( hMod, ORD_WIN32WINDOWFROMID,        NULL, (PFN *) &(txPfnApi.pfnWWFID));
            DosQueryProcAddr( hMod, ORD_WIN32SETWINDOWTEXT,       NULL, (PFN *) &(txPfnApi.pfnWSWT ));

            DosQueryProcAddr( hMod, ORD_WIN32OPENCLIPBRD,         NULL, (PFN *) &(txPfnApi.pfnWOC  ));
            DosQueryProcAddr( hMod, ORD_WIN32EMPTYCLIPBRD,        NULL, (PFN *) &(txPfnApi.pfnWEC  ));
            DosQueryProcAddr( hMod, ORD_WIN32SETCLIPBRDDATA,      NULL, (PFN *) &(txPfnApi.pfnWSCD ));
            DosQueryProcAddr( hMod, ORD_WIN32CLOSECLIPBRD,        NULL, (PFN *) &(txPfnApi.pfnWCC  ));
            DosQueryProcAddr( hMod, ORD_WIN32QUERYCLIPBRDFMTINFO, NULL, (PFN *) &(txPfnApi.pfnWQCFI));
            DosQueryProcAddr( hMod, ORD_WIN32QUERYCLIPBRDDATA,    NULL, (PFN *) &(txPfnApi.pfnWQCD ));

            TRACES(( "WinInit  : %p  SetWinTxt: %p  OpenClipB: %p\n", txPfnApi.pfnWOC, txPfnApi.pfnWSWT, txPfnApi.pfnWINIT));

            if (txPfnApi.pfnWINIT != NULL)      // try to morph to a PM program in-memory
            {

               if ((txPfnApi.hHAB = (txPfnApi.pfnWINIT)( 0)) != NULLHANDLE)
               {
                  TxSessionMorph2PM( TRUE);

                  //- Create a message queue (required for WinSetWindowText) and get our frame-window hwnd
                  if ((txPfnApi.hHMQ = (txPfnApi.pfnWCMQ)( txPfnApi.hHAB, 0)) != NULLHANDLE)
                  {
                     (txPfnApi.pfnWCS)( txPfnApi.hHMQ, TRUE); // Cancel Shutdown msg delivery

                     if (txPfnApi.pfnWQSH && txPfnApi.pfnWQSE)
                     {
                        SWCNTRL         swctl;
                        HSWITCH         hswitch;

                        memset( &swctl, 0, sizeof( SWCNTRL));
                        if ((hswitch = (txPfnApi.pfnWQSH)( 0, txwa->pib->pib_ulpid)) != NULLHANDLE)
                        {
                           if ((txPfnApi.pfnWQSE)( hswitch, &swctl) == NO_ERROR)
                           {
                              txPfnApi.hwndFrame = swctl.hwnd; // fast access to frame hwnd
                           }
                           else
                           {
                              TRACES(("QuerySwitchEntry failed\n"));
                           }
                        }
                        else
                        {
                           TRACES(("QuerySwitchHandle failed\n"));
                        }
                     }
                     TRACES(("Morphed into a PM process, type %u -> %u, hwnd: %8.8lx\n",
                              txwa->session, PT_PM, txPfnApi.hwndFrame));
                  }
                  else
                  {
                     TxSessionMorph2PM( FALSE);             // undo morph to PM
                     TRACES(("Failed to create Message-Queue, no PM-morph!\n"));
                  }
               }
               else
               {
                  TRACES(("Failed to get HAB, no PM-morph!\n"));
               }
            }
         }
         else
         {
            TRACES(("Error loading DLL: '%s'\n", TXF_PMWIN));
         }
      }
      else
      {
         TRACES(("PMCTLS dll is not present, so PM is NOT active, no morphing (SetTitle)\n"));
      }
      txPfnApiTested = TRUE;
   }

   if (entrypoints != NULL)
   {
      *entrypoints = &txPfnApi;
   }

   if ((txPfnApi.DosOpenL != NULL) && (txPfnApi.DosSeekL != NULL)  )
   {
      rc = TRUE;
   }
   BRETURN( rc);
}                                               // end 'TxInitDllApiOS2'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate the Morph-to-PM situation
/*****************************************************************************/
VOID TxTermDllApiOS2
(
   void
)
{
   ENTER();

   if ((txPfnApi.pfnWDMQ  != NULL) && (txPfnApi.hHMQ != NULLHANDLE))
   {
      ( txPfnApi.pfnWDMQ )(txPfnApi.hHMQ);
   }
   if ((txPfnApi.pfnWTERM != NULL) && (txPfnApi.hHAB != NULLHANDLE))
   {
      ( txPfnApi.pfnWTERM)( txPfnApi.hHAB);
   }
   VRETURN( rc);
}                                               // end 'TxTermDllApiOS2'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set sessiontype to Morphed-to-PM, or return to original VIO/FS sessiontype
// While running normal, the library is morphed-to-PM, make sure to morph
// back temporarily when running other (VIO) programs (system() call etc)
/*****************************************************************************/
VOID TxSessionMorph2PM
(
   BOOL                toPM                     // IN    Morph TO PM, not back
)
{
   ENTER();

   if (toPM)
   {
      txwa->session = txwa->pib->pib_ultype;    // save original type VIO/FS
      txwa->pib->pib_ultype = PT_PM;            // morph to PM process type
   }
   else
   {
      txwa->pib->pib_ultype = txwa->session;    // undo morph to PM
   }
   VRETURN( rc);
}                                               // end 'TxSessionMorph2PM'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Query current setting, and SET OS/2 system menu 'Mouse Actions' ON or OFF
/*****************************************************************************/
TXSETVALUE TxSysMenuMouseActions                // RET   OS-mouse-marking is ON
(
   TXSETVALUE          value                    // IN    New value or query-only
)
{
   TXSETVALUE          rc      = TX_UNKNOWN;    // function return

   ENTER();
   TRACES(("value: %d\n", value));

   if (txwa->api && txwa->api->pfnWPM && txwa->api->pfnWSM && txwa->api->pfnWSM)
   {
      HWND    sysmenu = (txwa->api->pfnWWFID)( txwa->api->hwndFrame, FID_SYSMENU);
      ULONG   query;

      //- SEND message to the system-menu child window, to query the menu state, incl submenus
      query = (ULONG) (txwa->api->pfnWSM)( sysmenu, MM_QUERYITEMATTR,
                                           MPFROM2SHORT( SYSMENU_MOUSE_ACTIONS, TRUE),
                                           MPFROMSHORT(  MIA_CHECKED));

      TRACES(( "query value: %8.8x\n", query));

      if (query & MIA_CHECKED)
      {
         rc = TX_CHECKED;
      }
      else
      {
         rc = TX_UNCHECKED;
      }

      if ((value != TX_QUERY) && (value != rc)) // need to toggle it?
      {
         //- POST message to the frame window, to toggle the menu state
         (txwa->api->pfnWPM)( txwa->api->hwndFrame, WM_SYSCOMMAND,
                              MPFROMSHORT( SYSMENU_MOUSE_ACTIONS),
                              MPFROMSHORT( CMDSRC_MENU));
      }
   }
   else
   {
      TRACES(("PM environment not available!\n"));
   }
   RETURN (rc);
}                                               // end 'TxSysMenuMouseActions'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Query current setting, and SET OS/2 DragText menu 'Mouse Marking' ON or OFF
/*****************************************************************************/
TXSETVALUE TxDragTextMouseMarking               // RET   DT-mouse-marking is ON
(
   TXSETVALUE          value                    // IN    New value or query-only
)
{
   TXSETVALUE          rc      = TX_UNKNOWN;    // function return

   ENTER();
   TRACES(("value: %d\n", value));

   if (txwa->api && txwa->api->pfnWPM && txwa->api->pfnWSM && txwa->api->pfnWSM)
   {
      HWND    sysmenu = (txwa->api->pfnWWFID)( txwa->api->hwndFrame, FID_SYSMENU);
      ULONG   query;

      //- SEND message to the system-menu child window, to query the menu state, incl submenus
      query = (ULONG) (txwa->api->pfnWSM)( sysmenu, MM_QUERYITEMATTR,
                                           MPFROM2SHORT( DRAGTXT_MOUSE_MARKING, TRUE),
                                           MPFROMSHORT(  MIA_CHECKED));

      TRACES(( "query value: %8.8x\n", query));

      if (query & MIA_CHECKED)
      {
         rc = TX_CHECKED;
      }
      else
      {
         rc = TX_UNCHECKED;
      }

      if ((value != TX_QUERY) && (value != rc)) // need to toggle it?
      {
         //- POST message to the frame window, to toggle the menu state
         (txwa->api->pfnWPM)( txwa->api->hwndFrame, WM_COMMAND,
                              MPFROMSHORT( DRAGTXT_MOUSE_MARKING),
                              MPFROMSHORT( CMDSRC_MENU));
      }
   }
   else
   {
      TRACES(("PM environment not available!\n"));
   }
   RETURN (rc);
}                                               // end 'TxDragTextMouseMarking'
/*---------------------------------------------------------------------------*/

#endif
