//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2005 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
// This file contains Original Code and/or Modifications of Original Code as
// defined in and that are subject to the GNU Lesser General Public License.
// You may not use this file except in compliance with the License.
// BY USING THIS FILE YOU AGREE TO ALL TERMS AND CONDITIONS OF THE LICENSE.
// A copy of the License is provided with the Original Code and Modifications,
// and is also available at http://www.dfsee.com/txwin/lgpl.htm
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License,
// or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; (lgpl.htm) if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// Questions on TxWin licensing can be directed to: txwin@fsys.nl
//
// ==========================================================================
//
// TX external command execution
//
// Author: J. van Wijk
//
// 24-07-2005  Restyled for TXWIN open version
// 20-05-2017  Allow file-redirect instead of pipes for OS/2 as well

#include <txlib.h>                              // TX library interface
#include <txwpriv.h>                            // TX private interfaces


#if defined (UNIX)
// #define TXC_TEMP_FILE_NAME "~/.txlibtemp"    // fails to open after creation ???
   #define TXC_TEMP_FILE_NAME "/tmp/_-!TX!-O.TMP"
   #define TXC_ITMP_FILE_NAME "/tmp/_-!TX!-I.TMP"
#else
   #define TXC_TEMP_FILE_NAME "_-!TX!-O.TMP"
   #define TXC_ITMP_FILE_NAME "_-!TX!-I.TMP"
#endif


#if !defined (DEV32)

// Execute, using FILE redirected stdout/stderr to TxPrint
static ULONG txcExecStdOutFile
(
   char               *cmd                      // IN    command to execute
);


#else

// temporary hack to het KBD handle (move IOCTl to txwmsg.c later)
extern HFILE   txw_hkeyboard;


#define TXC_PIPE_SIZE      16384                // size of output pipe
#define TXC_NEW_HANDLE     0xFFFFFFFF           // new from DosDup...
#define TXC_MAX_TIMEOUTS   9                    // max nr of receive timeouts
#define TXC_EOF            ((char) 0x1a)        // end-of-file indicator
#define TXC_CR             ((char) 0x0d)        // carriage return
#define TXC_LF             ((char) 0x0a)        // linefeed

#define STDIN              0
#define STDOUT             1
#define STDERR             2


typedef struct txc_io_handles
{
   TXHFILE             read;
   TXHFILE             write;
   TXHFILE             save;
   char                ind;                     // progress indicator
} TXC_IO_HANDLES;                               // end of struct


// Execute an external command, using PIPE redirected stdout/stderr to TxPrint
ULONG txcExecStdOutPipe
(
   char               *cmd                      // IN    command to execute
);

// TXC child-stdout thread, waiting for output by child-process; to TxPrint
static void txcPipeOutputPump
(
   void               *p                        // IN    pointer parameter
);

// Write to stream (redirected stdout) with logfile (TEE) and output hooks
// Includes SB filter for WGET, adds ANSI 'cursor-up' after each progress line
static void TxWriteStream
(
   TXHFILE             stream,                  // IN    stream to write to
   DEVICE_STATE        logging,                 // IN    log state to be restored
   char               *text                     // IN    string to write
);

#endif

#if defined (UNIX)
static char txcTrustedExternals[] = " LS LL CD RM CHMOD CAT CLEAR CP SU MV DU DF WGET UNZIP ZIP "
 #if defined (DARWIN)
                                    " DISKUTIL ";
 #else
                                    " MOUNT UMOUNT LSBLK ";
 #endif
#else
static char txcTrustedExternals[] = " DIR CD DEL REN RENAME ATTRIB SET TYPE MODE CLS CHCP WGET UNZIP ZIP "
 #if defined (DOS32)
                                    " MEM KEYB "
 #else
                                    " HELP MOVE "
 #endif
                                    " COPY ";
#endif




/*****************************************************************************/
// Execute an external command, redirect stdout/stderr to TxPrint
/*****************************************************************************/
ULONG TxExternalCommand
(
   char               *cmd                      // IN    command to execute
)
{
   ULONG               rc  = NO_ERROR;
   TXTS                first;                   // first word
   char               *s;

   ENTER();

   memset( first, ' ',    TXMAXTS);
   TxCopy( first +1, cmd, TXMAXTS -1);
   if ((s = strchr( first +1, ' ')) != NULL)
   {
      *(s+1) = 0;
   }
   TxStrToUpper( first);                        // uppercased first word

   TRACES(( "first: '%s' for cmd: '%s'\n", first, cmd));

   if ((TxaExeSwitch('b')) || (TxaOption('B'))       ||
       (strstr( txcTrustedExternals, first) != NULL) ||
       (TxConfirm( TXCM_HELPREG + 00, "%s%s%s is not a trusted command!\n"
                   "It could be an incorrectly typed internal command ...\n\n"
                   "Are you sure you want to have :\n\n%s%s%s\n\nexecuted as "
                   "an external command ? [Y/N] : ", CBR, first +1, CNN, CBG, cmd, CNN)))
   {
      #if defined (DEV32)
         #if defined (USEWINDOWING)
            txwa->KbdKill = TRUE;               // quit KBD thread (STDIN is needed!)

            TxSleep( 100);                      // Make sure KBD thread is gone ...

            rc = txcExecStdOutPipe( cmd);       // simple std IO used, using pipe

            TxInputDesktopInit();               // restart KBD thread
         #else
            rc = txcExecStdOutPipe( cmd);       // simple std IO used, using pipe
         #endif
      #else
         rc = txcExecStdOutFile( cmd);          // redirect using TEMP file
      #endif
   }
   #if defined (USEWINDOWING)
      if (txwIsWindow( TXHWND_DESKTOP))
      {
         txwInvalidateAll();                    // repaint windows, in case
      }                                         // the screen was 'damaged'
   #endif                                       // USEWINDOWING
   #if defined (UNIX)
      if (rc == (ULONG) -1)
      {
         rc = TX_UNIX_ERROR;                    // specific 16-bit error value
      }
      else if ((rc == ERROR_BAD_SHELL)     ||   // DARWIN only ? translate
               (rc == ERROR_CMD_NOT_FOUND) ||   // command not found
               (rc == ERROR_CMD_NOT_EXEC)   )   // command not executable
      {
         rc = ERROR_NOT_A_CMD;                  // use same value as Win or OS/2
      }
   #elif defined (WIN32)                        // Windows
      if ((rc == ERROR_BAD_EXT_CMD) ||          // bad commands with redirection
          (rc == 229L             ) ||          // invalid external cmd or parameter
          (rc == 9009L            )  )          // often used instead of 1041 (DNS auth)
      {
         rc = ERROR_NOT_A_CMD;                  // use same value as OS/2
      }
   #else
      if (rc == ERROR_BAD_EXT_CMD)              // RC 1, general command failure
      {
         rc = TX_EXT_CMD_ERROR;                 // some error on external command
      }
   #endif
   RETURN (rc & 0xffff);                        // 16-bit, preserve flag values
}                                               // end 'TxExternalCommand'
/*---------------------------------------------------------------------------*/


#define TXMAX1MIB ((ULN64)(1024*1024))
/*****************************************************************************/
// Execute an external command, using redirected stdin/stdout/stderr to memory
// Similar to UNIX shellscript `command` constructs, with added INPUT on stdin
// Output size will be limited to 1 MiB, possibly clipping some output lines
/*****************************************************************************/
ULONG txcExecRedirectIO
(
   char               *cmd,                     // IN    command to execute
   char               *input,                   // IN    input  string or NULL (stdin)
   char              **output                   // OUT   output string or NULL (stdout)
)                                               //       (needs FreeMem)
{
   ULONG               rc  = NO_ERROR;
   TXLN                command = {0};           // redirected command
   FILE               *fp;                      // std input or output file
   ULN64               fsize;                   // resulting file size
   char               *mem = NULL;              // allocated output memory

   ENTER();

   if (input != NULL)                           // redirect string to stdin
   {                                            // using cat/type of tmp file
      if (((fp = fopen( TXC_ITMP_FILE_NAME, "w" TXFMODE)) != NULL) &&
          (fwrite( input, 1, (size_t) strlen( input), fp) != 0))
      {
         fclose( fp);                           // close the input file

         #if defined (UNIX)
            sprintf( command, "cat  %s | ", TXC_ITMP_FILE_NAME);
         #else
            sprintf( command, "type %s | ", TXC_ITMP_FILE_NAME);
         #endif
      }
      else
      {
         rc = TX_FAILED;
      }
   }

   if (rc == NO_ERROR)
   {
      strcat( command, cmd);                    // command to execute
      strcat( command, " > ");
      #if defined (UNIX)
         strcat( command, (output) ? TXC_TEMP_FILE_NAME : "/dev/null");
      #else
         strcat( command, (output) ? TXC_TEMP_FILE_NAME : "NUL");
      #endif
      #if !defined (DOS32)                      // redirect stderr too
         strcat( command, " 2>&1");
      #endif

      #if defined (DEV32)
         TxSessionMorph2PM( FALSE);             // undo morph to PM
         rc = (ULONG) system( command);         // execute the actual command
         TxSessionMorph2PM( TRUE);              // and back to PM, for set Win Title
      #else
         rc = (ULONG) system( command);         // execute the command
      #endif

      TRACES(( "Command executed: '%s'   rc: %u\n", command, rc));

      if (input)
      {
         TxDeleteFile( TXC_ITMP_FILE_NAME);
      }

      if (output)
      {
         if (TxFileSize( TXC_TEMP_FILE_NAME, &fsize))
         {
            if (fsize > TXMAX1MIB)
            {
               fsize = TXMAX1MIB;
            }
            if ((mem = TxAlloc( 1, (size_t) fsize + 1)) != NULL)
            {
               if ((fp = fopen( TXC_TEMP_FILE_NAME, "r" TXFMODE)) != NULL)
               {
                  if (fread( mem, 1, (size_t) fsize, fp) != 0)
                  {
                     //- success, could do additional filtering here, but
                     //- mem-string holds null-terminated file contents
                  }
                  fclose( fp);
               }
               else
               {
                  TRACES(("Can not open '%s'\n", TXC_TEMP_FILE_NAME));
               }
            }
         }
         TxDeleteFile( TXC_TEMP_FILE_NAME);
         *output = mem;
      }
   }
   RETURN (rc);
}                                               // end 'txcExecRedirectIO'
/*---------------------------------------------------------------------------*/


#if !defined (DEV32)
/*****************************************************************************/
// Execute an external command, using FILE redirected stdout/stderr to TxPrint
// Includes SB filter for WGET, removes thousands of progress lines
/*****************************************************************************/
static ULONG txcExecStdOutFile
(
   char               *cmd                      // IN    command to execute
)
{
   ULONG               rc  = NO_ERROR;
   TXLN                command;                 // redirected command
   TXLN                output;                  // filtered output (TABS)
   FILE               *so;                      // std output file

   ENTER();

   strcpy( command, cmd);                       // command to execute
   strcat( command, " > ");
   strcat( command, TXC_TEMP_FILE_NAME);
   #if !defined (DOS32)                         // redirect stderr too
      strcat( command, " 2>&1");
   #endif

   TxDeleteFile( TXC_TEMP_FILE_NAME);
   rc = (ULONG) system( command);               // execute the command
   TRACES(( "External command: '%s'   rc:0x%8.8lx\n", command, rc));

   if ((so = fopen( TXC_TEMP_FILE_NAME, "r" TXFMODE)) != NULL)
   {
      while (!TxAbort() && !feof(so) && !ferror(so))
      {
         if (fgets( command, TXMAXLN, so) != NULL)
         {
            if ((strstr( command, "0K .") == NULL) ||
                (strstr( command, "% "  ) == NULL)  ) // not a WGET progress line
            {
               TxRepStr( command, 0x09, "        ", output, TXMAXLN); // crude TAB expand (fixed)

               TxPrint( "%s", output);
            }
         }
      }
      fclose( so);
   }
   else
   {
      TRACES(("Can not open '%s'\n", TXC_TEMP_FILE_NAME));
   }
   TxDeleteFile( TXC_TEMP_FILE_NAME);
   RETURN (rc);
}                                               // end 'txcExecStdOutFile'
/*---------------------------------------------------------------------------*/

#else
/*****************************************************************************/
// Execute an external command, using PIPE redirected stdout/stderr to TxPrint
/*****************************************************************************/
ULONG txcExecStdOutPipe
(
   char               *cmd                      // IN    command to execute
)
{
   ULONG               rc  = NO_ERROR;
   TXC_IO_HANDLES      ho;                      // handles stdout
   HFILE               wt;                      // temp write-handle
   TID                 tidout;                  // Thread-id stdout reader
   TXLN                command;

   ENTER();

   if ((rc = DosCreatePipe( &(ho.read), &(ho.write), TXC_PIPE_SIZE)) == NO_ERROR)
   {
      ho.save = TXC_NEW_HANDLE;
      if ((rc = DosDupHandle(  STDOUT, &(ho.save))) == NO_ERROR)
      {
         if (ho.write != STDOUT)                // unless we got right value
         {
            wt = STDOUT;                        // duplicate it to handle
            rc = DosDupHandle( ho.write, &wt);  // with correct value
            if (rc == NO_ERROR)
            {
               TxClose( ho.write);              // close original pipe wh
               ho.write = wt;
            }
         }
         if (rc == NO_ERROR)
         {
            tidout = TxBeginThread( txcPipeOutputPump, 16384, &ho);

            strcpy( command, cmd);
            strcat( command, " < NUL");
            if ((strstr(command, "2>") == NULL) &&
                (strstr(command, "&2") == NULL) )
            {                                   // let CMD.EXE combine stderr
               strcat( command, " 2>&1");       // with stdout, resulting in
            }                                   // better ordering of output
            TRACES(( "External 'system()' command: '%s'\n", command));

            TxSessionMorph2PM( FALSE);          // undo morph to PM
            rc = (ULONG) system( command);      // execute the actual command
            TxSessionMorph2PM( TRUE);           // and back to PM, for set Win Title

            fprintf( stdout, "%c", TXC_EOF);    // end stdout
            fflush(  stdout);

            DosWaitThread( &tidout, DCWW_WAIT); // wait for pump to complete

            wt = STDOUT;                        // reconnect stdout
            if (DosDupHandle( ho.save, &wt) == NO_ERROR)
            {
               TxClose( ho.save);               // close original save-handle
            }
         }
      }
      TxClose( ho.read);                        // close the pipe-handle
   }
   RETURN (rc);
}                                               // end 'txcExecStdOutPipe'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// TXC child-stdout thread, waiting for output by child-process; to TXPRINT
/*****************************************************************************/
void txcPipeOutputPump
(
   void               *p                        // IN    pointer parameter
)
{
   TXC_IO_HANDLES     *sh = (TXC_IO_HANDLES *) p;
   char                ch = 0;                  // temporary char
   ULONG               nr;                      // nr of chars read
   ULONG               i;
   TX1K                buf;
   DEVICE_STATE        sa = TxLogfileState(DEVICE_TEST); // save for filtering

   ENTER();

   while (ch != TXC_EOF)
   {
      for ( i = 0, ch = '\0';
           (i < TXMAX1K -1) && (ch != TXC_EOF) && (ch != '\n');
          )
      {
         if ((TxRead( sh->read, &ch, 1, &nr) == NO_ERROR) && nr)
         {
            switch (ch)
            {
               case TXC_EOF:                    // end of stream
               case TXC_CR:
                  break;

               case TXC_LF:                     // end of line
                  ch = '\n';                    // fall through
               default:
                  buf[i++] = ch;
                  break;
            }
         }
         else
         {
            ch = TXC_EOF;                       // terminate on errors
         }
      }
      if (i > 0)                                // anything to redirect ?
      {
         buf[i]  = '\0';                        // terminate string
         TxWriteStream( sh->save, sa, buf);
      }
   }
   TxLogfileState( sa);                         // restore logfile state
   VRETURN();
}                                               // end 'txcPipeOutputPump'
/*---------------------------------------------------------------------------                    */


/*****************************************************************************/
// Write to stream (redirected stdout) with logfile (TEE) and output hooks
// Includes SB filter for WGET, adds ANSI 'cursor-up' after each progress line
/*****************************************************************************/
static void TxWriteStream
(
   TXHFILE             stream,                  // IN    stream to write to
   DEVICE_STATE        logging,                 // IN    log state to be restored
   char               *text                     // IN    string to write
)                                               //       usually one whole line
{
   ULONG               size = strlen(text);

   if ((strstr( text, "0K .") != NULL) &&
       (strstr( text, "% "  ) != NULL)  )       // probable WGET progress line
   {
      TxLogfileState( DEVICE_OFF);              // logfile state explicit OFF
      TxPrint( "%s%s", text, CU1);              // back to start of same line (overwrite)
   }
   else
   {
      if (logging == DEVICE_ON)                 // logging was active before cmd
      {                                         // might be OFF now, by filter
         TxLogfileState( DEVICE_ON);            // restore logfile state
      }
      TxPrint( "%s", text);                     // output to logfile/hooks (scrollbuffer)
   }

   if (((TxScreenState(DEVICE_TEST)) == DEVICE_ON) &&  //- screen active
       (txwa->desktop == NULL))                        //- and not windowed!
   {                                                   //- direct to stdout
      if (text[size-1] == '\n')
      {
         text[--size]   = '\0';
         TxWrite( stream, text, size, &size);   // to our original stdout
         TxWrite( stream, "\r\n",  2, &size);
      }
      else
      {
         TxWrite( stream, text, size, &size);   // to our original stdout
      }
   }
}                                               // end 'TxWriteStream'
/*---------------------------------------------------------------------------*/
#endif
