//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2005 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
// This file contains Original Code and/or Modifications of Original Code as
// defined in and that are subject to the GNU Lesser General Public License.
// You may not use this file except in compliance with the License.
// BY USING THIS FILE YOU AGREE TO ALL TERMS AND CONDITIONS OF THE LICENSE.
// A copy of the License is provided with the Original Code and Modifications,
// and is also available at http://www.dfsee.com/txwin/lgpl.htm
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License,
// or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; (lgpl.htm) if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
//
// Questions on TxWin licensing can be directed to: txwin@fsys.nl
//
// ==========================================================================
//
// Command-window functions and clipboard handling
// and file-logging facilities
//
// Author: J. van Wijk
//
// JvW  23-06-2018 Initial version, split off from TXCON.C

#include <txlib.h>
#include <txwpriv.h>                            // private window interface
#include <txtpriv.h>                            // private text   interface

#if defined (DOS32)
   static char *localClipBoard = NULL;          // application specific clipboard
#endif


#if defined (WIN32)

// Set TEXT string to the clipboard, clipboard COPY function; Windows specific
static ULONG TxWinSetClipBoardText              // RET   result
(
   char               *text                     // IN    text to copy to clipboard
);

// Get TEXT contents from the clipboard return text and supply close handle
static char *TxWinGetClipBoardText              // RET   clipboard text or NULL
(
   HANDLE             *hClip                    // OUT   handle to release mem
);

// Release clipboard TEXT memory from GetClipBoardText and close clipboard
static void TxWinReleaseClipBoard
(
   HANDLE              hClip                    // IN    handle to release mem
);
#endif

#if defined (DEV32)

// Set TEXT string to the clipboard, clipboard COPY function; OS/2 specific
static ULONG TxOs2SetClipBoardText              // RET   result
(
   char               *text                     // IN    text to copy to clipboard
);

// Get TEXT contents from the clipboard return text and supply close handle
static char *TxOs2GetClipBoardText              // RET   clipboard text or NULL
(
   ULONG              *hClip                    // OUT   handle to release mem
);

// Release clipboard TEXT memory from GetClipBoardText
static void TxOs2ReleaseClipBoard
(
   ULONG               hClip                    // IN    handle to release mem
);
#endif



/*****************************************************************************/
// Set applications window/terminal title to specified string, or RESTORE
// the saved orginal Title when requested by 'magic' title contents
/*****************************************************************************/
void TxSetAppWindowTitle
(
   char               *newTitle                 // IN    new title string
)
{
   #if defined (DEV32) || defined (WIN32)
      static TXLN      orgTitle = {0};          // Title at startup (1st call)
   #endif

   ENTER();
   TRACES(("newTitle: '%s'\n", newTitle));

   #if   defined (WIN32)
      if (orgTitle[0] == 0)                     // no saved Title yet
      {
         GetConsoleTitle( orgTitle, TXMAXLN);
         TRACES(("orgTitle: '%s'\n", orgTitle));
      }
      if ((orgTitle[0] != 0) && (strstr( newTitle, TX_APPWIN_TRESTORE) != NULL))
      {
         SetConsoleTitle( orgTitle);            // restore original
      }
      else                                      // just set the given title
      {
         SetConsoleTitle( newTitle);
      }
   #elif defined (DOS32)
      //- not implemented for DOS
   #elif defined (UNIX)

    #if defined (LINUX)
      //- NOT to be done for Linux console, may cause beeps/screen corruption
      if ((getenv( "DISPLAY")) != NULL)   // seems to be genuine X environment
    #endif

      {
         printf( "\033]0;%s\007", newTitle);    // macOS and Linux X-terminals
      }
   #else                                        // OS/2, switchlist/WinTitle APIs (4OS2)
   {
      SWCNTRL          swctl;
      HSWITCH          hswitch;

      memset( &swctl, 0, sizeof( SWCNTRL));
      if (txwa->api && txwa->api->pfnWQSH && txwa->api->pfnWQSE && txwa->api->pfnWCSE)
      {
         if ((hswitch = (txwa->api->pfnWQSH)( 0, txwa->pib->pib_ulpid)) != NULLHANDLE)
         {
            if ((txwa->api->pfnWQSE)( hswitch, &swctl) == NO_ERROR)
            {
               if (orgTitle[0] == 0)            // no saved Title yet
               {
                  strcpy( orgTitle, swctl.szSwtitle);
                  TRACES(("orgTitle: '%s'\n", orgTitle));
               }
               //- change the name (will be clipped to 60 positions) then set it
               if ((orgTitle[0] != 0) && (strstr( newTitle, TX_APPWIN_TRESTORE) != NULL))
               {
                  strcpy( swctl.szSwtitle, orgTitle);
               }
               else                             // just set the given title
               {
                  strcpy( swctl.szSwtitle, newTitle);
               }
               if ((txwa->api->pfnWCSE)( hswitch, &swctl) == NO_ERROR)
               {
                  TRACES(("Set Switchlist name to '%s'\n", swctl.szSwtitle));

                  //- Now attempt to set the WindowTitle as well (2 methods)
                  if (txwa->api && txwa->api->pfnWSTAI && txwa->api->pfnWQSE && txwa->api->pfnWCSE)
                  {
                     //- to be refined, may need EXE/ICO path+name as second parameter ?? (see 4OS2)
                     //- may be needed for Warp 3/4 (works fine on 2.0 and later 4.50 etc)
                     (txwa->api->pfnWSTAI)( swctl.szSwtitle, swctl.szSwtitle);

                     //- Set WindowTitle on switchlist hwnd
                     (txwa->api->pfnWSWT)( swctl.hwnd, swctl.szSwtitle);
                  }
                  else
                  {
                     TRACES(("WinTitle APIs not available!\n"));
                  }
               }
               else
               {
                  TRACES(("ChangeSwitchEntry failed\n"));
               }
            }
            else
            {
               TRACES(("QuerySwitchEntry failed\n"));
            }
         }
         else
         {
            TRACES(("QuerySwitchHandle failed\n"));
         }
      }
      else
      {
         TRACES(("switchlist APIs not available!\n"));
      }
   }
   #endif

   VRETURN ();
}                                               // end 'TxSetAppWindowTitle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy TEXT to the clipboard (Linux: to both CLIPBOARD and PRIMARY X-select!)
/*****************************************************************************/
ULONG TxCopyClipBoardText
(
   char               *clipText                 // IN    C-string of UTF-8 text
)
{
   ULONG               rc = TX_FAILED;
   #if defined (UNIX)
      char            *utilOut = NULL;         // possible 'not found' message
   #endif

   ENTER();
   TRACES(("clipText length: %d\n", strlen( clipText)));

   //- Platform specific open clipboard and copy text to it
   #if   defined (WIN32)
      rc = TxWinSetClipBoardText(       clipText);
   #elif defined (DEV32)
      rc = TxOs2SetClipBoardText(       clipText);
   #elif defined (DARWIN)
      rc = txcExecRedirectIO( "pbcopy", clipText, &utilOut);
      if ((rc != 0) && (strstr( clipText, "Failed to execute the 'pbcopy'") == NULL)) // Don't repeat :)
      {
         TxNamedMessage( TRUE, TXCM_HELPREG + 10, " ERROR: 'pbcopy' execute failed ",
                                  "Failed to execute the 'pbcopy' clipboard utility: \n\n%s", utilOut);
      }
      TxFreeMem( utilOut);
   #elif defined (LINUX)
      rc = txcExecRedirectIO( "xclip -selection clipboard",  clipText, &utilOut);
      if (rc == NO_ERROR)                       // copy to CLIPBOARD X-selection OK
      {                                         // no copy to PRIMARY too
         rc = txcExecRedirectIO( "xclip -selection primary",  clipText, &utilOut);
      }
      if ((rc != 0) && (strstr( clipText, "Failed to execute the 'xclip'") == NULL)) // Don't repeat :)
      {
         TxNamedMessage( TRUE, TXCM_HELPREG + 10, " ERROR: 'xclip' execute failed ",
                                  "Failed to execute the 'xclip' clipboard utility: \n\n%s", utilOut);
      }
      TxFreeMem( utilOut);
   #else
      TxFreeMem( localClipBoard);               // free existing contents

      if ((localClipBoard = TxAlloc( 1, strlen( clipText) + 1)) != NULL)
      {
         TRACES(("Copying data into local clipboard\n"));
         strcpy( localClipBoard, clipText);
         rc = NO_ERROR;
      }
   #endif

   RETURN (rc);
}                                               // end 'TxCopyClipBoardText'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get ALL or just the first line of TEXT from the clipboard, return length
/*****************************************************************************/
ULONG TxPasteClipBoardLines                     // RET   length of text LINE(s)
(
   ULONG               size,                    // IN    Maximum length (buffer)
   BOOL                firstLineOnly,           // IN    Paste 1st line only
   char               *line                     // OUT   One line of TEXT
)
{
   ULONG               rc = 0;                  // function return
   char               *clipText = NULL;         // clipboard text
   char               *s;                       // string walk pointer
   char               *dest;                    // destination in copy

   #if   defined (WIN32)
      HANDLE           hCb;                     // clipboard handle
   #elif defined (DEV32)
      ULONG            hCb;                     // clipboard handle
   #elif defined (DARWIN)
   #elif defined (LINUX)
   #else
   #endif

   ENTER();

   //- Platform specific open clipboard and retrieve text
   #if   defined (WIN32)
      clipText = TxWinGetClipBoardText( &hCb);
   #elif defined (DEV32)
      clipText = TxOs2GetClipBoardText( &hCb);
   #elif defined (DARWIN)
      rc = txcExecRedirectIO( "pbpaste",  NULL, &clipText);
      if ((rc != 0) && (strstr( clipText, "pbpaste:") != NULL)) // likely a not-found message
      {
         TxNamedMessage( TRUE, TXCM_HELPREG + 10, " ERROR: 'pbpaste' execute failed ",
                                  "Failed to execute the 'pbpaste' clipboard utility: \n\n%s", clipText);
         strcpy( clipText, "");                 // don't paste the error message
      }
   #elif defined (LINUX)
      if (txwa->xSelClip)                       // target CLIPBOARD X-selection
      {
         rc = txcExecRedirectIO( "xclip -o -selection clipboard", NULL, &clipText);
      }
      else                                      // target PRIMARY   X-selection
      {
         rc = txcExecRedirectIO( "xclip -o -selection primary", NULL, &clipText);
      }
      if ((rc != 0) && (strstr( clipText, "xclip:") != NULL)) // likely a not-found message
      {
         TxNamedMessage( TRUE, TXCM_HELPREG + 10, " ERROR: 'xclip' execute failed ",
                                  "Failed to execute the 'xclip' clipboard utility: \n\n%s", clipText);
         strcpy( clipText, "");                 // don't paste the error message
      }
   #else
      clipText = localClipBoard;
   #endif


   //- Generic part, just handle the (string) TEXT from the clipboard
   if ((clipText != NULL) && (size > 0))
   {
      for (s = clipText, dest = line; --size; s++)
      {
         if ((((*s == 0x0a) || (*s == 0x0d)) && (firstLineOnly)) || (*s == 0))
         {
            *dest++ = 0;                        // terminate at CR/LF or end string
            break;
         }
         else
         {
            *dest++ = *s;
         }
      }
      *dest = 0;                                // make sure it is terminated
      rc = strlen( line);
   }
   TRACES(("line: '%s'\n", line));

   //- Platform specific release of text/close-clipboard
   #if   defined (WIN32)
      TxWinReleaseClipBoard( hCb);              // Release and close clipboard
   #elif defined (DEV32)
      TxOs2ReleaseClipBoard( hCb);              // Release clipboard
   #elif defined (DARWIN)
      TxFreeMem( clipText);
   #elif defined (LINUX)
      TxFreeMem( clipText);
   #else
   #endif

   RETURN (rc);
}                                               // end 'TxPasteClipBoardLines'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get TEXT from the clipboard, and write it to the output scrollbuffer (PASTE)
/*****************************************************************************/
ULONG TxPasteClipBoardText                      // RET   length of text LINE
(
   void
)
{
   ULONG               rc = 0;                  // function return
   char               *clipText = NULL;         // clipboard text

   #if   defined (WIN32)
      HANDLE           hCb;                     // clipboard handle
   #elif defined (DEV32)
      ULONG            hCb;                     // clipboard handle
   #elif defined (DARWIN)
   #elif defined (LINUX)
   #else
   #endif

   ENTER();

   //- Platform specific open clipboard and retrieve text
   #if   defined (WIN32)
      clipText = TxWinGetClipBoardText( &hCb);
   #elif defined (DEV32)
      clipText = TxOs2GetClipBoardText( &hCb);
   #elif defined (DARWIN)
      rc = txcExecRedirectIO( "pbpaste",  NULL, &clipText);
      if ((rc != 0) && (strstr( clipText, "pbpaste:") != NULL)) // likely a not-found message
      {
         TxNamedMessage( TRUE, TXCM_HELPREG + 10, " ERROR: 'pbpaste' execute failed ",
                                  "Failed to execute the 'pbpaste' clipboard utility: \n\n%s", clipText);
         //- paste the error message, for logging
      }
   #elif defined (LINUX)
      if (txwa->xSelClip)                       // target CLIPBOARD X-selection
      {
         rc = txcExecRedirectIO( "xclip -o -selection clipboard", NULL, &clipText);
      }
      else                                      // target PRIMARY   X-selection
      {
         rc = txcExecRedirectIO( "xclip -o -selection primary", NULL, &clipText);
      }
      rc = txcExecRedirectIO( "xclip -o", NULL, &clipText);
      if ((rc != 0) && (strstr( clipText, "xclip:") != NULL)) // likely a not-found message
      {
         TxNamedMessage( TRUE, TXCM_HELPREG + 10, " ERROR: 'xclip' execute failed ",
                                  "Failed to execute the 'xclip' clipboard utility: \n\n%s", clipText);
         //- paste the error message, for logging
      }
   #else
      clipText = localClipBoard;
   #endif


   //- Generic part, just handle the (string) TEXT from the clipboard
   if ((clipText != NULL) && (strlen( clipText) > 0))
   {
      TXLN             chunkBuffer;
      char            *chunkPointer = clipText;

      TxPrint(  "\n- - - - - - - - - - - - - - Pasted from the clipboard: - - - - - - - - - - - - -\n");

      //- Write buffer to output buffer, in chunks to avoid TxPrint buffer overflows
      while (strlen( chunkPointer) > (TXMAXLN - 1))
      {
         memcpy( chunkBuffer, chunkPointer, TXMAXLN - 1);
         chunkBuffer[ TXMAXLN - 1] = 0;
         chunkPointer += (TXMAXLN - 1);
         TxPrint( "%s", chunkBuffer);           // print one large chunk
      }
      TxPrint( "%s", chunkPointer);             // print the remainder ...

      rc = strlen( clipText);

      if (clipText[ rc - 1] != '\n')            // make sure last line gets EOL
      {
         TxPrint( "\n");
      }
      TxPrint(    "- - - - - - - - - - - - - - End paste from  clipboard  - - - - - - - - - - - - -\n");
   }


   //- Platform specific release of text/close-clipboard
   #if   defined (WIN32)
      TxWinReleaseClipBoard( hCb);              // Release and close clipboard
   #elif defined (DEV32)
      TxOs2ReleaseClipBoard( hCb);              // Release clipboard / free memory
   #elif defined (DARWIN)
      TxFreeMem( clipText);
   #elif defined (LINUX)
      TxFreeMem( clipText);
   #else
   #endif

   RETURN (rc);
}                                               // end 'TxPasteClipBoardText'
/*---------------------------------------------------------------------------*/


#if defined (WIN32)
/*****************************************************************************/
// Set TEXT string to the clipboard, clipboard COPY function; Windows specific
/*****************************************************************************/
static ULONG TxWinSetClipBoardText              // RET   result
(
   char               *text                     // IN    text to copy to clipboard
)
{
   ULONG               rc = TX_FAILED;
   HGLOBAL            *hClip;                   // memory handle global, unmovable
   char               *clipText;                // char * locked alias for memory

   ENTER();

   if ((hClip = GlobalAlloc( GPTR, strlen( text) + 1)) != NULL)
   {
      clipText = (char *) GlobalLock( hClip);   // lock before copying
      strcpy( clipText, text);                  // copy the data to locked memory
      GlobalUnlock( hClip);                     // release the memory lock

      if (OpenClipboard( NULL))
      {
         EmptyClipboard();                      // remove any existing contents

         if (SetClipboardData( CF_TEXT, hClip)) // transfer to clipboard
         {
            rc = NO_ERROR;
         }
         else
         {
            TRACES(("SetClipboardData fail, %s\n", txNtLastError()));
         }
         CloseClipboard();                      // and close clipboard
      }
   }
   RETURN (rc);
}                                               // end 'TxWinSetClipBoardText'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get TEXT contents from the clipboard return text and supply close handle
/*****************************************************************************/
static char *TxWinGetClipBoardText              // RET   clipboard text or NULL
(
// HANDLE             *hClip                    // OUT   handle to release mem
   HGLOBAL            *hClip                    // OUT   handle to release mem
)
{
   char               *rc = NULL;               // function return

   ENTER();

   *hClip = NULL;

   if (IsClipboardFormatAvailable( CF_TEXT))    // UTF-8 text available ?
   {
      if (OpenClipboard( NULL))
      {
         if ((*hClip = GetClipboardData( CF_TEXT)) != NULL)
         {
            rc = (char *) GlobalLock( *hClip);  // Lock and return as char pointer

            TRHEXS( 70, rc, strlen( rc), "CLIPBOARD");
         }
      }
   }
   RETURN (rc);
}                                               // end 'TxWinGetClipBoardText'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Release clipboard TEXT memory from GetClipBoardText and close clipboard
/*****************************************************************************/
static void TxWinReleaseClipBoard
(
// HANDLE              hClip                    // IN    handle to release mem
   HGLOBAL             hClip                    // IN    handle to release mem
)
{
   ENTER();

   if (hClip != NULL)                           // clipboard data retrieved
   {
      GlobalUnlock( hClip);                     // release the memory

      CloseClipboard();                         // and close clipboard
   }
   VRETURN ();
}                                               // end 'TxWinReleaseClipBoard'
/*---------------------------------------------------------------------------*/

#endif


#if defined (DEV32)

/*****************************************************************************/
// Set TEXT string to the clipboard, clipboard COPY function; OS/2 specific
/*****************************************************************************/
static ULONG TxOs2SetClipBoardText              // RET   result
(
   char               *text                     // IN    text to copy to clipboard
)
{
   ULONG               rc = TX_FAILED;
   char               *clipText = NULL;

   ENTER();

   if (txwa->api->pfnWSCD && txwa->api->pfnWEC && txwa->api->pfnWOC && txwa->api->pfnWCC)
   {
      if (txwa->api->pfnWOC( txwa->api->hHAB))  // Open clipboard
      {
         txwa->api->pfnWEC( txwa->api->hHAB);   // remove current contents

         DosAllocSharedMem( (PVOID) &clipText, NULL, strlen( text) + 1,
                             OBJ_GETTABLE | OBJ_GIVEABLE | OBJ_TILE | PAG_COMMIT | PAG_READ | PAG_WRITE );
         if (clipText != NULL)
         {
            strcpy( clipText, text);            // copy string to global memory object

            //- Set the global memory object in the clipboard
            if (txwa->api->pfnWSCD( txwa->api->hHAB, (ULONG) clipText, CF_TEXT, CFI_POINTER))
            {
               txwa->api->pfnWCC( txwa->api->hHAB); // Close clipboard
               rc = NO_ERROR;
            }
            else
            {
               TRACES(("SetClipboardData failed\n"));
            }
         }
         else
         {
            TRACES(("DosAllocSharedMem failed\n"));
         }
      }
   }
   RETURN (rc);
}                                               // end 'TxOs2SetClipBoardText'
/*---------------------------------------------------------------------------*/



   #define  TXCB_NOTOPEN  0UL
   #define  TXCB_NO_FREE ~0UL
/*****************************************************************************/
// Get TEXT contents from the clipboard return text and supply close handle
/*****************************************************************************/
static char *TxOs2GetClipBoardText              // RET   clipboard text or NULL
(
   ULONG              *hClip                    // OUT   handle to release mem
)
{
   char               *rc = NULL;               // function return
   ULONG               dummy;
   PVOID               tMem;                    // text memory

   ENTER();

   *hClip = TXCB_NOTOPEN;

   if (txwa->api->pfnWQCFI && txwa->api->pfnWOC && txwa->api->pfnWQCD)
   {
      if (txwa->api->pfnWQCFI(  txwa->api->hHAB, CF_TEXT, &dummy)) // Query format
      {
         if (txwa->api->pfnWOC( txwa->api->hHAB)) // Open clipboard
         {
            *hClip = TXCB_NO_FREE;              // no free desired by default
            if ((tMem = txwa->api->pfnWQCD( txwa->api->hHAB, CF_TEXT)) != NULL) // Query data
            {
               //- Verify the memory, and make readable when needed (avoids OS2 VIO bug)
               ULONG   size  = 0x1000;
               ULONG   flags = ~0UL;
               APIRET  dr    = DosQueryMem( tMem, &size, &flags);

               if (dr != NO_ERROR)
               {
                  dr = DosQueryMem( tMem, &size, &flags); // retry, avoiding an OS2 bug
               }
               if ((dr != NO_ERROR) || (flags & (PAG_READ | PAG_COMMIT)) != (PAG_READ | PAG_COMMIT))
               {
                  if ((dr = DosGetSharedMem( tMem, PAG_READ)) == NO_ERROR)
                  {
                     *hClip = (ULONG) tMem;     // handle is memory address, need to free
                  }
               }
               if (dr == NO_ERROR)              // return the text
               {
                  rc = (char *) tMem;

                  TRHEXS( 70, rc, strlen( rc), "CLIPBOARD");
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'TxOs2GetClipBoardText'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Release clipboard TEXT memory from GetClipBoardText
/*****************************************************************************/
static void TxOs2ReleaseClipBoard
(
   ULONG               hClip                    // IN    handle to release mem
)
{
   ENTER();

   if (hClip != TXCB_NOTOPEN)                   // clipboard was opened
   {
      txwa->api->pfnWCC( txwa->api->hHAB);      // Close clipboard

      if (hClip != TXCB_NO_FREE)                // need to free shared mem
      {
         DosFreeMem( (PVOID) hClip);            // handle is memory address
      }
   }
   VRETURN ();
}                                               // end 'TxOs2ReleaseClipBoard'
/*---------------------------------------------------------------------------*/

#endif
