/**************************************************************************************************
	$Header: /pub/cvsroot/yencode/src/ypost/prompt.c,v 1.2 2002/03/21 04:58:31 bboy Exp $
	Routines used by `ypost' to prompt the user for missing data and confirm posting schedule.

	Copyright (C) 2002  Don Moore <bboy@bboy.net>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at Your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
**************************************************************************************************/

#include "ypost.h"


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_AUTHOR
	Prompts the user to enter author information.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void
prompt_author(void)
{
	char input[BUFSIZ], *problem;

	do
	{
		printf("%s: ", _("Author"));
		fflush(stdout);
		fgets(input, sizeof(input)-1, stdin);

		if ((problem = usenet_valid_From(input, sizeof(input)-1)))
			printf("%s: %s\n", _("Invalid address"), problem);
	} while (problem);

	opt_author = xstrdup(input);
}
/*--- prompt_author() ---------------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_SUBJECT
	Prompts the user to enter subject information.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void
prompt_subject(void)
{
	char input[BUFSIZ], *problem;

	do
	{
		printf("%s: ", _("Subject"));
		fflush(stdout);
		fgets(input, sizeof(input)-1, stdin);

		if ((problem = usenet_valid_Subject(input, sizeof(input)-1)))
			printf("%s: %s\n", _("Invalid subject"), problem);

	} while (problem);

	opt_subject = (*input) ? xstrdup(input) : NULL;
}
/*--- prompt_subject() --------------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_COMMENT
	Prompts the user to enter comment information.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void
prompt_comment(void)
{
	char input[BUFSIZ], *problem;

	do
	{
		printf("%s: ", _("Comment"));
		fflush(stdout);
		fgets(input, sizeof(input)-1, stdin);

		if ((problem = usenet_valid_Subject(input, sizeof(input)-1)))
			printf("%s: %s\n", _("Invalid subject"), problem);

	} while (problem);

	opt_comment = (*input) ? xstrdup(input) : NULL;
}
/*--- prompt_comment() --------------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_NEWSGROUP
	Prompts the user to enter the newsgroup for posting.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void
prompt_newsgroup(void)
{
	char input[BUFSIZ];

	do
	{
		printf("%s: ", _("Newsgroup"));
		fflush(stdout);
		fgets(input, sizeof(input)-1, stdin);
		strtrim(input);
	} while (!strlen(input));
	opt_newsgroup = xstrdup(input);
}
/*--- prompt_newsgroup() ------------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_NNTP_SERVER
	Prompts the user to enter the news server for posting.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
static void
prompt_nntp_server(void)
{
	char input[BUFSIZ];

	do
	{
		printf("%s: ", _("News server"));
		fflush(stdout);
		fgets(input, sizeof(input)-1, stdin);
		strtrim(input);
	} while (!strlen(input));
	opt_nntp_server = xstrdup(input);
}
/*--- prompt_nntp_server() ----------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_FOR_MISSING
	Prompts the user for any missing data.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void
prompt_for_missing(void)
{
	char *problem;

	if (!opt_stdout && isatty(STDIN_FILENO) && isatty(STDOUT_FILENO))
	{
		/* If the server or group were not specified, prompt for them */
		if (!opt_nntp_server) prompt_nntp_server();
		if (!opt_newsgroup) prompt_newsgroup();

		if (opt_prompt_author) prompt_author();
		if (opt_prompt_subject) prompt_subject();
		if (opt_prompt_comment) prompt_comment();
	}

	/* If no author was specified, generate author line */
	if (!opt_author)
	{
		struct utsname uts;
		struct passwd	*pwd;
		char	authorbuf[1024];

		if (uname(&uts))
			ErrERR(_("unable to determine hostname"));
		if (!(pwd = getpwuid(getuid())))
			ErrERR(_("unable to determine username"));
		if (strlen(pwd->pw_gecos) > 1)
			snprintf(authorbuf, sizeof(authorbuf), "%s@%s (%s)", pwd->pw_name, uts.nodename, pwd->pw_gecos);
		else
			snprintf(authorbuf, sizeof(authorbuf), "%s@%s (%s)", pwd->pw_name, uts.nodename, pwd->pw_name);
		memset(pwd, 0, sizeof(struct passwd));
		opt_author = xstrdup(authorbuf);
	}

	/* Validate author (and set opt_sender) */
	if ((problem = usenet_valid_From(opt_author, sizeof(opt_author)-1)))
		Err("%s: %s\n", _("invalid author address"), problem);
}
/*--- prompt_for_missing() ----------------------------------------------------------------------*/


/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	PROMPT_CONFIRM_POST
	Output list of subjects to post and confirm with user.
+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
void
prompt_confirm_post(void)
{
	register int ct;
	YENCFILE	*y;
	char	*subject, input[5];
	int	indent = 0, total_parts;

	if (!isatty(STDOUT_FILENO) || !isatty(STDIN_FILENO))
		Err(_("unable to confirm posting: terminal unavailable"));

	indent = strlen(_("Server")) > indent ? strlen(_("Server")) : indent;
	indent = strlen(_("Newsgroup")) > indent ? strlen(_("Newsgroup")) : indent;
	indent = strlen(_("Author")) > indent ? strlen(_("Author")) : indent;
	indent += 4;

	puts("");
	printf("%*s: %s\n", indent, _("Server"), opt_nntp_server);
	printf("%*s: %s\n", indent, _("Newsgroup"), opt_newsgroup);
	printf("%*s: %s\n", indent, _("Author"), opt_author);
	puts("");

	for (ct = total_parts = 0; ct < num_input_files; ct++)
	{
		y = input_files[ct];

		/* Set temporary values into file_current, part_current, part_total so that the subject
			previews appear correctly */
		file_current = ct + 1;
		part_total = y->totalparts;
		part_current = 0;	// (makes 'x' appear)

		subject = usenet_make_subject(y, -1);
		printf("%3d. %s\n", ct+1, subject);
		free(subject);

		total_parts += y->totalparts;
	}
	file_current = 1;
	puts("");

	/* If the posting is being resumed, reassure the user that this is happening */
	if (opt_resume_msg)
	{
		printf(_("This post will be resumed, beginning with message number %d.\n"), opt_resume_msg);
		printf(_("As such, some of the parts listed above may not be posted, \n"
					"since they were posted successfully on the previous run.\n\n"));
	}

	/* Prompt the user */
	do
	{
		if (total_parts == 1)
			printf(_("Post this message?"));
		else
		{
			if (file_total == 1)
				printf(_("Post 1 file (%s parts)?"), comma(total_parts));
			else
				printf(_("Post %s files (%s parts)?"), comma1(file_total), comma2(total_parts));
		}
		printf(" [y]es, [n]o: ");
		fflush(stdout);

		fgets(input, sizeof(input)-1, stdin);
	} while (tolower(*input) != 'y' && tolower(*input) != 'n');
	puts("");

	if (*input != 'y')
		Err(_("aborted by user"));
}
/*--- prompt_confirm_post() ---------------------------------------------------------------------*/

/* vi:set ts=3: */
