/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "projectsview.h"

#include <QTreeView>
#include <QAction>
#include <QMenu>
#include <QTimer>
#include <QMessageBox>

#include "models/tablemodels.h"
#include "models/treeviewsettings.h"
#include "models/treeviewhelper.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/updatebatch.h"
#include "dialogs/projectdialogs.h"
#include "dialogs/watchesdialog.h"
#include "dialogs/notifydialog.h"
#include "xmlui/builder.h"
#include "connectionmanager.h"
#include "configdata.h"
#include "viewmanager.h"
#include "iconloader.h"

ProjectsView::ProjectsView( QObject* parent, QWidget* parentWidget ) : View( parent ),
    m_firstUpdateDone( false ),
    m_updateCounter( 0 ),
    m_activateReason( 0 )
{
    m_systemAdmin = connectionManager->connectionInfo()->access() == AdminAccess;

    QAction* action;

    action = new QAction( IconLoader::icon( "file-reload" ), tr( "&Update Projects" ), this );
    action->setShortcut( QKeySequence::Refresh );
    connect( action, SIGNAL( triggered() ), this, SLOT( updateProjects() ) );
    setAction( "updateProjects", action );

    action = new QAction( IconLoader::icon( "view-members" ), tr( "Project &Members" ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( showMembers() ) );
    setAction( "showMembers", action );

    action = new QAction( IconLoader::icon( "folder-watch" ), tr( "Folder &Watches..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( editWatches() ) );
    setAction( "editWatches", action );

    if ( connectionManager->connectionInfo()->checkFeature( "notify" ) ) {
        action = new QAction( IconLoader::icon( "notify" ), tr( "Enable &Notification" ), this );
        action->setCheckable( true );
        connect( action, SIGNAL( triggered() ), this, SLOT( toggleNotify() ) );
        setAction( "toggleNotify", action );
    }

    if ( m_systemAdmin ) {
        action = new QAction( IconLoader::icon( "project-new" ), tr( "Add &Project..." ), this );
        connect( action, SIGNAL( triggered() ), this, SLOT( addProject() ) );
        setAction( "addProject", action );
    }

    action = new QAction( IconLoader::icon( "folder-new" ), tr( "Add &Folder..." ), this );
    action->setShortcut( QKeySequence::New );
    connect( action, SIGNAL( triggered() ), this, SLOT( addFolder() ) );
    setAction( "addFolder", action );

    action = new QAction( IconLoader::icon( "edit-rename" ), tr( "&Rename..." ), this );
    action->setShortcut( tr( "F2" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( editRename() ) );
    setAction( "editRename", action );

    action = new QAction( IconLoader::icon( "edit-delete" ), tr( "&Delete" ), this );
    action->setShortcut( QKeySequence::Delete );
    connect( action, SIGNAL( triggered() ), this, SLOT( editDelete() ) );
    setAction( "editDelete", action );

    action = new QAction( IconLoader::icon( "folder-open" ), tr( "&Open Folder" ), this );
    action->setShortcut( QKeySequence::Open );
    connect( action, SIGNAL( triggered() ), this, SLOT( openFolder() ) );
    setAction( "openFolder", action );

    setTitle( "menuProjects", tr( "&Projects" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );

    setButtonStyle( "addProject", Qt::ToolButtonTextBesideIcon );
    setButtonStyle( "addFolder", Qt::ToolButtonTextBesideIcon );
    setButtonStyle( "updateProjects", Qt::ToolButtonTextBesideIcon );

    loadXmlUiFile( ":/resources/projectsview.xml" );

    m_list = new QTreeView( parentWidget );
    m_list->setRootIsDecorated( true );
    m_list->setSortingEnabled( true );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( contextMenu( const QPoint& ) ) );
    connect( m_list, SIGNAL( activated( const QModelIndex& ) ),
        this, SLOT( activated( const QModelIndex& ) ) );

    m_list->installEventFilter( this );
    m_list->viewport()->installEventFilter( this );

    setMainWidget( m_list );
}

ProjectsView::~ProjectsView()
{
    TreeViewSettings settings;
    settings.openProjectsTree();

    settings.saveColumnWidths( TreeViewHelper::readColumnWidths( m_list ) );
    if ( m_firstUpdateDone )
        settings.saveExpandedNodes( TreeViewHelper::readExpandedNodes( m_list ) );
}

void ProjectsView::initialUpdate()
{
    m_model = new RDB::TableItemModel( this );
    m_model->setRootTableModel( new ProjectsTableModel( m_model ), dataManager->projects()->index() );
    m_model->addChildTableModel( new FoldersTableModel( m_model ),
        dataManager->folders()->index(), dataManager->folders()->parentIndex() );

    TreeViewSettings settings;
    settings.openProjectsTree();

    m_model->setColumns( settings.loadColumns() );

    m_list->setModel( m_model );

    m_list->sortByColumn( 0, Qt::AscendingOrder );

    TreeViewHelper::applyColumnWidths( m_list, settings.loadColumnWidths() );

    setCaption( connectionManager->connectionInfo()->serverName() );

    connect( m_list->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
        this, SLOT( updateActions() ) );
    connect( m_model, SIGNAL( layoutChanged() ), this, SLOT( updateActions() ) );

    updateActions();
    updateSummary();

    periodicUpdate( true );

    QTimer* timer = new QTimer( this );
    connect( timer, SIGNAL( timeout() ), this, SLOT( updateTimeout() ) );
    timer->start( 60 * 1000 );

    setAccess( checkDataAccess(), true );
}

Access ProjectsView::checkDataAccess()
{
    m_anyProjectAdmin = m_systemAdmin;

    int userId = connectionManager->connectionInfo()->userId();
    RDB::ForeignConstIterator<MemberRow> it( dataManager->members()->index().first(), userId );
    while ( it.next() ) {
        const MemberRow* member = it.get();
        if ( member->access() == AdminAccess )
            m_anyProjectAdmin = true;
    }

    return m_anyProjectAdmin ? AdminAccess : NormalAccess;
}

void ProjectsView::updateAccess( Access /*access*/ )
{
    action( "addFolder" )->setVisible( m_anyProjectAdmin );
    action( "editRename" )->setVisible( m_systemAdmin || m_anyProjectAdmin );
    action( "editDelete" )->setVisible( m_systemAdmin || m_anyProjectAdmin );
}

void ProjectsView::updateTimeout()
{
    m_updateCounter++;

    if ( m_updateCounter >= configData->updateInterval() ) {
        periodicUpdate( false );

        m_updateCounter = 0;
    }
}

void ProjectsView::periodicUpdate( bool initial )
{
    UpdateBatch* batch = new UpdateBatch( initial ? -1 : -10 );
    batch->updateUsers();
    batch->updateTypes();
    if ( connectionManager->connectionInfo()->checkFeature( "notify" ) )
        batch->updateNotifications();
    batch->updateProjects();
    if ( connectionManager->connectionInfo()->checkVersion( "0.8.4" ) )
        batch->updatePreferences();

    executeUpdate( batch );
}

void ProjectsView::updateActions()
{
    m_selectedProjectId = 0;
    m_selectedFolderId = 0;
    m_currentProjectId = 0;

    QModelIndex index = selectedIndex();
    if ( index.isValid() ) {
        int level = m_model->data( index, RDB::TableItemModel::LevelRole ).toInt();
        int rowId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();
        if ( level == 0 ) {
            m_selectedProjectId = rowId;
            m_currentProjectId = rowId;
        } else {
            m_selectedFolderId = rowId;
            const FolderRow* folder = dataManager->folders()->find( rowId );
            if ( folder )
                m_currentProjectId = folder->projectId();
        }
    }

    m_currentProjectAdmin = m_systemAdmin;

    if ( m_currentProjectId != 0 ) {
        int userId = connectionManager->connectionInfo()->userId();
        const MemberRow* member = dataManager->members()->find( userId, m_currentProjectId );
        if ( member && member->access() == AdminAccess )
            m_currentProjectAdmin = true;
    }

    action( "showMembers" )->setEnabled( m_currentProjectId != 0 );
    action( "openFolder" )->setEnabled( m_selectedFolderId != 0 );
    action( "addFolder" )->setEnabled( m_currentProjectId != 0 && m_currentProjectAdmin );
    action( "editRename" )->setEnabled( ( m_selectedProjectId != 0 && m_systemAdmin ) || ( m_selectedFolderId != 0 && m_currentProjectAdmin ) );
    action( "editDelete" )->setEnabled( ( m_selectedProjectId != 0 && m_systemAdmin ) || ( m_selectedFolderId != 0 && m_currentProjectAdmin ) );
    action( "editWatches" )->setEnabled( m_selectedFolderId != 0 );

    if ( connectionManager->connectionInfo()->checkFeature( "notify" ) ) {
        action( "toggleNotify" )->setEnabled( m_selectedFolderId != 0 );
        if ( m_selectedFolderId != 0 ) {
            if ( dataManager->isFolderNotify( m_selectedFolderId ) ) {
                action( "toggleNotify" )->setText( tr( "Disable &Notification" ) );
                action( "toggleNotify" )->setChecked( true );
            } else {
                action( "toggleNotify" )->setText( tr( "Enable &Notification" ) );
                action( "toggleNotify" )->setChecked( false );
            }
        }
    }

    emit selectedFolderChanged( m_selectedFolderId );
}

void ProjectsView::setSelectedFolderId( int folderId )
{
    QModelIndex index = m_model->findCell( 1, folderId, Column_Name );
    if ( index.isValid() ) {
        m_list->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
        m_list->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
    } else {
        if ( m_selectedFolderId != folderId )
            m_list->clearSelection();
    }
}

void ProjectsView::updateProjects()
{
    if ( !isUpdating() ) {
        UpdateBatch* batch = new UpdateBatch();
        batch->updateProjects();

        executeUpdate( batch );
    }
}

void ProjectsView::showMembers()
{
    if ( m_currentProjectId != 0 )
        viewManager->openMembersView( m_currentProjectId );
}

void ProjectsView::addProject()
{
    if ( m_systemAdmin ) {
        AddProjectDialog dialog( mainWidget() );
        dialog.exec();
    }
}

void ProjectsView::addFolder()
{
    if ( m_currentProjectId != 0 && m_currentProjectAdmin ) {
        AddFolderDialog dialog( m_currentProjectId, mainWidget() );
        if ( dialog.exec() == QDialog::Accepted )
            m_list->expand( selectedIndex() );
    }
}

void ProjectsView::editRename()
{
    if ( m_selectedProjectId != 0 && m_systemAdmin ) {
        RenameProjectDialog dialog( m_selectedProjectId, mainWidget() );
        dialog.exec();
    } else if ( m_selectedFolderId != 0 && m_currentProjectAdmin ) {
        RenameFolderDialog dialog( m_selectedFolderId, mainWidget() );
        dialog.exec();
    }
}

void ProjectsView::editDelete()
{
    if ( m_selectedProjectId != 0 && m_systemAdmin ) {
        RDB::ForeignConstIterator<FolderRow> it( dataManager->folders()->parentIndex(), m_selectedProjectId );
        if ( it.next() ) {
            const ProjectRow* project = dataManager->projects()->find( m_selectedProjectId );
            QString name = project ? project->name() : QString();
            QMessageBox::warning( mainWidget(), tr( "Delete Project" ),
                tr( "<qt>Project <b>%1</b> cannot be deleted because it is not empty.</qt>" ).arg( name ) );
            return;
        }
        DeleteProjectDialog dialog( m_selectedProjectId, mainWidget() );
        dialog.exec();
    } else if ( m_selectedFolderId != 0 && m_currentProjectAdmin ) {
        const FolderRow* folder = dataManager->folders()->find( m_selectedFolderId );
        if ( folder && folder->stamp() > 0 ) {
            QString name = folder->name();
            QMessageBox::warning( mainWidget(), tr( "Delete Folder" ),
                tr( "<qt>Folder <b>%1</b> cannot be deleted because it is not empty.</qt>" ).arg( name ) );
            return;
        }
        DeleteFolderDialog dialog( m_selectedFolderId, mainWidget() );
        dialog.exec();
    }
}

void ProjectsView::editWatches()
{
    if ( m_selectedFolderId != 0 ) {
        WatchesDialog dialog( m_selectedFolderId, mainWidget() );
        dialog.exec();
    }
}

void ProjectsView::toggleNotify()
{
    if ( m_selectedFolderId != 0 ) {
        NotifyDialog dialog( mainWidget() );
        if ( dialog.notifyFolder( m_selectedFolderId, !dataManager->isFolderNotify( m_selectedFolderId ) ) )
            dialog.exec();
    }
}

void ProjectsView::openFolder()
{
    if ( m_selectedFolderId != 0  )
        viewManager->openFolderView( m_selectedFolderId );
}

void ProjectsView::updateSummary()
{
    int userId = connectionManager->connectionInfo()->userId();
    const UserRow* user = dataManager->users()->find( userId );
    QString name = user ? user->name() : QString();

    QPixmap pixmap = m_systemAdmin ? IconLoader::overlayedPixmap( "user", "overlay-admin" ) : IconLoader::pixmap( "user" );

    showSummary( pixmap, name );
}

void ProjectsView::updateEvent( UpdateEvent* e )
{
    setAccess( checkDataAccess() );

    if ( e->unit() == UpdateEvent::Users )
        updateSummary();

    if ( e->unit() == UpdateEvent::Projects && !m_firstUpdateDone ) {
        QTimer::singleShot( 0, this, SLOT( projectsPopulated() ) );
        m_firstUpdateDone = true;
    }
}

void ProjectsView::viewEvent( ViewEvent* e )
{
    if ( e->action() == ViewEvent::UpdateWatches )
        m_model->updateData();
    else
        View::viewEvent( e );
}

bool ProjectsView::eventFilter( QObject* obj, QEvent* e )
{
    if ( obj == m_list || obj == m_list->viewport() ) {
        QEvent::Type type = e->type();
        if ( type == QEvent::MouseButtonPress || type == QEvent::MouseButtonDblClick || type == QEvent::KeyPress )
            m_activateReason = type;
    }

    return View::eventFilter( obj, e );
}

void ProjectsView::projectsPopulated()
{
    TreeViewSettings settings;
    settings.openProjectsTree();

    TreeViewHelper::applyExpandedNodes( m_list, settings.loadExpandedNodes() );

    mainWidget()->setFocus();
}

void ProjectsView::contextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );

    if ( index.isValid() ) {
        m_list->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
        m_list->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
    }

    QString menuName;
    if ( index.isValid() ) {
        int level = m_model->data( index, RDB::TableItemModel::LevelRole ).toInt();
        if ( level == 0 )
            menuName = m_systemAdmin ? "contextProjectAdmin" : "contextProject";
        else
            menuName = "contextFolder";
    } else {
        menuName = "contextNull";
    }

    QMenu* menu = builder()->contextMenu( menuName );
    if ( menu )
        menu->exec( m_list->viewport()->mapToGlobal( pos ) );
}

void ProjectsView::activated( const QModelIndex& index )
{
    if ( index.isValid() && ( m_activateReason != QEvent::MouseButtonPress || configData->windowLayout() == LayoutSinglePane ) ) {
        int level = m_model->data( index, RDB::TableItemModel::LevelRole ).toInt();
        int rowId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();

        if ( level == 1 )
            viewManager->openFolderView( rowId );
    }
}

QModelIndex ProjectsView::selectedIndex()
{
    if ( !m_list->selectionModel() )
        return QModelIndex();

    QModelIndexList selection = m_list->selectionModel()->selectedRows();
    if ( selection.isEmpty() )
        return QModelIndex();

    return selection.at( 0 );
}
