/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "membersview.h"

#include <QTreeView>
#include <QAction>
#include <QMenu>

#include "models/tablemodels.h"
#include "models/treeviewsettings.h"
#include "models/treeviewhelper.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/updatebatch.h"
#include "dialogs/userdialogs.h"
#include "xmlui/builder.h"
#include "connectionmanager.h"
#include "iconloader.h"

MembersView::MembersView( QObject* parent, QWidget* parentWidget ) : View( parent ),
    m_model( NULL )
{
    QAction* action;

    action = new QAction( IconLoader::icon( "file-reload" ), tr( "&Update Members" ), this );
    action->setShortcut( QKeySequence::Refresh );
    connect( action, SIGNAL( triggered() ), this, SLOT( updateMembers() ) );
    setAction( "updateMembers", action );

    action = new QAction( IconLoader::icon( "user-new" ), tr( "&Add Members..." ), this );
    action->setShortcut( QKeySequence::New );
    connect( action, SIGNAL( triggered() ), this, SLOT( addMember() ) );
    setAction( "addMember", action );

    action = new QAction( IconLoader::icon( "edit-access" ), tr( "&Change Access..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( changeAccess() ) );
    setAction( "changeAccess", action );

    action = new QAction( IconLoader::icon( "edit-delete" ), tr( "&Remove Members" ), this );
    action->setShortcut( QKeySequence::Delete );
    connect( action, SIGNAL( triggered() ), this, SLOT( removeMember() ) );
    setAction( "removeMember", action );

    setTitle( "menuMain", tr( "&Members" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );

    setButtonStyle( "updateMembers", Qt::ToolButtonTextBesideIcon );
    setButtonStyle( "addMember", Qt::ToolButtonTextBesideIcon );

    loadXmlUiFile( ":/resources/membersview.xml" );

    m_list = new QTreeView( parentWidget );
    m_list->setSortingEnabled( true );
    m_list->setRootIsDecorated( false );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu );
    m_list->setSelectionMode( QAbstractItemView::ExtendedSelection );

    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( contextMenu( const QPoint& ) ) );
    connect( m_list, SIGNAL( activated( const QModelIndex& ) ),
        this, SLOT( activated( const QModelIndex& ) ) );

    setMainWidget( m_list );

    setViewerSizeHint( QSize( 300, 400 ) );
}

MembersView::~MembersView()
{
    if ( isEnabled() ) {
        TreeViewSettings settings;
        settings.openMembersList();

        settings.saveColumnWidths( TreeViewHelper::readColumnWidths( m_list ) );
    }
}

void MembersView::initialUpdate()
{
    m_model = new RDB::TableItemModel( this );
    m_model->setRootTableModel( new MembersTableModel( id(), m_model ),
        dataManager->members()->index().first(), dataManager->members()->index().second(), id() );

    connect( m_model, SIGNAL( layoutChanged() ), this, SLOT( updateActions() ) );

    setAccess( checkDataAccess(), true );
}

Access MembersView::checkDataAccess()
{
    m_projectAdmin = false;

    const ProjectRow* row = dataManager->projects()->find( id() );
    if ( !row )
        return NoAccess;

    if ( connectionManager->connectionInfo()->access() == AdminAccess ) {
        m_projectAdmin = true;
        return AdminAccess;
    }

    int userId = connectionManager->connectionInfo()->userId();
    const MemberRow* member = dataManager->members()->find( userId, id() );
    if ( !member )
        return NoAccess;

    if ( member->access() == AdminAccess )
        m_projectAdmin = true;

    return member->access();
}

void MembersView::enableView()
{
    TreeViewSettings settings;
    settings.openMembersList();

    m_model->setColumns( settings.loadColumns() );

    m_list->setModel( m_model );

    m_list->sortByColumn( 0, Qt::AscendingOrder );

    TreeViewHelper::applyColumnWidths( m_list, settings.loadColumnWidths() );

    connect( m_list->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
        this, SLOT( updateActions() ) );

    updateCaption();
    updateActions();
}

void MembersView::disableView()
{
    m_list->setModel( NULL );

    updateCaption();
}

void MembersView::updateAccess( Access /*access*/ )
{
    action( "addMember" )->setVisible( m_projectAdmin );
    action( "changeAccess" )->setVisible( m_projectAdmin );
    action( "removeMember" )->setVisible( m_projectAdmin );
}

void MembersView::updateCaption()
{
    QString name = tr( "Unknown Project" ) ;
    if ( isEnabled() ) {
        const ProjectRow* row = dataManager->projects()->find( id() );
        if ( row )
            name = row->name();
    }
    setCaption( tr( "Project Members - %1" ).arg( name ) );
}

void MembersView::updateActions()
{
    m_selectedUsers.clear();

    QModelIndexList indexes = selectedIndexes();
    for ( int i = 0; i < indexes.count(); i++ )
        m_selectedUsers.append( m_model->data( indexes.at( i ), RDB::TableItemModel::RowIdRole ).toInt() );

    action( "changeAccess" )->setEnabled( !m_selectedUsers.isEmpty() );
    action( "removeMember" )->setEnabled( !m_selectedUsers.isEmpty() );
}

void MembersView::updateMembers()
{
    if ( isEnabled() && !isUpdating() ) {
        UpdateBatch* batch = new UpdateBatch();
        batch->updateUsers();

        executeUpdate( batch );
    }
}

void MembersView::addMember()
{
    if ( isEnabled() && m_projectAdmin ) {
        AddMemberDialog dialog( id(), mainWidget() );
        dialog.exec();
    }
}

void MembersView::changeAccess()
{
    if ( isEnabled() && !m_selectedUsers.isEmpty() && m_projectAdmin ) {
        ChangeMemberAccessDialog dialog( m_selectedUsers, id(), mainWidget() );
        dialog.exec();
    }
}

void MembersView::removeMember()
{
    if ( isEnabled() && !m_selectedUsers.isEmpty() && m_projectAdmin ) {
        RemoveMemberDialog dialog( m_selectedUsers, id(), mainWidget() );
        dialog.exec();
    }
}

void MembersView::updateEvent( UpdateEvent* e )
{
    setAccess( checkDataAccess() );

    if ( isEnabled() && e->unit() == UpdateEvent::Projects )
        updateCaption();
}

void MembersView::contextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );

    if ( index.isValid() ) {
        m_list->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
        m_list->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
    }

    QString menuName;
    if ( index.isValid() )
        menuName = "contextMember";
    else
        menuName = "contextNull";

    QMenu* menu = builder()->contextMenu( menuName );
    if ( menu )
        menu->exec( m_list->viewport()->mapToGlobal( pos ) );
}

void MembersView::activated( const QModelIndex& index )
{
    if ( index.isValid() && m_projectAdmin ) {
        int userId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();

        QList<int> users;
        users.append( userId );

        ChangeMemberAccessDialog dialog( users, id(), mainWidget() );
        dialog.exec();
    }
}

QList<QModelIndex> MembersView::selectedIndexes()
{
    if ( !m_list->selectionModel() )
        return QModelIndexList();

    return m_list->selectionModel()->selectedRows();
}
