/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef FOLDERVIEW_H
#define FOLDERVIEW_H

#include "view.h"
#include "rdb/tableitemmodel.h"

class QTreeView;
class QModelIndex;
class QMenu;
class QActionGroup;

class IssueRowFilter;
class SearchEditBox;
class SeparatorComboBox;
class ColumnCondition;

/**
* View for displaying issues in a folder.
*/
class FolderView : public View
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent object.
    *
    * @param parentWidget The parent widget of the view's main widget.
    */
    FolderView( QObject* parent, QWidget* parentWidget );

    /**
    * Destructor.
    */
    ~FolderView();

public:
    /**
    * Return the name of the active filter.
    */
    QString selectedFilter() const;

    /**
    * Activate the filter with given name.
    */
    void setSelectedFilter( const QString& filter );

    /**
    * Return the identifier of the selected issue.
    */
    int selectedIssueId() const { return m_selectedIssueId; }

    /**
    * Select issue with given identifier.
    */
    void setSelectedIssueId( int issueId );

    /**
    * Navigate to the given issue, comment or attachment.
    */
    void gotoIssue( int issueId, int itemId );

public: // overrides
    void initialUpdate();

    bool eventFilter( QObject* obj, QEvent* e );

signals:
    /**
    * Emitted when the selected issue is changed.
    */
    void selectedIssueChanged( int issueId );

    /**
    * Emitted when the selected filter is changed.
    */
    void selectedFilterChanged( const QString& filter );

    /**
    * Emitted when given item should be activated.
    */
    void gotoItem( int itemId );

protected: // overrides
    void enableView();
    void disableView();

    void updateEvent( UpdateEvent* e );

    void viewEvent( ViewEvent* e );

private slots:
    void updateActions();
    void updateSummary();

    void updateFolder();
    void openIssue();
    void addIssue();

    void editWatches();
    void toggleMark();
    void markAllRead();
    void markAllNew();

    void toggleNotify();
    void toggleAutoNotify();

    void changeFilter();
    void saveFilterAs();
    void manageFilters();
    void configureColumns();

    void printReport();
    void exportCsv();
    void exportHtml();
    void exportPdf();

    void headerContextMenu( const QPoint& pos );
    void listContextMenu( const QPoint& pos );

    void activated( const QModelIndex& index );

    void quickSearchChanged( const QString& text );
    void searchActionTriggered( QAction* action );

    void filterActivated( int index );

    void applyColumns();
    void applyFilter();

private:
    void cleanUp();

    Access checkDataAccess();

    void updateFolderIfNeeded();

    void updateCaption();

    void updateColumnList( bool resort );
    void updateFilters();
    void updateSearchOptions();

    void markAllIssues( bool read );

    void loadCurrentFilter();

    QModelIndex selectedIndex();

    QList<int> visibleIssues();

private:
    enum FilterType
    {
        NoFilter,
        SettingsFilter,
        NewFilter
    };

private:
    QTreeView* m_list;
    RDB::TableItemModel* m_model;

    IssueRowFilter* m_filter;

    SeparatorComboBox* m_filterCombo;

    SearchEditBox* m_searchBox;

    QMenu* m_searchMenu;
    QActionGroup* m_searchActionGroup;

    FilterType m_filterType;
    QString m_filterName;

    QList<ColumnCondition> m_newConditions;

    int m_searchColumn;

    int m_typeId;
    int m_projectId;

    int m_gotoIssueId;
    int m_gotoItemId;

    int m_selectedIssueId;

    bool m_autoNotify;

    int m_activateReason;
};

#endif
