/*
 *  VME Linux/m68k Loader
 *
 *  (c) Copyright 1998 by Nick Holgate
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#define MVMEBugEntry			15		/* Trap #15 to call MVMEBug */

#define MB_OUTCHR			0x0020
#define MB_OUTSTR			0x0021
#define MB_WRITD			0x0028
#define MB_INCHR			0x0000
#define MB_INSTAT			0x0001
#define MB_DSKRD			0x0010
#define MB_SYMBOLTA			0x0130
#define MB_SYMBOLTD			0x0131
#define MB_RETURN			0x0063
#define MB_RTC_READ			0x0053
#define MB_NETRD			0x0018
#define MB_NETWR			0x0019

/*--------------------------------------------------------------------------*/
/* Board ID data structure
 *
 * Note, bytes 12 and 13 are board no in BCD (0162,0166,0167,0177,etc)
 */

typedef struct {
	unsigned long	bdid;
	unsigned char	rev;
	unsigned char	mth;
	unsigned char	day;
	unsigned char	yr;
	unsigned short	size;
	unsigned short	reserved;
	unsigned short	brdno;
	unsigned char	suffix[2];
	unsigned long	options;
	unsigned short	clun;
	unsigned short	dlun;
	unsigned short	ctype;
	unsigned short	dnum;
	unsigned long	option2;

} BRDID_PKT;

/*--------------------------------------------------------------------------*/
/* system call .NETRD/.NETWR packet template
 */

typedef struct niopcall {
	unsigned char	clun;			/* controller logical unit number		*/
	unsigned char	dlun;			/* device logical unit number			*/
	unsigned short	status;			/* error status word					*/
	unsigned long	address;		/* data transfer address				*/
	unsigned long	length;			/* maximum length of transfer			*/
	unsigned long	offset;			/* byte offset							*/
	unsigned long	time;			/* transfer time in seconds (status)	*/
	unsigned long	bytes;			/* transfer byte count (status)			*/
	unsigned char	filename[64];	/* boot filename buffer string			*/

} NIOPCALL;

/*--------------------------------------------------------------------------*/
/* network boot information block structure template
 */

typedef struct netbootinfo {
	unsigned long	cipa;			/* client IP address 					*/
	unsigned long	sipa;			/* server IP address					*/
	unsigned long	gipa;			/* gateway IP address					*/
	unsigned long	subnetmask;		/* subnet IP address mask				*/
	unsigned long	broadcast;		/* broadcast IP address					*/

} NETBOOTINFO;

/*--------------------------------------------------------------------------*/
/* Disk Read command block
 */
typedef struct {
	unsigned char	clun,
					dlun;
	unsigned short	status;
	unsigned long	address,
					block;
	unsigned short	count;
	unsigned char	flags,
					mod;
} DSKRD_CMND;

/*--------------------------------------------------------------------------*/

/*
        32 bit word at  0x14 block number of start of IPL
        16 bit word at  0x18 number of blocks to load
        32 bit word at  0x1E boot load address
        32 bit word at 0x90 to contain 0 (Media config area start block)
        Locations 0xF8 to 0xFF to contain "MOTOROLA"
*/

typedef struct {
	unsigned long	sp;					/*  00 initial stack pointer		*/
	unsigned long	pc;					/*  04 initial program counter		*/
	char			filler1[12];		/*  08								*/
	unsigned long	boot_start_block;	/*  14 boot file start block addr	*/
	unsigned short	boot_block_count;	/*  18 number of blocks in file		*/
	char			filler2[4];			/*  1a								*/
	unsigned long	boot_load_addr;		/*  1e 								*/
	char			boot_code[110];		/*  22 IPL code						*/
	unsigned long	cfg_block;			/*  90 config data start block		*/
	unsigned char	cfg_count;			/*  94 config data block count		*/
	char			filler3[1];			/*  95 word align frag_list			*/
	char			frag_list[90];		/*  96 loader file fragment list	*/
	char			boot_id[16];		/*  f0 id string "Linux68kMOTOROLA"	*/
} BOOTBLOCK;

#define BOOT_BLOCK_ID  "Linux68kMOTOROLA"

/*--------------------------------------------------------------------------*/
/* Call MVMEBug putchar subroutine.
 */

static
__inline__
void
MVMEBug_putchar
(	int		c
)
{	register unsigned long	d0 asm("d0") = (unsigned long) c;

	asm volatile (
		"	move.b	%%d0,-(%%sp)	\n\
			trap	#15				\n\
			dc.w	0x0020			\n\
		"
		: /* no outputs */
		: "r" (d0)
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug putstr subroutine.
 */

static
__inline__
void
MVMEBug_putstr
(	const char		*str,
	const char		*end
)
{	register void			*a0 asm("a0") = (void *) str;
	register void			*a1 asm("a1") = (void *) end;

	asm volatile (
		"	move.l	%%a1,-(%%sp)	\n\
			move.l	%%a0,-(%%sp)	\n\
			trap	#15				\n\
			dc.w	0x0021			\n\
		"
		: /* no outputs */
		: "r" (a0), "r" (a1)
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug print subroutine.
 */

static
__inline__
void
MVMEBug_print
(	const char		*str,
	void			*vap
)
{	register void			*a0 asm("a0") = (void *) str;
	register void			*a1 asm("a1") = (void *) vap;

	asm volatile (
		"	move.l	%%a1,-(%%sp)	\n\
			move.l	%%a0,-(%%sp)	\n\
			trap	#15				\n\
			dc.w	0x0028			\n\
		"
		: /* no outputs */
		: "r" (a0), "r" (a1)
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug getchar subroutine.
 */

static
__inline__
int
MVMEBug_getchar
(	void
)
{	register unsigned long	d0 asm("d0") = 0;

	asm volatile (
		"	subq.l	#2,%%sp			\n\
			trap	#15				\n\
			dc.w	0x0000			\n\
			move.b	(%%sp)+,%%d0	\n\
		"
		: "=r" (d0)
		: "r" (d0)
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug input status subroutine.
 */

static
__inline__
int
MVMEBug_getchar_status
(	void
)
{	register unsigned long	d0 asm("d0");

	asm volatile (
		"	move.l	#0,-(%%sp)		\n\
			trap	#15				\n\
			dc.w	0x0001			\n\
			beq.s	1f				\n\
			move.l	#1,(%%sp)		\n\
		1:							\n\
			move.l	(%%sp)+,%%d0	\n\
		"
		: "=r" (d0)
		: /* no inputs */
	);

	return d0;
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug RTC read subroutine.
 */

static
__inline__
void
MVMEBug_rtc_read
(	void			*buf
)
{	register void *			d0 asm("d0") = (void *) buf;

	asm volatile (
		"	move.l	%%d0,-(%%sp)	\n\
			trap	#15				\n\
			dc.w	0x0053			\n\
		"
		: /* no outputs */
		: "r" (d0)
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug disk_read subroutine.
 */

static
__inline__
void
MVMEBug_disk_read
(	DSKRD_CMND		*cmnd
)
{	register void	*d0 asm("d0") = (void *) cmnd;

	asm volatile (
		"	move.l	%%d0,-(%%sp)	\n\
			trap	#15				\n\
			dc.w	0x0010			\n\
		"
		: /* no outputs */
		: "r" (d0)
		: "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug net_read subroutine.
 */

static
__inline__
void
MVMEBug_net_read
(	NIOPCALL		*niop
)
{	register void	*d0 asm("d0") = (void *) niop;

	asm volatile (
		"	move.l	%%d0,-(%%sp)	\n\
			trap	#15				\n\
			dc.w	0x0018			\n\
		"
		: /* no outputs */
		: "r" (d0)
		: "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug.
 */

static
__inline__
void
MVMEBug_enter
(	void
)
{
	asm volatile (
		"	trap	#15				\n\
			dc.w	0x0063			\n\
		"
		: /* No outputs */
		: /* No inputs  */
		: "d0", "d1", "d2", "d3", "d4", "d5", "d6", "d7",
		  "a0", "a1", "a2", "a3", "a4", "a5", "a6", "memory"
	);
}

/*--------------------------------------------------------------------------*/
/* Call MVMEBug BRD_ID subroutine.
 */

static
__inline__
BRDID_PKT *
MVMEBug_brdid
(	void
)
{	register void *			d0 asm("d0");

	asm volatile (
		"	subq.l	#4,%%sp			\n\
			trap	#15				\n\
			dc.w	0x0070			\n\
			move.l	(%%sp)+,%%d0	\n\
		"
		: "=r" (d0)
		: /* no inputs */
	);

	return d0;
}

/*-----------------------------< end of file >------------------------------*/
