/* ========================================================================== */
/* === UMFPACK_triplet_to_col =============================================== */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* UMFPACK Version 3.2 (Jan. 1, 2002), Copyright (c) 2002 by Timothy A.       */
/* Davis, University of Florida, davis@cise.ufl.edu.  All Rights Reserved.    */
/* See README, umfpack.h, or type "umfpack_details" in Matlab for License.    */
/* -------------------------------------------------------------------------- */

/*
    User callable.  Converts triplet input to column-oriented form.  Duplicate
    entries may exist (they are summed in the output).  The columns of the
    column-oriented form are in sorted order.  The input is not modified.
    Returns 1 if OK, 0 if an error occured.  See umfpack_triplet_to_col.h for
    details.

    Dynamic memory usage:

	If numerical values are present (both Tx and Bx), then a workspace of
	size (nz+1)*sizeof(double) is allocated via UMF_malloc.  Next, 4
	calls to UMF_malloc are made to obtain workspace of size
	((nz+1) + (n+1) + 2*n) * sizeof(Int).  All of this workspace (4 or 5
	objects) are free'd via UMF_free on return.

*/

#include "umf_internal.h"
#include "umf_malloc.h"
#include "umf_free.h"

#ifndef NDEBUG
PRIVATE Int init_count ;
#endif

/* ========================================================================== */

GLOBAL Int UMFPACK_triplet_to_col
(
    Int n,
    Int nz,
    const Int Ti [ ],
    const Int Tj [ ],
    const double Tx [ ],
    Int Bp [ ],
    Int Bi [ ],
    double Bx [ ]
)
{

    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    Int i, j, k, p, cp, p1, p2, pdest, pj, *RowCount, *RowCount1, *ColCount,
	*Rp, *Rj, *W ;
    double *Rx ;

#ifndef NDEBUG
    init_count = UMF_malloc_count ;
#endif

    /* ---------------------------------------------------------------------- */
    /* check inputs */
    /* ---------------------------------------------------------------------- */

    if (!Bi || !Bp || !Ti || !Tj)
    {
	return (UMFPACK_ERROR_argument_missing) ;
    }

    if (n <= 0)		/* n must be > 0 */
    {
	return (UMFPACK_ERROR_n_nonpositive) ;
    }

    if (nz < 0)		/* nz must be >= 0 (singular matrices are OK) */
    {
	return (UMFPACK_ERROR_nz_negative) ;
    }

    /* ---------------------------------------------------------------------- */
    /* count the entries in each row (also counting duplicates) */
    /* ---------------------------------------------------------------------- */

    /* use Bi as workspace for RowCount */
    RowCount1 = Bi ;
    for (i = 0 ; i < n ; i++)
    {
	Bi [i] = 0 ;
    }

    for (k = 0 ; k < nz ; k++)
    {
	i = Ti [k] ;
	j = Tj [k] ;
	if (i < 0 || i >= n || j < 0 || j >= n)
	{
	    return (UMFPACK_ERROR_invalid_triplet) ;
	}
	RowCount1 [i]++ ;
    }

    /* ---------------------------------------------------------------------- */
    /* allocate workspace */
    /* ---------------------------------------------------------------------- */

    Rx = (double *) NULL ;
    if (Bx && Tx)
    {
	Rx = (double *) UMF_malloc (nz+1, sizeof (double)) ;
	if (!Rx)
	{
	    ASSERT (UMF_malloc_count == init_count) ;
	    return (UMFPACK_ERROR_out_of_memory) ;
	}
    }

    Rj = (Int *) UMF_malloc (nz+1, sizeof (Int)) ;
    Rp = (Int *) UMF_malloc (n+1, sizeof (Int)) ;
    RowCount = (Int *) UMF_malloc (n, sizeof (Int)) ;
    W = (Int *) UMF_malloc (n, sizeof (Int)) ;
    if (!Rj || !Rp || !RowCount || !W)
    {
	(void) UMF_free ((void *) Rx) ;
	(void) UMF_free ((void *) Rp) ;
	(void) UMF_free ((void *) Rj) ;
	(void) UMF_free ((void *) RowCount) ;
	(void) UMF_free ((void *) W) ;
	ASSERT (UMF_malloc_count == init_count) ;
	return (UMFPACK_ERROR_out_of_memory) ;
    }
    ASSERT (UMF_malloc_count == init_count + 4 + (Bx && Tx)) ;

    /* ---------------------------------------------------------------------- */
    /* compute the row pointers */
    /* ---------------------------------------------------------------------- */

    Rp [0] = 0 ;
    for (i = 0 ; i < n ; i++)
    {
	Rp [i+1] = Rp [i] + RowCount1 [i] ;
	W [i] = Rp [i] ;
    }

    /* ---------------------------------------------------------------------- */
    /* construct the row form */
    /* ---------------------------------------------------------------------- */

    for (k = 0 ; k < nz ; k++)
    {
	p = W [Ti [k]]++ ;
	Rj [p] = Tj [k] ;
	if (Rx && Tx)
	{
	    Rx [p] = Tx [k] ;
	}
    }

#ifndef NDEBUG
    for (i = 0 ; i < n ; i++)
    {
	ASSERT (W [i] == Rp [i+1]) ;
    }
#endif

    /* ---------------------------------------------------------------------- */
    /* sum up duplicates */
    /* ---------------------------------------------------------------------- */

    for (j = 0 ; j < n ; j++)
    {
	W [j] = EMPTY ;
    }

    for (i = 0 ; i < n ; i++)
    {
	p1 = Rp [i] ;
	p2 = Rp [i+1] ;
	pdest = p1 ;
	for (p = p1 ; p < p2 ; p++)
	{
	    j = Rj [p] ;
	    pj = W [j] ;
	    if (pj >= p1)
	    {
		/* this column index, j, is already in row i, at position pj */
		ASSERT (pj < p) ;
		ASSERT (Rj [pj] == j) ;
		/* print the entry if print level is high enough */
		/* sum the entry */
		if (Rx)
		{
		    Rx [pj] += Rx [p] ;
		}
	    }
	    else
	    {
		/* keep the entry */
		W [j] = pdest ;
		if (pdest != p)
		{
		    Rj [pdest] = j ;
		    if (Rx)
		    {
			Rx [pdest] = Rx [p] ;
		    }
		}
		pdest++ ;
	    }
	}
	RowCount [i] = pdest - p1 ;
    }

    /* ---------------------------------------------------------------------- */
    /* count the entries in each column */
    /* ---------------------------------------------------------------------- */

    /* [ use Bi as work space for ColCount */
    ColCount = Bi ;
    for (j = 0 ; j < n ; j++)
    {
	ColCount [j] = 0 ;
    }

    for (i = 0 ; i < n ; i++)
    {
	for (p = Rp [i] ; p < Rp [i] + RowCount [i] ; p++)
	{
	    j = Rj [p] ;
	    ColCount [j]++ ;
	}
    }

    /* ---------------------------------------------------------------------- */
    /* create the column pointers */
    /* ---------------------------------------------------------------------- */

    Bp [0] = 0 ;
    for (j = 0 ; j < n ; j++)
    {
	Bp [j+1] = Bp [j] + ColCount [j] ;
    }
    /* done using Bi as workspace for ColCount ] */
    for (j = 0 ; j < n ; j++)
    {
	W [j] = Bp [j] ;
    }

    /* ---------------------------------------------------------------------- */
    /* construct the column form */
    /* ---------------------------------------------------------------------- */

    for (i = 0 ; i < n ; i++)
    {
	for (p = Rp [i] ; p < Rp [i] + RowCount [i] ; p++)
	{
	    cp = W [Rj [p]]++ ;
	    Bi [cp] = i ;
	    if (Bx && Rx)
	    {
		Bx [cp] = Rx [p] ;
	    }
	}
    }

#ifndef NDEBUG
    for (j = 0 ; j < n ; j++)
    {
	ASSERT (W [j] == Bp [j+1]) ;
    }
#endif

    /* ---------------------------------------------------------------------- */
    /* free the workspace */
    /* ---------------------------------------------------------------------- */

    (void) UMF_free ((void *) Rx) ;
    (void) UMF_free ((void *) Rp) ;
    (void) UMF_free ((void *) Rj) ;
    (void) UMF_free ((void *) RowCount) ;
    (void) UMF_free ((void *) W) ;
    ASSERT (UMF_malloc_count == init_count) ;

    return (UMFPACK_OK) ;
}

