/************************************************************************/
/*									*/
/*  Ted, Screen drawing and forcing drawing through			*/
/*  appExposeRectangle().						*/
/*									*/
/************************************************************************/

#   include	"tedConfig.h"

#   include	<stddef.h>
#   include	<stdio.h>

#   include	"tedApp.h"
#   include	"tedLayout.h"
#   include	"docDraw.h"

#   include	<appDebugon.h>

# define DRDEB(dr) APP_DEB(appDebug( "%s(%3d) %s=[%4d..%4d,%4d..%4d]\n", \
			     __FILE__, __LINE__, #dr, \
			     (dr)->drX0, (dr)->drX1, (dr)->drY0, (dr)->drY1 ))

#   define	LOG_REDRAWS	0
#   define	BLINK_IBAR	1

#   if		LOG_REDRAWS
#	define	BLINK_IBAR	0
#   endif

typedef struct ScreenDrawingData
    {
    int			sddForegroundSet;
#			define	SDDpixelUNDEF	0
#			define	SDDpixelFORE	1
#			define	SDDpixelLINK	2
#			define	SDDpixelBACK	3
#			define	SDDpixelGRID	4
#			define	SDDpixelPAGE	5

    APP_COLOR_RGB	sddForeColor;
    APP_COLOR_RGB	sddLinkColor;
    APP_COLOR_RGB	sddBackColor;
    APP_COLOR_RGB	sddGridColor;
    APP_COLOR_RGB	sddPageColor;

    int			sddOx;
    int			sddOy;
    } ScreenDrawingData;

/************************************************************************/
/*									*/
/*  Cause the smallest rectangle that contains the selection to be	*/
/*  redrawn.								*/
/*									*/
/************************************************************************/

void tedExposeSelection(	const EditDocument *		ed,
				const DocumentSelection *	ds,
				int				scrolledX,
				int				scrolledY )
    {
    TedDocument *		td= (TedDocument *)ed->edPrivateData;
    BufferDocument *		bd= td->tdDocument;

    const AppDrawingData *	add= &(ed->edDrawingData);

    SelectionGeometry		sg;

    tedSelectionGeometry( &sg, ds, bd, add );

    appDocExposeRectangle( ed, &(sg.sgRectangle), scrolledX, scrolledY );

    return;
    }

/************************************************************************/
/*									*/
/*  Blinking cursor.							*/
/*									*/
/*  1)  Turned off when we are debugging exposures and redraws.		*/
/*									*/
/************************************************************************/

void tedUndrawIBar(	const EditDocument *	ed )
    {
#   if BLINK_IBAR

    TedDocument *		td= (TedDocument *)ed->edPrivateData;

    DocumentSelection		ds;
    SelectionGeometry		sg;

    const int			scrolledX= 0;
    const int			scrolledY= 0;

    if  ( tedGetSelection( &ds, &sg, td ) )
	{ LDEB(1); return;	}

    appDocExposeRectangle( ed, &(sg.sgRectangle), scrolledX, scrolledY );

#   endif

    return;
    }

/************************************************************************/
/*									*/
/*  Optimise GC changes.						*/
/*									*/
/************************************************************************/

static void tedDrawSetForegroundFun(	AppDrawingData *	add,
					ScreenDrawingData *	sdd,
					int			which )
    {
    APP_COLOR_RGB *	pix;

    if  ( sdd->sddForegroundSet == which )
	{ return;	}

    switch( which )
	{
	case SDDpixelFORE:	pix= &(sdd->sddForeColor);	break;
	case SDDpixelLINK:	pix= &(sdd->sddLinkColor);	break;
	case SDDpixelBACK:	pix= &(sdd->sddBackColor);	break;
	case SDDpixelGRID:	pix= &(sdd->sddGridColor);	break;
	case SDDpixelPAGE:	pix= &(sdd->sddPageColor);	break;

	case SDDpixelUNDEF:
	    sdd->sddForegroundSet= which;
	    return;
	default:
	    LDEB(which); return;
	}

    appDrawSetForegroundColor( add, pix );

    sdd->sddForegroundSet= which;
    }

#   define tedDrawSetForeground( add, sdd, which )		\
    if  ( (sdd)->sddForegroundSet != (which) )			\
	{ tedDrawSetForegroundFun( (add), (sdd), (which) );	}

/************************************************************************/
/*									*/
/*  Various Border drawing routines.					*/
/*									*/
/************************************************************************/

static void tedDrawHorizontalBorder(	const BorderProperties *	bp,
					int				asGrid,
					const DrawingContext *		dc,
					ScreenDrawingData *		sdd,
					int				above,
					int				x0,
					int				x1,
					int				y )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			wide;
    int			thick= tedBorderThick( &wide, bp, add );

    if  ( asGrid && thick == 0 )
	{ thick= 1;	}

    if  ( thick > 0 )
	{
	const DocumentRectangle *	drClip= dc->dcClipRect;
	DocumentRectangle		drBorder;

	if  ( above )
	    { y -= thick;	}

	drBorder.drX0= x0;
	drBorder.drX1= x1;
	drBorder.drY0= y;
	drBorder.drY1= y+ thick- 1;

	if  ( ! drClip							  ||
	      docIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    appDrawFillRectangle( add,
			drBorder.drX0- sdd->sddOx,
			drBorder.drY0- sdd->sddOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    return;
    }

static void tedDrawVerticalBorder(	const BorderProperties *	bp,
					int				asGrid,
					const DrawingContext *		dc,
					ScreenDrawingData *		sdd,
					int				x,
					int				y0,
					int				y1 )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    int			wide;
    int			thick= tedBorderThick( &wide, bp, add );

    if  ( asGrid && thick == 0 )
	{ thick= 1;	}

    if  ( thick > 0 )
	{
	const DocumentRectangle *	drClip= dc->dcClipRect;
	DocumentRectangle		drBorder;

	drBorder.drX0= x;
	drBorder.drX1= x+ thick- 1;
	drBorder.drY0= y0- 1;
	drBorder.drY1= y1;

	if  ( ! drClip							  ||
	      docIntersectRectangle( &drBorder, &drBorder, dc->dcClipRect ) )
	    {
	    appDrawFillRectangle( add,
			drBorder.drX0- sdd->sddOx,
			drBorder.drY0- sdd->sddOy,
			drBorder.drX1- drBorder.drX0+ 1,
			drBorder.drY1- drBorder.drY0+ 1 );
	    }
	}

    return;
    }

/************************************************************************/
/*									*/
/*  Draw a text item.							*/
/*									*/
/*  1)  Tabs need not to be drawn.					*/
/*									*/
/************************************************************************/

static int tedDrawTab(	const BufferItem *	bi,
			const TextParticule *	tp,
			int			baseLine,
			ScreenDrawingData *	sdd,
			DrawingContext *	dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    TabStop *		ts= bi->biParaTabStops+ tp->tpObjectNumber;

    int			x0;
    int			x1;

    AppPhysicalFont *	apf;

    switch( ts->tsLeader )
	{
	case DOCtlNONE:
	    break;

	case DOCtlDOTS:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return -1;	}
	    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;

	    x0= tp->tpX0+ apf->apfFullSizePixels/ 4;
	    x1= tp->tpX0+ tp->tpPixelsWide- apf->apfFullSizePixels/ 2;

	    x0= 3* ( ( x0+ 2 )/ 3 );

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		static char	boldDot[]= { 2, 1 };

#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    2, LineOnOffDash, CapButt, JoinMiter );

		XSetDashes( add->addDisplay, add->addGc,
					    0, boldDot, sizeof( boldDot ) );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			2, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

		gdk_gc_set_dashes( add->addGc, 0, boldDot, sizeof( boldDot ) );
#		endif

		}
	    else{
		static char	dot[]= { 1, 2 };

#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    1, LineOnOffDash, CapButt, JoinMiter );

		XSetDashes( add->addDisplay, add->addGc,
						0, dot, sizeof( dot ) );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

		gdk_gc_set_dashes( add->addGc, 0, dot, sizeof( dot ) );
#		endif
		}

	    appDrawDrawLine( add,
				    x0- sdd->sddOx, baseLine- sdd->sddOy,
				    x1- sdd->sddOx, baseLine- sdd->sddOy );

	    break;

	case DOCtlUNDERLINE:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return -1;	}
	    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;

	    x0= tp->tpX0+ apf->apfFullSizePixels/ 4;
	    x1= tp->tpX0+ tp->tpPixelsWide- apf->apfFullSizePixels/ 2;

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    2, LineSolid, CapButt, JoinMiter );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			2, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER );
#		endif
		}
	    else{
#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
					1, LineSolid, CapButt, JoinMiter );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			1, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER );
#		endif
		}

	    appDrawDrawLine( add,
				    x0- sdd->sddOx, baseLine- sdd->sddOy,
				    x1- sdd->sddOx, baseLine- sdd->sddOy );

	    break;

	case DOCtlHYPH:
	    if  ( tp->tpPhysicalFont < 0 )
		{ LDEB(tp->tpPhysicalFont); return -1;	}
	    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;

	    x0= tp->tpX0+ apf->apfFullSizePixels/ 4;
	    x1= tp->tpX0+ tp->tpPixelsWide- apf->apfFullSizePixels/ 2;

	    x0= 7* ( ( x0+ 6 )/ 7 );

	    if  ( x1 <= x0 )
		{ return 0;	}

	    if  ( tp->tpTextAttribute.taFontIsBold )
		{
		static char	boldDash[]= { 4, 3 };

#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    2, LineOnOffDash, CapButt, JoinMiter );

		XSetDashes( add->addDisplay, add->addGc,
					    0, boldDash, sizeof( boldDash ) );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			2, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

		gdk_gc_set_dashes( add->addGc,
					    0, boldDash, sizeof( boldDash ) );
#		endif

		}
	    else{
		static char	dash[]= { 3, 4 };

#		ifdef USE_MOTIF
		XSetLineAttributes( add->addDisplay, add->addGc,
				    1, LineOnOffDash, CapButt, JoinMiter );

		XSetDashes( add->addDisplay, add->addGc,
						0, dash, sizeof( dash ) );
#		endif

#		ifdef USE_GTK
		gdk_gc_set_line_attributes( add->addGc,
			1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

		gdk_gc_set_dashes( add->addGc, 0, dash, sizeof( dash ) );
#		endif
		}

	    appDrawDrawLine( add,
				    x0- sdd->sddOx, baseLine- sdd->sddOy,
				    x1- sdd->sddOx, baseLine- sdd->sddOy );

	    break;

	    break;

	case DOCtlTHICK:
	    LDEB(ts->tsLeader);
	    break;
	case DOCtlEQUAL:
	    LDEB(ts->tsLeader);
	    break;
	default:
	    LDEB(ts->tsLeader);
	    break;
	}

    return 0;
    }

static int tedDrawParticules(	const BufferItem *	bi,
				const TextParticule *	tp,
				int			count,
				int			baseLine,
				ScreenDrawingData *	sdd,
				DrawingContext *	dc )
    {
    AppDrawingData *	add= dc->dcDrawingData;
    unsigned char *	paraString= bi->biParaString;

    int			drawn;
    int			i;
    int			len;
    int			y;

    AppPhysicalFont *	apf;

    /*  1  */
    switch( tp->tpKind )
	{ 
	case DOCkindTAB:
	    if  ( tp->tpObjectNumber >= 0			&&
		  tp->tpObjectNumber < bi->biParaTabCount	)
		{
		if  ( tedDrawTab( bi, tp, baseLine, sdd, dc ) )
		    { LDEB(1);	}
		}

	    return drawn= 1;

	case DOCkindTEXT:
	    break;

	case DOCkindFIELDSTART:
	case DOCkindFIELDEND:
	case DOCkindXE:
	case DOCkindTC:
	case DOCkindLINEBREAK:
	case DOCkindPAGEBREAK:
	case DOCkindNOTE:
	    return drawn= 1;

	case DOCkindOBJECT:
	    if  ( tedDrawObject( bi, tp, baseLine,
					    sdd->sddOx, sdd->sddOy, add ) )
		{ LDEB(1); return -1;	}

	    return drawn= 1;

	case DOCkindCHFTNSEP:
#	    ifdef USE_MOTIF
	    XSetLineAttributes( add->addDisplay, add->addGc,
					1, LineSolid, CapButt, JoinMiter );
#	    endif

#	    ifdef USE_GTK
	    gdk_gc_set_line_attributes( add->addGc,
			1, GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER );
#	    endif

	    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;
	    y= baseLine;
	    y -= ( apf->apfXHeightPixels )/ 2;

	    appDrawDrawLine( add,
		    tp->tpX0- sdd->sddOx, y- sdd->sddOy,
		    tp->tpX0+ tp->tpPixelsWide- sdd->sddOx, y- sdd->sddOy );

	    return drawn= 1;

	default:
	    LDEB(tp->tpKind); return -1;
	}

    drawn= 1;
    len= tp[drawn-1].tpStroff+ tp[drawn-1].tpStrlen- tp->tpStroff;

    if  ( tp->tpPhysicalFont < 0 )
	{ LDEB(tp->tpPhysicalFont); return -1;	}

    apf= add->addPhysicalFontList.apflFonts+ tp->tpPhysicalFont;
    y= baseLine;

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUPERSCRIPT )
	{ y -= ( 10* apf->apfXHeightPixels )/ 6; }

    if  ( tp->tpTextAttribute.taSuperSub == DOCfontSUBSCRIPT )
	{ y += ( 3* apf->apfFullSizePixels )/ 10; }

    while( len > 0 && paraString[tp->tpStroff+ len- 1] == ' ' )
	{ len--;	}

    if  ( len > 0 )
	{
	if  ( tp->tpPhysicalFont != dc->dcCurrentPhysicalFont )
	    {
	    appDrawSetFont( add, apf->apfFontStruct );

	    dc->dcCurrentPhysicalFont= tp->tpPhysicalFont;
	    }

	appDrawDrawString( add,
				tp->tpX0- sdd->sddOx, y- sdd->sddOy,
				(char *)paraString+ tp->tpStroff, len );

	/*
	appDebug( "[%3d,%3d] -- %.*s\n",
		y, tp->tpX0,
		len, (char *)paraString+ tp->tpStroff );
	*/
	}

    i= 0;
    while( i < drawn )
	{
	int	x0;
	int	x1;
	int	y0;
	int	h;

	if  ( ! tp[i].tpTextAttribute.taIsUnderlined )
	    { i++; continue;	}

	x1= x0= tp[i].tpX0;
	y0= baseLine+ apf->apfUnderLinePositionPixels;
	h= apf->apfUnderlineThicknessPixels;

	while( i < drawn && tp[i].tpTextAttribute.taIsUnderlined )
	    { x1= tp[i].tpX0+ tp[i].tpPixelsWide; i++;	}

	appDrawFillRectangle( add, x0- sdd->sddOx, y0- sdd->sddOy, x1- x0, h );
	}

    return drawn;
    }

static int tedDrawTextLine(	const BufferItem *		bi,
				int				line,
				const ParagraphFrame *		pf,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;

    const TextLine *		tl= bi->biParaLines+ line;
    const TextParticule *	tp= bi->biParaParticules+ tl->tlFirstParticule;

    AppDrawingData *		add= dc->dcDrawingData;
    int				done;

    done= 0;
    while( done < tl->tlParticuleCount )
	{
	int		drawn;
	int		baseline;

	if  ( tp->tpTextAttribute.taShowAsLink )
	    { tedDrawSetForeground( add, sdd, SDDpixelLINK );	}
	else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

	baseline= TL_BASE_PIXELS( add, tl );

	drawn= tedDrawParticules( bi, tp, tl->tlParticuleCount- done,
							baseline, sdd, dc );
	if  ( drawn < 1 )
	    { LDEB(drawn); return -1;	}

	done += drawn; tp += drawn;
	}

    return done;
    }

static int tedLineRectangle(	DocumentRectangle *		drRedraw,
				const BufferItem *		bi,
				DrawingContext *		dc,
				int				line,
				const ParagraphFrame *		pf )
    {
    const DocumentSelection *	ds= dc->dcDocumentSelection;
    const SelectionGeometry *	sg= dc->dcSelectionGeometry;
    const DocumentRectangle *	drClip= dc->dcClipRect;

    DocumentSelection	dsLine;
    SelectionGeometry	sgLine;

    DocumentRectangle	drLine;

    const int		mindPart= 1;

    docLineSelection( &dsLine, bi, line );
    tedSelectionGeometry( &sgLine, &dsLine, dc->dcDocument, dc->dcDrawingData );

    drLine= sgLine.sgRectangle;

    if  ( docComparePositions( &(ds->dsBegin),
					&(dsLine.dsBegin), mindPart ) < 0 )
	{
	drLine.drX0= pf->pfX0Pixels;
	}
    else{ drLine.drX0= sg->sgBegin.pgXPixels;	}

    if  ( docComparePositions( &(ds->dsEnd), &(dsLine.dsEnd), mindPart ) > 0 )
	{ drLine.drX1= pf->pfX1Pixels;			}
    else{ drLine.drX1= sg->sgEnd.pgXPixels;		}

    if  ( ! drClip || docIntersectRectangle( drRedraw, drClip, &drLine ) )
	{ return 1;	}
    else{ return 0;	}
    }

static int tedDrawTextReverse(	const BufferItem *		bi,
				int				line,
				const ParagraphFrame *		pf,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    DocumentRectangle		drRedraw;
    int				done;

    if  ( tedLineRectangle( &drRedraw, bi, dc, line, pf ) )
	{
	drRedraw.drX0 -= sdd->sddOx;
	drRedraw.drX1 -= sdd->sddOx;
	drRedraw.drY0 -= sdd->sddOy;
	drRedraw.drY1 -= sdd->sddOy;

	appDrawSetClipRect( add, &drRedraw );

	done= tedDrawTextLine( bi, line, pf, vsdd, dc );
	}
    else{
	TextLine *	tl= bi->biParaLines+ line;

	done= tl->tlParticuleCount;
	}

    return done;
    }

static int tedHighlightBack(	const BufferItem *		bi,
				int				line,
				const ParagraphFrame *		pf,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    TextLine *			tl= bi->biParaLines+ line;

    DocumentRectangle		drRedraw;

    if  ( tedLineRectangle( &drRedraw, bi, dc, line, pf ) )
	{
	tedDrawSetForeground( add, sdd, SDDpixelBACK );

	appDrawFillRectangle( add,
		drRedraw.drX0- sdd->sddOx,
		drRedraw.drY0- sdd->sddOy,
		drRedraw.drX1- drRedraw.drX0+ 1,
		drRedraw.drY1- drRedraw.drY0+ 1 );
	}

    return tl->tlParticuleCount;
    }

/************************************************************************/
/*									*/
/*  Draw a single paragraph. The Graphical context has been prepared	*/
/*  by the caller.							*/
/*									*/
/*  7)	Do not print below the bottom of fixed height cells. ( Word	*/
/*	does not do so either ).					*/
/*									*/
/************************************************************************/

static int tedDrawParaBottom(	const BufferItem *		bi,
				const BorderProperties *	bp,
				const ParagraphFrame *		pf,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;
    double			xfac= add->addMagnifiedPixelsPerTwip;
    int				spaceAfterPixels;

    const int			asGrid= 0;

    spaceAfterPixels= TWIPStoPIXELS( xfac, bi->biParaSpaceAfterTwips );

    tedDrawSetForeground( add, sdd, SDDpixelFORE );

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 1,
			    pf->pfX0Pixels, pf->pfX1Pixels,
			    BI_BELOW_PIXELS( add, bi )- spaceAfterPixels -1 );

    return 0;
    }


static int tedDrawParaTop(	const BufferItem *		bi,
				const BorderProperties *	bp,
				const ParagraphFrame *		pf,
				void *				vsdd,
				DrawingContext *		dc )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;
    double			xfac= add->addMagnifiedPixelsPerTwip;
    int				spaceBeforePixels;

    const int			asGrid= 0;

    spaceBeforePixels= TWIPStoPIXELS( xfac, bi->biParaSpaceBeforeTwips );

    tedDrawSetForeground( add, sdd, SDDpixelFORE );

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 0,
			    pf->pfX0Pixels, pf->pfX1Pixels,
			    BI_TOP_PIXELS( add, bi )+ spaceBeforePixels );
    return 0;
    }

static int tedDrawCellTop(	const BorderProperties *	bp,
				int				asGrid,
				int				ign_x0Twips,
				int				x0Pixels,
				int				ign_x1Twips,
				int				x1Pixels,
				void *				vsdd,
				DrawingContext *		dc,
				const LayoutPosition *		lpTop )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 1,
				    x0Pixels, x1Pixels,
				    LP_YPIXELS( add, lpTop ) );

    return 0;
    }

static int tedDrawCellLeft(	const BorderProperties *	bp,
				int				asGrid,
				void *				vsdd,
				DrawingContext *		dc,
				int				x0Twips,
				int				x0Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawVerticalBorder( bp, asGrid, dc, sdd, x0Pixels,
				    LP_YPIXELS( add, lpTop ),
				    LP_YPIXELS( add, lpBelow ) );

    return 0;
    }

static int tedDrawCellRight(	const BorderProperties *	bp,
				int				asGrid,
				void *				vsdd,
				DrawingContext *		dc,
				int				x1Twips,
				int				x1Pixels,
				const LayoutPosition *		lpTop,
				const LayoutPosition *		lpBelow )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawVerticalBorder( bp, asGrid, dc, sdd, x1Pixels,
				    LP_YPIXELS( add, lpTop ),
				    LP_YPIXELS( add, lpBelow ) );

    return 0;
    }

static int tedDrawCellBottom(	const BorderProperties *	bp,
				int				asGrid,
				int				ign_x0Twips,
				int				x0Pixels,
				int				ign_x1Twips,
				int				x1Pixels,
				void *				vsdd,
				DrawingContext *		dc,
				const LayoutPosition *		lpBottom )
    {
    ScreenDrawingData *		sdd= (ScreenDrawingData *)vsdd;
    AppDrawingData *		add= dc->dcDrawingData;

    if  ( asGrid )
	{ tedDrawSetForeground( add, sdd, SDDpixelGRID );	}
    else{ tedDrawSetForeground( add, sdd, SDDpixelFORE );	}

    tedDrawHorizontalBorder( bp, asGrid, dc, sdd, 0,
				    x0Pixels, x1Pixels,
				    LP_YPIXELS( add, lpBottom ) );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Skip to the next page.						*/
/*									*/
/************************************************************************/

static void tedDrawPageGap(	ScreenDrawingData *		sdd,
				DrawingContext *		dc,
				int				page )
    {
    AppDrawingData *		add= dc->dcDrawingData;

    DocumentRectangle		drBetween;

    drBetween.drX0= add->addBackRect.drX0;
    drBetween.drX1= add->addBackRect.drX1;
    drBetween.drY0= page* add->addPageStepPixels- add->addPageGapPixels;
    drBetween.drY1= page* add->addPageStepPixels;

    if  ( dc->dcClipRect						   &&
	  ! docIntersectRectangle( &drBetween, &drBetween, dc->dcClipRect ) )
	{ return;	}

    tedDrawSetForeground( add, sdd, SDDpixelPAGE );

    appDrawFillRectangle( add,
					drBetween.drX0- sdd->sddOx,
					drBetween.drY0- sdd->sddOy,
					drBetween.drX1- drBetween.drX0+ 1,
					drBetween.drY1- drBetween.drY0+ 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw an I Bar.							*/
/*									*/
/************************************************************************/
#   define	IW	5

int tedDrawIBar(	const PositionGeometry *	pg,
			int				ox,
			int				oy,
			AppDrawingData *		add )
    {
    int		y0= PG_TOP_PIXELS( add, pg );

    appDrawFillRectangle( add, pg->pgXPixels- ox, y0- oy,
						1, pg->pgY1Pixels- y0 );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Draw the eight blocks around a selected object.			*/
/*									*/
/************************************************************************/

static void tedDrawObjectBlocks(	int			x0,
					int			y,
					int			wide,
					int			high,
					const DrawingContext *	dc,
					ScreenDrawingData *	sdd )
    {
    AppDrawingData *	add= dc->dcDrawingData;

    APP_POINT		xp[8];
    int			i;

    /*  bottom  */
    xp[0].x= x0- sdd->sddOx;
    xp[0].y= y- RESIZE_BLOCK- sdd->sddOy;

    xp[1].x= x0 +wide/2- RESIZE_BLOCK/2- sdd->sddOx;
    xp[1].y= y- RESIZE_BLOCK- sdd->sddOy;

    xp[2].x= x0 +wide- RESIZE_BLOCK- sdd->sddOx;
    xp[2].y= y- RESIZE_BLOCK- sdd->sddOy;

    /*  middle */
    xp[3].x= x0- sdd->sddOx;
    xp[3].y= y- high/2- RESIZE_BLOCK/2 -sdd->sddOy;

    xp[4].x= x0 +wide- RESIZE_BLOCK- sdd->sddOx;
    xp[4].y= y- high/2- RESIZE_BLOCK/2 -sdd->sddOy;

    /*  top  */
    xp[5].x= x0- sdd->sddOx;
    xp[5].y= y- high- sdd->sddOy;

    xp[6].x= x0 +wide/2- RESIZE_BLOCK/2- sdd->sddOx;
    xp[6].y= y- high- sdd->sddOy;

    xp[7].x= x0 +wide- RESIZE_BLOCK- sdd->sddOx;
    xp[7].y= y- high- sdd->sddOy;

    tedDrawSetForeground( add, sdd, SDDpixelBACK );

    for ( i= 0; i < 8; i++ )
	{
	appDrawFillRectangle( add, xp[i].x, xp[i].y, RESIZE_BLOCK, RESIZE_BLOCK );
	}

    tedDrawSetForeground( add, sdd, SDDpixelFORE );

    for ( i= 0; i < 8; i++ )
	{
	appDrawDrawRectangle( add,
			    xp[i].x, xp[i].y, RESIZE_BLOCK- 1, RESIZE_BLOCK- 1 );
	}

    appDrawDrawRectangle( add,
			x0- sdd->sddOx, y- high- sdd->sddOy, wide- 1, high- 1 );

    return;
    }

/************************************************************************/
/*									*/
/*  Draw the box around an active footnote, header or footer.		*/
/*									*/
/************************************************************************/

static void tedDrawExternalItemBox(
			    const BufferItem *		bodySectBi,
			    const ExternalItem *	ei,
			    ScreenDrawingData *		sdd,
			    DrawingContext *		dc,
			    const DocumentRectangle *	drClip )
    {
    AppDrawingData *		add= dc->dcDrawingData;
    
    DocumentRectangle		drIntersect;
    DocumentRectangle		drBox;

    const int			justUsed= 0;

    static char			dot[]= { 1, 2 };

    if  ( ei->eiPageFormattedFor < 0 )
	{ LDEB(ei->eiPageFormattedFor); return;	}

    if  ( docGetExternalItemBox( &drBox, bodySectBi, ei, justUsed,
			    ei->eiPageFormattedFor, dc->dcDocument, add ) )
	{ LDEB(1); return;	}

    if  ( ! docIntersectRectangle( &drIntersect, &drBox, drClip ) )
	{ return;	}

#   ifdef USE_MOTIF
    XSetLineAttributes( add->addDisplay, add->addGc,
				    1, LineOnOffDash, CapButt, JoinMiter );

    XSetDashes( add->addDisplay, add->addGc, 0, dot, sizeof( dot ) );
#   endif

#   ifdef USE_GTK
    gdk_gc_set_line_attributes( add->addGc,
		    1, GDK_LINE_ON_OFF_DASH, GDK_CAP_BUTT, GDK_JOIN_MITER );

    gdk_gc_set_dashes( add->addGc, 0, dot, sizeof( dot ) );
#   endif

    appDrawDrawLine( add,
			    drBox.drX0- sdd->sddOx, drBox.drY0- sdd->sddOy,
			    drBox.drX1- sdd->sddOx, drBox.drY0- sdd->sddOy );
    appDrawDrawLine( add,
			    drBox.drX0- sdd->sddOx, drBox.drY1- sdd->sddOy,
			    drBox.drX1- sdd->sddOx, drBox.drY1- sdd->sddOy );
    appDrawDrawLine( add,
			    drBox.drX0- sdd->sddOx, drBox.drY0- sdd->sddOy,
			    drBox.drX0- sdd->sddOx, drBox.drY1- sdd->sddOy );
    appDrawDrawLine( add,
			    drBox.drX1- sdd->sddOx, drBox.drY0- sdd->sddOy,
			    drBox.drX1- sdd->sddOx, drBox.drY1- sdd->sddOy );

    return;
    }

static int tedDrawCheckPageOfSelectedExtItem(
				    SelectionGeometry *		sg,
				    const DocumentSelection *	ds,
				    BufferDocument *		bd,
				    ExternalItem *		selRootEi,
				    AppDrawingData *		add )
    {
    int			changed= 0;
    DocumentRectangle	drChanged;

    if  ( tedCheckPageOfSelectedExtItem( &changed, &drChanged,
							bd, selRootEi, add ) )
	{ LDEB(1); return -1;	}

    if  ( changed )
	{ tedSelectionGeometry( sg, ds, bd, add );	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Expose handler for documents.					*/
/*									*/
/*  2)  Clear background.						*/
/*  2b) If the selected area overlaps with the exposed region, draw the	*/
/*	selection background.						*/
/*  3)  Draw text.							*/
/*  4)	Draw page separators.						*/
/*  5)  Draw I bar if necessary.					*/
/*									*/
/************************************************************************/

void tedRedrawRectangle(	APP_WIDGET		w,
				TedDocument *		td,
				DocumentRectangle *	drClip,
				AppDrawingData *	add,
				AppColors *		acSys,
				int			ox,
				int			oy )
    {
    BufferDocument *		bd= td->tdDocument;
    BufferItem *		docBi= &(bd->bdItem);

    DrawingContext		dc;
    ScreenDrawingData		sdd;

    BufferItem *		selRootBi= (BufferItem *)0;

    APP_COLOR_RGB		selColor;
    APP_COLOR_RGB		linkFore;

    DocumentSelection		ds;
    SelectionGeometry		sg;

    ExternalItem *		selRootEi= (ExternalItem *)0;
    BufferItem *		selRootBodySectBi= (BufferItem *)0;

    if  ( tedGetSelection( &ds, &sg, td ) )
	{
	docInitDocumentSelection( &ds );
	docInitSelectionGeometry( &sg );
	}

    if  ( td->tdVisibleSelectionCopied )
	{ selColor= td->tdXSelColor;		}
    else{ selColor= td->tdSelColor;		}

    linkFore= td->tdFieldColor;

    docInitDrawingContext( &dc );

    dc.dcDrawingData= add;
    dc.dcDocument= bd;
    dc.dcDrawTableGrid= td->tdDrawTableGrid;
    dc.dcClipRect= drClip;

    sdd.sddOx= ox;
    sdd.sddOy= oy;

    if  ( appDrawBlackColor( add, &(sdd.sddForeColor) ) )
	{ LDEB(1); return;	}
    if  ( appDrawWhiteColor( add, &(sdd.sddBackColor) ) )
	{ LDEB(1); return;	}
    sdd.sddLinkColor= linkFore;
    sdd.sddGridColor= td->tdTableColor;
    sdd.sddPageColor= add->addBackColor;

#   if LOG_REDRAWS
    docLogRectangle( "REDRAW", drClip );
#   endif

    /*  2  */
    tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
    tedDrawSetForeground( add, &sdd, SDDpixelBACK );

    appDrawFillRectangle( add,
			    drClip->drX0- ox, drClip->drY0-oy,
			    drClip->drX1- drClip->drX0+ 1,
			    drClip->drY1- drClip->drY0+ 1 );

    /*  2a  */
    if  ( tedHasSelection( td ) )
	{
	selRootBi= docGetSelectionRoot(
				&selRootEi, &selRootBodySectBi, bd, &ds );
	if  ( ! selRootBi )
	    { XDEB(selRootBi);	}
	}

    if  ( selRootBi && ! tedHasIBarSelection( td ) )
	{
	tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
	sdd.sddBackColor= selColor;

	dc.dcDrawTextLine= tedHighlightBack;
	dc.dcDrawParaTop= (DRAW_PARA_TOP)0;
	dc.dcDrawParaBottom= (DRAW_PARA_BOTTOM)0;
	dc.dcDrawCellTop= (DRAW_CELL_TOP)0;
	dc.dcDrawCellBottom= (DRAW_CELL_BOTTOM)0;
	dc.dcDrawCellLeft= (DRAW_CELL_LEFT)0;
	dc.dcDrawCellRight= (DRAW_CELL_RIGHT)0;
	dc.dcParaFramePixels= tedParagraphFramePixels;
	dc.dcFinishPage= (FINISH_PAGE)0;
	dc.dcStartPage= (START_PAGE)0;

	dc.dcLayoutExternal= (LAYOUT_EXTERNAL)0;
	dc.dcCloseObject= tedCloseObject;
	dc.dcDrawHeadersFooters= 0;

	dc.dcDocumentSelection= &ds;
	dc.dcSelectionGeometry= &sg;

	if  ( selRootBi->biInExternalItem != DOCinBODY			&&
	      tedDrawCheckPageOfSelectedExtItem( &sg, &ds, bd,
						    selRootEi, add )	)
	    { LDEB(1); return; }

	if  ( docDrawItem( selRootBi, (void *)&sdd, &dc ) )
	    { LDEB(1);	}
	}

    /*  3  */
    if  ( appDrawBlackColor( add, &(sdd.sddForeColor) ) )
	{ LDEB(1); return;	}
    if  ( appDrawWhiteColor( add, &(sdd.sddBackColor) ) )
	{ LDEB(1); return;	}

    dc.dcDrawTextLine= tedDrawTextLine;
    dc.dcDrawParaTop= tedDrawParaTop;
    dc.dcDrawParaBottom= tedDrawParaBottom;
    dc.dcDrawCellTop= tedDrawCellTop;
    dc.dcDrawCellBottom= tedDrawCellBottom;
    dc.dcDrawCellLeft= tedDrawCellLeft;
    dc.dcDrawCellRight= tedDrawCellRight;
    dc.dcParaFramePixels= tedParagraphFramePixels;

    dc.dcLayoutExternal= tedLayoutExternalItem;
    dc.dcCloseObject= tedCloseObject;
    dc.dcDrawHeadersFooters= 1;

    dc.dcDocumentSelection= (DocumentSelection *)0;
    dc.dcSelectionGeometry= (SelectionGeometry *)0;

    appDrawSetBackgroundWhite( add );

    tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
    tedDrawSetForeground( add, &sdd, SDDpixelFORE );
    dc.dcCurrentPhysicalFont= -1;

    if  ( docDrawItem( docBi, (void *)&sdd, &dc ) )
	{ LDEB(1);	}

    {
    int		firstPage;
    int		lastPage;

    int		page;
    int		sectNr;

    int		y0= drClip->drY0;
    int		y1= drClip->drY1;

    y0= drClip->drY0- add->addPageGapPixels;
    if  ( y0 < 0 )
	{ y0= 0;	}

    y1= drClip->drY1+ add->addPageStepPixels- 1;
    if  ( y1 > BI_BELOW_PIXELS( add, docBi ) )
	{ y1=  BI_BELOW_PIXELS( add, docBi );	}

    firstPage= y0/ add->addPageStepPixels;
    lastPage=  y1/ add->addPageStepPixels;

    if  ( lastPage > docBi->biBelowPosition.lpPage )
	{ lastPage=  docBi->biBelowPosition.lpPage;	}

    for ( page= firstPage; page <= lastPage; page++ )
	{ tedDrawPageGap( &sdd, &dc, page );	}

    for ( sectNr= 0; sectNr < docBi->biChildCount; sectNr++ )
	{
	BufferItem *	bodySectBi= docBi->biChildren[sectNr];
	BufferItem *	prevSectBi= (BufferItem *)0;
	int		first= bodySectBi->biTopPosition.lpPage;
	int		last= bodySectBi->biBelowPosition.lpPage;

	if  ( first < firstPage )
	    { first=  firstPage;	}
	if  ( last >  lastPage )
	    { last=   lastPage;		}

	if  ( sectNr > 0 )
	    { prevSectBi= docBi->biChildren[sectNr- 1];	}
	/*
	Word draws the footer of the first section on the page.
	if  ( sectNr < docBi->biChildCount- 1 )
	    { nextSectBi= docBi->biChildren[sectNr+ 1];	}
	*/

	for ( page= first; page <= last; page++ )
	    {
	    if  ( ! prevSectBi					||
		  prevSectBi->biBelowPosition.lpPage < page	)
		{ docDrawPageHeader( bodySectBi, (void *)&sdd, &dc, page ); }

	    docDrawFootnotesForColumn( page, (void *)&sdd, &dc );

	    /* Word draws the footer of the first section on the page. */
	    if  ( ! prevSectBi					||
		  prevSectBi->biBelowPosition.lpPage < page	)
		{ docDrawPageFooter( bodySectBi, (void *)&sdd, &dc, page ); }
	    }
	}
    }

    if  ( selRootBi					&&
	  ! tedHasIBarSelection( td )			&&
	  acSys->acAllocator.caDepth < 4		)
	{
	if  ( appDrawWhiteColor( add, &(sdd.sddForeColor) ) )
	    { LDEB(1); return;	}
	if  ( appDrawWhiteColor( add, &(sdd.sddLinkColor) ) )
	    { LDEB(1); return;	}

	tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
	tedDrawSetForeground( add, &sdd, SDDpixelFORE );

	dc.dcDrawTextLine= tedDrawTextReverse;
	dc.dcDrawParaTop= (DRAW_PARA_TOP)0;
	dc.dcDrawParaBottom= (DRAW_PARA_BOTTOM)0;
	dc.dcDrawCellTop= (DRAW_CELL_TOP)0;
	dc.dcDrawCellBottom= (DRAW_CELL_BOTTOM)0;
	dc.dcDrawCellLeft= (DRAW_CELL_LEFT)0;
	dc.dcDrawCellRight= (DRAW_CELL_RIGHT)0;
	dc.dcParaFramePixels= tedParagraphFramePixels;
	dc.dcFinishPage= (FINISH_PAGE)0;
	dc.dcStartPage= (START_PAGE)0;

	dc.dcLayoutExternal= (LAYOUT_EXTERNAL)0;
	dc.dcCloseObject= tedCloseObject;
	dc.dcDrawHeadersFooters= 0;

	dc.dcDocumentSelection= &ds;
	dc.dcSelectionGeometry= &sg;

	if  ( selRootBi->biInExternalItem != DOCinBODY			&&
	      tedDrawCheckPageOfSelectedExtItem( &sg, &ds, bd,
						    selRootEi, add )	)
	    { LDEB(1); return; }

	if  ( docDrawItem( selRootBi, (void *)&sdd, &dc ) )
	    { LDEB(1);	}
	}

    if  ( selRootBi					&&
	  selRootBi->biInExternalItem != DOCinBODY	)
	{
	BufferItem *	x_selRootBi;
	ExternalItem *	x_selRootEi;
	BufferItem *	bodySectBi;

	tedDrawSetForeground( add, &sdd, SDDpixelFORE );

	if  ( docGetRootOfSelectionScope( &x_selRootEi, &x_selRootBi,
				&bodySectBi, bd, &(ds.dsSelectionScope) ) )
	    { LDEB(1);	}
	else{
	    tedDrawExternalItemBox( bodySectBi, x_selRootEi,
							&sdd, &dc, drClip );
	    }
	}

    tedDrawSetForeground( add, &sdd, SDDpixelFORE );

    /*  5  */
    if  ( tedHasIBarSelection( td ) )
	{
	const PositionGeometry *	pgBegin= &(sg.sgBegin);

	if  ( selRootBi->biInExternalItem != DOCinBODY			&&
	      tedDrawCheckPageOfSelectedExtItem( &sg, &ds, bd,
						    selRootEi, add )	)
	    { LDEB(1); return; }

	if  ( drClip->drY0 <= pgBegin->pgY1Pixels		&&
	      drClip->drY1 >= PG_TOP_PIXELS( add, pgBegin )	&&
	      ! td->tdShowIBarId				)
	    { tedDrawIBar( pgBegin, ox, oy, add ); }
	}
    else{
	InsertedObject *	io;
	DocumentPosition	dpO;

	docInitDocumentPosition( &dpO );

	if  ( td->tdObjectSelected				&&
	      ! tedGetObjectSelection( td, &dpO, &io  )	)
	    {
	    PositionGeometry	pgO;

	    if  ( selRootBi->biInExternalItem != DOCinBODY		&&
		  tedDrawCheckPageOfSelectedExtItem( &sg, &ds, bd,
						    selRootEi, add )	)
		{ LDEB(1); return; }

	    tedDrawSetForeground( add, &sdd, SDDpixelUNDEF );
	    if  ( appDrawBlackColor( add, &(sdd.sddForeColor) ) )
		{ LDEB(1); return;	}
	    sdd.sddBackColor= selColor;

	    tedPositionGeometry( &pgO, &dpO, add );

	    if  ( drClip->drY0 <= pgO.pgY1Pixels		&&
		  drClip->drY1 >= PG_TOP_PIXELS( add, &pgO )	)
		{
		if  ( io->ioDragWide > 0 )
		    {
		    tedDrawObjectBlocks( pgO.pgXPixels,
			pgO.pgBaselinePixels+ io->ioDragHigh- io->ioPixelsHigh,
			io->ioDragWide, io->ioDragHigh, &dc, &sdd );
		    }
		else{
		    tedDrawObjectBlocks( pgO.pgXPixels,
					 pgO.pgBaselinePixels,
			io->ioPixelsWide, io->ioPixelsHigh, &dc, &sdd );
		    }
		}
	    }
	}

    return;
    }

