/*                                      -*- c-file-style: "bsd" -*-
 * rproxy -- dynamic caching and delta update in HTTP
 * $Id: rproxy.c,v 1.24 2000/08/16 10:08:58 mbp Exp $
 *
 * Copyright (C) 1999, 2000 by Martin Pool <mbp@linuxcare.com>
 * Copyright (C) 1999 by tridge@samba.org
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */



                              /*
                               | In order to succeed in any
                               | enterprise, one must be persistent
                               | and patient. Even if one has to run
                               | some risks, one must be brave and
                               | strong enough to meet and overcome
                               | vexing challenges to maintain a
                               | successful business in the long
                               | run. -- Hajime Karatsu
                               */


/*
 * TODO: Add an option to change the userid: chroot is kind of
 * pointless without this.  But does this functionality really belong
 * in this program, or would it be better to just rely on people
 * putting that in the startup script?
 *
 * TODO: Finish cleanup.  Security audit should be done before we put this
 * anywhere public.
 *
 * TODO: Progressively truncate the URL while looking for cache files, rather
 * than just an all-or-nothing match.
 *
 * TODO: Let people specify file patterns to exclude (e.g. jpegs), or perhaps
 * mime-types.
 *
 * TODO: If we're an HTTP accelerator, then we know we'll never get an
 * rsync-encoded response from upstream.  Therefore we may as well not cache
 * the responses.
 *
 * TODO: Is not caching useful?  How about restricting whether we'll encode
 * or decode?
 *
 * FIXME: I think the -r option will cause trouble if anyone upstream
 * understands hsync.  They'll see that we accept the encoding, but they'll
 * try to do a transfer from scratch, which will make things a bit wierd.
 *
 * TODO: Show the right PID when forking
 *
 * TODO: Show pids in the log
 *
 * TODO: Maybe change argv when serving requests?
 *
 * TODO: Show message when exiting.
 *
 * TODO: Make sure we close all file descriptors when we fork.
 *
 * TODO: Respond to SIGHUP by re-executing ourselves with the same
 * arguments.  Not very useful in general, however, because people can only
 * reconfigure it by specifying different arguments.
 */

#include "config.h"
#include "sysheaders.h"

#include <sys/socket.h>

#include "rproxy.h"
#include "cache.h"
#include "util.h"
#include "urllib.h"
#include "header.h"
#include "socklib.h"
#include "msgpage.h"
#include "trace.h"
#include "rsglue.h"
#include "xferlog.h"
#include "myname.h"
#include "daemon.h"
#include "logfile.h"
#include "error.h"

/* Configuration for the whole server. */
struct config   config;

static void
find_my_hostname(void)
{
    size_t          len = 100;

    config.my_hostname = xmalloc(len);
    if (gethostname(config.my_hostname, len - 1) == -1) {
        rp_log(LOGAREA_NET, LOG_ERR, "gethostname failed: %s\n", strerror(errno));
        exit(1);
    }
}



/* chroot into a dir, or die. */
static void
try_chroot(char const *dir)
{
    int             rc;

    rc = chroot(dir);
    if (rc) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "can't chroot to %s: %s", dir, strerror(errno));
        if (errno == EPERM) {
            rp_log(LOGAREA_PROCESS, LOG_ERR, "(perhaps you must be root to call chroot?)");
        }
        exit(1);
    }

    rc = chdir("/");
    if (rc) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "can't change into new root directory %s: %s\n",
               dir, strerror(errno));
        exit(1);
    }
}





static void
usage(void)
{
    fputs("Usage: rproxy OPTIONS\n"
          "rproxy adds differential encoding to HTTP to speed the web.\n"
          "This release of rproxy is ALPHA: use it at your own risk.\n"
          "For more information: http://linuxcare.com.au/rproxy/\n"
          "\n"
          "Options:\n"
          "  -1             serve a single request on stdin (inetd)\n"
          "  -C DIR         chroot to DIR\n"
          "  -L FACILITY    trace/error log to syslog facility\n"
          "  -S SOCKOPT     setsockopt on listen socket\n"
          "  -V             show version & license\n"
          "  -X             don't fork or leave the foreground\n"
          "  -c DIR         set cache directory\n"
          "  -h             help\n"
          "  -l LOGFILE     transfer log\n"
          "  -p PORT        port to listen on\n"
          "  -s STATSLOG    statistics log file\n"
          "  -r             remove Rsync-Signature from requests (dodgy)\n"
          "  -u HOST[:PORT] upstream server\n"
          ,
          stdout);

    /* TODO: Configuration parameter to allow CONNECT, saying which
     * ports it can go to.  Suggested is 443 563.*/
    write_trace_usage(stdout);
}


static void
show_version(void)
{
    printf("%s %s (%s) (%s)\n"
           "http://linuxcare.com.au/rproxy/\n"
           "Copyright (C) 1999-2000 by Martin Pool, and others.\n\n"
           "rproxy comes with NO WARRANTY, to the extent permitted by law.\n"
           "You may redistribute copies of rproxy under the terms of the\n"
           "GNU General Public License.  For more information about \n"
           "these matters, see the files named COPYING.\n",
           PACKAGE, VERSION, hs_libhsync_version, HOST_TYPE);
}

static void
rp_bad_usage(char opt)
{
    rp_log(LOGAREA_PROCESS, LOG_ERR, "bad command-line option -%c", opt);
    exit(EXIT_STARTUP_ERR);
}


int
main(int argc, char *argv[])
{
    int             opt;

    bzero(&config, sizeof config);
    config.upstream_port = 80;
    config.cache_root = "/var/cache/rproxy";
    config.stats_log_fd = -1;

    find_my_hostname();

    /* XXX: Emitting any error messages here is dodgy, because if
     * we're running from inetd they'll get spattered into the output
     * stream and ruin the file header.  So for the meantime you
     * should put -L first!  */

    opterr = 0;
    while ((opt = getopt(argc, argv, "1C:D:L:XS:Vc:l:rhp:s:u:")) != -1) {
        switch (opt) {
        case 'C':
            try_chroot(optarg);
            break;
        case 'D':
            set_trace_flags(optarg);
            break;
        case 'L':
            config.syslog_facility = xstrdup(optarg);
            break;
        case 'X':
            config.dont_fork = 1;
            break;
        case 'N':
            config.no_cache = 1;
            break;
        case 'S':
            if (config.out_sockopts) {
                rp_log(LOGAREA_PROCESS, LOG_ERR, "-S may only be used once sorry\n");
                exit(1);
            }
            config.out_sockopts = xstrdup(optarg);
            break;
        case 'V':
            show_version();
            return 0;
        case 'c':
            config.cache_root = xstrdup(optarg);
            break;
        case 'l':
            rp_open_transfer_log(xstrdup(optarg));
            break;
        case 'r':
            config.remove_sig = 1;
            break;
        case 's':
            config.stats_log_fd = rp_open_logfile(optarg);
            break;
        case 'u':
            _rp_split_host_port(optarg,
                               &config.upstream_host, &config.upstream_port);
            break;
        case 'p':
            config.listen_port = atoi(optarg);
            break;
        case '1':
            config.serve_one = 1;
            break;
        case 'h':
            usage();
            exit(1);
        default:
            rp_bad_usage(optopt);
        }
    }

    rp_my_name(argc, argv);

    rp_setup_syslog();
    hs_trace_to(hsync_glue_log);
    hs_trace_set_level(LOG_DEBUG);

    if (!config.serve_one == !config.listen_port) {
        rp_log(LOGAREA_PROCESS, LOG_ERR, "please specify either inetd (-1) or "
               "daemon (-p PORT) mode, or say -h for help");
        exit(1);
    }

    /* TODO: setuid.

       TODO: Close unnecessary fds.

       TODO: chdir(/) even if not chrooted.  According to Stevens this is
       good form. */

    /* OK, now we're essentially started up; switch to the logfile. */


    rp_log(LOGAREA_PROCESS, LOG_CRIT,
          "%s %s (%s) (%s)",
          PACKAGE, VERSION, hs_libhsync_version, HOST_TYPE);

    if (config.upstream_host) {
        rp_log(LOGAREA_PROCESS, LOG_INFO,
               "upstream server %s:%d",
               config.upstream_host, config.upstream_port);
    } else {
        rp_log(LOGAREA_PROCESS, LOG_INFO, "no upstream server");
    }

    if (!(config.dont_fork || config.serve_one))
        rp_become_daemon();
    else
        trace(LOGAREA_PROCESS,
              "not forking, serving direct from process %d", getpid());

    rp_hook_death();

    if (!config.no_cache)
        cache_init(config.cache_root);

    if (!config.serve_one) {
        start_accept_loop(config.listen_port, rp_serve,
                          config.dont_fork, config.out_sockopts);
    } else {
        serve_inetd(rp_serve);
    }
    return 0;
}
