#include "allegro.h"
#include "defs.h"
#include "hull.h"
#include "trigtable.h"
#include "spaceobj.h"
#include "gunobj.h"
#include "ebox.h"
#include "section.h"

#include <stdio.h>

/* Constructor:
 * parameters are:
 * _drawNum - the drawing number of the object. Used for determing which bitmap
 * out of the datafile to draw from.
 * _life - the amount of life of the object
 * _strength - the amount of damage the object will do when it collides with something
 * num_guns - the maximum number of guns allowed in this object
 * level - the level number of the game
 * _drawing_plane - the drawing level of the object
 * ec - an ECollide object use for collision detection.
 */
HullObject::HullObject( BITMAP * _drawBitmap, double _life, double _strength, int num_guns, int level, int _drawing_plane, bool _trans, ECollide * ec):
crash( false ),
trans( _trans ),
strength( _strength ),
drawLevel( _drawing_plane ),
guns( NULL ),
max_guns( num_guns ) {
	Bitmap = _drawBitmap;
	Shadow_Bitmap = NULL;
	if ( Bitmap != NULL )
		Shadow_Bitmap = getShadow();
	life = _life;
	original_life = _life;
	collide = ec;
	//life = (int)( (double)l * ( 1.0 + (double)level / 3.8 ) );
	guns = NULL;
	if ( max_guns != 0 ) {
		guns = new WeaponObject*[ max_guns ];
		for ( int q = 0; q < max_guns; q++ )
			guns[q] = NULL;
	}
	dx = 0;
	dy = 0;
}


/* setLife:
 * Sets the life of the object according to a parameter, zx, which
 * usually refers to the game level give in the constructor.
 */
void HullObject::setLife( double zx ) {
	//life = (int)( (double)life * ( 1.0 + (double)zx / 3.1 ) );
	life = (double)life * ( 1.0 + (double)zx / 3.1 );
	original_life = life;
}


/* Moved:
 * Called the container object, spaceobject, has its method MoveMe called.
 * Default is to do nothing.
 */
void HullObject::Moved( double _dx, double _dy, double _ax, double _ay ) {

	dx = _dx;
	dy = _dy;

}


/* Hurt:
 * Returns the amount of damage the object will do when it collides with something.
 */
double HullObject::Hurt() {
	return strength;
}
	
int HullObject::getLife(){
	return (int)life;
}

/* Draw:
 * Draws the object onto the specified bitmap. For most objects, this will
 * consist of drawing the bitmap at data[ drawNum ] at the
 * X coordinate - half the width of the bitamp and the
 * Y coordinate - half the height of the bitmap. If the object does not have a bitmap
 * to draw from, it needs to override this method with its own drawing routine.
 */
void HullObject::Draw( BITMAP * who, int x, int y ) {
	//int mx = ((BITMAP *)dat[ drawNum ].dat)->w/2;
	//int my = ((BITMAP *)dat[ drawNum ].dat)->h/2;
	//draw_character(who, (BITMAP *)dat[drawNum].dat, x-mx+mx*3/2, y-my+my*3/2, makecol(30,30,30) );
	//draw_trans_sprite( who, (BITMAP *)dat[ drawNum ].dat, x-mx+mx*5/4, y-my+my*3/2 );
	//draw_sprite( who, (BITMAP *)dat[ drawNum ].dat, x-mx, y-my );
	if ( Bitmap == NULL ) return;
	int mx = Bitmap->w/2;
	int my = Bitmap->h/2;
	if ( trans )
		draw_trans_sprite( who, Bitmap, x-mx, y-my );
	else
		draw_sprite( who, Bitmap, x-mx, y-my );
	//if( collide ) collide->display( who, x, y );

}

BITMAP * HullObject::getShadow(){
	
	if ( Shadow_Bitmap == NULL ){

		Shadow_Bitmap = create_bitmap( Bitmap->w, Bitmap->h );
		//clear_to_color( Shadow_Bitmap, makecol(255,0,255) );
		//draw_lit_sprite( Shadow_Bitmap, Bitmap, 0, 0, 255 );
		int m = 35;
		clear_to_color( Shadow_Bitmap, makecol(m,m,m) );
		//draw_character( Shadow_Bitmap, Bitmap, 0, 0, 1 );
		draw_mask( Shadow_Bitmap, Bitmap, 0, 0 );
		
	}

	return Shadow_Bitmap;
}

/* Shadow:
 * Draws a shadow of the picture almost the same as the Draw() method except
 * using draw_lit_sprite instead.
 */
void HullObject::Shadow( BITMAP * who, int x, int y ) {
	if ( Bitmap == NULL ) return;

	/*
		int mx = Bitmap->w/2;
		int my = Bitmap->h/2;
		int tx = (x - mx) - ( screen_x/2 - (x-mx) ) / 9;
		int ty = ( screen_y/2 + (y-my) ) / 15 + y-my;
		ty = y + my;
		draw_lit_sprite( who, Bitmap, tx, ty, 255 );
	*/

	Shadow_Bitmap = getShadow();
		
	int mx = Bitmap->w/2;
	int my = Bitmap->h/2;
	int tx = (x - mx) - ( screen_x/2 - (x-mx) ) / 9;
	int ty = y + my;

	draw_trans_sprite( who, Shadow_Bitmap, tx, ty );
	//draw_lit_sprite( who, Bitmap, (int)(x-mx+dx*9), (int)(y-my+dy*9), 250 );
	//fblend_add( (BITMAP *)dat[ drawNum ].dat, who, (int)(x-mx+dx*11), (int)(y-my+dy*11), 20 );
}


/* Inside:
 * Returns true if the specified coordinate (ax,ay) is within the
 * space of the hull centered at (mx,my)
 * See ebox.cpp for further details on the Single() method.
 */
bool HullObject::Inside( int mx, int my, int ax, int ay ) {
	if ( collide ) return collide->Single( mx, my, ax, ay );
	return ( mx == ax && my == ay );
}


/* giveWeapon:
 * Gives the hull a gunobject at the specified bay.
 * WARNING: does not delete the previous object. To do this call
 * eraseweapon.
 */
void HullObject::giveWeapon( WeaponObject * weap, int num ) {
	guns[ num ] = weap;
}


/* eraseWeapon:
 * Deletes the object at the specified bay.
 * Sets the bay to NULL.
 */
void HullObject::eraseWeapon( int num ) {
	if ( num < 0 || num >= max_guns ) return;
	if ( guns[num] != NULL ) delete guns[num];
	guns[num] = NULL;
}


/* Collide:
 * Can only detect a collision if this object and the other object have a
 * collide object in it. Returns true if ECollide->Collision() returns true.
 * See ebox.cpp for further details.
 */
bool HullObject::Collide( int mx, int my, SpaceObject * check ) {

	if ( collide && check->getHull()->collide )
		return collide->Collision( check->getHull()->collide, mx, my, check->getX(), check->getY() );
	//return check->HitMe( mx, my );
	return false;
}


/* debug:
 * Debugging crap for knowing where the object is on the screen.
 */
void HullObject::debug( BITMAP * work, int x, int y ) {
	circlefill( work, x, y, 4, makecol(255,255,255) );
}


/* guns:
 * Returns the array of weaponobjects the object contains.
 */
WeaponObject ** HullObject::Guns() {
	return guns;
}


/* maxGuns:
 * Returns the maximum number of guns the object can possibly contain.
 */
int HullObject::maxGuns() {
	return max_guns;
}


/* haveBeenHit:
 * Returns true if the hull has collided with something since the last call to
 * haveBeenHit. The flag is initially false, so the first call will return true
 * the first time the hull collides with something.
 */
bool HullObject::haveBeenHit() {
	bool cy = crash;
	crash = false;
	return cy;
}


/* addSection:
 * Puts the object into the Section object which is used for speeding up
 * collision detection. This is calculated by determining the upper most corner
 * and the lower most corner. Given this height, each y coordinate in a specific range
 * given by Section->spacer() will be used to add the object into the Section's database
 * of objects it should know about. The object will not add itself if it is translucent.
 * See section.cpp for further details about this method.
 */
void HullObject::addSection( SpaceObject * who, Section * onscreen, int x, int y ) {

	if ( Translucent() ) return;
	if ( onscreen == NULL ) return;

	int y1 = y;
	int y2 = y;
	if ( collide ) {
		y1 = collide->CY1( collide->ul_y + y );
		y2 = collide->CY1( collide->lr_y + y );
	}

	/*
	onscreen->add( who, x, y1 );
	for ( int q = y1+onscreen->spacer(); q < y2; q += onscreen->spacer() )
		onscreen->add( who, x, q );
	if ( y2 != y1 )
		onscreen->add( who, x, y2 );
	*/
	for ( int ay = y1; ay <= y2; ay += onscreen->spacer() )
		onscreen->add( who, x, ay );

}


/* takeDamage:
 * Enforces that the hull take so much damage. This method SHOULD NOT be overwritten.
 */
void HullObject::takeDamage( double r ) {
	life -= r;
}


/* Damage:
 * Allows the hull to operate on how much damage it is going to take.
 * Default is to just take the damage specified.
 */
double HullObject::Damage( double much ) {
	return much;
}


/* GetLife:
 * Returns how much life this object has left.
 */
double HullObject::GetLife() {
	return life;
}


/* Translucent:
 * Returns true if the object is translucent. Translucent means that it cannot
 * collide with any other object.
 */
bool HullObject::Translucent() {
	return trans;
}


/* Collided:
 * Does something when hull collides with another hull
 */
void HullObject::Collided() {
}


HullObject * HullObject::copy() {

	HullObject * who = new HullObject( Bitmap, life, strength, this->maxGuns(), 1, drawLevel, trans, collide );
	for ( int q = 0; q < this->maxGuns(); q++ )
		if ( this->Guns()[q] != NULL )
			who->guns[q] = this->Guns()[q]->copy();
	else    who->guns[q] = NULL;
	return who;
}


/* Destructor:
 * Deletes all the guns the object contains.
 * WARNING: does not delete the ECollide object, this must be done elsewhere.
 */
HullObject::~HullObject() {

	if ( guns != NULL ) {
		for ( int q = 0; q < max_guns; q++ )
			if ( guns[q] != NULL )
				delete guns[q];
		delete[] guns;
	}
	if ( Shadow_Bitmap ) destroy_bitmap( Shadow_Bitmap );

}
