/***************************************************************************
 *            field.c
 *
 *  Fri Aug 25 14:55:56 2006
 *  Copyright  2006-2007  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/
/** @file field.c
	@brief  Add and edit fields (columns) in the GtkSheet.
	@author Copyright 2006-2007  Neil Williams <linux@codehelp.co.uk>
	@author Copyright 1999  Robert Lissner, Jay MacDonald
	@author Copyright 1999  Sam Phillips, Keith Wesolowski.
*/
/*
    This package is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <gtkextra/gtksheet.h>
#include <monetary.h>
#include "types.h"
#include "dialog_initial.h"
#include "dim_list_menu.h"
#include "filein.h"
#include "main.h"

/* this is all done in a dialog box, so globals are OK */
static QlFieldInfo * field_db_start;
static QlFieldInfo * field_db_curr;
static GtkWidget *done_button;
/** so we can ignore signals during display */
static gboolean displaying;
static GtkWidget * text_button;
static GtkWidget * numeric_button;
static GtkWidget * date_button;
static GtkWidget * time_button;
static GtkWidget * left_button;
static GtkWidget * center_button;
static GtkWidget * right_button;
static GtkWidget * column_name_entry;
static GtkWidget * decimal_places_entry;
static GtkWidget * entry_dialogue;

/** \todo make locale sensitive */

/** \brief new locale sensitive number formatting

strfmon :
%^ - omit grouping £1234567.89, default is £1,234,567.89
%( - bracket negative values             (£12,345.99)
%+ - use the locale equivalent of +/-    -£12,345.99
%! - omit the currency symbol.             12,345.99
%- - left formatted instead of right formatted.
also have string width support and # for
number of digits to the LEFT of the decimal.
Use . for number of decimal places.
Also i for international currencies, n for national. TEST!

\bug take care how these are read from existing files!

Include new specifiers for:
%(		(£12,345.99)
%^(		(£12345.99)
%(!		12,345.99
%i      12,345.99GBP
%i(		(12,345.99)GBP
%^i(	(12345.99)GBP
%^i		12,345.99GBP
*/
static G_GNUC_UNUSED gchar * num_formats[] = {
	"1,234.56",		/*  %! formatting style for formatting == 0 */
	"1.234,56",     /*  omit - use for files only */
	"1234.56",      /*  %^! */
	"1234,56",      /* omit - use for files only */
	"$1,234.56",    /*  default (%n) */
	"1,234.56%",    /* omit - use for files only */
	"1.234,56%",    /* omit - use for files only */
	NULL
};

static void
cancel_clicked (GtkObject G_GNUC_UNUSED * object, gpointer G_GNUC_UNUSED entry)
{
	gtk_widget_destroy (entry_dialogue);
}

static void
done_clicked (GtkObject G_GNUC_UNUSED * object, gpointer data)
{
	QlTabData * tab = (QlTabData*)data;
	g_return_if_fail (tab);
	if (tab->file->last_field < 0)
		return;
	gtk_widget_destroy (entry_dialogue);
	build_basic_list_mode (tab);
	tab->view->display_mode = DISPLAY_LIST;
	add1row (tab);
	dim_all_menus (tab->qlc);
	gtk_widget_show_all (tab->qlc->parent);
	connect_signals (tab);
}

static void
just_clicked (GtkObject G_GNUC_UNUSED * object, gpointer entry)
{
	field_db_curr->justification = GPOINTER_TO_INT (entry);
}

static void
sm_formatting_changed (GtkObject G_GNUC_UNUSED * object,
					   gpointer G_GNUC_UNUSED entry)
{
	field_db_curr->formatting =
		gtk_combo_box_get_active (GTK_COMBO_BOX (object));
}

/** This function displays the field dialogue box. */
static void
update_field_db (QlTabData * tab)
{
	gchar linebuf[20];
	const gchar *text;

	displaying = TRUE;

	text = gtk_entry_get_text (GTK_ENTRY (column_name_entry));
	if (text && strlen (text) > 0)
		field_db_curr->name = g_strdup(text);
	gtk_entry_set_text (GTK_ENTRY (column_name_entry), field_db_curr->name);
	/* otherwise assume want to enter name */
	if (tab->view->dialog_mode != MODE_EDIT)
		gtk_editable_select_region (GTK_EDITABLE (column_name_entry), 0,
			strlen (field_db_curr->name));
	gtk_widget_hide (GTK_WIDGET (tab->view->sm_numeric_box));
	gtk_widget_hide (tab->view->dec_places_label);
	gtk_widget_hide (decimal_places_entry);
	gtk_widget_hide (GTK_WIDGET (tab->view->sm_date_box));
	gtk_widget_hide (GTK_WIDGET (tab->view->sm_time_box));

	switch (field_db_curr->type)
	{
		case FIELD_TYPE_TEXT:
		{
			gtk_toggle_button_set_active
				(GTK_TOGGLE_BUTTON (text_button), TRUE);
			break;
		}
		case FIELD_TYPE_NUMERIC:
		{
			gtk_toggle_button_set_active
			(GTK_TOGGLE_BUTTON (numeric_button), TRUE);
			gtk_combo_box_set_active (tab->view->sm_numeric_box,
				field_db_curr->formatting);
			if (field_db_curr->decimal_places >= 0 &&
				field_db_curr->decimal_places < 10)
				sprintf (linebuf, "%u", field_db_curr->decimal_places);
			else
				strcpy (linebuf, "0");

			gtk_entry_set_text (GTK_ENTRY (decimal_places_entry), linebuf);
			gtk_widget_show (GTK_WIDGET (tab->view->sm_numeric_box));
			gtk_widget_show (tab->view->dec_places_label);
			gtk_widget_show (decimal_places_entry);
			break;
		}
		case FIELD_TYPE_DATE:
		{
			gtk_toggle_button_set_active
				(GTK_TOGGLE_BUTTON (date_button), TRUE);
			gtk_combo_box_set_active (tab->view->sm_date_box,
				field_db_curr->formatting);
			gtk_widget_show (GTK_WIDGET (tab->view->sm_date_box));
			break;
		}
		case FIELD_TYPE_TIME:
		{
			gtk_toggle_button_set_active
				(GTK_TOGGLE_BUTTON (time_button), TRUE);
			gtk_combo_box_set_active (tab->view->sm_time_box,
				field_db_curr->formatting);
			gtk_widget_show (GTK_WIDGET (tab->view->sm_time_box));
			break;
		}
	}
	switch (field_db_curr->justification)
	{
		case GTK_JUSTIFY_LEFT:
		{
			gtk_toggle_button_set_active
				(GTK_TOGGLE_BUTTON (left_button), TRUE);
			break;
		}
		case GTK_JUSTIFY_CENTER:
		{
			gtk_toggle_button_set_active
				(GTK_TOGGLE_BUTTON (center_button), TRUE);
			break;
		}
		case GTK_JUSTIFY_RIGHT:
		{
			gtk_toggle_button_set_active
				(GTK_TOGGLE_BUTTON (right_button), TRUE);
			break;
		}
	}
	if (tab->view->dialog_mode == MODE_NEW && tab->file->last_field >= 0)
		gtk_widget_show (done_button);

	gtk_widget_grab_focus (GTK_WIDGET (column_name_entry));
	displaying = FALSE;
}

static void
type_clicked (GtkObject G_GNUC_UNUSED * object, gpointer data)
{
	QlTabData * tab;
	/* update_db causes another type_clicked */
	if (displaying)
		return;

	tab = (QlTabData*) g_object_get_data (G_OBJECT(entry_dialogue), QLTABID);
	g_return_if_fail (tab);
	field_db_curr->type = GPOINTER_TO_INT (data);
	field_db_curr->formatting = 0;
	if (field_db_curr->type == FIELD_TYPE_TEXT)
		field_db_curr->justification = GTK_JUSTIFY_LEFT;
	else
		field_db_curr->justification = GTK_JUSTIFY_RIGHT;
	update_field_db (tab);
}

static gchar *
get_datestamp (const gchar * format)
{
	GTimeVal tv;
	GDate * gd;
	gchar * d;

	if (!format)
		return NULL;
	gd = g_date_new ();
	g_get_current_time (&tv);
	g_date_set_time_val (gd, &tv);
	d = display_date (gd, format);
	g_date_free (gd);
	return d;
}

static gchar *
get_timestamp (const gchar * format)
{
	GTimeVal tv;
	gchar * d;

	if (!format)
		return NULL;
	g_get_current_time (&tv);
	d = display_time (&tv, format);
	return d;
}

/** \brief used for currency

 \param number The amount of currency.
 \param qc The QuickList currency format to use.
*/
static gchar *
display_currency (gdouble number, QlCurrFormat qc)
{
	gchar buf[MAX_DATE_BUFFER];
	gsize len;
	gchar * format;

	format = NULL;
	switch (qc)
	{
		/* "1,234.56" %! formatting style for formatting == 0 */
		case QL_NF_GRP :   { format = g_strdup("%!i"); break;  }
		/* "1234.56"  %^! */
		case QL_NF :       { format = g_strdup("%^!i"); break; }
		/* "$1,234.56"  default (%n) */
		case QL_NF_CURR :  { format = g_strdup("%n"); break;  }
		/* "1,234.56%" percentage numbers */
		case QL_NF_PERCENT : { format = g_strdup ("%!i%%"); break; }
		/* "1.234,56" used for files only */
		/* "1234,56" used for files only */
		/* "1.234,56%" used for files only */
		default : break;
	}
	if (!format)
		return NULL;
	buf[0] = '\1';
	len = strfmon (buf, MAX_DATE_BUFFER, format, number);
	g_free (format);
	if (len <= 0 && buf[0] != '\0')
		return NULL;
	return g_strdup (buf);
}

/** \brief Assemble the style menus for numeric, date and time */
static GtkWidget *
sub_menus (QlTabData * tab)
{
	GtkWidget *sub_menu_box;
	gint subx;

	/* add the individual display types to the menu.  We'll show them
	 later.  They only show one at a time. */
	sub_menu_box = gtk_hbox_new (FALSE, 5);
	gtk_widget_show (sub_menu_box);

	/* first the numeric menu */
	subx = 0;
	tab->view->sm_numeric_box = GTK_COMBO_BOX (gtk_combo_box_new_text ());
	for (subx = 0; subx < QL_NF_LAST; subx++)
	{
		gchar * nstr = display_currency(1234.56, subx);
		if (!nstr)
			continue;
		gtk_combo_box_insert_text (tab->view->sm_numeric_box, subx, nstr);
	}
	g_signal_connect (GTK_OBJECT (tab->view->sm_numeric_box), "changed",
		G_CALLBACK (sm_formatting_changed), &subx);
	gtk_box_pack_start (GTK_BOX (sub_menu_box),
		GTK_WIDGET (tab->view->sm_numeric_box), FALSE, FALSE, 0);

	decimal_places_entry = gtk_spin_button_new_with_range (0,9,1);
	gtk_spin_button_set_snap_to_ticks (GTK_SPIN_BUTTON(decimal_places_entry), TRUE);
	gtk_box_pack_end (GTK_BOX (sub_menu_box), decimal_places_entry, FALSE, FALSE,
		0);
	tab->view->dec_places_label = gtk_label_new (_("Decimal places"));
	gtk_box_pack_end (GTK_BOX (sub_menu_box), tab->view->dec_places_label,
		FALSE, FALSE, 0);

	/* now the date format menu */
	tab->view->sm_date_box = GTK_COMBO_BOX (gtk_combo_box_new_text ());
	subx = 0;
	for (subx = 0; subx < QL_DF_LAST; subx++)
	{
		gtk_combo_box_insert_text (tab->view->sm_date_box, subx,
			get_datestamp(convert_old_df(subx)));
	}
	g_signal_connect (GTK_OBJECT (tab->view->sm_date_box), "changed",
		G_CALLBACK (sm_formatting_changed), NULL);
	gtk_box_pack_start (GTK_BOX (sub_menu_box),
		GTK_WIDGET (tab->view->sm_date_box), FALSE, FALSE, 0);

	/* now the time format menu */
	tab->view->sm_time_box = GTK_COMBO_BOX (gtk_combo_box_new_text ());
	subx = 0;
	for (subx = 0; subx < QL_TF_LAST; subx++)
	{
		gtk_combo_box_insert_text (tab->view->sm_time_box, subx,
			get_timestamp(convert_old_tf(subx)));
	}
	g_signal_connect (GTK_OBJECT (tab->view->sm_time_box), "changed",
		G_CALLBACK (sm_formatting_changed), NULL);

	gtk_box_pack_start (GTK_BOX (sub_menu_box),
		GTK_WIDGET (tab->view->sm_time_box), FALSE, FALSE, 0);
	return sub_menu_box;
}								/* end of sub_menus */

/** setup a new text field as a standard text field.  User might
  change it as they edit it */
static void
new_std_field (QlTabData * tab)
{
	gtk_entry_set_text (GTK_ENTRY (column_name_entry), "");
	/* if last_field =2, then new is Field 4 */
	field_db_curr->name = g_strdup_printf (_("Column %u"),
		tab->file->last_field + 2);
	field_db_curr->sheet_column = tab->file->last_field + 1;
	field_db_curr->type = FIELD_TYPE_TEXT;
	field_db_curr->formatting = 0;
	field_db_curr->decimal_places = 0;
	field_db_curr->justification = GTK_JUSTIFY_LEFT;
	field_db_curr->width = 10;
}

static void
add_column_cb (gpointer G_GNUC_UNUSED field_index, gpointer field_ptr, gpointer user_data)
{
	gint sheet_column;
	QlFieldInfo * field = (QlFieldInfo*)field_ptr;
	sheet_column = GPOINTER_TO_INT (user_data);
	if (field->sheet_column >= sheet_column)
		field->sheet_column++;
}

/** \brief Create new field in the sheet

Called by ::build_field_db and therefore used by new and
existing files. Remember that the sheet isn't open yet if
this is a new file.
*/
static void
ok_clicked (GtkObject G_GNUC_UNUSED * object, gpointer data)
{
	gint fieldx;
	gint sheetx;
	gint rowx;
	const gchar *text;
	gchar *tmp;
	GtkSheetRange range;
	gdouble temp_double;
	gchar linebuf[48];
	QlTabData * tab;

	tab = (QlTabData*)data;
	g_return_if_fail (tab);
	text = gtk_entry_get_text (GTK_ENTRY (column_name_entry));
	tmp = g_strdup (text);
	if (check_entry (tab, tmp))
		return;
	g_free (tmp);
	field_db_curr->name = g_strdup(text);

	if (field_db_curr->type == FIELD_TYPE_NUMERIC)
	{
		field_db_curr->decimal_places = gtk_spin_button_get_value_as_int
			(GTK_SPIN_BUTTON(decimal_places_entry));
	}

	/* we have an existing file and we are adding a new field/column */
	if (tab->view->dialog_mode == MODE_ADD)
	{
		field_db_curr->sheet_column = sheetx = tab->view->sel_range.col0;
		ql_fieldinfo_foreach (tab, add_column_cb, GINT_TO_POINTER (sheetx));

		fieldx = ++tab->file->last_field;
		big_draw_start (tab);
		ql_add_fieldinfo (tab, field_db_curr);

		/* add column, add to sheet now */
		gtk_sheet_insert_columns (tab->view->sheet, sheetx, 1);
		gtk_sheet_set_column_width (tab->view->sheet, sheetx,
			field_db_curr->width * 8);
		gtk_sheet_column_button_add_label (tab->view->sheet,
			sheetx, field_db_curr->name);
		gtk_sheet_column_set_justification (tab->view->sheet,
			sheetx, field_db_curr->justification);

		reset_col_to_field (tab);
		big_draw_end (tab);
		front_is_changed (tab);
		return;
	}							/* end of add field */

	/* we have an existing file and editing the fields. */
	else if (tab->view->dialog_mode == MODE_EDIT)
	{
		QlFieldInfo * field;
		sheetx = tab->view->sel_range.col0;
		field = ql_get_fieldinfo (tab, sheetx);
		field = field_db_curr;
		gtk_sheet_column_button_add_label (tab->view->sheet,
			sheetx, field_db_curr->name);
		big_draw_start (tab);
		if (field_db_curr->justification != field_db_start->justification)
		{
			gtk_sheet_column_set_justification (tab->view->sheet,
				sheetx, field_db_curr->justification);
			range.row0 = 0;
			range.rowi = tab->file->last_row;
			range.col0 = range.coli = sheetx;
			gtk_sheet_range_set_justification (tab->view->sheet,
				&range, field_db_curr->justification);
		}

		/* check if numeric format changed */
		if (field_db_curr->type == FIELD_TYPE_TEXT ||
			!tab->file->last_row)
		{
			big_draw_end (tab);
			front_is_changed (tab);
			return;
		}
		if (field_db_curr->type != FIELD_TYPE_TEXT &&
			(field_db_start->type != field_db_curr->type ||
				field_db_start->formatting != field_db_curr->formatting ||
				field_db_start->decimal_places !=
				field_db_curr->decimal_places))
		{

			/* make sure the old formatting code is valid */
			/** \bug need type-specific GList based formatting. */
			if (field_db_start->type == FIELD_TYPE_TEXT)
				field_db_start->formatting = field_db_curr->formatting;	/* default */
			for (rowx = 0; rowx < tab->file->last_row; rowx++)
			{
				text = gtk_sheet_cell_get_text (tab->view->sheet,
					rowx, sheetx);
				if (text)
				{
					temp_double = qls2d (text, field_db_curr->type,
						field_db_start->formatting);
					d2qls (linebuf, temp_double, field_db_curr->type,
						field_db_curr->formatting,
						field_db_curr->decimal_places);
					gtk_sheet_set_cell_text (tab->view->sheet,
						rowx, sheetx, linebuf);
				}
			}
		}
		big_draw_end (tab);
		front_is_changed (tab);
		return;
	}

	/* we have a new file */
	fieldx = sheetx = ++tab->file->last_field;
	ql_add_fieldinfo (tab, field_db_curr);
	reset_col_to_field (tab);
	new_std_field (tab);
	field_db_start = field_db_curr;
	update_field_db (tab);
}

static void
swap_columns (QlTabData * tab, gint tocol, gint from)
{
	QlFieldInfo * fieldto, * fieldfrom;
	gchar *textfromp;
	gchar *texttop;
	gchar textfrom[1024];
	gint from_justification;
	gint to_justification;
	gint rowx;
	gint to_sheet_col;

	big_draw_start (tab);
	fieldfrom = ql_get_fieldinfo (tab, from);
	fieldto = ql_get_fieldinfo (tab, tocol);
	from_justification = fieldfrom->justification;
	to_justification = fieldto->justification;
	to_sheet_col = fieldto->sheet_column;
	fieldto->sheet_column = fieldfrom->sheet_column;
	fieldfrom->sheet_column = to_sheet_col;

	gtk_sheet_set_column_width (tab->view->sheet, from, fieldto->width * 8);
	gtk_sheet_set_column_width (tab->view->sheet, tocol, fieldfrom->width * 8);
	gtk_sheet_column_button_add_label (tab->view->sheet, from, fieldto->name);
	gtk_sheet_column_button_add_label (tab->view->sheet, tocol,
		fieldfrom->name);
	gtk_sheet_column_set_justification (tab->view->sheet, from,
		fieldto->justification);
	gtk_sheet_column_set_justification (tab->view->sheet, tocol,
		fieldfrom->justification);

	reset_col_to_field (tab);

	/* now move the row data */
	for (rowx = 0; rowx <= tab->file->last_row; rowx++)
	{
		textfromp = gtk_sheet_cell_get_text (tab->view->sheet,
			rowx, from);
		if (textfromp)
			strcpy (textfrom, textfromp);

		/* if text in the "to" column, move it to the "from" column */
		texttop = gtk_sheet_cell_get_text (tab->view->sheet,
			rowx, tocol);
		if (texttop)
			gtk_sheet_set_cell (tab->view->sheet, rowx, from,
				to_justification, texttop);
		else
			gtk_sheet_cell_clear (tab->view->sheet, rowx, from);

		/* put the "from" text, if any, into the "to" column */
		if (textfromp)
			gtk_sheet_set_cell (tab->view->sheet, rowx, tocol,
				from_justification, textfrom);
		else
			gtk_sheet_cell_clear (tab->view->sheet, rowx, tocol);
	}
	big_draw_end (tab);
	front_is_changed (tab);
	gtk_sheet_select_column (tab->view->sheet, tocol);
}

/** \brief Builds dialogue box to edit field parameters

\param front->sel_range.col0 the field being edited
\param field_db_curr  the information we want to display
\param field_db_start the settings at the time the dialog
   box was opened.  Zero is possible */
void
build_field_db (QlTabData * tab)
{
	GtkWidget *top_box;
	GtkWidget *hbox_name;
	GtkWidget *hbox_type;
	GtkWidget *hbox_just;
	GtkWidget *button_box;
	GtkWidget *ok_button;
	GtkWidget *cancel_button;
	GtkWidget *name_label;

	entry_dialogue = gtk_dialog_new ();
	gtk_window_set_modal (GTK_WINDOW (entry_dialogue), TRUE);
	gtk_window_set_position (GTK_WINDOW (entry_dialogue), GTK_WIN_POS_CENTER);
	gtk_window_set_resizable (GTK_WINDOW (entry_dialogue), TRUE);
	gtk_container_set_border_width (GTK_CONTAINER (entry_dialogue), 5);
	displaying = FALSE;
	field_db_start = field_db_curr;
	/* associate the tab context with the field dialogue. */
	g_object_set_data (G_OBJECT(entry_dialogue), QLTABID, tab);
	gtk_window_set_title (GTK_WINDOW (entry_dialogue),
		_("Edit Column Information"));
	top_box = GTK_DIALOG (entry_dialogue)->vbox;
	gtk_box_set_spacing (GTK_BOX (top_box), 10);
	button_box = GTK_DIALOG (entry_dialogue)->action_area;

	hbox_name = gtk_hbox_new (FALSE, 5);
	name_label = gtk_label_new (_("Column Name"));
	gtk_widget_show (name_label);
	gtk_box_pack_start (GTK_BOX (hbox_name), name_label, FALSE, FALSE, 0);

	column_name_entry = gtk_entry_new ();
	gtk_entry_set_max_length (GTK_ENTRY (column_name_entry), MAX_FIELD_NAME);
	gtk_box_pack_start (GTK_BOX (hbox_name), column_name_entry, TRUE, TRUE, 0);
	gtk_widget_show (column_name_entry);
	gtk_widget_show (hbox_name);
	gtk_box_pack_start (GTK_BOX (top_box), hbox_name, FALSE, FALSE, 0);

	/* Got the top box formatted, now go for type radio buttons */
	hbox_type = gtk_hbox_new (FALSE, 3);
	text_button = gtk_radio_button_new_with_label (NULL, _("Text"));
	g_signal_connect (GTK_OBJECT (text_button), "clicked",
		G_CALLBACK (type_clicked), GINT_TO_POINTER (FIELD_TYPE_TEXT));
	gtk_box_pack_start (GTK_BOX (hbox_type), text_button, TRUE,
		TRUE, 0);
	gtk_widget_show (text_button);

	numeric_button = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group
		(GTK_RADIO_BUTTON (text_button)), _("Numeric"));
	g_signal_connect (GTK_OBJECT (numeric_button), "clicked",
		G_CALLBACK (type_clicked), GINT_TO_POINTER (FIELD_TYPE_NUMERIC));
	gtk_box_pack_start (GTK_BOX (hbox_type), numeric_button, TRUE,
		TRUE, 0);
	gtk_widget_show (numeric_button);

	date_button = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group
		(GTK_RADIO_BUTTON (text_button)), _("Date"));
	g_signal_connect (GTK_OBJECT (date_button), "clicked",
		G_CALLBACK (type_clicked), GINT_TO_POINTER (FIELD_TYPE_DATE));
	gtk_box_pack_start (GTK_BOX (hbox_type), date_button, TRUE,
		TRUE, 0);
	gtk_widget_show (date_button);

	time_button = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group
		(GTK_RADIO_BUTTON (text_button)), _("Time"));
	g_signal_connect (GTK_OBJECT (time_button), "clicked",
		G_CALLBACK (type_clicked), GINT_TO_POINTER (FIELD_TYPE_TIME));
	gtk_box_pack_start (GTK_BOX (hbox_type), time_button, TRUE,
		TRUE, 0);
	gtk_widget_show (time_button);
	gtk_widget_show (hbox_type);
	gtk_box_pack_start (GTK_BOX (top_box), hbox_type, FALSE, FALSE, 0);

	/* Now the justification radio buttons */
	hbox_just = gtk_hbox_new (FALSE, 3);
	left_button = gtk_radio_button_new_with_label (NULL, _("Left"));
	g_signal_connect (GTK_OBJECT (left_button), "clicked",
		G_CALLBACK (just_clicked), GINT_TO_POINTER (GTK_JUSTIFY_LEFT));
	gtk_box_pack_start (GTK_BOX (hbox_just), left_button, TRUE,
		TRUE, 0);
	gtk_widget_show (left_button);

	center_button = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group
		(GTK_RADIO_BUTTON (left_button)), _("Center"));
	g_signal_connect (GTK_OBJECT (center_button), "clicked",
		G_CALLBACK (just_clicked), GINT_TO_POINTER (GTK_JUSTIFY_CENTER));
	gtk_box_pack_start (GTK_BOX (hbox_just), center_button, TRUE,
		TRUE, 0);
	gtk_widget_show (center_button);

	right_button = gtk_radio_button_new_with_label
		(gtk_radio_button_get_group
		(GTK_RADIO_BUTTON (left_button)), _("Right"));
	g_signal_connect (GTK_OBJECT (right_button), "clicked",
		G_CALLBACK (just_clicked), GINT_TO_POINTER (GTK_JUSTIFY_RIGHT));
	gtk_box_pack_start (GTK_BOX (hbox_just), right_button, TRUE,
		TRUE, 0);
	gtk_widget_show (right_button);
	gtk_widget_show (hbox_just);
	gtk_box_pack_start (GTK_BOX (top_box), hbox_just, FALSE, FALSE, 0);

	/* go get the display types buttons for numeric, date and time */
	gtk_box_pack_start (GTK_BOX (top_box), sub_menus (tab), FALSE, FALSE, 0);

	/* Now do OK, Done and Cancel buttons */
	ok_button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_widget_show (ok_button);
	gtk_box_pack_start (GTK_BOX (button_box), ok_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (ok_button), "clicked",
		G_CALLBACK (ok_clicked), tab);

	if (tab->view->dialog_mode == MODE_NEW)
	{
		done_button = gtk_button_new_from_stock (GTK_STOCK_APPLY);
		/* show is done later */
		gtk_box_pack_start (GTK_BOX (button_box),
			done_button, TRUE, TRUE, 0);
		g_signal_connect (GTK_OBJECT (done_button), "clicked",
			G_CALLBACK (done_clicked), tab);
	}
	cancel_button = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
	gtk_widget_show (cancel_button);
	gtk_box_pack_start (GTK_BOX (button_box),
		cancel_button, TRUE, TRUE, 0);
	g_signal_connect (GTK_OBJECT (cancel_button), "clicked",
		G_CALLBACK (cancel_clicked), tab);
	g_signal_connect (GTK_OBJECT (entry_dialogue),
		"delete_event", G_CALLBACK (cancel_clicked), tab);
	gtk_widget_show (entry_dialogue);
}

void
column_add (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_CHANGED(tab);
	tab->view->dialog_mode = MODE_ADD;
	build_field_db (tab);
	new_std_field (tab);
	update_field_db (tab);
}

void
column_edit (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint this_field, selected;

	/** \todo simplify the notation further */
	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_RANGE;
	CHECK_CHANGED(tab);
	if (tab->view->sel_type != SELECT_COLUMN)
		return;
	tab->view->dialog_mode = MODE_EDIT;
	selected = tab->view->sel_range.col0;
	this_field = tab->file->col_to_field[selected];
	field_db_curr = ql_get_fieldinfo (tab, this_field);
	build_field_db (tab);
	update_field_db (tab);
}

void
del_column_cb (gpointer field_index, gpointer field_ptr,
	gpointer G_GNUC_UNUSED user_data)
{
	QlFieldInfo * field = (QlFieldInfo*)field_ptr;
	gint colx = GPOINTER_TO_INT (field_index);
	g_return_if_fail (field);
	if (field->sheet_column > colx)
		field->sheet_column--;
}

void
column_delete (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	gint colx, field_leaving, groupx;
	gint linex, reportto, reportfrom, starting_reports;
	gint colto, colfrom, starting_cols;
	QlReportInfo temp_report;
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	/* someday we should have code to ask if the user
	really wants to delete a column. */
	CHECK_RANGE;
	CHECK_CHANGED(tab);
	front_is_changed (tab);
	colx = tab->view->sel_range.col0;
	field_leaving = tab->file->col_to_field[colx];
	gtk_sheet_delete_columns (tab->view->sheet, colx, 1);

/*	for (fieldx = field_leaving; fieldx < tab->file->last_field; fieldx++)
		tab->file->fields[fieldx] = tab->file->fields[fieldx + 1];*/

	tab->file->last_field--;
	ql_remove_fieldinfo (tab, field_leaving);
	ql_fieldinfo_foreach (tab, del_column_cb, GINT_TO_POINTER(colx));
/*	for (fieldx = 0; fieldx <= tab->file->last_field; fieldx++)
		if (tab->file->fields[fieldx]->sheet_column > colx)
			tab->file->fields[fieldx]->sheet_column--;*/

	reset_col_to_field (tab);

	/* then go through sorts */
	for (groupx = 0; groupx < tab->file->sort_ct; groupx++)
	{
		for (linex = 0; linex < tab->file->sorts[groupx].line_ct; linex++)
		{
			if (tab->file->sorts[groupx].line[linex].field == field_leaving)
				tab->file->sorts[groupx].line[linex].field = -1;
			else if (tab->file->sorts[groupx].line[linex].field >
				field_leaving)
				tab->file->sorts[groupx].line[linex].field--;
		}
	}

	/* now go through filters */
	for (groupx = 0; groupx < tab->file->filter_ct; groupx++)
	{
		for (linex = 0; linex < tab->file->filters[groupx].line_ct; linex++)
		{
			if (tab->file->filters[groupx].line[linex].field == field_leaving)
				tab->file->filters[groupx].line[linex].field = -1;
			else if (tab->file->filters[groupx].line[linex].field >
				field_leaving)
				tab->file->filters[groupx].line[linex].field--;
		}
	}

	/* now go through reports */
	reportto = 0;
	starting_reports = tab->file->report_ct;
	for (reportfrom = 0; reportfrom < starting_reports; reportfrom++)
	{
		temp_report = tab->file->reports[reportfrom];
		colto = 0;
		starting_cols = temp_report.last_column;
		for (colfrom = 0; colfrom <= starting_cols; colfrom++)
		{
			if (temp_report.column[colfrom].field != field_leaving)
			{
				if (temp_report.column[colfrom].field > field_leaving)
					temp_report.column[colfrom].field--;
				temp_report.column[colto++] = temp_report.column[colfrom];
			}
			else
				temp_report.last_column--;
		}
		if (temp_report.last_column >= 0)	/* only if valid fields in report */
			tab->file->reports[reportto++] = temp_report;
		else
			tab->file->report_ct--;
	}
}								/* end of delete field */

void
column_left (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint from, to;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_RANGE;
	CHECK_CHANGED(tab);
	from = tab->view->sel_range.col0 - 1;
	to = tab->view->sel_range.col0;
	swap_columns (tab, to, from);
}

void
column_right (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;
	gint from, to;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_get_tabdata (qlc);
	CHECK_RANGE;
	CHECK_CHANGED(tab);
	to = tab->view->sel_range.col0 + 1;
	from = tab->view->sel_range.col0;
	swap_columns (tab, to, from);
}

void
new_file (GtkAction G_GNUC_UNUSED * w, gpointer data)
{
	QlContext * qlc;
	QlTabData * tab;

	qlc = ql_get_context (GTK_WIDGET(data));
	tab = ql_new_tabdata (qlc);
	tab->file->last_field = -1;

	tab->view->dialog_mode = MODE_NEW;
	/* Translators: this is the name of a new file before being saved with a name */
	tab->file->file_path = g_strconcat ("~/", _("Untitled_New_List"), ".qlf", NULL);
	tab->file->file_name = g_strconcat (_("Untitled_New_List"), ".qlf", NULL);
	field_db_curr = g_new0(QlFieldInfo, 1);
	field_db_start = g_new0(QlFieldInfo, 1);
	build_field_db (tab);
	new_std_field (tab);
	update_field_db (tab);
}								/* End of new file */
