/* KInterbasDB Python Package - Implementation of Transactional Operations
 *
 * Version 3.2
 *
 * The following contributors hold Copyright (C) over their respective
 * portions of code (see license.txt for details):
 *
 * [Original Author (maintained through version 2.0-0.3.1):]
 *   1998-2001 [alex]  Alexander Kuznetsov   <alexan@users.sourceforge.net>
 * [Maintainers (after version 2.0-0.3.1):]
 *   2001-2002 [maz]   Marek Isalski         <kinterbasdb@maz.nu>
 *   2002-2006 [dsr]   David Rushby          <woodsplitter@rocketmail.com>
 * [Contributors:]
 *   2001      [eac]   Evgeny A. Cherkashin  <eugeneai@icc.ru>
 *   2001-2002 [janez] Janez Jere            <janez.jere@void.si>
 */

/* Distributed transaction support added 2003.04.27. */

/*************************** DECLARATIONS : begin ****************************/
typedef enum {
  OP_COMMIT   =  1,
  OP_ROLLBACK =  0
} WhichTransactionOperation;

typedef enum {
  OP_RESULT_OK     =   0,
  OP_RESULT_ERROR  =  -1
} TransactionalOperationResult;

static isc_tr_handle _Connection_get_transaction_handle_from_group(
    CConnection *con
  );

static isc_tr_handle begin_transaction(
    /* Either: */  isc_db_handle db_handle, char *tpb, Py_ssize_t tpb_len,
    /* Or: */      ISC_TEB *tebs, short teb_count,
    ISC_STATUS *status_vector
  );

static PyObject *trans___s__trans_handle;
static PyObject *trans___s__default_tpb_str_;
/**************************** DECLARATIONS : end *****************************/

/************************ CORE FUNCTIONALITY : begin *************************/

static int init_kidb_transaction_support(void) {
  #define INIT_TRANS_STRING_CONST(s) \
    trans___s_ ## s = PyString_FromString(#s); \
    if (trans___s_ ## s == NULL) { goto fail; }

  INIT_TRANS_STRING_CONST(_trans_handle);
  INIT_TRANS_STRING_CONST(_default_tpb_str_);

  return 0;

  fail:
    return -1;
} /* init_kidb_transaction_support */

/* In any context where the second clause (the call to
 * _Connection_get_transaction_handle_from_group) might execute, the GIL must be
 * held. */
#define CON_GET_TRANS_HANDLE(con) ( \
  (con->trans_handle != NULL_TRANS_HANDLE) ? \
      con->trans_handle \
    : _Connection_get_transaction_handle_from_group(con) \
  )

#define CON_HAS_TRANSACTION(con) \
  ((boolean) (CON_GET_TRANS_HANDLE(con) != NULL_TRANS_HANDLE))


static int Connection_ensure_transaction(CConnection *con) {
  int res = 0;
  PyObject *py_default_tpb = NULL;

  assert (con != NULL);
  assert (con->python_wrapper_obj != NULL);
  #ifdef ENABLE_CONNECTION_TIMEOUT
    /* This function does not activate the connection, so it should only be
     * called when the connection has already been activated: */
    assert (
          Connection_timeout_enabled(con)
        ? con->timeout->state == CONOP_ACTIVE
        : TRUE
      );
  #endif

  if (!CON_HAS_TRANSACTION(con)) {
    if (con->group == NULL) {
      py_default_tpb = PyObject_GetAttr(con->python_wrapper_obj,
          trans___s__default_tpb_str_
        );
      if (py_default_tpb == NULL) { goto fail; }
      if (!PyString_CheckExact(py_default_tpb)) {
        raise_exception(InternalError, "Connection._default_tpb_str_ must be"
            " of type str."
          );
        goto fail;
      }

      {
        char *tpb = PyString_AS_STRING(py_default_tpb);
        const Py_ssize_t tpb_len = PyString_GET_SIZE(py_default_tpb);
        con->trans_handle = begin_transaction(
            con->db_handle, tpb, tpb_len,
            NULL, -1, /* all TEB-related params are null */
            con->status_vector
          );
      }
      if (con->trans_handle == NULL_TRANS_HANDLE) { goto fail; }
    } else {
      /* Call the 'begin' method of con->group: */
      PyObject *py_ret = PyObject_CallMethod(con->group, "begin", NULL);
      if (py_ret == NULL) { goto fail; }
      Py_DECREF(py_ret);
    }
  }

  assert (res == 0);
  goto clean;
  fail:
    assert (PyErr_Occurred());
    res = -1;
    /* Fall through to clean: */
  clean:
    Py_XDECREF(py_default_tpb);
    return res;
} /* Connection_ensure_transaction */

static isc_tr_handle _Connection_get_transaction_handle_from_group(
    CConnection *con
  )
{
  PyObject *group = con->group;
  isc_tr_handle native_handle = NULL_TRANS_HANDLE;

  /* This function should never be called if the con has its own trans_handle
   * (use the convenience macro CON_GET_TRANS_HANDLE for situations where that
   * might be the case). */
  assert (con->trans_handle == NULL_TRANS_HANDLE);

  if (group != NULL) {
    PyObject *py_trans_handle = PyObject_GetAttr(group,
        trans___s__trans_handle
      );
    if (py_trans_handle == NULL) {
      goto fail;
    } else if (py_trans_handle != Py_None) {
      /* The Python layer of kinterbasdb's internal code should never set
       * ConnectionGroup._trans_handle to anything other than a
       * TransactionHandleObject or None; enforce this. */
      if (py_trans_handle->ob_type != &TransactionHandleType) {
        raise_exception(InternalError, "ConnectionGroup._trans_handle is not"
            " an instance of TransactionHandleType."
          );
        Py_DECREF(py_trans_handle);
        goto fail;
      }

      native_handle =
        ((TransactionHandleObject *) py_trans_handle)->native_handle;
    }
    Py_DECREF(py_trans_handle);
  }

  return native_handle;

  fail:
    assert (PyErr_Occurred());
    return NULL_TRANS_HANDLE;
} /* _Connection_get_transaction_handle_from_group */

static isc_tr_handle *CON_GET_TRANS_HANDLE_ADDR(CConnection *con) {
  if (con->trans_handle != NULL_TRANS_HANDLE) {
    return &con->trans_handle;
  } else {
    PyObject *group = con->group;
    isc_tr_handle *native_handle_addr = NULL;

    if (group != NULL) {
      PyObject *py_trans_handle = PyObject_GetAttr(group,
          trans___s__trans_handle
        );
      if (py_trans_handle == NULL) { goto fail; }
      /* The Python layer should not allow this function to be called if the
       * ConnectionGroup has not yet established a transaction handle. */
      assert (py_trans_handle != Py_None);
      if (py_trans_handle->ob_type != &TransactionHandleType) {
        raise_exception(InternalError, "ConnectionGroup._trans_handle is not"
            " an instance of TransactionHandleType."
          );
        Py_DECREF(py_trans_handle);
        goto fail;
      }

      native_handle_addr =
        &((TransactionHandleObject *) py_trans_handle)->native_handle;

      Py_DECREF(py_trans_handle);
    }

    return native_handle_addr;
  }

  fail:
    assert (PyErr_Occurred());
    return NULL;
} /* CON_GET_TRANS_HANDLE_ADDR */

static isc_tr_handle begin_transaction(
    /* Either: */  isc_db_handle db_handle, char *tpb, Py_ssize_t tpb_len,
    /* Or: */      ISC_TEB *tebs, short teb_count,
    ISC_STATUS *status_vector
  )
{
  isc_tr_handle trans_handle = NULL_TRANS_HANDLE;

  /* (db_handle+tpb+tpb_len) and (tebs+teb_count) are mutually exclusive
   * parameters. */
  assert (db_handle != NULL_DB_HANDLE ? tebs == NULL : tebs != NULL && tpb == NULL);

  /* 2003.02.21: A huge TPB such as 'con.begin(tpb='x'*50000)' crashes the
   * FB 1.0.2 server process, but responsibly raises an error with FB 1.5b2.
   * Since kinterbasdb only exposes some 20 TPB component values, many of which
   * are mutually exclusive, I decided to impose a reasonable limit right
   * here. */
  if (tpb_len > 255) {
    raise_exception(ProgrammingError, "Transaction parameter buffer (TPB) too"
        " large.  len(tpb) must be <= 255."
      );
    goto fail;
  }

  ENTER_GDAL
  if (tebs == NULL) {
    isc_start_transaction(status_vector,
        &trans_handle,
        /* Only one database handle is being passed. */
        1, &db_handle,
        (unsigned short) tpb_len, /* Cast is safe b/c already checked val. */
        tpb
      );
  } else {
    isc_start_multiple(status_vector, &trans_handle, teb_count, tebs);
  }
  LEAVE_GDAL

  if (DB_API_ERROR(status_vector)) {
    raise_sql_exception(OperationalError, "begin transaction: ", status_vector);
    goto fail;
  }

  assert (trans_handle != NULL_TRANS_HANDLE);
  return trans_handle;

  fail:
    assert (PyErr_Occurred());
    return NULL_TRANS_HANDLE;
} /* begin_transaction */

/* 2003.08.28:  Added option for manual control over phases of 2PC. */
static TransactionalOperationResult prepare_transaction(
    isc_tr_handle trans_handle, ISC_STATUS *status_vector
  )
{
  /* The Python DB API requires that commit_transaction and
   * rollback_transaction accept a nonexistent transaction without
   * complaint. */
  if (trans_handle == NULL_TRANS_HANDLE) {
    return OP_RESULT_OK;
  }

  ENTER_GDAL
  isc_prepare_transaction(status_vector, &trans_handle);
  LEAVE_GDAL
  if (DB_API_ERROR(status_vector)) {
    raise_sql_exception(OperationalError, "prepare: ", status_vector);
    return OP_RESULT_ERROR;
  }

  return OP_RESULT_OK;
} /* prepare_transaction */

static TransactionalOperationResult commit_transaction(
    isc_tr_handle trans_handle, boolean retaining,
    ISC_STATUS *status_vector
  )
{
  if (trans_handle == NULL_TRANS_HANDLE) {
    /* As discussed on the Python DB-SIG in message:
     *   http://mail.python.org/pipermail/db-sig/2003-February/003158.html
     * , allow a transaction to be committed even if its existence is only
     * implicit. */
    return OP_RESULT_OK;
  }

  ENTER_GDAL
  if (!retaining) {
    isc_commit_transaction(status_vector, &trans_handle);
  } else {
    isc_commit_retaining(status_vector, &trans_handle);
    assert (trans_handle != NULL_TRANS_HANDLE);
  }
  LEAVE_GDAL

  if (DB_API_ERROR(status_vector)) {
    raise_sql_exception(OperationalError, "commit: ", status_vector);
    return OP_RESULT_ERROR;
  }

  return OP_RESULT_OK;
} /* commit_transaction */

static TransactionalOperationResult rollback_transaction(
    isc_tr_handle trans_handle, boolean retaining,
    boolean allowed_to_raise, ISC_STATUS *status_vector
  )
{
  /* If there is not an active transaction, rolling back is meaningless, but
   * acceptable. */
  if (trans_handle == NULL_TRANS_HANDLE) {
    return OP_RESULT_OK;
  }

  ENTER_GDAL
  if (!retaining) {
    isc_rollback_transaction(status_vector, &trans_handle);
  } else {
    #ifdef INTERBASE_6_OR_LATER /* IB 5.5 lacks isc_rollback_retaining. */
      isc_rollback_retaining(status_vector, &trans_handle);
      assert (trans_handle != NULL_TRANS_HANDLE);
    #else
      raise_exception(OperationalError, "Versions of Interbase prior to 6.0"
          " do not support retaining rollback."
        );
      return OP_RESULT_ERROR;
    #endif
  }
  LEAVE_GDAL

  if (DB_API_ERROR(status_vector)) {
    raise_sql_exception(OperationalError, "rollback: ", status_vector);
    if (allowed_to_raise) {
      return OP_RESULT_ERROR;
    } else {
      SUPPRESS_EXCEPTION;
    }
  }

  return OP_RESULT_OK;
} /* rollback_transaction */

static TransactionalOperationResult rollback_ANY_transaction(
    CConnection *con, boolean allowed_to_raise
  )
{
  /* Given a connection, this function rolls back either the connection's own
   * non-distributed transaction or the connection's group's distributed
   * transaction.
   * Note that this function never performs a retaining rollback. */
  TransactionalOperationResult result = OP_RESULT_ERROR;
  if (con->group == NULL) {
    CLEAR_OPEN_BLOBREADER_LIST_IF_NECESSARY(con, return OP_RESULT_ERROR);
    result = rollback_transaction(
        con->trans_handle, FALSE /* not retaining */,
        allowed_to_raise, con->status_vector
      );
    con->trans_handle = NULL_TRANS_HANDLE;
    Connection_clear_transaction_stats(con);
  } else {
    assert (con->trans_handle == NULL_TRANS_HANDLE);
    {
      PyObject *py_result = PyObject_CallMethod(con->group, "rollback", NULL);
      if (py_result == NULL) {
        if (!allowed_to_raise) {
          SUPPRESS_EXCEPTION;
        }
      } else {
        result = OP_RESULT_OK;
        Py_DECREF(py_result);
      }
    }
  }
  return result;
} /* rollback_ANY_transaction */

/************************* CORE FUNCTIONALITY : end **************************/

/********** PYTHON WRAPPERS FOR NON-DISTRIBUTED TRANS OPS : begin ************/

static PyObject *pyob_Connection_begin(PyObject *self, PyObject *args) {
  PyObject *ret = NULL;
  CConnection *con;
  char *tpb = NULL;
  Py_ssize_t tpb_len = 0;

  if (!PyArg_ParseTuple(args, "O!s#", &ConnectionType, &con, &tpb, &tpb_len)) {
    goto fail;
  }
  CON_ACTIVATE(con, return NULL);

  /* Raise a more informative error message if the previous transaction is
   * still active when the client attempts to start another.  The old approach
   * was to go ahead and try to start the new transaction regardless.  If there
   * was already an active transaction, the resulting exception made no mention
   * of it, which was very confusing. */
  if (CON_HAS_TRANSACTION(con)) { /* 2003.10.15a:OK */
    raise_exception_with_numeric_error_code(ProgrammingError, -901,
        "Previous transaction still active; cannot start new transaction."
        "  Use commit() or rollback() to resolve the old transaction first."
      );
    goto fail;
  }

  con->trans_handle = begin_transaction(
      con->db_handle, tpb, tpb_len,
      NULL, -1, /* all TEB-related params are null */
      con->status_vector
    );

  if (con->trans_handle == NULL_TRANS_HANDLE) { goto fail; }

  Py_INCREF(Py_None);
  ret = Py_None;

  goto clean;
  fail:
    assert (PyErr_Occurred());
    /* Fall through to clean: */
  clean:
    CON_PASSIVATE(con);
    CON_MUST_NOT_BE_ACTIVE(con);
    return ret;
} /* pyob_Connection_begin */

/* 2003.08.28:  Added option for manual control over phases of 2PC. */
static PyObject *pyob_Connection_prepare(PyObject *self, PyObject *args) {
  PyObject *res = NULL;
  CConnection *con;

  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }
  CON_ACTIVATE(con, return NULL);

  if (   prepare_transaction(con->trans_handle, con->status_vector)
      != OP_RESULT_OK
     )
  { goto fail; }

  assert (!PyErr_Occurred());
  res = Py_None;
  Py_INCREF(Py_None);

  goto clean;
  fail:
    assert (PyErr_Occurred());
    assert (res == NULL);
    /* Fall through to clean: */
  clean:
    CON_PASSIVATE(con);
    CON_MUST_NOT_BE_ACTIVE(con);
    return res;
} /* pyob_Connection_prepare */

static TransactionalOperationResult commit_or_rollback(
    WhichTransactionOperation op, CConnection *con, boolean retaining
  )
{
  TransactionalOperationResult action_result = OP_RESULT_ERROR;

  /* This function is not intended to handle distributed transactions: */
  assert (con->group == NULL);

  CLEAR_OPEN_BLOBREADER_LIST_IF_NECESSARY(con, return OP_RESULT_ERROR);
  switch (op) {
    case OP_COMMIT:
      action_result = commit_transaction(
          CON_GET_TRANS_HANDLE(con), /* 2003.10.15a:OK */
          retaining, con->status_vector
        );
      break;

    case OP_ROLLBACK:
      action_result = rollback_transaction(
          CON_GET_TRANS_HANDLE(con), /* 2003.10.15a:OK */
          retaining, TRUE, con->status_vector
        );
      break;
  }

  if (action_result == OP_RESULT_OK) {
    if (!retaining) {
      con->trans_handle = NULL_TRANS_HANDLE;
      Connection_clear_transaction_stats(con);
    }
  }

  return action_result;
} /* commit_or_rollback */

static PyObject *_pyob_commit_or_rollback(
    WhichTransactionOperation op, PyObject *self, PyObject *args
  )
{
  PyObject *ret = NULL;
  CConnection *con;
  PyObject *py_retaining;
  int retaining_int;

  if (!PyArg_ParseTuple(args, "O!O", &ConnectionType, &con, &py_retaining)) {
    return NULL;
  }
  retaining_int = PyObject_IsTrue(py_retaining);
  if (retaining_int == -1) { return NULL; }

  CON_ACTIVATE(con, return NULL);

  if (commit_or_rollback(op, con, (boolean) retaining_int) == OP_RESULT_OK) {
    Py_INCREF(Py_None);
    ret = Py_None;
  }

  CON_PASSIVATE(con);
  CON_MUST_NOT_BE_ACTIVE(con);
  assert (PyErr_Occurred() ? ret == NULL : ret != NULL);
  return ret;
} /* _pyob_commit_or_rollback */

static PyObject *pyob_Connection_commit(PyObject *self, PyObject *args) {
  return _pyob_commit_or_rollback(OP_COMMIT, self, args);
} /* pyob_Connection_commit */

static PyObject *pyob_Connection_rollback(PyObject *self, PyObject *args) {
  return _pyob_commit_or_rollback(OP_ROLLBACK, self, args);
} /* pyob_Connection_rollback */

/*********** PYTHON WRAPPERS FOR NON-DISTRIBUTED TRANS OPS : end *************/

/*********** PYTHON WRAPPERS FOR DISTRIBUTED TRANS OPS : begin ***************/

static TransactionHandleObject *new_transaction_handle(void) {
  TransactionHandleObject *trans_handle =
    PyObject_New(TransactionHandleObject, &TransactionHandleType);
  if (trans_handle == NULL) { return NULL; }

  trans_handle->native_handle = NULL_TRANS_HANDLE;

  return trans_handle;
} /* new_transaction_handle */

static void pyob_transaction_handle_del(PyObject *obj) {
  TransactionHandleObject *trans_handle = (TransactionHandleObject *) obj;

  /* Normally, the database client library will have already set the
   * native_handle to NULL when it either committed or rolled back the
   * transaction.  If the client library was not able to do either of those,
   * we simply forget about the handle (don't free, because a handle is not
   * necessarily a pointer). */
  if (trans_handle->native_handle != NULL_TRANS_HANDLE) {
    trans_handle->native_handle = NULL_TRANS_HANDLE;
  }

  /* Delete the memory of the TransactionHandleObject struct itself: */
  PyObject_Del(trans_handle);
} /* pyob_transaction_handle_del */

static ISC_TEB *build_teb_buffer(PyObject *cons) {
  ISC_TEB *tebs = NULL;
  Py_ssize_t teb_count;
  Py_ssize_t tebs_size;
  CConnection *con = NULL;
  PyObject *tpb = NULL;
  Py_ssize_t i;

  /* The caller (internal kinterbasdb code) should have already ensured this: */
  assert (PyList_Check(cons));

  teb_count = PyList_GET_SIZE(cons);
  tebs_size = sizeof(ISC_TEB) * teb_count;
  tebs = kimem_main_malloc(tebs_size);
  if (tebs == NULL) { goto fail; }

  for (i = 0; i < teb_count; i++) {
    ISC_TEB *t = tebs + i;
    PyObject *py_con = PyList_GET_ITEM(cons, i); /* borrowed ref */

    /* PyObject_GetAttr returns a new reference.  These new references are
     * released at the end of each iteration of this loop (normally), or in the
     * fail clause in case of error. */
    con = (CConnection *) PyObject_GetAttr(py_con, shared___s__C_con);
    if (con == NULL) { goto fail; }
    tpb = PyObject_GetAttr(py_con, trans___s__default_tpb_str_);
    if (tpb == NULL) { goto fail; }
    /* The Python layer should have already ensured this: */
    assert (con->db_handle != NULL_DB_HANDLE);

    t->db_ptr = (long *) &con->db_handle;

    if (tpb == Py_None) {
      t->tpb_len = 0;
      t->tpb_ptr = NULL;
    } else if (PyString_Check(tpb)) {
      const Py_ssize_t tpb_len = PyString_GET_SIZE(tpb);
      #if PY_SSIZE_T_MAX__CIRCUMVENT_COMPILER_COMPLAINT > LONG_MAX
      if (tpb_len > LONG_MAX) {
        raise_exception(ProgrammingError, "TPB size must be <= LONG_MAX.");
        goto fail;
      }
      #endif
      t->tpb_len = (long) tpb_len;

      t->tpb_ptr = PyString_AS_STRING(tpb);
    } else {
      PyErr_SetString(InternalError, "Connection._default_tpb_str_ must be a"
          " str or None."
        );
      goto fail;
    }
    Py_DECREF(con);
    Py_DECREF(tpb);
    /* Set to NULL to prevent double-decref in case of error during next
     * iteration. */
    con = NULL;
    tpb = NULL;
  }
  /* Upon successful exit, all references will have been released. */

  return tebs;

  fail:
    assert (PyErr_Occurred());

    Py_XDECREF(con);
    Py_XDECREF(tpb);
    if (tebs != NULL) { kimem_main_free(tebs); }

    return NULL;
} /* build_teb_buffer */

static PyObject *pyob_distributed_begin(PyObject *self, PyObject *args) {
  /* cons, the input object from the Python level, is a list of instances of
   * Python class kinterbasdb.Connection, not a list of instances of C type
   * ConnectionType. */
  PyObject *cons = NULL;

  TransactionHandleObject *trans_handle = NULL;
  ISC_TEB *tebs = NULL;
  Py_ssize_t teb_count;
  ISC_STATUS status_vector[STATUS_VECTOR_SIZE];

  if (!PyArg_ParseTuple(args, "O!", &PyList_Type, &cons)) { goto exit; }

  teb_count = PyList_GET_SIZE(cons);
  /* The Python layer should prevent the programmer from starting a distributed
   * transaction with an empty group. */
  assert (teb_count > 0);
  /* The Python layer (ConnectionGroup class) should prevent the programmer
   * from exceeding the database engine's limits on the number of database
   * handles that can participate in a single distributed transaction. */
  assert (teb_count <= DIST_TRANS_MAX_DATABASES);

  tebs = build_teb_buffer(cons);
  if (tebs == NULL) { goto exit; }

  trans_handle = new_transaction_handle();
  if (trans_handle == NULL) { goto exit; }

  trans_handle->native_handle = begin_transaction(
      NULL_DB_HANDLE, NULL, -1, /* All parameters for singleton transactions are null. */
      tebs, (short) teb_count, /* Cast is safe b/c already checked val. */
      status_vector
    );
  if (trans_handle->native_handle == NULL_TRANS_HANDLE) { goto exit; }

  exit:
    if (tebs != NULL) { kimem_main_free(tebs); }

    if (trans_handle == NULL) {
      assert (PyErr_Occurred());
      return NULL;
    } else if (trans_handle->native_handle == NULL_TRANS_HANDLE) {
      Py_DECREF(trans_handle);
      return NULL;
    } else {
      return (PyObject *) trans_handle;
    }
} /* pyob_distributed_begin */

/* 2003.08.28:  Added option for manual control over phases of 2PC. */
static PyObject *pyob_distributed_prepare(PyObject *self, PyObject *args) {
  TransactionHandleObject *py_handle;
  ISC_STATUS status_vector[STATUS_VECTOR_SIZE];

  if (!PyArg_ParseTuple(args, "O!", &TransactionHandleType, &py_handle)) {
    goto fail;
  }

  if (   prepare_transaction(py_handle->native_handle, status_vector)
      != OP_RESULT_OK
     )
  { goto fail; }

  RETURN_PY_NONE;

  fail:
    assert (PyErr_Occurred());
    return NULL;
} /* pyob_distributed_prepare */

static PyObject *_pyob_distributed_commit_or_rollback(
    WhichTransactionOperation op, PyObject *self, PyObject *args
  )
{
  TransactionHandleObject *trans_handle;
  boolean retaining;
  ISC_STATUS status_vector[STATUS_VECTOR_SIZE];
  TransactionalOperationResult action_result;

  {
    PyObject *py_retaining;
    if (!PyArg_ParseTuple(args, "O!O",
           &TransactionHandleType, &trans_handle, &py_retaining
         )
       )
    { goto fail; }
    retaining = (boolean) PyObject_IsTrue(py_retaining);
  }

  if (op == OP_COMMIT) {
    action_result = commit_transaction(
        trans_handle->native_handle, retaining, status_vector
      );
  } else {
    assert (op == OP_ROLLBACK);
    action_result = rollback_transaction(
        trans_handle->native_handle, retaining, TRUE, status_vector
      );
  }

  if (action_result != OP_RESULT_OK) { goto fail; }

  if (retaining) {
    assert (trans_handle->native_handle != NULL_TRANS_HANDLE);
  } else {
    trans_handle->native_handle = NULL_TRANS_HANDLE;
  }

  RETURN_PY_NONE;

  fail:
    assert (PyErr_Occurred());
    return NULL;
} /* _pyob_distributed_commit_or_rollback */

static PyObject *pyob_distributed_commit(PyObject *self, PyObject *args) {
  return _pyob_distributed_commit_or_rollback(OP_COMMIT, self, args);
} /* pyob_distributed_commit */

static PyObject *pyob_distributed_rollback(PyObject *self, PyObject *args) {
  return _pyob_distributed_commit_or_rollback(OP_ROLLBACK, self, args);
} /* pyob_distributed_rollback */

/************* PYTHON WRAPPERS FOR DISTRIBUTED TRANS OPS : end ****************/

static PyObject *pyob_Connection_has_transaction(PyObject *self, PyObject *args) {
  CConnection *con;

  if (!PyArg_ParseTuple(args, "O!", &ConnectionType, &con)) { return NULL; }

  return PyBool_FromLong(CON_HAS_TRANSACTION(con)); /* 2003.10.15a:OK */
} /* pyob_Connection_has_transaction */
