########################################################################
#
# File Name:            StylesheetReader.py
#
# Documentation:        http://docs.4suite.org/4XSLT/StylesheetReader.py.html
#

"""
Create a stylesheet object
WWW: http://4suite.org/4XSLT        e-mail: support@4suite.org

Copyright (c) 1999-2001 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import string, os, urllib, urlparse, cStringIO, cPickle
from Ft.Lib import ReaderBase

#FIXME: we might want to do some meta-magic to __import__ element modules on demand
from xml.xslt.ApplyTemplatesElement import ApplyTemplatesElement
from xml.xslt.ApplyImportsElement import ApplyImportsElement
from xml.xslt.AttributeElement import AttributeElement
from xml.xslt.AttributeSetElement import AttributeSetElement
from xml.xslt.CallTemplateElement import CallTemplateElement
from xml.xslt.ChooseElement import ChooseElement
from xml.xslt.CopyElement import CopyElement
from xml.xslt.CopyOfElement import CopyOfElement
from xml.xslt.CommentElement import CommentElement
from xml.xslt.ElementElement import ElementElement
from xml.xslt.ForEachElement import ForEachElement
from xml.xslt.IfElement import IfElement
from xml.xslt.LiteralElement import LiteralElement
from xml.xslt.LiteralText import LiteralText
from xml.xslt.MessageElement import MessageElement
from xml.xslt.NumberElement import NumberElement
from xml.xslt.OtherwiseElement import OtherwiseElement
from xml.xslt.ParamElement import ParamElement
from xml.xslt.ProcessingInstructionElement import ProcessingInstructionElement
from xml.xslt.SortElement import SortElement
from xml.xslt import TemplateElement
from xml.xslt.TextElement import TextElement
from xml.xslt.VariableElement import VariableElement
from xml.xslt.ValueOfElement import ValueOfElement
from xml.xslt.WhenElement import WhenElement
from xml.xslt.WithParamElement import WithParamElement

from xml.xslt.OtherXslElement import ImportElement, IncludeElement, DecimalFormatElement, KeyElement, NamespaceAliasElement, OutputElement, PreserveSpaceElement, StripSpaceElement, FallbackElement

from xml.xslt.Stylesheet import StylesheetElement
from xml.xslt import XSL_NAMESPACE, XsltElement
from xml.xslt import XsltException, Error, RegisterExtensionModules

from xml import xslt

from xml.dom import Node
from xml.dom.ext import StripXml, GetAllNs, SplitQName
from xml.dom import implementation, ext, XML_NAMESPACE, XMLNS_NAMESPACE
from Ft.Lib import pDomlette, pDomletteReader

#Artificial limits to simplify the import precedence algorithm
MAX_IMPORTS_PER_LEVEL = 100
IMPORT_SPACING_FACTOR = 10000000


#NOTE: This could be built automatically from a list of functions and their local names
g_mappings = {XSL_NAMESPACE: {
    'apply-templates': ApplyTemplatesElement
    , 'apply-imports': ApplyImportsElement
    , 'attribute': AttributeElement
    , 'attribute-set': AttributeSetElement
    , 'call-template': CallTemplateElement
    , 'choose': ChooseElement
    , 'copy': CopyElement
    , 'copy-of': CopyOfElement
    , 'comment': CommentElement
    , 'element': ElementElement
    , 'for-each': ForEachElement
    , 'if': IfElement
    , 'message': MessageElement
    , 'number': NumberElement
    , 'otherwise': OtherwiseElement
    , 'param': ParamElement
    , 'processing-instruction': ProcessingInstructionElement
    , 'sort': SortElement
    , 'stylesheet': StylesheetElement
    , 'transform': StylesheetElement
    , 'template': TemplateElement.TemplateElement
    , 'text': TextElement
    , 'variable': VariableElement
    , 'value-of': ValueOfElement
    , 'when': WhenElement
    , 'fallback': FallbackElement
    , 'with-param': WithParamElement
    , 'import': ImportElement
    , 'include': IncludeElement
    , 'key': KeyElement
    , 'namespace-alias': NamespaceAliasElement
    , 'output': OutputElement
    , 'preserve-space': PreserveSpaceElement
    , 'strip-space': StripSpaceElement
    }}


def FromDocument(oldDoc, baseUri='', stylesheetReader=None):
    #FIXME: We really shouldn't mutate the given doc, but this is the easiest way to strip whitespace
    if baseUri and baseUri[-1] == '/':
        modBaseUri = baseUri
    else:
        modBaseUri = baseUri + '/'
    imports_allowed = 1
    NormalizeNode(oldDoc)
    extElements = xslt.g_extElements
    source_root = oldDoc.documentElement
    #Set up a new document for the stylesheet nodes
    if source_root.namespaceURI == XSL_NAMESPACE:
        if source_root.localName not in ['stylesheet', 'transform']:
            raise XsltException(Error.STYLESHEET_ILLEGAL_ROOT, source_root.nodeName)
        result_elem_root = 0
    else:
        result_elem_root = 1
    xsl_doc = pDomlette.Document()
    xsl_doc.refUri = hasattr(oldDoc, 'refUri') and oldDoc.refUri or ''
    ext_uris = []
    if result_elem_root:
        imports_allowed = 0
        vattr = source_root.getAttributeNodeNS(XSL_NAMESPACE, 'version')
        if not vattr:
            root_nss = GetAllNs(source_root)
            if filter(lambda x, n=root_nss: n[x] == XSL_NAMESPACE, root_nss.keys()):
                raise XsltException(Error.STYLESHEET_MISSING_VERSION,("Unknown",))
            else:
                raise XsltException(Error.STYLESHEET_MISSING_VERSION_NOTE1,("Unknown",))

        sheet = StylesheetElement(xsl_doc, XSL_NAMESPACE,
                                  'transform', vattr.prefix,
                                  baseUri)
        sheet.setAttributeNS('', 'version', vattr.value)
        tpl = TemplateElement.TemplateElement(xsl_doc, XSL_NAMESPACE, 'template',
                              vattr.prefix, baseUri)

        tpl.setAttributeNS('', 'match', '/')
        sheet.appendChild(tpl)
        sheet.__dict__['extensionNss'] = []
        xsl_doc.appendChild(sheet)
        DomConvert(source_root, tpl, xsl_doc, [], extElements, 0)
    else:
        sheet = StylesheetElement(xsl_doc, XSL_NAMESPACE, source_root.localName,
                                  source_root.prefix, baseUri)
        sty_nss = GetAllNs(source_root)
        for attr in source_root.attributes.values():
            if (attr.namespaceURI, attr.localName) == ('', 'extension-element-prefixes'):
                ext_prefixes = string.splitfields(attr.value)
                for prefix in ext_prefixes:
                    if prefix == '#default': prefix = ''
                    ext_uris.append(sty_nss[prefix])
            if not attr.namespaceURI and attr.localName not in sheet.legalAttrs:
                raise XsltException(Error.XSLT_ILLEGAL_ATTR,
                                    attr.nodeName, source_root)
            sheet.setAttributeNS(attr.namespaceURI, attr.nodeName, attr.value)
        sheet.__dict__['extensionNss'] = ext_uris
        if not sheet.getAttributeNS('', 'version'):
            raise XsltException(Error.STYLESHEET_MISSING_VERSION,("Unknown",))
        xsl_doc.appendChild(sheet)
        for child in source_root.childNodes:
            if getattr(child, 'nodeType', None) == Node.ELEMENT_NODE:
                if (child.namespaceURI, child.localName) != (XSL_NAMESPACE, 'import'):
                    imports_allowed = 0
                elif not imports_allowed:
##                    raise XsltException(Error.ILLEGAL_IMPORT, self.baseUri,
##                                        self._ft_lineNumber,
##                                        self._ft_columnNumber)
                    raise XsltException(Error.ILLEGAL_IMPORT)
            DomConvert(child, sheet, xsl_doc, ext_uris, extElements, 0)
    #Handle XSL imports and includes
    impincs = filter(lambda x: x.nodeType == Node.ELEMENT_NODE and x.namespaceURI == XSL_NAMESPACE and x.localName in ['include', 'import'], sheet.childNodes)
    import_count = 0
    import_ix = 0
    for inc in impincs:
        href = inc.getAttributeNS("", "href")
        if stylesheetReader is None:
            stylesheetReader = StylesheetReader()
        if inc.localName == 'import':
            #Import index increases as the inverse of import precedence
            #FIXME: replace this convoluted logic (with its artificial
            #limitations built in) with a 2-pass approach
            import_ix = (MAX_IMPORTS_PER_LEVEL - import_count) * \
                        IMPORT_SPACING_FACTOR
            import_count = import_count + 1
            if import_count > MAX_IMPORTS_PER_LEVEL:
                raise XsltException(Error.INTERNAL_ERROR)
        docfrag = stylesheetReader.fromUri(href, baseUri=baseUri,
                                           ownerDoc=xsl_doc,
                                           importIndex=import_ix,
                                           importDepth=import_count+1)
        sty = docfrag.firstChild
        included_nss = GetAllNs(sty)
        if inc.localName == 'import':
            sty.importingSty = sheet
            sheet._imports.append(sty)
        for child in sty.childNodes[:]:
            if child.nodeType != pDomlette.Node.ELEMENT_NODE:
                continue
            if inc.localName == 'import':
                if child.nodeType == Node.ELEMENT_NODE and (child.namespaceURI, child.localName) == (XSL_NAMESPACE, 'template'):
                    sheet._importedTemplates.append(child)
            sheet.insertBefore(child, inc)
            #migrate old nss from stylesheet directly to new child
            for prefix in included_nss.keys():
                if prefix:
                    child.setAttributeNS(XMLNS_NAMESPACE, 'xmlns:'+prefix,
                                                 included_nss[prefix])
                else:
                    child.setAttributeNS(XMLNS_NAMESPACE, 'xmlns',
                                         included_nss[prefix])

        sheet.removeChild(inc)
        pDomlette.ReleaseNode(inc)
        #sty.reclaim()
    try:
        sheet.setup()
    except:
        pDomlette.ReleaseNode(sheet.ownerDocument)
        raise
    return sheet


def DomConvert(node, xslParent, xslDoc, extUris, extElements, preserveSpace):
    if node.nodeType == Node.ELEMENT_NODE:
        mapping = g_mappings.get(node.namespaceURI, None)
        if mapping:
            if not mapping.has_key(node.localName):
                raise XsltException(Error.XSLT_ILLEGAL_ELEMENT, node.localName)
            xsl_class = mapping[node.localName]
            xsl_instance = xsl_class(xslDoc, baseUri=xslParent.baseUri)
            for attr in node.attributes.values():
                if not attr.namespaceURI and attr.localName not in xsl_instance.legalAttrs:
                    raise XsltException(Error.XSLT_ILLEGAL_ATTR,
                                        attr.nodeName, xsl_instance.nodeName)
                xsl_instance.setAttributeNS(attr.namespaceURI, attr.nodeName,
                                            attr.value)
            xslParent.appendChild(xsl_instance)
        elif node.namespaceURI in extUris:
            name = (node.namespaceURI, node.localName)
            if name in extElements.keys():
                ext_class = extElements[name]
            else:
                #Default XsltElement behavior effects fallback
                ext_class = XsltElement
            xsl_instance = ext_class(xslDoc, node.namespaceURI,
                                     node.localName, node.prefix,
                                     xslParent.baseUri)
            for attr in node.attributes.values():
                if (attr.namespaceURI, attr.localName) == (XSL_NAMESPACE, 'extension-element-prefixes'):
                    ext_prefixes = string.splitfields(attr.value)
                    for prefix in ext_prefixes:
                        if prefix == '#default': prefix = ''
                        extUris.append(node_nss[prefix])
                xsl_instance.setAttributeNS(attr.namespaceURI, attr.nodeName,
                                            attr.value)
            xslParent.appendChild(xsl_instance)
        else:
            xsl_instance = LiteralElement(xslDoc, node.namespaceURI,
                                          node.localName, node.prefix,
                                          xslParent.baseUri)
            node_nss = GetAllNs(node)
            for attr in node.attributes.values():
                if (attr.namespaceURI, attr.localName) == (XSL_NAMESPACE, 'extension-element-prefixes'):
                    ext_prefixes = string.splitfields(attr.value)
                    for prefix in ext_prefixes:
                        if prefix == '#default': prefix = ''
                        extUris.append(node_nss[prefix])
                xsl_instance.setAttributeNS(attr.namespaceURI,
                                            attr.name,
                                            attr.value
                                            )
            xslParent.appendChild(xsl_instance)
        if node.localName == 'text' and node.namespaceURI == XSL_NAMESPACE:
            ps = 1
        elif node.getAttributeNS(XML_NAMESPACE, 'space') == 'preserve':
            ps = 1
        else:
            ps = preserveSpace
        for child in node.childNodes:
            DomConvert(child, xsl_instance, xslDoc, extUris, extElements, ps)
    elif node.nodeType == Node.TEXT_NODE:
        if preserveSpace or string.strip(node.data):
            xsl_instance = LiteralText(xslDoc, node.data)
            xslParent.appendChild(xsl_instance)
    return



### Domlette Parser Interface ###

from xml.parsers import expat

class Handler(pDomletteReader.Handler):
    def __init__(self, resolveEntity=None, processIncludes=1,
                 visitedHrefs=None, indexStringValues=1, force8Bit=0,
                 uriResolver=None, extModules=None,
                 stylesheetIncPaths=None):
        pDomletteReader.Handler.__init__(
            self, resolveEntity, processIncludes, visitedHrefs,
            indexStringValues, force8Bit, uriResolver
            )
        self._stylesheetIncPaths = stylesheetIncPaths or []
        self._extModules = extModules or []
        self._extElements = xslt.g_extElements
        if extModules:
            RegisterExtensionModules(extModules, self._extElements)
        self._ssheetUri = ''
        return

    def initState(self, ownerDoc, refUri, importIndex=0, importDepth=0):
        pDomletteReader.Handler.initState(self, ownerDoc, refUri=refUri)
        ReaderBase.PreserveSpaceMixin._initState(self, 0)
        self._extUris = []
        self._extUriStack = []
        self._firstElement = 1
        self._importsAllowed = 1
        self._importIndex = importIndex
        self._importDepth = importDepth
        self._importCount = 0
        if not self._ssheetUri:
            self._ssheetUri = refUri
        return

    def processingInstruction(self, target, data):
        self._completeTextNode()
        return

    def comment(self, data):
        self._completeTextNode()
        return

    def startElement(self, name, attribs):
        name = self._checkString(name)
        attribs = self._checkDict(attribs)
        (name, qname, nsattribs) = self._handleStartElementNss(name, attribs)
        include_depth = self._handleIncludes(name, qname, nsattribs)
        if include_depth: return
        (nsuri, local, prefix) = ReaderBase.HandlerBase._startElementPrep(
            self, name, qname
            )
        mapping = g_mappings.get(nsuri, None)
        del_extu = []
        if mapping:
            if not mapping.has_key(local):
                if self._firstElement:
                    raise XsltException(Error.STYLESHEET_ILLEGAL_ROOT, name)
                else:
                    raise XsltException(Error.XSLT_ILLEGAL_ELEMENT, local)
            xsl_class = mapping[local]
            if xsl_class != ImportElement and not self._firstElement:
                self._importsAllowed = 0
            if xsl_class in [IncludeElement, ImportElement]:
                import_ix = self._importIndex
                if xsl_class == ImportElement:
                    if not self._importsAllowed:
##                      raise XsltException(Error.ILLEGAL_IMPORT, self.baseUri,
##                                          self._ft_lineNumber,
##                                          self._ft_columnNumber)
                        raise XsltException(Error.ILLEGAL_IMPORT)
                    #Import index increases as the inverse of import precedence
                    #FIXME: replace this convoluted logic (with its artificial
                    #limitations built in) with a 2-pass approach
                    import_ix = self._importIndex + \
                                (MAX_IMPORTS_PER_LEVEL - self._importCount) * \
                                IMPORT_SPACING_FACTOR / (self._importDepth + 1)
                    self._importCount = self._importCount + 1
                    if self._importCount > MAX_IMPORTS_PER_LEVEL:
                        raise XsltException(Error.INTERNAL_ERROR)
                #Can the included sheet have literal result element as root?
                inc = None
                href = nsattribs[('', 'href')]
                reader = self.reader.clone()
                try:
                    inc = reader.fromUri(
                        href, baseUri=self._ssheetUri, ownerDoc=self._ownerDoc,
                        importIndex=import_ix, importDepth=self._importDepth+1)
                    #print "Include or import attempt", self._ssheetUri, nsattribs[('', 'href')]
                except IOError, ioe:
                    for path in self._stylesheetIncPaths:
                        try:
                            inc = reader.fromUri(
                                href, baseUri=path,
                                ownerDoc=self._ownerDoc
                                )
                            break
                        except IOError:
                            pass
                    if not inc:
                        raise XsltException(Error.INCLUDE_NOT_FOUND, href, self._ssheetUri)

                sty = inc.firstChild
                included_nss = GetAllNs(sty)
                sheet = self._nodeStack[1]
                if xsl_class == ImportElement:
                    sty.importingSty = sheet
                    sheet._imports.append(sty)
                for child in sty.childNodes[:]:
                    if xsl_class == ImportElement:
                        if child.nodeType == Node.ELEMENT_NODE and (child.namespaceURI, child.localName) == (XSL_NAMESPACE, 'template'):
                            sheet._importedTemplates.append(child)
                    self._nodeStack[-1].appendChild(child)
                    #migrate old nss from stylesheet directly to new child
                    for prefix in included_nss.keys():
                        if prefix:
                            child.setAttributeNS(XMLNS_NAMESPACE,
                                                 'xmlns:'+prefix,
                                                 included_nss[prefix])
                        else:
                            child.setAttributeNS(XMLNS_NAMESPACE, 'xmlns',
                                                 included_nss[prefix])
                self._nodeStack.append(None)
                pDomlette.ReleaseNode(inc)
                return
            else:
                xsl_instance = xsl_class(self._ownerDoc,
                                         baseUri=self._ssheetUri)
                xsl_instance.importIndex=self._importIndex

            for aqname in nsattribs.getQNames():
                (ansuri, alocal) = nsattribs.getNameByQName(aqname)

                value = nsattribs.getValueByQName(aqname)
                if ansuri != XMLNS_NAMESPACE and xsl_class == StylesheetElement:
                    self._handleExtUris(ansuri, alocal, value, '',
                                        del_extu,xsl_instance)
                elif not ansuri and alocal not in xsl_instance.__class__.legalAttrs:
                    raise XsltException(Error.XSLT_ILLEGAL_ATTR,
                                        aqname, xsl_instance.nodeName)

                xsl_instance.setAttributeNS(ansuri, aqname, value)
        else:
            if nsuri in self._extUris and self._extElements:
                #Default XsltElement behavior effects fallback
                ext_class = self._extElements.get((nsuri, local), XsltElement)
                xsl_instance = ext_class(self._ownerDoc, nsuri, local,
                                         prefix, self._ssheetUri)
            else:
                xsl_instance = LiteralElement(self._ownerDoc, nsuri, local,
                                              prefix, self._ssheetUri)
            for aqname in nsattribs.getQNames():
                (ansuri, alocal) = nsattribs.getNameByQName(aqname)
                value = nsattribs.getValueByQName(aqname)
                if ansuri != XMLNS_NAMESPACE:
                    self._handleExtUris(ansuri, alocal, value, '',
                                        del_extu, xsl_instance)
                    if hasattr(xsl_instance.__class__, 'legalAttrs'):
                        if not ansuri and alocal not in xsl_instance.__class__.legalAttrs:
                            raise XsltException(Error.XSLT_ILLEGAL_ATTR,
                                                alocal, xsl_instance.nodeName)
                xsl_instance.setAttributeNS(ansuri, aqname, value)
        self._extUriStack.append(del_extu)
        #Encapsulate into PreserveSpaceMixin
        if (xsl_instance.namespaceURI, xsl_instance.localName) == (XSL_NAMESPACE, 'text') or xsl_instance.getAttributeNS(XML_NAMESPACE, 'space') == 'preserve':
            self._preserveStateStack.append(1)
        elif xsl_instance.getAttributeNS(XML_NAMESPACE, 'space') == 'default':
            self._preserveStateStack.append(0)
        else:
            self._preserveStateStack.append(self._preserveStateStack[-1])
        if self._firstElement:
            self._initializeSheet(xsl_instance)
        self._nodeStack.append(xsl_instance)

        #Add linenumber information
        xsl_instance._ft_lineNumber = self.parser.ErrorLineNumber
        xsl_instance._ft_columnNumber = self.parser.ErrorColumnNumber

        return

    def endElement(self, name):
        if not self._nodeStack[-1]:
            del self._nodeStack[-1]
            return
        initial_include_depth = self._checkIncludeDepth()
        pDomletteReader.Handler.endElement(self, name)
        if initial_include_depth: return
        del_extu = self._extUriStack[-1]
        del self._extUriStack[-1]
        for uri in del_extu:
            self._extUris.remove(uri)
        return

    def characters(self, data):
        data = self._checkString(data)
        if self._checkIncludeDepth(): return
        self._currText = self._currText + data
        return

    def _completeTextNode(self):
        #Note some parsers don't report ignorable white space properly
        if self._currText and len(self._nodeStack) and self._nodeStack[-1].nodeType != Node.DOCUMENT_NODE:
            if self._peekPreserveStateStack() or string.strip(self._currText):
                new_text = LiteralText(self._ownerDoc, self._currText)
                self._nodeStack[-1].appendChild(new_text)
        self._currText = ''
        return

    def _initializeSheet(self, rootNode):
        if rootNode.namespaceURI == XSL_NAMESPACE:
            if rootNode.localName in ['stylesheet', 'transform']:
                if not rootNode.getAttributeNS('', 'version'):
                    raise XsltException(Error.STYLESHEET_MISSING_VERSION,(self._ssheetUri,))
                for attr in rootNode.attributes.values():
                    if not attr.namespaceURI and attr.localName not in rootNode.legalAttrs:
                        raise XsltException(Error.XSLT_ILLEGAL_ATTR,
                                            attr.nodeName, rootNode)
                #rootNode.__dict__['extensionNss'] = []
            else:
                raise XsltException(Error.STYLESHEET_ILLEGAL_ROOT, rootNode.nodeName)
        else:
            vattr = rootNode.getAttributeNodeNS(XSL_NAMESPACE, 'version')
            if not vattr:
                root_nss = GetAllNs(rootNode)
                if filter(lambda x, n=root_nss: n[x] == XSL_NAMESPACE, root_nss.keys()):
                    raise XsltException(Error.STYLESHEET_MISSING_VERSION,(self._ssheetUri,))
                else:
                    raise XsltException(Error.STYLESHEET_MISSING_VERSION_NOTE1,(self._ssheetUri,))
            sheet = StylesheetElement(self._ownerDoc, XSL_NAMESPACE,
                                      'transform', vattr.prefix,
                                      self._ssheetUri)
            sheet.setAttributeNS('', 'version', vattr.value)
            tpl = TemplateElement.TemplateElement(self._ownerDoc, XSL_NAMESPACE, 'template',
                                  vattr.prefix, self._ssheetUri)
            tpl.setAttributeNS('', 'match', '/')
            sheet.appendChild(tpl)
            sheet.__dict__['extensionNss'] = []
            self._nodeStack[-1].appendChild(sheet)
            # Ensure the literal element is a child of the template
            # endElement appends to the end of the nodeStack
            self._nodeStack.append(tpl)
        self._firstElement = 0
        return

    def _handleExtUris(self, ns, local, value, extUri, delExtu, sheet):
        if (ns, local) == (extUri, 'extension-element-prefixes'):
            ext_prefixes = string.splitfields(value)
            for prefix in ext_prefixes:
                if prefix == '#default': prefix = ''
                uri = self._namespaces[-1].get(prefix, '')
                if uri not in self._extUris:
                    delExtu.append(uri)
                    self._extUris.append(uri)
                if sheet and not uri in sheet.extensionNss:
                    sheet.extensionNss.append(uri)
        return
    
    
class StylesheetReader(pDomletteReader.PyExpatReader):
    def __init__(self, resolveEntity=None, processIncludes=1,
                 visitedHrefs=None, force8Bit=0, extModules=None,
                 stylesheetIncPaths=None):
        pDomlette.PyExpatReader.__init__(self, resolveEntity,
                                         processIncludes, visitedHrefs,
                                         force8Bit)
        self._stylesheetIncPaths = stylesheetIncPaths or []
        self._extModules = extModules or []
        self._ssheetUri = ''
        self._importIndex = 0
        self._importDepth = 0
        return

    def fromString(self, st, refUri='', ownerDoc=None, stripElements=None,
                   importIndex=0, importDepth=0):
        #self._importIndex = importIndex
        #self._importDepth = importDepth
        self._ssheetUri = refUri
        return pDomletteReader.PyExpatReader.fromString(self,st, refUri='', ownerDoc=None, stripElements=None)

    def fromUri(self, uri, baseUri='', ownerDoc=None, stripElements=None,
                importIndex=0, importDepth=0):
        self._importIndex = importIndex
        self._importDepth = importDepth
        save_sheet_uri = self._ssheetUri
        self._ssheetUri = urllib.basejoin(baseUri, uri)
        result = ReaderBase.DomletteReader.fromUri(self, uri, baseUri,
                                                   ownerDoc, stripElements)
        self._ssheetUri = save_sheet_uri
        return result

    def fromStream(self, stream, refUri='', ownerDoc=None, stripElements=None,
                importIndex=0, importDepth=0):
        #self._importIndex = importIndex
        #self._importDepth = importDepth
        if not self._ssheetUri:
            self._ssheetUri = refUri

        if not xslt.g_registered:
            xslt.Register()
        self.initParser()
        self.handler.initState(ownerDoc, refUri, importIndex=self._importIndex,
                               importDepth=self._importDepth)
        self.parser.SetBase(refUri)
        try:
            success = self.parser.ParseFile(stream)
        except XsltException:
            raise
        except Exception, e:
            for s in self.handler._nodeStack:
                self.releaseNode(s)
            if self.parser.ErrorCode:
                self.releaseNode(self.handler._rootNode)
                self.releaseNode(self.handler._ownerDoc)
                ln = self.parser.ErrorLineNumber
                cn = self.parser.ErrorColumnNumber
                ec = self.parser.ErrorCode
                self.killParser()
                raise XsltException(Error.STYLESHEET_PARSE_ERROR, refUri,
                                    ln, cn, expat.ErrorString(ec))
            else:
                raise
        self._ssheetUri = ''
        self.handler._completeTextNode()

        root = self.handler._rootNode or self.handler._ownerDoc
        self.killParser()
        if root.nodeType == Node.DOCUMENT_NODE:
            sheet = root.documentElement
            try:
                sheet.setup()
            except:
                sheet.reclaim()
                self.releaseNode(root)
                raise
        else:
            sheet = None
        rt = sheet or root
        return rt

    def initParser(self):
        HandlerClass = Handler
        self.handler = HandlerClass(
            resolveEntity=self._resolveEntity,
            processIncludes=self._processIncludes,
            visitedHrefs=self._visitedHrefs,
            force8Bit=self._force8Bit,
            extModules=self._extModules,
            stylesheetIncPaths=self._stylesheetIncPaths
            )
        self.handler.reader = self
        self.parser=expat.ParserCreate()
        self.parser.StartElementHandler = self.handler.startElement
        self.parser.EndElementHandler = self.handler.endElement
        self.parser.CharacterDataHandler = self.handler.characters
        self.parser.ProcessingInstructionHandler = self.handler.processingInstruction
        self.parser.CommentHandler = self.handler.comment
        self.parser.ExternalEntityRefHandler = self.handler.entityRef
        self.handler.parser = self.parser
        return

    def killParser(self):
        self.handler.reader = None
        pDomlette.PyExpatReader.killParser(self)
        return


def CreateInstantStylesheet(sheet):
    return pDomlette.PickleDocument(sheet.ownerDocument)

def FromInstant(dump, forceBaseUri=None):
    return UnpickleDocument(dump, forceBaseUri).documentElement


##FIXME
##This unpickling code is Basically a transplant from pDomlette
##with the addition of baseUri overriding in unpickling.  It is
##somewhat experimental and not intended to diverge from the pDomlette
##code, and if it ever does so, it should be nixed, and some
##genericity found between this the pDomlette code

import cPickle
def UnpickleDocument(pickledXml, forceBaseUri=None):
    doc = pDomlette.Document()
    if forceBaseUri is not None:
        doc.refUri = forceBaseUri
    stream = cStringIO.StringIO(pickledXml)
    unpickler = cPickle.Unpickler(stream)
    _UnpickleChildren(unpickler, doc, forceBaseUri)
    return doc

def UnpickleNode(pickledXml, doc=None, forceBaseUri=None):
    doc = doc or Document()
    stream = cStringIO.StringIO(pickledXml)
    unpickler = cPickle.Unpickler(stream)
    topLevelNode = unpickler.load()
    if forceBaseUri is None and hasattr(topLevelNode, 'baseUri'):
        topLevelNode.refUri = forceRefUri
    doc.appendChild(topLevelNode)
    if topLevelNode.attributes:
        for attr in topLevelNode.attributes.values():
            attr.ownerDocument = topLevelNode.ownerDocument
    _UnpickleChildren(unpickler, topLevelNode, forceBaseUri)
    return topLevelNode

## Helper function for unpickling ##


def _UnpickleChildren(unpickler, node, forceBaseUri=None):
    children = unpickler.load()
    while children:
        child = unpickler.load()
        if forceBaseUri is None and hasattr(child, 'baseUri'):
            child.baseUri = forceBaseUri
        node.appendChild(child)
        if child.nodeType == Node.ELEMENT_NODE:
            for attr in child.attributes:
                attr.ownerDocument = child.ownerDocument
        _UnpickleChildren(unpickler, child, forceBaseUri)
        children = children - 1


import xml.xpath.Util
NormalizeNode = xml.xpath.Util.NormalizeNode
