/*
    plconv.m

    A simple command-line tool for converting between different
    property list formats.

    Copyright (C) 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import <Foundation/Foundation.h>

static char * const propertyListFormatDescriptions[] = {
  "GNUstep format",
  "GNUstep binary format",
  "OpenStep format",
  "XML format, version 1.0",
  "Apple binary format, version 1.0"
};

static inline void
PrintHelp (const char *progname)
{
  printf ("plconv version 0.1, written by Saso Kiselkov, released under the GNU GPL\n"
          "Usage: %s [-q] [-h|-help] [-g|-gb|-o|-x|-ab]\n"
          "This program reads a property list from standard input\n"
          "and outputs it to standard output again. The output is,\n"
          "unless any option overrides it, in the same format as the\n"
          "input was.\n"
          "Further options are:\n"
          " -q - when specified, the program doesn't print to stderr\n"
          "      the format of the input plist detected\n"
          " -h or -help - prints out this help screen\n"
          " -g - sets the output format to GNUstep plain-text format\n"
          " -gb - sets the output format to GNUstep binary format\n"
          " -o - sets the output format to OpenStep plain-text format\n"
          " -x - sets the output format to XML format v1.0\n"
          " -ab - sets the output format to Apple binary format v1.0\n",
          progname);
}

int
main (int argc, const char *argv[])
{
  CREATE_AUTORELEASE_POOL (pool);

  NSUserDefaults * df = [NSUserDefaults standardUserDefaults];
  NSFileHandle * stdinHandle, * stdoutHandle;
  NSData * inData, * outData;
  id propList;
  NSPropertyListFormat inFormat, outFormat;
  NSString * error;

  // check for a -h or -help argument
  if ([df objectForKey: @"h"] || [df objectForKey: @"help"])
    {
      PrintHelp (argv[0]);

      return 0;
    }

  stdinHandle = [NSFileHandle fileHandleWithStandardInput];
  stdoutHandle = [NSFileHandle fileHandleWithStandardOutput];

  inData = [stdinHandle readDataToEndOfFile];
  propList = [NSPropertyListSerialization
    propertyListFromData: inData
        mutabilityOption: NSPropertyListImmutable
                  format: &inFormat
        errorDescription: &error];
  if (propList == nil)
    {
      fprintf (stderr, "Error reading property list: %s\n",
        [error cString]);

      return 1;
    }

  if (![df objectForKey: @"q"])
    {
      fprintf (stderr, "Input property list format: %s\n",
        propertyListFormatDescriptions[inFormat]);
    }

  if ([df objectForKey: @"g"])
    {
      outFormat = NSPropertyListGNUstepFormat;
    }
  else if ([df objectForKey: @"gb"])
    {
      outFormat = NSPropertyListGNUstepBinaryFormat;
    }
  else if ([df objectForKey: @"o"])
    {
      outFormat = NSPropertyListOpenStepFormat;
    }
  else if ([df objectForKey: @"x"])
    {
      outFormat = NSPropertyListXMLFormat_v1_0;
    }
  else if ([df objectForKey: @"ab"])
    {
      outFormat = NSPropertyListBinaryFormat_v1_0;
    }
  else
    {
      outFormat = inFormat;
    }

  outData = [NSPropertyListSerialization
    dataFromPropertyList: propList
                  format: outFormat
        errorDescription: &error];
  if (outData == nil)
    {
      fprintf (stderr, "Error building output property list: %s\n",
        [error cString]);

      return 1;
    }

  [stdoutHandle writeData: outData];

  DESTROY (pool);

  return 0;
}
