/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2005 SUSE Linux Products GmbH                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

/** @defgroup examples_testclient Example client or testclient
 * @include testclient.c
 */

/*

Implementing a powersave client
-------------------------------

Useful information:

- Dbus interface documentation can be found at
  /usr/share/doc/packages/powersave/powersave_manual.html#DBus

- a sample testclient can be found in the cvs repository in
  testsuite/testclient/testclient.c. Or better, look at the kpowersave
  implementation.

In general, a powersave client has to perform the following tasks:

* Set up graphical client environment

* Set up dbus connection to system bus

* Powersave schemes
  - provide a list of schemes to the user
    --> get list of schemes from the powersave daemon via dbus
    --> set specific scheme on user interaction via dbus

* CPU frequency policies (list of policies can not be queried via dbus)
  Show available cpu policies
       - dynamic
       - performance
       - powersave
       --> set policies on user interaction via dbus

* Battery charge indicator
  - get current battery charge level via dbus
  - get current remaining time until fully loaded/ until empty via dbus
  --> show those values

* Suspend
  - check if suspend to disk is enabled
    --> give possibility to suspend the system to disk
  - check if suspend to ram is enabled
    --> give possibility to suspend the system to ram
  (- check if standby is enabled
    --> give possibility to put system into standby)

* Watch the dbus connection for the various PowersaveEvents and update
  obove mentioned values if needed

* Permanent daemon connection
  - setup permanent daemon connection with library function and listen on
    this connection
  - listen for notification events on dbus and show those notifications
  - listen for screenlock/screensaver events on dbus and perform
    appropriate action
  - listen for progress information on dbus and show a progress bar with
    the given message

* YaST2 configuration module
  - add possibility to opening the YaST2 power management configuration
    module

*/

#ifndef DBUS_API_SUBJECT_TO_CHANGE
#define DBUS_API_SUBJECT_TO_CHANGE
#endif

#include "powerlib.h"
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>
#include <liblazy.h>

static int CONNECT = 0;
static int CONNECT_AND_CLOSE = 0;

void usage(void);
DBusHandlerResult filter_function(DBusConnection *, DBusMessage *, void *);
void checkArgs(int argc, char **argv);

static GMainLoop *gmain;

void checkArgs(int argc, char **argv)
{
	int option_index = 0;

	struct option opts[] = {
		{"connect", 0, NULL, 'c'},
		{"connect-and-close", 0, NULL, 'C'},
		{"help", 0, NULL, 'h'},
		{NULL, 0, NULL, 0},
	};
	
	while (1) {
		int i = getopt_long(argc, argv, "cCh", opts, &option_index);
		if (i == -1) {
			break;
		}
		
		switch (i) {
		case 'c':
			CONNECT = 1;
			break;
		case 'C':
			CONNECT = 1;
			CONNECT_AND_CLOSE = 1;
			break;
	        case 'h':
			usage();
			exit(EXIT_SUCCESS);		
			break;
		default:
			exit(EXIT_SUCCESS);
		}
	
	}

}


int main(int argc, char *argv[])
{
	DBusError error;
	DBusConnection *connection;

	checkArgs(argc, argv);

	/* set up a new main loop with glib */
	gmain = g_main_loop_new(NULL, FALSE);

	dbus_error_init(&error);

	/* get connection from dbus */
	connection = dbus_bus_get(DBUS_BUS_SYSTEM, &error);

	if (dbus_error_is_set(&error)) {
		printf("Error: %s\n", error.message);
		dbus_error_free(&error);
		exit(EXIT_FAILURE);
	}

	/* set up the dbus connection with the glib main loop */
	dbus_connection_setup_with_g_main(connection, NULL);

	/* add the filter function which should be executed on events on the bus */
	if (!dbus_connection_add_filter(connection, filter_function, gmain, NULL)) {
		printf("Error: Not enough memory to add filter to dbus connection\n");
		exit(EXIT_FAILURE);
	}

	/* add a match rule to catch all signals going through th bus with
	 * powersave manager interface */
	dbus_bus_add_match(connection,
			   "type='signal',"
			   "interface='com.novell.powersave.manager'," "path='/com/novell/powersave',", NULL);

	/* we immediately shut down the connection if CONNECT_AND_CLOSE is set */
	if (CONNECT_AND_CLOSE) {
		printf("Closing connection because -C is given\n");
		exit(EXIT_SUCCESS);
	}

	/* run the main loop */
	g_main_loop_run(gmain);

	exit(EXIT_SUCCESS);
}

DBusHandlerResult filter_function(DBusConnection * connection, DBusMessage * message, void *data)
{

	DBusError error;
	/* the argument received */
	char *value;
	const char *signal;

	dbus_error_init(&error);

	if (dbus_message_get_type(message) != DBUS_MESSAGE_TYPE_SIGNAL) {
		return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	}

	/* get the name of the signal */
	signal = dbus_message_get_member(message);

	if (liblazy_dbus_message_get_basic_arg(message, DBUS_TYPE_STRING, &value, 0) < 0) {
		printf(" Warning: Received signal %s, but no string argument\n", error.message);
		return DBUS_HANDLER_RESULT_HANDLED;
	}

	/* our name is... */
	if (!strcmp(signal, "NameAcquired")) {
		printf("DBus connection acquired: %s\n" "Waiting for signals from powersave daemon...\n\n", value);
		return DBUS_HANDLER_RESULT_HANDLED;
	}
	/* acpi event received */
	else if (!strcmp(signal, "AcpiEvent")) {
		printf("Received acpi event. Value: %s\n", value);
	}
	/* powersave event received */
	else if (!strcmp(signal, "PowersaveEvent")) {
		printf("Received powersave event. Value: %s\n", value);

		/* daemon shut down, we do the same... */
		if (!strcmp(value, "daemon.terminate")) {
			g_main_loop_quit(gmain);
		}
	}
	/* progress information received */
	else if (!strcmp(signal, "Progress")) {
		printf("Received progress information. Value: %s\n", value);
	}
	/* notification received */
	else if (!strcmp(signal, "Notification")) {
		printf("Received notification. Value: %s\n", value);
	}
	/* screenlock request received */
	else if (!strcmp(signal, "Screenlock")) {
		printf("Received screenlock request. Value: %s\n", value);
	} else {
		printf("Received unknown signal %s. Value: %s\n", signal, value);
	}

	return 0;
}

void usage(void)
{
	printf("Test client for connection to powersaved and\n"
	       "getting informed about acpi events, powersave events,\n"
	       "progress information or notifications.\n"
	       "\n"
	       "Usage: ./testclient {-c|-C} [-i capabilities]\n"
	       "\n"
	       "This client can connect to the powersave daemon over DBus.\n\n"
	       "Capabilities are a sum of CAPABILITIES from powersave_dbus.h\n"
	       "All capabilities can be added up. They are simple integers.\n\n"
	       "Example 1: ./testclient -c\n"
	       "Means: Connect to the daemon with default capabilities (all)\n\n"
	       "Example 2: ./testclient -C\n"
	       "Means: Connect to the daemon and shut down immediately.\n"
	       "\tCan be used to test the connection and evaluate the return code.\n\n"
	       "Example 3: ./testclient -i 1\n"
	       "Means: Connect to the daemon with CAPABILITIES '1' (CAPABILITY NOTIFICTATIONS\n\n");
}
