/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include <unistd.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/fcntl.h>
#include <sys/types.h>
#include <errno.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <stdio.h>

#include "powerlib_local.h"
#include "powerlib.h"

#define ACPI_PROCESSOR_DIR "/proc/acpi/processor"
#define PROC_STAT_FILE "/proc/stat"
#define PROC_CPUINFO_FILE "/proc/cpuinfo"
#define SYSFS_CPU_PATH "/sys/devices/system/cpu"

static unsigned long *ps_last_total_time = NULL, *ps_last_working_time = NULL;
static int *ps_cpuload = NULL;
static int cpuCount = -1;

int getThrottlingInfo(int *num_states, int *current_state)
{
	char value[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	char file_path[MAX_FILE_PATH + 1] = "";
	char file[MAX_FILE_PATH + 1] = "";
	FILE *rdFile;
	int ret;

	*num_states = 0;
	*current_state = 0;
	if (checkACPI() != ACPI)
		return NO_ACPI_ERROR;

	if (access(ACPI_PROCESSOR_DIR, F_OK)) {
		return NO_MODULE_ERROR;
	}
	/* expect that for every CPU the same number of throttling states exist */
	getDirEntry(0, file, sizeof(file), ACPI_PROCESSOR_DIR);
	ret = snprintf(file_path, MAX_FILE_PATH, "%s/%s/throttling", ACPI_PROCESSOR_DIR, file);
	if (ret >= MAX_FILE_PATH) {
		pDebug(DBG_DIAG, "path too long: %s/%s/throttling", ACPI_PROCESSOR_DIR, file);
		return -1;
	}
	rdFile = fopen(file_path, "r");
	if (!rdFile) {
		return -1;
	}
	/* if we get out here, it is the <not supported> string */
	if (getColonValue(rdFile, value, sizeof(value), temp, sizeof(temp)) < 0) {
		fclose(rdFile);
		return -1;
	} else {
		*num_states = (int)strtol(value, NULL, 10);
	}
	if (getColonValue(rdFile, value, sizeof(value), temp, sizeof(temp)) < 0) {
		pDebug(DBG_WARN, "could not parse '%s'. Please report.", file_path);
		ret = -1;
	} else {
		*current_state = (int)strtol(value + 1, NULL, 10);
		/* test for error
		   pDebug(DBG_DIAG, "Could not convert throttling value to integer!: %s\n", strerror(errno));
		   fclose (rdFile);
		   return -1;
		 */
		ret = 1;
	}
	fclose(rdFile);
	return ret;
}

int getThrottlingInfoCPU(const int cpu, int *states, int *current)
{

	/*      char cpu_info_value[MAX_LINE_SIZE + 1];
	 */

	char value[MAX_LINE_SIZE + 1] = "";
	char temp[MAX_LINE_SIZE + 1] = "";
	char file_path[MAX_FILE_PATH + 1] = "";
	char file[MAX_FILE_PATH + 1] = "";
	FILE *rdFile;
	int ret;

	*states = 0;
	*current = 0;
	if (checkACPI() != ACPI)
		return NO_ACPI_ERROR;

	if (access(ACPI_PROCESSOR_DIR, F_OK)) {
		return NO_MODULE_ERROR;
	}

	/* no hyperthreading check yet */
	if (getDirEntry(cpu, file, sizeof(file), ACPI_PROCESSOR_DIR) < 0)
		return -2;

	ret = snprintf(file_path, MAX_FILE_PATH, "%s/%s/throttling", ACPI_PROCESSOR_DIR, file);
	if (ret >= MAX_FILE_PATH) {
		pDebug(DBG_DIAG, "path too long: %s/%s/throttling", ACPI_PROCESSOR_DIR, file);
		ret = -3;
	}
	rdFile = fopen(file_path, "r");
	if (!rdFile) {
		pDebug(DBG_DIAG, "Could not open '%s': '%s'", file_path, strerror(errno));
		ret = -3;
	}
	if (getColonValue(rdFile, value, sizeof(value), temp, sizeof(temp)) < 0) {
		/* this is normal if there is the string <not supported> at this point */
		pDebug(DBG_DIAG, "Throttling not supported");
		fclose(rdFile);
		return -3;
	} else {
		pDebug(DBG_INFO, "Throttling state: %s", value);
		*states = (int)strtol(value, NULL, 10);
	}
	if (getColonValue(rdFile, value, sizeof(value), temp, sizeof(temp)) < 0)
		ret = -3;
	else {
		pDebug(DBG_INFO, "Current Throttling state: %s", value);
		*current = (int)strtol(value + 1, NULL, 10);
		ret = 1;
	}
	fclose(rdFile);
	return ret;
}

int setThrottlingState(const int state)
{
	int x;

	if (checkACPI() != ACPI)
		return NO_ACPI_ERROR;
	for (x = 0; setThrottlingStateCPU(x, state) > 0; x++) ;
	/* I do not check whether cpus above 0 (first cpu is zero) return error */
	if (x == 0)
		return NO_MODULE_ERROR;
	else
		return 1;
}

int setThrottlingStateCPU(const int cpu, const int state)
{
	char file[MAX_FILE_PATH + 1] = "";
	char file_path[MAX_FILE_PATH + 1] = "";
	int ret = 0;

	if (checkACPI() != ACPI)
		return NO_ACPI_ERROR;
	if (getDirEntry(cpu, file, sizeof(file), ACPI_PROCESSOR_DIR) < 0)
		return NO_MODULE_ERROR;

	ret = snprintf(file_path, MAX_FILE_PATH, "%s/%s/throttling", ACPI_PROCESSOR_DIR, file);
	if (ret >= MAX_FILE_PATH) {
		pDebug(DBG_DIAG, "path too long: %s/%s/throttling", ACPI_PROCESSOR_DIR, file);
		return -1;
	}
	if (_write_line(file_path, "%d", state) < 0) {
		pDebug(DBG_DIAG, "setting throttling state %d in file %s failed.", state, file_path);
		return -1;
	}
	pDebug(DBG_INFO, "cpu %d set to throttling state %d", cpu, state);
	return 1;
}

int setThrottlingPercentCPU(const int cpu, const int percent)
{
	int states, current, requested;

	if (percent < 0 || percent > 100) {
		return -1;
	}
	requested = getThrottlingInfoCPU(cpu, &states, &current);
	if (requested < 0)
		return requested;
	requested = (int)roundf((percent * states) / 100);
	if (setThrottlingStateCPU(cpu, requested) < 0) {
		pDebug(DBG_DIAG, "settion throttling state on cpu %d"
		       " to %d - %d%%", cpu, requested, percent);
		return -1;
	}
	return 1;
}

int setThrottlingPercent(const int percent)
{
	int x;
	int cpus = 0;
	if ((cpus = getDevicesNum(ACPI_PROCESSOR_DIR)) < 0) {
		return -1;
	}

	for (x = 0; x < cpus; x++) {
		if (setThrottlingPercentCPU(x, percent) < 0)
			return -1;
	}
	return 1;
}

float getRealProcessorSpeed()
{

#if defined(__i386__) || defined(__x86_64__)
	unsigned long long int x;
	//      struct timezone tz;
	struct timeval tvstart, tvstop;
	u_int64_t cycles[2];
	unsigned int microseconds;	/* total time taken */
	int sleep_left, cpuinfo_fd, tsc_support;
	float ret;
	char buffer[MAX_LINE_SIZE + 1] = "";
	tsc_support = 0;

	/* search cpuinfo for tsc register support, if we find it, 
	   the calculation is supported for this machine */
	cpuinfo_fd = open("/proc/cpuinfo", O_RDONLY);
	if (cpuinfo_fd < 0)
		return -3.0;
	while (read(cpuinfo_fd, buffer, MAX_LINE_SIZE) > 0) {
		if (strstr(buffer, "tsc")) {
			tsc_support = 1;
			break;
		}
	}
	close(cpuinfo_fd);

	if (tsc_support != 1)
		return -2.0;

	/* get this function in cached memory */
	gettimeofday(&tvstart, NULL);
	__asm__ volatile ("rdtsc":"=A" (x));
	cycles[0] = x;
	gettimeofday(&tvstart, NULL);

	/* we don't trust that this is any specific length of time */
	sleep_left = sleep(1);
	if (sleep_left || errno == EINTR)
		return -1.0;
	__asm__ volatile ("rdtsc":"=A" (x));
	cycles[1] = x;
	gettimeofday(&tvstop, NULL);
	microseconds = ((tvstop.tv_sec - tvstart.tv_sec) * 1000000) + (tvstop.tv_usec - tvstart.tv_usec);

	ret = (float)(cycles[1] - cycles[0]) / microseconds;
	pDebug(DBG_DIAG, "%f MHz processor.", ret);
	return ret;

#else
	// not implemented
	return -2.0;
#endif
}

int calcCPULoad(const int consider_nice)
{
	unsigned long total_elapsed, working_elapsed;
	char what[32];
	unsigned long user_time, nice_time, system_time, idle_time, total_time, iowait_time;
	unsigned scan_ret;
	char line[256];
	char cpuString[7];
	FILE *fp;
	int new_cpu_count;
	
	new_cpu_count = sysconf(_SC_NPROCESSORS_CONF);

	if (new_cpu_count == -1 || new_cpu_count != cpuCount) {
		freeCPULoadData();
		cpuCount = new_cpu_count;
		if (cpuCount <= 0) {
			errno = ENODEV;
			return -20;
		}
		ps_last_total_time = (unsigned long *)calloc(cpuCount + 1, sizeof(unsigned long));
		ps_last_working_time = (unsigned long *)calloc(cpuCount + 1, sizeof(unsigned long));
		ps_cpuload = (int *)calloc(cpuCount + 1, sizeof(int));
	}

	if ((fp = fopen(PROC_STAT_FILE, "r")) == NULL) {
		pDebug(DBG_DIAG, "Could not open %s: %s", PROC_STAT_FILE, strerror(errno));
		return -1;
	}

	sprintf(cpuString, "cpu "); // start with the first line, "overall" cpu load
	// if cpuCount == 1, we do not need to evaluate "overall" and "per-cpu" load
	for (int i=0; i <= cpuCount - (cpuCount == 1); i++) {
		/* I think this could be needed for systems with more then 2 cpus in future */
//		if (getCPUOnline(i) <= 0) {
//			sprintf(cpuString, "cpu%d ", i);
//			continue;
//		}

		if (fgets(line,255,fp) == NULL) {
			pDebug(DBG_DIAG, "%s too short (%s)", PROC_STAT_FILE, cpuString);
			fclose(fp);
			return -1;
		}
		if (memcmp(line, cpuString, strlen(cpuString))) {
			pDebug(DBG_DIAG, "no '%s' string in %s line %d", cpuString, PROC_STAT_FILE, i);
			fclose(fp);
			return -1;
		}
		iowait_time = 0;	/* initialized, since it is simply not there in 2.4 */
		scan_ret = sscanf(line, "%s %lu %lu %lu %lu %lu", what, &user_time, &nice_time,
			  &system_time, &idle_time, &iowait_time);
		if (scan_ret < 5) {
			pDebug(DBG_DIAG, "only %d values in %s. Please report.", scan_ret, PROC_STAT_FILE);
			fclose(fp);
			return -1;
		}

		unsigned long working_time;
		if (consider_nice) {
			working_time = user_time + system_time + nice_time;
			idle_time += iowait_time;
		} else {
			working_time = user_time + system_time;
			idle_time += (nice_time + iowait_time);
		}
		total_time = working_time + idle_time;
		total_elapsed = total_time - ps_last_total_time[i];
		working_elapsed = working_time - ps_last_working_time[i];
		ps_last_working_time[i] = working_time;
		ps_last_total_time[i] = total_time;
		
		if (!total_elapsed) {
			if (!i) { // not once per CPU, only once per check.
				pDebug(DBG_DIAG, "%s not updated yet, poll slower.", PROC_STAT_FILE);
			}
		} else {
			// if (working_elapsed*100) gets too big for a long, we are in trouble anyway.
			ps_cpuload[i] = working_elapsed * 100 / total_elapsed;
		}
		sprintf(cpuString, "cpu%d ", i);
	}
	if (cpuCount == 1) // shortcut for UP systems.
		ps_cpuload[1] = ps_cpuload[0];

	fclose(fp);
	
	return 0;
}

int getCPULoad(const int cpu)
{

	if (cpu < -1 || cpu > MAX_SUPPORTED_CPUS) {
		errno = EINVAL;
		return -10;
	}
	/* just to be sure we get no segfault. Should I check array as well? 
	   Do I need a check at all? */
	if (ps_cpuload == NULL) {
		pDebug(DBG_WARN, "ps_cpuload uninitialized");
		errno = EFAULT;
		return -40;
	}

	if (cpu >= cpuCount) {
		errno = ENODEV;
		return -30;
	}

	return ps_cpuload[cpu + 1];
}

void freeCPULoadData()
{
	if (cpuCount != -1) {
		free(ps_last_working_time);
		free(ps_last_total_time);
		free(ps_cpuload);
	}
}

int getCPUOnline(int cpu_num)
{
	char file_path[MAX_FILE_PATH];
	char line[20];
	FILE *online_file;
	int online;

	sprintf(file_path, "%s/cpu%d/online", SYSFS_CPU_PATH, cpu_num);

	printf("cpupath: %s\n", file_path);

	if ((online_file = fopen(file_path, "r")) == NULL) {
		return -1;
	}
	
	if (fgets(line, 20, online_file) == NULL) {
		fclose(online_file);
		return -1;
	}

	fclose(online_file);

	online = atoi(line);

	printf("line read: %d\n", online);

	return online ? 1 : 0;
}
