#ifndef CPU_H
#define CPU_H

#include <string>

/** @brief class representing a generic CPU */
class CPU {
public:
	/** @brief constructor
	 *
	 * @param id the CPU is starting from 0
	 */
	CPU(int id);

	/** @brief set a CPU online
	 *
	 * CPU 0 can't be offlined
	 *
	 * @param cpu number of CPU starting from 1
	 *
	 * @retval < 0 error setting CPU online
	 * @retval 0   CPU CPU already online
	 * @retval > 0 successfully onlined CPU
	 */
	int enable();

	/** @brief set a CPU offline
	 *
	 * CPU 0 can't be offlined
	 *
	 * @param cpu number of CPU starting from 1
	 *
	 * @retval < 0 error setting CPU offline
	 * @retval 0   CPU CPU already offline
	 * @retval > 0 successfully offlined CPU
	 */
	int disable();

	/** @brief if the CPU is online
	 *
	 * @return true if CPU is online, false otherwise
	 */
	bool online();

	/** @brief enables all CPUs in system
	 *
	 * @return number of hotpluggable CPUs
	 */
	static int enableAll();

	/** @brief if the CPU can be offlined and onlined
	 *
	 * @return true if hotpluggable, false otherwise
	 */
	bool hotpluggable();

	/** @brief read a line from a file
	 *
	 * @param filename the filename to read from
	 * @param line char pointer to store the read line
	 * @param len chars to read
	 *
	 * @return false on error
	 */
	bool read_line(const char *filename, char *line, unsigned len);

	/** @brief write a line to a file
	 *
	 * @param filename the filename to read from
	 * @param fmt format to write
	 * @param ... variable argument list to write
	 *
	 * @return false on error
	 */
	bool write_line(const char *filename, const char *fmt, ...);

protected:
	/** @brief Stores which CPU this object corresponds to */
	int _cpu_base;

private:
	/** @brief filename for CPU hotplugging */
	std::string ONLINE_FILE;
};

#endif // CPU_HOTPLUG_H
