// ------------------------------------------------------------------------- //
// $Id: nodecontroller.h,v 1.16 2003/12/16 22:33:21 pandr Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// ------------------------------------------------------------------------- //

#ifndef _NODECONTROLLER_H_
#define _NODECONTROLLER_H_

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_PYTHON2_2
#  include <python2.2/Python.h>
#elif HAVE_PYTHON2_3
#  include <python2.3/Python.h>
#endif

#include "common.h"

// ------------------------------------------------------------------------- //

class Node;

/*!
 * The Parameters objects encapsulate a block of named parameters. It
 * is implemented in terms of a Python dictionary.
 */

class Parameters
{
public:
	Parameters(PyObject *py_dict);
	float get_float(const char* name) const;
	float get_float(const char* name, float def) const;
	void  set_float(const char* name, float f);
	int   get_int(const char* name) const;
	int   get_int(const char* name, int def) const;
	void  set_int(const char* name, int i);
private:
	PyObject* _py_dict;
};

// ------------------------------------------------------------------------- //

//! Controls the animation and/or mutation of a node
/*!
 * Every Node can have one or more controllers attached. A controller is
 * an object which will be given a chance each frame to do stuff to the
 * node to which it is attached.
 */
class NodeController
{
public:
	NodeController(Node* node);
	virtual ~NodeController() {};
	virtual	void frame_update() = 0;
	bool get_active() const { return _active; }
	void set_active(bool active) { _active = active; }
protected:
	Node* _node;
private:
	bool _active;
};

/*!
 * The EffectControllers are special NodeControllers that are used for 
 * making objects visible (possibly in a fancy way) during a slide show.
 */
class EffectController : public NodeController
{
public:
	enum state_enum {
		at_start,
		at_end,
		start_to_end,
		end_to_start
	};
	EffectController(Node* node, const Parameters& p);
	virtual void init(Parameters& p) { };
	virtual void frame_update() = 0;
	virtual void start() = 0;
	virtual void start_reverse() = 0;
protected:
	state_enum _state;
	int        _activate_at_step;
private:
	Parameters _parms;
};

/*!
 * The Rotator controller is a simple controller class which rotates the
 * node to which it is attached according to certain rules.
 */
class Rotator : public EffectController
{
public:
	Rotator(Node* node, const Parameters& p);
	void frame_update();
	void start();
	void start_reverse() { };
private:
	v3 _rotation;
	float _start_time;
};

/*!
 * The Fader controller fades in stuff by setting the alpha channel
 */
class Fader : public EffectController
{
public:
	Fader(Node* node, const Parameters& p);
	void frame_update();
	void start();
	void start_reverse() { };
private:
	float _start_time;
	float _duration;
	float _start_alpha;
	float _end_alpha;
};

/*!
 * The Slider controller can slide in/out the osd/ui stuff.
 * The constructor needs a starting point and a destination point
 * as well as the speed of the movement.
 */
class Slider : public EffectController
{
public:
	Slider(Node* node, const Parameters& p);
	void frame_update();
	void start();
private:
	v3    _start_pos;
	v3    _end_pos;
	v3    _direction;
	float _speed;
	float _length;
	float _start_time;
};

// ------------------------------------------------------------------------- //

//! Controls the animation and/or mutation of a node in the Python world
/*!
 * A controller object in python
 */
class PythonNodeController : public NodeController
{
public:
	PythonNodeController(Node* node, PyObject* py_controller);
	~PythonNodeController();
	void frame_update();
protected:
	PyObject* _py_controller;
};

//! Factory method used in the Python world
class PythonEffectController : public EffectController
{
public:
	PythonEffectController(Node* node, const Parameters& p);
	static EffectController* EffectControllerFactory(Node* node, 
			const char *effect_name, const Parameters& p);
};

// ------------------------------------------------------------------------- //

#endif /* _NODECONTROLLER_H_ */

// ------------------------------------------------------------------------- //

