// ------------------------------------------------------------------------- //
// $Id: logging.h,v 1.7 2004/01/04 17:41:39 weismann Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*! \file logging.h
 *	\brief The logging definitions provide a very simple but
 *	consistent way of printing messages. There is a C++ and a C
 *	interface to this functionality.
 *         
 *	The logging functions allows one to have printf-link
 *	functionality.
 */ 

#ifndef _LOGGING_H_ 
#define _LOGGING_H_

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_IOSTREAM
# include <iostream>
#endif

#if HAVE_FSTREAM
# include <fstream>
#endif

#if HAVE_STRING
# include <string>
#endif

// ------------------------------------------------------------------------- //
// C++ interface for logging

//! Log is the base class for every log instance
/*!
 *
 */
class Log
{
public:
	Log() : _logstream(std::cerr), _null("/dev/null"), 
		_currentstream(&std::cerr) { }
	Log(std::ostream& ls) : _logstream(ls), _null("/dev/null"),
		_currentstream(&ls) {};
	Log(unsigned int l) : _logstream(std::cerr), _null("/dev/null"),
		_currentstream(&std::cerr) { _level = l; }
	virtual ~Log() {};

	void pause() { _currentstream = &_null; } 
	void resume() { _currentstream = &_logstream; } 
	std::ostream& stream() { return *_currentstream; }
	virtual	std::string get_prefix() = 0;

	operator std::ostream& () { return stream(); }

private:
	std::ostream&  _logstream;
	std::ofstream  _null;
	std::ostream*  _currentstream;
	unsigned int   _level;
};

// overloaded output operator
template <class T> Log& operator<<(Log& l, T t)
{
	l.stream() << l.get_prefix() << t;
	return l;
}

class LogAbort : public Log
{
public:
	LogAbort() : Log() {};
	LogAbort(unsigned int l) : Log(l) {};
	LogAbort(std::ostream& ls) : Log(ls) {};
	~LogAbort() {};

	std::string get_prefix() {std::string tmp = "ABORT: "; return tmp; }
};

class LogWarn : public Log
{
public:
	LogWarn() : Log() {};
	LogWarn(unsigned int l) : Log(l) {};
	LogWarn(std::ostream& ls) : Log(ls) {};
	~LogWarn() {};

	std::string get_prefix() {std::string tmp = "WARN: "; return tmp;}
};

class LogInfo : public Log
{
public:
	LogInfo() : Log() {};
	LogInfo(unsigned int l) : Log(l) {};
	LogInfo(std::ostream& ls) : Log(ls) {};
	~LogInfo() {};

	std::string get_prefix() {std::string tmp = "Info: "; return tmp; }
};

class LogDebug : public Log
{
public:
	LogDebug() : Log() {};
	LogDebug(unsigned int l) : Log(l) {};
	LogDebug(std::ostream& ls) : Log(ls) {};
	~LogDebug() {};

	std::string get_prefix() {std::string tmp = "Debug: "; return tmp; }
};

// ------------------------------------------------------------------------- //
// C interface for logging

#ifdef __cplusplus
extern "C" {
#endif
extern void log_abort(const char *fmt, ...);
extern void log_warning(const char *fmt, ...);
extern void log_info(const char *fmt, ...);
extern void log_debug(const char *fmt, ...);
extern unsigned int log_level;
#ifdef __cplusplus
}	/* end extern C */
#endif

// ------------------------------------------------------------------------- //

#endif		// _LOGGING_H_

// ------------------------------------------------------------------------- //

