/*************************************************************************/
/* module:         XML scanner                                           */
/* file:           XLTDecXml.c                                           */
/* target system:  all                                                   */
/* target OS:      all                                                   */
/*************************************************************************/

/*
 * Copyright Notice
 * Copyright (c) Ericsson, IBM, Lotus, Matsushita Communication 
 * Industrial Co., LTD,Motorola, Nokia, Palm, Inc., Psion, 
 * Starfish Software (2001).
 * All Rights Reserved.
 * Implementation of all or part of any Specification may require 
 * licenses under third party intellectual property rights, 
 * including without limitation, patent rights (such a third party 
 * may or may not be a Supporter). The Sponsors of the Specification 
 * are not responsible and shall not be held responsible in any 
 * manner for identifying or failing to identify any or all such 
 * third party intellectual property rights.
 * 
 * THIS DOCUMENT AND THE INFORMATION CONTAINED HEREIN ARE PROVIDED 
 * ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND AND ERICSSON, IBM, 
 * LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO. LTD, MOTOROLA, 
 * NOKIA, PALM INC., PSION, STARFISH SOFTWARE AND ALL OTHER SYNCML 
 * SPONSORS DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION 
 * HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF 
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT 
 * SHALL ERICSSON, IBM, LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO., 
 * LTD, MOTOROLA, NOKIA, PALM INC., PSION, STARFISH SOFTWARE OR ANY 
 * OTHER SYNCML SPONSOR BE LIABLE TO ANY PARTY FOR ANY LOSS OF 
 * PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF 
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTAL, 
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH 
 * THIS DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 * 
 * The above notice and this paragraph must be included on all copies 
 * of this document that are made.
 * 
 */

/**
 * The XML scanner/tokenizer. Used by the SyncML parser.
 */

/*************************************************************************/
/* Definitions                                                           */
/*************************************************************************/
#include "xltdeccom.h"
#include "xlttags.h"

#include <libmem.h>
#include <libstr.h>
#include <xltencxml.h>
#include <smlerr.h>

/**
 * Private Interface for the XML scanner.
 */
typedef struct xmlScannerPriv_s xmlScannerPriv_t, *xmlScannerPrivPtr_t;
struct xmlScannerPriv_s
{
    /* public */
    Ret_t (*nextTok)(XltDecScannerPtr_t);
    Ret_t (*destroy)(XltDecScannerPtr_t);
    Ret_t (*pushTok)(XltDecScannerPtr_t);
    void (*setBuf)(XltDecScannerPtr_t pScanner, const MemPtr_t pBufStart, const MemPtr_t pBufEnd); 
    MemPtr_t (*getPos)(XltDecScannerPtr_t pScanner); 

    XltDecTokenPtr_t curtok;       /* current token */
    Long_t charset;                /* 0 */
    String_t charsetStr;           /* character set */
    Long_t pubID;                  /* 0 */
    String_t pubIDStr;             /* document public identifier */

    Flag_t finished;

    /* private */
    MemPtr_t pos;                  /* current position */
    MemPtr_t bufend;               /* end of buffer */
    String_t tagpre;               /*current tagprefix*/

};

/**
 * Public methods of the scanner interface.
 *
 * Description see XLTDecCom.h.
 */
static Ret_t _destroy(XltDecScannerPtr_t);
static Ret_t _nextTok(XltDecScannerPtr_t);
static Ret_t _pushTok(XltDecScannerPtr_t);
static void _setBuf(XltDecScannerPtr_t, const MemPtr_t, const MemPtr_t); 
static MemPtr_t _getPos(XltDecScannerPtr_t); 

/**
 * FUNCTION: readBytes
 *
 * Advance the current position pointer after checking whether the end of
 * the buffer has been reached. If the end of the buffer has been reached
 * the scanner's finished flag is set.
 *
 * PRE-Condition:
 * POST-Condition:
 *
 * IN:             bytes, read this many bytes
 *
 * IN/OUT:         pScanner, the scanner
 *
 * RETURNS:        1, if end of buffer has not been reached
 *                 0 otherwise
 */
static Boolean_t readBytes(xmlScannerPrivPtr_t pScanner, Long_t bytes);

/**
 * Skip whitespaces.
 */
static void skipS(xmlScannerPrivPtr_t pScanner);

static Ret_t xmlTag(xmlScannerPrivPtr_t pScanner, Byte_t endtag);
static Ret_t xmlName(xmlScannerPrivPtr_t pScanner, String_t *name);
static Ret_t xmlCharData(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlProlog(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlDocTypeDecl(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlXMLDecl(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlAttribute(xmlScannerPrivPtr_t pScanner, String_t *name, String_t *value);
static Ret_t xmlStringConst(xmlScannerPrivPtr_t pScanner, String_t *value);

static Ret_t xmlSkipPCDATA(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlSkipComment(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlSkipAttributes(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlSkipPI(xmlScannerPrivPtr_t pScanner);
static Ret_t xmlCDATA(xmlScannerPrivPtr_t pScanner);
Boolean_t isPcdata(XltTagID_t tagid);

/*************************************************************************/
/* External Functions                                                    */
/*************************************************************************/


Ret_t
xltDecXmlInit(const MemPtr_t pBufEnd, MemPtr_t *ppBufStart, XltDecScannerPtr_t *ppScanner)
{
        xmlScannerPrivPtr_t pScanner;
        Ret_t rc;

        pScanner = (xmlScannerPrivPtr_t)smlLibMalloc(sizeof(xmlScannerPriv_t));
        pScanner->finished = 0;
        pScanner->pos = *ppBufStart;
        pScanner->bufend = pBufEnd;
        pScanner->curtok = (XltDecTokenPtr_t)smlLibMalloc(sizeof(XltDecToken_t));
        pScanner->curtok->pcdata = NULL;
        pScanner->curtok->tagid = TN_UNDEF;
        pScanner->curtok->curpretok = NULL;
        pScanner->pubID = 0;
        pScanner->pubIDStr = NULL;
        pScanner->charset = 0;
        pScanner->charsetStr = NULL;
        pScanner->tagpre = NULL;

        /* point public/private methods to the right implementation */
        pScanner->nextTok = _nextTok;
        pScanner->destroy = _destroy;
        pScanner->pushTok = _pushTok;
        pScanner->setBuf = _setBuf;
        pScanner->getPos = _getPos;

        if ((rc = xmlProlog(pScanner)) != SML_ERR_OK) {
                smlLibFree(pScanner->curtok);
                smlLibFree(pScanner);
                *ppScanner = NULL;
                return rc;
        }

        *ppScanner = (XltDecScannerPtr_t)pScanner;

        return SML_ERR_OK;
}

/**
 * FUNCTION: destroy
 * 
 * Free memory. Description see XltDecAll.h.
 */
static Ret_t
_destroy(XltDecScannerPtr_t pScanner)
{
        xmlScannerPrivPtr_t pScannerPriv;

        if (pScanner == NULL)
            return SML_ERR_OK;

        pScannerPriv = (xmlScannerPrivPtr_t)pScanner;
        smlLibFree(pScannerPriv->curtok);
        smlLibFree(pScannerPriv->charsetStr);
        smlLibFree(pScannerPriv->pubIDStr);
        smlLibFree(pScannerPriv->tagpre);
        smlLibFree(pScannerPriv);

        return SML_ERR_OK;
}

/**
 * FUNCTION: nextTok
 *
 * Get next token. Description see XltDecAll.h.
 */
static Ret_t
_nextTok(XltDecScannerPtr_t pScanner)
{
        xmlScannerPrivPtr_t pScannerPriv;
        Ret_t rc;

        pScannerPriv = (xmlScannerPrivPtr_t)pScanner;

        pScannerPriv->curtok->start = pScannerPriv->pos;

        skipS(pScannerPriv);

        /* skip unsupported elements until we find a supported one */
        rc = 0;

        while (!rc) {
            if (smlLibStrncmp((String_t)pScannerPriv->pos, "<!--", 4) == 0) {
                rc = xmlSkipComment(pScannerPriv);
            } else if (smlLibStrncmp((String_t)pScannerPriv->pos, "<?", 2) == 0) {
                rc = xmlSkipPI(pScannerPriv);
            } else if (smlLibStrncmp((String_t)pScannerPriv->pos, "</", 2) == 0) {
                rc = xmlTag(pScannerPriv, 1);
                break;
            } else if (smlLibStrncmp((String_t)pScannerPriv->pos, "<![CDATA[", 9) == 0) {
                rc = xmlCDATA(pScannerPriv);
                break;
            } else if ((isPcdata(pScannerPriv->curtok->tagid)) && (pScannerPriv->curtok->type != TOK_TAG_END)) {
                rc = xmlSkipPCDATA(pScannerPriv);
                break;
            } else if (smlLibStrncmp((String_t)pScannerPriv->pos, "<", 1) == 0) {
                rc = xmlTag(pScannerPriv, 0);
                break;
            } else {
                rc = xmlCharData(pScannerPriv);
                break;
            }
        }
        if (rc)
            return rc;

        return SML_ERR_OK;
}

/**
 * FUNCTION: pushTok
 *
 * Reset the scanner to the starting position of the current token within
 * the buffer. Description see XltDecAll.h.
 */
static Ret_t _pushTok(XltDecScannerPtr_t pScanner)
{
        xmlScannerPrivPtr_t pScannerPriv;       

        pScannerPriv = (xmlScannerPrivPtr_t)pScanner;
        pScannerPriv->pos = pScannerPriv->curtok->start;

        /* invalidate curtok */
        smlLibMemset(pScannerPriv->curtok, 0, sizeof(XltDecTokenPtr_t));

        return SML_ERR_OK;
}

/**
 * FUNCTION: setBuf
 *
 * Set the working buffer of the scanner.
 */
static void
_setBuf(XltDecScannerPtr_t pScanner, const MemPtr_t pBufStart,
        const MemPtr_t pBufEnd)
{
    xmlScannerPrivPtr_t pScannerPriv = (xmlScannerPrivPtr_t)pScanner;
    pScannerPriv->pos = pBufStart;
    pScannerPriv->bufend = pBufEnd;
}

/**
 * FUNCTION: getPos
 *
 * Get the current position of the scanner within its working buffer.
 */
static MemPtr_t
_getPos(XltDecScannerPtr_t pScanner)
{
    return ((xmlScannerPrivPtr_t)pScanner)->pos;
}




/*************************************************************************/
/* Internal Functions                                                    */
/*************************************************************************/

/**
 * FUNCTION: readBytes
 *
 * Advance the position pointer. Description see above.
 */
static Boolean_t
readBytes(xmlScannerPrivPtr_t pScanner, Long_t bytes)
{
        if (pScanner->pos + bytes > pScanner->bufend) {
                pScanner->finished = 1;
                return 0;
        }
        pScanner->pos += bytes;
        return 1;
}

/**
 * FUNCTION: skipS
 *
 * Skip whitespace.
 */
static void skipS(xmlScannerPrivPtr_t pScanner)
{
    while (1) {
        switch (*pScanner->pos) {
            case  9: /* tab stop */
            case 10: /* line feed */
            case 13: /* carriage return */
            case 32: /* space */
                readBytes(pScanner, 1);
                break;
            default:
                return;
        }
    }
}

/**
 * FUNCTION: xmlProlog
 *
 * Scan the XML prolog (might be empty...).
 */
static Ret_t
xmlProlog(xmlScannerPrivPtr_t pScanner)
{
    Ret_t rc;

    if (pScanner->pos + 5 > pScanner->bufend)
        return SML_ERR_OK;
    if (smlLibStrncmp((String_t)pScanner->pos, "<?xml", 5) == 0)
        if ((rc = xmlXMLDecl(pScanner)) != SML_ERR_OK)
            return rc;

    skipS(pScanner);

    while ((pScanner->pos + 4 <= pScanner->bufend) &&
           ((smlLibStrncmp((String_t)pScanner->pos, "<!--", 4) == 0) ||
            (smlLibStrncmp((String_t)pScanner->pos, "<?", 2) == 0))) {
        if (smlLibStrncmp((String_t)pScanner->pos, "<!--", 4) == 0)
            rc = xmlSkipComment(pScanner);
        else
            rc = xmlSkipPI(pScanner);
        if (rc != SML_ERR_OK)
            return rc;
        skipS(pScanner);
    }

    if ((pScanner->pos + 9 <= pScanner->bufend) &&
        (smlLibStrncmp((String_t)pScanner->pos, "<!DOCTYPE", 9) == 0))
        if ((rc = xmlDocTypeDecl(pScanner)) != SML_ERR_OK)
            return rc;

    skipS(pScanner);

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlDocTypeDecl
 *
 * Part of the Prolog scanning
 */
static Ret_t
xmlDocTypeDecl(xmlScannerPrivPtr_t pScanner)
{
    Ret_t rc;
    String_t name = NULL;
    String_t syslit = NULL;
    String_t publit = NULL;

    readBytes(pScanner, 9);
    skipS(pScanner);
    if ((rc = xmlName(pScanner, &name)) != SML_ERR_OK) {
        smlLibFree(name);
        return rc;
    }
    skipS(pScanner);

    /* parse ExternalID */
    if ((pScanner->pos + 6 <= pScanner->bufend) &&
        (smlLibStrncmp((String_t)pScanner->pos, "SYSTEM", 6) == 0)) {
        readBytes(pScanner, 6);
        skipS(pScanner);
        if ((rc = xmlStringConst(pScanner, &syslit)) != SML_ERR_OK) {
            smlLibFree(name);
            smlLibFree(syslit);
            return rc;
        }
    } else if ((pScanner->pos + 6 <= pScanner->bufend) &&
         (smlLibStrncmp((String_t)pScanner->pos, "PUBLIC", 6) == 0)) {
        readBytes(pScanner, 6);
        skipS(pScanner);
        if ((rc = xmlStringConst(pScanner, &publit)) != SML_ERR_OK) {
            smlLibFree(name);
            smlLibFree(publit);
            return rc;
        }
        skipS(pScanner);
        if ((rc = xmlStringConst(pScanner, &syslit)) != SML_ERR_OK) {
            smlLibFree(name);
            smlLibFree(syslit);
            smlLibFree(publit);
            return rc;
        }
    }

    smlLibFree(name); 
    smlLibFree(syslit); 
    smlLibFree(publit); 

    skipS(pScanner);

    if (*pScanner->pos != '>')
        return SML_ERR_XLT_INVAL_XML_DOC;
    readBytes(pScanner, 1);

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlXMLDecl
 *
 * Part of the Prolog scanning
 */
static Ret_t
xmlXMLDecl(xmlScannerPrivPtr_t pScanner)
{
    String_t name, value;
    Ret_t rc;

    readBytes(pScanner, 5);
    skipS(pScanner);
    
    /* mandatory version info */
    if ((rc = xmlAttribute(pScanner, &name, &value)) != SML_ERR_OK)
        return rc;
    if (smlLibStrcmp(name, "version") != 0) {
        smlLibFree(name);
        smlLibFree(value);
        return SML_ERR_XLT_INVAL_XML_DOC;
    }
    smlLibFree(name);
    smlLibFree(value);

    skipS(pScanner);

    /* optional attributes are encoding and standalone */
    while ((pScanner->pos + 2 <= pScanner->bufend) &&
        (smlLibStrncmp((String_t)pScanner->pos, "?>", 2) != 0)) {
        if ((rc = xmlAttribute(pScanner, &name, &value)) != SML_ERR_OK)
            return rc;
        smlLibFree(name);
        smlLibFree(value);
        skipS(pScanner);
    }

    if (pScanner->pos + 2 > pScanner->bufend)
        return SML_ERR_XLT_END_OF_BUFFER;

    readBytes(pScanner, 2);

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlAttribute
 *
 * Handle Attributes //function can be used if attributes get necessary
 */
static Ret_t
xmlAttribute(xmlScannerPrivPtr_t pScanner, String_t *name, String_t *value)
{
    Ret_t rc;
    
    if ((rc = xmlName(pScanner, name)) != SML_ERR_OK)
        return rc;

    if (*pScanner->pos != '=') {
        smlLibFree(*name);
        smlLibFree(*value);
        *name = NULL;
        *value = NULL;
        return SML_ERR_XLT_INVAL_XML_DOC;
    }
    readBytes(pScanner, 1);

    skipS(pScanner);

    if ((rc = xmlStringConst(pScanner, value)) != SML_ERR_OK) {
        smlLibFree(*name);
        smlLibFree(*value);
        *name = NULL;
        *value = NULL;
        return rc;
    }

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlCharData
 *
 * Handle Pcdata String Constants
 */
static Ret_t
xmlStringConst(xmlScannerPrivPtr_t pScanner, String_t *value)
{
    String_t end;
    int len;
    char del;

    if ((*pScanner->pos != '"') && (*pScanner->pos != '\'')) {
        *value = NULL;
        return SML_ERR_XLT_INVAL_XML_DOC;
    }
    del = *pScanner->pos;
    readBytes(pScanner, 1);
        
    if ((end = smlLibStrchr((String_t)pScanner->pos, del)) == NULL) {
        *value = NULL;
        return SML_ERR_XLT_END_OF_BUFFER;
    }
    len = end - (String_t)pScanner->pos;
    if ((*value = (String_t)smlLibMalloc(len + 1)) == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    smlLibMemset(*value, 0, len + 1);
    smlLibStrncpy(*value, (String_t)pScanner->pos, len);
    readBytes(pScanner, len + 1);

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlCharData
 *
 * Handle Pcdata character data content
 */
static Ret_t
xmlCharData(xmlScannerPrivPtr_t pScanner)
{
    SmlPcdataPtr_t pPCData;
    MemPtr_t begin;
    int len;

    pPCData = (SmlPcdataPtr_t)smlLibMalloc(sizeof(SmlPcdata_t));
    if (pPCData == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    pPCData->contentType = SML_PCDATA_UNDEFINED;
    pPCData->length = 0;
    pPCData->content = NULL;

    begin = pScanner->pos;
    while (*pScanner->pos != '<') /* && (*pScanner->pos != '&') */
    {
      if (pScanner->pos >= pScanner->bufend)
      {
        smlLibFree(pPCData);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
      }
      readBytes(pScanner, 1);
    }
    len = pScanner->pos - begin;
    pPCData->content = smlLibMalloc(len + 1);
    smlLibMemset(pPCData->content, 0, len + 1);
    smlLibMemcpy(pPCData->content, begin, len);
    pPCData->contentType = SML_PCDATA_STRING;
    pPCData->length = len;

    pScanner->curtok->type = TOK_CONT;
    pScanner->curtok->pcdata = pPCData;

    return SML_ERR_OK;
}

#ifdef __META_EXT__
static Ret_t
getTagPrefix(String_t attname, String_t *tagprefix)
{
    String_t begin;
    String_t tmp;
    int len, i, itmp;
    
    begin = attname;
    itmp = smlLibStrlen(attname);
    for (i=1; i<=itmp; i++) 
    {
      if (!smlLibStrncmp(attname,":",1))
      {
        attname = attname+1;      
        break;
      }
      attname = attname+1;
    }
    if (i == itmp+1) return -1;
    len = smlLibStrlen(attname);
    tmp = (String_t)smlLibMalloc(len + 1);
    if (tmp == NULL) {
        *tagprefix = NULL;
        return SML_ERR_NOT_ENOUGH_SPACE;
    }
    smlLibMemset(tmp, 0, len + 1);
    smlLibStrncpy(tmp, (String_t)attname, len);
    *tagprefix = tmp;
    return SML_ERR_OK;  
}
#endif

/**
 * FUNCTION: xmlName
 *
 * Handle Name Elements
 */
static Ret_t
xmlName(xmlScannerPrivPtr_t pScanner, String_t *name)
{
    MemPtr_t begin;
    String_t tmp;
    int len;

    skipS(pScanner);
    begin = pScanner->pos;
    while (((*pScanner->pos >= 'a') && (*pScanner->pos <= 'z')) ||
           ((*pScanner->pos >= 'A') && (*pScanner->pos <= 'Z')) ||
           ((*pScanner->pos >= '0') && (*pScanner->pos <= '9')) ||
           (*pScanner->pos == '.') || (*pScanner->pos == '-') ||
           (*pScanner->pos == '_') || (*pScanner->pos == ':'))
        readBytes(pScanner, 1);
    len = pScanner->pos - begin;
    tmp = (String_t)smlLibMalloc(len + 1);
    if (tmp == NULL) {
        *name = NULL;
        return SML_ERR_NOT_ENOUGH_SPACE;
    }
    smlLibMemset(tmp, 0, len + 1);
    smlLibStrncpy(tmp, (String_t)begin, len);
    *name = tmp;
    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlTag
 *
 * Handle XML Tags
 */
static Ret_t
xmlTag(xmlScannerPrivPtr_t pScanner, Byte_t endtag)
{
    Ret_t rc;
    String_t name, tagprefix = NULL;
    XltTagID_t tagid;
    Byte_t codePage;
#ifdef __META_EXT__
    String_t attname, attvalue, stmp;
    int itmp;
#endif

    if (endtag) {
        if (!readBytes(pScanner, 2))
            return SML_ERR_XLT_END_OF_BUFFER;
    } else {
        if (!readBytes(pScanner, 1))
            return SML_ERR_XLT_END_OF_BUFFER;
    }

    if ((rc = xmlName(pScanner, &name)) != SML_ERR_OK)
    {
        return SML_ERR_XLT_INVAL_XML_DOC;
    }

#ifdef __META_EXT__
    attname = NULL; 
    attvalue = NULL;

    rc = xmlAttribute(pScanner, &attname, &attvalue);
    if (rc == SML_ERR_OK)
    {

   
      //if we encounter a metainf sub dtd
      if (!smlLibStrcmp(attvalue, META_INF))
      {
        if (pScanner->tagpre == NULL)
        {
          rc = getTagPrefix(attname, &tagprefix);
          if (rc != SML_ERR_OK) 
          {  
              smlLibFree(tagprefix);
          } 
          else
          {
            pScanner->tagpre = tagprefix;
          }
        }
      }

      if (pScanner->tagpre != NULL)
      {
        if (!smlLibStrncmp(name, pScanner->tagpre, smlLibStrlen(pScanner->tagpre))) 
        {  
           pScanner->curtok->curpretok = pScanner->tagpre;
        } 
        else
        {
           pScanner->curtok->curpretok = NULL;
        }
      }
     
      smlLibFree(attname);
      smlLibFree(attvalue);

    }


    if (pScanner->tagpre != NULL)
    {
      if (!smlLibStrncmp(pScanner->tagpre, name, smlLibStrlen(pScanner->tagpre)))
      {
        itmp = smlLibStrlen(name) - smlLibStrlen(pScanner->tagpre);
        stmp = smlLibMalloc(itmp);
        smlLibMemset(stmp,0,itmp);
        smlLibStrncpy(stmp, name+smlLibStrlen(pScanner->tagpre)+1, itmp-1);
        smlLibFree(name);
        name = smlLibMalloc(itmp);
        smlLibMemset(name,0,itmp);
        smlLibStrcpy(name, stmp);
        smlLibFree(stmp);
      }
    }
#endif

    rc = getTagIDByStringAndNamespace(name, "", &tagid);
    if ((tagid == TN_UNDEF) || (rc != SML_ERR_OK))
    {
      smlLibFree(name);
      return rc;
    }

    smlLibFree(name);

    pScanner->curtok->tagid = tagid;

    if ((rc = getCodePageById(pScanner->curtok->tagid, &codePage)) != SML_ERR_OK) return rc;

    if (codePage == 0)
    {
      pScanner->curtok->ext = SML_EXT_UNDEFINED;
    }
    else if (codePage == META_CP)
    {
      pScanner->curtok->ext = SML_EXT_META;
    }

    skipS(pScanner);

    if (endtag) {
        /* found end tag */
        if (smlLibStrncmp((String_t)pScanner->pos, ">", 1) != 0)
            return SML_ERR_XLT_INVAL_XML_DOC;
        pScanner->curtok->type = TOK_TAG_END;
        readBytes(pScanner, 1);
    } else {

        /* Attributes are not supported in SyncML -> skip them*/
        if ((rc = xmlSkipAttributes(pScanner)) != SML_ERR_OK) return rc;

        if (smlLibStrncmp((String_t)pScanner->pos, "/>", 2) == 0) {
            /* found empty tag */
            pScanner->curtok->type = TOK_TAG_EMPTY;
            readBytes(pScanner, 2);
        } else if (smlLibStrncmp((String_t)pScanner->pos, ">", 1) == 0) {
            pScanner->curtok->type = TOK_TAG_START;
            readBytes(pScanner, 1);
        } else {
            return SML_ERR_XLT_INVAL_XML_DOC;
        }
    }

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlSkipPI
 *
 * Skip PI elements
 */
static Ret_t
xmlSkipPI(xmlScannerPrivPtr_t pScanner)
{
    return SML_ERR_UNSPECIFIC;
}

/**
 * FUNCTION: xmlSkipComment
 *
 * Skip comments
 */
static Ret_t
xmlSkipComment(xmlScannerPrivPtr_t pScanner)
{
    readBytes(pScanner, 4);

    while ((pScanner->pos + 3 <= pScanner->bufend) &&
           (smlLibStrncmp((String_t)pScanner->pos, "-->", 3) != 0))
        readBytes(pScanner, 1);

    if (pScanner->pos + 3 > pScanner->bufend)
        return SML_ERR_XLT_END_OF_BUFFER;

    readBytes(pScanner, 3);

    skipS(pScanner);

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlSkipAttributes
 *
 * Skip attributes -> they are not supported in SyncML
 */
static Ret_t
xmlSkipAttributes(xmlScannerPrivPtr_t pScanner)
{ 

    while ((pScanner->pos + 1 <= pScanner->bufend) &&
           (smlLibStrncmp((String_t)pScanner->pos, ">", 1)) && (smlLibStrncmp((String_t)pScanner->pos, "/>", 2)))
        readBytes(pScanner, 1);   

    
    if (pScanner->pos + 1 > pScanner->bufend)
        return SML_ERR_XLT_END_OF_BUFFER;

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlCDATA
 *
 * Handle a CDATA content
 */
static Ret_t
xmlCDATA(xmlScannerPrivPtr_t pScanner)
{
    SmlPcdataPtr_t pPCData;
    MemPtr_t begin;
    int len;
    
    readBytes(pScanner, 9);

    pPCData = (SmlPcdataPtr_t)smlLibMalloc(sizeof(SmlPcdata_t));
    if (pPCData == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    pPCData->contentType = SML_PCDATA_UNDEFINED;
    pPCData->length = 0;
    pPCData->content = NULL;

    begin = pScanner->pos;
    while ((*pScanner->pos != ']') && (*pScanner->pos+1 != ']'))
        readBytes(pScanner, 1);
    len = pScanner->pos - begin;
    pPCData->content = smlLibMalloc(len + 1);
    smlLibMemset(pPCData->content, 0, len + 1);
    smlLibMemcpy(pPCData->content, begin, len);
    pPCData->contentType = SML_PCDATA_STRING;
    pPCData->length = len;

    pScanner->curtok->type = TOK_CONT;
    pScanner->curtok->pcdata = pPCData;

    readBytes(pScanner, 2);

    return SML_ERR_OK;
}

/**
 * FUNCTION: xmlSkipPCDATA
 *
 * Read over a Pcdata content
 */
static Ret_t
xmlSkipPCDATA(xmlScannerPrivPtr_t pScanner)
{
    XltTagID_t tagid = TN_UNDEF;
    SmlPcdataPtr_t pPCData;
    MemPtr_t begin;
    int len;
    Ret_t rc;
    String_t _tagString = NULL;
    String_t _tagString2 = NULL;
#ifdef __META_EXT__
    String_t _tagString3 = NULL;
#endif
    if (smlLibStrncmp((String_t)pScanner->pos, "<", 1) == 0) 
    {
      rc = xmlTag(pScanner, 0);
      return rc;
    }
    _tagString = smlLibMalloc(XML_MAX_TAGLEN+XML_MAX_PREFIXLEN+2);
    if (_tagString == NULL) return SML_ERR_NOT_ENOUGH_SPACE;
    if ((rc = getTagString(pScanner->curtok->tagid, _tagString)) != SML_ERR_OK) return rc;

#ifdef __META_EXT__    
    if (pScanner->curtok->curpretok != NULL)
    {
       _tagString3 = smlLibMalloc(XML_MAX_TAGLEN+XML_MAX_PREFIXLEN+2);
       if (_tagString3 == NULL) return SML_ERR_NOT_ENOUGH_SPACE;
       smlLibMemset(_tagString3, 0, XML_MAX_TAGLEN+XML_MAX_PREFIXLEN+2);
       smlLibStrcpy(_tagString3, pScanner->curtok->curpretok);
       smlLibStrcat(_tagString3, ":");
       smlLibStrcat(_tagString3, _tagString);
       smlLibStrcpy(_tagString, _tagString3);
       smlLibFree(_tagString3);
    }
#endif
    _tagString2 = smlLibMalloc(smlLibStrlen(_tagString) + 4);

    // build a end tag String to compate (e.g. </Meta>)
    if (_tagString2 == NULL) return SML_ERR_NOT_ENOUGH_SPACE;
    _tagString2 = smlLibStrcpy(_tagString2,"</");
    _tagString2 = smlLibStrcat(_tagString2,_tagString);
    _tagString2 = smlLibStrcat(_tagString2,">");
    smlLibFree(_tagString);

    pPCData = (SmlPcdataPtr_t)smlLibMalloc(sizeof(SmlPcdata_t));
    if (pPCData == NULL)
        return SML_ERR_NOT_ENOUGH_SPACE;
    pPCData->contentType = SML_PCDATA_UNDEFINED;
    pPCData->length = 0;
    pPCData->content = NULL;

    begin = pScanner->pos;

    // read Pcdata content until end tag appears
    while (smlLibStrncmp((String_t)pScanner->pos, _tagString2, smlLibStrlen(_tagString2)) != 0)
    {
      if (pScanner->pos >= pScanner->bufend)
      {
        smlLibFree(_tagString2);
        return SML_ERR_XLT_INVAL_SYNCML_DOC;
      }
      readBytes(pScanner, 1);
    }
    
    smlLibFree(_tagString2);

    len = pScanner->pos - begin;
    pPCData->content = smlLibMalloc(len + 1);
    smlLibMemset(pPCData->content, 0, len + 1);
    smlLibMemcpy(pPCData->content, begin, len);
    pPCData->contentType = SML_PCDATA_STRING;
    pPCData->length = len;

    pScanner->curtok->type = TOK_CONT;
    pScanner->curtok->pcdata = pPCData;


    return SML_ERR_OK;
}

/**
 * FUNCTION: isPcdata
 *
 * Check if the current tag id represents a Pcdata element
 */
Boolean_t isPcdata(XltTagID_t tagid)
{

  switch (tagid)
  {
    case TN_CMD:
    case TN_CMDID:
    case TN_CMDREF:
    case TN_LANG:
    case TN_LOCNAME:      
    case TN_LOCURI:   
    case TN_MSGID:   
    case TN_MSGREF:   
    case TN_RESPURI:   
    case TN_SESSIONID:   
    case TN_SOURCEREF:   
    case TN_TARGETREF:   
    case TN_VERSION:   
    case TN_PROTO:   
    case TN_DATA:
    case TN_META:
#ifdef __META_EXT__
    case TN_META_FORMAT:
    case TN_META_TYPE:
    case TN_META_MARK:
    case TN_META_SIZE:
    case TN_META_LAST:
    case TN_META_NEXT:
    case TN_META_VERSION:
    case TN_META_NEXTNONCE:
    case TN_META_MAXMSGSIZE:
    case TN_META_EMI:
    case TN_META_FREEMEM:
    case TN_META_FREEID:
#endif
      return 1;
    default:
      return 0;
  }
}
