/*************************************************************************/
/* module:          some helper functions                             	 */
/*                                                                       */   
/* file:            mgrutil.c		                                         */
/* target system:   all                                                  */
/* target OS:       all                                                  */   
/*                                                                       */   
/* Description:                                                          */   
/*************************************************************************/


/*
 * Copyright Notice
 * Copyright (c) Ericsson, IBM, Lotus, Matsushita Communication 
 * Industrial Co., LTD,Motorola, Nokia, Palm, Inc., Psion, 
 * Starfish Software (2001).
 * All Rights Reserved.
 * Implementation of all or part of any Specification may require 
 * licenses under third party intellectual property rights, 
 * including without limitation, patent rights (such a third party 
 * may or may not be a Supporter). The Sponsors of the Specification 
 * are not responsible and shall not be held responsible in any 
 * manner for identifying or failing to identify any or all such 
 * third party intellectual property rights.
 * 
 * THIS DOCUMENT AND THE INFORMATION CONTAINED HEREIN ARE PROVIDED 
 * ON AN "AS IS" BASIS WITHOUT WARRANTY OF ANY KIND AND ERICSSON, IBM, 
 * LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO. LTD, MOTOROLA, 
 * NOKIA, PALM INC., PSION, STARFISH SOFTWARE AND ALL OTHER SYNCML 
 * SPONSORS DISCLAIM ALL WARRANTIES, EXPRESS OR IMPLIED, INCLUDING 
 * BUT NOT LIMITED TO ANY WARRANTY THAT THE USE OF THE INFORMATION 
 * HEREIN WILL NOT INFRINGE ANY RIGHTS OR ANY IMPLIED WARRANTIES OF 
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. IN NO EVENT 
 * SHALL ERICSSON, IBM, LOTUS, MATSUSHITA COMMUNICATION INDUSTRIAL CO., 
 * LTD, MOTOROLA, NOKIA, PALM INC., PSION, STARFISH SOFTWARE OR ANY 
 * OTHER SYNCML SPONSOR BE LIABLE TO ANY PARTY FOR ANY LOSS OF 
 * PROFITS, LOSS OF BUSINESS, LOSS OF USE OF DATA, INTERRUPTION OF 
 * BUSINESS, OR FOR DIRECT, INDIRECT, SPECIAL OR EXEMPLARY, INCIDENTAL, 
 * PUNITIVE OR CONSEQUENTIAL DAMAGES OF ANY KIND IN CONNECTION WITH 
 * THIS DOCUMENT OR THE INFORMATION CONTAINED HEREIN, EVEN IF ADVISED 
 * OF THE POSSIBILITY OF SUCH LOSS OR DAMAGE.
 * 
 * The above notice and this paragraph must be included on all copies 
 * of this document that are made.
 * 
 */ 



/*************************************************************************
 *  Definitions
 *************************************************************************/


/* Include Headers */
#include <sml.h>
#include <smldtd.h>
#include <smldef.h>
#include <smlerr.h>
#include "libmem.h"
#include "libstr.h"
#include "mgr.h"


/* Prototypes of exported SyncML API functions */
SML_API Ret_t smlFreeProtoElement(VoidPtr_t pProtoElement);
SML_API void smlFreePcdata(SmlPcdataPtr_t pPcdata);
#ifndef __SML_LITE__  /* these API calls are NOT included in the Toolkit lite version */
  SML_API String_t smlPcdata2String( SmlPcdataPtr_t pcdata );
  SML_API SmlPcdataPtr_t smlString2Pcdata( String_t str );
  SML_API SmlPcdataPtr_t smlPcdataDup(SmlPcdataPtr_t pcdata); 
  SML_API MemSize_t smlGetFreeBuffer(InstanceID_t id);
#endif

/* Private function prototypes */
static void freeSyncHdr(SmlSyncHdrPtr_t pSyncHdr);
static void freeSync(SmlSyncPtr_t pSync);
static void freeGeneric(SmlGenericCmdPtr_t pGenericCmd);
static void freeAlert(SmlAlertPtr_t pAlert);
static void freeAtomic(SmlAtomicPtr_t pAtomic);
static void freeExec(SmlExecPtr_t pExec);
static void freeGetPut(SmlPutPtr_t pGetPut);
static void freeMap(SmlMapPtr_t pMap);
static void freeResults(SmlResultsPtr_t pResults);
static void freeSearch(SmlSearchPtr_t pSearch);
static void freeStatus(SmlStatusPtr_t pStatus);
static void freeCredPtr(SmlCredPtr_t pCred);
static void freeChalPtr(SmlChalPtr_t pChal);
static void freeSourceTargetPtr(SmlSourcePtr_t pSourceTarget);
static void freeSourceList(SmlSourceListPtr_t pSourceList);
static void freeSourceRefList(SmlSourceRefListPtr_t pSourceRefList);
static void freeTargetRefList(SmlTargetRefListPtr_t pTargetRefList);
static void freeItemPtr(SmlItemPtr_t pItem);
static void freeItemList(SmlItemListPtr_t pItemList);
static void freeMapItemPtr(SmlMapItemPtr_t pMapItem);
static void freeMapItemList(SmlMapItemListPtr_t pMapItemList);
#ifdef __META_EXT__
static void freeMeta(SmlMetaPtr_t pMeta);
static void freeAnchor(SmlAnchorPtr_t pAnchor);
static void freeMem(SmlMemPtr_t pAnchor);
static void freeEmiList(SmlEmiListPtr_t pEmiList);
#endif
/*************************************************************************
 *  Exported SyncML API functions
 *************************************************************************/

/**
 * FUNCTION: smlFreeProtoElement
 *
 * frees all allocated memory of a smlProtoElement
 *
 * IN:              VoidPtr_t
 *                  Element to free
 *
 * RETURN:          Ret_t
 *                  Return Code
 */
SML_API Ret_t smlFreeProtoElement(VoidPtr_t pProtoElement)
{
	if (! pProtoElement)
		return(SML_ERR_OK);

	switch (((SmlUnknownProtoElementPtr_t)pProtoElement)->elementType) {

		case SML_PE_HEADER:
			freeSyncHdr((SmlSyncHdrPtr_t)pProtoElement);
			break;

		case SML_PE_SYNC_START:
			freeSync((SmlSyncPtr_t)pProtoElement);
			break;

		case SML_PE_ADD:
		case SML_PE_COPY:
		case SML_PE_REPLACE:
		case SML_PE_DELETE:
		case SML_PE_GENERIC:
			freeGeneric((SmlGenericCmdPtr_t)pProtoElement);
			break;

		case SML_PE_ALERT:
			freeAlert((SmlAlertPtr_t)pProtoElement);
			break;

    case SML_PE_ATOMIC_START:
    case SML_PE_SEQUENCE_START:
		case SML_PE_CMD_GROUP:
			freeAtomic((SmlAtomicPtr_t)pProtoElement);
			break;

		case SML_PE_EXEC:
			freeExec((SmlExecPtr_t)pProtoElement);
			break;

    case SML_PE_PUT:
    case SML_PE_GET:
		case SML_PE_PUT_GET:
			freeGetPut((SmlPutPtr_t)pProtoElement);
			break;

		case SML_PE_MAP:
			freeMap((SmlMapPtr_t)pProtoElement);
			break;

		case SML_PE_RESULTS:
			freeResults((SmlResultsPtr_t)pProtoElement);
			break;

		case SML_PE_SEARCH:
			freeSearch((SmlSearchPtr_t)pProtoElement);
			break;

		case SML_PE_STATUS:
			freeStatus((SmlStatusPtr_t)pProtoElement);
			break;

		default:
			return(SML_ERR_A_UTI_UNKNOWN_PROTO_ELEMENT);
	}

	return(SML_ERR_OK);
}


/**
 * FUNCTION: smlFreePcdata
 *
 * frees the Memory of an allocated Pcdata memory object
 *
 * IN:              SmlPcdataPtr_t
 *                  A Pointer to a PcData structure, which should be freed
 *
 * RETURN:          ---
 *
 */
SML_API void smlFreePcdata(SmlPcdataPtr_t pPcdata)
{
	if (! pPcdata)
		return;

  if (pPcdata->contentType == SML_PCDATA_EXTENSION)
  {
#ifdef __META_EXT__
    if (pPcdata->extension == SML_EXT_META)
    {
      if (pPcdata->content)
          freeMeta(pPcdata->content);
    }
#endif
  }
	else if (pPcdata->content)
		smlLibFree(pPcdata->content);

	smlLibFree(pPcdata);
}



/*************************************************************************
 *  Exported SyncML API functions (FULL-SIZE TOOLKIT ONLY)
 *************************************************************************/

#ifndef __SML_LITE__  /* these API calls are NOT included in the Toolkit lite version */

/**
 * FUNCTION: smlGetFreeBuffer
 *
 * Return amount of unused Workspace memory
 *
 * RETURN:          MemSize_t
 *                  Amount of unused Workspace memory
 *
 */
SML_API MemSize_t smlGetFreeBuffer(InstanceID_t id)
{
	/* Definitions */ 
  MemSize_t  freeMem=0;
  
  /* ask the Workspace Buffer */
  wsmGetFreeSize(id, &freeMem);
   
	return (MemSize_t)freeMem;
}


/**
 * FUNCTION: smlString2Pcdata
 *
 * copy a string into a Pcdata structure
 *
 * IN:              String_t
 *                  Input String
 *
 * RETURN:          SmlPcdataPtr_t
 *                  A Pointer to a PcData structure
 *
 */
SML_API SmlPcdataPtr_t smlString2Pcdata(String_t str)
{
	/* Definitions */ 
	SmlPcdataPtr_t pcdata;

	/* Invalid Input */ 
    	if (! str) 
		return NULL;

	/* Allocate the PcData Structure */
	pcdata = (SmlPcdataPtr_t)smlLibMalloc((MemSize_t)sizeof(SmlPcdata_t));
	if (! pcdata) 
		return NULL;
    	smlLibMemset (pcdata, 0, (MemSize_t)sizeof(SmlPcdata_t));

	/* Set the PcData Structure */
	pcdata->contentType = SML_PCDATA_OPAQUE;
	pcdata->length = smlLibStrlen( str );
	pcdata->content = (VoidPtr_t)smlLibStrdup(str);

	return pcdata;
}


/**
 * FUNCTION: smlPcdata2String
 *
 * copy a Pcdata structure into a string
 *
 * IN:              SmlPcdataPtr_t
 *                  A Pointer to a PcData structure
 * RETURN:          String_t
 *                  Input String
 *
 */
SML_API String_t smlPcdata2String(SmlPcdataPtr_t pcdata)
{
	/* Definitions */ 
	String_t 	str;

	/* Invalid Input */ 
    	if (! pcdata) 
		return NULL;

    	/* Allocate the String */
	if (! (str = (String_t)smlLibMalloc((MemSize_t)(pcdata->length+1))))
		return NULL;

	/* Copy the string into the allocated data structure */
    	smlLibMemcpy((MemPtr_t)str, (MemPtr_t)pcdata->content, pcdata->length);
    	*(str + pcdata->length) = '\0';  

	return str;
}


/**
 * FUNCTION: smlPcdataDup
 *
 * Duplicates a Pcdata memory object
 *
 * IN:              SmlPcdataPtr_t
 *                  A Pointer to the original PcData structure
 *
 * RETURN:          SmlPcdataPtr_t
 *                  A Pointer to the copy of the PcData structure
 *
 */
SML_API SmlPcdataPtr_t smlPcdataDup(SmlPcdataPtr_t pcdata) 
{
	/* Definitions */ 
	SmlPcdataPtr_t newPcdata;

	/* Invalid Input */ 
	if (! pcdata) 
		return NULL;

	/* Allocate the new pcdata memory object */
	newPcdata = (SmlPcdataPtr_t)smlLibMalloc((MemSize_t)sizeof(SmlPcdata_t));
	if (! newPcdata) 
		return NULL;
	smlLibMemset (newPcdata, 0, (MemSize_t)sizeof(SmlPcdata_t));

	/* Set the PcData Structure */
	newPcdata->contentType = pcdata->contentType;
	newPcdata->length = pcdata->length;
	newPcdata->content =(VoidPtr_t)smlLibMalloc((MemSize_t)pcdata->length+1);
	smlLibMemset(newPcdata->content, 0, (MemSize_t)((pcdata->length)+1));	
	smlLibMemcpy(newPcdata->content, pcdata->content, (MemSize_t)pcdata->length);

	return newPcdata;
}

#endif




/*************************************************************************
 *  Private Functions of this Module
 *************************************************************************/

static void freeSyncHdr(SmlSyncHdrPtr_t pSyncHdr)
{
	if (! pSyncHdr)
		return;

	smlFreePcdata(pSyncHdr->version);
	smlFreePcdata(pSyncHdr->proto);
	smlFreePcdata(pSyncHdr->sessionID);
	smlFreePcdata(pSyncHdr->msgID);
	smlFreePcdata(pSyncHdr->respURI);
	smlFreePcdata(pSyncHdr->meta);

	freeSourceTargetPtr(pSyncHdr->source);
	freeSourceTargetPtr(pSyncHdr->target);

	freeCredPtr(pSyncHdr->cred);

	smlLibFree(pSyncHdr);
}

#ifdef __META_EXT__

static void freeAnchor(SmlAnchorPtr_t pAnchor)
{
	if (! pAnchor)
		return;

	smlFreePcdata(pAnchor->last);
	smlFreePcdata(pAnchor->next);

	smlLibFree(pAnchor);
}

static void freeMem(SmlMemPtr_t pMem)
{
	if (! pMem)
		return;

	smlFreePcdata(pMem->freeid);
	smlFreePcdata(pMem->freemem);

	smlLibFree(pMem);
}

static void freeEmiList(SmlEmiListPtr_t pEmiList)
{
	SmlEmiListPtr_t	pTmp;

	while (pEmiList) {
		pTmp = pEmiList->next;
		smlFreePcdata(pEmiList->emi);
		smlLibFree(pEmiList);
		pEmiList = pTmp;
	}
}

static void freeMeta(SmlMetaPtr_t pMeta)
{
	if (! pMeta)
		return;

	smlFreePcdata(pMeta->format);
	smlFreePcdata(pMeta->mark);
	smlFreePcdata(pMeta->nextnonce);
	smlFreePcdata(pMeta->size);
	smlFreePcdata(pMeta->type);
	smlFreePcdata(pMeta->version);
  smlFreePcdata(pMeta->maxmsgsize);
  
  freeEmiList(pMeta->emilist);

  freeMem(pMeta->mem);
	freeAnchor(pMeta->anchor);

	smlLibFree(pMeta);
}

#endif

static void freeSync(SmlSyncPtr_t pSync)
{
	if (! pSync)
		return;

	smlFreePcdata(pSync->cmdID);
	smlFreePcdata(pSync->meta);

	freeSourceTargetPtr(pSync->source);
	freeSourceTargetPtr(pSync->target);

	freeCredPtr(pSync->cred);

	smlLibFree(pSync);
}


static void freeGeneric(SmlGenericCmdPtr_t pGenericCmd)
{
	if (! pGenericCmd)
		return;

	smlFreePcdata(pGenericCmd->cmdID);
	smlFreePcdata(pGenericCmd->meta);

	freeCredPtr(pGenericCmd->cred);

	freeItemList(pGenericCmd->itemList);

	smlLibFree(pGenericCmd);
}


static void freeAlert(SmlAlertPtr_t pAlert)
{
	if (! pAlert)
		return;

	smlFreePcdata(pAlert->cmdID);
	smlFreePcdata(pAlert->data);

	freeCredPtr(pAlert->cred);

	freeItemList(pAlert->itemList);

	smlLibFree(pAlert);
}


static void freeAtomic(SmlAtomicPtr_t pAtomic)
{
	if (! pAtomic)
		return;

	smlFreePcdata(pAtomic->cmdID);
	smlFreePcdata(pAtomic->meta);

	smlLibFree(pAtomic);
}


static void freeExec(SmlExecPtr_t pExec)
{
	if (! pExec)
		return;

	smlFreePcdata(pExec->cmdID);

	freeCredPtr(pExec->cred);

	freeItemPtr(pExec->item);

	smlLibFree(pExec);
}


static void freeGetPut(SmlPutPtr_t pGetPut)
{
	if (! pGetPut)
		return;

	smlFreePcdata(pGetPut->cmdID);
	smlFreePcdata(pGetPut->meta);
	smlFreePcdata(pGetPut->lang);

	freeCredPtr(pGetPut->cred);

	freeItemList(pGetPut->itemList);

	smlLibFree(pGetPut);
}


static void freeMap(SmlMapPtr_t pMap)
{
	if (! pMap)
		return;

	smlFreePcdata(pMap->cmdID);
	smlFreePcdata(pMap->meta);

	freeCredPtr(pMap->cred);

	freeSourceTargetPtr(pMap->source);
	freeSourceTargetPtr(pMap->target);

	freeMapItemList(pMap->mapItemList);

	smlLibFree(pMap);
}


static void freeResults(SmlResultsPtr_t pResults)
{
	if (! pResults)
		return;

	smlFreePcdata(pResults->cmdID);
	smlFreePcdata(pResults->msgRef);
	smlFreePcdata(pResults->cmdRef);
	smlFreePcdata(pResults->meta);
	smlFreePcdata(pResults->targetRef);
	smlFreePcdata(pResults->sourceRef);

	freeItemList(pResults->itemList);

	smlLibFree(pResults);
}


static void freeSearch(SmlSearchPtr_t pSearch)
{
	if (! pSearch)
		return;

	smlFreePcdata(pSearch->cmdID);
	smlFreePcdata(pSearch->lang);
	smlFreePcdata(pSearch->meta);
	smlFreePcdata(pSearch->data);

	freeCredPtr(pSearch->cred);

	freeSourceTargetPtr(pSearch->target);

	freeSourceList(pSearch->sourceList);

	smlLibFree(pSearch);
}


static void freeStatus(SmlStatusPtr_t pStatus)
{
	if (! pStatus)
		return;

	smlFreePcdata(pStatus->cmdID);
	smlFreePcdata(pStatus->msgRef);
	smlFreePcdata(pStatus->cmdRef);
	smlFreePcdata(pStatus->cmd);
	smlFreePcdata(pStatus->data);

	freeCredPtr(pStatus->cred);
	freeChalPtr(pStatus->chal);

	freeTargetRefList(pStatus->targetRefList);
	freeSourceRefList(pStatus->sourceRefList);

	freeItemList(pStatus->itemList);

	smlLibFree(pStatus);
}


static void freeCredPtr(SmlCredPtr_t pCred)
{
	if (! pCred)
		return;

	smlFreePcdata(pCred->meta);
	smlFreePcdata(pCred->data);

	smlLibFree(pCred);
}


static void freeChalPtr(SmlChalPtr_t pChal)
{
	if (! pChal)
		return;

	smlFreePcdata(pChal->meta);

	smlLibFree(pChal);
}


static void freeSourceTargetPtr(SmlSourcePtr_t pSourceTarget)
{
	if (! pSourceTarget)
		return;

	smlFreePcdata(pSourceTarget->locURI);
	smlFreePcdata(pSourceTarget->locName);

	smlLibFree(pSourceTarget);
}


static void freeSourceList(SmlSourceListPtr_t pSourceList)
{
	SmlSourceListPtr_t	pTmp;

	while (pSourceList) {
		pTmp = pSourceList->next;
		freeSourceTargetPtr(pSourceList->source);
		smlLibFree(pSourceList);
		pSourceList = pTmp;
	}
}

static void freeSourceRefList(SmlSourceRefListPtr_t pSourceRefList)
{
	SmlSourceRefListPtr_t	pTmp;

	while (pSourceRefList) {
		pTmp = pSourceRefList->next;
		smlFreePcdata(pSourceRefList->sourceRef);
		smlLibFree(pSourceRefList);
		pSourceRefList = pTmp;
	}
}


static void freeTargetRefList(SmlTargetRefListPtr_t pTargetRefList)
{
	SmlTargetRefListPtr_t	pTmp;

	while (pTargetRefList) {
		pTmp = pTargetRefList->next;
		smlFreePcdata(pTargetRefList->targetRef);
		smlLibFree(pTargetRefList);
		pTargetRefList = pTmp;
	}
}


static void freeItemPtr(SmlItemPtr_t pItem)
{
	if (! pItem)
		return;

	smlFreePcdata(pItem->meta);
	smlFreePcdata(pItem->data);

	freeSourceTargetPtr(pItem->source);
	freeSourceTargetPtr(pItem->target);

	smlLibFree(pItem);
}


static void freeItemList(SmlItemListPtr_t pItemList)
{
	SmlItemListPtr_t	pTmp;

	while (pItemList) {
		pTmp = pItemList->next;
		freeItemPtr(pItemList->item);
		smlLibFree(pItemList);
		pItemList = pTmp;
	}
}


static void freeMapItemPtr(SmlMapItemPtr_t pMapItem)
{
	if (! pMapItem)
		return;

	freeSourceTargetPtr(pMapItem->source);
	freeSourceTargetPtr(pMapItem->target);

	smlLibFree(pMapItem);
}


static void freeMapItemList(SmlMapItemListPtr_t pMapItemList)
{
	SmlMapItemListPtr_t	pTmp;

	while (pMapItemList) {
		pTmp = pMapItemList->next;
		freeMapItemPtr(pMapItemList->mapItem);
		smlLibFree(pMapItemList);
		pMapItemList = pTmp;
	}
}
